#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Enum, List, Model, Str, Dict, Submodel, Int, Float
from CliPlugin.AclCliModel import IpRule, Ipv6Rule, MacRule
from CliPlugin.AclCliModel import IpAclList, Ipv6AclList, MacAclList
from LacpLib import printList
import Arnet, Tac, TableOutput, Ark

# Base class for Pbr action classes

# pylint: disable-next=inconsistent-return-statements
def matchConditionToStr( matchCondition ):
   if matchCondition == 'matchConditionAny':
      return 'match-any'

Action = Tac.Type( "PolicyMap::Action" )
ActionType = Tac.Type( "PolicyMap::ActionType" )
ClassMapMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
classMatchOptions = list( ClassMapMatchOption.attributes )

class ClassMatchOptionModel( Model ):
   option = Enum( values=classMatchOptions,
                  help='class match protocol options' )

def matchOptionToStr( option ):
   if option == ClassMapMatchOption.matchIpAccessGroup:
      return 'IP'
   elif option == ClassMapMatchOption.matchIpv6AccessGroup:
      return 'IPV6'
   elif option == ClassMapMatchOption.matchMacAccessGroup:
      return 'MAC'
   else:
      return 'MPLS'

classMatchConditions = list( \
                  Tac.Type( 'PolicyMap::ClassMapMatchCondition' ).attributes )

class MatchConditionModel( Model ):
   condition = Enum( values=classMatchConditions,
                     help='Class map match condition' )


mapTypes = list( Tac.Type( 'PolicyMap::MapType' ).attributes )

class MapTypeModel( Model ):
   mapType = Enum( values=mapTypes,
                   help='Map type' )


actionTypes = list( Tac.Type( 'PolicyMap::ActionType' ).attributes )
actionTypeAttrType = Enum( values=actionTypes,
                           help='class action type' )

class ClassMapCounter( Model ):
   packetCount = Int( help='Number of packets matching the rule',
                      optional=True )
   byteCount = Int( help='Number of bytes matching the rule',
                    optional=True )
   checkpointPacketCount = Int( help='Number of packets matching the '
                      'rule since last checkpoint',
                      optional=True )
   checkpointByteCount = Int( help='Number of bytes matching the '
                      'rule since last checkpoint',
                      optional=True )
   lastChangedTime = Float( help='Last time the counters changed',
                      optional=True )

   def render( self ):
      # We print class-map counters only on platforms that dont
      # support per-ACL counters.
      pass

   def getText( self ):
      counterString = ''
      if ( self.packetCount is not None and
           self.byteCount is not None and
           self.checkpointPacketCount is not None and
           self.checkpointByteCount is not None and
           self.lastChangedTime is not None ):
         if self.packetCount > self.checkpointPacketCount:
            if self.byteCount > self.checkpointByteCount:
               counterString = '[match %ld bytes in %ld packets, %s]' % (
                  self.byteCount - self.checkpointByteCount,
                  self.packetCount - self.checkpointPacketCount,
                  Ark.timestampToStr( self.lastChangedTime ) )
            else:
               counterString = '[match %ld packets, %s]' % (
                  self.packetCount - self.checkpointPacketCount,
                  Ark.timestampToStr( self.lastChangedTime ) )
      return counterString

class ClassMatchAclModel( Model ):
   name = Str( help='acl name' )
   ipAclList = Submodel( valueType=IpAclList,
                         optional=True, help='IP Acl list' )
   ipv6AclList = Submodel( valueType=Ipv6AclList,
                           optional=True, help='IPv6 Acl list' )
   macAclList = Submodel( valueType=MacAclList,
                          optional=True, help='MAC Acl list' )

class MplsRule( Model ):
   counterData = Submodel( valueType=ClassMapCounter, optional=True,
                           help='Number of packets that matched this class-map' )
   def render( self ):
      print( '        0 permit mpls any %s' % self.counterData.getText() )

class ClassMatchModel( Model ):
   option = ClassMatchOptionModel.option

   acl = Dict( keyType=int,
               valueType=ClassMatchAclModel,
               optional=True,
               help='Matching acl rule for class map' )
   mplsRule = Submodel( valueType=MplsRule,
                        optional=True,
                        help='Matching MPLS rule config' )
   ipRule = Dict( keyType=int,
                  valueType=IpRule,
                  optional=True,
                  help='Matching IP rule config' )
   ip6Rule = Dict( keyType=int,
                   valueType=Ipv6Rule,
                   optional=True,
                   help='Matching IPv6 rule config' )
   macRule = Dict( keyType=int,
                   valueType=MacRule,
                   optional=True,
                   help='Matching MAC rule config' )

   def render( self ):
      # This function assumes that only one of the collections has anything
      # in it. This is true in the current implementations.
      # XXX a hack to make 'show policy-map type pbr' work with mpls rule.
      if self.mplsRule is not None:
         print( '    Match:' )
         self.mplsRule.render()

      for prio in sorted( self.acl.keys() ):
         aclModel = self.acl[ prio ]
         if aclModel.ipAclList is not None:
            print( '    Match: %d' % prio, end=' ' )
            aclModel.ipAclList.render()
         elif aclModel.ipv6AclList is not None:
            print( '    Match: %d' % prio, end=' ' )
            aclModel.ipv6AclList.render()
         elif aclModel.macAclList is not None:
            print( '    Match: %d' % prio, end=' ' )
            aclModel.macAclList.render()
         else:
            print( '    Match: %d %s Access List %s' %
                   ( prio, matchOptionToStr( self.option ), aclModel.name ) )

      for prio in sorted( self.ipRule.keys() ):
         rule = self.ipRule[ prio ]
         print( '    Match:' )
         rule.render( indent=2 )
      for prio in sorted( self.ip6Rule.keys() ):
         rule = self.ip6Rule[ prio ]
         print( '    Match:' )
         rule.render( indent=2 )
      for prio in sorted( self.macRule.keys() ):
         rule = self.macRule[ prio ]
         print( '    Match:' )
         rule.render( indent=2 )

class ClassMapModel( Model ):
   name = Str( help='Class map name' )
   matchCondition = MatchConditionModel.condition
   counterData = Submodel( valueType=ClassMapCounter, optional=True,
                           help='Number of packets that matched this class-map' )
   match = Dict( valueType=ClassMatchModel,
                 optional=True,
                 help='Matching rule for class map' )

   def renderChildren( self ):
      for classMatch in self.match.values():
         classMatch.render()

   def render( self ):
      print( '  Class-map: %s (%s)' %
             ( self.name, matchConditionToStr( self.matchCondition ) ) )
      self.renderChildren()
      print()


class ClassMapAllModel( Model ):
   mapType = MapTypeModel.mapType
   classMaps = Dict( keyType=str,
                     valueType=ClassMapModel,
                     help='List of all class maps' )

   def render( self ):
      for classMap in sorted( self.classMaps ):
         self.classMaps[ classMap ].render()

   def append( self, cmapName, classMapModel ):
      self.classMaps[ cmapName ] = classMapModel

class ActionBase:
   actionClasses_ = {}
   @staticmethod
   def addActionType( actType, clazz ):
      ActionBase.actionClasses_[ actType ] = clazz

   @classmethod
   def createModel( cls, action ):
      actType = action.__class__
      clazz = ActionBase.actionClasses_[ actType ]
      return clazz.createModel( action )

class ActionModel( Model ):
   actionType = actionTypeAttrType
   def renderText( self ):
      raise NotImplementedError

   def render( self ):
      print( self.renderText() )

class NamedActionSetModel( Model ):
   name = Str( help='Action set name' )
   actions = Dict( keyType=str,
                   valueType=ActionModel,
                   optional=True,
                   help='A mapping from action type to action value' )

   @classmethod
   def createModel( cls, namedActionSet ):
      model = NamedActionSetModel()
      model.name = namedActionSet.name
      for attrType in namedActionSet.tacType.attributeQ:
         if hasattr( namedActionSet, attrType.name ):
            attr = getattr( namedActionSet, attrType.name )
            if ( attr is not None and
                 isinstance( attr, Action ) and
                 attr.actionType != ActionType.actionSet ):
               actType = attr.actionType
               model.actions[ actType ] = ActionBase.createModel( attr )
      return model

   def render( self ):
      actionStrList = [
            action.renderText()
            for action in self.actions.values() ]
      actionStrList.sort()
      actionStr = '    Action-set %s: ' % self.name
      actionStr += ', '.join( actionStrList )
      print( actionStr )

class ActionSetModel( Model ):
   actionSets = Dict( keyType=str,
                      valueType=NamedActionSetModel,
                      optional=True,
                      help='A mapping from action set name to actions' )

   @classmethod
   def createModel( cls, action ):
      model = ActionSetModel()
      for name, namedActionSet in action.namedActionSet.items():
         nasModel = NamedActionSetModel.createModel( namedActionSet )
         model.actionSets[ name ] = nasModel
      return model

   def render( self ):
      for name in sorted( self.actionSets ):
         actionSet = self.actionSets[ name ]
         actionSet.render()

class PolicyClassMapModel( Model ):
   name = Str( help='Class map name' )
   classMap = Submodel( valueType=ClassMapModel,
                        help='Class map configuration' )
   actionSetConfig = Submodel( valueType=ActionSetModel,
                               optional=True,
                               help='Action set configuration' )
   configuredAction = Dict( keyType=str,
                            valueType=ActionModel,
                            optional=True,
                            help='List of class actions' )

   def render( self ):
      self.classMap.renderChildren()
      actionStrList = [ x.renderText()
                        for x in self.configuredAction.values() ]
      actionStrList.sort()
      actionStr = '    Configured actions: '
      actionStr += ', '.join( actionStrList )
      print( actionStr )
      if self.actionSetConfig:
         self.actionSetConfig.render()

class PolicyMapModel( Model ):
   name = Str( help='Name of policy map' )
   mapType = MapTypeModel.mapType
   configuredIngressIntfs = List( valueType=str,
            optional=True,
            help='List of interfaces with the policy configured on ingress' )
   configuredIngressIntfsAsFallback = List( valueType=str,
            optional=True,
            help='List of interfaces with the policy configured '
                 'on ingress as fallback policy.' )
   appliedIngressIntfs = List( valueType=str,
            optional=True,
            help='List of interfaces with the policy applied on ingress' )
   rawClassMap = Dict( keyType=str,
                       optional=True,
                       valueType=ClassMapModel,
                       help='Pseudo class maps for single match statements' )
   classMap = Dict( keyType=int,
                    optional=True,
                    valueType=PolicyClassMapModel,
                    help='Mapping of priority to class map status' )

   def printIntfs( self, indent, firstColumnWidth, header, intfs ):
      table = TableOutput.TableFormatter( indent=indent )
      
      f = TableOutput.Format( justify="left",
                              minWidth=firstColumnWidth,
                              maxWidth=firstColumnWidth )
      f.padLimitIs( True )
      table.startColumn( f )
      table.newCell( header )
      f = TableOutput.Format( justify="left", wrap=True )
      f.noTrailingSpaceIs( True )
      f.padLimitIs( True )
      table.startColumn( f )
      table.newCell( ' '.join( intfs ) )
      print( table.output(), end=' ' )

   def render( self ):
      print( 'Service policy', self.name )
      intfLines = [ ( '  Configured on: ', self.configuredIngressIntfs,
                         'mandatory' ),
                    ( '  Configured as fallback policy on: ',
                         self.configuredIngressIntfsAsFallback, 'optional' ),
                    ( '  Applied on:    ', self.appliedIngressIntfs, 'madatory' ) ]
      #headerWidth = max( [ len( x[ 0 ] ) for x in intfLines ] )
      for header, intfCollection, printLine in intfLines:
         if printLine == 'optional' and not intfCollection:
            continue
         intfs = Arnet.sortIntf( intfCollection )
         printList( intfs, header, TableOutput.terminalWidth() - 1 )
         #self.printIntfs( 2, headerWidth, header, intfs )

      for prio in sorted( self.classMap.keys() ):
         classMapStatus = self.classMap[ prio ]
         className = classMapStatus.name
         if className in self.rawClassMap:
            print( '  %d: Single match statement' % prio )
         else:
            print( '  %d: Class-map: %s (%s)' % ( prio, className, 'match-any' ) )
         classMapStatus.render()
      print()

class PolicyMapAllModel( Model ):
   __revision__ = 2
   mapType = MapTypeModel.mapType
   policyMaps = Dict( keyType=str,
                      valueType=PolicyMapModel,
                      help='List of policy maps' )
   intf = Dict( keyType=str,
                valueType=str,
                help='Policy map to intf map' )

   def render( self ):
      for policyMap in sorted( self.policyMaps ):
         self.policyMaps[ policyMap ].render()

   def append( self, pmapName, policyMapModel ):
      self.policyMaps[ pmapName ] = policyMapModel
