#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports
import Cell
import CliParser
import ConfigMount
import LazyMount
import Plugins
import CliCommand
import ShowCommand
import CliMatcher
import BasicCli
import CliPlugin.TechSupportCli
import CliPlugin.IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import Tac
import Tracing
import AgentCommandRequest
import Smash
import SharedMem
from TrafficPolicyLib import tcpFlagAndMaskStrings
from CliPlugin.ClassificationCliLib import ( CommitAbortModelet,
                                             ProtocolMixin,
                                             ProtocolOredBase,
                                             PrefixIpv4SingletonCmd,
                                             PrefixIpv6SingletonCmd,
                                             generateTcpFlagExpression,
                                             tcpUdpProtoExpr,
                                             portKwNode,
                                             portExpression )
from CliMode.PostcardTelemetry import ( PostcardPolicyConfigMode,
                                        PostcardSamplePolicyConfigMode,
                                        PostcardSamplePolicyMatchRuleIpv4ConfigMode,
                                        PostcardSamplePolicyMatchRuleIpv6ConfigMode,
                                        PostcardSamplePolicyActionRuleConfigMode,
                                        ProfileConfigMode, FEATURE )
from CliPlugin.TrafficPolicyCli import ( matchRuleName,
                                         MatchRuleConfigCmdBase,
                                         ActionsConfigCmdBase )
from CliPlugin.PostcardTelemetryPolicyCliLib import (
   PostcardSamplePolicyMatchRuleContext,
   PostcardSamplePolicyContext )
from CliPlugin.TrafficPolicyCliModel import (
   NumericalRange,
   Protocol,
   Actions,
   Rule,
   Matches,
   Port,
   )
from CliPlugin.PostcardTelemetryPolicyCliModel import (
   PostcardTelemetry,
   ProfileModel,
   ModelPostcardProfile,
   SamplePolicyModel,
   SamplePoliciesModel,
   Collection,
   IntfList,
   SampleTcpUdpChecksum,
   PostcardTelemetryCounters,
   )
from Arnet import IpGenAddr
import CliToken
from CliToken.Monitor import monitorMatcher
from CliPlugin.SamplePolicyCliLib import ( SamplePolicyConfigCmd,
                                           SampleActionCmd )
from Toggles.PostcardTelemetryCommonToggleLib import (
   toggleFeaturePostcardTelemetryTcpControlEnabled )
from collections import defaultdict
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'PostcardTelemetryCli' )
t0 = Tracing.trace0

#------------------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------------------
postcardTelemetryConfig = None
postcardTelemetryStatus = None
allVrfStatus = None
hwCapability = None
policiesCliConfig = None
policiesStatusRequestDir = None
policiesStatus = None
tacSampleRate = Tac.Type( 'PostcardTelemetry::SampleRate' )
postcardDefaults = Tac.Type( 'PostcardTelemetry::Defaults' )
tacInstalledStatus = Tac.Type( 'PolicyMap::InstalledStatus' )
CounterKey = TacLazyType( 'Smash::PostcardTelemetry::CounterKey' )
PostcardPktCountersEntry = TacLazyType(
                           'Smash::PostcardTelemetry::PostcardPktCountersEntry' )
#------------------------------------------------------------------------------------
# Utility functions
#------------------------------------------------------------------------------------
# pylint: disable-msg=R1703

def identicalProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.samplePolicy != currentEntry.samplePolicy:
      return False
   return True

def identicalTelemetryConfig( oldConfig, newConfig ):
   if oldConfig.enable != newConfig.enable:
      return False
   oldPortProfile = list( oldConfig.portProfile )
   newPortProfile = list( newConfig.portProfile )
   oldIntfToPortProfile = list( oldConfig.intfToPortProfile.values() )
   newIntfToPortProfile = list( newConfig.intfToPortProfile.values() )
   if set( oldPortProfile ) != set( newPortProfile ):
      oldSet = set( oldPortProfile )
      newSet = set( newPortProfile )
      if list( oldSet - newSet ) != [ 'default' ] or \
         list( newSet - oldSet ):
         return False
   for profile in oldPortProfile:
      if profile == 'default':
         continue
      oldProfile = oldConfig.portProfile[ profile ]
      newProfile = newConfig.portProfile[ profile ]
      if not identicalProfile( oldProfile, newProfile ):
         return False
   if oldIntfToPortProfile != newIntfToPortProfile:
      return False
   if oldConfig.destinationConfig != newConfig.destinationConfig:
      return False
   if oldConfig.sampleRate != newConfig.sampleRate:
      return False
   if oldConfig.sampleDataAndMask != newConfig.sampleDataAndMask:
      return False
   if oldConfig.vxlanConfig != newConfig.vxlanConfig:
      return False
   return True

def isDefaultDestination( dstConfig ):
   return dstConfig.srcIp.v4Addr == '0.0.0.0' and \
      dstConfig.dstIp.v4Addr == '0.0.0.0' and \
      dstConfig.srcIp.v6Addr.stringValue == "::" and \
      dstConfig.dstIp.v6Addr.stringValue == "::"

def copyProfile( newProfile, oldProfile ):
   newProfile.samplePolicy = oldProfile.samplePolicy

def getPortProfileNames( mode, context ):
   return list( postcardTelemetryConfig.portProfile )

def defaultProfileGuard( mode, token ):
   if token == 'default':
      return CliParser.guardNotThisPlatform
   return None

profileNameMatcher = CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
      getPortProfileNames, "Profile name",
      passContext=True ), storeSharedResult=True )

def postcardTelemetryGuard( mode, token ):
   if hwCapability.postcardTelemetrySupported:
      return None
   return CliParser.guardNotThisPlatform

def postcardTelemetryTcpControlGuard( mode, token ):
   if hwCapability.postcardTelemetryTcpControlFlagsSupported:
      return None
   return CliParser.guardNotThisPlatform

#------------------------------------------------------------------------------------
# Modelet under Intf mode for postcard telemetry command
#------------------------------------------------------------------------------------

class IntfPostcardTelemetryModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( ( 'Ethernet', 'Port-Channel', 'Switch' ) )
               and not mode.intf.isSubIntf() )

CliPlugin.IntfCli.IntfConfigMode.addModelet( IntfPostcardTelemetryModelet )

#------------------------------------------------------------------------------------
# The "monitor telemetry postcard" mode context
#------------------------------------------------------------------------------------

class PostcardTelemetryContext:
   def __init__( self, mode ):
      self.mode = mode
      self.config = postcardTelemetryConfig
      self.currentEntry_ = None
      self.previousEntry_ = Tac.newInstance( 'PostcardTelemetry::PolicyConfig',
                                             "intConfig" )
      self.previousEntry_.sampleRate = postcardTelemetryConfig.sampleRate
      if postcardTelemetryConfig.sampleDataAndMask:
         self.previousEntry_.sampleDataAndMask = ( "sampleData", )
         self.previousEntry_.sampleDataAndMask.data = postcardTelemetryConfig.\
                                                      sampleDataAndMask.data
         self.previousEntry_.sampleDataAndMask.mask = postcardTelemetryConfig.\
                                                      sampleDataAndMask.mask
         self.previousEntry_.sampleDataAndMask.version = postcardTelemetryConfig.\
                                                         sampleDataAndMask.version
      else:
         self.previousEntry_.sampleDataAndMask = None
      if postcardTelemetryConfig.vxlanConfig:
         self.previousEntry_.vxlanConfig = ( "vxlanConfig", )
         self.previousEntry_.vxlanConfig.vxlanEnabled = postcardTelemetryConfig.\
                                                        vxlanConfig.vxlanEnabled
         self.previousEntry_.vxlanConfig.vxlanMarkerBit = postcardTelemetryConfig.\
                                                          vxlanConfig.vxlanMarkerBit
         self.previousEntry_.vxlanConfig.version = postcardTelemetryConfig.\
                                                   vxlanConfig.version
      else:
         self.previousEntry_.vxlanConfig = None
      self.previousEntry_.enable = postcardTelemetryConfig.enable

   def copyEditEntry( self ):
      newEntry = Tac.newInstance( 'PostcardTelemetry::PolicyConfig',
                                  "intConfig" )
      for profile in self.config.portProfile:
         newEntry.portProfile.newMember( profile )
         newProfile = newEntry.portProfile[ profile ]
         newProfile.samplePolicy = self.config.portProfile[
            profile ].samplePolicy
      for intf, profile in self.config.intfToPortProfile.items():
         newEntry.intfToPortProfile[ intf ] = profile
      newEntry.destinationConfig = self.config.destinationConfig
      newEntry.sampleRate = self.config.sampleRate
      if self.config.sampleDataAndMask:
         newEntry.sampleDataAndMask = ( "sampleData", )
         newEntry.sampleDataAndMask.data = self.config.sampleDataAndMask.data
         newEntry.sampleDataAndMask.mask = self.config.sampleDataAndMask.mask
         newEntry.sampleDataAndMask.version = self.config.sampleDataAndMask.version
      else:
         newEntry.sampleDataAndMask = None
      if self.config.vxlanConfig:
         newEntry.vxlanConfig = ( "vxlanConfig", )
         newEntry.vxlanConfig.vxlanEnabled = self.config.vxlanConfig.vxlanEnabled
         newEntry.vxlanConfig.vxlanMarkerBit = self.config.vxlanConfig.vxlanMarkerBit
         newEntry.vxlanConfig.version = self.config.vxlanConfig.version
      else:
         newEntry.vxlanConfig = None
      newEntry.enable = self.config.enable
      self.currentEntry_ = newEntry
      return newEntry

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if self.currentEntry_:
         if identicalTelemetryConfig( self.config, self.currentEntry_ ):
            self.config.enable = self.currentEntry_.enable
            return

         # Set enable upon successfully committing from
         # postcard telemetry mode's context.
         self.config.sampleRate = self.currentEntry_.sampleRate
         if self.currentEntry_.sampleDataAndMask:
            self.config.sampleDataAndMask = ( "sampleData", )
            self.config.sampleDataAndMask.data = self.currentEntry_.\
                                                 sampleDataAndMask.data
            self.config.sampleDataAndMask.mask = self.currentEntry_.\
                                                 sampleDataAndMask.mask
            self.config.sampleDataAndMask.version = self.currentEntry_.\
                                                    sampleDataAndMask.version
         else:
            self.config.sampleDataAndMask = None
         if self.currentEntry_.vxlanConfig:
            self.config.vxlanConfig = ( "vxlanConfig", )
            self.config.vxlanConfig.vxlanEnabled = self.currentEntry_.\
                                                   vxlanConfig.vxlanEnabled
            self.config.vxlanConfig.vxlanMarkerBit = self.currentEntry_.\
                                                     vxlanConfig.vxlanMarkerBit
            self.config.vxlanConfig.version = self.currentEntry_.\
                                              vxlanConfig.version
         else:
            self.config.vxlanConfig = None
         self.config.destinationConfig = self.currentEntry_.destinationConfig
         self.config.enable = self.currentEntry_.enable

#------------------------------------------------------------------------------------
# The "monitor telemetry MODE" mode command
#------------------------------------------------------------------------------------

class MonitorTelemetryPostcard( CliCommand.CliCommandClass ):
   syntax = '''monitor telemetry postcard policy'''
   noOrDefaultSyntax = syntax
   data = {
      'monitor': monitorMatcher,
      'telemetry': 'Telemetry configuration',
      'postcard': CliCommand.guardedKeyword( 'postcard',
                                             helpdesc='Postcard telemetry',
                                             guard=postcardTelemetryGuard ),
      'policy': 'Postcard telemetry of type match policy',
   }

   @staticmethod
   def handler( mode, args ):
      context = PostcardTelemetryContext( mode )
      context.copyEditEntry()
      childMode = mode.childMode( PostcardPolicyConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for name in policiesCliConfig.pmapType.pmap:
         # pylint: disable=protected-access
         if name != 'default':
            PostcardSamplePolicyConfigCmd._removePolicy( mode, name )
      profiles = list( postcardTelemetryConfig.portProfile )
      for profile in profiles:
         if profile != 'default':
            del postcardTelemetryConfig.portProfile[ profile ]
      postcardTelemetryConfig.sampleRate = tacSampleRate.sampleRateInvalid
      postcardTelemetryConfig.sampleDataAndMask = None
      postcardTelemetryConfig.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( '0.0.0.0' ), IpGenAddr( '0.0.0.0' ),
         postcardDefaults.aristaSubTypeVersion,
         postcardDefaults.ttl, postcardDefaults.dscp,
         postcardDefaults.vrf, postcardDefaults.greProtocolType )
      postcardTelemetryConfig.vxlanConfig = None
      postcardTelemetryConfig.enable = False

def postcardIpv6Guard( mode, token ):
   if hwCapability.postcardTelemetryIpv6Supported:
      return None
   else:
      return CliParser.guardNotThisPlatform

nodeIpv6 = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ipv6',
                            helpdesc='IPv6 match rule' ),
                            guard=postcardIpv6Guard )
#[ no ]  match  <match-rule-name> ( ipv4 | ipv6 )
class PostcardSamplePolicyMatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = 'match RULE_NAME ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'match RULE_NAME ( ipv4 | ipv6 )'
   data = {
      'match': 'Configure match rule',
      'RULE_NAME': matchRuleName,
      'ipv4': 'IPv4 match rule',
      'ipv6': nodeIpv6,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return PostcardSamplePolicyMatchRuleContext( trafficPolicyContext, ruleName,
                                                   matchOption )

class PostcardSamplePolicyActionsConfigCmd( ActionsConfigCmdBase ):
   data = ActionsConfigCmdBase.data.copy()

   @staticmethod
   def _feature():
      return FEATURE

#------------------------------------------------------------------------------------
# The profile  mode context
#------------------------------------------------------------------------------------

class ProfileModeContext:
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.currentEntry_ = None
      self.previousEntry_ = None
      portProfileConfig = postcardTelemetryConfig.portProfile
      if profileName in portProfileConfig:
         self.profile_ = portProfileConfig[ profileName ]
         prevProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                        profileName )
         copyProfile( prevProfile, self.profile_ )
         self.previousEntry_ = prevProfile

   def copyEditEntry( self ):
      newProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                    self.profileName_ )
      copyProfile( newProfile, self.profile_ )
      self.currentEntry_ = newProfile
      return newProfile

   def newEditEntry( self ):
      newProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                    self.profileName_ )
      self.currentEntry_ = newProfile

   def profileName( self ):
      return self.profileName_

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if identicalProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ is None:
         self.profile_ = postcardTelemetryConfig.portProfile.newMember(
            self.profileName_ )
      for intf in postcardTelemetryConfig.intfToPortProfile:
         if postcardTelemetryConfig.intfToPortProfile[ intf ] != \
            self.profileName_:
            continue
      self.profile_.samplePolicy = self.currentEntry_.samplePolicy

def getSamplePolicyNames( mode ):
   if not policiesCliConfig:
      return []
   return policiesCliConfig.pmapType.pmapInput

samplePolicyNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   getSamplePolicyNames, "Sample Policy Name",
   pattern=r'[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

class PostcardSamplePolicyConfigCmd( SamplePolicyConfigCmd ):
   data = {
      'POLICY_NAME': samplePolicyNameMatcher,
   }
   data.update( SamplePolicyConfigCmd._baseData )

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name, mode ):
      return PostcardSamplePolicyContext( policiesCliConfig,
                                          policiesStatusRequestDir,
                                          policiesStatus, name )

ingressMatcher = CliMatcher.KeywordMatcher( 'ingress',
                                            helpdesc='Configure ingress paramters' )

class GreDestinationCmd( CliCommand.CliCommandClass ):
   syntax = """ingress collection gre source SRC_ADDR
              destination DST_ADDR
              [ version VERSION ]"""
   # For phase 1, optional parameters are not supported in this CLI command
   #
   #            [ { ( ttl TTLVAL )
   #              | ( dscp DSCPVAL )
   #              | ( protocol PROTOVAL )
   #              | ( VRF ) } ]"""
   noOrDefaultSyntax = "ingress collection gre ..."
   data = {
      'ingress': ingressMatcher,
      'collection': 'Collector configuration',
      'gre': 'GRE conifguration',
      'source': 'GRE source configuration',
      'SRC_ADDR': IpGenAddrMatcher(
         helpdesc='Source IP address of GRE tunnel' ),
      'destination': 'GRE destination configuration',
      'DST_ADDR': IpGenAddrMatcher(
         helpdesc='Destination IP address of GRE tunnel' ),
      'version': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'version',
                                  helpdesc='Postcard version configuration' ) ),
      'VERSION': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 2, helpdesc='Postcard version' ),
         maxMatches=1 )

      # 'ttl': CliCommand.Node( CliMatcher.KeywordMatcher(
      #   'ttl', 'TTL of the GRE tunnel' ), maxMatches=1 ),
      # 'TTLVAL': CliCommand.Node(
      #   CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ), maxMatches=1 ),
      # 'dscp': CliCommand.Node(
      #   CliMatcher.KeywordMatcher( 'dscp', 'DSCP of the GRE tunnel' ),
      #   maxMatches=1 ),
      # 'DSCPVAL': CliCommand.Node(
      #   CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ), maxMatches=1 ),
      # 'protocol': CliCommand.Node(
      #   CliMatcher.KeywordMatcher( 'protocol', 'Protocol type in GRE header' ),
      #    maxMatches=1 ),
      # 'PROTOVAL': CliCommand.Node( CliMatcher.IntegerMatcher( 0, 65535,
      #   helpdesc='Protocol range', helpname='0x0000-0xFFFF' ), maxMatches=1 ),
      # 'VRF': VrfExprFactory( helpdesc='VRF name of the GRE tunnel', maxMatches=1 )
   }

   @staticmethod
   def handler( mode, args ):
      greSrcAddr = args[ 'SRC_ADDR' ].stringValue
      greDstAddr = args[ 'DST_ADDR' ].stringValue
      ttlVal = args.get( 'TTLVAL', postcardDefaults.ttl )
      dscpVal = args.get( 'DSCPVAL', postcardDefaults.dscp )
      protocol = args.get( 'PROTOVAL', postcardDefaults.greProtocolType )
      vrf = args.get( 'VRF', postcardDefaults.vrf )
      subTypeVersion = args.get( 'VERSION', postcardDefaults.aristaSubTypeVersion )
      mode.telemetryContext.currentEntry_.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( greSrcAddr ), IpGenAddr( greDstAddr ),
         subTypeVersion,
         ttlVal, dscpVal, vrf, protocol )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.telemetryContext.currentEntry_.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( '0.0.0.0' ), IpGenAddr( '0.0.0.0' ),
         postcardDefaults.aristaSubTypeVersion,
         postcardDefaults.ttl, postcardDefaults.dscp,
         postcardDefaults.vrf, postcardDefaults.greProtocolType )

#------------------------------------------------------------------------------------
# The "profile <profile-type> <profile-name>" mode command
#------------------------------------------------------------------------------------
profileNameMatcherWithGuard = CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
   getPortProfileNames, "Profile name", passContext=True ),
   storeSharedResult=True, guard=defaultProfileGuard )
class ProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''profile PROFILE'''
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'Configure postcard telemetry profile',
      'PROFILE': profileNameMatcherWithGuard,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE' ]
      context = ProfileModeContext( mode, profileName )

      if profileName in postcardTelemetryConfig.portProfile:
         context.copyEditEntry()
      else:
         context.newEditEntry()
      childMode = mode.childMode( ProfileConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE' ]
      del postcardTelemetryConfig.portProfile[ profileName ]

rateMatcher = CliMatcher.EnumMatcher( {
   '16384': 'Set sample rate to 1 in 16k packets',
   '32768': 'Set sample rate to 1 in 32k packets',
   '65536': 'Set sample rate to 1 in 64k packets',
} )

checksumMaskMatcher = CliMatcher.PatternMatcher(
   pattern='0x?[a-fA-F0-9]{1,4}', helpname='0x0-0xffff',
   helpdesc="16 bit hex mask for TCP/UDP or IP ID with atmost 2 unset bits" )

class ChecksumMaskExpression( CliCommand.CliExpression ):
   expression = 'mask CHECKSUM_MASK'
   data = {
      'mask' : 'Configure mask of TCP/UDP checksum or IP ID',
      'CHECKSUM_MASK': checksumMaskMatcher
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'mask' not in args:
         return
      checksumMask = args.pop( 'CHECKSUM_MASK' )
      checksumMask = int( checksumMask, 16 )

      numBitsSet = bin( checksumMask ^ 0xffff ).count( '1' )
      if numBitsSet > 2:
         raise CliParser.InvalidInputError()
      args[ 'MASK' ] = checksumMask

class TcpUdpChecksumExpression( CliCommand.CliExpression ):
   expression = '( tcp-udp-checksum value CHECKSUM_VAL MASK_EXPR )'
   data = {
      'tcp-udp-checksum' : 'Configure TCP/UDP checksum or IP ID parameters',
      'value' : 'Configure value of TCP/UDP checksum or IP ID',
      'CHECKSUM_VAL' : CliMatcher.IntegerMatcher(
         0, 0xffff, helpdesc='TCP/UDP checksum or IP ID value 0,1,2,3,0xffff' ),
      'MASK_EXPR' : ChecksumMaskExpression,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'tcp-udp-checksum' not in args:
         return
      checksumVal = args.pop( 'CHECKSUM_VAL' )
      sampleDataAndMask = Tac.newInstance( 'Classification::TcpUdpCheckSumOrIpId',
                                           'sampleData' )
      sampleDataAndMask.data = checksumVal
      sampleDataAndMask.mask = args[ 'MASK' ]
      args[ 'CHECKSUM' ] = sampleDataAndMask

class SampleRateCmd( CliCommand.CliCommandClass ):
   syntax = 'ingress sample ( rate RATE_VAL ) | TCP_UDP_CHECKSUM'
   noOrDefaultSyntax = 'ingress sample ( rate ... ) |  TCP_UDP_CHECKSUM '
   data = {
      'ingress': ingressMatcher,
      'sample': 'Configure sampling parameters',
      'rate': 'Configure sampling rate',
      'RATE_VAL': rateMatcher,
      'TCP_UDP_CHECKSUM' : TcpUdpChecksumExpression,
   }

   @staticmethod
   def handler( mode, args ):
      currEntry = mode.telemetryContext.currentEntry_
      if 'RATE_VAL' in args:
         if currEntry.sampleDataAndMask:
            msg = "Cannot apply sample rate configuration." \
                  " Sample data/mask is already configured"
            mode.addError( msg )
            return
         currEntry.sampleRate = int( args[ 'RATE_VAL' ] )
      elif 'CHECKSUM' in args:
         if currEntry.sampleRate != tacSampleRate.sampleRateInvalid:
            msg = "Cannot apply data/mask configuration." \
                  " Sample rate is already configured"
            mode.addError( msg )
            return
         currEntry.sampleDataAndMask = ( "sampleData", )
         currEntry.sampleDataAndMask.data = args[ 'CHECKSUM' ].data
         currEntry.sampleDataAndMask.mask = args[ 'CHECKSUM' ].mask
         currEntry.sampleDataAndMask.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currEntry = mode.telemetryContext.currentEntry_
      if 'CHECKSUM' in args:
         if currEntry.sampleDataAndMask and \
            currEntry.sampleDataAndMask.data == args[ 'CHECKSUM' ].data and \
            currEntry.sampleDataAndMask.mask == args[ 'CHECKSUM' ].mask:
            currEntry.sampleDataAndMask = None
      else:
         if CliCommand.isNoCmd( args ):
            currEntry.sampleRate = tacSampleRate.sampleRateInvalid
         else:
            currEntry.sampleRate = tacSampleRate.sampleRate1in32k

def postcardVxlanGuard( mode, token ):
   if hwCapability.postcardTelemetryVxlanSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def postcardVxlanMarkerBitConfigurableGuard( mode, token ):
   if hwCapability.postcardTelemetryVxlanMarkerBitConfigurable:
      return None
   else:
      return CliParser.guardNotThisPlatform

markerMatcher = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'marker', helpdesc='Configure marker mode parameters' ),
   guard=postcardVxlanGuard )

headerMatcher = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'header', helpdesc='VXLAN header marking' ),
   guard=postcardVxlanMarkerBitConfigurableGuard )

class VxlanMarkerCmd( CliCommand.CliCommandClass ):
   syntax = '''marker vxlan [ header word 0 bit MARKER_BIT ]'''
   noOrDefaultSyntax = '''marker vxlan'''
   data = {
      'marker': markerMatcher,
      'vxlan': 'Enable vxlan marking using default bit 0',
      'header': headerMatcher,
      'word': 'word',
      '0': 'VXLAN header word 0',
      'bit': 'VXLAN reserved bit offset',
      'MARKER_BIT': CliCommand.Node(
          CliMatcher.IntegerMatcher( 1, 31, helpdesc='VXLAN marker bit' ),
                                     maxMatches=1 ),
   }

   @staticmethod
   def handler( mode, args ):
      currEntry = mode.telemetryContext.currentEntry_
      currEntry.vxlanConfig = ( 'vxlanConfig', )
      currEntry.vxlanConfig.vxlanMarkerBit = args.get( 'MARKER_BIT', 0 )
      currEntry.vxlanConfig.vxlanEnabled = True
      currEntry.vxlanConfig.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.telemetryContext.currentEntry_.vxlanConfig = None

class SamplePolicyCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample policy POLICY_NAME'''
   noOrDefaultSyntax = '''ingress sample policy [ POLICY_NAME ]'''
   data = {
      'ingress': ingressMatcher,
      'sample': 'Configure sampling parameters',
      'policy': 'Configure sample policy',
      'POLICY_NAME': samplePolicyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      profile = mode.profileContext.currentEntry_
      profile.samplePolicy = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      policyName = args.get( 'POLICY_NAME' )
      if not policyName or policyName == profile.samplePolicy:
         profile.samplePolicy = ""

class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Enable or disable the postcard telemetry feature'
   }

   @staticmethod
   def handler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = True

class PostcardTelemetryInterface( CliCommand.CliCommandClass ):
   syntax = '''telemetry postcard policy [ profile PROFILE ]'''
   noOrDefaultSyntax = syntax
   data = {
      'postcard': CliCommand.guardedKeyword( 'postcard',
                                             helpdesc='Apply postcard variant',
                                             guard=postcardTelemetryGuard ),
      'telemetry': 'Apply telemetry feature',
      'policy' : 'Apply postcard telemetry profile of type match policy',
      'profile': 'Apply postcard telemetry profile',
      'PROFILE': profileNameMatcher
      }

   @staticmethod
   def handler( mode, args ):
      profileName = args.get( 'PROFILE', 'default' )
      intfName = mode.intf.name
      intfToPortProfile = postcardTelemetryConfig.intfToPortProfile
      if intfToPortProfile.get( intfName ) == profileName:
         return
      del postcardTelemetryConfig.intfToPortProfile[ intfName ]
      intfToPortProfile[ intfName ] = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE' )
      intfName = mode.intf.name
      intfToPortProfile = postcardTelemetryConfig.intfToPortProfile
      if intfName in intfToPortProfile:
         if not profileName or ( profileName and
                                 intfToPortProfile.get( intfName ) == profileName ):
            del intfToPortProfile[ intfName ]

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
      'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

class PostcardProtocolConfigCmd( ProtocolMixin ):
   syntax = '''protocol TCP_UDP
               ( ( source port [ SPORT [ destination port DPORT ] ] )
               | ( destination port [ DPORT ] ) )'''
   noOrDefaultSyntax = syntax

   data = {
      'protocol': 'Protocol',
      'TCP_UDP': tcpUdpProtoExpr,
      'port': portKwNode,
      'source': 'Source',
      'SPORT': portExpression( 'SPORT' ),
      'destination': 'Destination',
      'DPORT': portExpression( 'DPORT' )
   }

   @classmethod
   def handler( cls, mode, args ):
      proto = ( args.get( cls._tcpUdpArgsListName ) )
      assert proto, args
      source = 'source' in args
      destination = 'destination' in args
      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, add=True )
      cls._maybeHandleErrors( mode, args, proto, source, destination )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      proto = args.get( cls._tcpUdpArgsListName )
      source = 'source' in args
      destination = 'destination' in args
      if not proto and not source and not destination:
         # 'no protocol' removes all protocols
         proto = True
      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, add=False )

class PostcardProtocolConfigWithoutFlagsCmd( ProtocolOredBase ):
   syntax = '''protocol ( TCP_UDP | FLAGS_EXPR )
           [ ( source port SPORT [ destination port ( DPORT | all ) ] )
           | ( [ source port all ] destination port DPORT ) ]'''

   noOrDefaultSyntax = '''protocol [ ( TCP_UDP | FLAGS_EXPR )
             [ ( source port [ SPORT [ destination port ( DPORT | all ) ] ] )
             | ( source port all destination port DPORT )
             | ( destination port [ DPORT ] ) ] ]'''
   data = {
      'FLAGS_EXPR': generateTcpFlagExpression( tcpFlagsSupported=False )
   }
   data.update( ProtocolOredBase._baseData )

# ----------------------------------------------------------------------------------
# The "protocol tcp flags TCP_FLAGS"
# ----------------------------------------------------------------------------------
class ProtocolTcpFlagsOnlyCmd( ProtocolMixin ):
   syntax = ( 'protocol FLAGS_EXPR' )
   noOrDefaultSyntax = syntax

   data = {
      'FLAGS_EXPR': generateTcpFlagExpression(
         tcpFlagsSupported=True,
         notAllowed=True,
         guard=postcardTelemetryTcpControlGuard ),
      'protocol': 'Protocol',
   }

   @classmethod
   def handler( cls, mode, args ):
      proto = args.get( cls._tcpFlagArgsListName )
      if not proto and not args:
         return
      cls._updateProtoAndPort( mode, args, proto, flags=True, add=True )
      cls._maybeHandleErrors( mode, args, proto )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      proto = args.get( cls._tcpFlagArgsListName )
      cls._updateProtoAndPort( mode, args, proto, flags=True, add=False )


#------------------------------------------------------------------------------------
# "show monitor telemetry postcard policy profile [ NAME ]"
#------------------------------------------------------------------------------------

def populateIntfStatusByMatchOption( intf,
                                     intfStatus,
                                     matchOption,
                                     profileNameToAppliedIntf,
                                     mapErrStringToFailedIntfList ):
   installedStatus = intfStatus.installed.get( matchOption, None )
   if installedStatus is not None:
      if installedStatus == tacInstalledStatus( 'success', '' ):
         profileNameToAppliedIntf[ matchOption ].append( intf )
      else:
         # Failed intf case
         error = installedStatus.error
         mapErrStringToFailedIntfList[ matchOption ][ error ].append( intf )

def populateAppliedAndFailedIntfOnProfile( profileName,
                                           intf,
                                           profileNameToAppliedIntf,
                                           mapErrStringToFailedIntfList ):
   profile = postcardTelemetryConfig.portProfile.get( profileName, None )
   if profile is None:
      return
   samplePolicy = profile.samplePolicy
   if not samplePolicy:
      return
   if len( policiesStatus ) != 1:
      # If PD has published any status, we should see exactly
      # one Tac::Dir
      return
   pdStatusPath = next( iter( policiesStatus ), None )
   if pdStatusPath is None:
      return

   pdStatus = policiesStatus.get( pdStatusPath, None )
   if pdStatus is None:
      return

   policyStatus = pdStatus.status.get( samplePolicy, None )
   if policyStatus is not None:
      intfStatus = policyStatus.intfStatus.get( intf, None )
      if intfStatus is not None:
         populateIntfStatusByMatchOption( intf, intfStatus, 'matchIpAccessGroup',
            profileNameToAppliedIntf, mapErrStringToFailedIntfList )

         populateIntfStatusByMatchOption( intf, intfStatus, 'matchIpv6AccessGroup',
            profileNameToAppliedIntf, mapErrStringToFailedIntfList )
   return

def nameInterfaceLinker( profileName ):
   profileNameToIntf = []
   profileNameToAppliedIntf = defaultdict( list )
   mapErrStringToFailedIntfList = defaultdict( lambda: defaultdict( list ) )
   for intf, profName in \
     sorted( postcardTelemetryConfig.intfToPortProfile.items() ):
      if profName == profileName:
         profileNameToIntf.append( intf )
         populateAppliedAndFailedIntfOnProfile( profileName,
               intf, profileNameToAppliedIntf, mapErrStringToFailedIntfList )

   return profileNameToIntf, profileNameToAppliedIntf, mapErrStringToFailedIntfList

def getProfiles( profileName=None ):
   profiles = {}
   profileNameToIntf = {}
   if not profileName:
      profileList = postcardTelemetryConfig.portProfile
   else:
      profileList = [ profileName ]

   for name in profileList:
      profileNameToIntf, profileNameToAppliedIntf, mapErrStringToFailedIntfList = \
                                    nameInterfaceLinker( name )
      samplePolicy = None

      for matchOption in mapErrStringToFailedIntfList:
         for error in mapErrStringToFailedIntfList[ matchOption ]:
            errIntfList = mapErrStringToFailedIntfList[ matchOption ][ error ]
            mapErrStringToFailedIntfList[ matchOption ][ error ] =\
               IntfList( interfaces=errIntfList )

      appliedInterfaces = profileNameToAppliedIntf[ "matchIpAccessGroup" ]
      failures = dict( mapErrStringToFailedIntfList[ "matchIpAccessGroup" ] )
      appliedInterfaces6 = profileNameToAppliedIntf[ "matchIpv6AccessGroup" ]
      failures6 = dict( mapErrStringToFailedIntfList[ "matchIpv6AccessGroup" ] )

      profileConfig = postcardTelemetryConfig.portProfile.get( name, None )
      if profileConfig is not None:
         samplePolicy = profileConfig.samplePolicy
      profiles[ name ] = ProfileModel( interfaces=profileNameToIntf,
                                       appliedInterfaces=appliedInterfaces,
                                       samplePolicy=samplePolicy,
                                       failures=failures,
                                       appliedInterfaces6=appliedInterfaces6,
                                       failures6=failures6 )
   return profiles

def showMonitorTelemetryPostcardPolicyProfile( mode, args ):
   profileName = args.get( 'NAME' )
   profiles = getProfiles( profileName )
   return ModelPostcardProfile( profiles=profiles )

class ShowMonitorTelemetryPostcardPolicyProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard policy profile [ NAME ]'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=postcardTelemetryGuard ),
         'postcard': 'Show monitor postcard telemetry',
         'policy' : 'Show monitor postcard telemetry of type policy',
         'profile': 'Show monitor postcard telemetry profile',
         'NAME': profileNameMatcher,
         }
   cliModel = ModelPostcardProfile
   handler = showMonitorTelemetryPostcardPolicyProfile
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry postcard sample policy" command
#------------------------------------------------------------------------------------
def getPolicyModel( policyName ):
   rules = []
   tcpFlags = {}
   if policyName not in policiesCliConfig.pmapType.pmapInput:
      return SamplePolicyModel( rules=rules )
   pmapConfig = policiesCliConfig.pmapType.pmapInput[ policyName ]
   if not pmapConfig.currCfg:
      return SamplePolicyModel( rules=rules )
   ruleNames = list( pmapConfig.currCfg.classPrio.values() )
   for ruleName in ruleNames:
      protocols = []
      rule = pmapConfig.currCfg.rawClassMap[ ruleName ]
      match = list( rule.match.items() )[ 0 ][ 1 ]
      matchOpt = "ipv4" if match.option == 'matchIpAccessGroup' else "ipv6"
      for protoField in match.structuredFilter.proto.values():
         protocolRange = NumericalRange( low=protoField.protocolRange.rangeStart,
                                         high=protoField.protocolRange.rangeEnd )
         srcPorts = []
         destPorts = []
         if not toggleFeaturePostcardTelemetryTcpControlEnabled():
            assert len( protoField.port ) <= 1
         for portField in protoField.port.values():
            srcPorts = [ NumericalRange( low=sport.rangeStart,
                                         high=sport.rangeEnd )
                         for sport in portField.sport ]
            destPorts = [ NumericalRange( low=dport.rangeStart,
                                          high=dport.rangeEnd )
                          for dport in portField.dport ]
            ports = [ Port( srcPorts=srcPorts, destPorts=destPorts ) ]
            protocols.append( Protocol( protocolRange=protocolRange, ports=ports ) )
         if toggleFeaturePostcardTelemetryTcpControlEnabled() and \
            protoField.tcpFlagAndMask:
            tcpFlagAndMaskSet = protoField.tcpFlagAndMask
            tcpFlagStr = tcpFlagAndMaskStrings( sorted( tcpFlagAndMaskSet ) )
            t0( '%s' % tcpFlagStr )
            namesStr = ''
            for name in tcpFlagStr:
               addNewLine = False
               if name != tcpFlagStr[ -1 ]:
                  addNewLine = True
               name = name.replace( " flags ", "" )
               namesStr += name
               if addNewLine:
                  namesStr += '\n\t\t'

            t0( '%s' % namesStr )
            tcpFlags[ ruleName ] = namesStr

      policyActions = pmapConfig.currCfg.classAction[ ruleName ].policyAction
      actions = Actions( sample='sample' in policyActions,
                         sampleAll='sampleAll' in policyActions )
      rules.append( Rule( ruleString=ruleName, matchOption=matchOpt, matches=Matches(
         srcPrefixes=list( match.structuredFilter.source ),
         destPrefixes=list( match.structuredFilter.destination ),
         protocols=protocols ), actions=actions ) )
   return SamplePolicyModel( rules=rules, tcpFlags=tcpFlags )

def getPolicies( policyName=None ):
   config = {}
   if policyName:
      config[ policyName ] = getPolicyModel( policyName )
   else:
      for name in policiesCliConfig.pmapType.pmapInput:
         config[ name ] = getPolicyModel( name )
   return config

def showSamplePolicyPostcardHandler( mode, args ):
   policyName = args.get( 'POLICY_NAME' )
   policies = getPolicies( policyName )
   return SamplePoliciesModel( policies=policies )

class ShowSamplePostcardPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard sample policy [ POLICY_NAME ]'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=postcardTelemetryGuard ),
            'postcard': 'Show monitor postcard telemetry',
            'sample': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'sample', helpdesc='Show sampling information' ),
               guard=postcardTelemetryGuard ),
            'policy': 'Show sample policy information',
            'POLICY_NAME': samplePolicyNameMatcher,
   }
   cliModel = SamplePoliciesModel
   handler = showSamplePolicyPostcardHandler
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry postcard" command
#------------------------------------------------------------------------------------

def showPostcardTelemetryHandler( mode, args ):
   enabled = postcardTelemetryConfig.enable
   sampleRate = postcardTelemetryConfig.sampleRate
   # if sample rate is invalid, compute sample rate
   # from sample data and mask if exists
   if sampleRate == tacSampleRate.sampleRateInvalid:
      if postcardTelemetryConfig.sampleDataAndMask and \
         postcardTelemetryConfig.sampleDataAndMask.mask:
         mask = postcardTelemetryConfig.sampleDataAndMask.mask
         invertedMask = mask ^ 0xffff
         if mask == 0xffff:
            sampleRate = tacSampleRate.sampleRate1in64k
         elif ( invertedMask & ( invertedMask - 1 ) ) == 0:
            sampleRate = tacSampleRate.sampleRate1in32k
         else:
            sampleRate = tacSampleRate.sampleRate1in16k

   if sampleRate == tacSampleRate.sampleRate1in16k:
      sampleRate = 16384
   elif sampleRate in [ tacSampleRate.sampleRate1in32k,
                        tacSampleRate.sampleRateInvalid ]:
      sampleRate = 32768
   elif sampleRate == tacSampleRate.sampleRate1in64k:
      sampleRate = 65536

   sampleTcpUdpChecksum = None
   if postcardTelemetryConfig.sampleDataAndMask and \
      postcardTelemetryConfig.sampleDataAndMask.mask:
      sampleTcpUdpChecksum = SampleTcpUdpChecksum(
         value=postcardTelemetryConfig.sampleDataAndMask.data,
         mask=postcardTelemetryConfig.sampleDataAndMask.mask )

   vxlanEnabled = False
   vxlanMarkerBit = None
   if hwCapability.postcardTelemetryVxlanSupported:
      vxlanConfig = postcardTelemetryConfig.vxlanConfig
      if vxlanConfig:
         vxlanEnabled = vxlanConfig.vxlanEnabled
         vxlanMarkerBit = vxlanConfig.vxlanMarkerBit
   destinationConfig = postcardTelemetryConfig.destinationConfig
   collection = None
   srcIp = dstIp = None
   if not isDefaultDestination( destinationConfig ):
      if destinationConfig.srcIp.af == 'ipv4':
         srcIp = destinationConfig.srcIp.v4Addr
      elif destinationConfig.srcIp.af == 'ipv6':
         srcIp = destinationConfig.srcIp.v6Addr.stringValue
      if destinationConfig.dstIp.af == 'ipv4':
         dstIp = destinationConfig.dstIp.v4Addr
      elif destinationConfig.dstIp.af == 'ipv6':
         dstIp = destinationConfig.dstIp.v6Addr.stringValue
      version = destinationConfig.subTypeVersion
      collection = Collection( srcIp=srcIp, dstIp=dstIp, version=version )

   return PostcardTelemetry( enabled=enabled, sampleRate=sampleRate,
                             sampleTcpUdpChecksum=sampleTcpUdpChecksum,
                             vxlanEnabled=vxlanEnabled,
                             vxlanMarkerBit=vxlanMarkerBit,
                             collection=collection )

class ShowPostcardTelemetryPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard policy'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=postcardTelemetryGuard ),
         'postcard': 'Show monitor postcard telemetry',
         'policy' : 'Show monitor postcard telemetry of type policy',
         }
   cliModel = PostcardTelemetry
   handler = showPostcardTelemetryHandler
   privileged = True

def showPostcardTelemetryCountersHandler( mode, args ):
   ret = PostcardTelemetryCounters()
   mountPath = 'postcardTelemetry/counters'
   entityType = 'Smash::PostcardTelemetry::PostcardTelemetryCounters'
   shmemEm = SharedMem.entityManager( sysdbEm=mode.entityManager )
   countersColl = shmemEm.doMount( mountPath, entityType,
                                   Smash.mountInfo( 'reader' ) )
   key = CounterKey( 0 )
   counters = countersColl.postcardPktCounters.get( key )
   if not counters:
      counters = PostcardPktCountersEntry()
   ret.sampleRcvd = counters.sampleRcvdCount
   ret.sampleDiscarded = counters.sampleDiscardedCount
   ret.multiDstSampleRcvd = counters.multiDstSampleRcvdCount
   ret.sampleSent = counters.sampleSentCount
   ret.grePktSent = counters.grePktCount
   return ret

class ShowPostcardTelemetryCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard counters'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.guardedKeyword(
            'telemetry', helpdesc='Show monitor telemetry information',
            guard=postcardTelemetryGuard ),
         'postcard': 'Show monitor postcard telemetry',
         'counters': 'Show monitor postcard telemetry samples count',
         }
   cliModel = PostcardTelemetryCounters
   handler = showPostcardTelemetryCountersHandler
   privileged = True

class ClearPostcardTelemetryCounters( CliCommand.CliCommandClass ):
   syntax = 'clear monitor telemetry postcard counters'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'monitor': CliToken.Monitor.monitorMatcherForClear,
         'telemetry': CliCommand.guardedKeyword(
            'telemetry', helpdesc='Clear monitor telemetry information',
            guard=postcardTelemetryGuard ),
         'postcard': 'Clear monitor postcard telemetry',
         'counters': 'Clear monitor postcard telemetry samples count',
         }

   @staticmethod
   def handler( mode, args ):
      if not postcardTelemetryConfig.enable:
         return
      cmd = "clearPostcardTelemetryCounters"
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            "PostcardTelemetry",
                                            "PostcardTelemetry",
                                            cmd, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearPostcardTelemetryCounters )
BasicCli.addShowCommandClass( ShowMonitorTelemetryPostcardPolicyProfile )
BasicCli.addShowCommandClass( ShowPostcardTelemetryCounters )
BasicCli.addShowCommandClass( ShowSamplePostcardPolicy )
BasicCli.addShowCommandClass( ShowPostcardTelemetryPolicy )
BasicCli.GlobalConfigMode.addCommandClass( MonitorTelemetryPostcard )
PostcardPolicyConfigMode.addCommandClass( DisabledCmd )
PostcardPolicyConfigMode.addCommandClass( GreDestinationCmd )
PostcardPolicyConfigMode.addCommandClass( ProfileConfigCmd )
PostcardPolicyConfigMode.addCommandClass( SampleRateCmd )
PostcardPolicyConfigMode.addCommandClass( VxlanMarkerCmd )
PostcardPolicyConfigMode.addCommandClass( PostcardSamplePolicyConfigCmd )
PostcardSamplePolicyConfigMode.addCommandClass(
   PostcardSamplePolicyMatchRuleConfigCmd )
PostcardSamplePolicyConfigMode.addModelet( CommitAbortModelet )
PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
   PostcardSamplePolicyActionsConfigCmd )
PostcardSamplePolicyActionRuleConfigMode.addCommandClass( SampleActionCmd )
PostcardSamplePolicyMatchRuleIpv4ConfigMode.addModelet( CommitAbortModelet )
if toggleFeaturePostcardTelemetryTcpControlEnabled():
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
      PostcardProtocolConfigWithoutFlagsCmd )
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
      ProtocolTcpFlagsOnlyCmd )
else:
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
   PostcardProtocolConfigCmd )
PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass( PrefixIpv4SingletonCmd )
PostcardSamplePolicyMatchRuleIpv6ConfigMode.addCommandClass(
   PostcardSamplePolicyActionsConfigCmd )
PostcardSamplePolicyMatchRuleIpv6ConfigMode.addModelet( CommitAbortModelet )
if toggleFeaturePostcardTelemetryTcpControlEnabled():
   PostcardSamplePolicyMatchRuleIpv6ConfigMode.addCommandClass(
      PostcardProtocolConfigWithoutFlagsCmd )
   PostcardSamplePolicyMatchRuleIpv6ConfigMode.addCommandClass(
      ProtocolTcpFlagsOnlyCmd )
else:
   PostcardSamplePolicyMatchRuleIpv6ConfigMode.addCommandClass(
      PostcardProtocolConfigCmd )
PostcardSamplePolicyMatchRuleIpv6ConfigMode.addCommandClass( PrefixIpv6SingletonCmd )
PostcardSamplePolicyActionRuleConfigMode.addModelet( CommitAbortModelet )
IntfPostcardTelemetryModelet.addCommandClass( PostcardTelemetryInterface )
ProfileConfigMode.addCommandClass( SamplePolicyCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
# Clean up all the configuration set by cmds in interface config mode.
#-------------------------------------------------------------------------------
class PostcardTelemetryIntfJanitor( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      if intfName in postcardTelemetryConfig.intfToPortProfile:
         del postcardTelemetryConfig.intfToPortProfile[ intfName ]

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin( provides=( "PostcardTelemetryPolicyCli", ) )
def Plugin( entityManager ):
   global postcardTelemetryConfig, postcardTelemetryStatus, allVrfStatus
   global policiesCliConfig, policiesStatus, policiesStatusRequestDir
   global hwCapability
   postcardTelemetryConfig = ConfigMount.mount( entityManager,
                                                "postcardtelemetry/policyconfig",
                                                "PostcardTelemetry::PolicyConfig",
                                                "w" )
   postcardTelemetryStatus = LazyMount.mount( entityManager,
                                              "postcardtelemetry/status",
                                              "PostcardTelemetry::Status", "r" )
   allVrfStatus = LazyMount.mount( entityManager, "ip/vrf/status/global",
                                   "Ip::AllVrfStatusGlobal", "r" )
   hwCapability = LazyMount.mount( entityManager, "postcardtelemetry/hwCapability",
                                   "PostcardTelemetry::HwCapability", "r" )
   samplePoliciesConfigPath = 'postcardtelemetry/samplePolicies/input/cli'
   samplePoliciesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusPath = 'cell/%d/postcardtelemetry/samplePolicies/status' % \
                        Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatusRequestDirPath = (
      'postcardtelemetry/samplePolicies/statusRequest/cli' )
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesStatusPath, policiesStatusType, 'ri' )
   mountGroup.close( callback=None, blocking=False )

   policiesCliConfig = ConfigMount.mount( entityManager, samplePoliciesConfigPath,
                                          samplePoliciesCliConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType, 'wc' )

   CliPlugin.IntfCli.Intf.registerDependentClass( PostcardTelemetryIntfJanitor )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-10-14 15:00:00',
   cmds=[ 'show monitor telemetry postcard policy',
          'show monitor telemetry postcard policy profile',
          'show monitor telemetry postcard sample policy',
   ],
   cmdsGuard=lambda: hwCapability.postcardTelemetrySupported )
