# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import datetime
import json
import re
import ArPyUtils
import ConfigMount
import EntityMib
import LazyMount
import Tac
from CliPlugin import EnvironmentCli
from CliPlugin import TechSupportCli
from CliPlugin.EnvironmentModels import SystemEnvAllExtra
import Toggles.PowerToggleLib as PowerToggles
from CliModel import Bool, Dict, Enum, Int, Float, Model, Str, Submodel
from TableOutput import Format, createTable
from operator import attrgetter

MAX_POWER_BUDGET = 100000
DEFAULT_POWER_CONTROLLER_POLL_INTERVAL = 60

cliConfig = None
powerSupplyDir = None
voltageSensorCellDir = None
voltageSensorSystemDir = None
ecbCellDir = None
ecbSystemDir = None
currentSensorCellDir = None
currentSensorSystemDir = None
hardwareDir = None
powerConfig = None
archerCoolingStatus = None
envTempStatusDir = None
entityMib = None
genericStatus = None
powerControllerCliConfig = None

def makeTableBase( header ):
   fl = Format( justify="left" )
   fl.noPadLeftIs( True )
   fl.padLimitIs( True )
   fr = Format( justify="right" )
   fr.noPadLeftIs( True )
   fr.padLimitIs( True )
   return createTable( header ), fl, fr

def makeTable( header ):
   t, fl, fr = makeTableBase( header )
   t.formatColumns( fl, fl, fr, fr, fr, fr, fl, fr )
   return t

def makeBlackBoxTable( header, fieldCount ):
   _, fl, fr = makeTableBase( header )
   t = createTable( header, indent=len( _indentStr( 1 ) ) )
   columns = [ fl ] + ( [ fr ] * fieldCount )
   t.formatColumns( *columns )
   return t

def makeVoltageSensorsTable( header ):
   t, fl, fr = makeTableBase( header )
   t.formatColumns( fl, fl, fr, fr, fl )
   return t

def makeCurrentSensorsTable( header ):
   t, fl, fr = makeTableBase( header )
   t.formatColumns( fl, fr, fl )
   return t

def printTable( table ):
   # Strip whitespace from the end of the lines, because that's how it was
   # before and it's what some parsers expect. Also remove the last newline.
   print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ : -1 ] ) ) )

def printThreshold( threshold, totalCapacity, totalOutputPower ):
   if not totalCapacity:
      totalCapacity = float( 0 )
   if not totalOutputPower:
      totalOutputPower = float( 0 )
   powerToggle = PowerToggles.togglePowerThresholdEnabled()
   if powerToggle:
      if threshold == float( "inf" ):
         print( "Warning threshold is not set, using all {}W of total power".format(
                totalCapacity ) )
      else:
         print( "Warning threshold is {}W out of {}W of total power".format(
                threshold, totalCapacity ) )
      print( "Current power consumption is {}W".format(
             totalOutputPower ) )
      print()

def formatU16( x ):
   return f'0x{x:04x}'

def formatU8( x ):
   return f'0x{x:02x}'

TechSupportCli.registerShowTechSupportCmd(
   '2012-01-31 19:14:25',
   cmds=[ 'show system environment power detail' ] )

powerSpecificInfoFn = {}
def registerPowerSpecificInfoFn( typeName, fn ):
   assert typeName not in powerSpecificInfoFn
   powerSpecificInfoFn[ typeName ] = fn

#--------------------------------------------------
# 'show system environment power'
#
# [legacy]
# 'show environment power'
#--------------------------------------------------

def _indentStr( indentLevel ):
   return '   ' * indentLevel

def _tempSensorString( name, model, state, temp ):
   if temp is None:
      return 'Unknown'

   # See BUG11257
   if model == 'PWR-2900AC' and state == 'powerLoss' and \
      re.match( 'TempSensorP[1-4]/1', name ):
      return 'Unknown'
   return '%.0fC' % temp

def _voltageString( voltage, managed=True ):
   if not managed or voltage is None:
      return 'N/A'
   return '%0.2fV' % voltage

def _currentString( current, managed=True ):
   if not managed or current is None:
      return 'N/A'
   return '%0.2fA' % current

def _powerString( power, precision=1, managed=True ):
   if not managed or power is None:
      return 'N/A'
   fmtStr = '%%0.%dfW' % precision
   return fmtStr % power

def _statusString( status ):
   if status is None:
      return 'Unknown'
   return re.sub( '([A-Z][a-z]+)', r' \1', status ).title()

def _systemInitialized():
   return entityMib.root and entityMib.root.initStatus == 'ok'

def _isStandbyMode( mode ):
   return ( mode.session_ and
         mode.session_.entityManager_.redundancyStatus().mode == 'standby' )

def _getAllSlots( mode ):
   slots = {}
   entityMibRoot = entityMib.root
   if _systemInitialized():
      return { f"{slot.tag}{slot.label}": slot
               for slot in entityMibRoot.cardSlot.values() }
   return slots

def _renderVoltageSensors( sensors ):
   headers = ( "\nSensor", "\nDescription", "Measured\nVoltage",
               "Expected\nVoltage", "\nStatus" )
   table = makeVoltageSensorsTable( headers )
   for sensorId in sorted( sensors, key=int ):
      sensor = sensors[ sensorId ]
      table.newRow( sensorId,
                    sensor.description,
                   _voltageString( sensor.voltage ),
                   _voltageString( sensor.expectedVoltage ),
                   _statusString( sensor.status ) )
   # The last symbol of each table is `\n`, we need to remove it to avoid creating
   # two white lines between blocks
   print( table.output()[ :-1 ] )

def _renderCurrentSensors( sensors ):
   headers = ( "\nName", "Measured\nCurrent", "\nStatus" )
   table = makeCurrentSensorsTable( headers )
   for sensor in sorted( sensors.values(), key=attrgetter( 'name' ) ):
      table.newRow( sensor.name,
                   _currentString( sensor.current ),
                   _statusString( sensor.status ) )
   # The last symbol of each table is `\n`, we need to remove it to avoid creating
   # two white lines between blocks
   print( table.output()[ :-1 ] )

def _getVoltageSensor( name ):
   """
   Voltage sensors exist under the following paths:
   - environment/archer/power/status/voltageSensor/system (linecard, fabric, PSU)
   - environment/archer/power/status/voltageSensor/cell/%cellId (supervisor,
     cpuCard, switchCard)
   """
   return ( voltageSensorSystemDir.get( name ) or
            next( ( cell[ name ] for cell in voltageSensorCellDir.values()
                    if name in cell ), None ) )

def _getSlotByName( name ):
   modelCardSlot = FruSlot()
   modelCardSlot.entPhysicalClass = ''
   modelCardSlot.label = name

   # Parse the card's name (such as "Linecard5") into a tag and label
   # ("Linecard" for the tag and "5" for the label in this example).
   tagList = [ 'Supervisor', 'Linecard', 'Fabric', 'SwitchcardCes' ]
   for tag in tagList:
      if name.startswith( tag ):
         modelCardSlot.entPhysicalClass = tag
         modelCardSlot.label = name[ len( tag ) : ]
         break

   return modelCardSlot

def _isManaged( psEnvStatus ):
   return bool( psEnvStatus.inputVoltageSensor or
                psEnvStatus.outputVoltageSensor or
                psEnvStatus.inputCurrentSensor or
                psEnvStatus.outputCurrentSensor )

class Fan( Model ):
   status = Enum( values=( 'failed', 'ok', 'unknownHwStatus' ),
                  help='Fan status' )
   speed = Int( help='Fan speed (percentage)' )

class TempSensor( Model ):
   status = Enum( values=( 'disabled', 'failed', 'invalidReading',
                           'ok', 'unknownHwStatus' ),
                  help='Temp sensor status' )
   temperature = Float( help='Fan temperature (celsius)', optional=True )

class VoltageSensor( Model ):
   status = Enum( values=( 'failed', 'ok', 'unknownHwStatus' ),
                  help='Voltage sensor status' )
   description = Str( help='Hardware label describing the voltage sensor',
                      optional=True )
   voltage = Float( help='Measured sensor voltage (volts)', optional=True )
   expectedVoltage = Float( help='Expected sensor voltage (volts)', optional=True )

class CurrentSensor( Model ):
   status = Enum( values=( 'failed', 'ok', 'offline', 'unknownHwStatus' ),
                  help='Current sensor status' )
   name = Str( help='The current sensor\'s hardware label',
               optional=True )
   current = Float( help='Measured sensor current (amps)', optional=True )

class FruSlot( Model ):
   """Information for a generic fru slot in the system"""
   entPhysicalClass = Str( help="Physical entity class of the slot" )
   label = Str( help="Label of the slot" )
   voltageSensors = Dict( keyType=str, valueType=VoltageSensor,
                          help="A mapping of labels to voltage sensors" )
   currentSensors = Dict( keyType=str, valueType=CurrentSensor,
                          help="A mapping of labels to current sensors",
                          optional=True )
   def render( self ):
      if self.voltageSensors:
         _renderVoltageSensors( self.voltageSensors )
      if self.currentSensors:
         _renderCurrentSensors( self.currentSensors )

class VoltageSensors( Model ):
   __revision__ = 2
   voltageSensors = Dict( keyType=str, valueType=VoltageSensor,
                          help="A mapping of labels to voltage sensors" )
   cardSlots = Dict( keyType=str, valueType=FruSlot,
                     help="A mapping of labels to card slots" )
   powerSupplySlots = Dict( keyType=str, valueType=FruSlot,
                            help="A mapping of labels to power supply slots" )

   def render( self ):

      if self.voltageSensors:
         _renderVoltageSensors( self.voltageSensors )

      def cardSlotKey( x ):
         priorityList = [ 'Supervisor', 'Linecard', 'Fabric' ]
         priority = 0
         for prefix in priorityList:
            if x.startswith( prefix ):
               break
            priority += 1
         return [ priority ] + ArPyUtils.naturalOrderKey( x )

      for name in sorted( self.cardSlots, key=cardSlotKey ):
         slot = self.cardSlots[ name ]
         print( f'\n{slot.entPhysicalClass} {slot.label}:\n' )
         slot.render()

      for name in ArPyUtils.naturalsorted( self.powerSupplySlots ):
         slot = self.powerSupplySlots[ name ]
         print( f'\n{slot.entPhysicalClass} {slot.label}:\n' )
         slot.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for slotName, slot in ( list( dictRepr[ 'cardSlots' ].items() ) +
                                 list( dictRepr[ 'powerSupplySlots' ].items() ) ):
            for sensorId, sensor in slot[ 'voltageSensors' ].items():
               sensorName = f'{slotName}/{sensorId}'
               dictRepr[ 'voltageSensors' ][ sensorName ] = sensor
         del dictRepr[ 'cardSlots' ]
         del dictRepr[ 'powerSupplySlots' ]
      return dictRepr

class CurrentSensors( Model ):
   currentSensors = Dict( keyType=str, valueType=CurrentSensor,
                          help="A mapping of labels to current sensors" )
   
   cardSlots = Dict( keyType=str, valueType=FruSlot,
                     help="A mapping of labels to card slots" )

   def render( self ):

      if self.currentSensors:
         _renderCurrentSensors( self.currentSensors )

      def cardSlotKey( x ):
         priorityList = [ 'Supervisor', 'Linecard', 'Fabric' ]
         priority = 0
         for priority, prefix in enumerate( priorityList ):
            if x.startswith( prefix ):
               break
         return [ priority ] + ArPyUtils.naturalOrderKey( x )
 
      for name in sorted( self.cardSlots, key=cardSlotKey ):
         slot = self.cardSlots[ name ]
         print( f'\n{slot.entPhysicalClass} {slot.label}:' )
         slot.render()

class PowerSupply( Model ):
   modelName = Str( help='Type of power supply', )
   capacity = Float( help='Capacity (watts)', optional=True )
   dominant = Bool( help='dominant power supply', optional=True )
   inputCurrent = Float( help='Input current (amps)', optional=True )
   outputCurrent = Float( help='Output current (amps)', optional=True )
   outputBCurrent = Float( help='Output B current (amps)', optional=True )
   inputVoltage = Float( help='Input voltage (volts)', optional=True )
   outputVoltage = Float( help='Output voltage (volts)', optional=True )
   outputBVoltage = Float( help='Output B voltage (volts)', optional=True )
   inputBVoltage = Float( help='Input B voltage (volts)', optional=True )
   inputBCurrent = Float( help='Input B current (amps)', optional=True )
   inputPower = Float( help='Input power (watts)', optional=True )
   outputPower = Float( help='Output power (watts)', optional=True )
   state = Enum( values=( 'ok', 'unknown', 'powerLoss', 'failed' ),
                 help='State of the power supply', optional=True )
   uptime = Float( help="Timestamp in UTC (in seconds) when the last change"
                        " in PowerSupply's state occurred",
                   optional=True )
   fans = Dict( keyType=str, valueType=Fan,
                help='Power supply fans' )
   tempSensors = Dict( keyType=str, valueType=TempSensor,
                   help='Power supply temperature sensors' )
   blackBoxLog = Str( help='Debugging log', optional=True )
   specificInfo = Str( help='Power supply model-specific information',
                       optional=True )
   details = Bool( help='Details available', optional=True )
   managed = Bool( default=True, help='Power supply is managed', optional=True )
   mfrId = Str( help='Manufacturer ID of power supply', optional=True )
   mfrModel = Str( help='Manufacturer model of power supply', optional=True )
   mfrRevision = Str( help='Manufacturer revision of power supply', optional=True )
   priFirmware = Str( help='Primary firmware of power supply', optional=True )
   secFirmware = Str( help='Secondary firmware of power supply', optional=True )

   def status( self ):
      status = 'Ok'
      if self.state == 'unknown':
         status = 'Unknown'
      elif self.state == 'powerLoss':
         status = 'Power Loss'
      elif self.state == 'failed':
         status = 'Failed'
      elif [ x for x in self.fans.values() if x.status == 'failed' ]:
         status = 'Fan Failure'
      elif [ x for x in self.tempSensors.values()
             if x.status in [ 'failed', 'invalidReading' ] ]:
         status = 'Sensor Failure'
      return status

class PowerSupplies( SystemEnvAllExtra ):
   powerSupplies = Dict( keyType=str, valueType=PowerSupply,
                         help='Power supplies status' )

   class Details( Model ):
      pollInterval = Float( help='Polling interval (seconds)' )

   details = Submodel( valueType=Details,
                       help='Details about the power supplies',
                       optional=True )

   def render( self ):
      if self.powerSupplies: # pylint: disable=too-many-nested-blocks
         dualInput = any( ps.inputBCurrent is not None
                          for ps in self.powerSupplies.values() )
         dualOutput = any( ps.outputBCurrent is not None
                           for ps in self.powerSupplies.values() )
         assert not ( dualInput and dualOutput )
         singleInputOutput = not dualInput and not dualOutput

         totalCapacity = 0
         dominantPsu = False
         dominantCapacity = 0
         totalOutputPower = 0
         sortedPsuLabels = ArPyUtils.naturalsorted( self.powerSupplies )

         if singleInputOutput:
            tableColumns = [ [ "Power", [ "Supply" ], ],
                             [ "", [ "Model" ], ],
                             [ "", [ "Capacity" ], ],
                             [ "Input", [ "Current" ] ],
                             [ "Output", [ "Current" ], ],
                             [ "Output", [ "Power" ], ],
                             [ "", [ "Status" ], ],
                             [ "", [ "Uptime" ], ] ]
            outputTable = makeTable( tableColumns )

            for label in sortedPsuLabels:
               ps = self.powerSupplies[ label ]
               # We should actually check all attributes used below, but we are
               # guaranteed that they are set if 'state' is set for now.
               uptime = "Offline"
               if ps.state is not None:
                  if ps.state == 'ok':
                     upt = Tac.utcNow() - ps.uptime
                     td = datetime.timedelta( seconds=int( upt ) )
                     uptime = str( td )
                  if ps.capacity is not None and totalCapacity is not None:
                     if ps.state == 'ok':
                        totalCapacity += ps.capacity
                        if ps.dominant:
                           dominantPsu = True
                           dominantCapacity = ps.capacity
                  else:
                     totalCapacity = None
                  if ps.outputPower is not None and totalOutputPower is not None:
                     totalOutputPower += ps.outputPower
                  else:
                     totalOutputPower = None
                  outputTable.newRow( label, ps.modelName,
                     _powerString( ps.capacity, precision=0 ),
                     _currentString( ps.inputCurrent, managed=ps.managed ),
                     _currentString( ps.outputCurrent, managed=ps.managed ),
                     _powerString( ps.outputPower, managed=ps.managed ),
                     ps.status(), uptime )
            if dominantPsu:
               totalCapacity = dominantCapacity
            outputTable.newRow( "Total", '--',
                  _powerString( totalCapacity, precision=0 ),
                  '--', '--', _powerString( totalOutputPower ), '--', '--' )
            printThreshold( cliConfig.powerThreshold, totalCapacity,
                            totalOutputPower )
            printTable( outputTable )

         else:
            if dualInput:
               tableColumns = [ [ "Power", [ "Supply" ], ],
                                [ "", [ "Model" ], ],
                                [ "Input A", [ "Voltage" ] ],
                                [ "Input A", [ "Current" ] ],
                                [ "Input B", [ "Voltage" ], ],
                                [ "Input B", [ "Current" ], ], ]
               outputTable = makeTable( tableColumns )

               for label in sortedPsuLabels:
                  ps = self.powerSupplies[ label ]
                  outputTable.newRow( label, ps.modelName,
                     _voltageString( ps.inputVoltage, managed=ps.managed ),
                     _currentString( ps.inputCurrent, managed=ps.managed ),
                     _voltageString( ps.inputBVoltage, managed=ps.managed ),
                     _currentString( ps.inputBCurrent, managed=ps.managed ) )

               tableColumns = [ [ "Power", [ "Supply" ], ],
                                [ "", [ "Capacity" ], ], 
                                [ "Output", [ "Current" ], ],
                                [ "Output", [ "Power" ], ],
                                [ "", [ "Status" ], ],
                                [ "", [ "Uptime" ], ] ]

            else: # dualOutput
               tableColumns = [ [ "Power", [ "Supply" ], ],
                                [ "", [ "Model" ], ],
                                [ "Output A", [ "Voltage" ] ],
                                [ "Output A", [ "Current" ] ],
                                [ "Output B", [ "Voltage" ], ],
                                [ "Output B", [ "Current" ], ], ]
               outputTable = makeTable( tableColumns )

               for label in sortedPsuLabels:
                  ps = self.powerSupplies[ label ]
                  outputTable.newRow( label, ps.modelName,
                     _voltageString( ps.outputVoltage, managed=ps.managed ),
                     _currentString( ps.outputCurrent, managed=ps.managed ),
                     _voltageString( ps.outputBVoltage, managed=ps.managed ),
                     _currentString( ps.outputBCurrent, managed=ps.managed ) )

               tableColumns = [ [ "Power", [ "Supply" ], ],
                                [ "", [ "Capacity" ], ], 
                                [ "Input", [ "Current" ], ],
                                [ "Output", [ "Power" ], ],
                                [ "", [ "Status" ], ],
                                [ "", [ "Uptime" ], ] ]

            outputTable2 = makeTable( tableColumns )
            for label in sortedPsuLabels:
               ps = self.powerSupplies[ label ]
               # We should actually check all attributes used below, but we are
               # guaranteed that they are set if 'state' is set for now.
               uptime = "Offline"
               if ps.state is not None:
                  if ps.state == 'ok':
                     upt = Tac.utcNow() - ps.uptime
                     td = datetime.timedelta( seconds=int( upt ) )
                     uptime = str( td )
                  if ps.capacity is not None and totalCapacity is not None:
                     if ps.state == 'ok':
                        totalCapacity += ps.capacity
                  else:
                     totalCapacity = None
                  if ps.outputPower is not None and totalOutputPower is not None:
                     totalOutputPower += ps.outputPower
                  else:
                     totalOutputPower = None
                  if dualInput:
                     currentStr = _currentString( ps.outputCurrent,
                                                  managed=ps.managed )
                  else:
                     currentStr = _currentString( ps.inputCurrent,
                                                  managed=ps.managed )
                  outputTable2.newRow( label,
                     _powerString( ps.capacity, precision=0 ),
                     currentStr,
                     _powerString( ps.outputPower, managed=ps.managed ),
                     ps.status(), uptime )
            outputTable2.newRow( "Total",
                  _powerString( totalCapacity, precision=0 ), '--', 
                  _powerString( totalOutputPower ), '--', '--' )
            printThreshold( cliConfig.powerThreshold, totalCapacity,
                            totalOutputPower )
            printTable( outputTable )
            print()
            printTable( outputTable2 )

      if self.details is None:
         return

      print()
      print( 'Polling Interval: %ds' % ( self.details.pollInterval ) )

      for label in sortedPsuLabels:
         ps = self.powerSupplies[ label ]
         print( f'Power Supply Slot {label}, {ps.modelName}' )

         if not ps.details:
            print( _indentStr( 1 ) + 'Showing power detail information is not ' \
                  'supported for this platform yet' )
            continue

         if ps.specificInfo:
            print( ps.specificInfo )

         if ps.outputBVoltage is not None and ps.outputBCurrent is not None:
            print( _indentStr( 1 ) + 'Output A Voltage: ' \
                                   + _voltageString( ps.outputVoltage ) )
            print( _indentStr( 1 ) + 'Output A Current: ' \
                                   + _currentString( ps.outputCurrent ) )
            print( _indentStr( 1 ) + 'Output B Voltage: ' \
                                   + _voltageString( ps.outputBVoltage ) )
            print( _indentStr( 1 ) + 'Output B Current: ' \
                                   + _currentString( ps.outputBCurrent ) )
            print( _indentStr( 1 ) + 'Output Power: ' \
                                   + _powerString( ps.outputPower, 2 ) )
         else:
            print( _indentStr( 1 ) + 'Output Voltage: ' \
                                   + _voltageString( ps.outputVoltage ) )
            print( _indentStr( 1 ) + 'Output Current: ' \
                                   + _currentString( ps.outputCurrent ) )
            print( _indentStr( 1 ) + 'Output Power: ' \
                                   + _powerString( ps.outputPower, 2 ) )
         if ps.inputBVoltage is not None and ps.inputBCurrent is not None:
            print( _indentStr( 1 ) + 'Input A Voltage: ' \
                                   + _voltageString( ps.inputVoltage ) )
            print( _indentStr( 1 ) + 'Input A Current: ' \
                                   + _currentString( ps.inputCurrent ) )
            print( _indentStr( 1 ) + 'Input B Voltage: ' \
                                   + _voltageString( ps.inputBVoltage ) )
            print( _indentStr( 1 ) + 'Input B Current: ' \
                                   + _currentString( ps.inputBCurrent ) )
         else:
            print( _indentStr( 1 ) + 'Input Voltage: ' \
                                   + _voltageString( ps.inputVoltage ) )
            print( _indentStr( 1 ) + 'Input Current: ' \
                                   + _currentString( ps.inputCurrent ) )
         print( _indentStr( 1 ) + 'Input Power: ' \
                                + _powerString( ps.inputPower, 2 ) )

         for fan in sorted( ps.fans.values() ):
            print( _indentStr( 1 ) + 'Fan Speed: %s' % fan.speed )

         for sensorName, sensor in sorted( ps.tempSensors.items() ):
            tempString = _tempSensorString( sensorName, ps.modelName,
                                            ps.state,
                                            sensor.temperature )
            print( _indentStr( 1 ) + 'Temperature ({}): {}'.format( sensorName,
                                                               tempString ) )
         if not ps.blackBoxLog:
            continue
         try:
            blackBoxLog = ps.blackBoxLog.split( "|" )
            headerFields = blackBoxLog[ 0 ].strip().split( "," )
            blackBoxDict = json.loads( blackBoxLog[ 1 ].strip() )
         except ( ValueError, IndexError ):
            continue
         header = [ [ "Page" ] ] + [ [ field.strip() ] for field in headerFields ]
         blackBoxTable = makeBlackBoxTable( header, len( headerFields ) )
         for page, log in sorted( blackBoxDict.items() ):
            row = [ page ] + [ value.strip() for value in log.split( "," ) ]
            blackBoxTable.newRow( *row )
         print( _indentStr( 1 ) + 'Black box log:' )
         printTable( blackBoxTable )

def _fans( ps ):
   fans = {}
   for fan in ps.fan.values():
      fanName = EntityMib.componentName( fan )
      fanEnvStatus = archerCoolingStatus.get( fanName )
      if fanEnvStatus:
         fans[ fanName ] = Fan( status=fanEnvStatus.hwStatus,
                                speed=int( fanEnvStatus.speed ) )
   return fans

def _tempSensors( ps ):
   tempSensors = {}
   for sensor in ps.sensor.values():
      if sensor.tag == 'TempSensor':
         tempSensorName = EntityMib.componentName( sensor )
         tsEnvStatus = envTempStatusDir.get( tempSensorName )
         if tsEnvStatus:
            temperature = None
            if tsEnvStatus.temperature != float( '-Inf' ):
               temperature = tsEnvStatus.temperature
            tempSensors[ tempSensorName ] = \
                TempSensor( status=tsEnvStatus.hwStatus,
                            temperature=temperature )
   return tempSensors

def showPower( mode, args ):
   if not _systemInitialized():
      if _isStandbyMode( mode ):
         mode.addError( 'This command only works on the active supervisor.' )
      else:
         mode.addError( 'System is not yet initialized.' )
      return None

   slots = list( entityMib.root.powerSupplySlot.values() )
   # this should really never hapen
   if not slots:
      mode.addError( 'There seem to be no power supplies connected.' )
      return None

   powerSupplies = {}
   slots.sort( key=lambda s: s.label )
   for slot in slots:
      ps = slot.powerSupply
      if ps is not None:
         psEnvStatus = powerSupplyDir.get( 'PowerSupply%d' % ps.relPos )
         if not psEnvStatus:
            continue

         fans = _fans( ps )
         tempSensors = _tempSensors( ps )

         inputCurrent = None
         if psEnvStatus.inputCurrentSensor:
            if psEnvStatus.inputCurrentSensor.current != float( '-Inf' ):
               inputCurrent = psEnvStatus.inputCurrentSensor.current
         inputBCurrent = None
         if psEnvStatus.inputBCurrentSensor:
            if psEnvStatus.inputBCurrentSensor.current != float( '-Inf' ):
               inputBCurrent = psEnvStatus.inputBCurrentSensor.current
         outputCurrent = None
         if psEnvStatus.outputCurrentSensor:
            if psEnvStatus.outputCurrentSensor.current != float( '-Inf' ):
               outputCurrent = psEnvStatus.outputCurrentSensor.current
         outputBCurrent = None
         if psEnvStatus.outputBCurrentSensor:
            if psEnvStatus.outputBCurrentSensor.current != float( '-Inf' ):
               outputBCurrent = psEnvStatus.outputBCurrentSensor.current
         inputVoltage = None
         if psEnvStatus.inputVoltageSensor:
            if psEnvStatus.inputVoltageSensor.voltage != float( '-Inf' ):
               inputVoltage = psEnvStatus.inputVoltageSensor.voltage
         inputBVoltage = None
         if psEnvStatus.inputBVoltageSensor:
            if psEnvStatus.inputBVoltageSensor.voltage != float( '-Inf' ):
               inputBVoltage = psEnvStatus.inputBVoltageSensor.voltage
         outputVoltage = None
         if psEnvStatus.outputVoltageSensor:
            if psEnvStatus.outputVoltageSensor.voltage != float( '-Inf' ):
               outputVoltage = psEnvStatus.outputVoltageSensor.voltage
         outputBVoltage = None
         if psEnvStatus.outputBVoltageSensor:
            if psEnvStatus.outputBVoltageSensor.voltage != float( '-Inf' ):
               outputBVoltage = psEnvStatus.outputBVoltageSensor.voltage

         capacity = None
         if psEnvStatus.capacity != float( '-Inf' ):
            capacity = psEnvStatus.capacity

         outputPower = None
         if psEnvStatus.outputPower != float( '-Inf' ):
            outputPower = psEnvStatus.outputPower

         managed = _isManaged( psEnvStatus )

         if psEnvStatus.state == "ok":
            deltaUptime = Tac.now() - psEnvStatus.lastStateChangeTime
            uptime = Tac.utcNow() - deltaUptime
         else:
            uptime = None
         powerSupplies[ ps.label ] = \
             PowerSupply( modelName=ps.modelName,
                          capacity=capacity,
                          dominant=psEnvStatus.dominant,
                          inputCurrent=inputCurrent,
                          inputBCurrent=inputBCurrent,
                          outputCurrent=outputCurrent,
                          outputBCurrent=outputBCurrent,
                          inputVoltage=inputVoltage,
                          inputBVoltage=inputBVoltage,
                          outputVoltage=outputVoltage,
                          outputBVoltage=outputBVoltage,
                          outputPower=outputPower,
                          state=psEnvStatus.state,
                          fans=fans,
                          tempSensors=tempSensors,
                          managed=managed,
                          uptime=uptime,
                        )

   return PowerSupplies( powerSupplies=powerSupplies )

def showPowerDetail( mode, args ):
   if not _systemInitialized():
      if _isStandbyMode( mode ):
         mode.addError( 'This command only works on the active supervisor.' )
      else:
         mode.addError( 'System is not yet initialized.' )
      return None

   slots = list( entityMib.root.powerSupplySlot.values() )
   # this should really never hapen
   if not slots:
      mode.addError( 'There seem to be no power supplies connected.' )
      return None

   details = PowerSupplies.Details( pollInterval=cliConfig.pollInterval )

   powerSupplies = {}
   slots.sort( key=lambda s: s.label )
   for slot in slots:
      ps = slot.powerSupply
      if ps is not None:
         psName = 'PowerSupply%s' % ps.label

         psDetails = False
         specificInfoStr = None
         mfrId = None
         mfrModel = None
         mfrRevision = None
         priFirmware = None
         secFirmware = None

         status = genericStatus.status.get( int( ps.label ) )
         if status is not None:
            psDetails = True

            # power supply specific fields
            specificInfoFn = powerSpecificInfoFn.get( status.tacType.fullTypeName )

            if specificInfoFn:
               formatStr = _indentStr( 1 ) + '{}: {}'
               specificInfo = specificInfoFn( status, formatStr )
               specificInfoStr = specificInfo.getStrRep()

               specificInfoDict = specificInfo.getDictRep()
               mfrId = specificInfoDict.get( 'MFR_ID' )
               mfrModel = specificInfoDict.get( 'MFR_MODEL' )
               mfrRevision = specificInfoDict.get( 'MFR_REVISION' )
               fwRevision = specificInfoDict.get( 'FW_REVISION' )
               priFirmware = specificInfoDict.get( 'PRI_FIRMWARE', fwRevision )
               secFirmware = specificInfoDict.get( 'SEC_FIRMWARE', fwRevision )

         psEnvStatus = powerSupplyDir.get( psName )
         if not psEnvStatus:
            powerSupplies[ ps.label ] = \
                PowerSupply( modelName=ps.modelName,
                             details=False )
            continue

         fans = _fans( ps )
         tempSensors = _tempSensors( ps )

         inputCurrent = None
         if psEnvStatus.inputCurrentSensor:
            if psEnvStatus.inputCurrentSensor.current != float( '-Inf' ):
               inputCurrent = psEnvStatus.inputCurrentSensor.current
         inputBCurrent = None
         if psEnvStatus.inputBCurrentSensor:
            if psEnvStatus.inputBCurrentSensor.current != float( '-Inf' ):
               inputBCurrent = psEnvStatus.inputBCurrentSensor.current
         outputCurrent = None
         if psEnvStatus.outputCurrentSensor:
            if psEnvStatus.outputCurrentSensor.current != float( '-Inf' ):
               outputCurrent = psEnvStatus.outputCurrentSensor.current
         outputBCurrent = None
         if psEnvStatus.outputBCurrentSensor:
            if psEnvStatus.outputBCurrentSensor.current != float( '-Inf' ):
               outputBCurrent = psEnvStatus.outputBCurrentSensor.current
         inputVoltage = None
         if psEnvStatus.inputVoltageSensor:
            if psEnvStatus.inputVoltageSensor.voltage != float( '-Inf' ):
               inputVoltage = psEnvStatus.inputVoltageSensor.voltage
         inputBVoltage = None
         if psEnvStatus.inputBVoltageSensor:
            if psEnvStatus.inputBVoltageSensor.voltage != float( '-Inf' ):
               inputBVoltage = psEnvStatus.inputBVoltageSensor.voltage
         outputVoltage = None
         if psEnvStatus.outputVoltageSensor:
            if psEnvStatus.outputVoltageSensor.voltage != float( '-Inf' ):
               outputVoltage = psEnvStatus.outputVoltageSensor.voltage
         outputBVoltage = None
         if psEnvStatus.outputBVoltageSensor:
            if psEnvStatus.outputBVoltageSensor.voltage != float( '-Inf' ):
               outputBVoltage = psEnvStatus.outputBVoltageSensor.voltage
         capacity = None
         if psEnvStatus.capacity != float( '-Inf' ):
            capacity = psEnvStatus.capacity
         inputPower = None
         if psEnvStatus.inputPower != float( '-Inf' ):
            inputPower = psEnvStatus.inputPower
         outputPower = None
         if psEnvStatus.outputPower != float( '-Inf' ):
            outputPower = psEnvStatus.outputPower
         blackBoxLog = psEnvStatus.blackBoxLog

         managed = _isManaged( psEnvStatus )

         if psEnvStatus.state == "ok":
            deltaUptime = Tac.now() - psEnvStatus.lastStateChangeTime
            uptime = Tac.utcNow() - deltaUptime
         else:
            uptime = None
         powerSupplies[ ps.label ] = \
             PowerSupply( modelName=ps.modelName,
                          capacity=capacity,
                          dominant=psEnvStatus.dominant,
                          inputCurrent=inputCurrent,
                          outputCurrent=outputCurrent,
                          outputBCurrent=outputBCurrent,
                          inputVoltage=inputVoltage,
                          outputVoltage=outputVoltage,
                          outputBVoltage=outputBVoltage,
                          inputBVoltage=inputBVoltage,
                          inputBCurrent=inputBCurrent,
                          inputPower=inputPower,
                          outputPower=outputPower,
                          state=psEnvStatus.state,
                          fans=fans,
                          tempSensors=tempSensors,
                          specificInfo=specificInfoStr,
                          mfrId=mfrId,
                          mfrModel=mfrModel,
                          mfrRevision=mfrRevision,
                          priFirmware=priFirmware,
                          secFirmware=secFirmware,
                          details=psDetails,
                          managed=managed,
                          uptime=uptime,
                          blackBoxLog=blackBoxLog,
                        )

   return PowerSupplies( powerSupplies=powerSupplies,
                         details=details )

def showPowerVoltage( mode, args ):
   if not _systemInitialized():
      if _isStandbyMode( mode ):
         mode.addError( 'This command only works on the active supervisor.' )
      else:
         mode.addError( 'System is not yet initialized.' )
      return None

   model = VoltageSensors()

   def _populatePsSensors( model ):
      # pylint: disable-next=too-many-nested-blocks
      for slot in entityMib.root.powerSupplySlot.values():
         slotName = slot.tag + slot.label
         supply = slot.powerSupply
         if supply:
            modelPsSlot = FruSlot()
            modelPsSlot.entPhysicalClass = slot.tag
            modelPsSlot.label = slot.label
            for sensor in supply.sensor.values():
               if sensor.tag == "VoltageSensor":
                  name = EntityMib.componentName( sensor )
                  voltage = None
                  description = sensor.description
                  expectedVoltage = None
                  config = powerConfig.voltageSensor.get( name )
                  status = _getVoltageSensor( name )
                  if not status:
                     continue
                  if status.voltage != float( "-Inf" ):
                     voltage = status.voltage
                  if config:
                     if config.expectedVoltage != float( "-Inf" ):
                        expectedVoltage = config.expectedVoltage

                  modelPsSlot.voltageSensors[ sensor.label ] = VoltageSensor(
                     status=status.hwStatus,
                     description=description,
                     voltage=voltage,
                     expectedVoltage=expectedVoltage )
            if modelPsSlot.voltageSensors:
               model.powerSupplySlots[ slotName ] = modelPsSlot

   # pylint: disable-next=too-many-nested-blocks
   if entityMib.root.tacType.fullTypeName == "EntityMib::Chassis":
      for slot in entityMib.root.cardSlot.values():
         slotName = slot.tag + slot.label
         if slot.card:
            # Card is inserted
            modelCardSlot = FruSlot()
            modelCardSlot.entPhysicalClass = slot.tag
            modelCardSlot.label = slot.label
            for sensor in slot.card.sensor.values():
               if sensor.tag == "VoltageSensor":
                  name = EntityMib.componentName( sensor )
                  voltage = None
                  description = sensor.description
                  expectedVoltage = None
                  config = powerConfig.voltageSensor.get( name )
                  status = _getVoltageSensor( name )
                  if not status:
                     continue
                  if status.voltage != float( "-Inf" ):
                     voltage = status.voltage
                  if config:
                     if config.expectedVoltage != float( "-Inf" ):
                        expectedVoltage = config.expectedVoltage

                  modelCardSlot.voltageSensors[ sensor.label ] = VoltageSensor(
                     status=status.hwStatus,
                     description=description,
                     voltage=voltage,
                     expectedVoltage=expectedVoltage )
            if modelCardSlot.voltageSensors:
               model.cardSlots[ slotName ] = modelCardSlot
      _populatePsSensors( model )
   elif entityMib.root.tacType.fullTypeName == "EntityMib::FixedSystem":
      for sensor in entityMib.root.sensor.values():
         if sensor.tag == "VoltageSensor":
            name = EntityMib.componentName( sensor )
            voltage = None
            description = sensor.description
            expectedVoltage = None
            config = powerConfig.voltageSensor.get( name )
            status = _getVoltageSensor( name )
            if not status:
               continue
            if status.voltage != float( "-Inf" ):
               voltage = status.voltage
            if config:
               if config.expectedVoltage != float( "-Inf" ):
                  expectedVoltage = config.expectedVoltage

            model.voltageSensors[ sensor.label ] = VoltageSensor(
               status=status.hwStatus,
               description=description,
               voltage=voltage,
               expectedVoltage=expectedVoltage )
      _populatePsSensors( model )

   return model

def showPowerCurrent( mode, args ):
   if not _systemInitialized():
      if _isStandbyMode( mode ):
         mode.addError( 'This command only works on the active supervisor.' )
      else:
         mode.addError( 'System is not yet initialized.' )
      return None

   model = CurrentSensors()

   def _populateEcbSensors( model, cardStatusMap ):
      # Iterate over all the cards that use ECBs and add a FruSlot for each one
      for cardName, cardStatus in sorted( cardStatusMap.items() ):
         sensorStatusMap = cardStatus.ecb

         modelCardSlot = _getSlotByName( cardName )

         # Add each sensor to the FruSlot it belongs to
         for sensorName, sensorStatus in sorted( sensorStatusMap.items() ):
            # The Ecb agent represents an unknown status as 'unknown', while the
            # other sensors use 'unknownHwStatus'.
            ecbStatus = cardStatus.state
            if ecbStatus == 'unknown':
               ecbStatus = 'unknownHwStatus'

            modelCardSlot.currentSensors[ sensorName ] = CurrentSensor(
               status=ecbStatus,
               name=sensorName,
               current=sensorStatus.current )

         # Only add the FruSlot if it has at least one current sensor
         if modelCardSlot.currentSensors:
            model.cardSlots[ cardName ] = modelCardSlot

   def _populateVoltageRailSensors( deviceName, deviceStatus ):
      # Return a collection of current sensors corresponding to pages and rails
      currentSensorPages = {}

      # get the voltage rail status of the page/rail to add to FruSlot
      voltageRailStatus = ( getattr( deviceStatus, 'voltageRailStatus', None ) or
                            getattr( deviceStatus, 'voltageRail', None ) )
      if voltageRailStatus is None:
         return {}

      # On Fabrics and Linecards, currents measured on each page and rail have
      # different voltages. Hence, we need to list each of them out in separate
      # currentSensors

      negativeInfinity = float( '-Inf' )
      for vrStatus in voltageRailStatus.values():
         # first specify if vrStatus is page or rail
         voltageRailEnvStatus = getattr( vrStatus, 'voltageRailEnvStatus', None )
         if voltageRailEnvStatus:
            # if voltageRailEnvStatus exists, it means this is a current sensor
            # on dpm, we use rail number to match the show voltage print out
            currentSensorStatus = voltageRailEnvStatus
            pageName = f"Rail{ vrStatus.id }"
         else:
            currentSensorStatus = vrStatus
            pageName = currentSensorStatus.name
         # If the measured current is negative infinity, it indicates this sensor
         # does not support current measurement now, we only show all available 
         # current measurements.
         if currentSensorStatus.current != negativeInfinity:
            currentSensorPageName = deviceName + '_' + pageName
            currentSensorPages[ currentSensorPageName ] = CurrentSensor(
               status=currentSensorStatus.hwStatus, name=currentSensorPageName,
               current=currentSensorStatus.current )

      return currentSensorPages

   # Add the ecb sensors
   if 'ecb' in hardwareDir:
      _populateEcbSensors( model, ecbSystemDir.card )
      for cellNum in ecbCellDir.values():
         _populateEcbSensors( model, cellNum.card )

   # Add the other current sensors
   if entityMib.root.tacType.fullTypeName == "EntityMib::Chassis":
      for deviceName, deviceStatus in currentSensorSystemDir.items():
         # Find which FruSlot the device belongs to
         modelCardSlot = _getSlotByName( deviceName )
         modelCardSlot.label = ""
      
         # This assumes that the device name is a tag followed by a number, then by
         # a non-numeric substring.
         for c in deviceName[ len( modelCardSlot.entPhysicalClass ) : ]:
            if str.isdigit( c ):
               modelCardSlot.label += c
            else:
               break

         # Check if we've already added a FruSlot
         slotName = modelCardSlot.entPhysicalClass + modelCardSlot.label
         if slotName in model.cardSlots:
            modelCardSlot = model.cardSlots[ slotName ]

         # Create and add current sensors with the current read from each sensor on
         # the card slot.
         currentSensor = _populateVoltageRailSensors( deviceName, deviceStatus )
         for devicePageName, currentSensorPage in currentSensor.items():
            modelCardSlot.currentSensors[ devicePageName ] = currentSensorPage

         # Add the card slot to the model
         model.cardSlots[ slotName ] = modelCardSlot
   elif entityMib.root.tacType.fullTypeName == "EntityMib::FixedSystem":
      for cellNum in currentSensorCellDir.values():
         for deviceName, deviceStatus in cellNum.items():
            currentSensor = _populateVoltageRailSensors( deviceName, deviceStatus )
            for devicePageName, currentSensorPage in currentSensor.items():
               model.currentSensors[ devicePageName ] = currentSensorPage

   return model

#--------------------------------------------------
# 'power poll-interval <INTERVAL> [ milliseconds ]'
#--------------------------------------------------
def pollInterval( mode, args ):
   interval = args.get( 'INTERVAL', cliConfig.defaultPollInterval )
   if 'milliseconds' in args:
      interval /= 1000.0
   cliConfig.pollInterval = interval

#--------------------------------------------------------------------------------
# [ no | default ] environment power controller poll-interval
#--------------------------------------------------------------------------------
def setPowerControllerPollInterval( mode, args ):
   interval = args.get( 'INTERVAL', DEFAULT_POWER_CONTROLLER_POLL_INTERVAL )
   if 'milliseconds' in args:
      interval /= 1000.0
   powerControllerCliConfig.pollInterval = interval

#--------------------------------------------------------------------------------
# [ no | default ] power limit current NUM_AMPS amps
#--------------------------------------------------------------------------------
def setPowerLimitCurrent( mode, args ):
   defaultPowerCurrentLimit = Tac.Value( "Units::Amps" ).value
   cliConfig.currentLimit = args.get( 'NUM_AMPS', defaultPowerCurrentLimit )

#--------------------------------------------------------------------------------
# [ no | default ] power redundancy cold
#--------------------------------------------------------------------------------
def enablePowerRedundancyCold( mode, args ):
   cliConfig.coldRedundancyEnabled = True

def disablePowerRedundancyCold( mode, args ):
   cliConfig.coldRedundancyEnabled = False

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global powerConfig, archerCoolingStatus, envTempStatusDir, entityMib
   global genericStatus, cliConfig
   global powerSupplyDir, voltageSensorCellDir, voltageSensorSystemDir
   global ecbSystemDir, ecbCellDir, currentSensorCellDir, currentSensorSystemDir
   global hardwareDir
   global powerControllerCliConfig

   powerSupplyDir = LazyMount.mount( entityManager,
                                     'environment/archer/power/status/powerSupply',
                                     'Tac::Dir', 'ri' )

   voltageSensorCellDir = LazyMount.mount( entityManager,
         'environment/archer/power/status/voltageSensor/cell', 'Tac::Dir', 'ri' )

   voltageSensorSystemDir = LazyMount.mount( entityManager,
         'environment/archer/power/status/voltageSensor/system', 'Tac::Dir', 'ri' )

   ecbSystemDir = LazyMount.mount( entityManager,
         'hardware/ecb/status/system', 'Hardware::Ecb::EcbDirStatus', 'ri' )

   ecbCellDir = LazyMount.mount( entityManager,
         'hardware/ecb/status/cell', 'Tac::Dir', 'ri' )

   currentSensorCellDir = LazyMount.mount( entityManager,
         'hardware/archer/powercontroller/status/cell', 'Tac::Dir', 'ri' ) 

   currentSensorSystemDir = LazyMount.mount( entityManager,
         'hardware/archer/powercontroller/status/system', 'Tac::Dir', 'ri' ) 

   hardwareDir = LazyMount.mount( entityManager, 'hardware', 'Tac::Dir', 'ri' ) 

   powerConfig = LazyMount.mount( entityManager, 'environment/power/config',
                                  'Environment::Power::Config', 'r' )
   archerCoolingStatus = LazyMount.mount( entityManager,
                                    'environment/archer/cooling/status',
                                    'Tac::Dir', 'ri' )
   envTempStatusDir = LazyMount.mount( entityManager,
         'environment/archer/temperature/status/system', 'Tac::Dir', 'ri' )
   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                'EntityMib::Status', 'r' )
   # collection that holds pointers to all entities derived from
   # Hardware::PowerSupply::PowerStatus used in show system environment power detail
   genericStatus = LazyMount.mount( entityManager,
                                    "hardware/powerSupply/status/generic",
                                    "Hardware::PowerSupply::PowerStatusPtrDir", "r" )
   cliConfig = ConfigMount.mount( entityManager,
                                  "hardware/powerSupply/config/cli",
                                  "Hardware::PowerSupply::CliConfig", "w" )
   powerControllerCliConfig = ConfigMount.mount(
      entityManager,
      "hardware/powercontroller/cliconfig",
      "Hardware::PowerController::CliConfig", "w" )

   EnvironmentCli.extraShowSystemEnvAllModelHandlerHook.addExtension( showPower )
