# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import math
import BasicCli
import BasicCliModes
import CliCommand
import CliExtensions
import CliMode.PowerManager
import CliParser
from CliPlugin import ModuleCli
from CliPlugin import TechSupportCli
from CliPlugin import PowerManagerModel
from CliPlugin.FruCli import SlotExpressionFactory
from CliPlugin.ThermoMgrGlobalConfigMode import (
      matcherAction,
      matcherEnvironment as matcherEnvironmentForConfig )
from CliMatcher import IntegerMatcher, KeywordMatcher
from CliToken.System import systemMatcherForShow, systemMatcherForConfig
import ConfigMount
import LazyMount
import ShowCommand
import Tac
# pkgdeps: rpmwith %{_libdir}/preinit/PowerManager

MAX_POWER_BUDGET = PowerManagerModel.MAX_POWER_BUDGET

entityMib = None
powerManager = None
powerManagerHwConfig = None
powerManagerCliConfig = None
powerCliConfig = None
managementMode = Tac.Type( "PowerManager::PowerManagerCliConfig::ManagementMode" )
poeStatus = None
poeCliConfig = None

def _systemInitialized():
   return entityMib.root and entityMib.root.initStatus == 'ok'

def _isStandbyMode( mode ):
   return ( mode.session_ and
            mode.session_.entityManager_.redundancyStatus().mode == 'standby' )

#--------------------------------------------------
# '[ no | default ] power budget [n] watts'
#--------------------------------------------------
def powerSupplyBudget( mode, args ):
   powerBudgetAmount = float( args[ "WATTS" ] )
   powerManagerCliConfig.powerBudget = powerBudgetAmount
   if powerManager.totalPower < powerBudgetAmount <= MAX_POWER_BUDGET:
      warning = "Budget is set to {0:.1f}W, which is greater than {1:.1f}W of " \
                "total power"
      mode.addWarning( warning.format( powerBudgetAmount,
                                       powerManager.totalPower ) )

class PowerBudgetCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget WATTS watts'
   noOrDefaultSyntax = 'power budget ...'
   data = {
      'power': 'Configure power supplies',
      'budget': 'Configure power budget (excluding chassis power)',
      'WATTS': IntegerMatcher( 0, MAX_POWER_BUDGET,
         helpdesc='Amount of power budgeted to the system' ),
      'watts': 'Unit for power budget',
   }

   @staticmethod
   def handler( mode, args ):
      powerSupplyBudget( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      args[ "WATTS" ] = "inf"
      powerSupplyBudget( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( PowerBudgetCmd )

#--------------------------------------------------
# '[ no | default ] power threshold [n] watts'
#--------------------------------------------------
def powerSupplyThreshold( mode, args ):
   powerThresholdAmount = float( args.get( "WATTS", "inf" ) )
   powerCliConfig.powerThreshold = powerThresholdAmount
   if powerManager.totalPower < powerThresholdAmount <= MAX_POWER_BUDGET:
      warning = "Threshold is set to {0:.1f}W, which is greater than {1:.1f}W of " \
                "total power"
      mode.addWarning( warning.format( powerThresholdAmount,
                                       powerManager.totalPower ) )

class PowerThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'power threshold WATTS watts'
   noOrDefaultSyntax = 'power threshold ...'
   data = {
      'power': 'Configure power supplies',
      'threshold': 'Configure power threshold',
      'WATTS': IntegerMatcher( 0, MAX_POWER_BUDGET,
         helpdesc='Amount of power consumed that will trigger a warning' ),
      'watts': 'Unit for power threshold',
   }
   handler = powerSupplyThreshold
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PowerThresholdCmd )

#-----------------------------------------------------------------------------
# '[ no | default ] power budget exceeded action [ warning | hold-down ]
#-----------------------------------------------------------------------------
matcherWarning = KeywordMatcher(
   'warning', helpdesc='Raise a log message but allow the device to power on' )
matcherHoldDown = KeywordMatcher(
   'hold-down', helpdesc='Prevent the device from powering on' )

class PowerBudgetActionCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget exceeded action [ warning | hold-down ]'
   noOrDefaultSyntax = 'power budget exceeded action ...'
   data = {
      'power': 'Configure power supplies',
      'budget': 'Configure power budget (excluding chassis power)',
      'exceeded': 'Actions when devices exceed budget',
      'action': 'Actions when devices exceed budget',
      'warning': matcherWarning,
      'hold-down': matcherHoldDown,
   }

   @staticmethod
   def handler( mode, args ):
      if "warning" in args:
         powerManagerCliConfig.mode = managementMode.warn
      elif "hold-down" in args:
         powerManagerCliConfig.mode = managementMode.strict

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      powerManagerCliConfig.mode = managementMode.strict

BasicCliModes.GlobalConfigMode.addCommandClass( PowerBudgetActionCmd )

#-----------------------------------------------------------------------------
# '[ no | default ] power threshold exceeded action [ warning ]
#-----------------------------------------------------------------------------
class PowerThresholdActionCmd( CliCommand.CliCommandClass ):
   syntax = 'power threshold exceeded action [ warning ]'
   noOrDefaultSyntax = 'power threshold exceeded action ...'
   data = {
      'power': 'Configure power supplies',
      'threshold': 'Configure power threshold',
      'exceeded': 'Actions when power exceeds threshold',
      'action': 'Actions when power exceeds threshold',
      'warning': matcherWarning,
   }

   # Establish basis for future cli actions with regards to threshold commands
   # Exists now to indicate to the user what is being configured.
   # Currently only has the one option of warning so handlers do nothing
   @staticmethod
   def handler( mode, args ):
      return

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return

BasicCliModes.GlobalConfigMode.addCommandClass( PowerThresholdActionCmd )

#-----------------------------------------------------------------------------
# [ no | default ] power domain split
#-----------------------------------------------------------------------------
def powerDomainGuard( mode, token ):
   if len( powerManagerHwConfig.powerDomain ) < 2:
      return CliParser.guardNotThisPlatform
   return None

nodeDomain = CliCommand.guardedKeyword(
   'domain',
   helpdesc='Configure power domains',
   guard=powerDomainGuard )

class PowerDomainSplitCmd( CliCommand.CliCommandClass ):
   syntax = 'power domain split'
   noOrDefaultSyntax = syntax
   data = {
      'power': 'Configure power supplies',
      'domain': nodeDomain,
      'split': 'Use split power domain mode when system has only one power supply',
   }

   @staticmethod
   def handler( mode, args ):
      powerManagerCliConfig.splitDomainMode = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      powerManagerCliConfig.splitDomainMode = False

BasicCliModes.GlobalConfigMode.addCommandClass( PowerDomainSplitCmd )

#-----------------------------------------------------------------------------
# [ no | default ] environment insufficient-power action [ shutdown | ignore ]
#-----------------------------------------------------------------------------
class PowerBudgetModularActionCmd( CliCommand.CliCommandClass ):
   syntax = 'environment insufficient-power action [ shutdown | ignore ]'
   noOrDefaultSyntax = 'environment insufficient-power action ...'
   data = {
      'environment': matcherEnvironmentForConfig,
      'insufficient-power': 'Configure system behavior when there is not enough '
                            'power for a card',
      'action': matcherAction,
      'ignore': 'Log a message but allow the card to power on',
      'shutdown': 'Power off the card due to insufficient power',
   }

   handler = "PowerManagerCliHandler.envInsufficientPowerActionHandler"
   noOrDefaultHandler = \
      "PowerManagerCliHandler.envInsufficientPowerActionNoOrDefaultHandler"

BasicCliModes.GlobalConfigMode.addCommandClass( PowerBudgetModularActionCmd )

#--------------------------------------------------------------------------------
# show system environment power budget
#--------------------------------------------------------------------------------
matcherEnvironment = KeywordMatcher( 'environment',
                                     helpdesc='Show system environment status' )
matcherPower = KeywordMatcher( 'power',
                               helpdesc='Show system power status' )
matcherBudget = KeywordMatcher( 'budget',
                                helpdesc='Show usage levels for the power budget'
                                ' ( excluding chassis power )' )

class SystemEnvironmentPowerBudgetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system environment power budget'
   data = {
      'system': systemMatcherForShow,
      'environment': matcherEnvironment,
      'power': matcherPower,
      'budget': matcherBudget,
   }

   @staticmethod
   def handler( mode, args ):
      if not _systemInitialized():
         if _isStandbyMode( mode ):
            mode.addError( 'This command only works on the active supervisor.' )
         else:
            mode.addError( 'System is not yet initialized.' )
         return None

      model = PowerManagerModel.PowerManager()

      consumedPoePower = float( 0 )
      for consumer in powerManager.powerConsumers:
         if poeCliConfig.telemetryBasedPoeEnabled:
            # Use real-time power consumption
            if consumer in poeStatus:
               outputPower = poeStatus[ consumer ].outputPower
               model.powerConsumers[ consumer ] = \
                  PowerManagerModel.PowerConsumer( amount=outputPower )
               consumedPoePower += outputPower
         else:
            # Use static PoE limits
            device = powerManager.powerConsumers[ consumer ]
            model.powerConsumers[ consumer ] = \
               PowerManagerModel.PowerConsumer( amount=device.amount )
            consumedPoePower += device.amount

      model.totalAvailablePower = powerManager.totalPower
      model.powerBudget = powerManagerCliConfig.powerBudget
      model.consumedPower = consumedPoePower
      if len( powerManager.powerDomainStatus ) > 1:
         if powerManager.pseudoLoadSharing:
            model.powerDomains = {}
            for domainName in powerManager.powerDomainStatus:
               domain = powerManager.powerDomainStatus[ domainName ]
               model.powerDomains[ domainName[ 11 : ] ] = \
                     PowerManagerModel.PowerDomainStatus(
                           totalAvailablePower=domain.powerBudget,
                           consumedPower=domain.currentPower )
      else:
         model.powerDomains = None
      return model

   cliModel = PowerManagerModel.PowerManager

BasicCli.addShowCommandClass( SystemEnvironmentPowerBudgetCmd )

#--------------------------------------------------------------------------------
# show system environment power domain
#--------------------------------------------------------------------------------
nodeDomainShow = CliCommand.guardedKeyword(
   'domain',
   helpdesc='Show power domain information',
   guard=powerDomainGuard )

class SystemEnvironmentPowerDomainCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system environment power domain'
   data = {
      'system': systemMatcherForShow,
      'environment': matcherEnvironment,
      'power': matcherPower,
      'domain': nodeDomainShow,
   }
   cliModel = "PowerManagerModel.PowerDomainCollection"
   handler = "PowerManagerCliHandler.showSystemEnvPowerDomainHandler"

BasicCli.addShowCommandClass( SystemEnvironmentPowerDomainCmd )

#--------------------------------------------------------------------------------
# show system environment power histogram
#--------------------------------------------------------------------------------
matcherHistogram = KeywordMatcher( 'histogram',
                                   helpdesc='Show histogram of total power usage' )
class SystemEnvironmentPowerHistogramCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system environment power histogram'
   data = {
      'system': systemMatcherForShow,
      'environment': matcherEnvironment,
      'power': matcherPower,
      'histogram': matcherHistogram,
   }
   cliModel = PowerManagerModel.PowerHistogram

   @staticmethod
   def handler( mode, args ):
      percentPerBucket = 5
      # Each key is 1% granularity, aggregate them into percentPerBucket granularity
      numBuckets = int( math.ceil(
                        len( powerManager.histogram ) / percentPerBucket ) )
      sampleRate = 1 / powerCliConfig.pollInterval # samples per second

      powerHistogramDict = {}
      for idx in range( numBuckets ):
         powerHistogramDict[ idx * percentPerBucket ] = 0
      for percent, uptime in powerManager.histogram.items():
         idx = percent // percentPerBucket
         bucketPercent = idx * percentPerBucket
         powerHistogramDict[ bucketPercent ] += int( uptime )
      return PowerManagerModel.PowerHistogram( buckets=powerHistogramDict,
                                               sampleRate=sampleRate )

BasicCli.addShowCommandClass( SystemEnvironmentPowerHistogramCmd )

TechSupportCli.registerShowTechSupportCmd(
   '2023-07-20 14:17:48',
   cmds=[ 'show system environment power histogram' ] )

#-----------------------------------------------------------------------------------
# show system environment power histogram detail
#-----------------------------------------------------------------------------------
class SystemEnvironmentPowerHistogramDetailCmd(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show system environment power histogram detail'
   data = {
      'system': systemMatcherForShow,
      'environment': matcherEnvironment,
      'power': matcherPower,
      'histogram': matcherHistogram,
      'detail': 'Show detailed system power usage histogram',
   }
   cliModel = "PowerManagerModel.PowerHistogramDetail"
   handler = "PowerManagerCliHandler.showSystemEnvPowerHistogramDetailHandler"

BasicCli.addShowCommandClass( SystemEnvironmentPowerHistogramDetailCmd )

#------------------------------------------------------------------------------------
# power management config mode
#------------------------------------------------------------------------------------
class PowerManagementMode( CliMode.PowerManager.PowerManagementMode,
                           BasicCli.ConfigModeBase ):
   name = 'Power Management configuration'

   def __init__( self, parent, session ):
      CliMode.PowerManager.PowerManagementMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------------------------------
# power management card config mode
#------------------------------------------------------------------------------------
class PowerManagementCardMode( CliMode.PowerManager.PowerManagementCardMode,
                               BasicCli.ConfigModeBase ):
   name = 'Power Management card configuration'

   def __init__( self, parent, session, param ):
      CliMode.PowerManager.PowerManagementCardMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------------------------------
# system power
# in global config mode
#-----------------------------------------------------------------------------------
moduleCleanupHook = CliExtensions.CliHook()

matcherPowerConfig = KeywordMatcher( 'power',
                                     helpdesc='configure system power settings' )

class PowerManagement( CliCommand.CliCommandClass ):
   syntax = 'system power'
   noOrDefaultSyntax = syntax
   data = {
      'system': systemMatcherForConfig,
      'power': matcherPowerConfig
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PowerManagementMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # cleanup all modules
      moduleCleanupHook.notifyExtensions( None )

BasicCli.GlobalConfigMode.addCommandClass( PowerManagement )

#------------------------------------------------------------------------------------
# module MODNAME
# in power management config mode
#------------------------------------------------------------------------------------
class EnterModuleCardCmd( CliCommand.CliCommandClass ):
   syntax = 'module MODNAME'
   noOrDefaultSyntax = syntax
   data = {
      'module': ModuleCli.moduleNode,
      'MODNAME': SlotExpressionFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      card = args[ 'MODNAME' ]
      childMode = mode.childMode( PowerManagementCardMode,
                                  param=str( card.tag ) + str( card.label ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      moduleCleanupHook.notifyExtensions( args[ 'MODENAME' ] )

PowerManagementMode.addCommandClass( EnterModuleCardCmd )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global entityMib
   global powerManager, powerManagerCliConfig, powerCliConfig
   global powerManagerHwConfig
   global poeStatus
   global poeCliConfig

   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                'EntityMib::Status', 'r' )
   powerManagerHwConfig = LazyMount.mount( entityManager,
                          'environment/archer/power/config/powerManager',
                          'PowerManager::PowerManagerConfig', 'r' )
   powerManager = LazyMount.mount( entityManager,
                                   'environment/archer/power/status/powerManager',
                                   'PowerManager::PowerManagerStatus', 'r' )
   powerManagerCliConfig = ConfigMount.mount( entityManager,
                           'environment/archer/power/config/powerManagerCliConfig',
                           'PowerManager::PowerManagerCliConfig', 'w' )
   powerCliConfig = ConfigMount.mount( entityManager,
                                       "hardware/powerSupply/config/cli",
                                       "Hardware::PowerSupply::CliConfig", "w" )
   poeStatus = LazyMount.mount( entityManager,
         'environment/power/status/poePort', 'Tac::Dir', 'ri' )
   poeCliConfig = LazyMount.mount( entityManager,
         'hardware/poe/config/cli', 'Hardware::Poe::CliConfig', 'r' )
