# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMode.IcmpProbe import IcmpProbeBaseMode
from CliPlugin import IntfCli
from CliPlugin import IpGenAddrMatcher
from CliPlugin import IcmpResponderCliLib
from CliPlugin import NetworkToolsCli
from CliPlugin import NetworkUrlCli
from CliPlugin import VrfCli
from CliPlugin import RouterGeneralCli
import ConfigMount
import ShowCommand
from Toggles.IcmpResponderToggleLib import ( toggleIcmpProbeClientEnabled,
      toggleIcmpProbeResponderEnabled )

config = None

probeVrfExprFactory = VrfCli.VrfExprFactory(
            helpdesc='Probe in a VRF',
            inclDefaultVrf=True )

class ProbeInterfaceExpr( CliCommand.CliExpression ):
   expression = (
         '( ifindex IFINDEX ) | ( ifname IFNAME ) | ( addr ADDR ) | '
           '( remote NEIGHBOR )'
           )
   data = {
         'ifindex': 'probe interface by ifIndex',
         'IFINDEX': CliMatcher.IntegerMatcher( 1, 0x7fffffff,
            helpdesc='interface index on proxy node' ),
         'ifname': 'probe interface by ifName',
         # Note: no limitation on interface name, because
         # the limitations exist on the remote system.
         'IFNAME': CliMatcher.PatternMatcher( r'.{1,150}',
            helpname='IFNAME',
            helpdesc='interface name on proxy node' ),
         'addr': 'probe interface by address',
         'ADDR': IpGenAddrMatcher.IpGenAddrMatcher(
            'interface address on proxy node' ),
         'remote': 'probe neighbor of the proxy node',
         'NEIGHBOR': IpGenAddrMatcher.IpGenAddrMatcher(
            'address of neighbor of proxy node' ),
         }

dstMatcher = CliMatcher.PatternMatcher(
      NetworkToolsCli.dnsNameOrIpv4OrIpv6AddressPattern,
      helpname='DESTINATION',
      helpdesc='Proxy node address or hostname' )

class DoProbe( CliCommand.CliCommandClass ):
   syntax = ( 'probe [ VRF ] [ ip | ipv6 ] DESTINATION '
         'PROBE_INTERFACE '
         '[ { '
            '( repeat COUNT ) | '
            '( interval INTERVAL ) | '
            '( source ( IP_GEN_ADDR | INTF_WITH_IP ) ) '
         '} ]' )
   data = {
         'probe': 'probe a remote interface',
         'VRF': probeVrfExprFactory,
         'ip': 'probe via IPv4',
         'ipv6': 'probe via IPv6',
         'DESTINATION': dstMatcher,
         'PROBE_INTERFACE': ProbeInterfaceExpr,
         'repeat': 'number of times to send probe message',
         'COUNT': CliMatcher.IntegerMatcher( 1, 0x7fffffff,
            helpdesc='Count' ),
         'interval': 'interval between probe messages',
         'INTERVAL': CliMatcher.FloatMatcher( 0, 60,
            helpdesc='Interval in units of seconds', precisionString='%.3f' ),
         'source': ( 'specify source address or choose an address '
                     'from specified interface' ),
         'IP_GEN_ADDR': IpGenAddrMatcher.IpGenAddrMatcher( 'source address',
            helpdesc4='IP address of the source',
            helpdesc6='IPv6 address of the source' ),
         'INTF_WITH_IP': CliCommand.Node( IntfCli.Intf.matcher ),
         }
   handler = 'ProbeCli.doProbe'

class DoProbeUnpriv( CliCommand.CliCommandClass ):
   syntax = 'probe  [ ip | ipv6 ] DESTINATION PROBE_INTERFACE'
   data = {
         'probe': 'probe a remote interface',
         'ip': 'probe via IPv4',
         'ipv6': 'probe via IPv6',
         'DESTINATION': NetworkToolsCli.dstMatcher,
         'PROBE_INTERFACE': ProbeInterfaceExpr,
         }
   handler = 'ProbeCli.doProbe'

if toggleIcmpProbeClientEnabled():
   BasicCliModes.EnableMode.addCommandClass( DoProbe )
   BasicCliModes.UnprivMode.addCommandClass( DoProbeUnpriv )
   NetworkUrlCli.addCmdForProtocol( "probe", clientRequired=True )

class IcmpProbeMode( IcmpProbeBaseMode, BasicCli.ConfigModeBase ):
   name = "ICMP PROBE configuration commands"

   def __init__( self, parent, session ):
      IcmpProbeBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# ------------------------------------------------------------
# [no|default] icmp probe
# ------------------------------------------------------------
def gotoIcmpProbeMode( mode, args ):
   childMode = mode.childMode( IcmpProbeMode )
   mode.session_.gotoChildMode( childMode )

def noIcmpProbeMode( mode, args ):
   noIcmpProbeVrfDefault( mode, args )
   noIcmpProbeQueryTypes( mode, {} )

class IcmpProbeCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp probe'
   noOrDefaultSyntax = syntax
   data = {
      'icmp': IcmpResponderCliLib.matcherIcmp,
      'probe': 'PROBE',
   }
   handler = gotoIcmpProbeMode
   noOrDefaultHandler = noIcmpProbeMode

if toggleIcmpProbeResponderEnabled():
   RouterGeneralCli.RouterGeneralMode.addCommandClass( IcmpProbeCmd )

def icmpProbeModeCleanup( mode ):
   noIcmpProbeMode( mode, {} )

RouterGeneralCli.routerGeneralCleanupHook.addExtension( icmpProbeModeCleanup )

# ------------------------------------------------------------
# in icmp probe config mode:
# [no|default] probe vrf default
# (future syntax: [no|default] probe vrf (default|all) [disabled]
# ------------------------------------------------------------

def setIcmpProbeVrfDefault( mode, args ):
   config.enabled = True

def noIcmpProbeVrfDefault( mode, args ):
   config.enabled = False

class IcmpProbeVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'probe vrf default'
   noOrDefaultSyntax = syntax
   data = {
      'probe': 'PROBE',
      'vrf': 'Configure VRF',
      'default': 'enable ICMP PROBE responses in the default VRF',
   }
   handler = setIcmpProbeVrfDefault
   noOrDefaultHandler = noIcmpProbeVrfDefault

IcmpProbeMode.addCommandClass( IcmpProbeVrfCmd )

# ------------------------------------------------------------
# in icmp probe config mode:
# [no|default] query-types ( all | { ifname | ifindex | addr | remote } )
# ------------------------------------------------------------

def setIcmpProbeQueryTypes( mode, args ):
   isAll = 'all' in args
   config.allowIfname = isAll or 'ifname' in args
   config.allowIfindex = isAll or 'ifindex' in args
   config.allowAddr = isAll or 'addr' in args
   config.allowRemote = isAll or 'remote' in args

def noIcmpProbeQueryTypes( mode, args ):
   isAll = 'all' in args or not set( args ).intersection(
         set( ( 'all', 'ifname', 'ifindex', 'addr', 'remote' ) ) )
   if isAll or 'ifname' in args:
      config.allowIfname = False
   if isAll or 'ifindex' in args:
      config.allowIfindex = False
   if isAll or 'addr' in args:
      config.allowAddr = False
   if isAll or 'remote' in args:
      config.allowRemote = False

class IcmpProbeQueryTypes( CliCommand.CliCommandClass ):
   syntax = 'query-types ( all | { ifname | ifindex | addr | remote } )'
   noOrDefaultSyntax = 'query-types [ all | { ifname | ifindex | addr | remote } ]'
   data = {
      'query-types': 'Query types that are permitted',
      'all': 'Allow all query types',
      'ifname': 'Allow query by interface name',
      'ifindex': 'Allow query by interface index',
      'addr': 'Allow query by interface address',
      'remote': 'Allow query by remote node address',
   }
   handler = setIcmpProbeQueryTypes
   noOrDefaultHandler = noIcmpProbeQueryTypes

IcmpProbeMode.addCommandClass( IcmpProbeQueryTypes )

class ShowProbe( ShowCommand.ShowCliCommandClass ):
   syntax = 'show icmp probe'
   data = {
         'icmp': IcmpResponderCliLib.matcherIcmp,
         'probe': 'PROBE',
         }
   cliModel = "ProbeModel.ProbeResponderVrfModel"
   handler = "ProbeCli.showProbe"

if toggleIcmpProbeResponderEnabled():
   BasicCli.addShowCommandClass( ShowProbe )

def Plugin( entityManager ):
   global config

   config = ConfigMount.mount( entityManager, "icmp/probe/config",
                               "IcmpResponder::ProbeResponder::Config", "w" )
