#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
import os
import re
import AgentDirectory
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.ClockCli as ClockCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.PtpCliModel import (
   PtpClock, PtpClockQuality, PtpIntfSummary, PtpIntfVlanSummary, PtpSummary )
from CliPlugin.PtpSwitchportModelet import (
   matcherAnnounce, matcherDelayMechanism, matcherDelayReq, matcherInterval,
   matcherPdelayNeighborThreshold, matcherPdelayReq,
   matcherTimeout, matcherTransport )
import ConfigMount
import LazyMount
import ShowCommand
from PyWrappers.LinuxPtp import ( phc_ctl, pmc )
import Tac
from TypeFuture import TacLazyType

Ptp4lUdsAddress = TacLazyType( 'Ptp::ManagementIntfConfig' ).udsAddress

#-------------------------------------------------------------------------------
# This module implements Management Plane PTP configuration.
# In particular, it provides:
#
# Management Interface Configuration
# [no|default] ptp enable
# [no|default] ptp announce interval
# [no|default] ptp announce timeout
# [no|default] ptp delay-mechanism p2p|e2e|auto
# [no|default] ptp delay-req interval
# [no|default] ptp domain
# [no|default] ptp pdelay-neighbor-threshold
# [no|default] ptp pdelay-req interval
# [no|default] ptp priority1
# [no|default] ptp priority2
# [no|default] ptp transport ipv4|ipv6|layer2
# [no|default] ptp ttl
#
# Show command
# show ptp management-plane

PtpIntfAttrCmd = namedtuple( "PtpIntfAttrCmd", "attr,syntax,defaultValue" )
ptpIntfAttrCmds = (
   PtpIntfAttrCmd( 'logAnnounceInterval', 'ptp announce interval', None ),
   PtpIntfAttrCmd( 'logMinDelayReqInterval', 'ptp delay-req interval', None ),
   PtpIntfAttrCmd( 'logMinPdelayReqInterval', 'ptp pdelay-req interval', None ),
   PtpIntfAttrCmd( 'linkDelayThreshold', 'ptp pdelay-neighbor-threshold', None ),
   PtpIntfAttrCmd( 'announceReceiptTimeout', 'ptp announce timeout', None ),
   PtpIntfAttrCmd( 'delayMechanism', 'ptp delay-mechanism', None ),
   PtpIntfAttrCmd( 'transportMode', 'ptp transport', None ),
   PtpIntfAttrCmd( 'domainNumber', 'ptp domain', None ),
   PtpIntfAttrCmd( 'priority1', 'ptp priority1', None ),
   PtpIntfAttrCmd( 'priority2', 'ptp priority2', None ),
   PtpIntfAttrCmd( 'ttl', 'ptp ttl', None ),
   PtpIntfAttrCmd( 'enabled', 'ptp enable', False ),
)

CliConstants = TacLazyType( 'Ptp::Constants' )

ptpConfig = None
phyEthtoolConfig = None
ethIntfPhyStatusDir = None

# NOTE: Some of the commands defined here should be guarded on Mako platforms, but
# using some entity or utility residing in a Mako package will create an unwanted
# dependency. Thus, we just check for the existence of the MakoCentral agent
# which is always present in Mako platforms.
def isMako( sysname ):
   return AgentDirectory.agent( sysname, 'MakoCentral' ) or \
      os.environ.get( 'SIMULATION_MAKO_PRODUCT' )

def getPtpCapablePhyIntfNames():
   return list( set( [ p.intfId
            for p in phyEthtoolConfig.phy.values()
            if p.ptpHardwareTimestampingSupported ] + [
            intfId
            for intfId in ethIntfPhyStatusDir.intfStatus.keys()
            if intfId.lower().startswith( 'ptp' ) ] ) )

def intfPtpSupported( intfId=None ):
   def _isMakoPtpIntfId( intfId ):
      return intfId.lower().startswith( 'ptp' )

   # Ptp interfaces are always supported by definition
   if intfId and _isMakoPtpIntfId( intfId ):
      return True

   # Check if ptp is supported on a given management port.
   # If no intfId is specified, check if ptp is supported on any management port.
   if not phyEthtoolConfig:
      return False

   supportedIntfs = getPtpCapablePhyIntfNames()

   if not supportedIntfs:
      return False
   # We move the one restriction down to enabled; it's fine to do configuration
   # at any time as that is saved to sysdb, with only the enabled config
   # synced to ptp4l's config
   return not intfId or intfId in supportedIntfs

def ptpManagementPortIntfGuard( mode, token ):
   if intfPtpSupported( intfId=mode.intf.name ):
      return None
   return CliParser.guardNotThisPlatform

def ptpManagementPlaneGuard( mode, token ):
   if not isMako( mode.sysname ) and intfPtpSupported():
      return None
   return CliParser.guardNotThisPlatform

# This is for management port interface config
ptpMatcherForManagementPortConfig = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Precision Time Protocol',
   guard=ptpManagementPortIntfGuard )

def setManagementPortIntfAttr( intf, attr, value ):
   intfConfig = ptpConfig.managementIntfConfig.get( intf )
   if not intfConfig:
      intfConfig = ptpConfig.newManagementIntfConfig( intf, False )
   if value is None:
      # set to default, use mode 1588V2 if required
      value = getattr( CliConstants, attr + 'Default', None )
      if value is None:
         value = getattr( CliConstants, attr + 'Default1588V2', None )
   setattr( intfConfig, attr, value )

def getEnabledIntf():
   intfConfig = ptpConfig.managementIntfConfig
   enabledIntfs = [ intf
                    for intf in intfConfig.values()
                    if intf.enabled ]
   if enabledIntfs:
      return enabledIntfs[ 0 ]
   else:
      return None

class PtpManagementPortModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
               mode.intf.name.startswith( ( "Management", "Ptp" ) ) )

IntfCli.IntfConfigMode.addModelet( PtpManagementPortModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ptp domain VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortDomainValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp domain VALUE'
   noOrDefaultSyntax = 'ptp domain ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'domain': 'Set the PTP domain',
      'VALUE': CliMatcher.IntegerMatcher(
         CliConstants.minDomainNumberPtp4l,
         CliConstants.maxDomainNumberPtp4l,
         helpdesc='The PTP domain number' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'domainNumber', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'domainNumber', CliConstants.domainNumberDefaultPtp4l )

PtpManagementPortModelet.addCommandClass( PtpManagementPortDomainValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp priority1 VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortPriority1ValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp priority1 VALUE'
   noOrDefaultSyntax = 'ptp priority1 ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'priority1': 'Set the value of priority1',
      'VALUE': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Priority' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'priority1', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'priority1', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortPriority1ValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp priority2 VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortPriority2ValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp priority2 VALUE'
   noOrDefaultSyntax = 'ptp priority2 ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'priority2': 'Set the value of priority2',
      'VALUE': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Priority' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'priority2', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'priority2', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortPriority2ValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp enable
#--------------------------------------------------------------------------------
class PtpManagementPortEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp enable'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'enable': 'Enable PTP on this port',
   }
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'enabled', False )

   @staticmethod
   def handler( mode, args ):
      del args
      # first make sure no other intf is enabled before starting
      # previous code assumes that this cli belongs to a modelet which is guarded
      # such that only ma1 could be enabled, so this check still succeeds
      ptpEnabledIntf = getEnabledIntf()
      if ptpEnabledIntf and mode.intf.name != ptpEnabledIntf.intf:
         # pylint: disable-next=consider-using-f-string
         mode.addErrorAndStop( "Can't enable PTP as interface {} is already enabled"
                               .format( ptpEnabledIntf.intf ) )

      if mode.intf.name not in getPtpCapablePhyIntfNames():
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( "The PTP configuration will not take effect until "
                          "the interface {} is created.".format( mode.intf.name ) )

      setManagementPortIntfAttr( mode.intf.name, 'enabled', True )

PtpManagementPortModelet.addCommandClass( PtpManagementPortEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce interval VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortAnnounceIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce interval VALUE'
   noOrDefaultSyntax = 'ptp announce interval ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'announce': matcherAnnounce,
      'interval': matcherInterval,
      'VALUE': CliMatcher.IntegerMatcher(
         -3, 4, helpdesc='Log of announce interval (secs)' )
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logAnnounceInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logAnnounceInterval', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortAnnounceIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce timeout TIMEOUT
#--------------------------------------------------------------------------------
class PtpManagementPortAnnounceTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce timeout TIMEOUT'
   noOrDefaultSyntax = 'ptp announce timeout ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'announce': matcherAnnounce,
      'timeout': matcherTimeout,
      'TIMEOUT': CliMatcher.IntegerMatcher(
         2, 255, helpdesc='Announce interval timeout multiplier' )
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'announceReceiptTimeout', args[ 'TIMEOUT' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'announceReceiptTimeout', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortAnnounceTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-mechanism VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortDelayMechanismCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-mechanism VALUE'
   noOrDefaultSyntax = 'ptp delay-mechanism ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'delay-mechanism': matcherDelayMechanism,
      'VALUE': CliMatcher.EnumMatcher( {
         'e2e': 'Delay req/resp mechanism',
         'p2p': 'Peer-to-peer mechanism',
         #'auto': 'Configure delay mechanism automatically',
      } )
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'delayMechanism',
      { 'e2e': 'e2e', 'p2p': 'p2p', 'auto': 'autoConfigure' }[ args[ 'VALUE' ] ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'delayMechanism', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortDelayMechanismCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortDelayReqIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-req interval VALUE'
   noOrDefaultSyntax = 'ptp delay-req interval ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'delay-req': matcherDelayReq,
      'interval': matcherInterval,
      'VALUE': CliMatcher.IntegerMatcher(
         CliConstants.minlogMinDelayReqInterval,
         CliConstants.maxlogMinDelayReqInterval,
         helpdesc='Log of delay req interval (secs)' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logMinDelayReqInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logMinDelayReqInterval', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortDelayReqIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-neighbor-threshold VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortPdelayNeighborThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-neighbor-threshold VALUE'
   noOrDefaultSyntax = 'ptp pdelay-neighbor-threshold ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'pdelay-neighbor-threshold': matcherPdelayNeighborThreshold,
      'VALUE': CliMatcher.IntegerMatcher( 0, 10 **9,
         helpdesc='Threshold in nanoseconds' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'linkDelayThreshold', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'linkDelayThreshold', None )

PtpManagementPortModelet.addCommandClass(
   PtpManagementPortPdelayNeighborThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortPdelayReqIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-req interval VALUE'
   noOrDefaultSyntax = 'ptp pdelay-req interval ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'pdelay-req': matcherPdelayReq,
      'interval': matcherInterval,
      'VALUE': CliMatcher.IntegerMatcher(
         CliConstants.minlogMinPdelayReqInterval1588V2,
         CliConstants.maxlogMinPdelayReqInterval1588V2,
         helpdesc='Log of delay req interval (secs)' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logMinPdelayReqInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'logMinPdelayReqInterval', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortPdelayReqIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp transport TRANSPORT
#--------------------------------------------------------------------------------
class PtpManagementPortTransportCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp transport TRANSPORT'
   noOrDefaultSyntax = 'ptp transport ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'transport': matcherTransport,
      'TRANSPORT': CliMatcher.EnumMatcher( {
         'layer2': 'Layer 2 messaging',
         'ipv4': 'Ipv4 messaging',
         'ipv6': 'Ipv6 messaging',
      } )
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'transportMode', args[ 'TRANSPORT' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'transportMode', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortTransportCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp ttl VALUE
#--------------------------------------------------------------------------------
class PtpManagementPortTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp ttl VALUE'
   noOrDefaultSyntax = 'ptp ttl ...'
   data = {
      'ptp': ptpMatcherForManagementPortConfig,
      'ttl': 'Set the TTL of PTP packets',
      'VALUE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL of PTP packets' ),
   }
   handler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'ttl', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: setManagementPortIntfAttr(
      mode.intf.name, 'ttl', None )

PtpManagementPortModelet.addCommandClass( PtpManagementPortTtlCmd )

#-------------------------------------------------------------------------------
# show commands
#-------------------------------------------------------------------------------
ptpMatcherForManagementPlaneShow = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Show Precision Time Protocol information',
   guard=ptpManagementPlaneGuard )

pmcCmds = {
   'CURRENT_DATA_SET' : ( 'stepsRemoved', 'offsetFromMaster', 'meanPathDelay' ),
   'DEFAULT_DATA_SET' : (
      'numberPorts', 'priority1', 'clockClass', 'clockAccuracy',
      'offsetScaledLogVariance', 'priority2', 'clockIdentity', 'domainNumber' ),
   'PARENT_DATA_SET' : ( 'parentPortIdentity', 'grandmasterIdentity' ),
   'PORT_DATA_SET' : ( 'portIdentity', 'portState' ),
   'TIME_PROPERTIES_DATA_SET' : ( 'currentUtcOffset', 'leap59', 'leap61' ),
}

def getPmcInfo( pmcId, ic=None ):
   fields = pmcCmds[ pmcId ]
   d = {}
   try:
      cmd = 'GET ' + pmcId
      ic = ic or getEnabledIntf()
      domain = ic.domainNumber if ic else 0
      out = Tac.run( [ pmc(), '-b', '0', '-u', '-d', str( domain ), '-s',
                       Ptp4lUdsAddress, cmd ],
                     stdout=Tac.CAPTURE, stderr=Tac.DISCARD, asRoot=True )
   except Tac.SystemCommandError:
      return d
   # pylint: disable-next=consider-using-f-string
   m = re.search( r'([-\w.]+) seq \d+ RESPONSE MANAGEMENT %s' % pmcId, out )
   if m:
      d[ 'portIdentity' ] = m.group( 1 )
   else:
      return d
   for f in fields:
      # pylint: disable-next=consider-using-f-string
      d[ f ] = re.search( r'%s\s+([-.\w]+)\n' % f, out ).group( 1 )
   return d

def getPhcOffset( intfName ):
   offset = None
   output = Tac.run( [ phc_ctl(), intfName, 'cmp', '-q' ], stdout=Tac.CAPTURE,
                     stderr=Tac.DISCARD, asRoot=True )
   m = re.search( r'offset from CLOCK_REALTIME is (\-?\d+)ns', output )
   if m:
      offsetTai = int( m.group( 1 ) )
      tpDS = getPmcInfo( 'TIME_PROPERTIES_DATA_SET' )
      if tpDS:
         offset = offsetTai + int( tpDS[ 'currentUtcOffset' ] ) * 10 ** 9
   return offset

def showPtp4lClock( ic ):
   model = PtpClock()
   if ic and ic.enabled:
      defaultDS = getPmcInfo( 'DEFAULT_DATA_SET', ic )
      if not defaultDS:
         return model

      currentDS = getPmcInfo( 'CURRENT_DATA_SET', ic )
      if not currentDS:
         return model

      model.ptpMode = 'ptpBoundaryClock'
      ptpOrdinaryOrBC = PtpClock.PtpOrdinaryOrBoundaryClock()
      ptpOrdinaryOrBC.clockIdentity = defaultDS[ 'clockIdentity' ]
      ptpOrdinaryOrBC.domainNumber = int( defaultDS[ 'domainNumber' ] )
      ptpOrdinaryOrBC.numberPorts = int( defaultDS[ 'numberPorts' ] )
      ptpOrdinaryOrBC.priority1 = int( defaultDS[ 'priority1' ] )
      ptpOrdinaryOrBC.priority2 = int( defaultDS[ 'priority2' ] )

      clockQuality = PtpClockQuality()
      clockQuality.clockClass = int( defaultDS[ 'clockClass' ] )
      clockQuality.accuracy = int( defaultDS[ 'clockAccuracy' ], 16 )
      clockQuality.offsetScaledLogVariance = int(
         defaultDS[ 'offsetScaledLogVariance' ], 16 )
      ptpOrdinaryOrBC.clockQuality = clockQuality

      bcProperties = \
         PtpClock.PtpOrdinaryOrBoundaryClock.PtpBoundaryClockProperties()
      bcProperties.stepsRemoved = int( currentDS[ 'stepsRemoved' ] )
      bcProperties.offsetFromMaster = int(
         float( currentDS[ 'offsetFromMaster' ] ) )
      bcProperties.meanPathDelay = int( float( currentDS[ 'meanPathDelay' ] ) )

      ptpOrdinaryOrBC.boundaryClockProperties = bcProperties
      model.ptpOrdinaryOrBoundaryClock = ptpOrdinaryOrBC

   return model

def showPtp4lSummary( ic ):
   model = PtpSummary()
   clock = showPtp4lClock( ic )
   if not clock.ptpOrdinaryOrBoundaryClock:
      return model

   parentDS = getPmcInfo( 'PARENT_DATA_SET', ic )
   if not parentDS:
      return model

   model.ptpMode = 'ptpBoundaryClock'
   clockSummary = model.PtpClockSummary()
   clockSummary.numberOfSlavePorts = 0
   clockSummary.numberOfMasterPorts = 0

   clockSummary.gmClockIdentity = parentDS[ 'grandmasterIdentity' ]

   clockSummary.clockIdentity = clock.ptpOrdinaryOrBoundaryClock.clockIdentity
   clockSummary.stepsRemoved = \
      clock.ptpOrdinaryOrBoundaryClock.boundaryClockProperties.stepsRemoved
   clockSummary.meanPathDelay = \
      clock.ptpOrdinaryOrBoundaryClock.boundaryClockProperties.meanPathDelay
   clockSummary.offsetFromMaster = \
      clock.ptpOrdinaryOrBoundaryClock.boundaryClockProperties.offsetFromMaster

   model.ptpClockSummary = clockSummary

   model.ptpIntfSummaries = {}
   if ic and ic.enabled:
      portDS = getPmcInfo( 'PORT_DATA_SET', ic )
      if not portDS:
         return model

      intf = ic.intf
      intfSummary = PtpIntfSummary()
      intfSummary.portState = {
         'NONE' : 'psInitializing',
         'INITIALIZING' : 'psInitializing',
         'FAULTY': 'psFaulty',
         'DISABLED' : 'psDisabled',
         'LISTENING' : 'psListening',
         'PRE_MASTER' : 'psPreMaster',
         'MASTER' : 'psMaster',
         'PASSIVE' : 'psPassive',
         'UNCALIBRATED' : 'psUncalibrated',
         'SLAVE' : 'psSlave',
         'GRAND_MASTER' : 'psDisabled' } [ portDS[ 'portState' ] ]

      if intfSummary.portState == 'psMaster':
         clockSummary.numberOfMasterPorts += 1
      elif intfSummary.portState == 'psSlave':
         clockSummary.numberOfSlavePorts += 1
         clockSummary.slavePort = intf
         # Offset b/w system clock and phc only makes sense for hardware timestamping
         if ( ptpConfig.systemClockSourceIntf == intf and
              ClockCli.clockConfig.source == 'ptp' and
              ic.timeStampingMode == 'hardware' ):
            clockSummary.systemClockOffset = getPhcOffset(
               ethIntfPhyStatusDir.intfStatus[ intf ].deviceName
            )


      intfSummary.transportMode = ic.transportMode
      intfSummary.delayMechanism = { 'e2e': 'e2e',
                                     'p2p': 'p2p',
                                     'autoConfigure': 'auto' }[ ic.delayMechanism ]
      model.ptpIntfSummaries[ intf ] = PtpIntfVlanSummary( interface=intf )
      model.ptpIntfSummaries[ intf ].ptpIntfVlanSummaries.append( intfSummary )

   return model

def showManagementPlaneSummary( mode ):
   ic = getEnabledIntf()
   if not ic or not ic.intf.lower().startswith( 'ma' ):
      return showPtp4lSummary( None )
   return showPtp4lSummary( ic )

#-------------------------------------------------------------------------------
# show ptp management-plane
#-------------------------------------------------------------------------------
class PtpManagementPlaneShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp management-plane'
   data = {
      'ptp': ptpMatcherForManagementPlaneShow,
      'management-plane': 'Show management plane PTP information',
   }
   handler = lambda mode, args: showManagementPlaneSummary( mode )
   cliModel = PtpSummary

BasicCli.addShowCommandClass( PtpManagementPlaneShowCmd )

# ------------------------------------------------------------------------------
# PtpIntfJanitor ( interface cleanup )
# ------------------------------------------------------------------------------
class PtpIntfJanitor( IntfCli.IntfDependentBase ):

   def destroy( self ):
      IntfCli.IntfDependentBase.destroy( self )
      # if there is no interface config object, we delete our ptp config as well
      if self.intf_.name in ptpConfig.managementIntfConfig:
         if not self.intf_.config():
            del ptpConfig.managementIntfConfig[ self.intf_.name ]

   def setDefault( self ):
      if not self.intf_.name in ptpConfig.managementIntfConfig:
         return

      for ptpIntfAttrCmd in ptpIntfAttrCmds:
         setManagementPortIntfAttr( self.intf_.name,
                                    ptpIntfAttrCmd.attr,
                                    ptpIntfAttrCmd.defaultValue )

IntfCli.Intf.registerDependentClass( PtpIntfJanitor )

# ------------------------------------------------------------------------------
# Plugin ( entry point )
# ------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ptpConfig
   global phyEthtoolConfig
   global ethIntfPhyStatusDir
   ptpConfig = ConfigMount.mount(
      entityManager, "sys/time/ptp/config", "Ptp::SystemClockConfig", "w" )
   phyEthtoolConfig = LazyMount.mount(
      # pylint: disable-next=consider-using-f-string
      entityManager, "hardware/cell/%d/phy/ethtool/config" % Cell.cellId(),
      "Hardware::Phy::EthtoolConfig", "r" )
   ethIntfPhyStatusDir = LazyMount.mount(
      entityManager, 'interface/status/eth/phy/all',
      'Interface::AllEthPhyIntfStatusDir', 'r' )
