#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliParser
from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
from CliPlugin import ClockCli, PtpManagementPlaneCli
from CliPlugin import VirtualIntfRule
import ConfigMount
import Tac
from TypeFuture import TacLazyType

#-------------------------------------------------------------------------------
# This module implements PTP system clock configuration. In particular, it provides:
#
# Global Configuration
# [no|default] ptp system-clock source interface INTF
# clock source ptp

ptpConfig = None
TimeStampingMode = TacLazyType( "Ptp::TimeStampingMode" )

intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= PhysicalIntfMatcher( 'Management' )

def ptpSystemClockGuard( mode, token ):
   if PtpManagementPlaneCli.intfPtpSupported():
      return None
   return CliParser.guardNotThisPlatform

# This is for global ptp system clock commands. Reuse ptpMatcherForReset since
# it's just an unguarded ptp keyword
ptpMatcherForSystemClockConfig = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Precision Time Protocol',
   guard=ptpSystemClockGuard )

#-------------------------------------------------------------------------------
# [no|default] ptp system-clock source interface INTF
#-------------------------------------------------------------------------------
def setTimeStamping( intf, tsmode ):
   PtpManagementPlaneCli.setManagementPortIntfAttr( intf,
                                                    'timeStampingMode',
                                                    tsmode )

def setPtpSystemClockSourceInterface( mode, args ):
   nextSourceIntf = args.get( 'INTF', '' )
   if nextSourceIntf.lower().startswith( 'ptp' ):
      setTimeStamping( nextSourceIntf, TimeStampingMode.software )
   else:
      setTimeStamping( nextSourceIntf or ptpConfig.systemClockSourceIntf,
                       TimeStampingMode.hardware )

   ptpConfig.systemClockSourceIntf = Tac.Value( "Arnet::IntfId", nextSourceIntf )

class PtpSystemClockSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp system-clock source interface INTF'
   noOrDefaultSyntax = 'ptp system-clock source interface ...'
   data = {
      'ptp': ptpMatcherForSystemClockConfig,
      'system-clock': 'Configure system clock PTP parameters',
      'source': 'Configure the PTP source for the system clock',
      'interface': 'Configure an interface as the PTP source for the system clock',
      'INTF': intfMatcher,
   }
   handler = setPtpSystemClockSourceInterface
   noOrDefaultHandler = setPtpSystemClockSourceInterface

BasicCliModes.GlobalConfigMode.addCommandClass( PtpSystemClockSourceInterfaceCmd )

#-------------------------------------------------------------------------------
# clock source ptp
#-------------------------------------------------------------------------------
class ClockSourcePtpCommand( CliCommand.CliCommandClass ):
   syntax = 'clock source ptp'
   data = {
      'clock' : ClockCli.configClockMatcher,
      'source' : ClockCli.sourceMatcher,
      'ptp' : ptpMatcherForSystemClockConfig,
   }
   handler = lambda mode, args: ClockCli.doSetClockSource( mode, 'ptp' )

BasicCliModes.GlobalConfigMode.addCommandClass( ClockSourcePtpCommand )

def ptpClockSourceHook( mode ):
   '''
   This method returns the PTP grandmaster that the system-clock is synced to and,
   return an empty string, if ptp is not enabled ( or ) if ptp is enabled without
   any master, then, the medium intf would not be in slave state ( or ) if the
   system-clock is not set to sync. 

   Currently, The system-clock will be synced to a ptp master-clock in two scenarios:
   1. With ptp4l's hardware time stamping, the PHC on a NIC is synced to a
      master-clock, and then phc2sys is used to sync the system-clock to the PHC.
      Hence, ( model.ptpClockSummary.systemClockOffset is not None ) which signify
      the offset b/w system-clock and PHC, is being used as conditional check
   2. With ptp4l's software time stamping, the system-clock is directly synced to
      the master-clock.
      Hence, ( ic.timeStampingMode == 'software' ) which signify the ptp4l's
      timestamping mode, is being used as another conditional check
   '''
   ic = PtpManagementPlaneCli.getEnabledIntf()
   if ic:
      model = PtpManagementPlaneCli.showPtp4lSummary( ic )
      if model.ptpClockSummary and \
         model.ptpClockSummary.slavePort == ic.intf and \
         ( ic.timeStampingMode == 'software' or
           model.ptpClockSummary.systemClockOffset is not None ):
         return f"PTP grandmaster ({model.ptpClockSummary.gmClockIdentity})"
   return ""

ClockCli.registerClockSourceHook( ptpClockSourceHook )

def Plugin( entityManager ):
   global ptpConfig
   ptpConfig = ConfigMount.mount(
      entityManager, "sys/time/ptp/config", "Ptp::SystemClockConfig", "w" )
