# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import re
from CliModel import Dict, List, Enum, Int, Float, Model

configStates = Tac.Type( 'PtpTimeSync::ConfiguredState' )
operationalStates = Tac.Type( 'PtpTimeSync::OperationalState' )
scdStates = Tac.Type( 'PtpTimeSync::ScdState' )
switchStates = Tac.Type( 'PtpTimeSync::SwitchState' )
pidStates = Tac.Type( 'PtpTimeSync::PidState' )

configStateToEnum = {
   configStates.off : "off",
   configStates.on : "on",
}
configEnumToStr = {
   "off" : "Off",
   "on" : "On",
}

operationalStateToEnum = {
   operationalStates.disabled : "disabled",
   operationalStates.enabled : "enabled",
   operationalStates.activeScdProgrammed : "active",
   operationalStates.synchronizing : "synchronizing",
   operationalStates.stable : "stable",
}
statusEnumToStr = {
   "disabled" : "Disabled",
   "enabled" : "Enabled",
   "active" : "Active Master Enabled",
   "synchronizing" : "Synchronizing",
   "stable" : "Stable",
}

scdStateToEnum = {
   scdStates.scdUninitialized : "disabled",
   scdStates.scdTimeSyncDisabled : "disabled",
   scdStates.scdTimeSyncInitialized : "initialized",
   scdStates.scdTimeSyncFailed : "failed",
   scdStates.scdTimeSyncActive : "active",
   scdStates.scdTimeSyncSwitchover : "switchover",
   scdStates.scdTimeSyncStandby : "initializedStandby",
   scdStates.scdTimeSyncStandbySynchronizing : "synchronizingStandby",
   scdStates.scdTimeSyncStandbyStable : "stableStandby",
}
masterEnumToStr = {
   "disabled" : "Disabled",
   "failed" : "Failed",
   "initialized" : "Initialized",
   "active" : "Active Master",
   "initializedStandby" : "Initialized Standby Master",
   "synchronizingStandby" : "Synchronizing Standby Master",
   "stableStandby" : "Stable Standby Master",
   "switchover" : "Switchover",
}

switchStateToEnum = {
   switchStates.switchTimeSyncDisabled : "disabled",
   switchStates.switchTimeSyncFailed : "failed",
   switchStates.switchUninitialized : "disabled",
   switchStates.switchTimeSyncInitialized : "initialized",
   switchStates.switchSynchronizing : "synchronizing",
   switchStates.switchTimeSyncStable : "stable",
   switchStates.switchTimeSyncSwitchover : "switchover",
}
slaveEnumToStr = {
   "disabled" : "Disabled",
   "failed" : "Failed",
   "initialized" : "Initialized",
   "synchronizing" : "Synchronizing",
   "stable" : "Stable",
   "switchover" : "Switchover",
}

pidStateToEnum = {
   pidStates.servoDisabled : "servoDisabled",
   pidStates.watchSkew : "watchSkew",
   pidStates.adjustSkew : "adjustSkew",
   pidStates.jumpClock : "jumpClock",
   pidStates.runServo : "runServo",
}

pidStateEnumToStr = {
   "servoDisabled" : "disabled",
   "watchSkew" : "measuring clock skew",
   "adjustSkew" : "setting skew compensation",
   "jumpClock" : "setting offset compensation",
   "runServo" : "dynamic adjustment",
}

globalStr = "Global"
titleStr = " Hardware Time Synchronization Information:"
configuredStateStr = "Configured State"
hbStr = "Configured Heartbeat Interval (in ms)"
localCounterStr = "Latest Local Counter"
localTodStr = "Latest Local Time of Day (ToD)"
masterTodStr = "Latest Master Time of Day (ToD)"
stateStr = "Operational State"
pidStateStr = "Servo State"
updatesStr = "Sync Update(s)"
phaseDifferencesStr = "Last 4 phase differences (in ns)"
timeToStabilityStr = "Time to stability (in s)"
crcErrorsStr = "CRC Error(s)"
timeoutErrorsStr = "Timeout Error(s)"
invalidTimestampErrorsStr = "Invalid Timestamp Error(s)"
fifoRetryLimitExceededStr = "FIFO Retry Limit Exceeded Error(s)"
slaveFreqErrorsStr = "Slave Frequency Error(s)"
masterSkewStr = "Active Master Skew"
retriesLeftSelfTestStr = "Retries Left after self test"

delim = " : "
defaultStr = "N/A"

class MasterModel( Model ):
   state = Enum( values=list( masterEnumToStr ), help=stateStr )
   pidState = Enum( values=list( pidStateEnumToStr ), help=pidStateStr )
   localTod = Int( help=localTodStr, optional=True )
   masterTod = Int( help=masterTodStr, optional=True )
   phaseDifferences = List( valueType=int, help=phaseDifferencesStr,
                            optional=True )
   timeToStability = Int( help=timeToStabilityStr, optional=True )
   updates = Int( help=updatesStr, optional=True )
   skew = Float( help=masterSkewStr, optional=True )

   def render( self ):
      print( stateStr + delim + masterEnumToStr[ self.state ] )
      if self.masterTod is not None:
         print( masterTodStr + delim + "0x%x" % self.masterTod )
      if self.localTod is not None:
         print( localTodStr + delim + "0x%x" % self.localTod )
      if self.phaseDifferences is not None:
         msg = phaseDifferencesStr + delim
         if not self.phaseDifferences:
            msg += defaultStr
         else:
            for phase in self.phaseDifferences:
               msg = "%s%d " % ( msg, phase )
         print( msg )
      if self.timeToStability is not None:
         print( timeToStabilityStr + delim + str( self.timeToStability ) )
      if self.updates is not None:
         print( updatesStr + delim + str( self.updates ) )
      if self.state not in ( 'synchronizingStandby', 'stableStandby' ):
         if self.pidState is not None:
            print( pidStateStr + delim + pidStateEnumToStr[ self.pidState ] )
         if self.skew is not None:
            print( masterSkewStr + delim + str( self.skew ) )
      print()

class MastersModel( Model ):
   masters = Dict( valueType=MasterModel,
                   help="A Mapping between a TimeSync Master and its time "
                        "synchronization information" )

   def render( self ):
      for name in sorted( self.masters.keys() ):
         master = self.masters[ name ]
         title = name + titleStr
         hline = '-' * len( title )
         print( title )
         print( hline )
         master.render()

class SlaveModel( Model ):
   state = Enum( values=list( slaveEnumToStr ), help=stateStr )
   localCounter = Int( help=localCounterStr, optional=True )
   masterTod = Int( help=masterTodStr, optional=True )
   updates = Int( help=updatesStr, optional=True )
   phaseDifferences = List( valueType=int, help=phaseDifferencesStr,
                            optional=True )
   timeToStability = Int( help=timeToStabilityStr, optional=True )
   crcErrors = Int( help=crcErrorsStr, optional=True )
   slaveFreqErrors = Int( help=slaveFreqErrorsStr, optional=True )
   invalidTimestampErrors = Int( help=invalidTimestampErrorsStr, optional=True )
   fifoRetryLimitExceeded = Int( help=fifoRetryLimitExceededStr, optional=True )
   timeoutErrors = Int( help=timeoutErrorsStr, optional=True )
   retriesLeftSelfTest = Int( help=retriesLeftSelfTestStr, optional=True )

   def render( self ):
      print( stateStr + delim + slaveEnumToStr[ self.state ] )
      if self.localCounter is not None:
         print( localCounterStr + delim + "0x%x" % self.localCounter )
      if self.masterTod is not None:
         print( masterTodStr + delim + "0x%x" % self.masterTod )
      if self.phaseDifferences is not None:
         msg = phaseDifferencesStr + delim
         if not self.phaseDifferences:
            msg += defaultStr
         else:
            for phase in self.phaseDifferences:
               msg = "%s%d " % ( msg, phase )
         print( msg ) 
      if self.timeToStability is not None:
         print( timeToStabilityStr + delim + str( self.timeToStability ) )
      if self.updates is not None:
         print( updatesStr + delim + str( self.updates ) )
      if self.crcErrors is not None:
         print( crcErrorsStr + delim + str( self.crcErrors ) )
      if self.timeoutErrors is not None:
         print( timeoutErrorsStr + delim + str( self.timeoutErrors ) )
      if self.invalidTimestampErrors is not None:
         print( invalidTimestampErrorsStr +
                delim + str( self.invalidTimestampErrors ) )
      if self.fifoRetryLimitExceeded is not None:
         print( fifoRetryLimitExceededStr  + delim +
                str( self.fifoRetryLimitExceeded ) )
      if self.retriesLeftSelfTest is not None:
         print( retriesLeftSelfTestStr + delim + str( self.retriesLeftSelfTest ) )
      if self.slaveFreqErrors is not None:
         print( slaveFreqErrorsStr + delim + str( self.slaveFreqErrors ) )
      print()

slaveNameRe = re.compile( "^[A-Za-z]+([0-9]+)(/([0-9]+)){0,1}$" )

def slaveKey( slaveName ):
   m = slaveNameRe.match( slaveName )
   if m:
      # Sort based only on slot and unitId, name doesn't matter.
      moduleOrPrimaryId = int( m.group( 1 ) )
      secondaryId = int( m.group( 3 ) or "0" )
      return ( moduleOrPrimaryId, secondaryId )
   return slaveName

class SlavesModel( Model ):
   slaves = Dict( valueType=SlaveModel,
                  help="A Mapping between a TimeSync Slave and its time "
                       "synchronization information" )

   def render( self ):
      for name in sorted( self.slaves.keys(), key=slaveKey ):
         slave = self.slaves[ name ]
         title = name + titleStr
         hline = '-' * len( title )
         print( title )
         print( hline )
         slave.render()

class TimeSyncModel( Model ):
   configuredState = Enum( values=list( configEnumToStr ),
                           help="Configured State" )
   operationalState = Enum( values=list( statusEnumToStr ),
                            help="Operational State" )
   syncInterval = Float( help="Synchronization Interval (in milliseconds)" )
   masters = Dict( valueType=MasterModel,
                   help="A Mapping between a TimeSync Master and its time "
                        "synchronization information" )
   slaves = Dict( valueType=SlaveModel,
                  help="A Mapping between a TimeSync Slave and its time "
                       "synchronization information" )

   def render( self ):
      title = globalStr + titleStr
      hline = '-' * len( title )
      print( title )
      print( hline )
      print( configuredStateStr + delim + configEnumToStr[ self.configuredState ] )
      print( stateStr + delim + statusEnumToStr[ self.operationalState ] )
      print( hbStr + delim + "%d" % self.syncInterval )
      print()
      MastersModel( masters=self.masters ).render()
      SlavesModel( slaves=self.slaves ).render()
