# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements PVLAN configuration.  In particular, it provides:
# -  the "[no] private-vlan { isolated|community } primary vlan" command
# -  the "show vlan private-vlan" command
# -  the "show pvlan mapping interfaces" command
# -  the "[no] switchport pvlan mapping" command
# -  the "[no] switchport trunk private-vlan secondary" command
# -  the "[no] pvlan mapping" command
#-------------------------------------------------------------------------------

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.VlanCli import VlanConfigMode, VlanSet
from CliPlugin.VlanCli import SwitchportModelet, vlanMappingSupportedGuard
from CliPlugin.VlanCli import vlanIdMatcher
from CliPlugin.VlanIntfCli import VlanIntfModelet
from EbraLib import privateVlanAllowed
import Intf
import LazyMount
import MultiRangeRule
import ShowCommand

gv = CliGlobal.CliGlobal( bridgingHwCapabilities=None,
                          epochStatus=None )

def bridgingSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.bridgingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Rule to match the token class <vlan_set>, returning a corresponding VlanSet
# object.
#-------------------------------------------------------------------------------
def vlanIdListFunc( mode, grList ):
   return VlanSet( mode, vlanIds=list( grList.values() ),
                   vlanSetString=str( grList ) )

#-------------------------------------------------------------------------------
# The "[no] private-vlan { isolated|community } primary vlan", in "config-vlan" mode
#
# The full syntax of this command is:
#
#     private-vlan { isolated|community } primary vlan <vlan_id>
#     no private-vlan [ { isolated|community } primary vlan <vlan_id> ]
#     default private-vlan [ { isolated|community } primary vlan <vlan_id> ]
#-------------------------------------------------------------------------------
def privateVlansSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.privateVlansSupported:
      if privateVlanAllowed( gv.epochStatus ):
         return None
   return CliParser.guardNotThisPlatform

nodePrivateVlan = CliCommand.guardedKeyword( 'private-vlan',
   helpdesc='Configure a private VLAN',
   guard=privateVlansSupportedGuard )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
   helpdesc='Identifier for a Virtual LAN' )

class PrivateVlanPrimaryVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'private-vlan PVLAN_TYPE primary vlan VLAN_ID'
   noOrDefaultSyntax = 'private-vlan ...'
   data = {
      'private-vlan': nodePrivateVlan,
      'vlan': matcherVlan,
      'PVLAN_TYPE': CliMatcher.EnumMatcher( {
            'isolated': 'Configure the VLAN as an isolated private VLAN',
            'community': 'Configure the VLAN as a community private VLAN',
      } ),
      'primary': 'Configure the associated primary VLAN',
      'VLAN_ID': vlanIdMatcher,
   }
   handler = "PvlanCliHandler.setPrivateVlan"
   noOrDefaultHandler = "PvlanCliHandler.noPrivateVlan"

VlanConfigMode.addCommandClass( PrivateVlanPrimaryVlanVlanidCmd )

#-------------------------------------------------------------------------------
# The "show vlan private-vlan" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan private-vlan
#-------------------------------------------------------------------------------
nodePrivateVlanForShow = CliCommand.guardedKeyword( 'private-vlan',
   helpdesc='Private VLAN information',
   guard=privateVlansSupportedGuard )

class VlanPrivateVlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan private-vlan'
   data = {
      'vlan': 'Details on VLAN operation',
      'private-vlan': nodePrivateVlanForShow,
   }
   handler = "PvlanCliHandler.doShowPrivateVlan"
   cliModel = "PvlanModel.VlanPrivateVlan"

BasicCli.addShowCommandClass( VlanPrivateVlanCmd )

#-------------------------------------------------------------------------------
# The "show interfaces private-vlan mapping" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show interfaces private-vlan mapping
#-------------------------------------------------------------------------------
pvlanShowDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher(
      'private-vlan',
      helpdesc='Show interface private VLAN information' ),
   guard=privateVlansSupportedGuard,
   deprecatedByCmd='show pvlan mapping interfaces' )

pvlanShowKw = CliCommand.guardedKeyword( 'pvlan',
                                         'Show interface private VLAN information',
                                         privateVlansSupportedGuard )

mappingShowKw = CliMatcher.KeywordMatcher(
   'mapping',
   helpdesc='Private VLAN mapping information' )

class ShowIntfPVlan( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces private-vlan mapping'
   data = { 'private-vlan' : pvlanShowDeprecated,
            'mapping' : mappingShowKw }
   handler = "PvlanCliHandler.doShowPrivateVlanMapping"

BasicCli.addShowCommandClass( ShowIntfPVlan )

class ShowPVlanMapping( ShowCommand.ShowCliCommandClass ):
   syntax = "show pvlan mapping interfaces [ INTF ]"
   data = {
      'pvlan' : pvlanShowKw,
      'mapping' : mappingShowKw,
      'interfaces' : 'Private VLAN mapping interface information',
      'INTF' : Intf.IntfRange.intfRangeMatcher,
   }

   handler = "PvlanCliHandler.doShowPrivateVlanMapping"

BasicCli.addShowCommandClass( ShowPVlanMapping )

#-------------------------------------------------------------------------------
# The "[no] switchport pvlan mapping" command, in "config-if" mode.
#
# legacy:
# The "[no] switchport private-vlan mapping" command, in "config-if" mode.
#
#
# The full syntax of this command is:
#
#   switchport pvlan mapping <vlanSet>
#   switchport pvlan mapping add <vlanSet>
#   switchport pvlan mapping remove <vlanSet>
#   {no|default} pvlan mapping [ [add|remove] <vlanSet> ]
#-------------------------------------------------------------------------------
privateVlanSet = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'Secondary VLAN IDs of the private VLAN mapping',
   value=vlanIdListFunc )
matcherSwitchport = CliMatcher.KeywordMatcher( 'switchport',
   helpdesc='Set switching mode characteristics' )
nodePvlan = CliCommand.guardedKeyword( 'pvlan',
   helpdesc='Configure a private VLAN',
   guard=privateVlansSupportedGuard )
matcherPrivateVlan = CliMatcher.KeywordMatcher( 'private-vlan',
   helpdesc='Set the private VLAN configuration' )
matcherMapping = CliMatcher.KeywordMatcher( 'mapping',
   helpdesc='Set the private VLAN mapping' )

class SwitchportPrivatevlanMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport ( pvlan | private-vlan ) mapping [ OP ] VLAN_SET'
   noOrDefaultSyntax = 'switchport ( pvlan | private-vlan ) mapping ...'
   data = {
      'switchport': matcherSwitchport,
      'pvlan': nodePvlan,
      'private-vlan': CliCommand.Node( matcher=matcherPrivateVlan,
         deprecatedByCmd='switchport pvlan mapping',
         guard=privateVlansSupportedGuard ),
      'mapping': matcherMapping,
      'OP': CliMatcher.EnumMatcher( {
        'add': 'Add VLANs to the current list',
        'remove': 'Remove VLANs from the current list',
       } ),
      'VLAN_SET': privateVlanSet,
   }

   handler = "PvlanCliHandler.setPrivateVlanMapping"
   noOrDefaultHandler = "PvlanCliHandler.noPrivateVlanMapping"

SwitchportModelet.addCommandClass( SwitchportPrivatevlanMappingCmd )
#-------------------------------------------------------------------------------
# "[no|default] switchport trunk private-vlan secondary" command in "config-if" mode.
#-------------------------------------------------------------------------------
nodeSecondary = CliCommand.guardedKeyword( 'secondary',
   helpdesc='Enable secondary VLAN mapping',
   guard=vlanMappingSupportedGuard )
nodeTrunk = CliCommand.guardedKeyword( 'trunk',
   helpdesc='Set trunking characteristics of the interface',
   guard=bridgingSupportedGuard )

class SwitchportTrunkPrivateVlanSecondaryCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport trunk private-vlan secondary'
   noOrDefaultSyntax = 'switchport trunk private-vlan secondary ...'
   data = {
      'switchport': matcherSwitchport,
      'trunk': nodeTrunk,
      'private-vlan': nodePrivateVlan,
      'secondary': nodeSecondary,
   }
   handler = "PvlanCliHandler.setTrunkPrivateVlanMapping"
   noOrDefaultHandler = "PvlanCliHandler.setTrunkPrivateVlanMapping"

SwitchportModelet.addCommandClass( SwitchportTrunkPrivateVlanSecondaryCmd )

#-------------------------------------------------------------------------------
# new syntax:
# The "[no] pvlan mapping" command, in "config-vlan-if" mode.
#
# legacy:
# The "[no] private-vlan mapping" command, in "config-vlan-if" mode.
#
# The full syntax of this command is:
#
#   pvlan mapping <vlanSet>
#   pvlan mapping add <vlanSet>
#   pvlan mapping remove <vlanSet>
#   {no|default} pvlan mapping [ [add|remove] <vlanSet> ]
#-------------------------------------------------------------------------------
class PvlanMappingCmd( CliCommand.CliCommandClass ):
   syntax = '( pvlan | private-vlan ) mapping [ OP ] VLAN_SET'
   noOrDefaultSyntax = '( pvlan | private-vlan ) mapping ...'
   data = {
      'pvlan': nodePvlan,
      'private-vlan': CliCommand.Node( matcher=matcherPrivateVlan,
         deprecatedByCmd='pvlan mapping',
         guard=privateVlansSupportedGuard ),
      'mapping': matcherMapping,
      'OP': CliMatcher.EnumMatcher( {
         'add': 'Add VLANs to the current list',
         'remove': 'Remove VLANs from the current list',
       } ),
      'VLAN_SET': privateVlanSet,
   }

   handler = "PvlanCliHandler.setSviPrivateVlanMapping"
   noOrDefaultHandler = "PvlanCliHandler.noSviPrivateVlanMapping"

VlanIntfModelet.addCommandClass( PvlanMappingCmd )

def Plugin( entityManager ):
   mount = LazyMount.mount
   gv.bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                           "Bridging::HwCapabilities", "r" )
   gv.epochStatus = mount( entityManager, "hwEpoch/status", "HwEpoch::Status", "r" )

