# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import ( # pylint: disable=unused-import
   # import these two here to allow mocking in cohab test
   switchTimeToUtc,
   utcTimeRelativeToNowStr,
)
import BasicCli
from BasicCliUtil import anyCaseRegex
import CliParser
import CliCommand
import CliToken
import CliMatcher
import ShowCommand
import LazyMount
from CliPlugin import TechSupportCli
import CliPlugin.MplsCli as MplsCli # pylint: disable=consider-using-from-import
from CliPlugin.LdpCli import matcherLdp
from PseudowireLib import (
   ConnectorType,
   validPwNameRegex as validNameRegex,
)

pwConfig = None
pwHwCapability = None

def pwSupportedGuard( mode, token ):
   if pwHwCapability.mplsPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#---------------------------------------------
# Useful tokens for show commands
#---------------------------------------------
matcherPanel = CliMatcher.KeywordMatcher( 'panel',
      helpdesc='Show patch panel information' )
nodePatch = CliCommand.guardedKeyword( 'patch',
      helpdesc='Show patch panel information',
      guard=pwSupportedGuard )

disallowRegex = anyCaseRegex( "(detail|forwarding)" )
showConnNameRegex = f"^(?!{disallowRegex}$){validNameRegex}$"
staticPwNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( [ connKey.name for connKey in pwConfig.connector
                       if connKey.connectorType == ConnectorType.mplsStatic ] ),
      'MPLS static pseudowire name' )
ldpPwNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( [ connKey.name for connKey in pwConfig.connector
                               if connKey.connectorType == ConnectorType.ldp ] ),
      'MPLS LDP pseudowire name' )
mplsStaticPwNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( [ connKey.name for connKey in pwConfig.connector
                       if connKey.connectorType == ConnectorType.mplsStatic ] ),
      'MPLS static pseudowire name' )

def getPwProfiles( mode ):
   return [ connKey.name
            for connKey in pwConfig.connector
            if connKey.connectorType == ConnectorType.ldpAutoDiscoveryProfile ]

ldpPwProfileMatcher = CliMatcher.DynamicNameMatcher( getPwProfiles,
      'MPLS LDP pseudowire profile' )
patchNameMatcher = CliMatcher.DynamicNameMatcher(
                        lambda mode: sorted( pwConfig.patch.keys() ),
                        pattern=showConnNameRegex,
                        helpdesc='Patch name' )
pwIdMatcher = CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF, helpdesc='Pseudowire ID' )

# ---------------------------------------------
# Useful tokens for clear commands
# ---------------------------------------------

nodePseudowire = CliCommand.guardedKeyword( 'pseudowire',
   helpdesc='LDP pseudowire',
   guard=pwSupportedGuard )

#--------------------------------------------------------------------------------
# show patch panel summary
#--------------------------------------------------------------------------------
class PatchPanelSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel summary'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'summary': 'Show patch panel summary',
   }
   cliModel = 'PwaModel.PatchSummary'
   handler = 'PwaCli.patchPanelSummaryCmdHandler'

BasicCli.addShowCommandClass( PatchPanelSummaryCmd )

#--------------------------------------------------------------------------------
# show patch panel [ PATCH_NAME ] [ detail ]
#--------------------------------------------------------------------------------
class PatchPanelDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel [ PATCH_NAME ] [ detail ]'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'PATCH_NAME': patchNameMatcher,
      'detail': 'Show patch panel details',
   }
   cliModel = 'PwaModel.Patches'
   handler = 'PwaCli.patchPanelDetailCmdHandler'

BasicCli.addShowCommandClass( PatchPanelDetailCmd )

#--------------------------------------------------------------------------------
# show patch panel forwarding
#--------------------------------------------------------------------------------
class PatchPanelForwardingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel forwarding'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'forwarding': 'Show patch forwarding information',
   }
   cliModel = 'PwaModel.PatchForwardingTable'
   handler = 'PwaCli.patchPanelForwardingCmdHandler'

BasicCli.addShowCommandClass( PatchPanelForwardingCmd )

# --------------------------------------------------------------------------------
# clear mpls ldp pseudowire pwName
# --------------------------------------------------------------------------------
class ClearMplsLdpPseudowireCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ldp pseudowire pwName'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mpls': MplsCli.mplsMatcherForClear,
      'ldp': matcherLdp,
      'pseudowire': nodePseudowire,
      'pwName': ldpPwNameMatcher,
   }
   handler = 'PwaCli.clearMplsLdpPseudowireCmdHandler'

BasicCli.EnableMode.addCommandClass( ClearMplsLdpPseudowireCmd )

#-------------------------------------------------------------------------------
# Commands to be run for 'show tech-support'
#-------------------------------------------------------------------------------

def showTechCmdsGuard():
   return pwHwCapability.mplsPseudowireSupported and pwConfig.patch

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd(
   '2021-01-20 17:00:00',
   cmds=[
      'show patch panel detail',
      'show agent Pseudowire debug',  # CliPlugin/AgentLibShowCommands.py
   ],
   cmdsGuard=showTechCmdsGuard,
   summaryCmds=[ 'show patch panel summary' ],
   summaryCmdsGuard=showTechCmdsGuard )

#-------------------------------------------------------------------------------
# Mount necessary mountpoints for show commands
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global pwConfig
   global pwHwCapability

   # Pseudowire Cli config in Sysdb
   pwConfig = LazyMount.mount( entityManager, 'pseudowire/config',
                                'Pseudowire::Config', 'r' )

   # Indicates if pseudowire is supported on this platform
   pwHwCapability = LazyMount.mount( entityManager,
                                     "routing/hardware/pseudowire/capability",
                                     "Pseudowire::Hardware::Capability", "r" )
