#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
import os

import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CliPlugin
import CliRangeExpansion
import CliSession
import CommonGuards
import ConfigMount
import FileUrl
import LazyMount
import MultiRangeRule
import Plugins
import QosLib
import Tac
import Tracing
import Url
from Intf.IntfRange import IntfRangeMatcher
from CliMode.Qos import ( IntfTxQueueMode, QosProfileModeBase )
from CliPlugin import ( IntfCli, EthIntfCli, ConfigConvert )
from CliPlugin.IntfCli import ( IntfConfigMode, counterSupportedIntfs )
from CliPlugin.IntfRangeCli import IntfRangeConfigMode
from CliPlugin.QosCliIntfTypes import ( ethOrLagIntfTypes, ethOrLagIntfPrefixes )
from CliPlugin.QosCliModel import ( InterfaceQosModel, IntfSchedulingGroupModel,
                                    SchedulingElementModel, SchedulingModeModel,
                                    IntfSchedulingGroupWithMembersModel,
                                    SchedulingGroupWithMembersModel,
                                    GuaranteedBwModel, ShapeRateModel,
                                    QosProfileAllModel )
from QosLib import ( actionToEnum, fabricIntfName, fabricTokenName, isLagPort,
                     compareAttributeToDefault, tacTrustMode, schedModeEnumToStr,
                     convertTrustModeModelFromStr, tacGuaranteedBwVal,
                     tacSchedulerCompensation, tacShapeRateVal, QOS_CLI_TIMEOUT,
                     guaranteedBwUnitFromEnum, isDefaultDscpPreserveIpMplsEncapMode,
                     isDefaultPolicerPktSizeAdjProfileName, isDefaultTcToCosMap,
                     isDefaultSchedulerCompensationConfig, isDefaultDscpToTcMap,
                     isDefaultSchedulerGroupName, isDefaultExpToTcProfile,
                     isDefaultCosToTcProfile, isDefaultGuaranteedBw,
                     isDefaultShapeRate, isDefaultDscp, isDefaultCos,
                     isDefaultTrustMode, isDefaultIntfConfig, getSubIntfNum,
                     isDefaultTxQueueConfig, shapeRateUnitFromEnum,
                     isDefaultTcToExpMap, isTxQueueNotConfigurable )
from QosCliLib import ( populateGuaranteedBwModel, populatePolicerPktSizeAdjModel,
                        populateSchedulerCompensationModel, populateBurstSizeModel,
                        populateShapeRateModel, QosProfileModelContainer )
from QosTypes import ( tacCos, tacDscp, tacPolicerPktSizeAdj, tacPercent,
                       tacTxQueuePriority, tacBwWeight, tacEcnDelayThreshold,
                       tacPolicerPktSizeAdjProfileName, tacTcToCosMapName,
                       tacDscpToTcMapName, tacExpToTcProfileName, tacFeatureName,
                       tacCosToTcProfileName, tacGuaranteedBwUnit, tacQueueWeight,
                       tacShapeRateUnit, tacHwProgrammingStatus, tacQueueType,
                       tacDirection, tacPfcWatchdogAction, tacTcDpToExpMapName )

__defaultTraceHandle__ = Tracing.Handle( 'QosCliCommon' )
t0 = Tracing.trace0

DYNAMIC_CMAP_EDIT_WARN = 'Editing dynamically created ' \
   '(by agents and not CLI) class map'
CLI_TIMEOUT = 'Operation timed out'

# -----------------------------------------------------------------------------------
# Variables for Qos Common associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosAclHwStatus = None
qosHwStatus = None
qosInputConfig = None
qosConfig = None
qosStatus = None
qosSliceHwStatus = None
subIntfHwStatus = None
qosTmMapping = None
qosGlobalConfig = None
qosInputProfileConfig = None
lagInputConfig = None
qosSchedulerConfig = None
cliQosAclConfig = None
qosHwConfig = None
qosAclSliceHwStatus = None
profileConfigDir = None

# Ecn default weight
defaultWeight = tacQueueWeight.defaultWeight

tacSubIntfId = Tac.Type( 'Arnet::SubIntfId' )
isPortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" ).isPortChannelIntfId

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardIpAcls( mode, token ):
   if qosAclHwStatus.ipAclsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIp6Acls( mode, token ):
   if qosAclHwStatus.ip6AclsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardGlobalEcn( mode, token ):
   if qosHwStatus.globalEcnSupported and \
      qosHwStatus.ecnSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardConfigureGlobalWredAllowEct( mode, token ):
   # Default is False which signify mutually exclusive
   if not qosHwStatus.wredEcnMutuallyExclusive:
      return None
   return CliParser.guardNotThisPlatform

def guardQosRandomDetect( mode, token ):
   if guardGlobalEcn( mode, token ) and \
      guardConfigureGlobalWredAllowEct( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

def guardTxQueueRandomDetect( mode, token ):
   if isinstance( mode, IntfTxQueueConfigMode ):
      if isTxQueueNotConfigurable( qosHwStatus, mode.txQueue.id ):
         return CliParser.guardNotThisPlatform
            
   if isSubIntfConfigMode( mode ) or mode.queueToken == 'mc-txq':
      return CliParser.guardNotThisPlatform
   return None

def guardMacAccessGroupMatch( mode, token ):
   if qosAclHwStatus.matchMacInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpEcnMatch( mode, token ):
   if qosAclHwStatus.dscpEcnMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapQosAction( mode, token ):
   action = actionToEnum( token.lower() )
   if qosAclHwStatus.policyQosSupported and \
      action in qosAclHwStatus.policyQosActionsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPwDecapDscpQosMap( mode, token ):
   if qosHwStatus.pwDecapDscpQosMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpToTcPerInterface( mode, token ):
   if isinstance( mode, IntfConfigMode ):
      # When DscpTc map attachment is processed in IntfConfgMode
      if mode.intf.isSubIntf():
         if not qosHwStatus.dscpToTcPerSubInterfaceSupported:
            return CliParser.guardNotThisPlatform
         return None
      else:
         if not qosHwStatus.dscpToTcPerInterfaceSupported:
            return CliParser.guardNotThisPlatform
         return None

   if not qosHwStatus.dscpToTcPerInterfaceSupported and \
      not qosHwStatus.dscpToTcPerSubInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardDscpToTcSupported( mode, token ):
   # This is for 'show qos map' and 'create DscpTc map' command
   if guardPwDecapDscpQosMap( mode, token ) and \
      guardDscpToTcPerInterface( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

def guardTxQueueSchedulerProfiles( mode, token ):
   if ( qosHwStatus.txQueueSchedulerProfileFixedSupported or
        qosHwStatus.txQueueSchedulerProfileCustomSupported ):
      return None
   return CliParser.guardNotThisPlatform

def guardTxQueueSchedulerProfileFixed( mode, token ):
   if qosHwStatus.txQueueSchedulerProfileFixedSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTxQueueSchedulerProfileCustom( mode, token ):
   if qosHwStatus.txQueueSchedulerProfileCustomSupported:
      return None
   return CliParser.guardNotThisPlatform

def txQueueRangeForNameFn( mode=None, context=None ):
   return ( 0, qosHwStatus.numTxQueueSupported - 1 )

def trafficClassRangeFn( mode=None, context=None ):
   return ( 0, qosHwStatus.numTcSupported - 1 )

def guardShapedLagSubIntfLoadbalance( mode, token ):
   if qosHwStatus.shapedLagSubIntfLoadbalanceSupported:
      return None
   return CliParser.guardNotThisPlatform


# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherSize = CliMatcher.KeywordMatcher( 'size',
      helpdesc='Configure packet size parameters' )
matcherAdjustment = CliMatcher.KeywordMatcher( 'adjustment',
      helpdesc='Configure packet size adjustment' )
matcherPlus = CliMatcher.KeywordMatcher( 'plus',
      helpdesc='Positive adjustment to packet size' )
matcherMinus = CliMatcher.KeywordMatcher( 'minus',
      helpdesc='Negative adjustment to packet size' )
matcherBytes = CliMatcher.KeywordMatcher( 'bytes',
      helpdesc='Packet size adjustment unit bytes' )
matcherAccessGroup = CliMatcher.KeywordMatcher( 'access-group',
      helpdesc='Match with given access group' )
matcherIntf = IntfRangeMatcher( explicitIntfTypes=ethOrLagIntfTypes )
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
      helpdesc='Show QoS status for a specific interface' )
matcherMls = CliMatcher.KeywordMatcher( 'mls', helpdesc='MLS global commands' )
matcherMatch = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Match the access rule specified' )
matcherExpToTcProfileName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.expToTcProfile, helpdesc='Map Name',
      pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
matcherDscpToTcMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.dscpToTcNamedMap, helpdesc='DSCP to Tc Map Name',
      pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
matcherTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Set Traffic-Class value' )
matcherMapExp = CliMatcher.KeywordMatcher( 'exp',
      helpdesc='Map EXP values' )
matcherTrafficClassValue = CliMatcher.DynamicIntegerMatcher(
      trafficClassRangeFn,
      helpdesc='Traffic class value', priority=CliParser.PRIO_HIGH )
matcherTrafficClassList = MultiRangeRule.MultiRangeMatcher(
      rangeFn=trafficClassRangeFn, noSingletons=False,
      helpdesc='Traffic Class (TC) value(s) or range(s) of TC values' )
matcherMapDscp = CliMatcher.KeywordMatcher( 'dscp',
      helpdesc='Map Differentiated Services Code Point (DSCP) values' )
matcherSet = CliMatcher.KeywordMatcher( 'set',
      helpdesc="Set QoS values" )
matcherTxQueueIdForName = CliMatcher.DynamicIntegerMatcher(
      txQueueRangeForNameFn, helpdesc='Transmit queue ID' )
matcherTxQueueConfig = CliMatcher.KeywordMatcher( 'tx-queue',
      helpdesc='Configure transmit queue parameters' )
matcherSchProfileToken = CliMatcher.KeywordMatcher( 'profile',
      helpdesc='Set scheduler profile' )

# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeRandomDetect = CliCommand.guardedKeyword( 'random-detect',
      helpdesc='Set WRED based congestion control parameters (including ECN)',
      guard=guardQosRandomDetect )
nodeTxQueueRandomDetect = CliCommand.guardedKeyword( 'random-detect',
      helpdesc='Set WRED based congestion control parameters (including ECN)',
      guard=guardTxQueueRandomDetect )
nodeMac = CliCommand.guardedKeyword( 'mac',
      helpdesc='Specify MAC Access-groups', guard=guardMacAccessGroupMatch )
nodeMls = CliCommand.Node( matcher=matcherMls, hidden=True )
nodeDscpEcn = CliCommand.guardedKeyword( 'ecn',
      helpdesc='Specify ECN parameter', guard=guardDscpEcnMatch )
nodeDscpToTcMapTo = CliCommand.guardedKeyword( 'to',
      helpdesc="Map to Traffic-Class",
      guard=guardDscpToTcSupported )
nodeInterfacesStandbyGuard = CliCommand.Node( matcher=matcherInterfaces,
      guard=CommonGuards.standbyGuard )
nodeSchedulerToken = CliCommand.guardedKeyword( 'scheduler',
      helpdesc='Set scheduler parameters',
      guard=guardTxQueueSchedulerProfiles )
nodeSchProfileResponsive = CliCommand.guardedKeyword( 'responsive',
      helpdesc='Optimize latency at the expense of burstiness',
      guard=guardTxQueueSchedulerProfileFixed )

# -----------------------------------------------------------------------------------
# Helper functions
# -----------------------------------------------------------------------------------
def weightRangeFn( mode=None, context=None ):
   return ( qosHwStatus.minWeightValue, qosHwStatus.maxWeightValue )

def randomDetectCmdsAdapter( mode, args, argsList ):
   if CliCommand.isNoOrDefaultCmd( args ):
      return
   minThd, maxThd, _ = args[ 'THD_RANGE' ]
   if minThd > maxThd:
      mode.addError( 'Minimum-threshold cannot be more than Maximum-threshold' )
      raise CliParserCommon.AlreadyHandledError()

def getIntfListFromMode( intfMode ):
   if isinstance( intfMode, IntfCli.IntfConfigMode ):
      return [ intfMode.intf.name ]
   else:
      return intfMode.intfList

def isSubIntfConfigMode( mode ):
   if isinstance( mode.parent_, IntfCli.IntfConfigMode ):
      return mode.parent_.intf.isSubIntf()
   elif isinstance( mode.parent_, IntfRangeConfigMode ):
      for intfMode in mode.parent_.individualIntfModes_:
         if intfMode.intf.isSubIntf():
            return True
   return False

def invertIdToName( id_, nameToIdMap, name ):
   for key, value in nameToIdMap.items():
      if value == id_ and key != name:
         del nameToIdMap[ key ]
   if name:
      nameToIdMap[ name ] = id_

class QosSubIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( ethOrLagIntfPrefixes ) and
               mode.intf.isSubIntf() )


IntfCli.IntfConfigMode.addModelet( QosSubIntfModelet )

# From LldpConfigCli
class QosModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( ethOrLagIntfPrefixes ) and
               not mode.intf.isSubIntf() )


IntfCli.IntfConfigMode.addModelet( QosModelet )

# -------------------------------------------------------------------------------
# tx-queue config mode. A new instance of this mode is created when the
# user enters "tx-queue <queue-id>" inside config-if mode.
# -------------------------------------------------------------------------------
class IntfTxQueueConfigMode( IntfTxQueueMode, BasicCli.ConfigModeBase ):
   # ----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   # ----------------------------------------------------------------------------
   name = 'Tx-queue configuration'

   # ----------------------------------------------------------------------------
   # Constructs a new TxQueueMode instance.
   # ----------------------------------------------------------------------------
   def __init__( self, parent, session, tokenQueueType=None, txQueueId=None ):
      assert txQueueId is not None
      self.txQueue = Tac.Value( 'Qos::TxQueue' )
      self.txQueue.id = txQueueId
      if 'tx-queue' == tokenQueueType:
         self.queueToken = 'txq'
         self.txQueue.type = tacQueueType.unknown
      elif 'uc-tx-queue' == tokenQueueType:
         self.queueToken = 'uc-txq'
         self.txQueue.type = tacQueueType.ucq
      elif 'mc-tx-queue' == tokenQueueType:
         self.queueToken = 'mc-txq'
         self.txQueue.type = tacQueueType.mcq
      else:
         assert 0
      assert isinstance( parent, ( IntfCli.IntfConfigMode, IntfRangeConfigMode,
                                  QosProfileMode ) )
      # To support tx-queue mode in intf Range mode, we first create individual modes
      # (per intf queue pair) in IntfTxQueueMode and create a multimode in case of
      # interface range or a single mode in case of single interface. The new mode
      # is entered into, as a child mode in the caller of this function.
      if isinstance( parent, IntfCli.IntfConfigMode ):
         iName = f'if-{parent.intf.shortname}'
         IntfTxQueueMode.__init__( self, ( iName,
                                           self.queueToken,
                                           txQueueId, parent ) )
         BasicCli.ConfigModeBase.__init__( self, parent, session )
      elif isinstance( parent, QosProfileMode ):
         IntfTxQueueMode.__init__( self, ( parent.longModeKey,
                                           self.queueToken,
                                           txQueueId, parent ) )
         BasicCli.ConfigModeBase.__init__( self, parent, session )
      else:
         IntfTxQueueMode.__init__( self, ( parent.longModeKey,
                                           self.queueToken,
                                           txQueueId, parent ) )
         self.individualIntfChildModes_ = []
         for intfMode in parent.individualIntfModes_:
            mode = IntfTxQueueConfigMode( intfMode, session,
                                          tokenQueueType, txQueueId )
            self.individualIntfChildModes_.append( mode )
         self.modeKey = self.queueToken
         self.longModeKey = f'{parent.longModeKey}-{self.queueToken}-{txQueueId}'
         BasicCli.ConfigModeBase.__init__(
            self, parent, session, multiInstance=True,
            multiModes=self.individualIntfChildModes_ )

   def showActive( self ):
      parent = self.parent_
      if not isinstance( parent, IntfRangeConfigMode ):
         BasicCli.ConfigModeBase.showActive( self )
         return
      # If parent mode is IntfRange then run show active for each intf mode
      url = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ) )
      content = url.open().readlines()
      for intfMode in parent.individualIntfModes_:
         BasicCliUtil.showRunningConfigWithFilter( content, self.filterExp(),
                                                   [ intfMode.filterExp() ] )

   def showActiveAll( self, showDetail ):
      parent = self.parent_
      if not isinstance( parent, IntfRangeConfigMode ):
         BasicCli.ConfigModeBase.showActiveAll( self, showDetail )
         return
      # If parent mode is IntfRange then run show active for each intf mode
      url = FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( self ),
                                           showDetail=showDetail )
      content = url.open().readlines()
      for intfMode in parent.individualIntfModes_:
         BasicCliUtil.showRunningConfigWithFilter( content, self.filterExp(),
                                                   [ intfMode.filterExp() ] )

# This modelet is used for ECN only so we test guard here too.
class IntfUcTxQueueModelet( CliParser.Modelet ):
   pass


IntfTxQueueConfigMode.addModelet( IntfUcTxQueueModelet )

# Function that handles tx-queue range
def goToTxQueueRangeMode( mode, tokenQueueType, txQueueIds ):
   ( txQueueId, txQueueMode ) = ( txQueueIds[ 0 ], IntfTxQueueConfigMode )\
                                if len( txQueueIds ) == 1 else ( txQueueIds,
                                                         IntfTxQueueRangeConfigMode )
   childMode = mode.childMode( txQueueMode, tokenQueueType=tokenQueueType,
                               txQueueId=txQueueId )
   mode.session_.gotoChildMode( childMode )

#
# Filters counter-supporting ethernet interfaces from a given interface range.
#
def counterSupportedEthIntfs( mode, intf, mod ):
   return counterSupportedIntfs( mode, intf, mod, intfType=EthIntfCli.EthIntf ) or ()

#
# Filters counter-supporting physical ethernet interfaces from a given interface
# range.
#
def counterSupportedEthPhyIntfs( mode, intf, mod ):
   intfType = EthIntfCli.EthPhyIntf
   return counterSupportedIntfs( mode, intf, mod, intfType=intfType ) or ()

def getIntfShapeRatePercentReference( intfStatus ):
   isSubIntf = tacSubIntfId.isSubIntfId( intfStatus.intfId )
   if not isSubIntf:
      return 'lineRate'
   parentIntfId = tacSubIntfId.parentIntfId( intfStatus.intfId )
   parentStatus = qosStatus.intfStatus.get( parentIntfId )
   if not parentStatus:
      return 'lineRate'
   sgName = intfStatus.schedulerGroupName
   if sgName and parentStatus.schedulerGroupStatus.get( sgName ):
      sgStatus = parentStatus.schedulerGroupStatus[ sgName ]
      if sgStatus.shapeRate.rate != tacShapeRateVal.invalid:
         return 'schedulingGroupShapeRate'
   if parentStatus.shapeRate.rate != tacShapeRateVal.invalid:
      return 'parentInterfaceShapeRate'
   return 'lineRate'

def getIntfSubObjectShapeRatePercentReference( intfStatus ):
   if intfStatus.shapeRate.rate == tacShapeRateVal.invalid:
      return getIntfShapeRatePercentReference( intfStatus )
   if tacSubIntfId.isSubIntfId( intfStatus.intfId ):
      return 'subinterfaceShapeRate'
   else:
      return 'parentInterfaceShapeRate'

def isValidShapeRate( shapeRate ):
   return shapeRate.rate != tacShapeRateVal.invalid

def getConfiguredLagMembers( lagIntf ):
   if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( lagIntf ):
      lagIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( lagIntf )

   lagMem = []
   for phyIntf, phyIntfVal in lagInputConfig.phyIntf.items():
      if phyIntfVal.lag and phyIntfVal.lag.intfId == lagIntf:
         lagMem.append( phyIntf )
   return lagMem

def getHwShapeRate( intf, txQ=None, group=None ):
   def getOpSr( sliceHwStatus, intf, txQ=None, group=None ):
      intfHwShapeRate = sliceHwStatus.hwShapeRates.get( intf )
      if not intfHwShapeRate:
         return None

      opSr = None
      if txQ:
         if txQ in intfHwShapeRate.txQueueShapeRate:
            opSr = intfHwShapeRate.txQueueShapeRate[ txQ ].shapeRate
      elif group:
         if group in intfHwShapeRate.schedulerGroupShapeRate:
            opSr = intfHwShapeRate.schedulerGroupShapeRate[ group ].shapeRate
      else:
         opSr = intfHwShapeRate.intfShapeRate
      return opSr

   for sliceHwStatus in qosSliceHwStatus.values():
      if not isPortChannelIntfId( intf ):
         # Check getOpSr, checked, should work
         opSr = getOpSr( sliceHwStatus, intf, txQ=txQ, group=group )
         if opSr and isValidShapeRate( opSr ):
            return opSr
      else:
         lagIntf = intf
         if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( lagIntf ):
            lagIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( lagIntf )
         members = getConfiguredLagMembers( lagIntf )
         for member in members:
            memberIntf = member
            if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intf ):
               subIntfNum = getSubIntfNum( intf )
               memberIntf = member + '.' + str( subIntfNum )

            opSr = getOpSr( sliceHwStatus, memberIntf, txQ=txQ, group=group )
            if opSr and isValidShapeRate( opSr ):
               return opSr
   return None

def populateBwModel( qosGuaranteedBw ):
   operBw = GuaranteedBwModel()
   operBw.bandwidth = qosGuaranteedBw.bw
   operBw.unit = guaranteedBwUnitFromEnum( qosGuaranteedBw.unit )
   return operBw

def populateConfiguredSr( configSr, statusSr, reference ):
   shapeRate = ShapeRateModel()
   if configSr.percent != tacPercent.invalid:
      shapeRate.percent = configSr.percent
      shapeRate.percentOf = reference
      shapeRate.rate = tacShapeRateVal.invalid
   if statusSr is not None:
      shapeRate.rate = statusSr.rate
      shapeRate.unit = shapeRateUnitFromEnum( statusSr.unit )
   elif configSr.percent == tacPercent.invalid:
      shapeRate.rate = configSr.rate
      shapeRate.unit = shapeRateUnitFromEnum( configSr.unit )
   return shapeRate

def isValidGuaranteedBw( guaranteedBw ):
   return guaranteedBw.bw != tacGuaranteedBwVal.invalid

def getHwGuaranteedBw( intf, txQ=None, group=None ):
   def getOpBw( sliceHwStatus, intf, txQ=None, group=None ):
      intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( intf )
      if not intfHwGuaranteedBw:
         return None

      opBw = None
      if txQ:
         if txQ in intfHwGuaranteedBw.txQueueGuaranteedBw:
            opBw = intfHwGuaranteedBw.txQueueGuaranteedBw[ txQ ].guaranteedBw
      elif group:
         if group in intfHwGuaranteedBw.schedulerGroupGuaranteedBw:
            opBw = \
                  intfHwGuaranteedBw.schedulerGroupGuaranteedBw[ group ].guaranteedBw
      else:
         opBw = intfHwGuaranteedBw.intfGuaranteedBw
      return opBw

   for sliceHwStatus in qosSliceHwStatus.values():
      if not isPortChannelIntfId( intf ):
         # check getOpBw, checked, should work
         opBw = getOpBw( sliceHwStatus, intf, txQ=txQ, group=group )
         if opBw and isValidGuaranteedBw( opBw ):
            return opBw
      else:
         lagIntf = intf
         if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( lagIntf ):
            lagIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( lagIntf )
         members = getConfiguredLagMembers( lagIntf )
         for member in members:
            memberIntf = member
            if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intf ):
               subIntfNum = getSubIntfNum( intf )
               memberIntf = member + '.' + str( subIntfNum )

            opBw = getOpBw( sliceHwStatus, memberIntf, txQ=txQ, group=group )
            if opBw and isValidGuaranteedBw( opBw ):
               return opBw
   return None

def populateConfiguredBw( configBw, statusBw, reference ):
   guaranteedBw = GuaranteedBwModel()
   if configBw.unit == tacGuaranteedBwUnit.guaranteedBwPercent:
      guaranteedBw.percent = configBw.bw
      guaranteedBw.percentOf = reference
      guaranteedBw.bandwidth = tacGuaranteedBwVal.invalid
   if statusBw:
      guaranteedBw.bandwidth = statusBw.bw
      guaranteedBw.unit = guaranteedBwUnitFromEnum( statusBw.unit )
   elif configBw.unit != tacGuaranteedBwUnit.guaranteedBwPercent:
      guaranteedBw.bandwidth = configBw.bw
      guaranteedBw.unit = guaranteedBwUnitFromEnum( configBw.unit )
   return guaranteedBw

def getIntfStatus( intf ):
   intfStatus = None
   if isinstance( intf, str ):
      intfName = intf
   else:
      intfName = intf.name

   if isLagPort( intfName ):
      members = getConfiguredLagMembers( intfName )
      if members:
         firstMember = members[ 0 ]
         if firstMember in qosStatus.intfStatus:
            if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfName ):
               subIntfNum = getSubIntfNum( intfName )
               ethSubIntf = firstMember + '.' + str( subIntfNum )
               if ethSubIntf in qosStatus.intfStatus:
                  intfStatus = qosStatus.intfStatus[ ethSubIntf ]
            else:
               intfStatus = qosStatus.intfStatus[ firstMember ]
   else:
      if intfName in qosStatus.intfStatus:
         intfStatus = qosStatus.intfStatus[ intfName ]
   return intfStatus

def populateQosSchedulingGroupsForIntf( intf, group=None, members=None ):
   if members is None:
      schedulingGroupModel = IntfSchedulingGroupModel()
      groupModel = SchedulingElementModel
   else:
      schedulingGroupModel = IntfSchedulingGroupWithMembersModel()
      groupModel = SchedulingGroupWithMembersModel

   intfName = intf.name
   intfConfig = qosConfig.intfConfig.get( intfName )
   intfStatus = getIntfStatus( intf )
   if not intfConfig:
      return schedulingGroupModel
   for name, groupConfig in intfConfig.schedulerGroupConfig.items():
      if group and name != group:
         continue
      schedulingGroup = groupModel()

      statusBw = None
      statusSr = None
      if intfStatus and name in intfStatus.schedulerGroupStatus:
         statusBw = intfStatus.schedulerGroupStatus[ name ].guaranteedBw
         statusSr = intfStatus.schedulerGroupStatus[ name ].shapeRate
      # Configured guaranteed bandwidth
      if isValidGuaranteedBw( groupConfig.guaranteedBw ):
         schedulingGroup.configuredGuaranteedBw = \
            populateConfiguredBw( groupConfig.guaranteedBw, statusBw,
                                  getIntfSubObjectShapeRatePercentReference(
                                     intfStatus ) )

      # Configured shape rate
      if isValidShapeRate( groupConfig.shapeRate ):
         schedulingGroup.configuredShapeRate = \
            populateConfiguredSr( groupConfig.shapeRate, statusSr,
                                  getIntfSubObjectShapeRatePercentReference(
                                     intfStatus ) )

      # Operational guaranteed bandwidth
      opBw = getHwGuaranteedBw( intfName, group=name )
      if opBw and isValidGuaranteedBw( opBw ):
         schedulingGroup.operationalGuaranteedBw = populateBwModel( opBw )

      # Operational shape rate
      opSr = getHwShapeRate( intfName, group=name )
      if opSr and isValidShapeRate( opSr ):
         schedulingGroup.operationalShapeRate = populateShapeRateModel( opSr )

      if members:
         for subIntf in members:
            intfConfig = qosConfig.intfConfig.get( subIntf )
            if intfConfig and intfConfig.schedulerGroupName == name:
               params = SchedulingElementModel()
               subIntfStatus = getIntfStatus( subIntf )
               if subIntfStatus:
                  statusBw = subIntfStatus.guaranteedBw
                  statusSr = subIntfStatus.shapeRate

               # configured value
               if isValidGuaranteedBw( intfConfig.guaranteedBw ):
                  params.configuredGuaranteedBw = \
                     populateConfiguredBw( intfConfig.guaranteedBw, statusBw,
                                           getIntfShapeRatePercentReference(
                                              subIntfStatus ) )

               if isValidShapeRate( intfConfig.shapeRate ):
                  params.configuredShapeRate = \
                     populateConfiguredSr( intfConfig.shapeRate, statusSr,
                                           getIntfShapeRatePercentReference(
                                              subIntfStatus ) )

               # populate operational value
               opBw = getHwGuaranteedBw( subIntf )
               if opBw and isValidGuaranteedBw( opBw ):
                  params.operationalGuaranteedBw = populateBwModel( opBw )

               opSr = getHwShapeRate( subIntf )
               if opSr and isValidShapeRate( opSr ):
                  params.operationalShapeRate = populateShapeRateModel( opSr )

               schedulingGroup.interfaces[ subIntf ] = params

      schedulingGroupModel.schedulingGroups[ name ] = schedulingGroup
   return schedulingGroupModel

def getSubIntfTrustAttr( intf, parentIntf ):
   trustMode = tacTrustMode.invalid
   defaultCos = tacCos.invalid
   defaultDscp = tacDscp.invalid

   parentQosIntfStatus = getIntfStatus( parentIntf )
   intfStatus = getIntfStatus( intf )
   if intfStatus:
      trustMode = intfStatus.trustMode
      defaultCos = intfStatus.defaultCos
      defaultDscp = intfStatus.defaultDscp
   if parentQosIntfStatus:
      if trustMode == tacTrustMode.invalid:
         trustMode = parentQosIntfStatus.trustMode
      if defaultCos == tacCos.invalid:
         defaultCos = parentQosIntfStatus.defaultCos
      if defaultDscp == tacDscp.invalid:
         defaultDscp = parentQosIntfStatus.defaultDscp
   return [ trustMode, defaultCos, defaultDscp ]

def showQosIntfTxQueues( intf, sliceName=None, mode=None ):
   txQueueQosModel = InterfaceQosModel.TxQueueQosModel()
   intfConfig = None
   intfStatus = None

   intfConfig = qosConfig.intfConfig.get( intf.name )
   intfStatus = getIntfStatus( intf )

   # SubIntf related parameters.
   isSubIntf = tacSubIntfId.isSubIntfId( intf.name )
   # Set to True if according to qosConfig the default SubIntf Schedulng scheme
   # is in effect.
   shapedSubIntfDefaultSchConfigPresent = False
   # Checks the above in qosStatus.
   shapedSubIntfDefaultSchStatusPresent = False
   # In case there is no scheduling / shaping config on a SubIntf, it uses its
   # parent's VOQs. This should be set to True in that case.
   subIntfUseParentQueues = False
   # If bandwidth weight is configured on any 1 of the tx-queues, operational
   # bandwidth is calculated using weight, else using percentage
   isOperationalBwModeWeight = False

   if isSubIntf:
      subIntfTxQueueSchPresent = False
      subIntfOrTxQueueShpPresent = False
      if intfStatus:
         for txQStatus in intfStatus.txQueueStatus.values():
            if txQStatus.shapeRate.rate != tacShapeRateVal.invalid:
               subIntfOrTxQueueShpPresent = True
            if txQStatus.priority != tacTxQueuePriority.priorityInvalid:
               subIntfTxQueueSchPresent = True
            if subIntfOrTxQueueShpPresent and subIntfTxQueueSchPresent:
               break
         if intfStatus.shapeRate.rate != tacShapeRateVal.invalid:
            subIntfOrTxQueueShpPresent = True
         if intfStatus.schedulerGroupName:
            schGroupName = intfStatus.schedulerGroupName
            parentIntfName = tacSubIntfId.parentIntfId( intf.name )
            parentIntfStatus = getIntfStatus( parentIntfName )
            if schGroupName in parentIntfStatus.schedulerGroupStatus:
               schGroupStatus = parentIntfStatus.schedulerGroupStatus[ schGroupName ]
               if schGroupStatus.shapeRate.rate != tacShapeRateVal.invalid:
                  subIntfOrTxQueueShpPresent = True
      shapedSubIntfDefaultSchStatusPresent = ( subIntfOrTxQueueShpPresent and
                                               not subIntfTxQueueSchPresent )
      subIntfUseParentQueues = not ( subIntfTxQueueSchPresent or
                                     subIntfOrTxQueueShpPresent )

      subIntfTxQueueSchPresent = False
      subIntfOrTxQueueShpPresent = False
      if intfConfig:
         for txQConfig in intfConfig.txQueueConfig.values():
            if txQConfig.shapeRate.rate != tacShapeRateVal.invalid:
               subIntfOrTxQueueShpPresent = True
            if txQConfig.priority != tacTxQueuePriority.priorityInvalid:
               subIntfTxQueueSchPresent = True
            if subIntfOrTxQueueShpPresent and subIntfTxQueueSchPresent:
               break
         if intfConfig.shapeRate.rate != tacShapeRateVal.invalid:
            subIntfOrTxQueueShpPresent = True
         if intfConfig.schedulerGroupName:
            groupName = intfConfig.schedulerGroupName
            parentIntfName = tacSubIntfId.parentIntfId( intf.name )
            parentIntfConfig = qosConfig.intfConfig.get( parentIntfName )
            if parentIntfConfig:
               groupConfig = parentIntfConfig.schedulerGroupConfig.get( groupName )
               if groupConfig and groupConfig.shapeRate.rate != \
                     tacShapeRateVal.invalid:
                  subIntfOrTxQueueShpPresent = True
      shapedSubIntfDefaultSchConfigPresent = ( subIntfOrTxQueueShpPresent and
                                               not subIntfTxQueueSchPresent )
      if isLagPort( intf.name ):
         # For lag subIntfs, this value is decided from intfConfig.
         subIntfUseParentQueues = not ( subIntfTxQueueSchPresent or
                                        subIntfOrTxQueueShpPresent )

      hwShapeRate = None
      parentHwShapeRate = None
      parentIntfName = tacSubIntfId.parentIntfId( intf.name )
      if not subIntfUseParentQueues and not isLagPort( intf.name ):
         for sliceHwStatus in qosSliceHwStatus.values():
            # Shape rate of parent and subinterface may appear on different
            # hardware slices, hence lookup through all slices to find them.
            if not hwShapeRate:
               hwShapeRate = sliceHwStatus.hwShapeRates.get( intf.name )
            if not parentHwShapeRate:
               parentHwShapeRate = sliceHwStatus.hwShapeRates.get( parentIntfName )
            if hwShapeRate and parentHwShapeRate:
               break
         subIntfUseParentQueues = True
         if hwShapeRate:
            subIntfUseParentQueues = False
            for txQ in hwShapeRate.txQueueShapeRate.values():
               if txQ.shapeRate.rate != tacShapeRateVal.invalid:
                  subIntfUseParentQueues = False
                  break
         if parentHwShapeRate:
            schGroupName = intfStatus.schedulerGroupName
            if schGroupName in parentHwShapeRate.schedulerGroupShapeRate:
               if parentHwShapeRate.schedulerGroupShapeRate[
                     schGroupName ].shapeRate.rate != tacShapeRateVal.invalid:
                  subIntfUseParentQueues = False

   if subIntfUseParentQueues:
      parentIntf = IntfCli.Intf( tacSubIntfId.parentIntfId( intf.name ), mode )
      return showQosIntfTxQueues( parentIntf, sliceName=sliceName, mode=mode )

   txQueueQosModel.wrrSupported = qosHwStatus.wrrSupported
   txQueueQosModel.guaranteedBwSupported = qosHwStatus.guaranteedBwSupported
   txQueueQosModel.ecnSupported = qosHwStatus.ecnSupported
   txQueueQosModel.delayEcnSupported = qosHwStatus.ecnDelaySupported
   txQueueQosModel.wredSupported = qosHwStatus.wredSupported
   txQueueQosModel.nonEctSupported = qosHwStatus.nonEctThdSupported
   txQueueQosModel.multipleSchedGroupsSupported = qosHwStatus.schedulerGroupSupported
   portTmMap = qosTmMapping.portTmMap
   if intf.name in portTmMap:
      txQueueQosModel.numTxQueuePrint = portTmMap[ intf.name ].numberOfQueues
   if not isSubIntf:
      txQueueQosModel.numTxQueue = qosHwStatus.numTxQueueSupported
   else:
      txQueueQosModel.numTxQueue = qosGlobalConfig.numTxqsPerSubIntf

   txQueueQosModel.burstSizeConfigSupported = \
                                       qosHwStatus.shapeRateBurstSizeConfigSupported
   if txQueueQosModel.numTxQueue == 0:
      return txQueueQosModel

   if intfConfig:
      for txQConfig in intfConfig.txQueueConfig.values():
         if txQConfig.bandwidthWeight != tacBwWeight.invalid:
            isOperationalBwModeWeight = True
            break

   for hwtxqid in range( txQueueQosModel.numTxQueue - 1, -1, -1 ):
      schGroupId = qosHwStatus.hwTxQueue[ hwtxqid ].schGroupId
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         txQName = clitxq.type[ : 2 ].upper() + str( clitxq.id )
      else:
         txQName = str( clitxq.id )

      txQueueModel = InterfaceQosModel.TxQueueQosModel.TxQueue()
      txQueueModel.txQueue = txQName
      txQueueModel.txQueuePriority = hwtxqid
      txQueueModel.schedGroupId = schGroupId
      txQueueModel.operationalBurstSize = None
      txQueueModel.configuredBurstSize = None

      subIntfTxQueueDefaultWrrBw = None
      subIntfTxQueueDefaultWrrPriority = None
      # Default scheduling for SubIntfs with port / tx-queue shaping may not be all
      # SP. If different, the PD agent is supposed to pusblish it in qosHwStatus.
      if clitxq in qosHwStatus.subIntfDefaultSchConfig:
         subIntfTxQueueDefaultWrrBw = qosHwStatus.subIntfDefaultSchConfig[
            clitxq ].bandwidth
         subIntfTxQueueDefaultWrrPriority = (
            qosHwStatus.subIntfDefaultSchConfig[ clitxq ].priority )

      txQueueQosModel.txQueueList.append( txQueueModel )

      txQueueConfig = None
      txQueueStatus = None

      if intfConfig:
         txQueueConfig = intfConfig.txQueueConfig.get( clitxq )

      if intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )

      if shapedSubIntfDefaultSchConfigPresent:
         if subIntfTxQueueDefaultWrrBw != tacPercent.invalid:
            # This value will get overwritten if txQueueConfig is not None
            # and txQueueConfig.bandwidth != tacPercent.invalid
            txQueueModel.configuredWrrBw = subIntfTxQueueDefaultWrrBw
         txQueueModel.configuredSchedMode = SchedulingModeModel()
         txQueueModel.configuredSchedMode.schedulingMode = schedModeEnumToStr(
            subIntfTxQueueDefaultWrrPriority )

      # Configured parameters
      if txQueueConfig:

         # WRR Bandwidth
         if txQueueConfig.bandwidthWeight != tacBwWeight.invalid:
            txQueueModel.configuredWrrBwWeight = txQueueConfig.bandwidthWeight
         elif txQueueConfig.bandwidth != tacPercent.invalid:
            txQueueModel.configuredWrrBw = txQueueConfig.bandwidth

         # Guaranteed bandwidth
         if txQueueConfig.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
            txQueueModel.configuredGuaranteedBw = GuaranteedBwModel()
            cfgGuaranteedBw = txQueueModel.configuredGuaranteedBw
            cfgGuaranteedBw.bandwidth = txQueueConfig.guaranteedBw.bw
            cfgGuaranteedBw.unit = \
               guaranteedBwUnitFromEnum( txQueueConfig.guaranteedBw.unit )

         # Shape rate
         txQueueModel.configuredShapeRate = \
            populateShapeRateModel(
               txQueueConfig.shapeRate,
               statusShapeRate=txQueueStatus.shapeRate
               if txQueueStatus else None,
               reference=getIntfSubObjectShapeRatePercentReference( intfStatus )
               if intfStatus else 'lineRate' )
         # Burst-size config
         if qosHwStatus.shapeRateBurstSizeConfigSupported:
            txQueueModel.configuredBurstSize = populateBurstSizeModel(
               txQueueConfig.shapeRate.burstSize )

         # Scheduling priority/mode
         if not shapedSubIntfDefaultSchConfigPresent:
            txQueueModel.configuredSchedMode = SchedulingModeModel()
            cfgSchedMode = txQueueModel.configuredSchedMode
            cfgSchedMode.schedulingMode = schedModeEnumToStr(
               txQueueConfig.priority )

      # ECN Status
      ecnStatus = None
      delayEcnEnabled = \
            qosStatus.ecnDelayThreshold != tacEcnDelayThreshold and \
            qosStatus.ecnDelayThreshold.unit != tacEcnDelayThreshold.unit
      if not intfStatus:
         if txQueueConfig:
            ecnStatus = txQueueConfig.ecnConfig
            delayEcnEnabled &= txQueueConfig.delayEcnEnabled
            # Txq based and global ecn should never be enabled together
            assert not ( bool( txQueueConfig.ecnDelayConfig ) and delayEcnEnabled )
            if not delayEcnEnabled:
               delayEcnEnabled = bool( txQueueConfig.ecnDelayConfig )
         else:
            delayEcnEnabled = False
      elif clitxq in intfStatus.txQueueStatus:
         ecnStatus = intfStatus.txQueueStatus[ clitxq ].ecnStatus
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]
         delayEcnEnabled &= intfStatus.txQueueStatus[ clitxq ].delayEcnEnabled
         assert not ( bool( intfStatus.txQueueStatus[ clitxq ].ecnDelayStatus )
                      and delayEcnEnabled )
         if not delayEcnEnabled:
            delayEcnEnabled = \
               bool( intfStatus.txQueueStatus[ clitxq ].ecnDelayStatus )
      else:
         delayEcnEnabled = False

      if ecnStatus and delayEcnEnabled:
         txQueueModel.ecnStatus = 'lengthAndDelayEnabled'
      elif ecnStatus:
         txQueueModel.ecnStatus = 'enabled'
      elif delayEcnEnabled:
         txQueueModel.ecnStatus = 'delayEnabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.ecnStatus = 'not applicable'
         else:
            txQueueModel.ecnStatus = 'disabled'

      # Dynamic ECN status
      if qosHwStatus.bufferBasedDecnSupported:
         txQueueModel.dynamicEcnStatus = 'disabled'
         if intfStatus and clitxq in intfStatus.txQueueStatus:
            if intfStatus.txQueueStatus[ clitxq ].bufferBasedDecnStatus:
               txQueueModel.dynamicEcnStatus = 'enabled'
      else:
         txQueueModel.dynamicEcnStatus = 'not applicable'

      # WRED status
      wredStatus = None
      dpWredStatus = None
      if not intfStatus:
         if txQueueConfig:
            wredStatus = txQueueConfig.wredConfig
            if txQueueConfig.dpConfig:
               dpWredStatus = txQueueConfig.dpConfig.dpWredConfig
      elif clitxq in intfStatus.txQueueStatus:
         wredStatus = intfStatus.txQueueStatus[ clitxq ].wredStatus
         dpStatus = intfStatus.txQueueStatus[ clitxq ].dpStatus
         if dpStatus:
            dpWredStatus = dpStatus.dpWredStatus
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]

      if wredStatus or dpWredStatus:
         txQueueModel.wredStatus = 'enabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.wredStatus = 'not applicable'
         else:
            txQueueModel.wredStatus = 'disabled'

      # NON-ECT status
      nonEctStatus = None
      if not intfStatus:
         if txQueueConfig:
            nonEctStatus = txQueueConfig.nonEctConfig
      elif clitxq in intfStatus.txQueueStatus:
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]
         nonEctStatus = txQueueStatus.nonEctStatus

      if nonEctStatus:
         txQueueModel.nonEctStatus = 'enabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.nonEctStatus = 'not applicable'
         else:
            txQueueModel.nonEctStatus = 'disabled'

      # Operational parameters
      # WRR Bandwidth operational status
      if txQueueStatus and isOperationalBwModeWeight:
         if txQueueStatus.bandwidthWeight != tacBwWeight.invalid:
            txQueueModel.operationalWrrBwWeight = txQueueStatus.bandwidthWeight
      elif txQueueStatus:
         if txQueueStatus.bandwidth != tacPercent.invalid:
            txQueueModel.operationalWrrBw = txQueueStatus.bandwidth

      # Check if detailed interface status is supported
      # ( i.e., Would hwStatus be populated by this platform )
      if txQueueStatus and qosHwStatus.interfaceDetailsSupported:

         # Guaranteed bandwidth operation status
         intfHwGuaranteedBw = None
         if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
            for sliceHwStatus in qosSliceHwStatus.values():
               intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( intf.name )
               if intfHwGuaranteedBw is not None:
                  break
         if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid and \
            intf.name == fabricIntfName:
            for sliceHwStatus in qosSliceHwStatus.values():
               if sliceHwStatus.name == sliceName:
                  hwGuaranteedBw = sliceHwStatus.hwGuaranteedBw
                  intfHwGuaranteedBw = hwGuaranteedBw.get( intf.name )
                  break
         if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid and \
            intfHwGuaranteedBw and clitxq in intfHwGuaranteedBw.txQueueGuaranteedBw:
            txQueueHwGuaranteedBw = \
               intfHwGuaranteedBw.txQueueGuaranteedBw[ clitxq ]

            if txQueueHwGuaranteedBw.guaranteedBw.bw != \
                  tacGuaranteedBwVal.invalid:
               txQueueModel.operationalGuaranteedBw = GuaranteedBwModel()
               operGuaranteedBw = txQueueModel.operationalGuaranteedBw
               bw = txQueueHwGuaranteedBw.guaranteedBw.bw
               unit = txQueueHwGuaranteedBw.guaranteedBw.unit
               operGuaranteedBw.bandwidth = bw
               operGuaranteedBw.unit = guaranteedBwUnitFromEnum( unit )

         # Shape rate operation status
         if txQueueStatus.shapeRate.rate != tacShapeRateVal.invalid:
            intfHwShapeRate = None
            for sliceHwStatus in qosSliceHwStatus.values():
               intfHwShapeRate = sliceHwStatus.hwShapeRates.get( intf.name )
               if intfHwShapeRate is not None:
                  break
            if intfHwShapeRate and \
                  clitxq in intfHwShapeRate.txQueueShapeRate:

               txQueueHwShapeRate = intfHwShapeRate.txQueueShapeRate[ clitxq ]
               txQueueModel.operationalShapeRate = \
                  populateShapeRateModel( txQueueHwShapeRate.shapeRate )
               # Burst-size operational Value
               if qosHwStatus.shapeRateBurstSizeConfigSupported:
                  txQueueModel.operationalBurstSize = \
                           populateBurstSizeModel(
                              txQueueHwShapeRate.shapeRate.burstSize )

      elif txQueueStatus:  # Interface details not supported/no hw status available

         # Guaranteed Bandwidth
         if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
            txQueueModel.operationalGuaranteedBw = GuaranteedBwModel()
            operGuaranteedBw = txQueueModel.operationalGuaranteedBw
            operGuaranteedBw.bandwidth = txQueueStatus.guaranteedBw.bw
            operGuaranteedBw.unit = \
               guaranteedBwUnitFromEnum( txQueueStatus.guaranteedBw.unit )

         # Shape Rate
         txQueueModel.operationalShapeRate = \
            populateShapeRateModel( txQueueStatus.shapeRate )

         # Burst Size
         if txQueueStatus.shapeRate:
            txQueueModel.operationalBurstSize = populateBurstSizeModel(
               txQueueStatus.shapeRate.burstSize )

      # Scheduling priority/mode
      if not shapedSubIntfDefaultSchStatusPresent:
         txQueueModel.operationalSchedMode = SchedulingModeModel()
         if txQueueStatus:
            txQueueModel.operationalSchedMode.schedulingMode = (
               schedModeEnumToStr( txQueueStatus.priority ) )
      else:
         if subIntfTxQueueDefaultWrrBw != tacPercent.invalid:
            txQueueModel.operationalWrrBw = subIntfTxQueueDefaultWrrBw
         txQueueModel.operationalSchedMode = SchedulingModeModel()
         txQueueModel.operationalSchedMode.schedulingMode = schedModeEnumToStr(
            subIntfTxQueueDefaultWrrPriority )

   return txQueueQosModel

def showQosInterface( intf, sliceName=None, detail=None, mode=None ):
   # detail specifies whether to display the detailed output for
   # fabric interface corresponding to the sliceName
   intfQosModel = InterfaceQosModel()

   if ( intf.name == fabricIntfName ) and detail:
      intfQosModel.sliceName = sliceName

   isLagIntf = isLagPort( intf.name )
   isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intf.name )
   isLagSubIntf = isSubIntf and isLagIntf

   intfQosModel.interface = intf.name.replace( fabricIntfName, fabricTokenName )

   intfQosModel.intfIsLag = isLagIntf

   trustMode = tacTrustMode.invalid
   defaultCos = tacCos.invalid
   defaultDscp = tacDscp.invalid

   lagMem = None
   if isLagIntf:
      lagMem = getConfiguredLagMembers( intf.name )

   if isLagIntf and not lagMem:
      # For a lag having no members, show intfConfig
      if intf.name in qosConfig.intfConfig:
         intfConfig = qosConfig.intfConfig[ intf.name ]
         trustMode = intfConfig.trustMode
         defaultCos = intfConfig.defaultCos
         defaultDscp = intfConfig.defaultDscp
   else:
      # For phyIntf or lag with atleast one member, show intfStatus
      intfStatus = getIntfStatus( intf )
      if intfStatus:
         trustMode = intfStatus.trustMode
         defaultCos = intfStatus.defaultCos
         defaultDscp = intfStatus.defaultDscp

   if isSubIntf:
      if ( not isLagSubIntf ) or ( isLagSubIntf and lagMem ):
         parentIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intf.name )
         [ trustMode, defaultCos, defaultDscp ] = \
               getSubIntfTrustAttr( intf, parentIntf )

   # Now we populate the intfQosModel( trustMode, defaultCos, defaultDscp )
   intfQosModel.intfQosInfoAvailable = True
   trustModeStr = compareAttributeToDefault( qosHwStatus, "defaultTrustMode",
                                          trustMode, tacTrustMode.invalid )
   intfQosModel.trustMode = convertTrustModeModelFromStr( trustModeStr )
   intfQosModel.intfQosDetailedInfoAvailable = qosHwStatus.interfaceDetailsSupported
   if qosHwStatus.defaultCosSupported is True:
      intfQosModel.defaultCosSupported = True
      intfQosModel.defaultCos = int( compareAttributeToDefault(
                                              qosHwStatus, "defaultCos",
                                              defaultCos, tacCos.invalid ) )

   if qosHwStatus.defaultDscpSupported is True:
      intfQosModel.defaultDscpSupported = True
      intfQosModel.defaultDscp = int( compareAttributeToDefault(
                                             qosHwStatus, "defaultDscp",
                                             defaultDscp, tacDscp.invalid ) )

   if qosHwStatus.cosToTcPerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.intfToCosToTcProfile:
               intfQosModel.cosToTcProfile = sliceHwStatus.intfToCosToTcProfile[
                  intf.name ]
      else:
         # For LAG, we populate configured value
         if intf.name in qosConfig.intfConfig:
            intfQosModel.cosToTcProfile = qosConfig.intfConfig[
               intf.name ].cosToTcProfileName

   if qosHwStatus.expToTcPerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.intfToExpToTcProfile:
               intfQosModel.expToTcProfile = sliceHwStatus.intfToExpToTcProfile[
                  intf.name ]
      else:
         # For LAG, we populate configured value
         if intf.name in qosConfig.intfConfig:
            intfQosModel.expToTcProfile = qosConfig.intfConfig[
               intf.name ].expToTcProfileName

   if qosHwStatus.dscpToTcPerSubInterfaceSupported:
      if not isLagIntf:
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.intfToDscpToTcNamedMap:
               intfQosModel.dscpToTcMap = sliceHwStatus.intfToDscpToTcNamedMap[
                  intf.name ]
      else:
         # For Lag, we populate configured value from CLI
         if intf.name in qosConfig.intfConfig:
            intfQosModel.dscpToTcMap = qosConfig.intfConfig[
               intf.name ].dscpToTcMapName

   if qosHwStatus.tcToCosPerInterfaceSupported:
      for sliceHwStatus in qosSliceHwStatus.values():
         if intf.name in sliceHwStatus.intfToTcToCosNamedMap:
            intfQosModel.tcToCosNamedMap = sliceHwStatus.intfToTcToCosNamedMap[
               intf.name ]
            break

   if qosHwStatus.cpuTcToCosPerInterfaceSupported or \
      qosHwStatus.cpuTcToCosPerSubInterfaceSupported or \
      qosHwStatus.cpuTcToCosPerLagSupported:
      if intf.name in qosConfig.intfConfig:
         intfQosModel.cpuTcToCosNamedMap = qosConfig.intfConfig[
            intf.name ].cpuTcToCosMapName

   if qosHwStatus.dscpRewritePerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.intfToTcToDscpRewriteMap:
               intfQosModel.dscpRewriteMap = sliceHwStatus.intfToTcToDscpRewriteMap[
                  intf.name ]
               break
      else:
         # For LAG, we populate configured value
         if intf.name in qosConfig.intfConfig:
            intfQosModel.dscpRewriteMap = qosConfig.intfConfig[
               intf.name ].dscpRewriteMapName

   if qosHwStatus.expRewriteSupported:
      for sliceHwStatus in qosSliceHwStatus.values():
         expRewriteMapName = sliceHwStatus.intfToTcDpToExpRewriteMap.get( intf.name )
         if expRewriteMapName:
            intfQosModel.tcDpToExpMap = expRewriteMapName
            break
   
   if qosHwStatus.dscpPreserveIpMplsEncapSupported:
      if not isLagIntf:
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.dscpPreserveIpMplsEncap:
               intfQosModel.dscpPreserveIpMplsEncapMode = True
               break
      else:
         # For LAG, we populate from QosIntfConfig[intf]
         if intf.name in qosConfig.intfConfig:
            intfQosModel.dscpPreserveIpMplsEncapMode = qosConfig.intfConfig[
               intf.name ].dscpPreserveIpMplsEncapMode

   if trustMode == tacTrustMode.dscp:
      # Incase of DSCP trust mode and COS rewrite is enabled but not active,
      # then indicate to the user.
      hwProgrammingSuccess = True
      # Flag sliceHwStatus.dscpToCosRewriteEnabled is set to False when
      # hardware programming fails.
      for sliceHwStatus in qosSliceHwStatus.values():
         if not sliceHwStatus.dscpToCosRewriteEnabled:
            hwProgrammingSuccess = False
            break
      if qosStatus.cosRewriteEnabled and not hwProgrammingSuccess:
         intfQosModel.dscpToCosRewriteActive = False
   else:
      # For other trust modes if DSCP rewrite is enabled but not active,
      # then indicate to the user.
      hwProgrammingSuccess = True
      for sliceHwStatus in qosSliceHwStatus.values():
         if not sliceHwStatus.cosToDscpRewriteEnabled:
            hwProgrammingSuccess = False
            break
      if qosStatus.dscpRewriteEnabled and not hwProgrammingSuccess:
         intfQosModel.dscpRewriteActive = False

   # Fetch the correct intfStatus and intfConfig for the interface
   intfStatus = getIntfStatus( intf )
   intfConfig = qosConfig.intfConfig.get( intf.name )

   # configured port shape rate
   if intfConfig:
      if isSubIntf and lagMem and '.' not in lagMem[ 0 ]:
         memSubIntf = lagMem[ 0 ] + intf.name[ intf.name.find( '.' ) : ]
         memSubIntfStatus = getIntfStatus( memSubIntf )
         statusShapeRate = memSubIntfStatus.shapeRate if memSubIntfStatus else None
         reference = getIntfShapeRatePercentReference( memSubIntfStatus ) \
            if memSubIntfStatus else 'lineRate'
      else:
         statusShapeRate = intfStatus.shapeRate if intfStatus else None
         reference = getIntfShapeRatePercentReference( intfStatus ) \
            if intfStatus else 'lineRate'
      intfQosModel.configuredPortShapeRate = \
         populateShapeRateModel( intfConfig.shapeRate,
                                 statusShapeRate=statusShapeRate,
                                 lagSize=len( lagMem ) if lagMem else None,
                                 reference=reference )

   # operational port shape rate
   if not isLagIntf or lagMem:
      # only for physical interfaces or lag having atleast one member
      # operational shape rate is populated
      if qosHwStatus.interfaceDetailsSupported:
         intfHwShapeRate = None
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.hwShapeRates and  \
               sliceHwStatus.hwShapeRates[ intf.name ].intfShapeRate.rate != \
                  tacShapeRateVal.invalid:
               intfHwShapeRate = sliceHwStatus.hwShapeRates[ intf.name ].\
                                 intfShapeRate
               intfQosModel.operationalPortShapeRate = \
                  populateShapeRateModel( intfHwShapeRate )
               break
      else:
         if intfStatus and intfStatus.shapeRate.rate != tacShapeRateVal.invalid:
            intfQosModel.operationalPortShapeRate = \
               populateShapeRateModel( intfStatus.shapeRate )

   if not isSubIntf or subIntfHwStatus.subIntfShapeRateSupported:
      intfQosModel.txQueueQosModel = showQosIntfTxQueues( intf, sliceName=sliceName,
                                                          mode=mode )

   if getIntfStatus( intf ) and subIntfHwStatus.subIntfSchedulingGroupSupported \
      and not isSubIntf:
      intfQosModel.schedulingGroupModel = populateQosSchedulingGroupsForIntf( intf )

   # configured burst-size
   if qosHwStatus.shapeRateBurstSizeConfigSupported:
      intfQosModel.burstSizeConfigSupported = True
      intfQosModel.configuredBurstSize = None
      intfQosModel.operationalBurstSize = None
      if intfConfig and intfConfig.shapeRate:
         intfQosModel.configuredBurstSize = populateBurstSizeModel(
            intfConfig.shapeRate.burstSize )
      if qosHwStatus.interfaceDetailsSupported:
         for sliceHwStatus in qosSliceHwStatus.values():
            if intf.name in sliceHwStatus.hwShapeRates:
               intfQosModel.operationalBurstSize = populateBurstSizeModel(
                  sliceHwStatus.hwShapeRates[ intf.name ].intfShapeRate.burstSize )
               break
      else:
         if intfStatus and intfStatus.shapeRate:
            intfQosModel.operationalBurstSize = populateBurstSizeModel(
               intfStatus.shapeRate.burstSize )

   # configured scheduler-compensation
   if qosHwStatus.perPortSchedulerCompensationSupported and \
      ( intf.name != fabricIntfName ):
      intfQosModel.schedulerCompensationConfigSupported = True
      intfQosModel.configuredSchedulerCompensation = None
      intfQosModel.operationalSchedulerCompensation = None
      if intfConfig:
         intfQosModel.configuredSchedulerCompensation = \
         populateSchedulerCompensationModel( intfConfig.schedulerCompensation )
      if qosHwStatus.interfaceDetailsSupported:
         compensationsPublished = [ sliceHwStatus.hwSchedulerCompensation[
            intf.name ]
            if intf.name in sliceHwStatus.hwSchedulerCompensation
            else tacSchedulerCompensation.invalid
            for sliceHwStatus in qosSliceHwStatus.values() ]
         slicesNotPusblished = compensationsPublished.count(
            tacSchedulerCompensation.invalid )
         if qosHwStatus.schedulerCompensationCheckAllSlice:
            if not slicesNotPusblished:
               # All slices have published a value.
               intfQosModel.operationalSchedulerCompensation = \
                  populateSchedulerCompensationModel( compensationsPublished[ 0 ] )
         elif slicesNotPusblished < len( qosSliceHwStatus.values() ):
            # Atleast one slice has published a value.
            intfQosModel.operationalSchedulerCompensation = \
               populateSchedulerCompensationModel(
                  [ i for i in compensationsPublished if i !=
                  tacSchedulerCompensation.invalid ][ 0 ] )
      else:
         if intfStatus:
            intfQosModel.operationalSchedulerCompensation = \
            populateSchedulerCompensationModel(
               intfStatus.schedulerCompensation )

   # configured policer packet size adjustment
   if qosHwStatus.perPortPolicerPacketSizeAdjSupported and \
      ( intf.name != fabricIntfName ):
      intfQosModel.policerPktSizeAdjConfigSupported = True
      intfQosModel.configuredPolicerPktSizeAdj = None
      intfQosModel.operationalPolicerPktSizeAdj = None
      if intfConfig:
         profile = intfConfig.policerPktSizeAdjProfileName
         if qosConfig.policingConfig and \
               profile in qosConfig.policingConfig.policerPktSizeAdjProfile:
            value = qosConfig.policingConfig.policerPktSizeAdjProfile[ profile ].\
               policerPktSizeAdj
         else:
            value = tacPolicerPktSizeAdj.invalid
         intfQosModel.configuredPolicerPktSizeAdj = \
            populatePolicerPktSizeAdjModel( profile, value )
      for sliceHwStatus in qosSliceHwStatus.values():
         if intf.name in sliceHwStatus.intfToPolicerPktSizeAdjProfile:
            profile = sliceHwStatus.intfToPolicerPktSizeAdjProfile[ intf.name ]
            if profile in qosConfig.policingConfig.policerPktSizeAdjProfile:
               value = qosConfig.policingConfig.\
                  policerPktSizeAdjProfile[ profile ].policerPktSizeAdj
            else:
               value = tacPolicerPktSizeAdj.invalid
            intfQosModel.operationalPolicerPktSizeAdj = \
               populatePolicerPktSizeAdjModel( profile, value )

   if ( ( isSubIntf and qosHwStatus.subIntfGuaranteedBwSupported ) or
        ( not isSubIntf and qosHwStatus.parentGuaranteedBwSupported )
   ):
      intfQosModel.interfaceGuaranteedBwSupported = True
      intfQosModel.configuredPortGuaranteedBw = None
      intfQosModel.operationalPortGuaranteedBw = None
      if intfConfig and intfConfig.guaranteedBw:
         statusGuaranteedBw = intfStatus.guaranteedBw if intfStatus else None
         intfQosModel.configuredPortGuaranteedBw = populateGuaranteedBwModel(
            intfConfig.guaranteedBw, statusGuaranteedBw=statusGuaranteedBw,
            reference=getIntfShapeRatePercentReference( intfStatus )
            if intfStatus else 'lineRate' )
      if qosHwStatus.interfaceDetailsSupported:
         for sliceHwStatus in qosSliceHwStatus.values():
            if ( intf.name in sliceHwStatus.hwGuaranteedBw and
                 sliceHwStatus.hwGuaranteedBw[ intf.name ].intfGuaranteedBw.bw !=
                 tacGuaranteedBwVal.invalid ):
               intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw[ intf.name ].\
                  intfGuaranteedBw
               intfQosModel.operationalPortGuaranteedBw = \
                  populateGuaranteedBwModel( intfHwGuaranteedBw )
               break
      else:
         if intfStatus and intfStatus.guaranteedBw.bw != \
               tacGuaranteedBwVal.invalid:
            intfQosModel.operationalPortGuaranteedBw = \
               populateGuaranteedBwModel( intfStatus.guaranteedBw )

   return intfQosModel

def convertPolicerConfigCmdVersion( mode ):
   for mapType in cliQosAclConfig.pmapType:
      qosPMapConfig = cliQosAclConfig.pmapType[ mapType ]
      for pmapName in qosPMapConfig.pmap:
         pmapConfig = qosPMapConfig.pmap.get( pmapName )
         for classMapName in pmapConfig.classAction:
            classMapConfig = pmapConfig.classAction.get( classMapName )
            if classMapConfig.policer:
               classMapConfig.policer.cmdVersion = 2


# ------------------------------------------------------------------------
# Register convertPolicerConfigCmdVersion via "config convert new-syntax"
# ------------------------------------------------------------------------
ConfigConvert.registerConfigConvertCallback(
   convertPolicerConfigCmdVersion )

def getSubIntfGroupMembership( subIntf ):
   if '.' not in subIntf:
      return ''

   parentIntf = subIntf.split( '.' )[ 0 ]
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if parentIntf not in qosSchedulerIntfConfig:
      return ''
   parentIntfConfig = qosSchedulerIntfConfig[ parentIntf ]
   for groupName in parentIntfConfig.group:
      groupConfig = parentIntfConfig.group[ groupName ]
      if subIntf in groupConfig.members:
         return groupName
   return ''

def setIntfConfig( intf, cfgTrustMode=None, cfgDefaultCos=None,
                   cfgDefaultDscp=None, cfgShapeRate=None, cfgShapeRateUnit=None,
                   profile=False, cfgShapeRateShared=None,
                   cfgShapeRatePercent=None, cfgDscpToTcMapName=None,
                   cfgCosToTcProfileName=None, cfgExpToTcProfileName=None,
                   cfgTcToCosMapName=None,
                   cfgCpuTcToCosMapName=None, burstSizeVal=None,
                   burstSizeUnit=None,
                   cfgGuaranteedBw=None, cfgGuaranteedBwUnit=None,
                   cfgSchedulerGroupName=None, cfgSchedulerCompensation=None,
                   cfgPolicerPktSizeAdjProfileName=None,
                   cfgDscpPreserveIpMplsEncap=None,
                   cfgTcDpToExpMapName=None ):

   intfConfig = None
   if profile:
      cfg = qosInputProfileConfig
   else:
      cfg = qosInputConfig
   intfCfgd = intf in cfg.intfConfig
   if intfCfgd:
      intfConfig = cfg.intfConfig[ intf ]

   if cfgDscpPreserveIpMplsEncap is None:
      if not intfCfgd:
         cfgDscpPreserveIpMplsEncap = False
      else:
         cfgDscpPreserveIpMplsEncap = intfConfig.dscpPreserveIpMplsEncapMode

   if cfgTrustMode is None:
      if not intfCfgd:
         cfgTrustMode = tacTrustMode.invalid
      else:
         cfgTrustMode = intfConfig.trustMode

   if cfgDefaultCos is None:
      if not intfCfgd:
         cfgDefaultCos = tacCos.invalid
      else:
         cfgDefaultCos = intfConfig.defaultCos

   if cfgDefaultDscp is None:
      if not intfCfgd:
         cfgDefaultDscp = tacDscp.invalid
      else:
         cfgDefaultDscp = intfConfig.defaultDscp

   if cfgShapeRate is None:
      if not intfCfgd:
         cfgShapeRate = tacShapeRateVal.invalid
         cfgShapeRateUnit = tacShapeRateUnit.shapeRateKbps
         cfgShapeRateShared = False
      else:
         cfgShapeRate = intfConfig.shapeRate.rate
         cfgShapeRateUnit = intfConfig.shapeRate.unit
         cfgShapeRateShared = intfConfig.shapeRate.shared

   if cfgShapeRateUnit is None:
      cfgShapeRateUnit = tacShapeRateUnit.shapeRateKbps

   if cfgShapeRatePercent is None:
      if intfCfgd:
         cfgShapeRatePercent = intfConfig.shapeRate.percent
      else:
         cfgShapeRatePercent = Tac.Value( 'Qos::Percent' )

   tempShapeRate = Tac.Value( 'Qos::ShapeRate' )
   burstSizeCfg = Tac.Value( 'Qos::BurstSize' )
   if burstSizeVal:
      burstSizeCfg.value = burstSizeVal
   else:
      if intfCfgd:
         burstSizeCfg.value = intfConfig.shapeRate.burstSize.value
   if burstSizeUnit:
      burstSizeCfg.unit = burstSizeUnit
   else:
      if intfCfgd:
         burstSizeCfg.unit = intfConfig.shapeRate.burstSize.unit

   tempShapeRate.burstSize = burstSizeCfg
   tempShapeRate.rate = cfgShapeRate
   tempShapeRate.unit = cfgShapeRateUnit
   tempShapeRate.shared = bool( cfgShapeRateShared )
   tempShapeRate.percent = cfgShapeRatePercent

   if cfgGuaranteedBw is None:
      if not intfCfgd:
         cfgGuaranteedBw = tacGuaranteedBwVal.invalid
         cfgGuaranteedBwUnit = tacGuaranteedBwUnit.guaranteedBwKbps
      else:
         cfgGuaranteedBw = intfConfig.guaranteedBw.bw
         cfgGuaranteedBwUnit = intfConfig.guaranteedBw.unit

   if cfgGuaranteedBwUnit is None:
      cfgGuaranteedBwUnit = tacGuaranteedBwUnit.guaranteedBwKbps

   tempGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   tempGuaranteedBw.bw = cfgGuaranteedBw
   tempGuaranteedBw.unit = cfgGuaranteedBwUnit

   if cfgCosToTcProfileName is None:
      if not intfCfgd:
         cfgCosToTcProfileName = tacCosToTcProfileName.defaultProfileName
      else:
         cfgCosToTcProfileName = intfConfig.cosToTcProfileName

   if cfgExpToTcProfileName is None:
      if not intfCfgd:
         cfgExpToTcProfileName = tacExpToTcProfileName.defaultProfileName
      else:
         cfgExpToTcProfileName = intfConfig.expToTcProfileName

   if cfgDscpToTcMapName is None:
      if not intfCfgd:
         cfgDscpToTcMapName = tacDscpToTcMapName.defaultMapName
      else:
         cfgDscpToTcMapName = intfConfig.dscpToTcMapName

   if cfgTcToCosMapName is None:
      if not intfCfgd:
         cfgTcToCosMapName = tacTcToCosMapName.defaultMapName
      else:
         cfgTcToCosMapName = intfConfig.tcToCosMapName

   if cfgTcDpToExpMapName is None:
      if not intfCfgd:
         cfgTcDpToExpMapName = tacTcDpToExpMapName.defaultMapName
      else:
         cfgTcDpToExpMapName = intfConfig.tcDpToExpMapName

   if cfgCpuTcToCosMapName is None:
      if not intfCfgd:
         cfgCpuTcToCosMapName = tacTcToCosMapName.defaultMapName
      else:
         cfgCpuTcToCosMapName = intfConfig.cpuTcToCosMapName

   if cfgSchedulerGroupName is None:
      # Source this field from the qos scheduling config
      cfgSchedulerGroupName = getSubIntfGroupMembership( intf )

   if cfgSchedulerCompensation is None:
      if not intfCfgd:
         cfgSchedulerCompensation = tacSchedulerCompensation.invalid
      else:
         cfgSchedulerCompensation = intfConfig.schedulerCompensation

   if cfgPolicerPktSizeAdjProfileName is None:
      if not intfCfgd:
         cfgPolicerPktSizeAdjProfileName = \
            tacPolicerPktSizeAdjProfileName.defaultProfileName
      else:
         cfgPolicerPktSizeAdjProfileName = intfConfig.policerPktSizeAdjProfileName

   if intfCfgd:  # Interface config exist ( update the current config )
      intfConfig.trustMode = cfgTrustMode
      intfConfig.defaultCos = cfgDefaultCos
      intfConfig.defaultDscp = cfgDefaultDscp
      intfConfig.shapeRate = tempShapeRate
      intfConfig.guaranteedBw = tempGuaranteedBw
      intfConfig.cosToTcProfileName = cfgCosToTcProfileName
      intfConfig.expToTcProfileName = cfgExpToTcProfileName
      intfConfig.dscpToTcMapName = cfgDscpToTcMapName
      intfConfig.tcToCosMapName = cfgTcToCosMapName
      intfConfig.tcDpToExpMapName = cfgTcDpToExpMapName
      intfConfig.cpuTcToCosMapName = cfgCpuTcToCosMapName
      intfConfig.schedulerGroupName = cfgSchedulerGroupName
      intfConfig.schedulerCompensation = cfgSchedulerCompensation
      intfConfig.policerPktSizeAdjProfileName = cfgPolicerPktSizeAdjProfileName
      intfConfig.dscpPreserveIpMplsEncapMode = cfgDscpPreserveIpMplsEncap
      # Will delete if the values are invalid or hw defaults
      for txQueue, txQueueConfig in intfConfig.txQueueConfig.items():
         if isDefaultTxQueueConfig( txQueueConfig ):
            del intfConfig.txQueueConfig[ txQueue ]

      if isDefaultIntfConfig( qosHwStatus, intfConfig ):
         del cfg.intfConfig[ intf ]
   else:  # interface config doesn't exist
      if ( isDefaultTrustMode( cfgTrustMode ) and
           isDefaultCos( intf, cfgDefaultCos, qosHwStatus ) and
           isDefaultDscp( intf, cfgDefaultDscp, qosHwStatus ) and
           isDefaultShapeRate( tempShapeRate ) and
           isDefaultGuaranteedBw( tempGuaranteedBw ) and
           isDefaultCosToTcProfile( cfgCosToTcProfileName ) and
           isDefaultExpToTcProfile( cfgExpToTcProfileName ) and
           isDefaultDscpToTcMap( cfgDscpToTcMapName ) and
           isDefaultTcToCosMap( cfgTcToCosMapName ) and
           isDefaultTcToCosMap( cfgCpuTcToCosMapName ) and
           isDefaultTcToExpMap( cfgTcDpToExpMapName ) and
           isDefaultSchedulerGroupName( cfgSchedulerGroupName ) and
           isDefaultSchedulerCompensationConfig( cfgSchedulerCompensation ) and
           isDefaultPolicerPktSizeAdjProfileName( cfgPolicerPktSizeAdjProfileName )
           and isDefaultDscpPreserveIpMplsEncapMode( cfgDscpPreserveIpMplsEncap ) ):
         # If there is no interface configuration and we are doing
         # default 'qos' command OR passing HwDefaults, then lets not create the
         # interface config
         return
      else:
         # Add a new config only if there is non-default qos config to be configured
         intfConfig = cfg.intfConfig.newMember( intf )
         intfConfig.trustMode = cfgTrustMode
         intfConfig.defaultCos = cfgDefaultCos
         intfConfig.defaultDscp = cfgDefaultDscp
         intfConfig.shapeRate = tempShapeRate
         intfConfig.guaranteedBw = tempGuaranteedBw
         intfConfig.cosToTcProfileName = cfgCosToTcProfileName
         intfConfig.expToTcProfileName = cfgExpToTcProfileName
         intfConfig.dscpToTcMapName = cfgDscpToTcMapName
         intfConfig.tcToCosMapName = cfgTcToCosMapName
         intfConfig.tcDpToExpMapName = cfgTcDpToExpMapName
         intfConfig.cpuTcToCosMapName = cfgCpuTcToCosMapName
         intfConfig.schedulerGroupName = cfgSchedulerGroupName
         intfConfig.schedulerCompensation = cfgSchedulerCompensation
         intfConfig.policerPktSizeAdjProfileName = cfgPolicerPktSizeAdjProfileName
         intfConfig.dscpPreserveIpMplsEncapMode = cfgDscpPreserveIpMplsEncap

# -----------------------------------------------------------------------------------
# Blocking Cli utility functions and vars
# -----------------------------------------------------------------------------------

def hwConfigAclVerificationSupported():
   return ( qosInputConfig.hwConfigVerificationEnabled and
            qosAclHwStatus.hwStatusReportingSupported )

def hwConfigVerificationSupported():
   return ( qosInputConfig.hwConfigVerificationEnabled and
            qosHwStatus.hwStatusReportingSupported )


supportedFeatureListOnStrataQosV2 = [ tacFeatureName.ecn,
                                      tacFeatureName.defaultCos,
                                      tacFeatureName.defaultDscp,
                                      tacFeatureName.trustMode,
                                      tacFeatureName.nonEct,
                                      tacFeatureName.wred,
                                      tacFeatureName.weight,
                                      tacFeatureName.cosRewrite,
                                      tacFeatureName.dscpRewrite,
                                      tacFeatureName.map,
                                      tacFeatureName.ecnCount,
                                      tacFeatureName.bufferBasedDecn,
                                      tacFeatureName.others ]

# Features that are programmed entirely asynchronously by PD, that do not live in
# QosBasicSm. Wait for the feature-specific hwProgrammingVersionAndStatus version
# to be the same or newer than the qosHwConfig.configVersionUpdateTime.
featuresToWaitForHwProgrammingVersion = [ tacFeatureName.bufferBasedDecn, ]

def featureSupportedOnStrataQosV2( feature ):
   return feature in supportedFeatureListOnStrataQosV2

# Waits for config to be applied at hardware, polls for ack sent
# by forwarding agents
def cliBlockingToApplyConfigChange( mode, configVersionUpdateTime, feature,
                                    description, policyMap=False ):
   def waitForHwStatusUpdate():
      qosFASupported = qosHwStatus.qosFeatureAgentSupported
      statusUpdated = True
      for sliceType in hwStatusPerSlice:
         for sliceHwStatus in sliceType.values():
            # Break and return when sliceHwStatus.values() is [ None ]
            if not sliceHwStatus:
               statusUpdated = False
               break
            if qosFASupported:
               featureSup = featureSupportedOnStrataQosV2( feature )
               if featureSup and sliceHwStatus.isSlice:
                  continue
               if not featureSup and not sliceHwStatus.isSlice:
                  continue

            if feature in featuresToWaitForHwProgrammingVersion:
               progStatus = sliceHwStatus.hwProgrammingVersionAndStatus.get(
                  feature, None )
               # If PD hasn't populated the status yet, or the status version is not
               # the latest config update time or newer, then PD has not yet updated
               # the status.
               if ( not progStatus
                    or progStatus.version < qosHwConfig.configVersionUpdateTime ):
                  statusUpdated = False
                  break
            else:
               delta = sliceHwStatus.configVersionUpdateTime - \
                   qosHwConfig.configVersionUpdateTime
               if abs( delta ) > 0.001:
                  statusUpdated = False
                  break
      return statusUpdated

   if policyMap:
      hwStatusPerSlice = [ qosAclSliceHwStatus ]
   elif feature in featuresToWaitForHwProgrammingVersion:
      # Features waiting for feature-level version time currently only populate
      # qosSliceHwStatus; not qosAclSliceHwStatus.
      hwStatusPerSlice = [ qosSliceHwStatus ]
   else:
      # All the features that use the configVersionUpdateTime reacts
      # on the same variable. All features of basicQos expect the policyQos
      # uses qosSliceHwStatus. This works well if all the SM updates the
      # variable at the same time or we can always trust the caller will
      # set policyMap=True, if they are waiting for policy map programming.
      # But in case of config session, maybe we cannot be sure what configuration
      # are changed, hence maybe we don't set policyMap=True in handleSessionCommit.
      # Now we are moving the policyQos programming to be asynchronous operation
      # that means the CLI needs to wait for longer period for the status from the
      # platform. Whereas the basicQos can still be a synchronous and will write
      # status back to qosSliceHwStatus.
      # To the solve the issue, we are going to now wait for both Qos feature to be
      # write back from the platform. Since both basicQos and policyQos react
      # to same  variable this seems to fair assumption to make both sm
      # will write back. one Sm will be slower than the other but we will
      # make sure all Sms have updated before we proceeed.
      hwStatusPerSlice = [ qosAclSliceHwStatus, qosSliceHwStatus ]
   hwStatusPerSliceCopy = hwStatusPerSlice.copy()
   for sliceType in hwStatusPerSliceCopy:
      if sliceType == qosAclSliceHwStatus:
         if not hwConfigAclVerificationSupported() or \
            ( mode and mode.session_.startupConfig() ):
            hwStatusPerSlice.remove( sliceType )
      elif sliceType == qosSliceHwStatus:
         if not hwConfigVerificationSupported() or \
            ( mode and mode.session_.startupConfig() ):
            hwStatusPerSlice.remove( sliceType )
   if not hwStatusPerSlice:
      return True, ""

   # configVersionUpdateTime is a token that is sent by the CLI and which
   # is acknowledged by forwarding agent which implies that all prior config
   # updates sent to forwarding were processed and their status are updated
   # appropriately
   # For features that wait for feature-level HW programming version, this
   # configVersionUpdateTime will have already been populated by the CLI caller.
   if feature not in featuresToWaitForHwProgrammingVersion:
      qosHwConfig.configVersionUpdateTime = configVersionUpdateTime
   t0( 'Waiting for Version: ', qosHwConfig.configVersionUpdateTime )
   result = False
   errMsg = ""
   timeout = int( os.getenv( 'QOS_TEST_CLI_TIMEOUT', str( QOS_CLI_TIMEOUT ) ) )
   try:
      Tac.waitFor( waitForHwStatusUpdate,
                   warnAfter=3.0, sleep=True,
                   maxDelay=0.1, timeout=timeout,
                   description=description )
      result = True
   except Tac.Timeout:
      for sliceType in hwStatusPerSlice:
         for sliceName, sliceHwStatus in sliceType.items():
            t0( 'Version:', sliceHwStatus.configVersionUpdateTime,
                ' seen at slice: ', sliceName )
      errMsg = "Operation timed out"
      result = False
   except KeyboardInterrupt:
      errMsg = "Keyboard Interrupt"
      result = False

   return result, errMsg

def handleSessionCommit( mode ):

   hwAck, errMsg = cliBlockingToApplyConfigChange(
      mode, Tac.now(), tacFeatureName.others, "Cli Session" )

   if hwAck:
      result = True
      for sliceHwStatus in qosSliceHwStatus.values():
         if sliceHwStatus.status != tacHwProgrammingStatus.hwPrgmSuccess:
            result = False
            errMsg = "Hardware Programming Failed"
            break
      for aclSliceHwStatus in qosAclSliceHwStatus.values():
         if aclSliceHwStatus.status != tacHwProgrammingStatus.hwPrgmSuccess:
            result = False
            errMsg = "Hardware Programming Failed"
            break
   else:
      result = False

   if not result:
      if errMsg == CLI_TIMEOUT:
         mode.addWarning( QosLib.qosTimeoutWarning() )
      else:
         warning = f"Error in QOS commit, configuration may differ " \
             f"from hardware ({errMsg})"
         mode.addWarning( warning )

def cliBlockingFail( mode, timestamp, feature, description, intf=None ):

   if ( not hwConfigVerificationSupported() or mode.session_.startupConfig() ):
      return False

   if mode.session.inConfigSession():
      CliSession.registerSessionOnCommitHandler(
         mode.session_.entityManager, "QOS",
         lambda m, onSessionCommit: handleSessionCommit( m ) )
      qosHwConfig.clearHwStatusErrorRequestTime = Tac.now()
      return False

   errMessage = None
   result, errMsg = cliBlockingToApplyConfigChange( mode, timestamp, feature,
                    description )
   if result:
      for sliceHwStatus in qosSliceHwStatus.values():
         status = sliceHwStatus.hwProgrammingVersionAndStatus.get( feature )
         if( status and
             status.hwProgrammingStatus != tacHwProgrammingStatus.hwPrgmSuccess ):
            errMsg = status.errMessage
            result = False
      for aclSliceHwStatus in qosAclSliceHwStatus.values():
         status = aclSliceHwStatus.hwProgrammingVersionAndStatus.get( feature )
         if( status and
             status.hwProgrammingStatus != tacHwProgrammingStatus.hwPrgmSuccess ):
            errMsg = status.errMessage
            result = False

   if not result:
      if not errMsg:
         errMsg = "Unknown reason"
      elif errMsg == CLI_TIMEOUT:
         mode.addWarning( QosLib.qosTimeoutWarning() )
         return False
      description = ( description + f" on {intf}" ) if intf else description
      errMessage = "Error: Programming " + description + f" ({errMsg})"
      mode.addError( errMessage )

   return errMessage

# -----------------------------------------------------------------------------------
# "traffic-class <tc> to dscp <d> in DscpRewrite mode
# -----------------------------------------------------------------------------------

class TrafficClassExpression( CliCommand.CliExpression ):
   expression = 'TC_LIST | { TC_VALUE }'
   data = {
      'TC_LIST': matcherTrafficClassList,
      'TC_VALUE': matcherTrafficClassValue,
   }

# --------------------------------------------------
# Cli Command Registration
# --------------------------------------------------

# Tx-Queue Range config mode for tx-queue range feature.
class IntfTxQueueRangeConfigMode( BasicCli.ConfigModeBase ):
   # Necessary attributes for every Mode class
   name = "Tx-Queue Range Configuration"

   def __init__( self, parent, session, tokenQueueType=None, txQueueId=None ):
      assert txQueueId is not None
      self.individualTxQueueChildModes_ = []
      for txQ in txQueueId:
         mode = IntfTxQueueConfigMode( parent, session, tokenQueueType, txQ )
         self.individualTxQueueChildModes_.append( mode )
      if tokenQueueType == 'tx-queue':
         self.queueToken = 'txq'
      elif 'uc-tx-queue' == tokenQueueType:
         self.queueToken = 'uc-txq'
      elif 'mc-tx-queue' == tokenQueueType:
         self.queueToken = 'mc-txq'
      else:
         assert 0, 'Incorrect Tx-Queue type'
      self.longModeKey = f'{parent.longModeKey}-{self.queueToken}-' \
         f'{MultiRangeRule.multiRangeToCanonicalString( txQueueId )}'
      self.modeKey = self.queueToken
      # Calling base class constructor
      BasicCli.ConfigModeBase.__init__(
         self, parent, session, multiInstance=True,
         multiModes=self.individualTxQueueChildModes_ )

   def getCompletions( self, tokens, partialToken, startWithPartialToken=False ):
      # Overriding getCompletions.
      # Any of my member modes is as good as any other for completions.
      return self.individualModes_[ 0 ].getCompletions( tokens,
                                                      partialToken,
                                                      startWithPartialToken )

   def parse( self, tokens, autoComplete=True, authz=True, acct=True ):

      # If there is a range expression, eg. {1,3,1,1} then this function will handle
      # that.
      if CliRangeExpansion.tryRangeParse( self, tokens, autoComplete=autoComplete,
                                          authz=authz, acct=acct ):
         return None
      try:
         # This is for handling commands like 'exit'.
         return BasicCli.ConfigModeBase.parse( self, tokens,
                                               autoComplete=autoComplete,
                                               authz=authz, acct=acct )
      except CliParser.ParseError:
         # If the command was not parsed then, we try to parse ourselves.
         pass

      # If the command is not parsed by any of the above parsers successfully
      # then we try to parse it with each individual child tx-queue modes.

      results = {}
      for m in self.individualModes_:
         results[ m ] = m.parse( tokens, autoComplete=autoComplete, authz=authz,
                                acct=acct )

      CmdHandler = collections.namedtuple( 'CmdHandler', [ 'handler', 'dummy' ] )
      cmdHandler = CmdHandler( self._invokeIndividualChildValueFunc, True )
      return { "cmdHandler": cmdHandler,
               "allowCache": False,
               "kargs": { "results": results, "authz": authz, "acct": acct },
               "aaa": None,
               "cmdDeprecatedBy": None }

   def _invokeIndividualChildValueFunc( self, mode, results, authz, acct ):
      try:
         # Try invoking the valueFunc of each child
         for m in self.individualModes_:
            # pylint: disable-msg=W0212
            m.session._invokeValueFunc( m, results[ m ], authz, acct )

            # Suppressing accounting for further calls
            acct = False

      finally:
         pass

   def showActive( self ):
      for m in self.individualModes_:
         m.showActive()

   def showActiveAll( self, showDetail ):
      for m in self.individualModes_:
         m.showActiveAll( showDetail )

class QosProfileMode( QosProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Qos Profile Configuration"
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, context ):
      self.qosProfileModeContext = context
      self.profileName_ = context.profileName()
      param = ( self.profileName_ )
      QosProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosProfileModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosProfileModeContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.qosProfileModeContext
      self.qosProfileModeContext = None
      context.commit()

   # print one of more profiles
   @staticmethod
   def showQosProfile( mode, args ):
      qosProfileName = args.get( 'PROFILENAME' )
      qosProfileConfig = profileConfigDir.config
      qosProfileAllModel = QosProfileAllModel()
      qosProfilesContainer = QosProfileModelContainer( qosProfileConfig,
                                                       qosHwStatus,
                                                       qosProfileAllModel )
      if qosProfileName is None:
         qosProfilesContainer.populateAll()
      else:
         qosProfilesContainer.populateQosProfile( qosProfileName )
      return qosProfileAllModel

def findServicePolicyForIntf( intf, emapType, directions=None ):
   if directions is None:
      directions = [ tacDirection.input, tacDirection.output ]
   pmapsFound = []
   if isLagPort( intf.name ):
      spConfigOrStatus = qosConfig.servicePolicyConfig
      for direction in directions:
         for key, spObj in spConfigOrStatus.items():
            if intf.name in spObj.intfIds:
               if key.direction == direction and key.type == emapType:
                  pmapsFound.append( ( key.pmapName, key.direction ) )
   else:
      spConfigOrStatus = qosStatus.servicePolicyStatus
      for direction in directions:
         for key, spObj in spConfigOrStatus.items():
            for intfPair in spObj.intfIds:
               if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfPair.lagIntf ):
                  # Members of port-channels will show policy information
                  # only from parent port-channel and not from port-channel
                  # subinterfaces.
                  continue
               if intfPair.intfId == intf.name and \
                  key.direction == direction and key.type == emapType:
                  pmapsFound.append( ( key.pmapName, key.direction ) )
   return pmapsFound

def setQosProfileConfig( profile ):
   for txQueue, txQueueConfig in profile.txQueueConfig.items():
      if isDefaultTxQueueConfig( txQueueConfig ):
         del profile.txQueueConfig[ txQueue ]
   if ( ( profile.pfcPortConfig ) and
        ( not profile.pfcPortConfig.enabled ) and
        ( not profile.pfcPortConfig.priorities ) and
        ( profile.pfcPortConfig.watchdogEnabled ) and
        ( not profile.pfcPortConfig.portTimerConfig or
          profile.pfcPortConfig.portTimerConfig ==
          Tac.Value( "Pfc::PortTimerConfig" ) ) and
        ( profile.pfcPortConfig.watchdogPortAction ==
          tacPfcWatchdogAction.invalid ) ):
      profile.pfcPortConfig = None

def convertRoutedPortSubIntfAclSharing( mode ):
   cliQosAclConfig.usingQosRoutedPortSubIntfAclSharingCli = True


# -------------------------------------------------------------------------------
# Register convertRoutedPortSubIntfAclSharing via "config convert new-syntax"
# -------------------------------------------------------------------------------
CliPlugin.ConfigConvert.registerConfigConvertCallback(
                                             convertRoutedPortSubIntfAclSharing )

@Plugins.plugin( provides=( "QosCliCommon", ) )
def Plugin( entityManager ):
   global qosAclHwStatus, qosHwStatus, qosInputConfig, qosConfig, qosStatus, \
      qosSliceHwStatus, subIntfHwStatus, qosTmMapping, qosGlobalConfig, \
      qosInputProfileConfig, lagInputConfig, qosSchedulerConfig, \
      cliQosAclConfig, qosHwConfig, qosAclSliceHwStatus, profileConfigDir
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
   qosTmMapping = LazyMount.mount( entityManager, "hardware/tm/mapping",
                                  "Qos::TmMapping", "r" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
   qosInputProfileConfig = ConfigMount.mount(
         entityManager, "qos/input/config/qosProfile",
         "Qos::Input::Config", "w" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
   qosSchedulerConfig = ConfigMount.mount( entityManager, "qos/scheduler/config",
                                           "Qos::QosSchedulerConfig", "w" )
   cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   qosHwConfig = LazyMount.mount( entityManager, "qos/hardware/config",
                                  "Qos::HwConfig", "w" )
   qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   profileConfigDir = ConfigMount.mount( entityManager, "qos/profile",
                                         "Qos::QosProfileConfigDir", "w" )
