#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Plugins
import SharedMem
import ShowCommand
import Smash
import Tac
import Tracing
from BasicCliModes import GlobalConfigMode
from CliPlugin import FabricConfigCli
from CliPlugin.QosCliIntfTypes import ( ethIntfPrefixes, ethOrLagIntfTypes,
                                        ethIntfTypes )
from CliPlugin.QosCliModel import ( EcnCountersModel, EcnDelayModel,
                                   EcnIntfQueueCountersModel, GlobalAllEcnModel,
                                   GlobalEcnModel, IntfEcnCollectionModel,
                                   IntfAllEcnCollectionModel,
                                   IntfAllBufferBasedDecnModel )
from CliPlugin.QosGlobalConfigMode import matcherMlsGlobal
from CliPlugin.QosCli import ( nodeQosForConfig, nodeQosForShow )
from CliPlugin.QosCliCommon import ( nodeRandomDetect, guardGlobalEcn,
                                     weightRangeFn, nodeTxQueueRandomDetect,
                                     randomDetectCmdsAdapter, IntfUcTxQueueModelet,
                                     getIntfListFromMode, isSubIntfConfigMode,
                                     cliBlockingFail, QosProfileMode, defaultWeight )
from CliPlugin.QosCliWred import ( RandomDetectCmdsThdWithLatencyFactory,
                                   RandomDetectCmdsThdFactory,
                                   segmentsKwMatcher,
                                   bytesKwMatcher,
                                   kbytesKwMatcher,
                                   mbytesKwMatcher )
from Intf.IntfRange import IntfRangeMatcher
from QosTypes import tacFeatureName

__defaultTraceHandle__ = Tracing.Handle( 'QosCliEcn' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

# -----------------------------------------------------------------------------------
# The Qos globals
# -----------------------------------------------------------------------------------
qosInputConfig = None
qosInputProfileConfig = None
qosConfig = None
qosStatus = None
qosHwConfig = None
qosHwStatus = None
entMan = None
qosSliceHwStatus = None
qosHwCounter = None
ecnCounterTable = None
ecnSnapshotTable = None
qosAclHwStatus = None

FapId = Tac.Type( 'FlexCounters::FapId' )

# -----------------------------------------------------------------------------------
# Guards begin
# -----------------------------------------------------------------------------------
def guardEcnCounters( mode, token ):
   if qosHwStatus.ecnCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnShow( mode, token ):
   if qosHwStatus.ecnDelaySupported or qosHwStatus.ecnCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcn( mode, token ):
   if qosHwStatus.ecnSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnDynamic( mode, token ):
   if qosHwStatus.bufferBasedDecnSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnWeight( mode, token ):
   if qosHwStatus.ecnWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardGlobalEcnWeight( mode, token ):
   if qosHwStatus.globalEcnWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnConfigureNonEct( mode, token ):
   if qosHwStatus.ecnConfigureNonEctSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnAllowNonEctChipBased( mode, token ):
   if qosHwStatus.tcamAndWredProfilesBasedAllowNonEctSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnQueueCounter( mode, token ):
   if not qosHwStatus.ecnQueueCounterSupported:
      return CliParser.guardNotThisPlatform
   if isinstance( mode.parent_, QosProfileMode ):
      if qosHwStatus.ecnQCtrViaQosProfileSupported:
         return None
      else:
         return CliParser.guardNotThisPlatform
   intfList = getIntfListFromMode( mode.parent_ )
   for intf in intfList:
      if not intf.startswith( ethIntfPrefixes ):
         return CliParser.guardNotThisPlatform
   return None

def guardShowEcnQueueCounter( mode, token ):
   if not qosHwStatus.ecnQueueCounterSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardEcnMarkProb( mode, token ):
   if not qosHwStatus.ecnMarkProbSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardEcnDelay( mode, token ):
   if isSubIntfConfigMode( mode ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.ecnDelaySupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnDelayPerTxQ( mode, token ):
   if qosHwStatus.ecnDelayPerTxQSupported:
      return None
   return CliParser.guardNotThisPlatform

def delayEcnPerTxQThresholdRangeMillisecondsFn( mode, context=None ):
   return ( 1, qosHwStatus.maxEcnDelayPerTxQThresholdNs // 1000000 )  # milliseconds

def delayEcnPerTxQThresholdRangeMicrosecondsFn( mode, context=None ):
   return ( 1, qosHwStatus.maxEcnDelayPerTxQThresholdNs // 1000 )  # microseconds

# -----------------------------------------------------------------------------------
# Guards end
# -----------------------------------------------------------------------------------


EcnCounterKey = Tac.Type( 'Qos::EcnCounterKey' )

# -----------------------------------------------------------------------------------
# Tokens begin
# -----------------------------------------------------------------------------------
matcherDroprate = CliMatcher.IntegerMatcher( 1, 100,
      helpdesc='Specify maximum drop rate' )
matcherMinimumThresholdEcn = CliMatcher.KeywordMatcher( 'minimum-threshold',
      helpdesc='Set minimum threshold for ECN' )
thresholdMatcher = RandomDetectCmdsThdFactory()
thresholdLatencyMatcher = RandomDetectCmdsThdWithLatencyFactory()
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
      helpdesc='Show QoS status for a specific interface' )
matcherMls = CliMatcher.KeywordMatcher( 'mls',
      helpdesc='MLS global commands' )
matcherRandomDetect = CliMatcher.KeywordMatcher( 'random-detect',
      helpdesc='Show WRED based congestion control parameters' )
matcherIntf = IntfRangeMatcher( explicitIntfTypes=ethOrLagIntfTypes )
nodeMls = CliCommand.Node( matcher=matcherMls, hidden=True )
nodeEcnShow = CliCommand.guardedKeyword( 'ecn', "Show ECN parameters",
      guard=guardEcn )
nodeEcnSet = CliCommand.guardedKeyword( 'ecn',
      helpdesc='Set ECN parameters', guard=guardEcn )
nodeEcnDynamic = CliCommand.guardedKeyword(
      'dynamic', helpdesc='Dynamic ECN marking', guard=guardEcnDynamic )
nodeFabric = CliCommand.guardedKeyword( 'fabric', "Fabric interface",
      guard=FabricConfigCli.isConfigurableFabricSystem )
nodeEcnQueueCount = CliCommand.guardedKeyword( 'count',
      helpdesc='Count ECN marked packets', guard=guardEcnQueueCounter )
nodeEcnDelay = CliCommand.guardedKeyword( 'ecn',
      helpdesc='General ECN configuration', guard=guardEcnDelay )
helpStrDelayEcn = 'Configure parameters for delay-based ECN'
nodeDelay = CliCommand.guardedKeyword( 'delay',
               helpdesc=helpStrDelayEcn,
               guard=guardEcnDelay )
nodeDelayTxQ = CliCommand.guardedKeyword( 'delay',
                  helpdesc=helpStrDelayEcn,
                  guard=guardEcnDelayPerTxQ )

helpStrDelayEcnThd = 'Configure delay threshold'
nodeThreshold = CliCommand.guardedKeyword( 'threshold',
                   helpdesc=helpStrDelayEcnThd,
                   guard=guardEcnDelay )
helpStrThdMicroseconds = 'Threshold value in microseconds'
helpStrThdMilliseconds = 'Threshold value in milliseconds'
millisecondThdMatcher = CliCommand.Node(
                            matcher=CliMatcher.DynamicIntegerMatcher(
                            delayEcnPerTxQThresholdRangeMillisecondsFn,
                            helpdesc=helpStrThdMilliseconds ) )
microsecondThdMatcher = CliCommand.Node(
                            matcher=CliMatcher.DynamicIntegerMatcher(
                            delayEcnPerTxQThresholdRangeMicrosecondsFn,
                            helpdesc=helpStrThdMicroseconds ) )
# Ecn drop probability
ecnMaxDroprate = 100

def ecnDelayThresholdUsRangeFn( mode, context=None ):
   return ( qosHwStatus.minEcnDelayThreshold.threshold / 1000,
            qosHwStatus.maxEcnDelayThreshold.threshold / 1000 )


matcherEcnDelayThresholdUsValue = CliMatcher.DynamicIntegerMatcher(
      rangeFn=ecnDelayThresholdUsRangeFn,
      helpdesc=helpStrThdMicroseconds )

def setTxQueueNonEct( mode, txQueueConfig, noOrDefaultKw=True,
                      minThd=None, maxThd=None,
                      unit=None, weight=defaultWeight ):
   prevNonEctConfig = txQueueConfig.nonEctConfig
   if noOrDefaultKw:
      if prevNonEctConfig:
         txQueueConfig.nonEctConfig = None
   else:
      warningMsg = "NON-ECT thresholds will not take effect " \
      "as ECN is not enabled on this tx-queue"
      if not ( prevNonEctConfig and
               prevNonEctConfig.minThd == minThd and
               prevNonEctConfig.maxThd == maxThd and
               prevNonEctConfig.unit == unit and
               prevNonEctConfig.weight == weight ):
         if txQueueConfig.ecnConfig is None:
            mode.addWarning( warningMsg )
         txQueueConfig.nonEctConfig = ( minThd, maxThd, unit,
                                        ecnMaxDroprate, weight )
   return txQueueConfig

# -----------------------------------------------------------------------------------
# "random-detect ecn count" command under tx-queue
# -----------------------------------------------------------------------------------
def setEcnQueueCounterConfig( mode, intf, txQueueId, queueCounterCfg, enable ):
   if enable == queueCounterCfg.counterEnable:
      return

   queueCounterCfg.counterEnable = enable

   timestamp = Tac.now()
   if enable:
      # Disable ecn count should always success, no need to block cli.
      if intf and cliBlockingFail( mode, timestamp, tacFeatureName.ecnCount,
                                   "ECN Count", intf ):
         queueCounterCfg.counterEnable = False
         enable = False
   if not enable:
      if intf:
         intfCounterCfg = qosInputConfig.ecnIntfCounterConfig.get( intf )
         if intfCounterCfg:
            del intfCounterCfg.ecnTxQueueCounterConfig[ txQueueId ]
         if len( intfCounterCfg.ecnTxQueueCounterConfig ) == 0:
            del qosInputConfig.ecnIntfCounterConfig[ intf ]
      else:
         profile = mode.parent_.qosProfileModeContext.currentEntry_
         del profile.ecnTxQueueCounterConfig[ txQueueId ]
      qosHwConfig.configVersionUpdateTime = timestamp

# -----------------------------------------------------------------------------------
# Tokens end
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# Commands begin
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# [ mls ] qos ecn delay threshold <n> [microseconds], in "global config" mode
# ecn delay, in "uc-tx-queue" mode
# ecn delay threshold <n> [microseconds], in "uc-tx-queue" mode
# -----------------------------------------------------------------------------------
class IntfUcTxQueueEcnDelayThCmd( CliCommand.CliCommandClass ):
   syntax = 'ecn delay [ threshold THRESHOLD [ microseconds ] ]'
   noOrDefaultSyntax = 'ecn delay [ threshold ... ]'
   data = {
      'ecn': nodeEcnDelay,
      'delay': nodeDelay,
      'threshold': nodeThreshold,
      'THRESHOLD': matcherEcnDelayThresholdUsValue,
      'microseconds': 'Microseconds',
   }

   handler = 'QosCliEcnHandler.setTxQueueEcnDelayThresholdUs'
   noOrDefaultHandler = handler


IntfUcTxQueueModelet.addCommandClass( IntfUcTxQueueEcnDelayThCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos random-detect ecn allow non-ect
# --------------------------------------------------------------------------------
class QosRandomDetectEcnAllowNonEctCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos random-detect ecn allow non-ect'
   noOrDefaultSyntax = 'qos random-detect ecn allow non-ect'
   data = {
      'mls': 'MLS global commands',
      'qos': nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'ecn': CliCommand.guardedKeyword( 'ecn',
                                        helpdesc='Set ECN parameters',
                                        guard=guardGlobalEcn ),
      'allow': CliCommand.guardedKeyword( 'allow',
                                          helpdesc='Allow forwarding of packets '
                                                   'regardless of WRED parameters',
                                          guard=guardEcnConfigureNonEct ),
      'non-ect': 'Non ECN-Capable Transport',
   }
   handler = 'QosCliEcnHandler.configureAllowNonEct'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosRandomDetectEcnAllowNonEctCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos random-detect ecn allow non-ect chip-based
# --------------------------------------------------------------------------------
class QosRandomDetectEcnAllowNonEctChipBasedCmd(
      CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos random-detect ecn allow non-ect chip-based'
   noOrDefaultSyntax = 'qos random-detect ecn allow non-ect chip-based'
   data = {
      'mls': 'MLS global commands',
      'qos': nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'ecn': CliCommand.guardedKeyword( 'ecn',
                                        helpdesc='Set ECN parameters',
                                        guard=guardGlobalEcn ),
      'allow': CliCommand.guardedKeyword( 'allow',
                                          helpdesc='Allow forwarding of packets '
                                          'regardless of WRED parameters',
                                          guard=guardEcnConfigureNonEct ),
      'non-ect': 'Non ECN-Capable Transport',
      'chip-based': CliCommand.guardedKeyword( 'chip-based',
                                               helpdesc='Use forwarding chip '
                                               'mechanisms other than the TCAM',
                                               guard=guardEcnAllowNonEctChipBased ),
      }
   handler = 'QosCliEcnHandler.configureAllowNonEctChipBased'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosRandomDetectEcnAllowNonEctChipBasedCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos random-detect ecn global-buffer minimum-threshold MINTHD segments
# maximum-threshold MAXTHD segments [ weight GLOBALECNWEIGHTVALUE ]
# --------------------------------------------------------------------------------
class QosRandomDetectEcnGlobalBufferCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos random-detect ecn global-buffer minimum-threshold '\
            'THD_EXPR [ weight WEIGHT ]'
   noOrDefaultSyntax = '[ mls ] qos random-detect ecn global-buffer ...'
   data = {
      'mls': 'MLS global commands',
      'qos': nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'ecn': CliCommand.guardedKeyword( 'ecn',
                                        helpdesc='Set ECN parameters',
                                        guard=guardGlobalEcn ),
      'global-buffer': 'Set global shared memory thresholds',
      'minimum-threshold': 'Set minimum threshold for ECN',
      'THD_EXPR': thresholdMatcher,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set global ECN weight',
                                           guard=guardGlobalEcnWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           weightRangeFn,
                                           helpdesc='Value of global ECN weight' ),
   }

   adapter = randomDetectCmdsAdapter
   handler = 'QosCliEcnHandler.setGlobalEcn'
   noOrDefaultHandler = 'QosCliEcnHandler.setNoOrDefaultGlobalEcn'


GlobalConfigMode.addCommandClass( QosRandomDetectEcnGlobalBufferCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos ecn delay threshold ECNDELAYTHRESHOLDUSVALUE [ microseconds ]
# [ mls ] qos ecn delay threshold ...
# --------------------------------------------------------------------------------
class QosEcnDelayThresholdEcndelayCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos ecn delay threshold '
              'ECNDELAYTHRESHOLDUSVALUE [ microseconds ]' )
   noOrDefaultSyntax = '[ mls ] qos ecn delay threshold ...'
   data = {
      'mls': matcherMlsGlobal,
      'qos': nodeQosForConfig,
      'ecn': nodeEcnDelay,
      'delay': nodeDelay,
      'threshold': nodeThreshold,
      'ECNDELAYTHRESHOLDUSVALUE': matcherEcnDelayThresholdUsValue,
      'microseconds': 'Microseconds',
   }

   handler = 'QosCliEcnHandler.setGlobalEcnDelayThresholdUs'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosEcnDelayThresholdEcndelayCmd )

# -----------------------------------------------------------------------------------
# "random-detect ecn count" command under tx-queue
# -----------------------------------------------------------------------------------
class UcTxQueueModeRandDetectEcnCmd( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn count'
   noOrDefaultSyntax = syntax
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'ecn': nodeEcnSet,
      'count': nodeEcnQueueCount,
   }

   handler = 'QosCliEcnHandler.setEcnQueueCount'
   noOrDefaultHandler = handler


IntfUcTxQueueModelet.addCommandClass( UcTxQueueModeRandDetectEcnCmd )

# --------------------------------------------------------------------------------
# show qos ecn counters
# --------------------------------------------------------------------------------
class QosEcnCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos ecn counters'
   data = {
      'qos': nodeQosForShow,
      'ecn': CliCommand.guardedKeyword( 'ecn', "Show general ECN parameters",
         guard=guardEcnShow ),
      'counters': CliCommand.guardedKeyword( 'counters',
         "Show per chip ECN counters", guard=guardEcnCounters ),
   }

   handler = 'QosCliEcnHandler.showEcnCounters'
   cliModel = EcnCountersModel


BasicCli.addShowCommandClass( QosEcnCountersCmd )

# --------------------------------------------------------------------------------
# [ no | default ] random-detect ecn minimum-threshold <MIN_THD> <unit>
# maximum-threshold <MAX_THD> <unit> [ max-mark-probability MAX_DROPRATE ]
# [ weight WEIGHT ]
# unit can be one of bytes, kbytes, mbytes, segments, milliseconds, or nanoseconds
# --------------------------------------------------------------------------------
class RandomDetectEcnCmds( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn minimum-threshold THD_EXPR '\
            '[ max-mark-probability MAX_DROPRATE ] [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect ecn minimum-threshold...'
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'ecn': nodeEcnSet,
      'minimum-threshold': matcherMinimumThresholdEcn,
      'THD_EXPR': thresholdLatencyMatcher,
      'max-mark-probability': CliCommand.guardedKeyword( 'max-mark-probability',
                                    helpdesc='Set maximum marking probability rate',
                                    guard=guardEcnMarkProb ),
      'MAX_DROPRATE': matcherDroprate,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set ECN weight',
                                           guard=guardEcnWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           weightRangeFn,
                                           helpdesc='Value of ECN weight' ),
   }

   adapter = randomDetectCmdsAdapter
   handler = 'QosCliEcnHandler.setEcn'
   noOrDefaultHandler = 'QosCliEcnHandler.setNoOrDefaultEcn'


IntfUcTxQueueModelet.addCommandClass( RandomDetectEcnCmds )

# --------------------------------------------------------------------------------
# [ no | default ] random-detect ecn delay threshold <THD> <unit>
# unit can be one of milliseconds, microseconds
# --------------------------------------------------------------------------------
class RandomDetectEcnDelayPerTxQCmds( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn delay threshold ( MILLISECONDS milliseconds ) | '\
            '( MICROSECONDS microseconds )'
   noOrDefaultSyntax = 'random-detect ecn delay...'
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'ecn': nodeEcnSet,
      'delay': nodeDelayTxQ,
      'threshold': helpStrDelayEcnThd,
      'MILLISECONDS': millisecondThdMatcher,
      'milliseconds': 'Set threshold in milliseconds',
      'MICROSECONDS': microsecondThdMatcher,
      'microseconds': 'Set threshold in microseconds'
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      milliStr, microStr = 'milliseconds', 'microseconds'
      if milliStr in args:
         thd = args.pop( 'MILLISECONDS' )
         args.pop( milliStr )
         args[ 'THRESHOLD' ] = ( thd, milliStr )
      elif microStr in args:
         thd = args.pop( 'MICROSECONDS' )
         args.pop( microStr )
         args[ 'THRESHOLD' ] = ( thd, microStr )

   handler = 'QosCliEcnHandler.setDelayEcnTxQ'
   noOrDefaultHandler = 'QosCliEcnHandler.setNoOrDefaultEcnDelayTxQ'


IntfUcTxQueueModelet.addCommandClass( RandomDetectEcnDelayPerTxQCmds )

# --------------------------------------------------------------------------------
# show qos interfaces [ INTF ] ecn counters queue
# --------------------------------------------------------------------------------
class QosInterfacesEcnCountersQueueCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces [ INTF ] ecn counters queue'
   data = {
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': IntfRangeMatcher( explicitIntfTypes=ethIntfTypes ),
      'ecn': nodeEcnShow,
      'counters': CliCommand.guardedKeyword( 'counters',
         "ECN interface counters", guard=guardShowEcnQueueCounter ),
      'queue': 'ECN queue counters',
   }

   handler = 'QosCliEcnHandler.showEcnIntfQueueCounters'
   cliModel = EcnIntfQueueCountersModel


BasicCli.addShowCommandClass( QosInterfacesEcnCountersQueueCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos random-detect ecn
# --------------------------------------------------------------------------------
class QosRandomDetectEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos random-detect ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'random-detect': CliCommand.guardedKeyword( 'random-detect',
         "Show WRED based congestion control parameters",
         guard=guardGlobalEcn ),
      'ecn': nodeEcnShow,
   }
   hidden = True

   handler = 'QosCliEcnHandler.showGlobalEcn'
   cliModel = GlobalEcnModel


BasicCli.addShowCommandClass( QosRandomDetectEcnCmd )

# --------------------------------------------------------------------------------
# Buffer-based DECN helpers
# --------------------------------------------------------------------------------

segmentsFloorMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxFloorSegments ),
   helpdesc='Floor value (in segments)' )
segmentsOffsetMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxOffsetSegments ),
   helpdesc='Offset value (in segments)' )
bytesFloorMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxFloorSegments *
                          qosHwStatus.ecnSegmentSizeInBytes ),
   helpdesc='Floor value (in bytes)' )
bytesOffsetMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxOffsetSegments *
                          qosHwStatus.ecnSegmentSizeInBytes ),
   helpdesc='Offset value (in bytes)' )
kbytesFloorMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxFloorSegments *
                          qosHwStatus.ecnSegmentSizeInBytes // 1000 ),
   helpdesc='Floor value (in kbytes)' )
kbytesOffsetMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxOffsetSegments *
                          qosHwStatus.ecnSegmentSizeInBytes // 1000 ),
   helpdesc='Offset value (in kbytes)' )
mbytesFloorMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxFloorSegments *
                          qosHwStatus.ecnSegmentSizeInBytes // 1000000 ),
   helpdesc='Floor value (in mbytes)' )
mbytesOffsetMatcher = CliMatcher.DynamicIntegerMatcher(
   lambda mode, ctx: ( 0, qosHwStatus.bufferBasedDecnMaxOffsetSegments *
                          qosHwStatus.ecnSegmentSizeInBytes // 1000000 ),
   helpdesc='Offset value (in mbytes)' )

class BufferBasedDecnFloorOffsetExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class Expr( CliCommand.CliExpression ):
         # The expression is set up to enforce the same unit to be used for both the
         # floor and offset.
         expression = (
            'floor '
            '( ( FLOOR_SEGMENTS FLOOR_UNIT_SEGMENTS '
            'offset OFFSET_SEGMENTS OFFSET_UNIT_SEGMENTS ) | '
            '( FLOOR_BYTES FLOOR_UNIT_BYTES '
            'offset OFFSET_BYTES OFFSET_UNIT_BYTES ) | '
            '( FLOOR_KBYTES FLOOR_UNIT_KBYTES '
            'offset OFFSET_KBYTES OFFSET_UNIT_KBYTES ) | '
            '( FLOOR_MBYTES FLOOR_UNIT_MBYTES '
            'offset OFFSET_MBYTES OFFSET_UNIT_MBYTES ) )' )
         data = {
            'floor': CliMatcher.KeywordMatcher( 'floor', helpdesc=(
               "if (Queue's shared limit - ECN Offset) >= ECN Floor then ECN "
               "Threshold = (Queue's shared limit - ECN Offset); "
               "if Queue's shared limit >= ECN Floor >= (Queue's shared limit - "
               "ECN Offset) then ECN Threshold = ECN Floor; "
               "if ECN Floor >=  Queue's shared limit then ECN Threshold = Queue's "
               "shared limit" ) ),
            'offset': CliMatcher.KeywordMatcher( 'offset', helpdesc=(
               "if (Queue's shared limit - ECN Offset) >= ECN Floor then ECN "
               "Threshold = (Queue's shared limit - ECN Offset); "
               "if Queue's shared limit >= ECN Floor >= (Queue's shared limit - "
               "ECN Offset) then ECN Threshold = ECN Floor; "
               "if ECN Floor >=  Queue's shared limit then ECN Threshold = Queue's "
               "shared limit" ) ),
            'FLOOR_SEGMENTS': segmentsFloorMatcher,
            'OFFSET_SEGMENTS': segmentsOffsetMatcher,
            'FLOOR_UNIT_SEGMENTS': segmentsKwMatcher,
            'OFFSET_UNIT_SEGMENTS': segmentsKwMatcher,
            'FLOOR_BYTES': bytesFloorMatcher,
            'OFFSET_BYTES': bytesOffsetMatcher,
            'FLOOR_UNIT_BYTES': bytesKwMatcher,
            'OFFSET_UNIT_BYTES': bytesKwMatcher,
            'FLOOR_KBYTES': kbytesFloorMatcher,
            'OFFSET_KBYTES': kbytesOffsetMatcher,
            'FLOOR_UNIT_KBYTES': kbytesKwMatcher,
            'OFFSET_UNIT_KBYTES': kbytesKwMatcher,
            'FLOOR_MBYTES': mbytesFloorMatcher,
            'OFFSET_MBYTES': mbytesOffsetMatcher,
            'FLOOR_UNIT_MBYTES': mbytesKwMatcher,
            'OFFSET_UNIT_MBYTES': mbytesKwMatcher,
         }

         @staticmethod
         def adapter( mode, args, argList ):
            for unitName in [ 'SEGMENTS', 'BYTES', 'KBYTES', 'MBYTES', ]:
               unit = args.get( f'FLOOR_UNIT_{unitName}', None )
               if unit is None:
                  continue
               floor = args[ f'FLOOR_{unitName}' ]
               offset = args[ f'OFFSET_{unitName}' ]
               args[ 'OFFSET_FLOOR_EXPR' ] = ( offset, floor, unit )
               return
            # If we reach this point, then the no or default handler was called

      return Expr

bufferBasedDecnFloorOffsetMatcher = BufferBasedDecnFloorOffsetExprFactory()

# --------------------------------------------------------------------------------
# [ no | default ] random-detect ecn dynamic floor <FLOOR> <unit> offset <OFFSET>
# <unit>
# unit can be one of bytes, kbytes, mbytes, segments
# --------------------------------------------------------------------------------
class BufferBasedDecnConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn dynamic OFFSET_FLOOR_EXPR'
   noOrDefaultSyntax = 'random-detect ecn dynamic...'
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'ecn': nodeEcnSet,
      'dynamic': nodeEcnDynamic,
      'OFFSET_FLOOR_EXPR': bufferBasedDecnFloorOffsetMatcher,
   }

   handler = 'QosCliEcnHandler.setBufferBasedDecn'
   noOrDefaultHandler = 'QosCliEcnHandler.setNoOrDefaultBufferBasedDecn'

IntfUcTxQueueModelet.addCommandClass( BufferBasedDecnConfigCmd )

# --------------------------------------------------------------------------------
# show qos interfaces [ INTF ] ecn dynamic
# --------------------------------------------------------------------------------
class ShowBufferBasedDecnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces [ INTF ] ecn dynamic'
   data = {
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'ecn': nodeEcnShow,
      'dynamic': nodeEcnDynamic,
   }

   handler = 'QosCliEcnHandler.showInterfacesBufferBasedDecn'
   cliModel = IntfAllBufferBasedDecnModel

BasicCli.addShowCommandClass( ShowBufferBasedDecnCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos ecn delay
# --------------------------------------------------------------------------------
class QosEcnDelayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos ecn delay'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'ecn': CliCommand.guardedKeyword( 'ecn', "Show general ECN parameters",
         guard=guardEcnShow ),
      'delay': CliCommand.guardedKeyword( 'delay',
         "Show parameters for delay-based ECN", guard=guardEcnDelay ),
   }
   hidden = True

   handler = 'QosCliEcnHandler.showEcnDelay'
   cliModel = EcnDelayModel


BasicCli.addShowCommandClass( QosEcnDelayCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos ecn
# --------------------------------------------------------------------------------
class QosEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'ecn': nodeEcnShow,
   }

   handler = 'QosCliEcnHandler.showGlobalEcn'
   cliModel = GlobalAllEcnModel


BasicCli.addShowCommandClass( QosEcnCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn
# --------------------------------------------------------------------------------
class QosInterfacesRandomDetectEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'fabric': nodeFabric,
      'random-detect': matcherRandomDetect,
      'ecn': nodeEcnShow,
   }
   hidden = True

   handler = 'QosCliEcnHandler.showInterfacesEcn'
   cliModel = IntfEcnCollectionModel


BasicCli.addShowCommandClass( QosInterfacesRandomDetectEcnCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF | fabric ] ecn
# --------------------------------------------------------------------------------
class QosInterfacesEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF | fabric ] ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'fabric': nodeFabric,
      'ecn': nodeEcnShow,
   }

   handler = 'QosCliEcnHandler.showInterfacesEcn'
   cliModel = IntfAllEcnCollectionModel


BasicCli.addShowCommandClass( QosInterfacesEcnCmd )
# -----------------------------------------------------------------------------------
# Commands end
# -----------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosCliEcn", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosInputProfileConfig, qosConfig, qosHwConfig, \
          qosStatus, qosHwStatus, qosAclHwStatus, \
          entMan, qosSliceHwStatus, \
          qosHwCounter, \
          ecnCounterTable, ecnSnapshotTable

   entMan = entityManager
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosInputProfileConfig = ConfigMount.mount(
         entityManager, "qos/input/config/qosProfile",
         "Qos::Input::Config", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosHwConfig = LazyMount.mount( entityManager, "qos/hardware/config",
                                  "Qos::HwConfig", "w" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosHwCounter = LazyMount.mount( entityManager,
                                   "qos/hardware/counter",
                                   "Qos::HwCounter", "ri" )

   # Mount ECN Queue counter current and snapshot smash tables.
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   mountPath = f'flexCounters/counterTable/Ecn/{FapId.allFapsId}'
   ecnCounterTable = shmemEm.doMount( mountPath, "Qos::EcnCounterTable",
                                      readerInfo )
   mountPath = f'flexCounters/snapshotTable/Ecn/{FapId.allFapsId}'
   ecnSnapshotTable = shmemEm.doMount( mountPath, "Qos::EcnCounterTable",
                                       readerInfo )
