# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=consider-using-f-string
# pylint: disable=singleton-comparison

# pylint: disable-msg=ungrouped-imports
import sys
from collections import OrderedDict
from Ark import utcTimestampToStr
import ArPyUtils
import AclCliLib
import Arnet
import QosLib
import Tac
from CliMode.Intf import IntfMode
from CliModel import Bool, Dict, Enum, Float, Int, List, Model, Str, Submodel
from HumanReadable import formatValueSi, scaleValueSi, formatValueSigFigs
from Intf.IntfRange import intfListToCanonical
from IntfModels import Interface
import CliPlugin.IntfCli as IntfCli  # pylint: disable=consider-using-from-import
from QosLib import ( burstUnitSymbolFromEnum, fabricIntfName, fabricTokenName,
                     qosMapType, tcDpToDscpSupportedDpValues, coppMapType,
                     tcDpToExpSupportedDpValues )
from QosTypes import ( tacBurstUnit, tacBurstVal, tacCosToTcProfileName,
                      tacExpToTcProfileName,
                       tacDscpToTcMapName, tacDscpRewriteMapName,
                       tacLatencyThreshold, tacPercent, tacPolicyMapType,
                       tacTcToCosMapName, tacSchedulerCompensation,
                       tacPolicerPktSizeAdjProfileName, tacPolicerPktSizeAdj,
                       tacDirection, tacQueueWeight, tacTcDpToExpMapName,
                       tacTxQueueSchedulerProfileName )

from TableOutput import Format, TableFormatter, createTable, terminalWidth
from TypeFuture import TacLazyType
from CliPlugin.QueueCountersModel import ShapeRateModel, percentReferenceStr
from functools import reduce  # pylint: disable=redefined-builtin
from Toggles.QosToggleLib import ( toggleQosRateCounterEnabled,
                                   toggleWredEcnNotMutuallyExclusiveEnabled )

QueueThresholdUnit = TacLazyType( "Qos::QueueThresholdUnit" )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )

INVALID = -1
dontCareEcn = Tac.Type( "Acl::Ecn" ).dontCare

def pfcDegradeHelper( dictRepr, revision, keyList ):
   if revision == 1:
      modelShort = {}
      for key in keyList:
         modelShort[ key ] = intfModelShort = {}
         for intf, model in dictRepr[ key ].items():
            intfShortname = IntfMode.getShortname( intf )
            intfModelShort[ intfShortname ] = model
         dictRepr[ key ] = intfModelShort
   return dictRepr

def degradePfcWdDropsOptionalParamsHelper( dictRepr, revision, keyList ):
   if revision < 3:
      for model in dictRepr[ keyList[ 0 ] ].values():
         for subModel in model[ 'egressCounters' ].values():
            if not 'queueType' in subModel:
               subModel[ 'queueType' ] = 'unicast'
            if not 'outCurrentDrops' in subModel:
               subModel[ 'outCurrentDrops' ] = 0
            if not 'outCurrentByteDrops' in subModel:
               subModel[ 'outCurrentByteDrops' ] = 0
            if not 'outAggregateDrops' in subModel:
               subModel[ 'outAggregateDrops' ] = 0
            if not 'outAggregateByteDrops' in subModel:
               subModel[ 'outAggregateByteDrops' ] = 0

         for subModel in model[ 'ingressCounters' ].values():
            if not 'currentDrops' in subModel:
               subModel[ 'currentDrops' ] = 0
            if not 'currentByteDrops' in subModel:
               subModel[ 'currentByteDrops' ] = 0
            if not 'aggregateDrops' in subModel:
               subModel[ 'aggregateDrops' ] = 0
            if not 'aggregateByteDrops' in subModel:
               subModel[ 'aggregateByteDrops' ] = 0

def wrrBwIsValid( bandwidth ):
   if bandwidth:  # pylint: disable=simplifiable-if-statement
      return True
   else:
      return False

def shapeRateIsValid( shaperate ):
   if shaperate:  # pylint: disable=simplifiable-if-statement
      return True
   else:
      return False

def guaranteedBwIsValid( guaranteedBw ):
   if guaranteedBw:  # pylint: disable=simplifiable-if-statement
      return True
   else:
      return False

# formats ShapeRateModel into Qos::ShapeRate
def formatShapeRate( model ):
   rate = model.rate
   unit = model.unit
   percent = model.percent
   minRate = QosLib.tacShapeRateVal.min
   maxRate = QosLib.tacShapeRateVal.maxPps if unit == 'pps' \
       else QosLib.tacShapeRateVal.max
   if rate < minRate or rate > maxRate:
      shapeRate = QosLib.invalidShapeRate
   else:
      shapeRate = QosLib.Tac.Value( 'Qos::ShapeRate' )
      shapeRate.rate = rate
      shapeRate.unit = QosLib.shapeRateUnitToEnum( unit )
      shapeRate.shared = bool( model.shared )
      if percent:
         shapeRate.percent = percent
   return shapeRate

class BandwidthModel( Model ):
   rate = Int( help="Bandwith supported" )
   unit = Enum( values=( 'kbps', 'pps', 'gbps', 'unspecified' ),
                help="Bandwidth unit",
                default='unspecified' )

# Policing counters per interface
class IntfPoliceCounters( Model ):
   # Green Packets and Bytes
   conformedPackets = Int( help="Conformed number of packets to policer", default=0 )
   conformedBytes = Int( help="Conformed number of bytes to policer", default=0 )
   # Yellow Packets and Bytes
   yellowPackets = Int( help="Conformed number of packets to PIR and not CIR",
                        default=0 )
   yellowBytes = Int( help="Conformed number of bytes to PIR and not CIR",
                      default=0 )
   # Red packets and Bytes
   exceededPackets = Int( help="Exceeded number of packets to policer", default=0 )
   exceededBytes = Int( help="Exceeded number of bytes to policer", default=0 )

   # Dropped bits rate (Red packets)
   droppedBitsRate = Float( help="Dropped bits rate in bps", optional=True )

   # Comformed bits rate (Green packets)
   conformedBitsRate = Float( help="Conformed bits rate in bps", optional=True )

   # Excess bit rate (Yellow packets)
   exceededBitsRate = Float( help="Excess bits rate in bps", optional=True )

   updateInterval = Float( help="Stats update interval in seconds", optional=True )

# Policing counter per direction
class PoliceCounters( Model ):
   interfaces = Dict( valueType=IntfPoliceCounters, help='Police Counters',
                      optional=True )

class PoliceModel( Model ):
   __revision__ = 3

   # Yellow actions parameters
   class PolicerActions( Model ):
      dropPrecedence = Bool( help="Mark Drop-Prcedence", optional=True )
      dscp = Int( help="DSCP value", optional=True )
      dscpConfiguredAsName = Bool( help="The DSCP value was configured as a name",
                                   optional=True )

   cir = Int( help="Committed Information Rate" )
   cirUnit = Enum( values=( 'bps', 'kbps', 'mbps', 'pps' ),
                   help="CIR unit",
                   default='bps' )
   bc = Int( help="Committed Burst Rate" )
   bcUnit = Enum( values=( 'bytes', 'kbytes', 'mbytes', 'packets' ),
                  help="burst unit",
                  default='bytes' )
   pir = Int( help="Peak Information Rate" )
   pirUnit = Enum( values=( 'bps', 'kbps', 'mbps', 'pps' ),
                   help="PIR unit",
                   default='bps' )
   be = Int( help="Excess Burst Size" )
   beUnit = Enum( values=( 'bytes', 'kbytes', 'mbytes', 'packets' ),
                  help="burst unit",
                  default='bytes' )

   policerName = Str( help="Policer name", optional=True )
   yellowActions = Submodel( valueType=PolicerActions,
                             help="Yellow Actions applicable only for trTCM Policer",
                             optional=True )

   counters = Dict( valueType=PoliceCounters, help="A mapping from direction "
                    "( 'input' or 'output' ) to per interface police counters",
                    optional=True )

   greenCounter = Bool( help="Green counter is supported",
                        optional=True )
   redCounter = Bool( help="Red counter is supported",
                      optional=True )
   perSviRedCounterWithSharedPolicer = Bool(
      help="Red counter per interface in shared policer mode is supported",
      optional=True )

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         if 'counters' in dictRepr:
            if 'input' in dictRepr[ 'counters' ].keys():
               dictRepr[ 'counters' ] = dictRepr[ 'counters' ][ 'input' ]
            elif 'output' in dictRepr[ 'counters' ].keys():
               dictRepr[ 'counters' ] = dictRepr[ 'counters' ][ 'output' ]
            else:
               dictRepr[ 'counters' ] = {}
      return dictRepr

class MatchL2Params( Model ):
   vlanValue = Str( help="VLAN match value configured", optional=True )
   vlanMask = Int( help="VLAN mask used", optional=True )
   vlanMaskValid = Bool( help="Flag to identify if vlan range type is used "
                         "or (vlan id, mask)", optional=True )
   innerVlanValue = Str( help="Inner VLAN match value configured", optional=True )
   innerVlanMask = Int( help="Inner VLAN mask used", optional=True )
   innerVlanMaskValid = Bool( help="Flag to identify if inner vlan range type is "
                              "used or (innervlan id, mask) is used", optional=True )
   cosValue = Str( help="COS value configured", optional=True )
   deiValue = Str( help="DEI value configured", optional=True )
   innerCosValue = Str( help="Inner COS value configured", optional=True )

class PolicyMapModel( Model ):
   __revision__ = 4

   class HwStatus( Model ):
      unitsProgrammed = Int( help="Number of units programmed", optional=True )
      status = Enum( values=( QosLib.tacPMapHwPrgmStatus.hwPrgmStatusSuccess,
                              QosLib.tacPMapHwPrgmStatus.hwPrgmStatusFail,
                              QosLib.tacPMapHwPrgmStatus.hwPrgmStatusInProgress ),
                     help="Hardware programming status" )

   class InterfaceList( Model ):
      interfaces = List( valueType=Interface,
                         optional=True,
                         help="List of interfaces" )

   class ClassMap( Model ):
      __revision__ = 4

      class Match( Model ):
         matchRule = Enum( values=( 'matchIpAccessGroup',
                                    'matchIpv6AccessGroup',
                                    'matchL2Params',
                                    'matchDscpEcn',
                                    'matchMplsTrafficClass',
                                    'matchMacAccessGroup' ),
                           help='Matching rule for the type' )
         matchEcnValue = Enum( values=list( AclCliLib.ecnAclNames ),
                               help='ECN value configured',
                               optional=True )
         matchDscpNameValid = Bool( help="Flag to identify if string value of DSCP "
                                    "is used or numeric value",
                                    optional=True )
         dscpValue = Int( help="DSCP equivalent value in digits",
                          optional=True )
         l2ParamsValue = Submodel( valueType=MatchL2Params,
                                   help="L2 Params values configured",
                                   optional=True )

      # Interface parameters
      class IntfPacketCounters( Model ):
         outPackets = Int( help="Number of out packets", optional=True )
         dropPackets = Int( help="Number of packets dropped", optional=True )

      # Qos related parameters
      class QosParams( Model ):
         cos = Int( help="COS value", optional=True )
         dscp = Int( help="DSCP value", optional=True )
         drop = Bool( help="Drop Packets", optional=True )
         trafficClass = Int( help="Traffic Class value", optional=True )
         dropPrecedence = Int( help="Drop Precedence value", optional=True )
         dropActionRestricted = Bool( help="Is drop action restricted "
                                       "on the system", optional=True )
         dscpConfiguredAsName = Bool( help="Flag to identify if the DSCP value "
                                           "was configured as a name",
                                      optional=True )

      class ClassMapMatchPacketCounterCollection( Model ):
         # Qos policy map counter parameters
         class ClassMapMatchPacketCounter( Model ):
            packets = Int( help="Number of packets matched", optional=True )
         interfaces = Dict( valueType=ClassMapMatchPacketCounter,
                            help='Interface Matched Packets',
                            optional=True )

      class ClassMapStatistics( Model ):
         droppedBitsRate = Float( help="Dropped bits rate in bps", optional=True )
         conformedBitsRate = Float( help="Conformed bits rate in bps",
                                    optional=True )
         exceededBitsRate = Float( help="Excess bits rate in bps", optional=True )
         updateInterval = Float( help="Stats update interval in seconds",
                                 optional=True )

      name = Str( help='Name of class map' )
      classPrio = Int( help='Class priority - lower value has precedence' )
      mapType = Enum( values=( "mapQos", "mapControlPlane", "mapPdp" ),
                       help="Determines if the class map is for "
                            "CoPP or Qos or PDP policy map",
                       optional=True )
      matchedPackets = Dict( valueType=ClassMapMatchPacketCounterCollection,
                             help="A mapping from direction ('input' or 'output') "
                             "to number of packets matched",
                             optional=True )
      match = Dict( valueType=Match, help='Matching access groups',
                    optional=True )
      matchCondition = Enum( values=( 'matchConditionAny', 'unspecified' ),
                              help="Match condition",
                              optional=True )
      intfPacketCounters = Dict( valueType=IntfPacketCounters,
                                 help="A mapping from direction ('input' or "
                                 " 'output') to per interface packet counters",
                                 optional=True )
      qosParams = Submodel( valueType=QosParams,
                            help="Value of Qos parameters,"
                                 " applicable only for Qos policy maps",
                            optional=True )

      classMapStatistics = Dict( valueType=ClassMapStatistics,
                                 help="A mapping from direction ('input' or "
                                 "'output') to class maps counter statistics",
                                 optional=True )

      # Policing related parameters
      police = Submodel( valueType=PoliceModel,
                         help="Ingress policing associated with the policy class",
                         optional=True )
      count = Bool( help="Count number of packets matching the policy class",
                    optional=True )

      # Copp related parameters
      shape = Submodel( valueType=ShapeRateModel,
                        help="Shape rate associated with the policy class, "
                             " applicable only for CoPP policy maps",
                        optional=True )
      bandwidth = Submodel( valueType=BandwidthModel,
                            help="Bandwidth associated with the policy class, "
                                 " applicable only for CoPP policy maps",
                            optional=True )
      coppActionPoliceSupported = Bool( help="Control plane policer supported",
                                        optional=True )

      def degrade( self, dictRepr, revision ):
         if revision < 3:
            # matchedPackets
            if 'matchedPackets' in dictRepr:
               dirKey = None
               if 'input' in dictRepr[ 'matchedPackets' ]:
                  dirKey = 'input'
               elif 'output' in dictRepr[ 'matchedPackets' ]:
                  dirKey = 'output'
               else:
                  dictRepr[ 'matchedPackets' ] = int( 0 )

               if dirKey and dictRepr[ 'matchedPackets' ].get( dirKey ):
                  if dictRepr[ 'matchedPackets' ][ dirKey ].get( 'interfaces' ):
                     if dictRepr[ 'matchedPackets' ][ dirKey ][ 'interfaces' ].\
                           get( 'all' ):
                        dictRepr[ 'matchedPackets' ] = int(
                           dictRepr[ 'matchedPackets' ][ dirKey ][ 'interfaces' ]
                           [ 'all' ][ 'packets' ] )

            # intfPacketCounters
            if 'intfPacketCounters' in dictRepr:
               if 'input' in dictRepr[ 'intfPacketCounters' ].keys():
                  dictRepr[ 'intfPacketCounters' ] = \
                     dictRepr[ 'intfPacketCounters' ][ 'input' ]
               elif 'output' in dictRepr[ 'intfPacketCounters' ].keys():
                  dictRepr[ 'matchedPackets' ] = \
                     dictRepr[ 'intfPacketCounters' ][ 'output' ]
               else:
                  dictRepr[ 'intfPacketCounters' ] = {}

            # classMapStatistics
            if 'classMapStatistics' in dictRepr:
               if 'input' in dictRepr[ 'classMapStatistics' ].keys():
                  dictRepr[ 'classMapStatistics' ] = \
                     dictRepr[ 'classMapStatistics' ][ 'input' ]
               elif 'output' in dictRepr[ 'classMapStatistics' ].keys():
                  dictRepr[ 'matchedPackets' ] = \
                     dictRepr[ 'classMapStatistics' ][ 'output' ]
               else:
                  dictRepr[ 'classMapStatistics' ] = {}
         elif revision == 3:
            # matchedPackets
            if 'matchedPackets' in dictRepr:
               dirKey = None
               if 'input' in dictRepr[ 'matchedPackets' ]:
                  dirKey = 'input'
               elif 'output' in dictRepr[ 'matchedPackets' ]:
                  dirKey = 'output'
               else:
                  dictRepr[ 'matchedPackets' ] = {}

               if dirKey and dictRepr[ 'matchedPackets' ].get( dirKey ):
                  if dictRepr[ 'matchedPackets' ][ dirKey ].get( 'interfaces' ):
                     if dictRepr[ 'matchedPackets' ][ dirKey ][ 'interfaces' ].\
                           get( 'all' ):
                        dictRepr[ 'matchedPackets' ][ dirKey ][ 'packets' ] = \
                              int( dictRepr[ 'matchedPackets' ][ dirKey ]
                              [ 'interfaces' ][ 'all' ][ 'packets' ] )

         return dictRepr

      def getRate( self, bps ):
         ( scaledBps, bpsPrefix ) = scaleValueSi( bps )
         # If the number has been scaled with some SI prefix
         # (e.g. Mbps) then it's meaningful to display digits after
         # the decimal point
         if bpsPrefix:
            formattedBps = formatValueSigFigs( scaledBps, 3 )
         else:
            formattedBps = '%.0f' % scaledBps
         return ( formattedBps, bpsPrefix )

      def renderMatch( self, match, accessGroupName ):
         output = ""
         if match.matchRule == 'matchL2Params':
            output += '    Match:'
            l2Params = match.l2ParamsValue
            if l2Params.vlanValue:
               if l2Params.vlanMaskValid:
                  output += ' %s %s %#5.3x' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                              l2Params.vlanValue, l2Params.vlanMask )
               else:
                  output += ' %s %s' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                              l2Params.vlanValue )
            if l2Params.innerVlanValue:
               if l2Params.innerVlanMaskValid:
                  output += ' %s %s %#5.3x' % (
                     QosLib.matchFromEnum( match.matchRule )[ 2 ],
                     l2Params.innerVlanValue, l2Params.innerVlanMask )
               else:
                  output += ' %s %s' % (
                     QosLib.matchFromEnum( match.matchRule )[ 2 ],
                     l2Params.innerVlanValue )
            if l2Params.cosValue:
               output += ' %s %s' % \
                         ( QosLib.matchFromEnum( match.matchRule )[ 1 ],
                           l2Params.cosValue )
            if l2Params.innerCosValue:
               output += ' %s %s' % \
                         ( QosLib.matchFromEnum( match.matchRule )[ 3 ],
                           l2Params.innerCosValue )
            if l2Params.deiValue:
               output += ' %s %s' % \
                         ( QosLib.matchFromEnum( match.matchRule )[ 4 ],
                           l2Params.deiValue )
            output += '\n'
         elif match.matchRule == 'matchDscpEcn':
            ecnName = ""
            if match.matchEcnValue != AclCliLib.ecnNameFromValue( dontCareEcn ):
               ecnName = " %s %s" % \
                  ( QosLib.matchFromEnum( match.matchRule )[ 1 ],
                    match.matchEcnValue )
            if accessGroupName in AclCliLib.ecnAclNames:
               # only ecn case
               output += '    Match:%s\n' % ( ecnName )
            else:
               output += '    Match: %s %s%s\n' % \
                         ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                           accessGroupName, ecnName )
         elif match.matchRule == 'matchMplsTrafficClass':
            mplsTrafficClassStr = 'mpls traffic-class'
            output += '    Match: %s %s\n' % \
                ( mplsTrafficClassStr, accessGroupName )
         else:
            output += '    Match: %s name %s\n' % \
                ( QosLib.matchFromEnum( match.matchRule ),
                  accessGroupName )
         return output

      def renderMatches( self ):
         output = ""
         for accessGroupName in self.match:
            match = self.match[ accessGroupName ]
            output += self.renderMatch( match, accessGroupName )
         return output

      def renderMatchedPackets( self, direction ):
         output = ""
         if not direction in self.matchedPackets:
            return output
         if 'all' in self.matchedPackets[ direction ].interfaces:
            matchedPackets = self.matchedPackets[ direction ].interfaces[ "all" ]
            output += "      Matched Packets: %d" % matchedPackets.packets
            if len( self.matchedPackets[ direction ].interfaces ) == 1:
               return output
            output += "\n"
         for ifName in sorted( self.matchedPackets[ direction ].interfaces ):
            # Display sum of counters of all intfs
            if 'all' in ifName:
               continue
            output += "        Interface: %s\n" % ifName
            output += \
                  "          Matched Packets: %d\n" % \
                  self.matchedPackets[ direction ].interfaces[ ifName ].packets
         return output

      def renderIntfPacketCounters( self, direction ):
         output = ""
         if direction not in self.intfPacketCounters:
            return output
         intfPacketCounters = self.intfPacketCounters[ direction ]
         if intfPacketCounters.outPackets != None:
            output += "      Out Packets : %d\n" % \
               intfPacketCounters.outPackets
         if intfPacketCounters.dropPackets != None:
            output += "      Drop Packets : %d\n" % \
               intfPacketCounters.dropPackets
         return output

      def renderIntfPolicerCounters( self, ifName, ifCounter ):
         output = ""
         if self.mapType == coppMapType:
            conformedStr = 'Conformed %d packets, %d bytes\n' % (
               ifCounter.conformedPackets, ifCounter.conformedBytes )
            if 'all' in ifName:
               # display accumulated counters
               output += '        ' + conformedStr
               return output
            output += '        Interface: %s\n' % ifName
            output += '          ' + conformedStr
            return output

         conformedStr = 'Below the rate limit' \
                        if self.mapType == qosMapType else 'Conformed'
         conformedStr += ( ' %d packets, %d bytes\n' % ( ifCounter.conformedPackets,
                                                        ifCounter.conformedBytes ) )
         conformedStrTrTcm = ( 'Below the low rate limit %d packets, %d '
                           'bytes\n' ) % ( ifCounter.conformedPackets,
                                           ifCounter.conformedBytes )
         yellowStr = ( 'Above the low rate limit and below the high rate '
                     'limit %d packets, %d bytes\n' ) % ( ifCounter.yellowPackets,
                                                           ifCounter.yellowBytes )
         exceededStr = 'Above the rate limit %d packets, %d bytes\n' % (
            ifCounter.exceededPackets, ifCounter.exceededBytes )
         exceededStrTrTcm = 'Above the high rate limit %d packets,' \
                       ' %d bytes\n' % ( ifCounter.exceededPackets,
                                     ifCounter.exceededBytes )
         conformedRateStr = ''
         yellowRateStr = ''
         exceededRateStr = ''

         if self.mapType == 'mapQos' and toggleQosRateCounterEnabled() \
            and 'all' not in ifName:
            intervalStr = None
            if ifCounter.updateInterval != None:
               intervalStr = IntfCli.getLoadIntervalPrintableString(
                     ifCounter.updateInterval )
            if intervalStr != None:
               rateType = ''
               if self.police and self.police.pir:
                  rateType = ' low'
                  yellowRateStr = self.rateInterfaceCountersHelper( 'above',
                        rateType, intervalStr, ifCounter.exceededBitsRate,
                        ' and below the high rate limit' )
                  if self.police.redCounter:
                     exceededRateStr = self.rateInterfaceCountersHelper( 'above', '',
                           intervalStr, ifCounter.droppedBitsRate, '' )
               else:
                  exceededRateStr = self.rateInterfaceCountersHelper( 'above', '',
                        intervalStr, ifCounter.droppedBitsRate, '' )
               conformedRateStr = self.rateInterfaceCountersHelper( 'below',
                     rateType, intervalStr, ifCounter.conformedBitsRate, '' )
         # display accumulated counters
         if 'all' in ifName:
            if self.police.perSviRedCounterWithSharedPolicer:
               return output
            if self.police.pir:
               output += '        ' + conformedStrTrTcm
               output += '        ' + yellowStr
               if self.police.redCounter:
                  output += '        ' + exceededStrTrTcm
            else:
               output += '        ' + conformedStr
               if self.police.redCounter:
                  output += '        ' + exceededStr
            return output
         # We will reach here if we support unshared policers.
         # Print stats for every port
         output += '        Interface: %s\n' % ifName
         output2 = ''
         if self.police.pir:
            output += '          ' + conformedStrTrTcm
            output += '          ' + yellowStr
            output2 += '          ' + conformedRateStr
            output2 += '          ' + yellowRateStr
            if self.police.redCounter:
               output += '          ' + exceededStrTrTcm
               output2 += '          ' + exceededRateStr

         else:
            if self.police.greenCounter:
               output += '          ' + conformedStr
               output2 += '          ' + conformedRateStr
            if self.police.redCounter:
               output += '          ' + exceededStr
               output2 += '          ' + exceededRateStr
         output += output2
         return output

      def rateInterfaceCountersHelper( self, direction, rateType, intervalStr,
                                       rate, additionalText ):
         if rate is None:
            return ''
         ( rateStr, ratePrefix ) = self.getRate( rate )
         return '%s rate %s the%s rate limit%s %s %sbps\n' % ( intervalStr,
                                             direction, rateType, additionalText,
                                             rateStr, ratePrefix )

      def renderPolicerCounters( self, direction ):
         output = ""
         if direction not in self.police.counters:
            return output
         counter = self.police.counters[ direction ].interfaces
         for ifName, ifCounter in sorted( counter.items() ):
            output += self.renderIntfPolicerCounters( ifName, ifCounter )
         return output

      def renderClassMapStatistics( self, direction ):
         output = ""
         if direction not in self.classMapStatistics:
            return output
         classMapStats = self.classMapStatistics[ direction ]
         intervalStr = None
         if classMapStats.updateInterval != None:
            intervalStr = IntfCli.getLoadIntervalPrintableString(
                        classMapStats.updateInterval )
         if intervalStr != None:
            if self.police and self.police.pir:
               if classMapStats.conformedBitsRate != None:
                  ( outRateStr, outRatePrefix ) = self.getRate(
                        classMapStats.conformedBitsRate )
                  packetRateStr = '%s rate below the low rate ' \
                                  'limit %s %sbps\n' % ( intervalStr,
                                 outRateStr, outRatePrefix )
                  output += '        ' + packetRateStr
               if classMapStats.exceededBitsRate != None:
                  ( eirStr, eirPrefix ) = self.getRate(
                        classMapStats.exceededBitsRate )
                  packetRateStr = '%s rate above the low rate limit' \
                                  ' and below the high rate limit ' \
                                  '%s %sbps\n' % ( intervalStr,
                                                   eirStr,
                                                   eirPrefix )
                  output += '        ' + packetRateStr
               if classMapStats.droppedBitsRate != None and \
                     self.police.redCounter:
                  ( dropRateStr, dropRatePrefix ) = self.getRate(
                        classMapStats.droppedBitsRate )
                  packetRateStr = '%s rate above the high rate ' \
                                  'limit %s %sbps\n' % ( intervalStr,
                                        dropRateStr, dropRatePrefix )
                  output += '        ' + packetRateStr
            else:
               if classMapStats.conformedBitsRate != None:
                  ( outRateStr, outRatePrefix ) = self.getRate(
                        classMapStats.conformedBitsRate )
                  packetRateStr = '%s rate below the rate limit ' \
                                  '%s %sbps\n' % ( intervalStr,
                                 outRateStr, outRatePrefix )
                  output += '        ' + packetRateStr

               if classMapStats.droppedBitsRate != None \
                     and self.police.redCounter:
                  ( dropRateStr, dropRatePrefix ) = self.getRate(
                        classMapStats.droppedBitsRate )
                  packetRateStr = '%s rate above the rate limit ' \
                                  '%s %sbps\n' % ( intervalStr,
                                 dropRateStr, dropRatePrefix )
                  output += '        ' + packetRateStr
         return output

      def renderPolicerAction( self ):
         output = ""
         if self.mapType == coppMapType:
            if self.police.bc:
               output += '       police rate %d %s burst-size %d %s\n' % \
                  ( self.police.cir, self.police.cirUnit,
                    self.police.bc, self.police.bcUnit )
            else:
               output += '       police rate %d %s\n' % \
                  ( self.police.cir, self.police.cirUnit )
            return output

         if self.police.policerName:
            output += '       police shared %s' % ( self.police.policerName )
         else:
            output += '       police rate %d %s burst-size %d %s' % (
                  self.police.cir, self.police.cirUnit, self.police.bc,
                  self.police.bcUnit )
         if self.police.pir:
            if self.police.yellowActions.dropPrecedence:
               output += ' action set drop-precedence'
            if self.police.yellowActions.dscp is not None:
               if self.police.yellowActions.dscpConfiguredAsName:
                  name = AclCliLib.dscpNameFromValue( self.police
                                                      .yellowActions
                                                      .dscp )
                  output += ' action set dscp %s' % name
               else:
                  output += ' action set dscp %d' % self.police.yellowActions\
                                                         .dscp
            output += ' rate %d %s burst-size %d %s\n' % (
               self.police.pir, self.police.pirUnit, self.police.be,
               self.police.beUnit )
         else:
            output += '\n'
         return output

      def renderControlPlanePolicer( self, direction ):
         output = ""
         if self.police:
            output += self.renderPolicerAction()
         if direction is None:
            return output
         if self.police and self.police.counters:
            output += self.renderPolicerCounters( direction )
         elif self.matchedPackets != None:
            output += self.renderMatchedPackets( direction )
         return output

      def renderControlPlaneShaperAction( self ):
         output = ""
         if self.shape:
            output += '       shape : %s %s\n' % \
               ( self.shape.rate, self.shape.unit )
         else:
            output += '       shape : None\n'
         if self.bandwidth:
            output += '       bandwidth : %s %s\n' % \
               ( self.bandwidth.rate, self.bandwidth.unit )
         else:
            output += '       bandwidth : None\n'
         return output

      def renderControlPlaneShaperCounter( self, direction ):
         output = self.renderIntfPacketCounters( direction )
         return output

      def renderControlPlaneShaper( self, direction ):
         output = self.renderControlPlaneShaperAction()
         if direction is None:
            return output
         if self.intfPacketCounters != None:
            output += self.renderControlPlaneShaperCounter( direction )
         return output

      def renderControlPlaneAction( self, direction ):
         output = ""
         if self.coppActionPoliceSupported:
            output += self.renderControlPlanePolicer( direction )
         else:
            output += self.renderControlPlaneShaper( direction )
         return output

      def renderQosActionType( self ):
         output = ""
         if self.qosParams.cos != None:
            output += '       set cos %d\n' % self.qosParams.cos
         if self.qosParams.dropPrecedence != None:
            output += '       set drop-precedence %d\n' % \
                  self.qosParams.dropPrecedence
         if self.qosParams.dscp != None:
            if self.qosParams.dscpConfiguredAsName:
               name = AclCliLib.dscpNameFromValue( self.qosParams.dscp )
               output += '       set dscp %s\n' % name
            else:
               output += '       set dscp %d\n' % self.qosParams.dscp
         if self.qosParams.drop != None:
            if self.qosParams.dropActionRestricted:  # pylint: disable-msg=W0212
               output += '       no action\n'
            else:
               output += '       drop\n'
         if self.qosParams.trafficClass != None:
            output += '       set traffic-class %d\n' % \
                  self.qosParams.trafficClass
         if self.police:
            output += self.renderPolicerAction()
         if self.count:
            output += '       count\n'
         return output

      def renderQosAction( self, direction ):
         output = self.renderQosActionType()
         if direction is None:
            return output
         if self.police and self.police.counters:
            output += self.renderPolicerCounters( direction )
            if self.mapType == qosMapType and toggleQosRateCounterEnabled() and \
                  self.classMapStatistics != None:
               output += self.renderClassMapStatistics( direction )
         elif self.matchedPackets != None:
            output += self.renderMatchedPackets( direction )
         return output

      def render( self, direction=None ):  # pylint: disable=arguments-differ
         output = '  Class-map: %s%s (%s)\n' % \
                   ( "built-in " if self.mapType == 'mapPdp' and self.name in
                     QosLib.builtInClassMapNames else "", self.name,
                     QosLib.strFromEnum( self.matchCondition ) )
         output += self.renderMatches()
         if self.mapType == coppMapType:
            output += self.renderControlPlaneAction( direction )
         else:
            output += self.renderQosAction( direction )
         print( output )

   name = Str( help="Name of policy map" )
   mapType = Enum( values=( "mapQos", "mapControlPlane", "mapPdp" ),
                   help="Determines if it is a CoPP or Qos or PDP policy map",
                   default='mapControlPlane' )
   shared = Bool( help="True if the policy map is shared", optional=True )
   spApplied = Dict( valueType=bool,
                     help="True for the directions( 'input', 'output' )"
                          "having service policy" )
   hwStatus = Dict( valueType=HwStatus, help="Hardware status for "
                                              " direction ( 'input', 'output' ) " )
   classMaps = Dict( valueType=ClassMap,
                     help="List of class maps for a given policy map" )

   configuredIntfs = Dict( valueType=InterfaceList,
                           optional=True,
                           help="Dictionary of list of interfaces with the policy "
                           "configured for the directions( "
                           "'input', 'output' )" )
   activeIntfs = Dict( valueType=InterfaceList,
                       optional=True,
                       help="Dictionary of list of interfaces with the policy"
                       " active for the directions( "
                       "'input', 'output' )" )
   inactiveIntfs = Dict( valueType=InterfaceList,
                         optional=True,
                         help="Dictionary of list of interfaces with the"
                         " policy inactive for the directions( "
                         "'input', 'output' )" )
   summary = Bool( help="Print summary info", default=False )

   def _printIntfs( self, intfs, title, lineLen ):
      titleLen = len( title )
      tableLen = lineLen - ( titleLen + 1 )
      # More than one line needed
      # pylint: disable-next=superfluous-parens
      if ( reduce( lambda tot, item: tot + len( item ) + 1, intfs, 0 ) > tableLen ):
         listLengths = [ len( x ) for x in sorted( intfs, key=len, reverse=True ) ]
         width = 0
         columns = 0
         for length in listLengths:
            if width + length + 1 > tableLen - 1:
               break
            width += length + 1
            columns += 1
         t = TableFormatter( tableWidth=lineLen )
         f = Format( justify='left' )
         f.padLimitIs( True )
         f.noPadLeftIs( True )
         t.formatColumns( *[ f ] * ( columns + 1 ) )
         # pylint: disable-next=consider-using-enumerate
         for cellCount in range( len( intfs ) ):
            if cellCount == 0:
               t.startRow( rowFormat=f )
               t.newCell( title )
            elif cellCount % columns == 0:
               t.startRow( rowFormat=f )
               t.newCell( titleLen * ' ' )
            t.newCell( intfs[ cellCount ] )
         nameList = t.output( ).rstrip( )
      else:
         nameList = title
         for name in intfs:
            nameList += "%s " % name
      print( nameList )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         types = ( 'configured', 'active', 'inactive' )
         for t in types:
            key = '%sIntfs' % t
            if key in dictRepr:
               oldKey = '%sIngressIntfs' % t
               dictRepr[ oldKey ] = dictRepr[ key ].get( 'input', [] )
               del dictRepr[ key ]
      return dictRepr

   def isCountersPopulated( self ):
      if not self.classMaps:
         return None
      cMap = next( iter( self.classMaps.items() ) )[ 1 ]
      return ( cMap.police and cMap.police.counters ) or \
         cMap.matchedPackets or cMap.intfPacketCounters

   def renderCmapsDir( self, direction ):
      # Currently only counters are aware of the direction. So when we pass
      # direction, we will show counter based on the direction
      # The class actions and class matches are not aware of direction,
      # therefore we will display the content independent of direction.
      # Currently for class actions CLI will throw a guard error based
      # on which direction the service-policy is applied on. So policy
      # can be applied on both direction iff all the actions are supported
      # in both directions.
      # we might need redsign the model such that we have class-maps and its
      # associated action / matches set correctly by platform based on direction.
      # once this done then class model itself can be direction based.
      for cMapName in ( sorted( self.classMaps,
                     key=lambda name: self.classMaps[ name ].classPrio ) ):
         cMap = self.classMaps[ cMapName ]
         cMap.render( direction=direction )

   def renderSummaryIntfs( self, configuredIntfs, activeIntfs, inactiveIntfs ):
      if self.mapType == tacPolicyMapType.mapQos:
         print( '  Number of class maps: %d' % len( self.classMaps ) )
      intfs = Arnet.sortIntf( configuredIntfs )
      self._printIntfs( intfs, '  Configured on: ', terminalWidth() - 1 )
      intfs = Arnet.sortIntf( activeIntfs )
      self._printIntfs( intfs, '  Active on: ', terminalWidth() - 1 )
      intfs = Arnet.sortIntf( inactiveIntfs )
      self._printIntfs( intfs, '  Inactive on: ', terminalWidth() - 1 )
      print()

   def renderServicePolicyDir( self, direction ):
      if direction:
         output = 'Service-policy %s: %s%s\n' % (
            direction, self.name, " (shared)" if self.shared else "" )
      else:
         output = 'Service-policy %s%s' % (
            self.name, " (shared)" if self.shared else "" )
      return output

   def renderServicePolicy( self ):
      if True in self.spApplied.values():
         # If service policy is applied in either direction, print the policy
         # with direction and programming status.
         for direction in sorted( self.hwStatus ):
            output = self.renderServicePolicyDir( direction )
            hwStatus = self.hwStatus[ direction ]
            if hwStatus.unitsProgrammed != None:
               output += "  Number of units programmed: %d\n" % \
                  hwStatus.unitsProgrammed
            output += "  Hardware programming status: %s" % \
               QosLib.programmedPMapHwStatusFromEnum( hwStatus.status )
            print( output )
            if self.summary:
               self.renderSummaryIntfs(
                  self.configuredIntfs[ direction ].interfaces,
                  self.activeIntfs[ direction ].interfaces,
                  self.inactiveIntfs[ direction ].interfaces )
               continue
            counterPopulated = self.isCountersPopulated()
            if counterPopulated:
               print()
               self.renderCmapsDir( direction )
         if ( not self.summary ) and ( not counterPopulated ):
            print()
            self.renderCmapsDir( None )
         return

      # no True in self.spApplied.values
      output = self.renderServicePolicyDir( None )
      print( output )
      if self.summary:
         intfs = []
         self.renderSummaryIntfs( intfs, intfs, intfs )
         return
      print()
      # even if direction is not specified set direction for
      # control-plane policy
      direction = None if self.mapType != coppMapType else \
         tacDirection.input
      self.renderCmapsDir( direction )

   def render( self ):
      self.renderServicePolicy()

# Wrapper class around class map so as to return a ( empty ) class map from a
# show* function in QosCli without making all parameters of ClassMap optional
class ClassMapWrapper( Model ):
   __revision__ = 4
   classMap = Submodel( valueType=PolicyMapModel.ClassMap,
                        help="Wrapper around a class map", optional=True )

   def render( self ):
      if self.classMap:
         self.classMap.render()

class PolicyMapAllModel( Model ):
   __revision__ = 4
   policyMaps = Dict( keyType=str, valueType=PolicyMapModel,
                          help="List of all policy maps" )

   maxCmapsPerPolicy = \
         Int( help="Maximum number of class maps supported per policy map",
              optional=True, default=0 )
   _summary = Bool( help="Print summary info", default=False )
   mapType = Enum( values=( tacPolicyMapType.mapQos,
                            tacPolicyMapType.mapControlPlane,
                            tacPolicyMapType.mapCoppCli,
                            tacPolicyMapType.mapPdp ),
                   help="Determines if it is a CoPP or Qos or PDP policy map",
                   default='mapControlPlane' )

   def maxCmapIs( self, policyQosNumCMapSupported ):
      self.maxCmapsPerPolicy = policyQosNumCMapSupported

   def summaryIs( self, summary ):
      self._summary = bool( summary )

   def mapTypeIs( self, mapType ):
      self.mapType = mapType

   def append( self, pMapName, policyMapModel ):
      self.policyMaps[ pMapName ] = policyMapModel

   def render( self ):
      if self._summary and self.mapType == tacPolicyMapType.mapQos:
         print( 'Maximum number of class maps supported per policy map: %d' %
                self.maxCmapsPerPolicy )
      for pMapName in sorted( self.policyMaps.keys() ):
         pMap = self.policyMaps[ pMapName ]
         pMap.render()


class InterfaceIdModel( Model ):

   intfIds = List( valueType=Interface, help='List of associated interfaces' )

   def append( self, intfId ):
      self.intfIds.append( intfId )

   def extend( self, intfId ):
      self.intfIds.extend( intfId )

   def render( self ):
      for intf in self.intfIds:
         print( intf )

class SchedulingModeModel( Model ):
   schedulingMode = Enum( values=( 'roundRobin',
                                   'strictPriority',
                                   'unspecified' ),
                          help="Scheduling mode",
                          default='unspecified' )

   def formatSchedulingMode( self ):
      model = self
      if model and model.schedulingMode == 'roundRobin':
         return 'RR'
      else:
         return 'SP'

class GuaranteedBwModel( Model ):
   bandwidth = Int( help="Guaranteed bandwidth" )
   unit = Enum( values=( 'kbps', 'pps', 'percent', 'unspecified' ),
                help="Guaranteed bandwidth unit",
                default='unspecified' )
   # used by interface scheduling group
   percent = Int( help="Guaranteed bandwidth as percentage of reference",
                  optional=True )
   percentOf = Enum( values=( 'lineRate', 'schedulingGroupShapeRate',
                              'parentInterfaceShapeRate', 'subinterfaceShapeRate',
                              'unknown' ),
                     help="Reference for guaranteed bandwidth percent",
                     optional=True )

   # formats GuaranteedBwModel into Qos::GuaranteedBw
   def formatGuaranteedBw( self ):
      model = self
      bw = model.bandwidth
      unit = model.unit
      if bw < QosLib.tacGuaranteedBwVal.min or bw > QosLib.tacGuaranteedBwVal.max:
         guaranteedBw = QosLib.invalidGuaranteedBw
      else:
         guaranteedBw = QosLib.Tac.Value( 'Qos::GuaranteedBw' )
         guaranteedBw.bw = bw
         guaranteedBw.unit = QosLib.guaranteedBwUnitToEnum( unit )

      return guaranteedBw

def formatBurstSize( burstSizeModel ):
   # We expect the input to always be tacBurstUnit.burstUnitBytes
   # Returns a string of the format:
   #    '%s %s' % ( value represented in 'unit',
   #                'unit' -> least precise unit that an accurately show the value. )
   assert burstSizeModel.unit == tacBurstUnit.burstUnitBytes
   unitStrList = [ burstUnitSymbolFromEnum( tacBurstUnit.burstUnitBytes ),
                   burstUnitSymbolFromEnum( tacBurstUnit.burstUnitKBytes ),
                   burstUnitSymbolFromEnum( tacBurstUnit.burstUnitMBytes ) ]
   unitStrIdx = 0
   burstSizeVal = burstSizeModel.value
   while ( burstSizeVal % 1000 == 0 ) and unitStrIdx < ( len( unitStrList ) - 1 ):
      unitStrIdx += 1
      burstSizeVal //= 1000
   return '%s %s' % ( burstSizeVal, unitStrList[ unitStrIdx ] )

class BurstSizeModel( Model ):
   value = Int( help="Burst-size value" )
   unit = Enum( values=( tacBurstUnit.burstUnitBytes,
                         tacBurstUnit.burstUnitKBytes,
                         tacBurstUnit.burstUnitMBytes ),
                help="Burst-size unit",
                default=tacBurstUnit.burstUnitBytes )

def formatSchedulerCompensation( schedulerCompensationModel ):
   assert schedulerCompensationModel.unit == tacBurstUnit.burstUnitBytes
   schedulerCompensationStr = '%s %s' % ( schedulerCompensationModel.value,
                      burstUnitSymbolFromEnum( schedulerCompensationModel.unit ) )
   if schedulerCompensationModel.value > 0:
      schedulerCompensationStr = '+' + schedulerCompensationStr
   return schedulerCompensationStr

class SchedulerCompensationModel( Model ):
   value = Int( help="Scheduler Compensation value" )
   unit = Enum( values=( tacBurstUnit.burstUnitBytes, ),
                help="Scheduler Compensation unit",
                default=tacBurstUnit.burstUnitBytes )

class PolicerPktSizeAdjModel( Model ):
   profile = Str( help="Packet size adjustment profile" )
   value = Int( help="Packet size adjustment configuration"
                     " associated with the profile" )

def formatPolicerPktSizeAdj( policerPktSizeAdjModel ):
   valueStr = str( policerPktSizeAdjModel.value )
   if policerPktSizeAdjModel.value > 0:
      valueStr = '+' + valueStr
   if policerPktSizeAdjModel.value == tacPolicerPktSizeAdj.invalid:
      policerPktSizeAdjStr = ' %s ( - )' % ( policerPktSizeAdjModel.profile )
   else:
      policerPktSizeAdjStr = '%s ( %sB )' % ( policerPktSizeAdjModel.profile,
                                              valueStr )
   return policerPktSizeAdjStr

class SchedulingElementModel( Model ):
   configuredGuaranteedBw = Submodel( valueType=GuaranteedBwModel,
                                      help="Configured Guaranteed bandwidth",
                                      optional=True )
   operationalGuaranteedBw = Submodel( valueType=GuaranteedBwModel,
                                       help="Operational Guaranteed bandwidth",
                                       optional=True )

   configuredShapeRate = Submodel( valueType=ShapeRateModel,
                                   help="Configured Shape rate",
                                   optional=True )
   operationalShapeRate = Submodel( valueType=ShapeRateModel,
                                    help="Operational Shape rate",
                                    optional=True )

   def renderSgShapeRate( self, tableCol=True ):
      shapeRateUnits = '-'
      cfgShapeRateUnits = '-'
      operShapeRateUnits = '-'

      if self.configuredShapeRate and \
         self.configuredShapeRate.rate != QosLib.invalidShapeRate.rate:
         tmpCfgShapeRate = formatShapeRate( self.configuredShapeRate )
         cfgShapeRate, cfgShapeRateUnits = \
                        QosLib.cliConvertShapeRateToSIUnit( tmpCfgShapeRate )
         shapeRateUnits = cfgShapeRateUnits
      else:
         cfgShapeRate = '-'

      if self.operationalShapeRate and \
         self.operationalShapeRate.rate != QosLib.invalidShapeRate.rate:
         tmpOperShapeRate = formatShapeRate( self.operationalShapeRate )
         operShapeRate, operShapeRateUnits = \
            QosLib.cliConvertShapeRateToSIUnit( tmpOperShapeRate )
         shapeRateUnits = operShapeRateUnits
      else:
         operShapeRate = '-'

      if operShapeRate != '-' and cfgShapeRate != '-' and \
         operShapeRateUnits != '-' and cfgShapeRateUnits != '-' and \
         operShapeRateUnits != cfgShapeRateUnits and \
         not self.configuredShapeRate.percent:
         cfgShapeRate, operShapeRate, shapeRateUnits = \
               QosLib.cliConvertBothShapeRatesToSIUnit( tmpCfgShapeRate,
                                                        tmpOperShapeRate )

      if tableCol:
         shapeRate = f'{operShapeRate:>5} / {cfgShapeRate:>5}'
      else:
         shapeRate = operShapeRate + ' / ' + cfgShapeRate

      shapeRateUnits = ' (' + shapeRateUnits + ')'

      if self.configuredShapeRate and self.configuredShapeRate.percent and \
         self.configuredShapeRate.percent != tacPercent.invalid:
         if tableCol:
            template = '(%s%%)'
         else:
            template = ', %s%% of ' + percentReferenceStr(
               self.configuredShapeRate.percentOf )
         shapeRatePercent = template % self.configuredShapeRate.percent
      else:
         shapeRatePercent = ''

      return shapeRate + shapeRateUnits + shapeRatePercent

   def renderSgGuaranteedBw( self, tableCol=True ):
      guaranteedBwUnits = '-'

      if self.configuredGuaranteedBw and \
         self.configuredGuaranteedBw.bandwidth != QosLib.invalidGuaranteedBw.bw:
         tmpCfgGuaranteedBw = self.configuredGuaranteedBw.formatGuaranteedBw()
         cfgGuaranteedBw, guaranteedBwUnits = \
            QosLib.cliConvertGuaranteedBwToSIUnit( tmpCfgGuaranteedBw )
      else:
         cfgGuaranteedBw = '-'

      if self.operationalGuaranteedBw and \
         self.operationalGuaranteedBw.bandwidth != QosLib.invalidGuaranteedBw.bw:
         tmpOperGuaranteedBw = self.operationalGuaranteedBw.formatGuaranteedBw()
         operGuaranteedBw, guaranteedBwUnits = \
            QosLib.cliConvertGuaranteedBwToSIUnit( tmpOperGuaranteedBw )
      else:
         operGuaranteedBw = '-'

      if tableCol:
         guaranteedBw = f'{operGuaranteedBw:>5} / {cfgGuaranteedBw:>5}'
      else:
         guaranteedBw = operGuaranteedBw + ' / ' + cfgGuaranteedBw
      guaranteedBwUnits = ' (' + guaranteedBwUnits + ')'

      if self.configuredGuaranteedBw and self.configuredGuaranteedBw.percent and \
         self.configuredGuaranteedBw.percent != tacPercent.invalid:
         if tableCol:
            template = '(%s%%)'
         else:
            template = ', %s%% of ' + percentReferenceStr(
               self.configuredGuaranteedBw.percentOf )
         guaranteedBwPercent = template % self.configuredGuaranteedBw.percent
      else:
         guaranteedBwPercent = ''

      return guaranteedBw + guaranteedBwUnits + guaranteedBwPercent

class SchedulingGroupWithMembersModel( SchedulingElementModel ):
   interfaces = Dict( keyType=Interface, valueType=SchedulingElementModel,
                      help="Member subinterfaces of the scheduling group",
                      optional=True )

   def renderMemberTable( self ):
      table = createTable( ( "Member", "Bandwidth (units)",
                             "Shape Rate (units)" ) )
      for intf in sorted( self.interfaces ):
         params = self.interfaces[ intf ]
         guaranteedBw = params.renderSgGuaranteedBw()
         shapeRate = params.renderSgShapeRate()
         table.newRow( intf, guaranteedBw, shapeRate )

      f1 = Format( justify="left" )
      table.formatColumns( f1, f1, f1 )
      print( '\n' + table.output() )

class IntfSchedulingGroupModel( Model ):
   schedulingGroups = Dict( keyType=str, valueType=SchedulingElementModel,
                            help="Scheduling Groups configured for the interface,"
                            " indexed by their name" )

   def renderIntfSchedulingGroupTable( self ):
      table = createTable( ( "Scheduling Group", "Bandwidth (units)",
                             "Shape Rate (units)" ) )
      for groupName in sorted( self.schedulingGroups ):
         group = self.schedulingGroups[ groupName ]
         guaranteedBw = group.renderSgGuaranteedBw()
         shapeRate = group.renderSgShapeRate()
         table.newRow( groupName, guaranteedBw, shapeRate )

      f1 = Format( justify="left" )
      table.formatColumns( f1, f1, f1 )
      print( '\n' + table.output() )
      print( 'Note: Values are displayed as Operational/Configured' )

class IntfSchedulingGroupWithMembersModel( Model ):
   schedulingGroups = Dict( keyType=str, valueType=SchedulingGroupWithMembersModel,
                            help="SchedulingGroups configured for the interface,"
                            " indexed by their name" )

   def renderIntfSchedulingGroupWithMembers( self ):
      for groupName in sorted( self.schedulingGroups ):
         group = self.schedulingGroups[ groupName ]
         print( 'Scheduling Group Name:', groupName )
         guaranteedBw = group.renderSgGuaranteedBw( False )
         shapeRate = group.renderSgShapeRate( False )
         print( 'Bandwidth:', guaranteedBw )
         print( 'Shape Rate:', shapeRate )
         group.renderMemberTable()

class InterfaceQosModel( Model ):
   interface = Str( help="Qos on Interface" )
   sliceName = Str( help="Slice name corresponding to the Fabric interface",
                    optional=True )
   intfIsLag = Bool( help="Is a LAG Interface", default=False )
   intfQosInfoAvailable = Bool( help="Interface Qos info availability" )
   intfQosDetailedInfoAvailable = Bool( help="Detailed Qos info of the interface"
                                             " is available", default=False )

   trustMode = Enum( values=( 'dscpTrusted', 'cosTrusted', 'unTrusted' ),
                     help="Qos Trustmode of interface", default='unTrusted' )

   defaultCosSupported = Bool( help="Is default COS supported", default=False )
   defaultCos = Int( help="Default COS value, if supported", optional=True )

   defaultDscpSupported = Bool( help="Is default DSCP supported", default=False )
   defaultDscp = Int( help="Default DSCP value, if supported", optional=True )

   dscpToCosRewriteActive = Bool( help="DSCP To COS Rewrite active status,"
                                       " if hardware supports reporting",
                                       optional=True )
   dscpRewriteActive = Bool( help="DSCP Rewrite Active,"
                                  " if hardware supports reporting",
                             optional=True )
  
   expRewriteActive = Bool( help="EXP Rewrite Active,"
                                  " if hardware supports reporting", 
                            optional=True )

   dscpPreserveIpMplsEncapMode = Bool( help="DSCP Preserve for MPLS IP encap",
                                       default=False )

   cosToTcProfile = Str( help="COS-to-TC profile",
                         default=tacCosToTcProfileName.defaultProfileName )

   expToTcProfile = Str( help="EXP-to-TC profile",
                         default=tacExpToTcProfileName.defaultProfileName )

   dscpToTcMap = Str( help="DSCP-to-TC map",
                      default=tacDscpToTcMapName.defaultMapName )

   tcToCosNamedMap = Str( help="TC-to-COS map",
                          default=tacTcToCosMapName.defaultMapName )

   cpuTcToCosNamedMap = Str( help="CPU TC-to-COS map",
                             default=tacTcToCosMapName.defaultMapName )

   dscpRewriteMap = Str( help="TC-to-DSCP map",
                         default=tacDscpRewriteMapName.defaultMapName )

   tcDpToExpMap = Str( help="TC-to-EXP map",
                       default=tacTcDpToExpMapName.defaultMapName )

   configuredPortShapeRate = Submodel( valueType=ShapeRateModel,
                                        help="Configured port shape rate",
                                        optional=True )
   operationalPortShapeRate = Submodel( valueType=ShapeRateModel,
                                        help="Operational port shape rate",
                                        optional=True )
   burstSizeConfigSupported = Bool( help="Burst-size config is supported",
                                    default=False )
   configuredBurstSize = Submodel( valueType=BurstSizeModel,
                                   help="Configured burst-size",
                                   optional=True )
   operationalBurstSize = Submodel( valueType=BurstSizeModel,
                                    help="Operational burst-size",
                                    optional=True )
   schedulerCompensationConfigSupported = Bool(
      help="Scheduler compensation config is supported", default=False )
   configuredSchedulerCompensation = Submodel(
      valueType=SchedulerCompensationModel, help="Configured scheduler compensation",
      optional=True )
   operationalSchedulerCompensation = Submodel(
      valueType=SchedulerCompensationModel,
      help="Operational scheduler compensation", optional=True )
   policerPktSizeAdjConfigSupported = Bool(
      help="Policer packet size config is supported", default=False )
   configuredPolicerPktSizeAdj = Submodel(
      valueType=PolicerPktSizeAdjModel,
      help="Configured policer packet size adjustment", optional=True )
   operationalPolicerPktSizeAdj = Submodel(
      valueType=PolicerPktSizeAdjModel,
      help="Operational policer packet size adjustment", optional=True )
   interfaceGuaranteedBwSupported = Bool(
      help="Guaranteed bandwidth config is supported", default=False )
   configuredPortGuaranteedBw = Submodel(
      valueType=GuaranteedBwModel, help="Configured port guaranteed bandwidth",
      optional=True )
   operationalPortGuaranteedBw = Submodel(
      valueType=GuaranteedBwModel, help="Operational port guaranteed bandwidth",
      optional=True )

   class TxQueueQosModel( Model ):

      # Tx Queue
      wrrSupported = Bool( help="WRR Scheduling supported", default=False )
      guaranteedBwSupported = Bool( help="Guaranteed bandwidth supported",
                                    default=False )
      ecnSupported = Bool( help="ECN supported", default=False )
      delayEcnSupported = Bool( help="Delay based ECN supported", default=False )
      wredSupported = Bool( help="WRED supported", default=False )
      nonEctSupported = Bool( help="NON-ECT supported", default=False )
      numTxQueue = Int( help="Number of Transmit Queues", default=0 )
      multipleSchedGroupsSupported = \
         Bool( help="Multiple scheduling groups supported", default=False )
      burstSizeConfigSupported = Bool( help="Burst-size config is supported",
                                       default=False )
      numTxQueuePrint = \
            Int( help="Number of Transmit Queues to print", optional=True )

      class TxQueue( Model ):
         txQueue = Str( help="Transmit Queue" )
         txQueuePriority = Int( help="Transmit Queue Priority - higher \
               txQueuePriority value has precedence" )

         configuredWrrBw = Int( help="Configured WRR Bandwidth in percent",
                                optional=True )
         operationalWrrBw = Int( help="Operational WRR Bandwidth in percent",
                                 optional=True )

         configuredWrrBwWeight = Int( help="Configured WRR Bandwidth weight",
                                optional=True )
         operationalWrrBwWeight = Int( help="Operational WRR Bandwidth weight",
                                 optional=True )

         configuredGuaranteedBw = Submodel( valueType=GuaranteedBwModel,
                                            help="Configured Guaranteed bandwidth",
                                            optional=True )
         operationalGuaranteedBw = Submodel( valueType=GuaranteedBwModel,
                                             help="Operational Guaranteed bandwidth",
                                             optional=True )

         configuredShapeRate = Submodel( valueType=ShapeRateModel,
                                         help="Configured Shape rate",
                                         optional=True )
         operationalShapeRate = Submodel( valueType=ShapeRateModel,
                                          help="Operational Shape rate",
                                          optional=True )

         configuredSchedMode = Submodel( valueType=SchedulingModeModel,
                                         help="Configured Scheduling mode",
                                         optional=True )
         operationalSchedMode = Submodel( valueType=SchedulingModeModel,
                                          help="Operational Scheduling mode",
                                          optional=True )

         ecnStatus = Enum( values=( 'enabled', 'disabled', 'not applicable',
                                    'delayEnabled', 'lengthAndDelayEnabled' ),
                           help="ECN Status on Transmit Queue" )
         dynamicEcnStatus = Enum(
            values=( 'enabled', 'disabled', 'not applicable', ),
            help="Dynamic ECN Status on Transmit Queue" )
         wredStatus = Enum( values=( 'enabled', 'disabled', 'not applicable' ),
                            help="WRED Status on Transmit Queue" )
         nonEctStatus = Enum( values=( 'enabled', 'disabled', 'not applicable' ),
                              help="NON-ECT Status on Transmit Queue" )
         schedGroupId = Int( help="Scheduling Group Id", default=0 )
         configuredBurstSize = Submodel( valueType=BurstSizeModel,
                                         help="Configured burst-size",
                                         optional=True )
         operationalBurstSize = Submodel( valueType=BurstSizeModel,
                                          help="Operational burst-size",
                                          optional=True )

      # List of txQueues on the interface
      txQueueList = List( valueType=TxQueue, help="List of Transmit Queues" )

      def renderTxQueueQosModel( self, lagIntf, isFabricIntf, isSubIntf,
                                 intfQosDetailedInfoAvailable, displayLegend=True,
                                 mixedNumTxqPortsInSystem=False ):
         # WRR Bandwidth
         def renderWrrBw( self, cfgWrrBw, operWrrBw, cfgWrrBwWeight,
               operWrrBwWeight ):
            if not self.wrrSupported:
               return None
            if wrrBwIsValid( cfgWrrBwWeight ):
               cfgWrrBwStr = str( cfgWrrBwWeight )
            elif wrrBwIsValid( cfgWrrBw ):
               cfgWrrBwStr = str( cfgWrrBw ) + '%'
            else:
               cfgWrrBwStr = '-'

            if wrrBwIsValid( operWrrBwWeight ):
               operWrrBwStr = str( operWrrBwWeight )
            elif wrrBwIsValid( operWrrBw ):
               operWrrBwStr = str( operWrrBw ) + '%'
            else:
               operWrrBwStr = '-'

            if not lagIntf:
               wrrBandwidth = operWrrBwStr.rjust( 5 ) + \
                              ' / ' + \
                              cfgWrrBwStr.ljust( 5 )
            else:
               wrrBandwidth = cfgWrrBwStr
               if not wrrBwIsValid( cfgWrrBw ):
                  wrrBandwidth = operWrrBwStr
            return wrrBandwidth

         # Guaranteed bandwidth
         def renderGuaranteedBw( self, cfgGuaranteedBwModel, operGuaranteedBwModel,
                                 lagIntf, intfQosDetailedInfoAvailable ):
            if not self.guaranteedBwSupported:
               return None

            tmpCfgGuaranteedBw = QosLib.invalidGuaranteedBw
            tmpOperGuaranteedBw = QosLib.invalidGuaranteedBw

            if cfgGuaranteedBwModel:
               tmpCfgGuaranteedBw = cfgGuaranteedBwModel.formatGuaranteedBw()

            if operGuaranteedBwModel:
               tmpOperGuaranteedBw = operGuaranteedBwModel.formatGuaranteedBw()

            if lagIntf:
               guaranteedBw, guaranteedBwUnits = \
                  QosLib.cliConvertGuaranteedBwToSIUnit( tmpCfgGuaranteedBw )
            elif not intfQosDetailedInfoAvailable:
               guaranteedBw, guaranteedBwUnits = \
                  QosLib.cliConvertGuaranteedBwToSIUnit( tmpOperGuaranteedBw )
            else:
               cfgGuaranteedBw, operGuaranteedBw, guaranteedBwUnits = \
                  QosLib.cliConvertBothGuaranteedBwsToSIUnit( tmpCfgGuaranteedBw,
                                                              tmpOperGuaranteedBw )
               guaranteedBw = operGuaranteedBw.rjust( 5 ) + \
                              ' / ' + \
                              cfgGuaranteedBw.ljust( 5 )

            if guaranteedBwUnits == '':
               guaranteedBw += ' ( - )'  # no units
            elif guaranteedBwUnits == '%':
               guaranteedBw += ' ( % )'
            else:
               guaranteedBw += ' (' + guaranteedBwUnits + ')'

            return guaranteedBw

         # Shape Rate
         def renderShapeRate( self, cfgShapeRateModel, operShapeRateModel, lagIntf,
                              intfQosDetailedInfoAvailable ):

            tmpCfgShapeRate = QosLib.invalidShapeRate
            tmpOperShapeRate = QosLib.invalidShapeRate

            if cfgShapeRateModel:
               tmpCfgShapeRate = formatShapeRate( cfgShapeRateModel )
            if operShapeRateModel:
               tmpOperShapeRate = formatShapeRate( operShapeRateModel )

            if lagIntf:
               shapeRate, shapeRateUnits = \
                  QosLib.cliConvertShapeRateToSIUnit( tmpCfgShapeRate )
            elif not intfQosDetailedInfoAvailable:
               shapeRate, shapeRateUnits = \
                  QosLib.cliConvertShapeRateToSIUnit( tmpOperShapeRate )
            else:
               cfgShapeRate, operShapeRate, shapeRateUnits = \
                  QosLib.cliConvertBothShapeRatesToSIUnit( tmpCfgShapeRate,
                                                           tmpOperShapeRate )
               if tmpCfgShapeRate.percent != tacPercent.invalid:
                  cfgShapeRate = str( tmpCfgShapeRate.percent )
                  shapeRateUnits = '%'

               shapeRate = operShapeRate.rjust( 5 ) + \
                           ' / ' + \
                           cfgShapeRate.ljust( 5 )

            if shapeRateUnits == '':
               shapeRateUnits = ' - '  # no units
            shapeRateUnits = '(' + shapeRateUnits + ')'

            if tmpCfgShapeRate.percent != tacPercent.invalid:
               shapeRatePercent = '(' + str( tmpCfgShapeRate.percent ) + '%)'
            else:
               shapeRatePercent = ''

            return shapeRate, shapeRatePercent, shapeRateUnits

         # Burst-size
         def renderBurstSize( self, cfgBurstSize, operBurstSize, lagIntf,
                              intfQosDetailedInfoAvailable ):
            if cfgBurstSize and cfgBurstSize.value != tacBurstVal.invalid:
               cfgBurstSizeStr = formatBurstSize( cfgBurstSize )
            else:
               cfgBurstSizeStr = ' - '
            if operBurstSize and operBurstSize.value != tacBurstVal.invalid:
               operBurstSizeStr = formatBurstSize( operBurstSize )
               if cfgBurstSizeStr == ' - ':
                  cfgBurstSizeStr = operBurstSizeStr
            else:
               operBurstSizeStr = ' - '
            if lagIntf:
               burstSize = cfgBurstSizeStr.center( 26 )
            else:
               if intfQosDetailedInfoAvailable:
                  burstSize = ( operBurstSizeStr.rjust( 12 ) + ' / ' +
                                cfgBurstSizeStr.ljust( 11 ) )
               else:
                  burstSize = operBurstSizeStr.center( 26 )
            return burstSize

         # Scheduling Mode
         def renderSchedMode( self, cfgSchedMode, operSchedMode, lagIntf ):
            tmpCfgSchedMode = 'SP'
            tmpOperSchedMode = 'SP'

            if cfgSchedMode:
               tmpCfgSchedMode = cfgSchedMode.formatSchedulingMode()
            if operSchedMode:
               tmpOperSchedMode = operSchedMode.formatSchedulingMode()

            if lagIntf:
               schedMode = tmpCfgSchedMode
               if tmpOperSchedMode != tmpCfgSchedMode and tmpOperSchedMode != 'SP':
                  schedMode = tmpOperSchedMode
            else:
               schedMode = tmpOperSchedMode.rjust( 4 ) + \
                           ' / ' + \
                           tmpCfgSchedMode.ljust( 4 )

            return schedMode

         # ECN or WRED Status
         def renderEcnOrWredStatus( self, ecnStatus, wredStatus, dynamicEcnStatus ):
            ecnWredStatus = ''
            if ( ecnStatus == 'disabled' and wredStatus == 'disabled' and
                 dynamicEcnStatus != 'enabled' ):
               return 'D'
            if ( ecnStatus == 'enabled' or ecnStatus == 'lengthAndDelayEnabled' or
                 dynamicEcnStatus == 'enabled' ):
               assert ( toggleWredEcnNotMutuallyExclusiveEnabled() or
                        wredStatus != 'enabled' )
               ecnWredStatus += 'L'
            if ecnStatus == 'delayEnabled' or ecnStatus == 'lengthAndDelayEnabled':
               assert ( toggleWredEcnNotMutuallyExclusiveEnabled() or
                        wredStatus != 'enabled' )
               ecnWredStatus += 'T'
            if wredStatus == 'enabled':
               assert ( toggleWredEcnNotMutuallyExclusiveEnabled() or
                        ( ecnStatus != 'enabled' and
                          ecnStatus != 'lengthAndDelayEnabled' ) )
               ecnWredStatus += 'W'
            if not ecnWredStatus:
               ecnWredStatus = 'N/A'
            return ecnWredStatus

         if self.numTxQueue == 0:
            return
         print()

         txQShowHeader1 = '  Tx   '
         txQShowHeader2 = ' Queue '

         if self.wrrSupported:
            txQShowHeader1 += '  Bandwidth  '
            txQShowHeader2 += '             '

         if self.guaranteedBwSupported:
            txQShowHeader1 += '    Bandwidth           '
            txQShowHeader2 += '    Guaranteed (units)  '

         if not isFabricIntf:
            txQShowHeader1 += 'Shape Rate'.center( 25 )
            txQShowHeader2 += '(units)'.center( 25 )
            if self.burstSizeConfigSupported:
               txQShowHeader1 += 'Burst-Size'.center( 26 )
               txQShowHeader2 += '(units)'.center( 26 )
            txQShowHeader1 += 'Priority'.center( 11 )
         else:
            txQShowHeader1 += ' Priority'
            txQShowHeader2 += '         '

         if self.multipleSchedGroupsSupported:
            txQShowHeader1 += '    Priority'
            txQShowHeader2 += '     Group  '

         ecnOrWredTitle = ''
         if self.ecnSupported or self.wredSupported:
            if self.ecnSupported:
               ecnOrWredTitle += 'ECN'
            if self.wredSupported:
               ecnOrWredTitle += ( '/' if ecnOrWredTitle else '' ) + 'WRED'
            txQShowHeader1 += '  ' + ecnOrWredTitle + ' '

         txQShowMinus = ' ' + '-' * ( len( txQShowHeader1 ) - 1 )

         print( txQShowHeader1 )
         print( txQShowHeader2 )
         print( txQShowMinus )

         txQueueList = self.txQueueList
         if self.numTxQueuePrint:
            txQueueList = txQueueList[ 0 - ( self.numTxQueuePrint ) : ]
         for txQueue in txQueueList:

            wrrBandwidth = renderWrrBw( self, txQueue.configuredWrrBw,
                                        txQueue.operationalWrrBw,
                                        txQueue.configuredWrrBwWeight,
                                        txQueue.operationalWrrBwWeight )
            guaranteedBw = renderGuaranteedBw( self, txQueue.configuredGuaranteedBw,
                                               txQueue.operationalGuaranteedBw,
                                               lagIntf,
                                               intfQosDetailedInfoAvailable )
            if not isFabricIntf:
               shapeRate, shapeRatePercent, shapeRateUnits = \
                  renderShapeRate( self,
                                   txQueue.configuredShapeRate,
                                   txQueue.operationalShapeRate,
                                   lagIntf,
                                   intfQosDetailedInfoAvailable )
               if self.burstSizeConfigSupported:
                  burstSize = renderBurstSize( self, txQueue.configuredBurstSize,
                                               txQueue.operationalBurstSize,
                                               lagIntf,
                                               intfQosDetailedInfoAvailable )

            schedMode = renderSchedMode( self, txQueue.configuredSchedMode,
                                         txQueue.operationalSchedMode,
                                         lagIntf )
            schedGroupId = str( txQueue.schedGroupId )

            ecnOrWredStatus = renderEcnOrWredStatus(
               self, txQueue.ecnStatus, txQueue.wredStatus,
               txQueue.dynamicEcnStatus )

            # Prepare string to be displayed
            txQShowValues = txQueue.txQueue.center( 7 )

            if self.wrrSupported:
               txQShowValues += wrrBandwidth.center( 11 )

            if self.guaranteedBwSupported:
               txQShowValues += guaranteedBw.center( 23 )

            if not isFabricIntf:
               txQShowValues += shapeRate.center( 13 ) + \
                                shapeRatePercent.center( 6 ) + \
                                shapeRateUnits.center( 6 )
               if self.burstSizeConfigSupported:
                  txQShowValues += burstSize.center( 26 )
               txQShowValues += schedMode.center( 11 )
            else:
               txQShowValues += schedMode.center( 11 )
            if self.multipleSchedGroupsSupported:
               txQShowValues += schedGroupId.center( 11 )
            if self.ecnSupported or self.wredSupported:
               txQShowValues += ecnOrWredStatus.center(
                                    len( ecnOrWredTitle ) + 3 )

            print( txQShowValues )

         if not lagIntf:
            print()
            print( 'Note: Values are displayed as Operational/Configured' )

         if displayLegend:
            print()
            print( 'Legend:' )
            if mixedNumTxqPortsInSystem:
               print()
               print( 'Tc - tx-queue map for interface with 2 tx-queues:' )
               printFormattedArray( "tc:      ", "tx-queue:", 0, 7,
                                    [ 0, 0, 0, 0, 1, 1, 1, 1 ] )
            print( 'RR -> Round Robin' )
            print( 'SP -> Strict Priority' )
            print( ' - -> Not Applicable / Not Configured' )
            print( ' % -> Percentage of reference' )
            if self.ecnSupported or self.wredSupported:
               legendDesc = ''
               legendStr = ''
               if self.ecnSupported:
                  legendDesc = 'ECN'
                  legendStr = 'L -> Queue Length ECN Enabled     '
                  if self.delayEcnSupported:
                     legendStr += 'T -> Queue Delay ECN Enabled     '
               if self.wredSupported:
                  legendDesc += ( '/' if legendDesc else '' ) + 'WRED'
                  legendStr += 'W -> WRED Enabled     '
               legendStr += 'D -> Disabled'
               print( legendDesc + ':', legendStr )

   txQueueQosModel = Submodel( valueType=TxQueueQosModel,
                               help="Transmit Queue Model", optional=True )

   schedulingGroupModel = Submodel( valueType=IntfSchedulingGroupModel,
                                    help="Scheduling Group Model", optional=True )

   def renderInterfaceQosModel( self, displayLegend=True,
                                mixedNumTxqPortsInSystem=False ):
      if self.intfQosInfoAvailable == False:
         print( 'No QOS info available for %s' % self.interface )
         return

      intfStr = str( self.interface )
      intfStr = intfStr.strip( '\'' )

      isFabricIntf = ( intfStr == fabricTokenName )
      isSubIntf = not isFabricIntf and SubIntfId.isSubIntfId( self.interface )

      if self.sliceName:
         print( '%s:' % self.sliceName.split( '-', 1 )[ -1 ] )
      else:
         print( '%s:' % intfStr )
      if not isFabricIntf:
         print( '   Trust Mode: %s' % QosLib.convertTrustModeModelToStr(
            self.trustMode ) )

      if self.defaultCosSupported and not isFabricIntf:
         print( '   Default COS: %s' % self.defaultCos )

      if self.defaultDscpSupported and not isFabricIntf:
         print( '   Default DSCP: %s' % self.defaultDscp )

      if self.dscpToCosRewriteActive == False:
         print( '   DSCP to COS rewrite: Not active' )
      elif self.dscpRewriteActive == False:
         print( '   DSCP rewrite: Not active' )

      if self.expRewriteActive == False:
         print( '   EXP rewrite: Not active' )

      if self.dscpPreserveIpMplsEncapMode:
         print( '   DSCP rewrite for MPLS encap packets: Disabled' )

      if self.cosToTcProfile != tacCosToTcProfileName.defaultProfileName:
         print( '   COS to TC profile: %s' % self.cosToTcProfile )

      if self.expToTcProfile != tacExpToTcProfileName.defaultProfileName:
         print( '   EXP to TC profile: %s' % self.expToTcProfile )

      if self.dscpToTcMap != tacDscpToTcMapName.defaultMapName:
         print( '   DSCP to TC profile: %s' % self.dscpToTcMap )

      if self.dscpRewriteMap != tacDscpRewriteMapName.defaultMapName:
         print( '   TC to DSCP Profile: %s' % self.dscpRewriteMap )

      if self.tcDpToExpMap != tacTcDpToExpMapName.defaultMapName:
         print( '   TC to EXP Profile: %s' % self.tcDpToExpMap )

      if self.tcToCosNamedMap != tacTcToCosMapName.defaultMapName:
         print( '   TC to COS Profile: %s' % self.tcToCosNamedMap )

      if self.cpuTcToCosNamedMap != tacTcToCosMapName.defaultMapName:
         print( '   CPU TC to COS Profile: %s' % self.cpuTcToCosNamedMap )
      print()

      cfgPortShapeRate = self.configuredPortShapeRate
      operPortShapeRate = self.operationalPortShapeRate

      tmpCfgPortShapeRate = QosLib.invalidShapeRate
      if cfgPortShapeRate:
         tmpCfgPortShapeRate = formatShapeRate( cfgPortShapeRate )

      tmpOperPortShapeRate = QosLib.invalidShapeRate
      if operPortShapeRate:
         tmpOperPortShapeRate = formatShapeRate( operPortShapeRate )

      if self.intfIsLag:
         # Operational shape rate is not supported for LAG interfaces
         # BUG498435: Support this for LAG subinterfaces, because we have an
         # anchor interface.
         shapeRate, shapeRateUnit = \
            QosLib.cliConvertShapeRateToSIUnit( tmpCfgPortShapeRate )
      elif not self.intfQosDetailedInfoAvailable:
         shapeRate, shapeRateUnit = \
            QosLib.cliConvertShapeRateToSIUnit( tmpOperPortShapeRate )
      else:
         cfgStr, operStr, shapeRateUnit = \
            QosLib.cliConvertBothShapeRatesToSIUnit( tmpCfgPortShapeRate,
                                                     tmpOperPortShapeRate )
         if cfgStr + operStr == '--':
            shapeRate = '-'
         else:
            shapeRate = operStr + ' / ' + cfgStr

      if not isFabricIntf:
         if shapeRate == '-':
            print( '   Port shaping rate: disabled' )
         else:
            output = '   Port shaping rate'
            if self.intfIsLag and cfgPortShapeRate and \
                  cfgPortShapeRate.percent == cfgPortShapeRate.rate:
               # No LAG members. show only configured value in percent
               output += ': %s%% of %s' % ( cfgPortShapeRate.percent,
                                            percentReferenceStr(
                                               cfgPortShapeRate.percentOf ) )
            else:
               if shapeRateUnit:
                  output += ' (%s)' % shapeRateUnit
               # Operational / Configured shape rate
               output += ': %s' % shapeRate
               if cfgPortShapeRate and cfgPortShapeRate.percent:
                  output += ', %s%% of %s' % ( cfgPortShapeRate.percent,
                                               percentReferenceStr(
                                                  cfgPortShapeRate.percentOf ) )
            print( output )

      if self.burstSizeConfigSupported:
         configuredBurstSizeVal = '-'
         operationalBurstSizeVal = '-'
         output = '   Burst-size: '
         if self.configuredBurstSize is not None and \
            self.configuredBurstSize.value != tacBurstVal.invalid:
            # There exists a valid burst-size config.
            configuredBurstSizeVal = formatBurstSize( self.configuredBurstSize )
         if self.operationalBurstSize is not None and \
            self.operationalBurstSize.value != tacBurstVal.invalid:
            # There is an operational burst-size value.
            operationalBurstSizeVal = formatBurstSize( self.operationalBurstSize )
         if configuredBurstSizeVal == operationalBurstSizeVal and \
            configuredBurstSizeVal == '-':
            output += 'disabled'
         else:
            if configuredBurstSizeVal == '-':
               assert operationalBurstSizeVal != '-'
               configuredBurstSizeVal = operationalBurstSizeVal
            if self.intfIsLag:
               # Show only configured values for LAGs.
               output += ' %s' % ( configuredBurstSizeVal )
            else:
               if self.intfQosDetailedInfoAvailable:
                  # Operational / Configured burst-size
                  output += ' %s / %s' % ( operationalBurstSizeVal,
                                           configuredBurstSizeVal )
               else:
                  output += ' %s' % ( operationalBurstSizeVal )
         print( output )

      if self.schedulerCompensationConfigSupported:
         configuredSchedulerCompensationVal = '-'
         operationalSchedulerCompensationVal = '-'
         output = '   Scheduler packet size adjustment: '
         if self.configuredSchedulerCompensation is not None and \
            self.configuredSchedulerCompensation.value != \
            tacSchedulerCompensation.invalid:
            # There exists a scheduler compensation config
            configuredSchedulerCompensationVal = formatSchedulerCompensation(
               self.configuredSchedulerCompensation )
         if self.operationalSchedulerCompensation is not None and \
            self.operationalSchedulerCompensation.value != \
            tacSchedulerCompensation.invalid:
            # There exists a scheduler compensation operational value
            operationalSchedulerCompensationVal = formatSchedulerCompensation(
               self.operationalSchedulerCompensation )
         if configuredSchedulerCompensationVal == \
            operationalSchedulerCompensationVal \
            and configuredSchedulerCompensationVal == '-':
            output += 'default'
         else:
            if configuredSchedulerCompensationVal == '-':
               assert operationalSchedulerCompensationVal != '-'
               configuredSchedulerCompensationVal = \
               operationalSchedulerCompensationVal
            if self.intfIsLag:
               # show only configured values for lag
               output += ' %s' % ( configuredSchedulerCompensationVal )
            else:
               if self.intfQosDetailedInfoAvailable:
                  output += ' %s / %s' % ( operationalSchedulerCompensationVal,
                                           configuredSchedulerCompensationVal )
               else:
                  output += ' %s' % ( operationalSchedulerCompensationVal )
         print( output )

      if self.policerPktSizeAdjConfigSupported:
         configuredPolicerPktSizeAdjStr = '-'
         operationalPolicerPktSizeAdjStr = '-'
         output = '   Policer packet size adjustment: '
         if self.configuredPolicerPktSizeAdj is not None and \
            self.configuredPolicerPktSizeAdj.profile != \
               tacPolicerPktSizeAdjProfileName.defaultProfileName:
             # There exists a valid packet size adj config
            configuredPolicerPktSizeAdjStr = formatPolicerPktSizeAdj(
               self.configuredPolicerPktSizeAdj )
         if self.operationalPolicerPktSizeAdj is not None and \
            self.operationalPolicerPktSizeAdj.profile != \
               tacPolicerPktSizeAdjProfileName.defaultProfileName:
             # There exist a valid packet size adj status
            operationalPolicerPktSizeAdjStr = formatPolicerPktSizeAdj(
               self.operationalPolicerPktSizeAdj )
         if configuredPolicerPktSizeAdjStr == operationalPolicerPktSizeAdjStr and \
            configuredPolicerPktSizeAdjStr == '-':
            output += 'default'
         else:
            if self.intfIsLag:
               # show only configured values for lag
               output += ' %s' % ( configuredPolicerPktSizeAdjStr )
            else:
               if self.intfQosDetailedInfoAvailable:
                  output += ' %s / %s' % ( operationalPolicerPktSizeAdjStr,
                                           configuredPolicerPktSizeAdjStr )
               else:
                  output += ' %s' % ( operationalPolicerPktSizeAdjStr )
         print( output )

      if not isFabricIntf and self.interfaceGuaranteedBwSupported:
         cfgPortGuaranteedBw = self.configuredPortGuaranteedBw
         operPortGuaranteedBw = self.operationalPortGuaranteedBw

         tmpCfgGuaranteedBw = QosLib.invalidGuaranteedBw
         tmpOperGuaranteedBw = QosLib.invalidGuaranteedBw

         if cfgPortGuaranteedBw:
            tmpCfgGuaranteedBw = cfgPortGuaranteedBw.formatGuaranteedBw()

         if operPortGuaranteedBw:
            tmpOperGuaranteedBw = operPortGuaranteedBw.formatGuaranteedBw()

         if self.intfIsLag:
            # Operational guaranteed bandwidth is not supported for LAG interfaces
            # BUG498435: Support this for LAG subinterfaces, because we have an
            # anchor interface.
            guaranteedBw, guaranteedBwUnits = \
               QosLib.cliConvertGuaranteedBwToSIUnit( tmpCfgGuaranteedBw )
         elif not self.intfQosDetailedInfoAvailable:
            guaranteedBw, guaranteedBwUnits = \
               QosLib.cliConvertGuaranteedBwToSIUnit( tmpOperGuaranteedBw )
         else:
            cfgStr, operStr, guaranteedBwUnits = \
               QosLib.cliConvertBothGuaranteedBwsToSIUnit( tmpCfgGuaranteedBw,
                                                           tmpOperGuaranteedBw )
            if cfgStr + operStr == '--':
               guaranteedBw = 'disabled'
            else:
               guaranteedBw = operStr + ' / ' + cfgStr

         if guaranteedBw == '-' or guaranteedBw == 'disabled':
            print( '   Port bandwidth: disabled' )
         else:
            output = '   Port bandwidth'
            if self.intfIsLag and cfgPortGuaranteedBw and \
                  cfgPortGuaranteedBw.percent:
               # No LAG members. show only configured value in percent
               output += ': %s%% of %s' % (
                  cfgPortGuaranteedBw.percent,
                  percentReferenceStr( cfgPortGuaranteedBw.percentOf ) )
            else:
               if guaranteedBwUnits:
                  output += ' (%s)' % guaranteedBwUnits
                  # Operational / Configured shape rate
                  output += ': %s' % guaranteedBw
                  if cfgPortGuaranteedBw and cfgPortGuaranteedBw.percent:
                     output += ', %s%% of %s' % (
                        cfgPortGuaranteedBw.percent,
                        percentReferenceStr( cfgPortGuaranteedBw.percentOf ) )
            print( output )

      if self.txQueueQosModel:
         self.txQueueQosModel.renderTxQueueQosModel(
            self.intfIsLag, isFabricIntf, isSubIntf,
            self.intfQosDetailedInfoAvailable,
            displayLegend=displayLegend,
            mixedNumTxqPortsInSystem=mixedNumTxqPortsInSystem )
      if self.schedulingGroupModel:
         self.schedulingGroupModel.renderIntfSchedulingGroupTable()

class PolicyQosModel( Model ):
   name = Str( help="Name of the policy map" )
   directionString = Enum( values=( 'input', 'output' ),
                           help="Direction in which policy is applied" )
   status = Enum( values=( QosLib.tacPMapHwPrgmStatus.hwPrgmStatusSuccess,
                           QosLib.tacPMapHwPrgmStatus.hwPrgmStatusFail,
                           QosLib.tacPMapHwPrgmStatus.hwPrgmStatusInProgress ),
                  help="Hardware programming status" )

class IntfServicePolicyQosModel( Model ):
   intfPolicyMaps = Dict( valueType=PolicyQosModel,
                          help="Policy maps list for an interface \
                                for different directions" )

   def render( self ):  # pylint: disable=useless-return
      print()
      for ipm in self.intfPolicyMaps.values():
         print( "Service-policy %s %s ( programming: %s )\n" %
               ( ipm.name, ipm.directionString,
                 QosLib.programmedPMapHwStatusFromEnum( ipm.status ) ) )
      return

class EcnParametersModel( Model ):
   minThreshold = Int( help="Minimum Threshold for ECN" )
   maxThreshold = Int( help="Maximum Threshold for ECN" )
   unit = Enum( values=QueueThresholdUnit.attributes,
                help="Threshold unit for ECN" )
   segmentSizeInBytes = Int( help="Size of segment unit in bytes",
                             optional=True )
   maxDroprate = Int( help="Mark probability rate", optional=True )
   operationalMaxDroprate = Int( help="Operational Mark probability rate",
                                 optional=True )
   weight = Int( help="Weight for averaging queue size", optional=True )

class DecnParametersModel( Model ):
   offset = Int( help="Dynamic ECN offset value" )
   floor = Int( help="Dynamic ECN floor value" )
   unit = Enum( values=QueueThresholdUnit.attributes,
                help="Dynamic ECN threshold unit" )

class IntfBufferBasedDecnModel( Model ):
   txQueues = Dict( keyType=str,
                    valueType=DecnParametersModel,
                    valueOptional=True, # DECN disabled will be null
                    help="Transmit queue dynamic ECN settings" )

class IntfAllBufferBasedDecnModel( Model ):
   intfs = Dict( keyType=Interface,
                 valueType=IntfBufferBasedDecnModel,
                 help="Interface dynamic ECN" )

   def render( self ):
      fmtL = Format( justify="right" )
      fmtL.padLimitIs( True )
      fmtL.noPadLeftIs( True )
      fmt = Format( justify="right", minWidth=10 )
      fmt.padLimitIs( True )
      fmt.noPadLeftIs( True )

      for intf, model in self.intfs.items():
         print( f"{intf}:\n" )

         table = createTable( ( "Tx-Queue", "Floor", "Offset", "Unit", ) )
         for txQueue, decnParams in model.txQueues.items():
            if decnParams:
               table.newRow(
                  txQueue, decnParams.floor, decnParams.offset, decnParams.unit )
            else:
               table.newRow( txQueue, "-", "-", "-" )

         table.formatColumns( fmtL, fmt, fmt, fmt )
         print( table.output() )

class EcnDelayTxQParametersModel( Model ):
   maxThreshold = Int( help="Maximum Threshold for ECN" )
   unit = Enum( values=QueueThresholdUnit.attributes,
                help="Threshold unit for ECN" )

class WredParametersModel( Model ):
   minThreshold = Int( help="Minimum Threshold for WRED" )
   maxThreshold = Int( help="Maximum Threshold for WRED" )
   unit = Enum( values=QueueThresholdUnit.attributes,
                help="Threshold unit for WRED" )
   maxDroprate = Int( help="Drop probability rate" )
   operationalMaxDroprate = Int( help="Operational Mark probability rate",
                                 optional=True )
   segmentSizeInBytes = Int( help="Size of segment unit in bytes",
                             optional=True )
   weight = Int( help="Weight for averaging queue size", optional=True )
   dp = Int( help="Drop Precedence value", optional=True )

class EcnDelayConfigurationModel( Model ):
   configuredThreshold = Float( help="Configured Threshold value in microseconds",
                                optional=True )
   source = Enum( values=( 'global', 'local', 'none' ), help="Configuration source",
                  optional=True )

   def renderSource( self ):
      if self.source == "global":
         return "G"
      elif self.source == "local":
         return "L"
      else:
         return "-"

class EcnDelayParametersModel( EcnDelayConfigurationModel ):
   threshold = Float( help="Actual Threshold value in microseconds." )

class DropPrecedenceThresholdsModel( Model ):
   percent = Int( help="Tail drop-thresholds in percent of corresponding"
                  "Drop Precedence 0 threshold value" )

class LatencyThresholdModel( Model ):
   threshold = Int( help='Configured latency threshold' )
   unit = Str( help='Latency threshold unit' )

class TxQueueProfileModel( Model ):

   class TxQueue( Model ):
      txQueue = Int( help="Transmit Queue id" )
      # On certain platforms there are separate unicast and multicast tx-queues
      txQueueType = Enum( values=( 'uc-tx-queue', 'mc-tx-queue', 'tx-queue' ),
                          help="Type of Transmit Queue. It is tx-queue for \
                          platforms that don't support multicast and unicast queues",
                          default='tx-queue', optional=True )
      shapeRate = Submodel( valueType=ShapeRateModel,
                            help="Configured shape rate",
                            optional=True )
      guaranteedBw = Submodel( valueType=GuaranteedBwModel,
                               help="Configured guaranteed bandwidth",
                               optional=True )
      ecn = Submodel( valueType=EcnParametersModel,
                      help="Transmit Queue ECN Parameters. \
                            Not Applicable if Transmit Queue ECN is disabled.",
                      optional=True )
      decn = Submodel( valueType=DecnParametersModel,
                       help=( "Transmit Queue Dynamic ECN Parameters. "
                              "Not Applicable if Transmit Queue Dynamic ECN is "
                              "disabled." ),
                       optional=True )
      ecnDelayConfig = Submodel( valueType=EcnDelayTxQParametersModel,
                          help="Transmit Queue ECN Delay Parameters. \
                               Not Applicable if Transmit Queue ECN is disabled.",
                           optional=True )

      ecnDelay = Submodel( valueType=EcnDelayConfigurationModel,
                     help="Transmit Queue Delay based ECN Parameters. \
                           Not Applicable if Transmit Queue delay ECN is disabled.",
                     optional=True )
      ecnCounter = Bool( help="Count congested packets.", optional=True )
      wred = Submodel( valueType=WredParametersModel,
                      help="Transmit Queue WRED Parameters. \
                            Not Applicable if Transmit Queue WRED is disabled.",
                      optional=True )
      dpWred = Dict( keyType=int,
                     valueType=WredParametersModel,
                     help="Transmit Queue Dp WRED Parameters."
                             "Not Applicable if Transmit Queue Dp WRED is disable",
                     optional=True )
      nonEct = Submodel( valueType=EcnParametersModel,
                      help="Transmit Queue NON-ECT Parameters. \
                            Not Applicable if Transmit Queue NON-ECT is disabled.",
                      optional=True )

      wrrBandwidthPercent = Int( help="Configured WRR bandwidth in percent",
                                 optional=True )

      weightConfig = Int( help="Configured weight", optional=True )

      dpThresholds = Dict( keyType=int,
                           valueType=DropPrecedenceThresholdsModel,
                           help="Tail drop thresholds per Drop Precedence",
                           optional=True )

      latencyThreshold = Submodel( valueType=LatencyThresholdModel,
                                   help='Configured latency threshold',
                                   optional=True )

      noPriority = Bool( help="Transmit Queue priority", optional=True )
      ecnWeightSupported = Bool( help="ECN Weight Supported", default=False )
      wredWeightSupported = Bool( help="WRED Weight Supported", default=False )
      nonEctWeightSupported = Bool( help="NON-ECT Weight Supported", default=False )
      ecnMarkProbSupported = Bool( help="ECN Mark Probability Supported,",
                                   default=False )
      schedulerProfile = Str( help="Scheduler profile", optional=True )

      def render( self ):
         if self.txQueue != None:
            if self.txQueueType:
               # pylint: disable-next=bad-string-format-type
               output = "    %s %d\n\t" % ( self.txQueueType, self.txQueue )
            else:
               output = "    tx-queue %d\n\t" % ( self.txQueue )
            if self.noPriority:
               output += "no priority\n\t"
            if self.wrrBandwidthPercent:
               output += "bandwidth percent %d\n\t" % ( self.wrrBandwidthPercent )
            if self.guaranteedBw:
               if self.guaranteedBw.unit != 'percent':
                  output += "bandwidth guaranteed %d %s\n\t" \
                      % ( self.guaranteedBw.bandwidth, self.guaranteedBw.unit )
               else:
                  output += "bandwidth guaranteed %s %d\n\t" \
                      % ( self.guaranteedBw.unit, self.guaranteedBw.bandwidth )
            if self.shapeRate:
               if self.shapeRate.percent:
                  output += "shape rate %d%% of %s\n\t" % (
                     self.shapeRate.percent,
                     percentReferenceStr( self.shapeRate.percentOf, txQueue=True ) )
               else:
                  output += "shape rate %d %s\n\t" % \
                            ( self.shapeRate.rate, self.shapeRate.unit )
            if self.weightConfig:
               if tacQueueWeight.defaultWeight != self.weightConfig:
                  avg_que = "queue length "
                  weight = "weight %d\n" % ( self.weightConfig )
                  output = avg_que + weight
            if self.ecn:
               randomDetectStr = "random-detect ecn "
               minThdStr = "minimum-threshold %d %s " \
                     % ( self.ecn.minThreshold, self.ecn.unit )
               maxThdStr = "maximum-threshold %d %s"\
                     % ( self.ecn.maxThreshold, self.ecn.unit )
               output += randomDetectStr + minThdStr + maxThdStr
               if self.ecnMarkProbSupported:
                  output += " max-mark-probability %d" % ( self.ecn.maxDroprate )
               if self.ecnWeightSupported:
                  output += " weight %d" % ( self.ecn.weight )
               output += "\n\t"

            if self.decn:
               output += ( f"random-detect ecn dynamic floor {self.decn.floor} "
                           f"{self.decn.unit} offset {self.decn.offset} "
                           f"{self.decn.unit}" )

            if self.ecnDelay:
               if self.ecnDelay.source == "global":
                  output += "ecn delay"
               elif self.ecnDelay.source == "local":
                  output += "ecn delay threshold %d \n" % \
                            self.ecnDelay.configuredThreshold
            if self.ecnCounter:
               output += "random-detect ecn count \n\t"

            if self.ecnDelayConfig:
               output += "random-detect ecn delay threshold %d %s\n\t" \
                         % ( self.ecnDelayConfig.maxThreshold,
                             self.ecnDelayConfig.unit )
            if self.wred:
               randomDetectStr = "random-detect drop "
               minThdStr = "minimum-threshold %d %s " \
                     % ( self.wred.minThreshold, self.wred.unit )
               maxThdStr = "maximum-threshold %d %s "\
                     % ( self.wred.maxThreshold, self.wred.unit )
               dropRateStr = "drop-probability %d"\
                     % ( self.wred.maxDroprate )
               output += randomDetectStr + minThdStr + maxThdStr + dropRateStr
               if self.wredWeightSupported:
                  output += " weight %d\n\t" % ( self.wred.weight )
            if self.dpWred:
               randomDetectStr = "random-detect drop "
               for dp in sorted( self.dpWred.keys() ):
                  dpWred = self.dpWred[ dp ]
                  dpStr = "drop-precedence %d " % dpWred.dp
                  minThdStr = "minimum-threshold %d %s " \
                        % ( dpWred.minThreshold, dpWred.unit )
                  maxThdStr = "maximum-threshold %d %s "\
                        % ( dpWred.maxThreshold, dpWred.unit )
                  dropRateStr = "drop-probability %d"\
                        % ( dpWred.maxDroprate )
                  output += randomDetectStr + dpStr + minThdStr + maxThdStr + \
                            dropRateStr
                  if self.wredWeightSupported:
                     output += " weight %d" % ( dpWred.weight )
                  output += "\n\t"
            if self.nonEct:
               randomDetectStr = "random-detect non-ect "
               minThdStr = "minimum-threshold %d %s " \
                           % ( self.nonEct.minThreshold, self.nonEct.unit )
               maxThdStr = "maximum-threshold %d %s" \
                           % ( self.nonEct.maxThreshold, self.nonEct.unit )
               output += randomDetectStr + minThdStr + maxThdStr
               if self.nonEctWeightSupported:
                  output += " weight %d\n\t" % ( self.nonEct.weight )
            for dp in sorted( self.dpThresholds.keys() ):
               dpThreshold = self.dpThresholds[ dp ]
               output += "drop-precedence %d drop-threshold percent %d\n\t" % \
                         ( dp, dpThreshold.percent )

            if self.latencyThreshold and \
               self.latencyThreshold.threshold != tacLatencyThreshold.invalid:
               output += "latency maximum %d %s\n\t" % \
                         ( self.latencyThreshold.threshold,
                           self.latencyThreshold.unit )

            if self.schedulerProfile:
               output += "scheduler profile %s\n\t" % self.schedulerProfile

            print( output )

   txQueueModel = Submodel( valueType=TxQueue,
                      help="Transmit Queue model" )

   def render( self ):
      if self.txQueueModel:
         self.txQueueModel.render()

class TxQueueShapeRateAdaptiveModel( Model ):
   configuredMode = Bool( help="Configured txQueue percentage \
                          based allocation",
                          default=False )
   operationalMode = Bool( help="Operational txQueue percentage \
                           based allocation",
                           default=False )

   def render( self ):
      outStr = "Shape rate percent : "\
                f"{'Non-' if not self.operationalMode else '' }Adaptive"
      if self.configuredMode != self.operationalMode:
         outStr += " (reboot required for "\
                f"{'Non-' if not self.configuredMode else '' }Adaptive)"
      print( outStr )

class TxQueueSchedulerMapModel( Model ):
   memberIntfs = Dict( keyType=Interface, valueType=bool,
                       optional=True,
                       help='A mapping of member interfaces with their '
                            'configuration status' )
   txQueues = Dict( keyType=int, valueType=str,
                    help='A mapping of tx queue to its scheduler profile' )

   def render( self ):
      if self.memberIntfs:
         # If members, this is a lag interface
         mapping = ", ".join( f"{intfName}{'' if isConf else '*'}"
                              for intfName, isConf
                              in self.memberIntfs.items() )
         print( f"  Members: {mapping}\n" )
      if self.txQueues:
         headings = ( ( 'Tx Queue', 'r' ), ( 'Scheduler profile', 'l' ) )
         table = createTable( headings, indent=2 )
         for txQueueId, profileName in sorted( self.txQueues.items() ):
            if profileName == tacTxQueueSchedulerProfileName.defaultProfile:
               table.newRow( txQueueId, '-' )
            else:
               table.newRow( txQueueId, profileName )
         print( table.output() )
      else:
         print( "  No scheduler profile configuration found" )

class SchedulerProfileRecapModel( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=TxQueueSchedulerMapModel,
                      help='A mapping of interfaces to their '
                           'tx queue to scheduler profile mapping' )

   def render( self ):
      if self.interfaces:
         for interface, txQueueSchedulerMap in self.interfaces.items():
            print( f"{interface}:" )
            txQueueSchedulerMap.render()
      else:
         print( "No scheduler profile configuration found on requested interfaces" )
      print()

class PfcProfileModel( Model ):
   dynamicLoadBalancing = Bool( help="Dynamic load balancing",
                                default=False )

class LagSubIntfLoadbalancingModel( Model ):
   configuredLagSubIntfLoadBalancing = Bool( help="Configured Hardware based \
                                             port-channel load balancing",
                                             default=False )
   operationalLagSubIntfLoadBalancing = Bool( help="Operational Hardware based \
                                              port-channel load balancing",
                                              default=False )

   def render( self ):
      if not self.configuredLagSubIntfLoadBalancing and \
         not self.operationalLagSubIntfLoadBalancing:
         print( "Load-balancing on lag subinterfaces with Queues is disabled" )
      elif self.configuredLagSubIntfLoadBalancing and \
           not self.operationalLagSubIntfLoadBalancing:
         print( "Load-balancing on lag subinterfaces with Queues is configured "
                "but inactive (reboot required to activate)" )
      elif self.configuredLagSubIntfLoadBalancing and\
           self.operationalLagSubIntfLoadBalancing:
         print( "Load-balancing on lag subinterfaces with Queues is active" )
      elif self.operationalLagSubIntfLoadBalancing and\
           not self.configuredLagSubIntfLoadBalancing:
         print( "Load-balancing on lag subinterfaces with Queues is active "
                "but not configured (reboot required to deactivate)" )

class PfcWatchdogPortTimerModel( Model ):
   timeout = Float( help="Configure timeout after which port should be \
                    errdisabled or should start dropping on congested priorities \
                    or just notify about the event",
                    default=0.0 )
   pollingInterval = Float( help="Configure the interval at which the \
                            watchdog should poll the queue(s)", default=0.0 )
   recoveryTime = Float( help="Configure recovery-time after which stuck queue \
                         should recover and start forwarding", default=0.0 )
   forcedRecovery = Bool( help="Force recover any stuck queue(s) after the \
                          recoveryTimer duration, irrespective of whether PFC \
                          frames are being received or not", default=False )

class QosProfileModel( Model ):
   numTxQueueSupported = Int( help="Number of Transmit Queues supported",
                              default=0 )
   trustMode = Enum( values=( 'dscp', 'cos', 'untrusted', 'invalid', ),
                     help="Qos Trustmode of interface", default='invalid' )
   defaultCosSupported = Bool( help="Default COS supported", default=False )
   defaultCos = Int( help="Default COS value", optional=True )
   defaultDscpSupported = Bool( help="Default DSCP supported", default=False )
   defaultDscp = Int( help="Default DSCP value", optional=True )
   schedulerCompensationSupported = Bool( help="Scheduler compensation supported",
                                          default=False )
   configuredSchedulerCompensation = Submodel(
      valueType=SchedulerCompensationModel,
      help="Configured port scheduler compensation", optional=True )
   dscpPreserveIpMplsEncapMode = Bool( help="DSCP Preserve for MPLS IP encap",
                                       default=False )
   configuredPortShapeRate = Submodel( valueType=ShapeRateModel,
                                        help="Configured port shape rate",
                                        optional=True )
   configuredPortGuaranteedBw = Submodel(
         valueType=GuaranteedBwModel,
         help="Configured port Guaranteed Bandwidth",
         optional=True )

   pfcEnabled = Bool( help="PFC enabled", default=False )
   pfcPriorities = Dict( keyType=int,
                         valueType=PfcProfileModel,
                         help="A mapping between PFC priority \
                               and PFC configuration" )
   pfcWatchdogEnabled = Bool( help="PFC Watchdog enabled",
                             default=True )
   pfcWatchdogPortAction = Str( help="Configured per-port action value \
                                for PFC Watchdog", optional=True )
   pfcWatchdogPortTimerValues = Submodel( valueType=PfcWatchdogPortTimerModel,
                                          help="Configured per-port timer values \
                                          for PFC Watchdog", optional=True )
   configuredServicePolicy = Submodel( valueType=PolicyQosModel,
                                       help="Configured Service Policy",
                                       optional=True )
   txQueueProfiles = Dict( keyType=int,
                           valueType=TxQueueProfileModel,
                           help="A mapping between Transmit queue id \
                                 and Transmit Queue configuration" )

   def append( self, txQueueId, txQueueType, txQueueModel, txQueueProfileModel ):
      self.txQueueProfiles[ txQueueId ] = txQueueProfileModel
      self.txQueueProfiles[ txQueueId ].txQueueModel = txQueueModel
      if txQueueType[ : 2 ] in [ 'uc', 'mc' ]:
         prefix = txQueueType[ : 2 ] + '-'
      else:
         prefix = ''
      self.txQueueProfiles[ txQueueId ].txQueueModel.txQueue = txQueueId
      self.txQueueProfiles[ txQueueId ].txQueueModel.txQueueType = prefix + \
         'tx-queue'

   def renderShapeRate( self ):
      if not self.configuredPortShapeRate:
         return

      if self.configuredPortShapeRate.unit == 'unspecified':
         shapeRateUnit = ''
      else:
         shapeRateUnit = self.configuredPortShapeRate.unit

      shapeRateShared = "shared" if self.configuredPortShapeRate.shared else ""

      if shapeRateIsValid( self.configuredPortShapeRate.rate ):
         if self.configuredPortShapeRate.rate != QosLib.invalidShapeRate.rate:
            shapeRate = str( self.configuredPortShapeRate.rate )
            if self.configuredPortShapeRate.percent:
               reference = percentReferenceStr(
                  self.configuredPortShapeRate.percentOf )
               print( '    Port shaping rate: %s%% of %s %s' %
                      ( shapeRate, reference, shapeRateShared ) )
            else:
               print( '    Port shaping rate: %s %s %s' %
                      ( shapeRate, shapeRateUnit, shapeRateShared ) )

   def renderGuaranteedBw( self ):
      if not self.configuredPortGuaranteedBw:
         return

      if self.configuredPortGuaranteedBw.unit == 'unspecified':
         guaranteedBwUnit = ''
      else:
         guaranteedBwUnit = self.configuredPortGuaranteedBw.unit

      if guaranteedBwIsValid( self.configuredPortGuaranteedBw.bandwidth ):
         if self.configuredPortGuaranteedBw.bandwidth != \
               QosLib.invalidGuaranteedBw.bw:
            guaranteedBw = str( self.configuredPortGuaranteedBw.bandwidth )
            if self.configuredPortGuaranteedBw.percent:
               reference = percentReferenceStr(
                  self.configuredPortGuaranteedBw.percentOf )
               print( '    Port guaranteed Bw: %s%% of %s' %
                      ( guaranteedBw, reference ) )
            else:
               print( '    Port guaranteed Bw: %s %s' %
                      ( guaranteedBw, guaranteedBwUnit ) )

   def render( self ):
      if self.trustMode != QosLib.tacTrustMode.invalid:
         print( '    Trust Mode: %s' % self.trustMode )

      if self.defaultCosSupported and self.defaultCos != QosLib.tacCos.invalid:
         print( '    Default COS: %s' % self.defaultCos )

      if self.defaultDscpSupported and self.defaultDscp != QosLib.tacDscp.invalid:
         print( '    Default DSCP: %s' % self.defaultDscp )

      if self.pfcEnabled:
         print( "    priority-flow-control on " )
      if not self.pfcWatchdogEnabled:
         print( "    no priority-flow-control pause watchdog" )
      for priority in sorted( self.pfcPriorities.keys() ):
         output = ""
         output += "    priority-flow-control priority %d no-drop " % ( priority )
         if self.pfcPriorities[ priority ].dynamicLoadBalancing:
            output += "dlb"
         print( output )
      if self.pfcWatchdogPortTimerValues:
         portTimerConfig = self.pfcWatchdogPortTimerValues
         cmdString = 'priority-flow-control pause watchdog port timer'
         timeoutString = ' timeout %2.2f' % ( portTimerConfig.timeout )
         cmdString += timeoutString
         pollingIntervalValue = "auto" if not \
                                portTimerConfig.pollingInterval else \
                                "%.3f" % ( portTimerConfig.pollingInterval )
         pollingIntervalString = " polling-interval %s" % ( pollingIntervalValue )
         cmdString += pollingIntervalString
         recoveryTimeValue = "0" if not portTimerConfig.recoveryTime else \
                             "%2.2f" % ( portTimerConfig.recoveryTime )
         if portTimerConfig.forcedRecovery:
            recoveryTimeValue += " forced"
         recoveryTimeString = " recovery-time %s" % ( recoveryTimeValue )
         cmdString += recoveryTimeString
         output = "    %s" % ( cmdString )
         print( output )
      if self.pfcWatchdogPortAction and \
         self.pfcWatchdogPortAction != 'invalid':
         watchdogPortAction = self.pfcWatchdogPortAction
         if self.pfcWatchdogPortAction == 'notifyOnly':
            watchdogPortAction = 'notify-only'
         print( "    priority-flow-control pause watchdog port action %s"
                % ( watchdogPortAction ) )
      cfgSrvPolicy = self.configuredServicePolicy
      if cfgSrvPolicy:
         print( "    Service-policy %s %s\n" %
                ( cfgSrvPolicy.name, cfgSrvPolicy.directionString ) )

      if self.schedulerCompensationSupported and \
         self.configuredSchedulerCompensation.value != \
            tacSchedulerCompensation.invalid:
         compensation = formatSchedulerCompensation(
            self.configuredSchedulerCompensation )
         print( "    Scheduler packet size adjustment: %s" % compensation )

      if self.dscpPreserveIpMplsEncapMode:
         print( "    qos rewrite dscp mpls encapsulation disabled" )

      self.renderShapeRate()
      self.renderGuaranteedBw()
      for txQueueId in sorted( self.txQueueProfiles.keys() ):
         txQueueProfile = self.txQueueProfiles[ txQueueId ]
         txQueueProfile.render()

class QosProfileAllModel( Model ):
   qosProfiles = Dict( keyType=str, valueType=QosProfileModel,
                       help="A mapping between QoS profile name and QoS profiles" )

   def append( self, qosProfileName, qosProfileModel ):
      self.qosProfiles[ qosProfileName ] = qosProfileModel

   def render( self ):
      for qosProfileName in sorted( self.qosProfiles.keys() ):
         qosProfile = self.qosProfiles[ qosProfileName ]
         print( "qos profile %s" % ( qosProfileName ) )
         qosProfile.render()

class IntfQosProfileModel( Model ):
   qosProfileName = Str( help="QoS profile name" )

   def render( self ):
      if self.qosProfileName:
         print( "    service-profile %s " % ( self.qosProfileName ) )

class QosProfileConfiguredIntfs( Model ):
   intfs = List( valueType=str, help="Interface name" )

class QosProfileSummaryModel( Model ):
   qosProfileApplication = Dict( keyType=str, valueType=QosProfileConfiguredIntfs,
                             help="A mapping between QoS Profile name "
                                "and configured interfaces" )

   def render( self ):
      tableWidth = terminalWidth() - 1
      t = TableFormatter( tableWidth=tableWidth )
      f = Format( justify="left", wrap=True, maxWidth=tableWidth, minWidth=1 )
      for profileName in sorted( self.qosProfileApplication ):
         t.newRow( 'Qos Profile: %s' % profileName )
         intfList = intfListToCanonical(
            self.qosProfileApplication[ profileName ].intfs, strLimit=60 )
         firstLine = True
         if not len( intfList ):  # pylint: disable=use-implicit-booleaness-not-len
            t.newRow( '   Configured on: ' )
         for intfs in intfList:
            if firstLine:
               firstLine = False
               t.newRow( '   Configured on: ' + intfs )
            else:
               t.newRow( '                  ' + intfs )
      t.formatColumns( f )
      print( t.output() )

class IntfQosAllModel( Model ):
   intfQosModel = Submodel( valueType=InterfaceQosModel,
                            help="Qos on Interface" )
   intfServicePolicyQosModel = Submodel( valueType=IntfServicePolicyQosModel,
                                         help="Service Policy on Interface" )
   def renderIntfQosAllModel( self, displayLegend=True,
                              mixedNumTxqPortsInSystem=False ):
      if self.intfQosModel:
         self.intfQosModel.renderInterfaceQosModel( displayLegend=displayLegend,
               mixedNumTxqPortsInSystem=mixedNumTxqPortsInSystem )
      if self.intfServicePolicyQosModel:
         self.intfServicePolicyQosModel.render()

class QosAllSchedulingGroupModel( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=IntfSchedulingGroupWithMembersModel,
                      help="A mapping of interface to its QoS scheduling groups" )

   def insert( self, interface, intfSchedulingGroupModel ):
      self.interfaces[ interface ] = intfSchedulingGroupModel

   def render( self ):
      if not self.interfaces:
         return
      interfaces = Arnet.sortIntf( self.interfaces )
      for intf in interfaces:
         print( "Interface:", intf )
         self.interfaces[ intf ].renderIntfSchedulingGroupWithMembers()
      print( "Note: Values are displayed as Operational/Configured" )
      print()

class QosSchedulingHierarchyModel( Model ):

   class HierarchyModel( SchedulingElementModel ):
      name = Str( help="Name of the entity in the hierarchy" )
      interfaceSpeed = Int( help="linkSpeed (bps)", optional=True )

   txQueues = List( valueType=HierarchyModel,
                    help="List of transmit queues on the subinterface",
                    optional=True )

   subinterface = Submodel( valueType=HierarchyModel,
                            help="Scheduling information for the subinterface" )

   schedulingGroup = Submodel( valueType=HierarchyModel,
                               help="Scheduling group containing the subinterface",
                               optional=True )

   parentInterface = Submodel( valueType=HierarchyModel,
                           help="Scheduling information for the parent interface" )

class QosAllSchedulingHierarchyModel( Model ):
   interfaces = Dict( keyType=Interface, valueType=QosSchedulingHierarchyModel,
                  help="A mapping of sub interface to its QoS scheduling hierarchy" )

   def render( self ):
      if not self.interfaces:
         return
      headings = ( "Interface", "Hierarchy Level", "Bandwidth (units)",
                   "Shape Rate (units)" )
      table = createTable( headings, tableWidth=120 )

      for intf in Arnet.sortIntf( self.interfaces ):
         hierarchyModel = self.interfaces[ intf ]
         intfStr = intf
         for q in sorted( hierarchyModel.txQueues, key=lambda q: q.name ):
            hierarchy = "tx queue (%s)" % q.name
            guaranteedBw = q.renderSgGuaranteedBw()
            shapeRate = q.renderSgShapeRate()
            table.newRow( intfStr, hierarchy, guaranteedBw, shapeRate )
            intfStr = ''

         levels = [ ( "subinterface", hierarchyModel.subinterface ),
                    ( "group", hierarchyModel.schedulingGroup ),
                    ( "parent", hierarchyModel.parentInterface ) ]

         for hierarchy, model in levels:
            if not model:
               continue
            if model.interfaceSpeed is not None:
               speedStr = formatValueSi( model.interfaceSpeed, unit="bps" )
               hierarchy = "%s (%s, %s)" % ( hierarchy, model.name, speedStr )
            else:
               hierarchy = "%s (%s)" % ( hierarchy, model.name )
            guaranteedBw = model.renderSgGuaranteedBw()
            shapeRate = model.renderSgShapeRate()
            table.newRow( intfStr, hierarchy, guaranteedBw, shapeRate )
            intfStr = ''

      f1 = Format( justify="left" )
      table.formatColumns( f1, f1, f1, f1 )
      print()
      print( table.output() )
      print( "Note: Values are displayed as Operational/Configured" )
      print()

class IntfAllQosAllModel( Model ):
   intfAllQosAll = Dict( keyType=Interface, valueType=IntfQosAllModel,
                         help="All the Interfaces with all Qos details for it" )
   _mixedNumTxqPortsInSystem = Bool( help="There exists mixed type of interfaces",
                              default=False )

   def insert( self, interface, intfQosAllModel ):
      self.intfAllQosAll[ interface ] = intfQosAllModel

   def render( self ):
      intfAllQosAll = Arnet.sortIntf( self.intfAllQosAll )
      for intf in intfAllQosAll:
         if intfAllQosAll[ -1 ] == intf:
            self.intfAllQosAll[ intf ].renderIntfQosAllModel(
                  mixedNumTxqPortsInSystem=self._mixedNumTxqPortsInSystem )
         else:
            self.intfAllQosAll[ intf ].renderIntfQosAllModel( displayLegend=False )

class FabricAllQosAllModel( Model ):
   fabricAllQosAll = Dict(
      keyType=str, valueType=IntfQosAllModel,
      help="A mapping between Fabric Interfaces to Qos details" )
   _mixedNumTxqPortsInSystem = Bool( help="There exists mixed type of interfaces",
                              default=False )

   def insert( self, interface, intfQosAllModel ):
      self.fabricAllQosAll[ interface ] = intfQosAllModel

   def render( self ):
      fabricAllQosAll = sorted( self.fabricAllQosAll )
      for intf in fabricAllQosAll:
         if fabricAllQosAll[ -1 ] == intf:
            self.fabricAllQosAll[ intf ].renderIntfQosAllModel(
                  mixedNumTxqPortsInSystem=self._mixedNumTxqPortsInSystem )
         else:
            self.fabricAllQosAll[ intf ].renderIntfQosAllModel( displayLegend=False )

class GlobalEcnModel( Model ):

   globalEcnSupported = Bool( help="Global Buffer ECN supported" )
   globalEcnWeightSupported = Bool( help="Global ECN Supported", default=False )
   globalEcnParameters = Submodel(
      valueType=EcnParametersModel,
      help="Global length based ECN Parameters. "
           "Not applicable if Global length based ECN is disabled.",
      optional=True )
   delayEcnSupported = Bool( help="Global Delay based ECN supported" )
   globalEcnDelayParameters = Submodel(
      valueType=EcnDelayParametersModel,
      help="Global delay based ECN Parameters. "
           "Not applicable if Global delay based ECN is disabled.",
      optional=True )

   def renderBufferEcn( self ):
      if self.globalEcnSupported and self.globalEcnParameters:
         print( 'Global Buffer ECN' )
         print( '   Minimum Threshold: ', self.globalEcnParameters.minThreshold )
         print( '   Maximum Threshold: ', self.globalEcnParameters.maxThreshold )
         if self.globalEcnParameters.unit == 'segments':
            unitStr = '%s (%s bytes)' \
                % ( self.globalEcnParameters.unit,
                    self.globalEcnParameters.segmentSizeInBytes )
         else:
            unitStr = self.globalEcnParameters.unit
         print( '   Threshold Unit: ', unitStr )
         if self.globalEcnWeightSupported:
            print( '   Weight: ', self.globalEcnParameters.weight )
         print()
      elif self.globalEcnSupported:
         print( '   Global Buffer ECN Disabled' )
         print()

   def renderDelayEcn( self ):
      if self.delayEcnSupported and self.globalEcnDelayParameters:
         print( 'Global Delay based ECN' )
         print( '   Configured Threshold: %.3f %s' % (
               self.globalEcnDelayParameters.configuredThreshold,
               "microseconds" ) )
         print( '   Actual Threshold: %.3f %s' % (
               self.globalEcnDelayParameters.threshold,
               "microseconds" ) )
         print()
      elif self.delayEcnSupported:
         print( '   Global Delay based ECN Disabled' )
         print()

   def render( self ):
      self.renderBufferEcn()

class GlobalAllEcnModel( GlobalEcnModel ):
   def render( self ):
      self.renderBufferEcn()
      self.renderDelayEcn()

class EcnDelayModel( Model ):

   ecnDelayParameters = Submodel(
      valueType=EcnDelayParametersModel,
      help="General delay ECN parameters", optional=True )

   def render( self ):
      if self.ecnDelayParameters:
         print( 'Global Delay based ECN' )
         print( '   Configured Threshold: %.3f %s' % (
               self.ecnDelayParameters.configuredThreshold,
               "microseconds" ) )
         print( '   Actual Threshold: %.3f %s' % (
               self.ecnDelayParameters.threshold,
               "microseconds" ) )
      else:
         print( '   Global Delay based ECN Disabled' )
      print()

class EcnCountersModel( Model ):
   chipEcnCounter = Dict( keyType=str, valueType=int,
                          help="A mapping between a chip name and ECN counter" )

   def insert( self, chipName, count ):
      self.chipEcnCounter[ chipName ] = int( count )

   def render( self ):
      formatStr = ' %15s   %20s'
      print( formatStr % ( 'Chip Name', 'Marked Packets', ) )
      print( '-' * 40 )
      for chipName, chipCount in sorted( self.chipEcnCounter.items() ):
         print( formatStr % ( chipName, chipCount ) )
      print()


class EcnQueueCounterModel( Model ):
   queueCounters = Dict( keyType=int, valueType=str,
                         help='Mapping from tx-queue to ecn counters' )

   def insert( self, queueId, pkts ):
      self.queueCounters[ queueId ] = pkts

   def renderTxQCounters( self ):
      print( "   Tx-Queue                Marked Packets" )
      print( "  ----------      -----------------------" )
      for queueId in sorted( self.queueCounters.keys() ):
         pkts = self.queueCounters[ queueId ]
         print( "%7d           %23s" % ( queueId, pkts ) )

class EcnIntfQueueCountersModel( Model ):
   intfQueueCounters = Dict( keyType=Interface,
                             valueType=EcnQueueCounterModel,
                             help='Mapping from interface to tx-queue ecn counters' )

   def insert( self, intf, queueCounterModel ):
      self.intfQueueCounters[ intf ] = queueCounterModel

   def render( self ):
      for intf in Arnet.sortIntf( self.intfQueueCounters ):
         print( "%s:" % intf )
         self.intfQueueCounters[ intf ].renderTxQCounters()

class MetaWredQueueCounterModel( Model ):
   enqueuePkts = Int( help="Total enqueued packets", optional=True )
   enqueueBytes = Int( help="Total enqueued bytes", optional=True )
   wredDropPkts = Int( help="Packets dropped due to WRED" )
   wredDropBytes = Int( help="Bytes dropped due to WRED", optional=True )
   tailDropPkts = Int( help="Tail dropped packets", optional=True )
   tailDropBytes = Int( help="Tail dropped bytes", optional=True )
   ecnPkts = Int( help="ECN marked packets", optional=True )
   ecnBytes = Int( help="ECN marked bytes", optional=True )

class WredQueueCounterModel( Model ):
   queueCounters = Dict( keyType=int, valueType=MetaWredQueueCounterModel,
                         help='Mapping from tx-queue to wred counters' )

   def insert( self, queueId, meta ):
      self.queueCounters[ queueId ] = meta

   def renderTxQCounters( self ):
      if not self.queueCounters:
         return

      fl = Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = Format( justify="right" )

      allHeadings = OrderedDict( [
            ( "Enqueue/Pkts", "enqueuePkts" ),
            ( "Enqueue/Bytes", "enqueueBytes" ),
            ( "WredPreTail/Pkts", "wredDropPkts" ),
            ( "WredPreTail/Bytes", "wredDropBytes" ),
            ( "TailDrop/Pkts", "tailDropPkts" ),
            ( "TailDrop/Bytes", "tailDropBytes" ),
            ( "EcnMarked/Pkts", "ecnPkts" ),
            ( "EcnMarked/Bytes", "ecnBytes" )
            ] )

      anyCounter = next( iter( self.queueCounters.values() ) )
      # Filter out all the headers that won't have values
      countHeadings = OrderedDict( [
            ( a, b ) for a, b in allHeadings.items()
            if getattr( anyCounter, b ) is not None
         ] )
      tableHeadings = [ "Tx-Queue" ] + list( countHeadings )
      table = createTable( tableHeadings )
      table.formatColumns( fl, *( [ fr ] * len( countHeadings ) ) )

      for queueId in sorted( self.queueCounters.keys() ):
         counters = self.queueCounters[ queueId ]
         row = [ getattr( counters,
               field ) for field in countHeadings.values() ]
         table.newRow( queueId, *row )
      print( table.output() )

class WredIntfQueueCountersModel( Model ):
   intfQueueCounters = Dict( keyType=Interface,
                             valueType=WredQueueCounterModel,
                             help='Mapping from interface to tx-q wred counters' )

   def insert( self, intf, queueCounterModel ):
      self.intfQueueCounters[ intf ] = queueCounterModel

   def render( self ):
      for intf in Arnet.sortIntf( self.intfQueueCounters ):
         print( "%s:" % intf )
         self.intfQueueCounters[ intf ].renderTxQCounters()

class TxQueueEcnModel( Model ):

   txQueue = Str( help="Name of Egress Queue" )
   txQueueEcnParameters = Submodel(
      valueType=EcnParametersModel,
      help="Transmit Queue ECN Parameters. \
            Not Applicable if Transmit Queue buffer ECN is disabled.",
      optional=True )
   txQueueEcnDelaySupported = Bool( help="Trasmit Queue delay ECN supported." )
   txQueueEcnDelayParameters = Submodel(
      valueType=EcnDelayParametersModel,
      help="Transmit Queue Delay based ECN Parameters. \
            Not Applicable if Transmit Queue delay ECN is disabled.",
      optional=True )
   ecnMarkProbSupported = Bool( help="ECN mark-probability supported",
                                default=False )
   ecnWeightSupported = Bool( help="ECN Weight Supported", default=False )

class TxQueueWredModel( Model ):
   txQueue = Str( help="Name of Egress Queue" )
   txQueueWredParameters = Submodel(
      valueType=WredParametersModel,
      help="Transmit Queue WRED Parameters. \
            Not Applicable if Transmit Queue WRED is disabled.",
      optional=True )
   wredWeightSupported = Bool( help="WRED weight Supported", default=False )
   dpWredSupported = Bool( help="WRED per Drop-Precedence Supported", default=False )

class TxQueueNonEctModel( Model ):
   txQueue = Str( help="Name of Egress Queue" )
   txQueueNonEctParameters = Submodel(
      valueType=EcnParametersModel,
      help="Transmit Queue NON-ECT Parameters. \
            Not Applicable if Transmit Queue NON-ECT is disabled.",
      optional=True )
   nonEctWeightSupported = Bool( help="NON-ECT Weight Supported", default=False )

class IntfEcnModel( Model ):
   txQueueEcnList = List( valueType=TxQueueEcnModel,
                          help="List of Transmit Queue ECN Settings" )
   txQueueNonEctList = List( valueType=TxQueueNonEctModel,
                             help="List of Transmit Queue NON-ECT Settings" )

class IntfWredModel( Model ):
   txQueueWredList = List( valueType=TxQueueWredModel,
                          help="List of Transmit Queue WRED Settings" )

class IntfNonEctModel( Model ):
   txQueueNonEctList = List( valueType=TxQueueNonEctModel,
                          help="List of Transmit Queue NON-ECT Settings" )

class IntfEcnCollectionModel( Model ):

   intfEcnCollection = Dict( keyType=Interface, valueType=IntfEcnModel,
                             help="Interface ECN" )

   def renderHeader( self, delayEcnSupported, ecnMarkProbSupported,
                     ecnWeightSupported ):
      print()
      header1 = '   Tx-Queue    Minimum Threshold    Maximum Threshold' + \
                '         Threshold Unit'
      header2 = '   ---------------------------------------------------' + \
                '-----------------------'

      if ecnMarkProbSupported:
         header1 += '      Mark-Prob'
         header2 += '---------------'
      if ecnWeightSupported:
         header1 += '      Weight'
         header2 += '------------'

      print( header1 )
      print( header2 )

   def renderTable( self, txQueueEcn, ecnMarkProbSupported, ecnWeightSupported ):
      txQEcnParam = txQueueEcn.txQueueEcnParameters
      if txQEcnParam:
         if txQEcnParam.unit == 'segments':
            unitStr = '%s (%s bytes)' % ( txQEcnParam.unit,
                                          txQEcnParam.segmentSizeInBytes )
         else:
            unitStr = txQEcnParam.unit

         ecnStr = '%17s   %18s   %18s' % ( txQEcnParam.minThreshold,
                                           txQEcnParam.maxThreshold,
                                           unitStr )
         if ecnMarkProbSupported:
            ecnStr += '%13s' % ( txQEcnParam.maxDroprate )
            ecnStr += ' / %s' % ( txQEcnParam.operationalMaxDroprate )
         if ecnWeightSupported:
            ecnStr += "%11s" % ( txQEcnParam.weight )
      else:
         ecnStr = '%17s   %18s   %18s' % ( '-', '-', '-' )
         if ecnMarkProbSupported:
            ecnStr += '%13s' % ( '-' )
            ecnStr += ' / %s' % ( '-' )
         if ecnWeightSupported:
            ecnStr += '%11s' % ( '-' )

      print( '   %8s    %s' % ( txQueueEcn.txQueue, ecnStr ) )

   def renderFooter( self, delayEcnSupported ):
      print( ' Note: Values are displayed as Configured/Operational' )
      print()

   def render( self ):
      for intf in Arnet.sortIntf( self.intfEcnCollection ):
         if str( intf ) == fabricIntfName:
            print( '%s:' % fabricTokenName )
         else:
            print( '%s:' % intf )
         intfEcn = self.intfEcnCollection[ intf ]
         if not intfEcn.txQueueEcnList:
            continue
         delayEcnSupported = any( txq.txQueueEcnDelaySupported for txq in
                                  intfEcn.txQueueEcnList )
         ecnMarkProbSupported = any( txq.ecnMarkProbSupported for txq in
                                     intfEcn.txQueueEcnList )
         ecnWeightSupported = any( txq.ecnWeightSupported for txq in
                                   intfEcn.txQueueEcnList )
         self.renderHeader( delayEcnSupported, ecnMarkProbSupported,
                            ecnWeightSupported )
         for txQueueEcn in intfEcn.txQueueEcnList:
            self.renderTable( txQueueEcn, ecnMarkProbSupported, ecnWeightSupported )
         self.renderFooter( delayEcnSupported )

class IntfAllEcnCollectionModel( IntfEcnCollectionModel ):

   def renderFooter( self, delayEcnSupported ):
      if delayEcnSupported:
         print()
         print( ' Delay Threshold Type: G -> Global     L -> Local' )
      print( ' Note: Values are displayed as Configured/Operational' )
      print()

   def createEcnHeader( self, ecnMarkProbSupported, ecnWeightSupported,
                        delayEcnSupported, tableHeader, formatList, fl, fr ):

      tableHeader.extend( [ "Threshold Unit",
                            "ECN\nMinimum\nThreshold",
                            "ECN\nMaximum\nThreshold" ] )

      formatList.extend( [ fl, fr, fr ] )
      if delayEcnSupported:
         tableHeader.append( "ECN\nDelay\nThreshold" )
         formatList.append( fl )
      if ecnMarkProbSupported:
         tableHeader.append( "ECN \nMark-Prob" )
         formatList.append( fl )
      if ecnWeightSupported:
         tableHeader.append( "ECN\nWeight" )
         formatList.append( fr )

   def createNonEctHeader( self, nonEctWeightSupported, tableHeader,
                           formatList, fl, fr ):
      tableHeader.extend( [ "Non-ECT\nMinimum\nThreshold",
                            "Non-ECT\nMaximum\nThreshold",
                            "Non-ECT\nMark-Prob" ] )
      formatList.extend( [ fr, fr, fl ] )

      if nonEctWeightSupported:
         tableHeader.append( "Non-ECT\nWeight " )
         formatList.append( fr )

   def createEcnRow( self, ecnMarkProbSupported, ecnWeightSupported,
                        delayEcnSupported, txQueueEcn, rowList ):
      txQEcnParam = txQueueEcn.txQueueEcnParameters
      txQEcnDelayParam = txQueueEcn.txQueueEcnDelayParameters
      minThresh = str( txQEcnParam.minThreshold ) if txQEcnParam else '-'
      maxThresh = str( txQEcnParam.maxThreshold ) if txQEcnParam else '-'
      if txQEcnParam:
         if txQEcnParam.unit == 'segments':
            unitStr = '%s (%s bytes)' % ( txQEcnParam.unit,
                                          txQEcnParam.segmentSizeInBytes )
         else:
            unitStr = txQEcnParam.unit
      else:
         unitStr = '-'
      rowList.extend( [ unitStr, minThresh, maxThresh ] )
      if delayEcnSupported:
         actualDelayThresh = "%.3f" % txQEcnDelayParam.threshold \
                             if ( txQEcnDelayParam and 
                                  txQEcnDelayParam.threshold ) \
                             else '-'
         configDelayThresh = "%.3f" % txQEcnDelayParam.configuredThreshold \
                             if ( txQEcnDelayParam and 
                                 txQEcnDelayParam.configuredThreshold ) \
                             else '-'
         delayThreshSource = txQEcnDelayParam.renderSource() \
                             if txQEcnDelayParam else '-'
         delayThreshStr = "%s / %s (%s)" % ( configDelayThresh, actualDelayThresh,
                                           delayThreshSource ) \
                          if txQEcnDelayParam else '-'
         rowList.append( delayThreshStr )
      if ecnMarkProbSupported:
         maxProb = str( txQEcnParam.maxDroprate ) if txQEcnParam else '-'
         operMaxProb = str( txQEcnParam.operationalMaxDroprate ) \
                       if txQEcnParam else '-'
         markProbStr = '%s / %s' % ( maxProb, operMaxProb )
         rowList.append( markProbStr )
      if ecnWeightSupported:
         weight = str( txQEcnParam.weight ) if txQEcnParam else '-'
         rowList.append( weight )

   def createNonEctRow( self, nonEctWeightSupported,
                        txQueueNonEct, rowList ):
      txQNonEctParam = txQueueNonEct.txQueueNonEctParameters
      minThresh = str( txQNonEctParam.minThreshold ) if txQNonEctParam else '-'
      maxThresh = str( txQNonEctParam.maxThreshold ) if txQNonEctParam else '-'
      maxProb = str( txQNonEctParam.maxDroprate ) if txQNonEctParam else '-'
      operMaxProb = str( txQNonEctParam.operationalMaxDroprate ) \
                       if txQNonEctParam else '-'
      markProbStr = '%s / %s' % ( maxProb, operMaxProb )
      rowList.extend( [ minThresh, maxThresh, markProbStr ] )
      if nonEctWeightSupported:
         weight = str( txQNonEctParam.weight ) if txQNonEctParam else '-'
         rowList.append( weight )

   def render( self ):
      for intf in Arnet.sortIntf( self.intfEcnCollection ):
         if str( intf ) == fabricIntfName:
            print( '%s:' % fabricTokenName )
         else:
            print( '%s:' % intf )
         intfEcn = self.intfEcnCollection[ intf ]

         if not intfEcn.txQueueEcnList:
            continue
         delayEcnSupported = any( txq.txQueueEcnDelaySupported for txq in
                                  intfEcn.txQueueEcnList )
         ecnMarkProbSupported = any( txq.ecnMarkProbSupported for txq in
                                     intfEcn.txQueueEcnList )
         ecnWeightSupported = any( txq.ecnWeightSupported for txq in
                                   intfEcn.txQueueEcnList )
         nonEctThdSupported = len( intfEcn.txQueueNonEctList ) > 0
         nonEctWeightSupported = nonEctThdSupported and \
                                    any( txq.nonEctWeightSupported
                                      for txq in intfEcn.txQueueNonEctList )
         fl = Format( justify="left" )
         fl.padLimitIs( True )
         fl.noPadLeftIs( True )
         fr = Format( justify="right" )
         fr.padLimitIs( True )
         fr.noPadLeftIs( True )
         fKey = Format( justify="right", isHeading=True )
         fKey.padLimitIs( True )
         fKey.noPadLeftIs( True )
         tableHeader = [ "Tx-Queue" ]
         formatList = [ fKey ]
         self.createEcnHeader( ecnMarkProbSupported, ecnWeightSupported,
         delayEcnSupported, tableHeader, formatList, fl, fr )
         if nonEctThdSupported:
            self.createNonEctHeader( nonEctWeightSupported, tableHeader, formatList,
         fl, fr )
         table = createTable( tuple( tableHeader ) )
         index = 0
         for txQueueEcn in intfEcn.txQueueEcnList:
            txQueueNonEct = intfEcn.txQueueNonEctList[ index ] \
               if nonEctThdSupported else None
            rowList = [ txQueueEcn.txQueue ]
            self.createEcnRow( ecnMarkProbSupported, ecnWeightSupported,
                               delayEcnSupported, txQueueEcn, rowList )
            if nonEctThdSupported:
               self.createNonEctRow( nonEctWeightSupported, txQueueNonEct, rowList )
            table.newRow( *rowList )
            index += 1
         table.formatColumns( *formatList )
         print( '\n' + table.output() )
         self.renderFooter( delayEcnSupported )

class IntfWredCollectionModel( Model ):

   intfWredCollection = Dict( keyType=Interface, valueType=IntfWredModel,
                              help="Interface WRED" )

   def render( self ):
      for intf in Arnet.sortIntf( self.intfWredCollection ):
         print( '%s:' % intf )
         intfWred = self.intfWredCollection[ intf ]
         if not intfWred.txQueueWredList:
            continue
         wredWeightSupported = any( txq.wredWeightSupported
                                    for txq in intfWred.txQueueWredList )
         dpWredSupported = any( txq.dpWredSupported
                                for txq in intfWred.txQueueWredList )

         header1 = '   Tx-Queue     %sMinimum Threshold    Maximum Threshold' + \
                   '   MaxDroprate        Threshold Unit'
         header2 = '   ---------------------------------------------------' + \
                   '------------------------------------%s'
         if dpWredSupported:
            header1 = header1 % ( "Drop Precedence     " )
            header2 = header2 % ( "--------------------" )
         else:
            header1 = header1 % ( "" )
            header2 = header2 % ( "" )

         if wredWeightSupported:
            header1 += '    Weight'
            header2 += '----------'

         print()
         print( header1 )
         print( header2 )

         for txQueueWred in intfWred.txQueueWredList:
            txQWredParam = txQueueWred.txQueueWredParameters
            dp = "-"
            if txQWredParam:
               dp = txQWredParam.dp if txQWredParam.dp is not None else '-'
               if txQWredParam.unit == 'segments':
                  unitStr = '%s (%s bytes)' % ( txQWredParam.unit,
                                                txQWredParam.segmentSizeInBytes )
               else:
                  unitStr = txQWredParam.unit
               wredStr = '%18s   %18s  %12s  %20s' % ( txQWredParam.minThreshold,
                                                       txQWredParam.maxThreshold,
                                                       txQWredParam.maxDroprate,
                                                       unitStr )
               if wredWeightSupported:
                  wredStr += "%10s" % ( txQWredParam.weight )
            else:
               wredStr = '%18s   %18s  %12s  %20s' % ( '-', '-', '-', '-' )
               if wredWeightSupported:
                  wredStr += '%10s' % ( '-' )
            if dpWredSupported:
               print( '   %8s     %15s    %s' % ( txQueueWred.txQueue, dp,
                                                   wredStr ) )
            else:
               print( '   %8s    %s' % ( txQueueWred.txQueue, wredStr ) )
         print()

class IntfNonEctCollectionModel( Model ):
   intfNonEctCollection = Dict( keyType=Interface, valueType=IntfNonEctModel,
                             help="Interface NON-ECT" )

   def render( self ):
      for intf in Arnet.sortIntf( self.intfNonEctCollection ):
         print( '%s:' % intf )
         intfNonEct = self.intfNonEctCollection[ intf ]
         if not intfNonEct.txQueueNonEctList:
            continue
         nonEctWeightSupported = any( txq.nonEctWeightSupported
                                      for txq in intfNonEct.txQueueNonEctList )
         header1 = '   Tx-Queue     Minimum Threshold    Maximum Threshold' + \
               '   MaxDroprate        Threshold Unit'
         header2 = '   ---------------------------------------------------' + \
             '------------------------------------'
         if nonEctWeightSupported:
            header1 += "    Weight"
            header2 += "----------"

         print()
         print( header1 )
         print( header2 )

         for txQueueNonEct in intfNonEct.txQueueNonEctList:
            txQNonEctParam = txQueueNonEct.txQueueNonEctParameters
            if txQNonEctParam:
               if txQNonEctParam.unit == 'segments':
                  unitStr = '%s (%s bytes)' % ( txQNonEctParam.unit,
                                                txQNonEctParam.segmentSizeInBytes )
               else:
                  unitStr = txQNonEctParam.unit
               nonEctStr = '%18s   %18s  %12s  %20s' % ( txQNonEctParam.minThreshold,
                                                      txQNonEctParam.maxThreshold,
                                                      txQNonEctParam.maxDroprate,
                                                      unitStr )
               if nonEctWeightSupported:
                  nonEctStr += "%10s" % ( txQNonEctParam.weight )
            else:
               nonEctStr = '%18s   %18s  %12s  %20s' % ( '-', '-', '-', '-' )
               if nonEctWeightSupported:
                  nonEctStr += "%10s" % ( '-' )
            print( '   %8s    %s' % ( txQueueNonEct.txQueue, nonEctStr ) )
         print()

class TxQueueDropThresholdsModel( Model ):
   txQueue = Str( help="Name of Egress Queue" )
   dropThresholds = Dict( keyType=int, valueType=int,
                          help="Drop Thresholds per drop precedence in percent" )

class IntfDropThresholdsModel( Model ):
   txQueueDropThresholdsList = List(
      valueType=TxQueueDropThresholdsModel,
      help="List of Transmit Queue Drop Threshold Settings" )

class IntfDropThresholdsCollectionModel( Model ):
   dropThresholdsSupported = Bool( help="Drop Precedence thresholds supported" )
   intfDropThresholdsCollection = Dict(
      keyType=Interface, valueType=IntfDropThresholdsModel,
      help="A mapping of an interface to its drop thresholds" )

   def renderHeader( self ):
      print()
      header1 = '   Tx-Queue        Drop Precedence       Drop Thresholds '
      header2 = '                                           ( percent )   '
      header3 = '---------------------------------------------------------'

      print( header1 )
      print( header2 )
      print( header3 )

   def renderTable( self, txQueueDropThresholds ):
      txQDropThresholds = txQueueDropThresholds.dropThresholds
      for dp in sorted( txQDropThresholds ):
         dpStr = '%15s   %15s' % ( dp, txQDropThresholds[ dp ] )
         print( '   %8s    %s' % ( txQueueDropThresholds.txQueue, dpStr ) )

   def renderFooter( self ):
      print()

   def render( self ):
      if not self.dropThresholdsSupported:
         return

      for intf in Arnet.sortIntf( self.intfDropThresholdsCollection ):
         intfDropThresholds = self.intfDropThresholdsCollection[ intf ]
         if not intfDropThresholds.txQueueDropThresholdsList:
            continue
         print( '%s:' % intf )
         self.renderHeader()
         for txQueueDropThreshold in intfDropThresholds.txQueueDropThresholdsList:
            self.renderTable( txQueueDropThreshold )
         self.renderFooter()

class DscpRewriteMapModel( Model ):
   tcToDscpMap = Dict( keyType=int, valueType=int,
                       help="Traffic-Class to DSCP mapping " )
   # map is of format ( tc-%d dp-%d ) -> dscp
   tcDpToDscpMap = Dict( keyType=str, valueType=int,
                       help="Traffic-Class, Drop-Precedence to DSCP mapping " )

   def getTcList( self ):
      tcList = {}
      for key in self.tcDpToDscpMap:
         tc = int( key.split()[ 0 ].split( '-' )[ 1 ] )
         tcList[ tc ] = True
      return tcList

   def renderTcToDscpMap( self, space='' ):
      lines = [ space for _ in range( 3 ) ]
      lines[ 0 ] += 'TC:   '
      lines[ 1 ] += '-------'
      lines[ 2 ] += 'DSCP: '
      for tc in self.tcToDscpMap:
         lines[ 0 ] += '%3d' % tc
         lines[ 1 ] += '---'
         lines[ 2 ] += '%3d' % self.tcToDscpMap[ tc ]
      for line in lines:
         print( line )
      print( '' )

   def renderTcDpToDscpMap( self, space='' ):
      tcList = sorted( self.getTcList( ) )
      dpList = sorted( tcDpToDscpSupportedDpValues( None, dpSupported=True ) )
      lines = [ space for _ in range( 2 + len( dpList ) ) ]
      lines[ 0 ] += 'DP : TC :'
      lines[ 1 ] += '----------'
      for dp in dpList:
         lines[ dp + 2 ] += '%2d :     ' % dp
      for tc in tcList:
         lines[ 0 ] += '%3d' % tc
         lines[ 1 ] += '---'
         for dp in dpList:
            key = "tc-%d dp-%d" % ( tc, dp )
            lines[ dp + 2 ] += '%3s' % self.tcDpToDscpMap.get( key, '-' )
      for line in lines:
         print( line )
      print( '' )

   def render( self ):
      if self.tcDpToDscpMap:
         self.renderTcDpToDscpMap()
      else:
         self.renderTcToDscpMap()

class DscpRewriteMapAllModel( Model ):
   dscpRewriteMaps = Dict( keyType=str, valueType=DscpRewriteMapModel,
                           help="A mapping between Dscp RewriteMap names \
                           and Dscp RewriteMap" )

   def render( self ):
      for mapName in sorted( self.dscpRewriteMaps.keys() ):
         dscpRewriteMap = self.dscpRewriteMaps[ mapName ]
         print( " MapName : %s " % ( mapName ) )
         dscpRewriteMap.render()

class TcDpToExpMapModel( Model ):
   # map is of format ( tc-%d dp-%d ) -> exp
   tcDpToExpMap = Dict( keyType=str, valueType=int,
                       help="Traffic-Class, Drop-Precedence to EXP mapping " )
   
   def getTcList( self ):
      tcList = {}
      for key in self.tcDpToExpMap:
         tc = int( key.split()[ 0 ].split( '-' )[ 1 ] )
         tcList[ tc ] = True
      return tcList
 
   def renderTcDpToExpMap( self, space='' ):
      tcList = sorted( self.getTcList( ) )
      dpList = tcDpToExpSupportedDpValues()
      lines = [ space for _ in range( 2 + len( dpList ) ) ]
      lines[ 0 ] += 'DP : TC :'
      lines[ 1 ] += '----------'
      for dp in dpList:
         lines[ dp + 2 ] += '%2d :     ' % dp
      for tc in tcList:
         lines[ 0 ] += '%3d' % tc
         lines[ 1 ] += '---'
         for dp in dpList:
            key = "tc-%d dp-%d" % ( tc, dp )
            lines[ dp + 2 ] += f"{self.tcDpToExpMap.get( key, '-' ):>3}"
      for line in lines:
         print( line )
      print( '' )

   def render( self ):
      self.renderTcDpToExpMap()
         
class TcDpToExpMapAllModel( Model ):
   tcDpToExpMaps = Dict( keyType=str, valueType=TcDpToExpMapModel,
                           help="A mapping between TcDpToExp Map names \
                           and TcDpToExp Maps" )
   def render( self ):
      for mapName in sorted( self.tcDpToExpMaps ):
         tcDpToExpMap = self.tcDpToExpMaps[ mapName ]
         print( f" MapName : {mapName} " )
         tcDpToExpMap.render()

def printFormattedArray( col1, col2, low, high, array ):
   # pylint: disable-next=consider-using-ternary
   hlen = len( col1 ) > len( col2 ) and len( col1 ) or len( col2 )
   space = '     '
   print( space + col1, end='' )
   for index in range( low, high + 1 ):
      sys.stdout.softspace = 0
      print( '%3d' % index, end='' )
   print()
   print( space + '-' * hlen, end='' )
   for index in range( low, high + 1 ):
      sys.stdout.softspace = 0
      print( '---', end='' )
   print()
   print( space + col2, end='' )
   for index in range( low, high + 1 ):
      sys.stdout.softspace = 0
      if array[ index ] == -1:
         print( '  -', end='' )
      else:
         print( '%3d' % ( array[ index ] ), end='' )
   print( '\n' )

def printArrayDecFormat( array, highIndex ):
   space5 = ' ' * 5
   space3 = ' ' * 3
   print( space5 + 'd1 :  d2 %s' % ( "  ".join( [ str( i ) for i in
                                                    range( 10 ) ] ) ) )
   print( space5 + '--------------------------------------' )

   for i in range( len( array ) // 10 + 1 ):
      print( space5 + ' %d :' % i + space3, end='' )
      for j in range( 10 ):
         sys.stdout.softspace = 0
         if i * 10 + j > highIndex:
            break
         if array[ i * 10 + j ] == INVALID:
            print( '  -' )
         else:
            print( '%3d' % array[ i * 10 + j ], end='' )
      print()
   print()

class CosToTcProfileModel( Model ):
   cosToTcMap = Dict( keyType=int, valueType=int,
                      help="CoS to Traffic-Class mapping" )

   def render( self ):
      printFormattedArray( 'cos:', 'tc: ', 0, 7, self.cosToTcMap )

class CosToTcProfileAllModel( Model ):
   cosToTcProfiles = Dict(
      keyType=str, valueType=CosToTcProfileModel,
      help="CoS to Traffic-Class profiles, indexed by their name" )

   def render( self ):
      for profileName in sorted( self.cosToTcProfiles.keys() ):
         cosToTcProfile = self.cosToTcProfiles[ profileName ]
         print( ' CoS-TC map: %s ' % ( profileName ) )
         cosToTcProfile.render()

class ExpToTcProfileModel( Model ):
   expToTcMap = Dict( keyType=int, valueType=int,
                     help="EXP to Traffic-Class mapping" )

   def render( self ):
      printFormattedArray( 'exp:', 'tc: ', 0, 7, self.expToTcMap )

class ExpToTcProfileAllModel( Model ):
   expToTcProfiles = Dict(
      keyType=str, valueType=ExpToTcProfileModel,
      help="EXP to Traffic-Class profiles, indexed by their name" )

   def render( self ):
      for profileName in sorted( self.expToTcProfiles.keys() ):
         expToTcProfile = self.expToTcProfiles[ profileName ]
         print( ' EXP-TC map: %s ' % ( profileName ) )
         expToTcProfile.render()

class DscpToTcMapModel( Model ):
   dscpToTcNamedMap = Dict( keyType=int, valueType=int,
                           help="DSCP To Traffic-Class mapping" )

   def render( self ):
      printArrayDecFormat( self.dscpToTcNamedMap, 63 )

class DscpToTcMapAllModel( Model ):
   dscpToTcNamedMaps = Dict(
      keyType=str, valueType=DscpToTcMapModel,
      help="DSCP To Traffic-Class maps, indexed by their name" )

   def render( self ):
      for mapName in sorted( self.dscpToTcNamedMaps ):
         dscpToTcMapModel = self.dscpToTcNamedMaps[ mapName ]
         print( '   DSCP to TC map: %s ' % ( mapName ) )
         dscpToTcMapModel.render()

class TcToCosMapModel( Model ):
   tcToCosNamedMap = Dict( keyType=str, valueType=int,
                           help="Traffic-Class to CoS mapping" )

   def printTcToCosNamedMap( self, col1, col2, col3, low, maxTc, maxDp, array ):
      hlen = max( len( col1 ), max( len( col2 ), len( col3 ) ) )
      space = '     '
      line = [ '' for _ in range( 4 ) ]
      line[ 0 ] += space + str( col1 ) + ' ' * ( hlen - len( col1 ) )
      line[ 1 ] += space + str( col2 ) + ' ' * ( hlen - len( col2 ) )
      line[ 2 ] += space + '-' * hlen
      line[ 3 ] += space + str( col3 ) + ' ' * ( hlen - len( col3 ) )
      for index1 in range( low, maxTc + 1 ):
         for index2 in range( low, maxDp + 1 ):
            sys.stdout.softspace = 0
            line[ 0 ] += '%3d' % index1
            line[ 1 ] += '%3d' % index2
            line[ 2 ] += '---'
            tcDpIndex = 'tc-%d dp-%d' % ( index1, index2 )
            line[ 3 ] += '%3s' % array.get( tcDpIndex, '-' )
      print( '\n'.join( line ) )
      print()

   def render( self ):
      self.printTcToCosNamedMap( "tc:", "dp:", "cos:", 0, 7, 2,
                                 self.tcToCosNamedMap )

class TcToCosMapAllModel( Model ):
   tcToCosNamedMaps = Dict(
      keyType=str, valueType=TcToCosMapModel,
      help="Traffic-Class to CoS profiles, indexed by their name" )

   def render( self ):
      for mapName in sorted( self.tcToCosNamedMaps ):
         tcToCosNamedMap = self.tcToCosNamedMaps[ mapName ]
         print( ' TC to CoS map: %s ' % ( mapName ) )
         tcToCosNamedMap.render()

class GlobalQosMapsModel( Model ):
   numTcSupported = Int( help="Number of Traffic Classes supported" )
   numTxQueueSupported = Int( help="Number of Transmit Queues supported" )
   numMulticastQueueSupported = Int( help="Number of Multicast Queues supported" )

   cosRewriteSupported = Bool( help="COS rewrite supported" )
   cosRewriteDisableSupported = Bool( help="COS rewrite disable supported" )
   dscpRewriteSupported = Bool( help="DSCP rewrite supported" )
   tcToDscpMapSupported = Bool( help="Traffic-Class to DSCP mapping supported" )
   dscpToDpMapSupported = Bool( help="DSCP to Drop-Precedence mapping supported" )
   dynamicExpMappingSupported = Bool( help="Configuration of EXP to "
                                           "Traffic-Class and Traffic-Class "
                                           "to EXP mapping supported" )

   dscpRewritePerInterfaceSupported = Bool(
      help="Configuration of Traffic-Class to DSCP Map on interface supported" )
   cosToTcProfilePerInterfaceSupported = Bool(
      help="Configuration of CoS to Traffic-Class Map on interface supported" )
   expToTcProfilePerInterfaceSupported = Bool(
      help="Configuration of EXP to Traffic-Class Map on interface supported" )
   tcToCosMapPerInterfaceSupported = Bool(
      help="Configuration of Traffic-Class to CoS Map on interface supported" )
   pwDecapDscpQosMapSupported = Bool(
      help="Configuration of DSCP To Traffic-Class Map on pseudowire supported" )
   dscpToTcMapPerInterfaceSupported = Bool(
      help="Configuration of DSCP To Traffic-Class Map on interface supported" )
   dscpToTcMapPerSubInterfaceSupported = Bool(
      help="Configuration of DSCP To Traffic-Class Map on subInterface supported" )
   expRewriteSupported = Bool(
      help="Configuration of Traffic-Class to Exp Map on interface supported" )

   cosRewriteEnabled = Bool( help="COS rewrite enabled" )
   dscpRewriteEnabled = Bool( help="DSCP rewrite enabled" )
   deiToDpMapSupported = Bool( help="DEI to Drop-Precedence mapping supported" )

   cosToTcMap = Dict( keyType=int, valueType=int,
                      help="COS to Traffic-Class mapping" )
   deiToDpMap = Dict( keyType=int, valueType=int,
                      help="DEI to Drop-Precedence mapping" )
   dscpToTcMap = Dict( keyType=int, valueType=int,
                       help="DSCP to Traffic-Class mapping" )
   dscpToDpMap = Dict( keyType=int, valueType=int,
                       help="DSCP to Drop-Precedence mapping",
                       optional=True )
   expToTcMap = Dict( keyType=int, valueType=int,
                      help="EXP to Traffic-Class mapping "
                           "when dynamic EXP mapping is supported",
                      optional=True )

   tcToCosMap = Dict( keyType=int, valueType=int,
                      help="Traffic-Class to COS mapping "
                         "when COS rewrite is supported",
                      optional=True )
   tcToDscpMap = Dict( keyType=int, valueType=int,
                       help="Traffic-Class to DSCP mapping "
                          "when DSCP rewrite is supported",
                       optional=True )
   tcToExpMap = Dict( keyType=int, valueType=int,
                      help="Traffic-Class to EXP mapping "
                            "when dynamic EXP mapping is supported",
                      optional=True )
   
   tcDpToExpMap = Dict( keyType=int, valueType=int, 
                        help="Traffic-Class and Drop-Prededence to EXP mapping " \
                           "when EXP rewrite is supported",
                        optional=True )   

   tcToTxQueueMap = Dict( keyType=int, valueType=int,
                          help="Traffic-Class to tx-queue mapping "
                             "when unicast queues are supported",
                          optional=True )
   tcToMcTxQueueMap = Dict( keyType=int, valueType=int,
                            help="Traffic-Class to mc-tx-queue mapping "
                               "when multicast queues are supported",
                            optional=True )

   tcToPriorityGroupMapSupported = Bool( help="Traffic-Class to Priority Group "
                                            "supported" )
   tcToPriorityGroupMap = Dict( keyType=int, valueType=int,
                          help="Traffic-Class to PriorityGroup mapping ",
                          optional=True )

   cosToTcProfileAllModel = Submodel(
      valueType=CosToTcProfileAllModel,
      help="User configured COS to Traffic-Class profiles", optional=True )

   expToTcProfileAllModel = Submodel(
      valueType=ExpToTcProfileAllModel,
      help="User configured EXP to Traffic-Class profiles", optional=True )

   tcToCosMapAllModel = Submodel(
      valueType=TcToCosMapAllModel,
      help="User configured Traffic-Class to CoS profiles", optional=True )
   
   tcDpToExpMapAllModel = Submodel(
      valueType=TcDpToExpMapAllModel,
      help="User configured Traffic-Class and Drop-Precedence to EXP profiles",
      optional=True )   

   dscpRewriteMapAllModel = Submodel(
      valueType=DscpRewriteMapAllModel,
      help="User configured Traffic-Class and Drop-Precedence to DSCP profiles",
      optional=True )

   dscpToTcMapAllModel = Submodel(
      valueType=DscpToTcMapAllModel,
      help="User configured Dscp to Traffic-Class maps", optional=True )

   def render( self ):
      print( '   Number of Traffic Classes supported: %s' % self.numTcSupported )
      if self.numTxQueueSupported:
         print( '   Number of Transmit Queues supported: %s' %
                self.numTxQueueSupported )
      if self.cosRewriteSupported and self.cosRewriteDisableSupported:
         print( '   Cos Rewrite:  ',
                'Enabled' if self.cosRewriteEnabled else 'Disabled' )
      if self.dscpRewriteSupported:
         print( '   Dscp Rewrite: ',
                'Enabled' if self.dscpRewriteEnabled else 'Disabled' )
      print()

      print( '   Cos-tc map:' )
      printFormattedArray( 'cos:', 'tc: ', 0, 7, self.cosToTcMap )

      if self.deiToDpMapSupported:
         print( '   DEI-DP map:' )
         printFormattedArray( 'dei:', 'dp: ', 0, 1, self.deiToDpMap )

      print( '   Dscp-tc map:' )
      printArrayDecFormat( self.dscpToTcMap, 63 )

      if self.dscpToDpMapSupported:
         print( '   Dscp-DP map:' )
         printArrayDecFormat( self.dscpToDpMap, 63 )

      if self.dynamicExpMappingSupported:
         print( '   Exp-tc map:' )
         printFormattedArray( 'exp:', 'tc: ', 0, 7, self.expToTcMap )

      if self.cosRewriteSupported:
         print( '   Tc-cos map:' )
         printFormattedArray( 'tc: ', 'cos:', 0, self.numTcSupported - 1,
                              self.tcToCosMap )

      if self.dscpRewriteSupported:
         print( '   Tc-dscp map:' )
         printFormattedArray( 'tc:  ', 'dscp:', 0, self.numTcSupported - 1,
                              self.tcToDscpMap )

      if self.dynamicExpMappingSupported:
         print( '   Tc-exp map:' )
         printFormattedArray( 'tc: ', 'exp:', 0, self.numTcSupported - 1,
                              self.tcToExpMap )

      if self.numTxQueueSupported:
         if self.numMulticastQueueSupported:
            print( '   Tc - uc-tx-queue map:' )
            printFormattedArray( 'tc:         ', 'uc-tx-queue:', 0,
                                 ( self.numTcSupported - 1 ),
                                 self.tcToTxQueueMap )
            print( '   Tc - mc-tx-queue map:' )
            printFormattedArray( 'tc:         ', 'mc-tx-queue:', 0,
                                 ( self.numTcSupported - 1 ),
                                 self.tcToMcTxQueueMap )
         else:
            print( '   Tc - tx-queue map:' )
            printFormattedArray( 'tc:      ', 'tx-queue:', 0,
                                 ( self.numTcSupported - 1 ),
                                 self.tcToTxQueueMap )

      if self.tcToPriorityGroupMapSupported:
         print( '   Tc - PriorityGroup map:' )
         printFormattedArray( 'tc:            ',
                              'priority-group:', 0,
                              ( self.numTcSupported - 1 ),
                              self.tcToPriorityGroupMap )

      if self.cosToTcProfilePerInterfaceSupported:
         cosToTcProfiles = self.cosToTcProfileAllModel.cosToTcProfiles
         profiles = sorted( cosToTcProfiles )
         if self.cosToTcProfileAllModel:
            for profileName in profiles:
               cosToTcProfile = cosToTcProfiles[ profileName ]
               print( '   CoS-TC map: %s ' % ( profileName ) )
               cosToTcProfile.render()

      if self.expToTcProfilePerInterfaceSupported:
         expToTcProfiles = self.expToTcProfileAllModel.expToTcProfiles
         profiles = sorted( expToTcProfiles )
         if self.expToTcProfileAllModel:
            for profileName in profiles:
               expToTcProfile = expToTcProfiles[ profileName ]
               print( '   EXP-TC map: %s ' % ( profileName ) )
               expToTcProfile.render()

      if self.tcToCosMapPerInterfaceSupported and self.tcToCosMapAllModel:
         self.tcToCosMapAllModel.render()

      if self.dscpToTcMapAllModel and \
         ( self.pwDecapDscpQosMapSupported or \
           self.dscpToTcMapPerInterfaceSupported or \
           self.dscpToTcMapPerSubInterfaceSupported ):
         self.dscpToTcMapAllModel.render()

      if self.dscpRewritePerInterfaceSupported and self.dscpRewriteMapAllModel:
         dscpRewriteMaps = self.dscpRewriteMapAllModel.dscpRewriteMaps
         maps = sorted( dscpRewriteMaps )
         for mapName in maps:
            dscpRewriteMap = dscpRewriteMaps[ mapName ]
            if dscpRewriteMap.tcDpToDscpMap:  # pylint: disable=superfluous-parens
               print( '   TC DP to DSCP map: %s ' % ( mapName ) )
               dscpRewriteMap.renderTcDpToDscpMap( space='     ' )
            else:
               print( '   TC to DSCP map: %s ' % ( mapName ) )
               printFormattedArray( 'TC:  ', 'DSCP:', 0,
                                 ( self.numTcSupported - 1 ),
                                 dscpRewriteMap.tcToDscpMap )
               
      if self.expRewriteSupported and self.tcDpToExpMapAllModel:
         tcDpToExpMaps = self.tcDpToExpMapAllModel.tcDpToExpMaps
         maps = sorted( tcDpToExpMaps )
         for mapName in maps:
            tcDpToExpMapModel = tcDpToExpMaps[ mapName ]
            if tcDpToExpMapModel.tcDpToExpMap:
               print(  f"   TC DP to EXP map: {mapName} " )
               tcDpToExpMapModel.renderTcDpToExpMap( space='     ' )    

class ClassMapAllModel( Model ):
   def append( self, cMapName, classMapModel ):
      self.classMaps[ cMapName ] = classMapModel

   def render( self ):
      for cMapName in ( sorted( self.classMaps,
                        key=lambda name: self.classMaps[ name ].classPrio ) ):
         cMap = self.classMaps[ cMapName ]
         matchCondition = QosLib.strFromEnum( cMap.matchCondition )
         output = '  Class-map: %s (%s)' % ( cMapName, matchCondition )
         print( output )
         for accessGroupName in cMap.match:
            match = cMap.match[ accessGroupName ]
            if match.matchRule == 'matchL2Params':
               output = '    Match:'
               l2Params = match.l2ParamsValue
               if l2Params.vlanValue:
                  if l2Params.vlanMaskValid:
                     output += ' %s %s %#5.3x' % \
                               ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                                 l2Params.vlanValue, l2Params.vlanMask )
                  else:
                     output += ' %s %s' % \
                               ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                                 l2Params.vlanValue )
               if l2Params.innerVlanValue:
                  if l2Params.innerVlanMaskValid:
                     output += ' %s %s %#5.3x' % (
                        QosLib.matchFromEnum( match.matchRule )[ 2 ],
                        l2Params.innerVlanValue, l2Params.innerVlanMask )
                  else:
                     output += ' %s %s' % (
                        QosLib.matchFromEnum( match.matchRule )[ 2 ],
                        l2Params.innerVlanValue )
               if l2Params.cosValue:
                  output += ' %s %s' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 1 ],
                              l2Params.cosValue )
               if l2Params.innerCosValue:
                  output += ' %s %s' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 3 ],
                              l2Params.innerCosValue )
               if l2Params.deiValue:
                  output += ' %s %s' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 4 ],
                              l2Params.deiValue )
            elif match.matchRule == 'matchDscpEcn':
               ecnName = ""
               if match.matchEcnValue != AclCliLib.ecnNameFromValue( dontCareEcn ):
                  ecnName = " %s %s" % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 1 ],
                              match.matchEcnValue )
               if accessGroupName in AclCliLib.ecnAclNames:
                  # only ecn case
                  output = '    Match:%s' % ( ecnName )
               else:
                  output = '    Match: %s %s%s' % \
                            ( QosLib.matchFromEnum( match.matchRule )[ 0 ],
                              accessGroupName, ecnName )
            elif match.matchRule == 'matchMplsTrafficClass':
               mplsTrafficClassStr = 'mpls traffic-class'
               output = '    Match: %s %s' % \
                   ( mplsTrafficClassStr, accessGroupName )
            else:
               output = '    Match: %s name %s' % \
                   ( QosLib.matchFromEnum( match.matchRule ), accessGroupName )
            print( output )

   class ClassMapModel( Model ):
      classPrio = Int( help="Class priority" )
      matchCondition = Enum( values=( 'matchConditionAny', 'unspecified' ),
                             help="Match condition" )

      class Match( Model ):
         matchRule = Enum( values=( 'matchIpAccessGroup',
                                    'matchIpv6AccessGroup',
                                    'matchL2Params',
                                    'matchDscpEcn',
                                    'matchMplsTrafficClass',
                                    'matchMacAccessGroup' ),
                           help='Matching rule for the type' )
         matchEcnValue = Enum( values=list( AclCliLib.ecnAclNames ),
                               help='ECN value configured',
                               optional=True )
         matchDscpNameValid = Bool( help="Flag to identify if string value of DSCP "
                                    "is used or numeric value",
                                    optional=True )
         dscpValue = Int( help="DSCP equivalent value in digits",
                          optional=True )
         l2ParamsValue = Submodel( valueType=MatchL2Params,
                                   help="L2 Params values configured",
                                   optional=True )

      match = Dict( valueType=Match, help='Matching access groups',
                    optional=True )

   classMaps = Dict( keyType=str, valueType=ClassMapModel,
                         help="List of all class maps" )

class IntfTrustModeParametersModel( Model ):
   operationalTrustMode = Enum( values=( 'DSCP', 'COS', 'UNTRUSTED' ),
                                help="Operational trust mode" )
   configuredTrustMode = Enum( values=( 'DSCP', 'COS', 'UNTRUSTED', 'DEFAULT' ),
                               help="Configured trust mode" )

class IntfTrustModeModel( Model ):
   intfTrustModeParameters = Submodel(
      valueType=IntfTrustModeParametersModel,
      help="Trust Mode Parameters.",
      optional=True )

class IntfTrustModeCollectionModel( Model ):

   intfTrustModeCollection = Dict( keyType=Interface, valueType=IntfTrustModeModel,
                               help="Interface trust mode" )

   def render( self ):
      col1Len = 31
      col2Len = len( 'Operational' ) + 1 + len( 'TrustMode' ) + 1
      col3Len = len( 'Configured' )
      tableHdr1 = 'Port'.ljust( col1Len ) + ' ' * ( len( 'Operational' ) + 1 ) + \
          'Trust Mode'
      tableHdr2 = ' ' * col1Len + 'Operational'.ljust( col2Len ) + \
          'Configured'
      tableHdrHyphen = '-' * ( col1Len + col2Len + col3Len )
      print( tableHdr1 )
      print( tableHdr2 )
      print( tableHdrHyphen )

      for intf in Arnet.sortIntf( self.intfTrustModeCollection ):
         intfTrustMode = self.intfTrustModeCollection[ intf ]
         intfTrustModeParameters = intfTrustMode.intfTrustModeParameters
         if not intfTrustModeParameters:
            operTrustModeStr = '-'
            cfgTrustModeStr = '-'
         else:
            operTrustModeStr = intfTrustModeParameters.operationalTrustMode
            cfgTrustModeStr = intfTrustModeParameters.configuredTrustMode
         print( intf.ljust( col1Len ) + operTrustModeStr.ljust( col2Len ) +
                cfgTrustModeStr.ljust( col2Len ) )
      print()

class IntfDscpPreserveCollectionModel( Model ):
   interfaces = List( valueType=Interface,
                      help="Interfaces with DSCP Preserve enabled" )

   def render( self ):
      headings = "Interface"
      table = createTable( ( headings, ) )
      for intf in Arnet.sortIntf( self.interfaces ):
         table.newRow( intf )

      f1 = Format( justify="left" )
      table.formatColumns( f1 )
      print( '\n' + table.output() )

#
# -----------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) CLI Model Render Functions
# -----------------------------------------------------------------------
# This section defines the render functions for the priority flow control
# CLI models.
#
# Utility routines:
# . prioritiesToStr
# . printColumns
#
# Sub-Model Render functions:
# . pfcStatusModelRender
# . pfcWatchdogStatusModelRender
# . pfcIntfStatusModelRender
# . pfcIntfCountersModelRender
# . pfcIntfDetailedCountersModelRender
# . pfcIntfHistoryCountersModelRender
# . pfcWatchdogIntfCountersModelRender
#
# Model Render functions:
# . pfcFabricIntfStatusModelRender
# . pfcIntfRangeStatusModelRender
# . pfcIntfRangeCountersModelRender
# . pfcIntfRangeDetailedCountersModelRender
# . pfcIntfRangeHistoryCountersModelRender
# . pfcWatchdogIntfRangeCountersModelRender
# . pfcIntfRangeModelRender
# -----------------------------------------------------------------------
#


#
# Number of priority levels supported by priority flow control.
#
numPriority = 8

#
# Converts a list of priorities into a string.
#
def prioritiesToStr( priorities, ignoreMissing=True ):
   if ignoreMissing:
      return " ".join( str( p ) for p in sorted( priorities ) )

   return "".join( str( p ) if p in priorities else " "
                     for p in range( numPriority ) )

#
# Prints a list of fields as a column-formatted string.
#
def printColumns( fields, columnWidths ):
   string = ""

   for i in range( len( fields ) ):  # pylint: disable=consider-using-enumerate
      if columnWidths[ i ] < 0:
         # Left justification.
         string += fields[ i ].ljust( -columnWidths[ i ] )
      else:
         # Right justification.
         string += fields[ i ].rjust( columnWidths[ i ] )

   print( string )

#
# Render function for PfcStatusModel.
#
# Note: The PFC Watchdog status is printed inline, and this requires that
#       PfcWatchdogStatusModel be also passed to this function.
#
def pfcStatusModelRender( model, watchdogModel, summary=False ):
   if model.supported:
      print( "The hardware supports PFC on priorities", end=' ' )
      print( prioritiesToStr( model.priorities ) )

      rxPfcPrioritiesHonored = prioritiesToStr( model.rxPfcPrioritiesHonored )
      assert rxPfcPrioritiesHonored
      print( "PFC receive processing is enabled on priorities", end=' ' )
      print( prioritiesToStr( model.rxPfcPrioritiesHonored ) )

      #
      # Print the Watchdog status.
      #
      pfcWatchdogStatusModelRender( watchdogModel, summary )
   else:
      print( "The hardware does not support priority flow control." )

   print( "Global PFC :", end=' ' )
   print( "Enabled" if model.enabled else "Disabled" )
   print()

#
# Render function for PfcWatchdogStatusModel.
#
def pfcWatchdogStatusModelRender( model, summary ):

   # Trivial case: Nothing to do.
   if not model.supported:
      return

   # Only summarize.
   if summary:
      print( "The PFC watchdog default timeout is ", end=' ' )
      print( model.timer.timeout )
      return

   # The unit of time.
   unit = "second(s)"

   # Timeout.
   if not model.timer.timeout:
      print( "! Please configure the timeout "
             "for watchdog to be operationally active" )

   print( "The PFC watchdog timeout is", model.timer.timeout, unit, end=' ' )
   if not model.timer.timeout:
      print( "(default)" )
   else:
      print()

   # Recovery time.
   print( "The PFC watchdog recovery-time is",
          model.timer.recoveryTime, unit, end=' ' )
   if not model.timer.recoveryTime:
      print( "(auto) (default)" )
   else:
      print( "(forced)" if model.timer.forcedRecovery else "(auto)" )

   # Polling interval.
   print( "The PFC watchdog polling-interval is",
          model.timer.operPollInterval, unit, end=' ' )
   if not model.timer.operPollInterval:
      print( "(default)" )
   else:
      print()

      #
      # Informational note on the value of the polling interval.
      #
      pollIntervalStr = \
         "  (Configured value %s %s" % ( str( model.timer.pollInterval ), unit )

      if model.timer.pollInterval == model.timer.operPollInterval:
         pass

      elif ( model.timer.timeout and model.timer.recoveryTime and
             ( not model.timer.forcedRecovery ) and
             ( model.timer.recoveryTime < model.timer.timeout ) and
             ( model.timer.recoveryTime < ( 2 * model.timer.pollInterval ) ) ):
         print( pollIntervalStr, "is greater than half of recovery-time)" )

      elif model.timer.timeout < ( 2 * model.timer.pollInterval ):
         print( pollIntervalStr, "is greater than half of timeout)" )

   # Action.
   print( "The PFC watchdog action is", end=' ' )
   print( model.action if model.action else "invalid" )

   # Override action.
   print( "The PFC watchdog override action drop is " + str(
         model.overrideAction ).lower() )

   # Non-disruptive priorities.
   if model.nDPriorities.supported:
      print( "The PFC watchdog non-disruptive priorities are", end=' ' )

      if model.nDPriorities.priorities:
         print( prioritiesToStr( model.nDPriorities.priorities ) )
      else:
         print( "not configured" )

      print( "The PFC watchdog non-disruptive action is", end=' ' )
      if model.nDAction == "invalid":
         print( "not configured" )
      else:
         print( model.nDAction )

      print( "The PFC watchdog port non-disruptive-only is " + str(
         model.forcedNd ).lower() )

   # Hardware Monitored priorities
   if model.hwMonitoredPri is not None:
      print( "The PFC watchdog hardware monitored priorities are", end=' ' )

      if model.hwMonitoredPri:
         print( prioritiesToStr( model.hwMonitoredPri ) )
      else:
         print( "not configured" )

#
# Render function for PfcIntfStatusModel.
#
def pfcIntfStatusModelRender( intfName, model, columnWidths ):

   # Status.
   status = "E " if model.enabled else "D "
   status += "A " if model.active else "- "
   status += "W" if model.watchdogActive else "-"

   # Priorities.
   priorities = prioritiesToStr( model.priorities, ignoreMissing=False )

   #
   # Watchdog attributes.
   #
   action = model.watchdogAction if model.watchdogAction else "-"

   timeout = str( model.watchdogTimer.timeout ) + "s" if model.watchdogTimer else "-"

   if model.watchdogTimer:
      recovery = str( model.watchdogTimer.recoveryTime ) + "s / "
      recovery += "forced" if model.watchdogTimer.forcedRecovery else "auto"
   else:
      recovery = "- / -"

   if model.watchdogTimer:
      polling = str( model.watchdogTimer.pollInterval ) + "s / "
      polling += str( model.watchdogTimer.operPollInterval ) + "s" \
                  if model.watchdogTimer.operPollInterval is not None else "-"
   else:
      polling = "- / -"

   # Optional note.
   note = model.note if model.note else ""

   printColumns( [ intfName, status, priorities, action, timeout, recovery,
                   polling, note ], columnWidths )

#
# Render function for PfcFrameCountersModel for an interface.
#
def pfcIntfCountersModelRender( intfName, model, columnWidths ):
   rxFrames = str( model.rxFrames )
   txFrames = str( model.txFrames )

   printColumns( [ intfName, rxFrames, txFrames ], columnWidths )

#
# Render function for PfcIntfDetailedCountersModel.
#
def pfcIntfDetailedCountersModelRender( intfName, model, counterName, columnWidths ):
   priorityCounters = []

   for p in range( numPriority ):
      framesCount = getattr( model.priorities[ p ], counterName )
      if framesCount is None:
         priorityCounters.append( 'n/a' )
      else:
         priorityCounters.append( str( framesCount ) )

   printColumns( [ intfName ] + priorityCounters, columnWidths )

#
# Render function for PfcIntfHistoryCountersModel.
#
def pfcIntfHistoryCountersModelRender( intfName, model, columnWidths ):
   counterSequence = [ 'totalRxPfcOnTransitions', 'maxRxPfcOnTimeNs',
                       'maxRxPfcOnTimestamp', 'totalRxPfcOnTimeNs' ]

   for p in range( numPriority ):
      prioStr = 'PFC' + str( p )
      priorityCounters = []
      for counter in counterSequence:
         counterVal = getattr( model.priorities[ p ], counter )
         if counterVal is None:
            priorityCounters.append( 'n/a' )
         else:
            if counter == 'maxRxPfcOnTimestamp':
               priorityCounters.append( utcTimestampToStr( counterVal ) )
            else:
               priorityCounters.append( f'{counterVal:,}' )
      priorityCounters.insert( 2, '' )
      printColumns( [ intfName, prioStr ] + priorityCounters, columnWidths )

#
# Render function for PfcWatchdogIntfCountersModel.
#
def pfcWatchdogIntfCountersModelRender( intfName, model, columnWidths ):

   for txQueueId in sorted( model.txQueues.keys() ):
      txQueueModel = model.txQueues[ txQueueId ]

      prefix = "UC" if txQueueModel.queueType == 'unicast' else "MC"

      txQueueName = prefix + "%u" % txQueueId

      stuckCount = str( txQueueModel.stuckCount )
      recoveryCount = str( txQueueModel.recoveryCount )
      lastStuckTimestamp = \
         utcTimestampToStr( txQueueModel.lastStuckTimestampUtc )
      lastRecoveryTimestamp = \
         utcTimestampToStr( txQueueModel.lastRecoveryTimestampUtc )
      printColumns( [ intfName, txQueueName, stuckCount, recoveryCount,
                      lastStuckTimestamp, lastRecoveryTimestamp ],
                    columnWidths )

#
# Render function for PfcWatchdogIntfEgressDropCountersModel.egressCounters
#
def pfcWatchdogIntfEgressDropCountersModelRender( intfName, model, columnWidths ):
   for txQueueId, egressDropCountersModel in sorted( model.egressCounters.items() ):
      # egressDropCountersModel is None for unsupported platforms
      prefix = 'UC' if ( egressDropCountersModel.queueType is None or
                         egressDropCountersModel.queueType == 'unicast' ) else 'MC'
      txQueueName = "%s%u" % ( prefix, txQueueId )

      row = [ intfName, txQueueName ]
      if egressDropCountersModel.outAggregateDrops is not None:
         drops = ( egressDropCountersModel.outAggregateDrops,
                   egressDropCountersModel.outAggregateByteDrops,
                   egressDropCountersModel.outCurrentDrops,
                   egressDropCountersModel.outCurrentByteDrops )

         row += [ str( drop ) for drop in drops ]
      else:
         # Return n/a for platforms which don't support egress drop counters
         row += [ 'n/a' for _ in range( 4 ) ]

      printColumns( row, columnWidths )

#
# Render function for PfcWatchdogIntfDropCountersModel.ingressCounters
#
def pfcWatchdogIntfIngressDropCountersModelRender( intfName, model, columnWidths ):
   for tc, ingressDropCountersModel in sorted( model.ingressCounters.items() ):
      row = [ intfName, str( tc ) ]
      if ingressDropCountersModel.aggregateDrops is not None:
         drops = ( ingressDropCountersModel.aggregateDrops,
                   ingressDropCountersModel.aggregateByteDrops,
                   ingressDropCountersModel.currentDrops,
                   ingressDropCountersModel.currentByteDrops )

         row += [ str( drop ) for drop in drops ]
      else:
         # Return n/a for platforms which don't support ingress drop counters
         row += [ 'n/a' for _ in range( 4 ) ]

      printColumns( row, columnWidths )

#
# Render function for PfcFabricIntfStatusModel.
#
def pfcFabricIntfStatusModelRender( model ):
   #
   # Print the global PFC [Watchdog] status.
   #
   pfcStatusModelRender( model.status, model.watchdogStatus, summary=True )

   # Column format: Negative widths for left-justification.
   columnWidths = [ -8, -8, -64 ]

   # Heading.
   printColumns( [ "Port", "Enabled", "Priorities" ], columnWidths )

   #
   # Print PFC status for the Fabric interface.
   #
   enabled = "Yes" if model.enabled else "No"
   priorities = prioritiesToStr( model.priorities, ignoreMissing=False )

   printColumns( [ model.intfName, enabled, priorities ], columnWidths )

#
# Render function for PfcIntfRangeStatusModel.
#
def pfcIntfRangeStatusModelRender( model ):

   intfNames = Arnet.sortIntf( model.interfaceStatuses )

   # Trivial case: Nothing to do.
   if not intfNames:
      return

   #
   # Print the global PFC [Watchdog] status.
   #
   pfcStatusModelRender( model.status, model.watchdogStatus )

   # Legend.
   print( "E: PFC Enabled, D: PFC Disabled, A: PFC Active, W: PFC Watchdog Active" )

   # Column formats: Negative width for left-justification.
   columnWidths1 = [ -11, -8, -12, -12, -9, -16, -16, -53 ]
   columnWidths2 = [ -50, -16, -16, -53 ]

   # Headings.
   printColumns( [ "Port", "Status", "Priorities", "Action", "Timeout", "Recovery",
                   "Polling", "Note" ], columnWidths1 )
   printColumns( [ "", "Interval/Mode", "Config/Oper", "" ], columnWidths2 )

   print( '-' * 87 )

   #
   # Print PFC status for each interface.
   #
   for intfName in intfNames:
      intfShortname = IntfMode.getShortname( intfName )
      pfcIntfStatusModelRender( intfShortname,
                                model.interfaceStatuses[ intfName ],
                                columnWidths1 )

#
# Render function for PfcIntfRangeCountersModel.
#
def pfcIntfRangeCountersModelRender( model ):

   intfNames = Arnet.sortIntf( model.interfaceCounters )

   # Trivial case: Nothing to do.
   if not intfNames:
      return

   # Column format: Negative width for left-justification.
   columnWidths = [ -10, 16, 16 ]

   # Heading.
   printColumns( [ "Port", "RxPfc", "TxPfc" ], columnWidths )

   #
   # Print PFC counters for each interface.
   #
   for intfName in intfNames:
      intfShortname = IntfMode.getShortname( intfName )
      pfcIntfCountersModelRender( intfShortname, model.interfaceCounters[ intfName ],
                                  columnWidths )

#
# Render function for PfcIntfRangeDetailedCountersModel.
#
def pfcIntfRangeDetailedCountersModelRender( model ):

   intfNames = Arnet.sortIntf( model.interfaces )

   # Trivial case: Nothing to do.
   if not intfNames:
      return

   # Column format: Negative width for left-justification.
   columnWidths = [ -10 ] + ( [ 13 ] * numPriority )

   # PFC priority names.
   priorityNames = [ "PFC%u" % p for p in range( numPriority ) ]

   # A mapping of direction to counter name.
   directions = { "Rx": "rxFrames", "Tx": "txFrames" }

   #
   # Print PFC counters for each direction.
   #
   for d in sorted( directions.keys() ):

      # Heading.
      printColumns( [ "Port %s" % d ] + priorityNames, columnWidths )

      #
      # Print PFC counters for each interface.
      #
      for intfName in intfNames:
         intfShortname = IntfMode.getShortname( intfName )
         pfcIntfDetailedCountersModelRender( intfShortname,
                                             model.interfaces[ intfName ],
                                             directions[ d ],
                                             columnWidths )

      # Delimiter for the next direction.
      print()

   # Warning if missing counters
   def _maybePrintNote():
      for intfModel in model.interfaces.values():
         for prio in intfModel.priorities.values():
            for direction in directions.values():
               if getattr( prio, direction ) is None:
                  print( "Note: Interfaces printing n/a counter values lack "
                          "hardware support for these counters" )
                  return
   _maybePrintNote()

#
# Render function for PfcIntfRangeHistoryCountersModel.
#
def pfcIntfRangeHistoryCountersModelRender( model ):

   # Trivial case: Nothing to do.
   if not model.interfaces:
      return

   intfNames = Arnet.sortIntf( model.interfaces )

   # Column format: Negative width for left-justification.
   columnWidths = [ -11, 10, 21, 25, 2, -23, 25 ]

   # Heading.
   print( "Last Clear Timestamp: %s" %
          utcTimestampToStr( model.lastClearTimestamp ) )
   printColumns( [ "", "", "RX PFC Off", "Max RX PFC", "", "Max RX PFC",
                   "Total RX PFC" ],
                 columnWidths )

   printColumns( [ "Interface", "Priority", "to On Count", "On Time (ns)", "",
                   "Timestamp", "On Time (ns)" ],
                 columnWidths )

   print( "-" * 11 + "  " + "-" * 8 + "  " + "-" * 19 + "  " + "-" * 23 + "  " +
          "-" * 21 + "  " + "-" * 25 )

   #
   # Print PFC counters for each interface.
   #
   for intfName in intfNames:
      intfShortname = IntfMode.getShortname( intfName )
      pfcIntfHistoryCountersModelRender( intfShortname,
                                         model.interfaces[ intfName ],
                                         columnWidths )

   # Final delimiter.
   print()

#
# Render function for PfcWatchdogIntfRangeCountersModel.
#
def pfcWatchdogIntfRangeCountersModelRender( model, cliOutput ):

   # Trivial case: Nothing to do.
   if not cliOutput:
      return

   intfNames = Arnet.sortIntf( model.interfaces )

   # Column format: Negative width for left-justification.
   columnWidths = [ -11, -3, 14, 14, 22, 22 ]

   # Heading.
   print( "Port       TxQ   Total Times   Total Times     Last Stuck Time  "
          "   Last Recovery Time" )
   print( "                 Stuck         Recovered" )
   print( "-------   ----   -----------   -----------  ---------------------"
          " ---------------------" )

   #
   # Print PFC Watchdog counters for each interface.
   #
   for intfName in intfNames:
      intfShortname = IntfMode.getShortname( intfName )
      pfcWatchdogIntfCountersModelRender( intfShortname,
                                          model.interfaces[ intfName ],
                                          columnWidths )

   # Final delimiter.
   print()

#
# Render function for PfcWatchdogIntfRangeCountersModel.
#
def pfcWatchdogIntfRangeDropCountersModelRender( model, cliOutput ):
   if not cliOutput:
      return
   # Column format: Negative width for left-justification.
   columnWidths = ( -11, -3, 20, 21, 18, 19 )
   template = "%%%ds" * len( columnWidths ) % columnWidths
   headings = ( 'Port', 'TxQ', 'Aggregate Pkts', 'Aggregate Bytes',
                'Current Pkts', 'Current Bytes' )
   print( "-" * 37, "Egress Drops", "-" * 41 )
   print( template % headings )
   colHeadLen1, colHeadLen2 = ( 7, 4 )
   print( '-' * colHeadLen1 + '   ' + '-' * colHeadLen2 + '      ' +
          '      '.join( [ '-' * len( heading ) for heading in headings[ 2 : ] ] ) )

   #
   # Print PFC Watchdog counters for each interface.
   #
   for intfName in Arnet.sortIntf( model.interfaces ):
      intfShortname = IntfMode.getShortname( intfName )
      pfcWatchdogIntfEgressDropCountersModelRender( intfShortname,
                                                    model.interfaces[ intfName ],
                                                    columnWidths )

   # Heading ingress counters.
   print()
   print( "-" * 37, "Ingress Drops", "-" * 40 )
   headings = ( 'Port', 'Tc', 'Aggregate Pkts', 'Aggregate Bytes',
                'Current Pkts', 'Current Bytes' )
   print( template % headings )
   print( '-' * colHeadLen1 + '   ' + '-' * colHeadLen2 + '      ' +
          '      '.join( [ '-' * len( heading ) for heading in headings[ 2 : ] ] ) )

   for intfName in Arnet.sortIntf( model.interfaces ):
      intfShortname = IntfMode.getShortname( intfName )
      pfcWatchdogIntfIngressDropCountersModelRender( intfShortname,
                                                     model.interfaces[ intfName ],
                                                     columnWidths )

#
# Render function for PfcIntfRangeModel.
#
# Note-1: The 'interfaceStatuses' and 'interfaceCounters' collections have
#         identical keys. Either collection can be used to check against
#         emptiness.
#
# Note-2: The attributes of PfcIntfRangeModel are a union of the attributes
#         of PfcIntfRangeStatusModel and PfcIntfRangeCountersModel. This
#         allows for the render functions of these latter models to be
#         invoked in a transparent fashion.
#
def pfcIntfRangeModelRender( model ):

   # Trivial case: Nothing to do.
   if not model.interfaceStatuses:
      return

   # Print PFC status.
   pfcIntfRangeStatusModelRender( model )

   print()

   # Print PFC counters.
   pfcIntfRangeCountersModelRender( model )


#
# -----------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) CLI Models
# -----------------------------------------------------------------------
# This section defines the CLI models (and sub-models) associated with
# the priority flow control show commands.
#
# Sub-Models:
# . PfcStatusModel
# . PfcWatchdogTimerModel
# . PfcWatchdogNDPrioritiesModel
# . PfcWatchdogStatusModel
# . PfcIntfStatusModel
# . PfcFrameCountersModel
# . PfcIntfDetailedCountersModel
# . PfcHistoryCountersModel
# . PfcIntfHistoryCountersModel
# . PfcWatchdogTxQueueCountersModel
# . PfcWatchdogIntfCountersModel
#
# Models:
# . PfcFabricIntfStatusModel
# . PfcIntfRangeStatusModel
# . PfcIntfRangeCountersModel
# . PfcIntfRangeDetailedCountersModel
# . PfcIntfRangeHistoryCountersModel
# . PfcWatchdogIntfRangeCountersModel
# . PfcIntfRangeModel
# -----------------------------------------------------------------------
#

#
# Model for global PFC status.
# Usage: Sub-Model.
#
class PfcStatusModel( Model ):
   supported = Bool( help="Indicates whether priority flow control is supported "
                          "by the hardware" )

   priorities = List( help="Priority values for which priority flow control is "
                           "supported by the hardware", valueType=int,
                      optional=True )

   rxPfcPrioritiesHonored = List( help="Priorities for which PFC pause frames"
                                       "are honored",
                                  valueType=int )

   enabled = Bool( help="Indicates whether priority flow control is enabled "
                        "at a global level" )


#
# Model for PFC Watchdog timer details.
# Usage: Sub-Model.
#
class PfcWatchdogTimerModel( Model ):
   timeout = Float( help="Time (in seconds) for the priority flow control "
                         "watchdog to expire" )

   pollInterval = Float( help="Polling interval (in seconds) configured for the "
                              "priority flow control watchdog" )

   operPollInterval = Float( help="Polling interval (in seconds) actually "
                                  "operational for the priority flow control "
                                  "watchdog", optional=True )

   recoveryTime = Float( help="Wait time (in seconds) after which the priority "
                              "flow control watchdog marks a transmit queue as "
                              "recovered" )

   forcedRecovery = Bool( help="Indicates whether forced recovery has been "
                               "configured for the priority flow control watchdog" )


#
# Model for PFC Watchdog non-disruptive priorities.
# Usage: Sub-Model.
#
class PfcWatchdogNDPrioritiesModel( Model ):
   supported = Bool( help="Indicates whether non-disruptive priorities are "
                          "supported by the priority flow control watchdog" )

   priorities = List( help="Priority values configured as non-disruptive for the "
                           "priority flow control watchdog", valueType=int,
                      optional=True )


#
# Model for global PFC Watchdog status.
# Usage: Sub-Model.
#
class PfcWatchdogStatusModel( Model ):
   supported = Bool( help="Indicates whether priority flow control watchdog "
                          "is supported at a global level" )

   action = Enum( help="Action taken by the priority flow control watchdog",
                  values=( 'errdisable', 'drop', 'notify-only' ), optional=True )

   timer = Submodel( help="Timer attributes of the priority flow control watchdog",
                     valueType=PfcWatchdogTimerModel, optional=True )

   nDPriorities = Submodel( help="Non-disruptive priorities supported by the "
                                 "priority flow control watchdog",
                            valueType=PfcWatchdogNDPrioritiesModel, optional=True )

   nDAction = Enum( help="Action taken on non-disruptive queues by the priority "
                       "flow control watchdog",
                  values=( 'invalid', 'forward' ), optional=True )

   forcedNd = Bool( help="Indicates whether port action non-disruptive-only"
                    " behaviour is on or off", optional=True )

   overrideAction = Bool( help="Indicates whether override action has been enabled"
                          " to unconditionally drop packets", optional=True )

   hwMonitoredPri = List( help="Priority values configured as Hardware monitored "
                               "priorities", valueType=int, optional=True )

#
# Model for interface-level PFC status.
# Usage: Sub-Model.
#
class PfcIntfStatusModel( Model ):
   enabled = Bool( help="Indicates whether priority flow control is enabled" )

   priorities = List( help="Priorities for which priority flow control is enabled",
                      valueType=int )

   active = Bool( help="Indicates whether priority flow control is operationally "
                       "active" )

   watchdogEnabled = Bool( help="Indicates whether priority flow control watchdog "
                                "is enabled" )

   watchdogActive = Bool( help="Indicates whether priority flow control watchdog "
                                "is operationally active" )

   watchdogAction = Enum( help="Action taken by the priority flow control watchdog",
                          values=( 'errdisable', 'drop', 'notify-only' ),
                          optional=True )

   watchdogTimer = Submodel( help="Timer attributes of the priority flow control "
                                  "watchdog", valueType=PfcWatchdogTimerModel,
                             optional=True )

   note = Str( help="Informational note on the status of priority flow control",
               optional=True )


#
# Model for PFC frame counters.
# Usage: Sub-Model.
#
class PfcFrameCountersModel( Model ):
   __revision__ = 3
   rxFrames = Int(
         optional=True, help="Number of priority flow control frames received" )

   txFrames = Int(
         optional=True, help="Number of priority flow control frames transmitted" )

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         if not dictRepr.get( 'rxFrames' ):
            dictRepr[ 'rxFrames' ] = 0
         if not dictRepr.get( 'txFrames' ):
            dictRepr[ 'txFrames' ] = 0
      return dictRepr

#
# Model for interface-level PFC detailed counters.
# Usage: Sub-Model.
#
class PfcIntfDetailedCountersModel( Model ):
   priorities = Dict( help="A mapping of priority value to priority flow control "
                           "counters",
                      valueType=PfcFrameCountersModel,
                      keyType=int )

#
# Model for PFC history counters.
# Usage: Sub-Model.
#
class PfcHistoryCountersModel( Model ):
   totalRxPfcOnTransitions = Int( help="Total RX PFC on transitions", optional=True )
   maxRxPfcOnTimeNs = Int( help="Max RX PFC on duration in ns", optional=True )
   maxRxPfcOnTimestamp = Float( help="UTC timestamp when max RX PFC on duration was "
                                     "recorded",
                                optional=True )
   totalRxPfcOnTimeNs = Int( help="Total RX PFC on duration in ns", optional=True )

#
# Model for interface-level PFC history counters.
# Usage: Sub-Model.
#
class PfcIntfHistoryCountersModel( Model ):
   priorities = Dict( help="A mapping of priority value to priority flow control "
                           "history counters",
                      valueType=PfcHistoryCountersModel,
                      keyType=int )

#
# Model for per-transmit-queue PFC Watchdog counters.
# Usage: Sub-Model.
#
class PfcWatchdogTxQueueCountersModel( Model ):
   queueType = Enum( help="Type of the transmit queue",
                     values=( 'unicast', 'multicast' ) )

   stuckCount = Int( help="Number of times the transmit queue was stuck" )
   recoveryCount = Int( help="Number of times the transmit queue recovered" )

   lastStuckTimestamp = Float(
      help="Seconds since start-up to "
           "last recorded stuck event for transmit queue" )
   lastStuckTimestampUtc = Float(
      help="UTC timestamp of "
           "last recorded stuck event for transmit queue" )

   lastRecoveryTimestamp = Float(
      help="Seconds since start-up to "
           "last recorded recovery event for transmit queue" )
   lastRecoveryTimestampUtc = Float(
      help="UTC timestamp of "
           "last recorded recovery event for transmit queue" )

#
# Model for per-transmit-queue PFC Watchdog counters.
# Usage: Sub-Model.
#
class PfcWatchdogEgressDropCountersModel( Model ):
   queueType = Enum( help="Type of the transmit queue",
                     values=( 'unicast', 'multicast' ),
                     optional=True )

   outCurrentDrops = Int( help="Packets dropped in current stuck-recovery cycle",
                          optional=True )
   outCurrentByteDrops = Int( help="Bytes dropped in current stuck-recovery cycle",
                              optional=True )
   outAggregateDrops = Int( help="Packets dropped at the transmit queue",
                            optional=True )
   outAggregateByteDrops = Int( help="Bytes dropped at the transmit queue ",
                                optional=True )
#
# Model for per-tc ingress PFC Watchdog drop counters.
# Usage: Sub-Model.
#

class PfcWatchdogIngressDropCountersModel( Model ):
   currentDrops = Int( help="Number of packets dropped in current"
                         " stuck-recovery cycle on the ingress side",
                       optional=True )
   currentByteDrops = Int( help="Bytes dropped in current stuck-recovery cycle",
                           optional=True )
   aggregateDrops = Int( help="Packets dropped on the ingress side",
                         optional=True )
   aggregateByteDrops = Int( help="Bytes dropped on the ingress side",
                             optional=True )

#
# Model for interface-level PFC Watchdog counters.
# Usage: Sub-Model.
#
class PfcWatchdogIntfCountersModel( Model ):
   txQueues = Dict( help="A mapping of transmit queue id to priority flow control "
                         "watchdog counters",
                    valueType=PfcWatchdogTxQueueCountersModel, keyType=int )

#
# Model for interface-level PFC Watchdog drop counters.
# Usage: Sub-Model.
#
class PfcWatchdogIntfEgressDropCountersModel( Model ):
   egressCounters = Dict( help="A mapping of transmit queue id to priority flow "
                         "control watchdog drop counters",
                    valueType=PfcWatchdogEgressDropCountersModel, keyType=int )
   ingressCounters = Dict(
      help="A mapping of traffic class to priority flow control watchdog ingress"
      " drop counters", valueType=PfcWatchdogIngressDropCountersModel, keyType=int )

#
# Model for PFC status on the Fabric interface.
#
# Usage: Model associated with these commands:
#  - show interfaces Fabric priority-flow-control
#  - show priority-flow-control interfaces Fabric status
#
class PfcFabricIntfStatusModel( Model ):
   intfName = Str( help="Fabric interface name" )

   enabled = Bool( help="Indicates whether priority flow control is enabled" )

   priorities = List( help="Priorities for which priority flow control is enabled",
                      valueType=int )

   status = Submodel( help="Status of priority flow control at a global level",
                      valueType=PfcStatusModel )

   watchdogStatus = Submodel( help="Status of priority flow control watchdog at "
                                   "a global level",
                              valueType=PfcWatchdogStatusModel, optional=True )

   def render( self ):
      return pfcFabricIntfStatusModelRender( self )


#
# Model for PFC status on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control status
#  - show priority-flow-control [ interfaces <range> ] status
#
class PfcIntfRangeStatusModel( Model ):
   __revision__ = 2
   interfaceStatuses = Dict( help="A mapping of interface name to priority flow "
                                  "control status",
                             keyType=Interface,
                             valueType=PfcIntfStatusModel )

   status = Submodel( help="Status of priority flow control at a global level",
                      valueType=PfcStatusModel )

   watchdogStatus = Submodel( help="Status of priority flow control watchdog "
                                   "at a global level",
                              valueType=PfcWatchdogStatusModel, optional=True )

   def render( self ):
      return pfcIntfRangeStatusModelRender( self )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaceStatuses' ]
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model


#
# Model for PFC counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control counters
#  - show priority-flow-control [ interfaces <range> ] counters
#
class PfcIntfRangeCountersModel( Model ):
   __revision__ = 3
   interfaceCounters = Dict( help="A mapping of interface name to priority flow "
                                  "control counters",
                             keyType=Interface,
                             valueType=PfcFrameCountersModel )

   def render( self ):
      return pfcIntfRangeCountersModelRender( self )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaceCounters' ]
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model

#
# Model for PFC detailed counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control counters detail
#  - show priority-flow-control [ interfaces <range> ] counters detail
#
class PfcIntfRangeDetailedCountersModel( Model ):
   __revision__ = 3
   interfaces = Dict( help="A mapping of interface name to priority flow control "
                           "detailed counters",
                      keyType=Interface,
                      valueType=PfcIntfDetailedCountersModel )

   def render( self ):
      return pfcIntfRangeDetailedCountersModelRender( self )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaces' ]
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model

#
# Model for PFC history counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show priority-flow-control [ interfaces <range> ] counters history
#
class PfcIntfRangeHistoryCountersModel( Model ):
   lastClearTimestamp = Float( help="UTC timestamp when clear command was last "
                                    "issued" )
   interfaces = Dict( help="A mapping of interface name to priority flow control "
                           "history counters",
                      keyType=Interface,
                      valueType=PfcIntfHistoryCountersModel )

   def render( self ):
      return pfcIntfRangeHistoryCountersModelRender( self )

#
# Model for PFC Watchdog counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control counters watchdog
#  - show priority-flow-control [ interfaces <range> ] counters watchdog
#
class PfcWatchdogIntfRangeCountersModel( Model ):
   __revision__ = 2
   interfaces = Dict( help="A mapping of interface name to priority flow control "
                           "watchdog counters",
                      keyType=Interface,
                      valueType=PfcWatchdogIntfCountersModel )

   _cliOutput = Bool( help="Indicates whether a CLI output is needed. This handles "
                           "the quirky expectation that there should be a non-empty "
                           "output, even when no interfaces have priority flow "
                           "control watchdog configured" )

   def cliOutputIs( self, cliOutput ):
      self._cliOutput = cliOutput

   def render( self ):
      return pfcWatchdogIntfRangeCountersModelRender( self, self._cliOutput )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaces' ]
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model

#
# Model for PFC Watchdog drop counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control counters watchdog drop
#  - show priority-flow-control [ interfaces <range> ] counters watchdog drop
#
class PfcWatchdogIntfRangeDropCountersModel( Model ):
   __revision__ = 3
   interfaces = Dict( help="A mapping of interface name to priority flow control "
                           "watchdog drop counters",
                      keyType=Interface,
                      valueType=PfcWatchdogIntfEgressDropCountersModel )

   _cliOutput = Bool( help="Indicates whether a CLI output is needed. This handles "
                           "the quirky expectation that there should be a non-empty "
                           "output, even when no interfaces have priority flow "
                           "control watchdog configured" )

   def cliOutputIs( self, cliOutput ):
      self._cliOutput = cliOutput

   def render( self ):
      return pfcWatchdogIntfRangeDropCountersModelRender( self, self._cliOutput )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaces' ]
      degradePfcWdDropsOptionalParamsHelper( dictRepr, revision, keyList )
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model

#
# Model for PFC status and counters on interfaces.
#
# Usage: Model associated with these commands:
#  - show interfaces [ <range> ] priority-flow-control
#  - show priority-flow-control [ interfaces <range> ]
#
class PfcIntfRangeModel( Model ):
   __revision__ = 3
   interfaceCounters = Dict( help="A mapping of interface name to priority flow "
                                  "control counters",
                             keyType=Interface,
                             valueType=PfcFrameCountersModel )

   interfaceStatuses = Dict( help="A mapping of interface name to priority flow "
                                  "control status",
                             keyType=Interface,
                             valueType=PfcIntfStatusModel )

   status = Submodel( help="Status of priority flow control at a global level",
                      valueType=PfcStatusModel )

   watchdogStatus = Submodel( help="Status of priority flow control watchdog "
                                   "at a global level",
                              valueType=PfcWatchdogStatusModel, optional=True )

   def render( self ):
      return pfcIntfRangeModelRender( self )

   def degrade( self, dictRepr, revision ):
      keyList = [ 'interfaceCounters', 'interfaceStatuses' ]
      model = pfcDegradeHelper( dictRepr, revision, keyList )
      return model


class TxQueueLatencyThresholdModel( Model ):
   txQueue = Str( help="Name of transmit queue" )
   threshold = Int( help="Transmit queue latency threshold" )
   unit = Str( help='Latency threshold unit' )

class IntfLatencyThresholdModel( Model ):
   txQueues = List( valueType=TxQueueLatencyThresholdModel,
                    help="List of transmit queue latency "
                    "threshold settings" )

class IntfLatencyThresholdCollectionModel( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=IntfLatencyThresholdModel,
                      help="A mapping of interface to "
                           "transmit queue latency threshold" )

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         print( intf + ':' )
         print()
         table = createTable( ( 'Tx Queue', 'Maximum latency' ) )
         for txqLatencyThreshold in self.interfaces[ intf ].txQueues:
            if txqLatencyThreshold.threshold:
               maxLatency = str( txqLatencyThreshold.threshold ) + ' ' + \
                            txqLatencyThreshold.unit
            else:
               maxLatency = '-'
            table.newRow( txqLatencyThreshold.txQueue, maxLatency )
         print( table.output() )

class ProfileModel( Model ):
   __revision__ = 2
   profileName = Str( help="Profile name" )
   police = Submodel( valueType=PoliceModel,
                      help="Policing configuration associated with profiles" )
   configuredIngressIntfs = Dict( keyType=Interface, valueType=bool,
                                  optional=True,
            help="A mapping of interface to whether it is configured on ingress " )
   policerAssociations = List( valueType=str, optional=True,
                               help="Policers to which the profile is attached" )

   def render( self ):
      output = '%s: ' % ( self.profileName )
      if self.police:
         output += 'rate %d %s burst-size %d %s ' % (
            self.police.cir, self.police.cirUnit, self.police.bc,
            self.police.bcUnit )
         if self.police.pir:
            output += 'rate %d %s burst-size %d %s' % (
               self.police.pir, self.police.pirUnit, self.police.be,
               self.police.beUnit )
      if self.configuredIngressIntfs:
         output += '\n  Ingress interfaces: '
         for intf in self.configuredIngressIntfs:
            output += '%s' % intf
            output += '*, ' if self.configuredIngressIntfs[ intf ] is False else ', '
         output = output[ : -2 ]
      if self.policerAssociations:
         output += '\n  Policers: '
         for policer in self.policerAssociations:
            output += '%s, ' % policer
         output = output[ : -2 ]
      print( output )

class PolicerInstanceModel( Model ):
   policerName = Str( help="Policer instance name" )
   profileName = Str( help="Profile associated with Policer instance" )
   configuredIngressIntfs = Dict( keyType=Interface, valueType=bool,
                                  optional=True,
            help="A mapping of interface to whether it is configured on ingress" )

   def render( self ):
      output = '%s: Profile %s' % ( self.policerName, self.profileName )
      if self.configuredIngressIntfs:
         output += '\n  Ingress interfaces: '
         for intf in self.configuredIngressIntfs:
            output += '%s' % intf
            output += '*, ' if self.configuredIngressIntfs[ intf ] is False else ', '
         output = output[ : -2 ]
      print( output )

class HwProgrammingEnumModel( Model ):
   hwState = Enum( values=( "active", "inactive", "partial" ),
                   help="Hardware programming state" )

class PolicerPktSizeAdjProfileModel( Model ):
   profileName = Str( help="Profile name" )
   pktSizeAdj = Int(
               help="Packet size adjustment configuration associated with profiles" )
   configuredIngressIntfs = Dict( keyType=Interface,
                                  valueType=HwProgrammingEnumModel,
                                  optional=True,
            help="A mapping of interface to whether it is configured on ingress " )

   def render( self ):
      output = '%s: ' % ( self.profileName )
      sign = "plus"
      if self.pktSizeAdj < 0:
         sign = "minus"
      output += "packet size adjustment %s %d bytes" % (
         sign, abs( self.pktSizeAdj ) )
      if self.configuredIngressIntfs:
         suffixes = {
            'partial': '#',
            'inactive': '*',
            'active': '',
         }
         output += '\n  Ingress interfaces: '
         output += ', '.join( '%s%s' % ( intf, suffixes[ data.hwState ] )
                              for intf, data in self.configuredIngressIntfs.items() )
      print( output )

class ModePolicingModel( Model ):
   __revision__ = 2
   profiles = Dict( keyType=str, valueType=ProfileModel,
                    help="Profiles indexed by their name" )
   policerInstances = Dict( keyType=str, valueType=PolicerInstanceModel,
                            help="Policer instances indexed by their name" )
   pktSizeAdjProfiles = Dict( keyType=str, valueType=PolicerPktSizeAdjProfileModel,
                              help="Policer packet size adjustment profiles index"
                                   " by their name" )
   _allOutput = Bool( help="Print output for all profiles, policerInstances and"
                           " policer packet size adjustment profiles",
                      default=False, optional=True )

   def populateProfile( self, profileName, profileModel ):
      self.profiles[ profileName ] = profileModel

   def populatePolicerInstance( self, policerName, policerInstanceModel ):
      self.policerInstances[ policerName ] = policerInstanceModel

   def populatePktSizeAdjProfile( self, profileName, profileModel ):
      self.pktSizeAdjProfiles[ profileName ] = profileModel

   def populateAllOutput( self, allOutput ):
      self._allOutput = bool( allOutput )

   def render( self ):
      print( 'Legend' )
      print( '* Not active, # Partially active' )
      if self.profiles:
         print( 'Profiles:' )
         for profile in self.profiles:
            profile = self.profiles[ profile ]
            profile.render()
      if self.policerInstances:
         if self._allOutput:
            print( '\nPolicers:' )
         else:
            print( 'Policers:' )
         for policer in self.policerInstances:
            police = self.policerInstances[ policer ]
            police.render()
      if self.pktSizeAdjProfiles:
         if self._allOutput:
            print( '\nPacket size adjustment profiles:' )
         else:
            print( 'Packet size adjustment profiles:' )
         for profile in self.pktSizeAdjProfiles:
            profile = self.pktSizeAdjProfiles[ profile ]
            profile.render()

class InterfacePolicerModel( Model ):
   profileName = Str( help="Profile name", optional=True )
   policerName = Str( help="Group policer", optional=True )
   isConfigured = Submodel( valueType=HwProgrammingEnumModel,
                            help="Interface programming status" )
   portChannelMembers = Dict( keyType=Interface, valueType=HwProgrammingEnumModel,
                  help="A mapping of member interface to whether it is configured",
                  optional=True )

class InterfacePolicerAllModel( Model ):
   intfPolicerInfo = Dict( keyType=Interface, valueType=InterfacePolicerModel,
                           help="Ingress interface-policer info keyed by interface" )
   intfEgrPolicerInfo = Dict( keyType=Interface, valueType=InterfacePolicerModel,
                           help="Egress interface-policer info keyed by interface" )
   intfCpuPolicerInfo = Dict( keyType=Interface, valueType=InterfacePolicerModel,
                              help="CPU interface-policer info keyed by interface" )
   _detail = Bool( help="Print detailed interface output", default=False )
   _policerDir = Str( help="Print interface-policer info by policer direction" )
   _ingPolicerSupported = Bool( help="Print ingress policers info", default=False )
   _egrPolicerSupported = Bool( help="Print egress policers info", default=False )
   _cpuPolicerSupported = Bool( help="Print CPU policers info", default=False )

   def populateIntfPolicerInfo( self, interface, interfacePolicerModel, policerDir ):
      if policerDir == 'input':
         self.intfPolicerInfo[ interface ] = interfacePolicerModel
      elif policerDir == 'output':
         self.intfEgrPolicerInfo[ interface ] = interfacePolicerModel
      elif policerDir == 'cpu':
         self.intfCpuPolicerInfo[ interface ] = interfacePolicerModel

   def detailIs( self, detail ):
      self._detail = bool( detail )

   def policerDirIs( self, policerDir ):
      self._policerDir = str( policerDir )

   def ingPolicerSupportedIs( self, ingPolicerSupported ):
      self._ingPolicerSupported = ingPolicerSupported

   def egrPolicerSupportedIs( self, egrPolicerSupported ):
      self._egrPolicerSupported = egrPolicerSupported

   def cpuPolicerSupportedIs( self, cpuPolicerSupported ):
      self._cpuPolicerSupported = cpuPolicerSupported

   def populateTableForInterface( self, table, policerDir ):
      isLagIntfPresent = False
      intfPolicerInfo = None
      if policerDir == 'cpu':
         intfPolicerInfo = self.intfCpuPolicerInfo
      elif policerDir == 'output':
         intfPolicerInfo = self.intfEgrPolicerInfo
      else:
         intfPolicerInfo = self.intfPolicerInfo

      for intf in ArPyUtils.naturalsorted( intfPolicerInfo ):
         interface = str( intf )
         profile = intfPolicerInfo[ intf ].profileName
         group = intfPolicerInfo[ intf ].policerName
         suffix = ""
         if intfPolicerInfo[ intf ].portChannelMembers:
            isLagIntfPresent = True
         if intfPolicerInfo[ intf ].isConfigured.hwState == 'inactive':
            suffix = '*'
         elif intfPolicerInfo[ intf ].isConfigured.hwState == 'partial':
            suffix = '#'
         profile = profile + suffix if profile else '-'
         group = group + suffix if group else '-'
         table.newRow( interface, profile, group )
      return isLagIntfPresent

   def printDetailOutput( self, isLagIntfPresent, policerDir ):
      intfPolicerInfo = None
      if policerDir == 'cpu':
         intfPolicerInfo = self.intfCpuPolicerInfo
      elif policerDir == 'output':
         intfPolicerInfo = self.intfEgrPolicerInfo
      else:
         intfPolicerInfo = self.intfPolicerInfo

      if isLagIntfPresent:
         table = createTable( ( "Port-Channel", "Member Interface", "Profile",
                                 "Group" ) )
         for intf in intfPolicerInfo:
            interface = str( intf )
            profile = intfPolicerInfo[ intf ].profileName
            group = intfPolicerInfo[ intf ].policerName
            for member in intfPolicerInfo[ intf ].portChannelMembers:
               memberInfo = intfPolicerInfo[ intf ].portChannelMembers[
                  member ]
               suffix = ""
               if memberInfo.hwState == 'inactive':
                  suffix = "*"
               elif memberInfo.hwState == 'partial':
                  suffix = "#"
               newProfile = profile + suffix if profile else '-'
               newGroup = group + suffix if group else '-'
               table.newRow( interface, member, newProfile, newGroup )
         print( table.output() )

   def render( self ):
      print( 'Legend' )
      print( '* Not active, # Partially active' )

      if self._ingPolicerSupported and \
         ( self._policerDir == 'input' or self._policerDir == 'all' ):
         print( '\nIngress policing' )
         table = createTable( ( "Interface", "Profile", "Group" ) )
         isLagIntfPresent = self.populateTableForInterface( table, 'input' )
         print( table.output() )
         if self._detail:
            self.printDetailOutput( isLagIntfPresent, 'input' )

      if self._egrPolicerSupported and \
         ( self._policerDir == 'output' or self._policerDir == 'all' ):
         print( '\nEgress policing' )
         table = createTable( ( "Interface", "Profile", "Group" ) )
         isLagIntfPresent = self.populateTableForInterface( table, 'output' )
         print( table.output() )
         if self._detail:
            self.printDetailOutput( isLagIntfPresent, 'output' )

      if self._cpuPolicerSupported and \
         ( self._policerDir == 'cpu' or self._policerDir == 'all' ):
         print( '\nCPU policing' )
         table = createTable( ( "Interface", "Profile", "Group" ) )
         isLagIntfPresent = self.populateTableForInterface( table, 'cpu' )
         print( table.output() )
         if self._detail:
            self.printDetailOutput( isLagIntfPresent, 'cpu' )

class InterfacePolicingCounterModel( Model ):
   profileName = Str( help="Profile name", optional=True )
   policerName = Str( help="Group policer", optional=True )
   mode = Enum( values=( "committed", "trtcm" ), help="Policer mode" )
   counters = Submodel( valueType=IntfPoliceCounters, help='Policer Counters' )

   def countersDict( self ):
      return {
         "greenPkt": self.counters.conformedPackets,
         "greenBytes": self.counters.conformedBytes,
         "yellowPkt": self.counters.yellowPackets,
         "yellowBytes": self.counters.yellowBytes,
         "redPkt": self.counters.exceededPackets,
         "redBytes": self.counters.exceededBytes
      }

   def render( self ):
      output = ""
      profilePolicerString = ""
      if self.profileName:
         profilePolicerString += f"Profile: {self.profileName}"
      if self.profileName and self.policerName:
         profilePolicerString += ", "
      if self.policerName:
         profilePolicerString += f"Group: {self.policerName}"

      output += profilePolicerString

      indent = "\n  "  # Indented by 2 spaces
      conformedStr = "Below the rate limit {greenPkt} packets, {greenBytes} bytes"
      conformedStrTrTcm = ( "Below the low rate limit {greenPkt} packets, "
                            "{greenBytes} bytes" )
      yellowStr = ( "Above the low rate limit and below the high rate "
                    "limit {yellowPkt} packets, {yellowBytes} bytes" )
      exceededStr = "Above the rate limit {redPkt} packets, {redBytes} bytes"

      if self.mode == 'committed':
         policeStr = ( indent + conformedStr +
                       indent + exceededStr )
      else:
         policeStr = ( indent + conformedStrTrTcm +
                       indent + yellowStr +
                       indent + exceededStr )

      countersDict = self.countersDict()
      output += policeStr.format( **countersDict )
      print( output )

class InterfacePolicingCounterAllModel( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfacePolicingCounterModel,
                      help=( "Interface ingress policing counter info "
                             "keyed by interface" ) )
   egrInterfaces = Dict( keyType=Interface, valueType=InterfacePolicingCounterModel,
                         help=( "Interface egress policing counter info "
                                "keyed by interface" ) )
   cpuInterfaces = Dict( keyType=Interface, valueType=InterfacePolicingCounterModel,
                         help=( "Interface CPU policing counter info "
                                "keyed by interface" ) )

   def populateIntfCounterInfo( self, intf, intfCounterModel, policerType ):
      if policerType == 'input':
         self.interfaces[ intf ] = intfCounterModel
      elif policerType == 'output':
         self.egrInterfaces[ intf ] = intfCounterModel
      elif policerType == 'cpu':
         self.cpuInterfaces[ intf ] = intfCounterModel

   def render( self ):
      print( "\n===== Interface Policing Counters =====" )

      if self.interfaces:
         print( "\nIngress policing" )

         for intf, counterInfo in self.interfaces.items():
            print( f"\n{intf}:" )
            counterInfo.render()

      if self.egrInterfaces:
         print( "\nEgress policing" )

         for intf, counterInfo in self.egrInterfaces.items():
            print( f"\n{intf}:" )
            counterInfo.render()

      if self.cpuInterfaces:
         print( "\nCPU policing" )

         for intf, counterInfo in self.cpuInterfaces.items():
            print( f"\n{intf}:" )
            counterInfo.render()
