#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import LazyMount
import Plugins
from BasicCliModes import GlobalConfigMode
from CliPlugin import QosCli
from CliPlugin.AclCli import inKwMatcher
from CliToken.Hardware import hardwareMatcherForConfig

# -----------------------------------------------------------------------------------
# Variables for Qos Wred associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosAclHwStatus = None

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardRoutedPortSubIntfAclSharing( mode, token ):
   if qosAclHwStatus.routedPortSubIntfQosAclSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardSharingVlan( mode, token ):
   if ( qosAclHwStatus.sviPolicyQosSupported and
        qosAclHwStatus.sviPolicyQosSharingSupported and
        not qosAclHwStatus.sviPolicyQosDefaultSharing ):
      return None
   return CliParser.guardNotThisPlatform


# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherAccessList = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc="Hardware parameter related to access-list" )
matcherResource = CliMatcher.KeywordMatcher( 'resource',
      helpdesc="Hardware resources" )
matcherSharing = CliMatcher.KeywordMatcher( 'sharing',
      helpdesc="Sharing behaviour of hardware resources" )

# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeRoutedport = CliCommand.guardedKeyword( 'routedport', helpdesc="routed port",
      guard=guardRoutedPortSubIntfAclSharing )
nodeSubinterface = CliCommand.guardedKeyword( 'subinterface',
      helpdesc="sub-interface", guard=guardRoutedPortSubIntfAclSharing )

# --------------------------------------------------------------------------------
# [ no | default ] hardware access-list qos resource sharing vlan in
# --------------------------------------------------------------------------------
class HwAccessListQosResourceSharingVlanInCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list qos resource sharing vlan in'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'qos': QosCli.nodeQosForConfig,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': CliCommand.guardedKeyword( 'vlan',
         helpdesc='Share hardware resources when qos access-list is '
                  'attached to VLAN interfaces', guard=guardSharingVlan ),
      'in': inKwMatcher,
   }

   handler = 'QosCliPolicyHandler.enableQosAclSharing'
   noOrDefaultHandler = 'QosCliPolicyHandler.disableQosAclSharing'


GlobalConfigMode.addCommandClass( HwAccessListQosResourceSharingVlanInCmd )

# --------------------------------------------------------------------------------
# [ no | default ] hardware access-list [ qos ] resource sharing
#                                    routedport subinterface
# --------------------------------------------------------------------------------
class RoutedPortSubintfCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list [ qos ] resource sharing routedport subinterface'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'qos': QosCli.nodeQosForConfig,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'routedport': nodeRoutedport,
      'subinterface': nodeSubinterface,
   }

   handler = 'QosCliPolicyHandler.gotoRoutedPortSubIntfAclSharing'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( RoutedPortSubintfCmd )

@Plugins.plugin( provides=( "QosCliPolicy", ) )
def Plugin( entityManager ):
   global qosAclHwStatus
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
