#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliExtensions
import CliParser
import CliMatcher
import ShowCommand
import LazyMount
import Plugins
import QosLib
from BasicCliModes import GlobalConfigMode
from Intf.IntfRange import IntfRangeMatcher
from CliPlugin import QosCli
from CliPlugin.QosCliIntfTypes import ethIntfTypes
from CliPlugin.QosCliCommon import ( nodeRandomDetect, nodeMls,
                                     weightRangeFn, nodeTxQueueRandomDetect,
                                     randomDetectCmdsAdapter, matcherInterfaces,
                                     matcherIntf, guardConfigureGlobalWredAllowEct,
                                     IntfUcTxQueueModelet )
from CliPlugin.QosCliModel import ( WredIntfQueueCountersModel,
                                    IntfWredCollectionModel )
from CliPlugin.QosCli import nodeQosForShow
from QosTypes import tacShapeRateVal

# -----------------------------------------------------------------------------------
# Variables for Qos Wred associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosHwStatus = None

# -----------------------------------------------------------------------------------
# Hooks
# -----------------------------------------------------------------------------------
showWredIntfQueueCountersHook = CliExtensions.CliHook()

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardWred( mode, token ):
   if qosHwStatus.wredSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDpWred( mode, token ):
   if qosHwStatus.dpWredSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardWredWeight( mode, token ):
   if qosHwStatus.wredWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardShowWredQueueCounter( mode, token ):
   if not qosHwStatus.wredQueueCounterSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardSegment( mode, token ):
   if qosHwStatus.segmentUnitForQueueThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardLatency( mode, token ):
   if qosHwStatus.latencyUnitsForQueueThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def ucTxQueueEcnThresholdRangeSegmentsFn( mode, context=None ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes //
               qosHwStatus.ecnSegmentSizeInBytes )

def ucTxQueueEcnThresholdRangeBytesFn( mode, context=None ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes )

def ucTxQueueEcnThresholdRangeKBytesFn( mode, context=None ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes // 1000 )

def ucTxQueueEcnThresholdRangeMBytesFn( mode, context=None ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes // 1000000 )

def maxThresholdInSeconds( mult ):
   # formula for converting max bytes to max milliseconds is
   # 1 sec = ( max_bytes * mult * 8 bits ) / ( bandwith_val * bits/sec )
   # max bps is 400 Gbps = 4 * 1e+9bps
   maxBitrate = tacShapeRateVal.maxKbpsForThresholdAndPpsFeatures * 1000
   # multiplier can be added to convert secs to millis/micros/etc
   return max( 1, ( qosHwStatus.ecnMaxQueueThresholdInBytes * mult * 8 ) /
                  ( maxBitrate ) )

def ucTxQueueEcnThresholdRangeMillisecondsFn( mode, context=None ):
   # 1000ms = 1s
   maxThresholdInMilliseconds = maxThresholdInSeconds( 1000 )
   return ( 1, maxThresholdInMilliseconds )

def ucTxQueueEcnThresholdRangeMicrosecondsFn( mode, context=None ):
   # 1e+6us = 1s
   maxThresholdInMicroseconds = maxThresholdInSeconds( 10 ** 6 )
   return ( 1, maxThresholdInMicroseconds )


# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherDroprate = CliMatcher.IntegerMatcher( 1, 100,
                     helpdesc='Specify maximum drop rate' )
matcherRandomDetect = CliMatcher.KeywordMatcher( 'random-detect',
      helpdesc='Show WRED based congestion control parameters' )
validWredDropPrecedenceValues = sorted( QosLib.validDpWredConfigValues() )
bytesKwMatcher = CliMatcher.KeywordMatcher( 'bytes',
                                 helpdesc='Set threshold in bytes' )
kbytesKwMatcher = CliMatcher.KeywordMatcher( 'kbytes',
                                 helpdesc='Set threshold in kbytes' )
mbytesKwMatcher = CliMatcher.KeywordMatcher( 'mbytes',
                                 helpdesc='Set threshold in mbytes' )
segmentsKwMatcher = CliCommand.guardedKeyword(
      'segments', helpdesc='Set threshold in segments', guard=guardSegment )
bytesThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeBytesFn,
                                 helpdesc='Threshold value (in bytes)' )
kbytesThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeKBytesFn,
                                 helpdesc='Threshold value (in kbytes)' )
mbytesThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeMBytesFn,
                                 helpdesc='Threshold value (in mbytes)' )
segmentsThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeSegmentsFn,
                                 helpdesc='Threshold value (in segments)' )
millisecondsThdMatcher = CliCommand.Node(
                         matcher=CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeMillisecondsFn,
                                 helpdesc='Threshold value (in milliseconds)' ),
                         guard=guardLatency )
microsecondsThdMatcher = CliCommand.Node(
                         matcher=CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeMicrosecondsFn,
                                 helpdesc='Threshold value (in microseconds)' ),
                         guard=guardLatency )
minThdBytes = "MIN_VAL_BYTES"
maxThdBytes = "MAX_VAL_BYTES"
minThdKbytes = "MIN_VAL_KBYTES"
maxThdKbytes = "MAX_VAL_KBYTES"
minThdMbytes = "MIN_VAL_MBYTES"
maxThdMbytes = "MAX_VAL_MBYTES"
minThdSegments = "MIN_VAL_SEGMENTS"
maxThdSegments = "MAX_VAL_SEGMENTS"
minThdMilliseconds = "MIN_VAL_MILLISECONDS"
maxThdMilliseconds = "MAX_VAL_MILLISECONDS"
minThdMicroseconds = "MIN_VAL_MICROSECONDS"
maxThdMicroseconds = "MAX_VAL_MICROSECONDS"
thdsUnitsEnum = {
      'BYTES1': [ minThdBytes, maxThdBytes ],
      'KBYTES1': [ minThdKbytes, maxThdKbytes ],
      'MBYTES1': [ minThdMbytes, maxThdMbytes ],
      'segments': [ minThdSegments, maxThdSegments ],
      'milliseconds': [ minThdMilliseconds, maxThdMilliseconds ],
      'microseconds': [ minThdMicroseconds, maxThdMicroseconds ],
}

# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeWred = CliCommand.guardedKeyword( 'wred', "Show WRED parameters",
      guard=guardWred )

class RandomDetectCmdsThdFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class ThdExpr( CliCommand.CliExpression ):
         expression = \
            f'( {minThdBytes} BYTES1 maximum-threshold ' \
            f'{maxThdBytes} BYTES2 ) | ' \
            f'( {minThdKbytes} KBYTES1 maximum-threshold ' \
            f'{maxThdKbytes} KBYTES2 ) | ' \
            f'( {minThdMbytes} MBYTES1  maximum-threshold ' \
            f'{maxThdMbytes} MBYTES2 ) | ' \
            f'( {minThdSegments} SEGMENTS_KW maximum-threshold ' \
            f'{maxThdSegments} segments ) | '
         data = {
            'maximum-threshold': 'Set maximum threshold for WRED',
            'BYTES1': bytesKwMatcher,
            'BYTES2': bytesKwMatcher,
            minThdBytes: bytesThdMatcher,
            maxThdBytes: bytesThdMatcher,
            'KBYTES1': kbytesKwMatcher,
            'KBYTES2': kbytesKwMatcher,
            minThdKbytes: kbytesThdMatcher,
            maxThdKbytes: kbytesThdMatcher,
            'MBYTES1': mbytesKwMatcher,
            'MBYTES2': mbytesKwMatcher,
            minThdMbytes: mbytesThdMatcher,
            maxThdMbytes: mbytesThdMatcher,
            'SEGMENTS_KW': segmentsKwMatcher,
            'segments': 'Set threshold in segments',
            minThdSegments: segmentsThdMatcher,
            maxThdSegments: segmentsThdMatcher,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            for unit, thds in thdsUnitsEnum.items():
               if unit not in args:
                  continue
               minThd = args.pop( thds[ 0 ] )
               maxThd = args.pop( thds[ 1 ] )
               unit = args.pop( unit )
               args[ 'THD_RANGE' ] = ( minThd, maxThd, unit )
               break
      return ThdExpr

class RandomDetectCmdsThdWithLatencyFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      baseCmdInfo = RandomDetectCmdsThdFactory().generate( name )

      class ThdExpr( CliCommand.CliExpression ):
         expression = baseCmdInfo.expression + \
            f'( {minThdMilliseconds} MILLISECONDS_KW maximum-threshold' \
            f' {maxThdMilliseconds} milliseconds ) | ' \
            f'( {minThdMicroseconds} MICROSECONDS_KW maximum-threshold' \
            f' {maxThdMicroseconds} microseconds ) '
         data = baseCmdInfo.data
         data.update( {
            'MILLISECONDS_KW': CliCommand.guardedKeyword( 'milliseconds',
                                       helpdesc='Set threshold in milliseconds',
                                       guard=guardLatency ),
            'milliseconds': 'Set threshold in milliseconds',
            minThdMilliseconds: millisecondsThdMatcher,
            maxThdMilliseconds: millisecondsThdMatcher,
            'MICROSECONDS_KW': CliCommand.guardedKeyword( 'microseconds',
                                       helpdesc='Set threshold in microseconds',
                                       guard=guardLatency ),
            'microseconds': 'Set threshold in microseconds',
            minThdMicroseconds: microsecondsThdMatcher,
            maxThdMicroseconds: microsecondsThdMatcher,
         } )

         adapter = baseCmdInfo.adapter
      return ThdExpr


thresholdLatencyMatcher = RandomDetectCmdsThdWithLatencyFactory()

# --------------------------------------------------------------------------------
# show qos interfaces [ INTF ] wred counters queue
# --------------------------------------------------------------------------------
class QosInterfacesWredCountersQueueCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces [ INTF ] wred counters queue'
   data = {
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': IntfRangeMatcher( explicitIntfTypes=ethIntfTypes ),
      'wred': nodeWred,
      'counters': CliCommand.guardedKeyword( 'counters',
         "WRED interface counters", guard=guardShowWredQueueCounter ),
      'queue': 'WRED queue counters',
   }

   handler = "QosCliWredHandler.showWredIntfQueueCounters"
   cliModel = WredIntfQueueCountersModel


BasicCli.addShowCommandClass( QosInterfacesWredCountersQueueCmd )

# --------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF ] random-detect drop
# --------------------------------------------------------------------------------
class QosInterfacesIntfRandomDetectDropCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF ] random-detect drop'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'random-detect': matcherRandomDetect,
      'drop': CliCommand.guardedKeyword( 'drop', "Show WRED parameters",
         guard=guardWred ),
   }

   handler = "QosCliWredHandler.showInterfacesWred"
   cliModel = IntfWredCollectionModel


BasicCli.addShowCommandClass( QosInterfacesIntfRandomDetectDropCmd )

# --------------------------------------------------------------------------------
# [ no | default ] random-detect drop [ drop-precedence <DP> ] minimum-threshold
# <MIN_THD> <unit> maximum-threshold <MAX_THD> <unit> drop-probability MAX_DROPRATE
# [ weight WEIGHT ]
# unit can be one of bytes, kbytes, mbytes, segments, milliseconds, or nanoseconds
# --------------------------------------------------------------------------------
class RandomDetectDropCmds( CliCommand.CliCommandClass ):
   syntax = 'random-detect drop [ drop-precedence DP ] minimum-threshold THD_EXPR '\
            'drop-probability MAX_DROPRATE [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect drop [ drop-precedence DP ] ...'
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'drop': CliCommand.guardedKeyword( 'drop',
                                         helpdesc='Set WRED parameters',
                                         guard=guardWred ),
      'drop-precedence': CliCommand.guardedKeyword( 'drop-precedence',
                             helpdesc="Configure drop-precedence parameters",
                             guard=guardDpWred ),
      'DP': CliMatcher.IntegerMatcher( validWredDropPrecedenceValues[ 0 ],
                                        validWredDropPrecedenceValues[ -1 ],
                                        helpdesc="Specify drop-precedence value" ),
      'minimum-threshold': 'Set minimum threshold for WRED',
      'THD_EXPR': thresholdLatencyMatcher,
      'drop-probability': 'Set maximum drop probability rate',
      'MAX_DROPRATE': matcherDroprate,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set WRED weight',
                                           guard=guardWredWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           weightRangeFn,
                                           helpdesc='Value of WRED weight' ),
   }

   adapter = randomDetectCmdsAdapter
   handler = "QosCliWredHandler.setWred"
   noOrDefaultHandler = "QosCliWredHandler.setNoOrDefaultWred"


IntfUcTxQueueModelet.addCommandClass( RandomDetectDropCmds )

# --------------------------------------------------------------------------------
# [ no | default ] qos random-detect drop allow ect
# --------------------------------------------------------------------------------
class QosRandomDetectWredAllowEctCmd( CliCommand.CliCommandClass ):
   syntax = 'qos random-detect drop allow ect'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'drop': CliCommand.guardedKeyword( 'drop',
                                        helpdesc='Set WRED parameters',
                                        guard=guardConfigureGlobalWredAllowEct ),
      'allow': 'Allow forwarding of ECN capable packets through WRED '
               'configuration on a tx-queue',
      'ect': 'ECN capable transport',
   }
   handler = "QosCliWredHandler.configureGlobalWredAllowEct"
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosRandomDetectWredAllowEctCmd )

@Plugins.plugin( provides=( "QosCliWred", ) )
def Plugin( entityManager ):
   global qosHwStatus
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
