# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import base64
import binascii
import os
import uuid

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import Tracing

from CliPlugin import QuoteCliModel, TpmCliLib
from CliPlugin import ConfigMgmtMode
from MultiRangeRule import MultiRangeMatcher
import TpmPcrLib

th = Tracing.Handle( 'QuotePcrCli' )
debug = th.trace4

quoteConfigDir = None
quoteStatusDir = None
configDir = None
tpmStatus = None
aikCertFile = "/persist/local/tpm-data/aik.cert"

def getPcrRange():
   return ( 0, 23 )

def guardTpm( mode, token ):
   if not tpmStatus.tpmVersion:
      return CliParser.guardNotThisPlatform
   return None

tpmMatcher = CliCommand.guardedKeyword( 'tpm',
                                        helpdesc='Show TPM related information',
                                        guard=guardTpm )
noncePatternMatcher = CliMatcher.PatternMatcher( pattern='[0-9a-fA-F]+',
                                                 helpdesc=( 'Nonce should be a '
                                                            'hexadecimal string' ),
                                                 helpname='nonce' )
pcrRangeMatcher = MultiRangeMatcher( getPcrRange, False,
                                     ( 'PCR selection should be 0~23, '
                                       'for example: 1,2,7-9' ) )

def populateQuote( mode, model, stat ):
   for i in stat.pcrValue:
      model.registers[ i ] = \
         binascii.hexlify( stat.pcrValue[ i ].value ).decode( 'utf-8' )
   model.tpmVersion = tpmStatus.tpmVersion
   model.quoteInfo = base64.b64encode( stat.quoteInfo ).decode( encoding='ascii' )
   model.quoteData = base64.b64encode( stat.quoteData ).decode( encoding='ascii' )

   if not os.path.isfile( aikCertFile ):
      mode.addError( "No AIK certificate found" )
      return False

   with open( aikCertFile ) as f:
      model.aikCert = f.read()

   return True

def showQuote( mode, args ):
   result = QuoteCliModel.PcrQuote()
   nonce = args.get( 'NONCE_VAL' )
   pcrRange = args.get( 'PCR_RANGE' )
   pcrSelect = pcrRange.values()

   if not nonce:
      nonce = TpmPcrLib.getEmptyNonce( tpmStatus.tpmVersion )
   else:
      errMsg = TpmPcrLib.checkNonce( tpmStatus.tpmVersion, nonce )
      if errMsg:
         mode.addErrorAndStop( errMsg )

   quoteConfig = quoteConfigDir.pcrQuoteConfig
   quoteStatus = quoteStatusDir.pcrQuoteStatus

   randomId = str( uuid.uuid4() )
   conf = configDir.newEntity( 'Hardware::TpmLib::PcrQuoteConfig', randomId )
   conf.nonce = nonce
   for i in pcrSelect:
      conf.quoteSelection.add( i )
   quoteConfig.addMember( conf )

   try:
      Tac.waitFor( lambda: randomId in quoteStatus,
                           description="quote result to be returned",
                           sleep=True, timeout=10 )

      stat = quoteStatus[ randomId ]

      if stat.failed:
         mode.addError( "TPM internal failure." )
         return None

      if not populateQuote( mode, result, stat ):
         return None

      return result
   except Tac.Timeout:
      mode.addError( "Timeout waiting for TPM results." )
      return None
   finally:
      del quoteConfig[ randomId ]
      configDir.deleteEntity( randomId )

class ShowQuoteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management security tpm pcr [ nonce NONCE_VAL ] '
              'PCR_RANGE' )
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'security': 'Show security status',
      'tpm': TpmCliLib.tpmMatcher,
      'pcr': 'Show PCR attestation information',
      'nonce': 'Optional nonce value',
      'NONCE_VAL': noncePatternMatcher,
      'PCR_RANGE': pcrRangeMatcher,
   }
   cliModel = QuoteCliModel.PcrQuote
   handler = showQuote

BasicCli.addShowCommandClass( ShowQuoteCmd )

def Plugin( entityManager ):
   global quoteConfigDir, quoteStatusDir, configDir, tpmStatus

   quoteConfigDir = LazyMount.mount( entityManager,
                                     Cell.path( 'hardware/tpm/pcrQuoteConfigDir' ),
                                     'Hardware::TpmLib::PcrQuoteConfigDir', 'w' )
   quoteStatusDir = LazyMount.mount( entityManager,
                                     Cell.path( 'hardware/tpm/pcrQuoteStatusDir' ),
                                     'Hardware::TpmLib::PcrQuoteStatusDir', 'r' )
   tpmStatus = LazyMount.mount( entityManager,
                                Cell.path( 'hardware/tpm/status' ),
                                'Hardware::TpmLib::Status', 'r' )
   configDir = ConfigMount.mount( entityManager,
                                  'hardware/tpm/quoteConfig',
                                  'Tac::Dir', 'wi' )
