# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import ConfigMount
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.LagIntfCli as LagIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.SubIntfCli as SubIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.VirtualIntfRule as VirtualIntfRule
from CliPlugin.GeneratorCli import (
   Rfc2544ProfilesMode,
)
from CliPlugin.GeneratorCliLib import (
   interfaceNode,
   rfc2544ConfigKwMatcher,
   generatorsNode,
   isFeatureEnabled,
)
from CliToken.Generator import(
   startKwForGenerator,
   stopKwForGenerator,
)
from CliToken.Clear import (
   clearKwNode,
)
from CliToken.Monitor import (
   monitorMatcher,
   monitorMatcherForClear,
)
from EoamTypes import (
   rateUnitToEnumMap,
   tacRateUnit,
   FeatureEnabledEnum,
)
from GeneratorCliUtils import (
   generatorDirectionEnum,
)
from RFC2544InitiatorCliUtils import (
   benchmarkTestSupportedKwStr,
   benchmarkThroughput,
   benchmarkFrameLossRate,
   PacketSizeInBytesEnum,
   ExecRequestEnum,
   TestStateEnum,
)
from TypeFuture import TacLazyType
from Arnet import EthAddr
import LazyMount
import Tac

# Globals
rfc2544ProfileConfigDir = None
rfc2544ExecConfigDir = None
rfc2544HwCapabilities = None
rfc2544ExecStatusDir = None

TrafficDuration = TacLazyType( 'Rfc2544Initiator::TrafficDuration' )
TacEthAddr = TacLazyType( 'Arnet::EthAddr' )
IntfId = TacLazyType( 'Arnet::IntfId' )
TrafficRateDefaultLimits = TacLazyType(
   'Rfc2544Initiator::TrafficRateDefaultLimits' )
EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )
PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )
supportedBenchmarkTests = {
   benchmarkTestSupportedKwStr[ benchmarkThroughput ] : 'Throughput test',
}
supportedBenchmarkTests[
   benchmarkTestSupportedKwStr[
      benchmarkFrameLossRate ] ] = 'Frame loss rate test'

def getRfc2544ProfileNames( mode ):
   return rfc2544ProfileConfigDir.profileConfig.keys()

def maxPacketRateInKbps():
   if rfc2544HwCapabilities:
      return rfc2544HwCapabilities.maxPacketRateAndUnit.packetRateInKbps()
   return TrafficRateDefaultLimits.maxRateKbps

def minPacketRateInKbps():
   if rfc2544HwCapabilities:
      return rfc2544HwCapabilities.minPacketRateAndUnit.packetRateInKbps()
   return TrafficRateDefaultLimits.minRateKbps

rfc2544ProfileMatcher = CliMatcher.DynamicNameMatcher(
   getRfc2544ProfileNames,
   'Name of the profile',
   pattern='[a-zA-Z0-9_-]+',
   helpname='WORD' )

#-----------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] test { testName(s) }
#-----------------------------------------------------------------------------
class Rfc2544TestCmd( CliCommand.CliCommandClass ):
   syntax = 'test TEST_NAMES'
   noOrDefaultSyntax = 'test ...'
   # Even though TEST_NAMES has only one test for now, it will have multiple
   # tests like latency, back-to-back, etc in very near future, so using
   # SetEnumMatcher from future perspective.
   data = { 'test' : 'Benchmark tests to be run',
            'TEST_NAMES' : CliCommand.SetEnumMatcher ( supportedBenchmarkTests )
      }

   @staticmethod
   def handler( mode, args ):
      tests = args.get( 'TEST_NAMES' )
      # The command is a single line command, and the latest command is supposed
      # to overwrite the existing config
      mode.profile.benchmarkTest.clear()
      for testType, testKw in benchmarkTestSupportedKwStr.items():
         if testKw in tests:
            mode.profile.benchmarkTest[ testType ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.benchmarkTest.clear()

Rfc2544ProfilesMode.addCommandClass( Rfc2544TestCmd )

#-------------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] direction ( in | out )
#-------------------------------------------------------------------------------
class Rfc2544DirectionCmd( CliCommand.CliCommandClass ):
   # Currently only "in" is supported
   syntax = 'direction in'
   noOrDefaultSyntax = 'direction ...'
   data = { 'direction' : 'Direction of packets injected by generator',
            'in' : 'Ingress direction' }

   @staticmethod
   def handler( mode, args ):
      if 'in' in args:
         mode.profile.direction = generatorDirectionEnum.flowIn

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.direction = generatorDirectionEnum.flowDirectionUnknown

Rfc2544ProfilesMode.addCommandClass( Rfc2544DirectionCmd )

#-------------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] packet size
# { packet_size } bytes
#-------------------------------------------------------------------------------
class Rfc2544PacketSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'packet size PACKET_SIZE bytes'
   noOrDefaultSyntax = 'packet size ...'
   data = { 'packet' : 'Packet parameters for test traffic',
            'size' : 'Packet size for test traffic',
            'PACKET_SIZE' : CliCommand.SetEnumMatcher( {
               '64' : '64 bytes',
               '128' : '128 bytes',
               '256' : '256 bytes',
               '512' : '512 bytes',
               '768' : '768 bytes',
               '1024' : '1024 bytes',
               '1280' : '1280 bytes',
               '1518' : '1518 bytes',
               '1600' : '1600 bytes',
               '1728' : '1728 bytes',
               '2000' : '2000 bytes',
               '2048' : '2048 bytes',
               '2496' : '2496 bytes',
               '3584' : '3584 bytes',
               '4016' : '4016 bytes',
               '4096' : '4096 bytes',
               '8192' : '8192 bytes',
               '9104' : '9104 bytes',
               '9136' : '9136 bytes',
               '9600' : '9600 bytes', } ),
            'bytes' : 'Bytes'
          }

   @staticmethod
   def handler( mode, args ):
      packetSizes = args.get( 'PACKET_SIZE' )
      # The command is a single line command, and the latest command is supposed
      # to overwrite the existing config
      mode.profile.packetSizeInBytes.clear()
      for packetSize in packetSizes:
         tacPacketSizeEnum = getattr( PacketSizeInBytesEnum,
                                      # pylint: disable-next=consider-using-f-string
                                      'pktSize%sBytes' % packetSize )
         mode.profile.packetSizeInBytes[ tacPacketSizeEnum ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.packetSizeInBytes.clear()

Rfc2544ProfilesMode.addCommandClass( Rfc2544PacketSizeCmd )

#-------------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] traffic duration
# TIME seconds
#-------------------------------------------------------------------------------
class Rfc2544TrafficDurationCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic duration TIME seconds'
   noOrDefaultSyntax = 'traffic duration ...'
   data = { 'traffic' : 'Traffic',
            'duration' : 'Traffic duration in seconds',
            'TIME' : CliMatcher.IntegerMatcher( TrafficDuration.min,
                     TrafficDuration.max, helpdesc='Traffic duration in seconds' ),
            'seconds' : 'Traffic duration time unit' }

   @staticmethod
   def handler( mode, args ):
      # default traffic duration = 60 seconds.
      duration = args.get( 'TIME', TrafficDuration.defaultDuration )
      mode.profile.testDurationInSeconds = duration

   noOrDefaultHandler = handler

Rfc2544ProfilesMode.addCommandClass( Rfc2544TrafficDurationCmd )

#-------------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] mac address
# [ source <H.H.H> ] destination <H.H.H>
#-------------------------------------------------------------------------------
class Rfc2544MacAddressCmd( CliCommand.CliCommandClass ):
   # Source MAC address is optional
   syntax = 'mac address [ source SMAC ] destination DMAC'
   noOrDefaultSyntax = 'mac address ...'
   data = { 'mac' : 'Destination and/or source Ethernet address to be used',
            'address' : 'Ethernet address',
            'source' : 'Source Ethernet address',
            'SMAC' : MacAddr.MacAddrMatcher(),
            'destination' : 'Destination Ethernet address',
            'DMAC' : MacAddr.MacAddrMatcher()
            }

   @staticmethod
   def handler( mode, args ):
      # The command is a single line command, and the latest command is supposed
      # to overwrite the existing config
      sourceMac = args.get( 'SMAC', TacEthAddr.ethAddrZero )
      destinationMac = args.get( 'DMAC', TacEthAddr.ethAddrZero )

      if not EthAddr( sourceMac ).isUnicast or \
         not EthAddr( destinationMac ).isUnicast:
         mode.addErrorAndStop(
            'Please configure unicast source and destination MAC address' )
      mode.profile.sourceMac = sourceMac
      mode.profile.destinationMac = destinationMac

   noOrDefaultHandler = handler

Rfc2544ProfilesMode.addCommandClass( Rfc2544MacAddressCmd )

#-------------------------------------------------------------------------------
# (config-mon-2544-gen-prof-<profile_name>)# [no|default] traffic rate RATE UNIT
#-------------------------------------------------------------------------------
class Rfc2544TrafficRateCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic rate RATE UNIT'
   noOrDefaultSyntax = 'traffic rate ...'
   data = { 'traffic' : 'Traffic',
            'rate' : 'Traffic rate',
            'RATE' : CliMatcher.IntegerMatcher( 1, maxPacketRateInKbps(),
                                                helpdesc='Traffic rate' ),
            'UNIT' : CliMatcher.EnumMatcher( {
                           'kbps' : 'Traffic unit in kbps',
                           'mbps' : 'Traffic unit in mbps',
                           'gbps' : 'Traffic unit in gbps',
                        } )
            }

   @staticmethod
   def handler( mode, args ):
      # For RFC2544, traffic rate is a mandatory config.
      # no or default version will set the traffic rate to default of 0.
      rate = args.get( 'RATE', TrafficRateDefaultLimits.defaultRateKbps )
      unit = args.get( 'UNIT' )
      unitEnum = rateUnitToEnumMap[ unit ] if unit is not None else \
                 tacRateUnit.rateUnitInvalid
      rateKbps = rate
      if unitEnum == tacRateUnit.rateUnitMbps:
         rateKbps = rateKbps * 1000
      elif unitEnum == tacRateUnit.rateUnitGbps:
         rateKbps = rateKbps * 1000 * 1000

      minRateKbps = minPacketRateInKbps()
      maxRateKbps = maxPacketRateInKbps()
      if unitEnum != tacRateUnit.rateUnitInvalid and rateKbps < minRateKbps:
         mode.addErrorAndStop( 'Rate configured below supported minimum of ' +
                               str( minRateKbps ) + ' kbps' )
      if rateKbps > maxRateKbps:
         mode.addErrorAndStop( 'Rate configured above supported maximum of ' +
                               str( maxRateKbps ) + ' kbps' )

      mode.profile.packetRateAndUnit = Tac.Value(
            "FlowGenerator::PacketRateAndUnit", rate, unitEnum )
   noOrDefaultHandler = handler

Rfc2544ProfilesMode.addCommandClass( Rfc2544TrafficRateCmd )

intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.subMatcher

def validateRfc2544IntfOrErrorAndStop( mode, rfc2544Intf ):
   intfSupported = False
   if SubIntfId.isSubIntfId( rfc2544Intf ):
      if EthIntfId.isEthIntfId( rfc2544Intf ):
         intfSupported = rfc2544HwCapabilities.rfc2544SubIntfSupported
      elif PortChannelIntfId.isPortChannelIntfId( rfc2544Intf ):
         intfSupported = rfc2544HwCapabilities.rfc2544LagSubIntfSupported
   else:
      if EthIntfId.isEthIntfId( rfc2544Intf ):
         intfSupported = rfc2544HwCapabilities.rfc2544EthIntfSupported
      elif PortChannelIntfId.isPortChannelIntfId( rfc2544Intf ):
         intfSupported = rfc2544HwCapabilities.rfc2544LagIntfSupported
   if not intfSupported:
      mode.addErrorAndStop(
         'Interface type not supported on this hardware platform' )

# ----------------------------------------------------------------------------------
# # start monitor rfc2544 generators interface <interface> profile <profile_name>
# ----------------------------------------------------------------------------------
class Rfc2544GeneratorStartExecCmd( CliCommand.CliCommandClass ):
   # Keeping the implementation of "start" and "stop" separate because:
   # - "start ..." exec command should exit if profile is not configured or
   #   profile config is incomplete.
   # - "stop ..." exec command will succeed even if profile is not configured
   #   profile config is incomplete because:
   #   - there is only one test that could be running on an interface at a time
   #   - user may have deleted the profile that was used to start the test,
   #   and may now want to stop that test. It does not make sense to force the
   #   user in such cases to have profile configured.
   syntax = 'start monitor rfc2544 generators interface INTF profile PROFILE'
   data = { 'start' : startKwForGenerator,
            'monitor' : monitorMatcher,
            'rfc2544' : rfc2544ConfigKwMatcher,
            'generators' : generatorsNode,
            'interface' : interfaceNode,
            'INTF' : CliCommand.Node( intfMatcher ),
            'profile' : 'Generator profile',
            'PROFILE' : rfc2544ProfileMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      featureEnabledError = isFeatureEnabled( FeatureEnabledEnum.rfc2544 )
      if featureEnabledError:
         mode.addErrorAndStop( featureEnabledError )

      intf = args[ 'INTF' ].name
      validateRfc2544IntfOrErrorAndStop( mode, intf )

      profileName = args[ "PROFILE" ]
      profileConfig = rfc2544ProfileConfigDir.profileConfig.get( profileName )

      intfId = IntfId( intf )
      intfExecStatus = rfc2544ExecStatusDir.intfExecStatus.get( intfId )
      # If there is a pending "start" exec request for this interface, then
      # we cannot allow new start request. However, if all the testing
      # corresponding go this start request has completed, then we can let
      # a new start request go through.
      intfExecRequest = rfc2544ExecConfigDir.intfExecRequest.get( intfId )
      pendingStartRequest = False
      # pylint: disable=too-many-nested-blocks
      if intfExecRequest and \
         intfExecRequest.execRequest == ExecRequestEnum.execStart:
         if intfExecStatus:
            testExecStatus = intfExecStatus.testExecStatus.get(
               intfExecRequest.execId )
            if testExecStatus:
               # testExecStatus exists corresponding to the existing "start" request,
               # so check if all the tests have finished.
               if testExecStatus.testStatus:
                  for testStatus in testExecStatus.testStatus.values():
                     if testStatus.testState in [ TestStateEnum.flowStateNone,
                                                  TestStateEnum.flowStateRunning ]:
                        pendingStartRequest = True
               else:
                  # If testExecStatus.testStatus does not exist, it
                  # means testing is still pending.
                  pendingStartRequest = True
            else:
               # If testExecStatus does not exist corresponding to the existing
               # "start" request, it means testing has not started.
               pendingStartRequest = True
         else:
            # If intfExecStatus does not exist, then it means testing has not
            # started for the existing "start" request.
            pendingStartRequest = True
      if pendingStartRequest:
         mode.addErrorAndStop( 'There is a pending start exec request'
                               ' for interface ' + intf )

      # If there is a test already running on a interface, then we can not
      # allow new "start" exec request.
      # The case handled above takes care of the case where there is a "start"
      # exec request in rfc2544ExecConfigDir. However, it is possible that
      # there is no "start" request, say there is a "clear" request, and the
      # test may still be running. To handle such cases, we need to check
      # independently if there is any test running on an interface, and
      # not let the "start" command go through in that case.
      if intfExecStatus is not None:
         # For a particular intfExecStatus, there can be at max 6
         # tests that could be run, so the iteration will be max 6 times.
         # Number of testExecStatus depends on how much history of
         # tests we are saving.
         for testExecStatus in intfExecStatus.testExecStatus.values():
            for testStatus in testExecStatus.testStatus.values():
               if testStatus.testState == TestStateEnum.flowStateRunning:
                  mode.addErrorAndStop( "Test " +
                     benchmarkTestSupportedKwStr[ testStatus.benchmarkTest ] +
                     " already running on interface " + intf )
      if profileConfig is None:
         mode.addErrorAndStop( "profile " + profileName + " not configured" )

      invalidProfileConfigs = []
      if profileConfig.direction == generatorDirectionEnum.flowDirectionUnknown:
         invalidProfileConfigs.append( "direction" )

      if not profileConfig.benchmarkTest:
         invalidProfileConfigs.append( "test" )

      if not profileConfig.packetRateAndUnit:
         invalidProfileConfigs.append( "traffic rate" )

      if not profileConfig.packetSizeInBytes:
         invalidProfileConfigs.append( "packet size" )

      if profileConfig.destinationMac == TacEthAddr.ethAddrZero:
         invalidProfileConfigs.append( "destination Ethernet address" )

      # If the action is "start" and the profile config is missing something,
      # then we don't proceed with the exec command.
      if invalidProfileConfigs:
         errorStr = "profile " + profileName + " has no "
         i = 0
         for invalidConfig in invalidProfileConfigs:
            if i > 0:
               errorStr += ", "
            errorStr += invalidConfig
            i += 1
         errorStr += " configured"
         mode.addErrorAndStop( errorStr )

      rfc2544ExecConfigDir.intfExecRequest.addMember(
         Tac.Value( "Rfc2544Initiator::IntfExecRequest",
                    intfId,
                    ExecRequestEnum.execStart,
                    Tac.Value( "Ark::UniqueId" ),
                    profileName ) )

# ----------------------------------------------------------------------------------
# # stop monitor rfc2544 generators interface <interface>
# ----------------------------------------------------------------------------------
class Rfc2544GeneratorStopExecCmd( CliCommand.CliCommandClass ):
   # Keeping the implementation of "start" and "stop" separate because:
   # - "start ..." exec command should exit if profile is not configured or
   #   profile config is incomplete.
   # - "stop ..." exec command will succeed even if profile is not configured
   #   or profile config is incomplete because:
   #   - there is only one test that could be running on an interface at a time
   #   - user may have deleted the profile that was used to start the test,
   #   and may now want to stop that test. It does not make sense to force the
   #   user in such cases to have profile configured.
   syntax = 'stop monitor rfc2544 generators interface INTF'
   data = { 'stop' : stopKwForGenerator,
            'monitor' : monitorMatcher,
            'rfc2544' : rfc2544ConfigKwMatcher,
            'generators' : generatorsNode,
            'interface' : interfaceNode,
            'INTF' : CliCommand.Node( intfMatcher ),
          }

   @staticmethod
   def handler( mode, args ):
      featureEnabledError = isFeatureEnabled( FeatureEnabledEnum.rfc2544 )
      if featureEnabledError:
         mode.addErrorAndStop( featureEnabledError )

      intf = args[ 'INTF' ].name
      validateRfc2544IntfOrErrorAndStop( mode, intf )

      intfId = IntfId( intf )
      rfc2544ExecConfigDir.intfExecRequest.addMember(
         Tac.Value( "Rfc2544Initiator::IntfExecRequest",
                    intfId,
                    ExecRequestEnum.execStop,
                    Tac.Value( "Ark::UniqueId" ), '' ) )

# ----------------------------------------------------------------------------------
# # clear monitor rfc2544 generators interface <interface>
# ----------------------------------------------------------------------------------
class Rfc2544GeneratorClearExecCmd( CliCommand.CliCommandClass ):
   # Keeping the implementation of "clear" and "stop" separate because in future
   # clear command will have more filters like "last <N>" etc.
   syntax = 'clear monitor rfc2544 generators interface INTF'
   data = { 'clear' : clearKwNode,
            'monitor' : monitorMatcherForClear,
            'rfc2544' : rfc2544ConfigKwMatcher,
            'generators' : generatorsNode,
            'interface' : interfaceNode,
            'INTF' : CliCommand.Node( intfMatcher ),
          }

   @staticmethod
   def handler( mode, args ):
      featureEnabledError = isFeatureEnabled( FeatureEnabledEnum.rfc2544 )
      if featureEnabledError:
         mode.addErrorAndStop( featureEnabledError )

      intf = args[ 'INTF' ].name
      validateRfc2544IntfOrErrorAndStop( mode, intf )

      intfId = IntfId( intf )
      rfc2544ExecConfigDir.intfExecRequest.addMember(
         Tac.Value( "Rfc2544Initiator::IntfExecRequest",
                    intfId,
                    ExecRequestEnum.execClear,
                    Tac.Value( "Ark::UniqueId" ), '' ) )

BasicCli.EnableMode.addCommandClass( Rfc2544GeneratorStartExecCmd )
BasicCli.EnableMode.addCommandClass( Rfc2544GeneratorStopExecCmd )
BasicCli.EnableMode.addCommandClass( Rfc2544GeneratorClearExecCmd )

def Plugin( entityManager ):
   global rfc2544ProfileConfigDir
   global rfc2544ExecConfigDir
   global rfc2544HwCapabilities
   global rfc2544ExecStatusDir

   rfc2544ProfileConfigDir = ConfigMount.mount(
         entityManager, 'rfc2544Initiator/profileConfigDir',
         'Rfc2544Initiator::ProfileConfigDir', 'w' )
   rfc2544ExecConfigDir = ConfigMount.mount(
         entityManager, 'rfc2544Initiator/execConfig',
         'Rfc2544Initiator::ExecConfigDir', 'w' )
   rfc2544HwCapabilities = LazyMount.mount(
      entityManager, 'rfc2544Initiator/hardware/capabilities',
      'Rfc2544Initiator::HwCapabilities', 'r' )
   rfc2544ExecStatusDir = LazyMount.mount(
         entityManager, 'rfc2544Initiator/execStatus',
         'Rfc2544Initiator::ExecStatusDir', 'r' )
