# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
from CliPlugin import RcfCliLib
from CliPlugin import RcfCliHelpers
from CliPlugin.RouterGeneralCli import RouterGeneralMode
from CliToken import RcfCliTokens

import ConfigMount
import ShowCommand

gv = CliGlobal.CliGlobal( dict(
      rcfConfig=None,
      codeUnitScratchpadHelper=None,
      openConfigScratchpadHelper=None,
   )
)

unitNameMatcher = CliMatcher.DynamicNameMatcher(
   # Wrap with a lambda so we don't access codeUnitScratchpadHelper before it's
   # initialized.
   # pylint: disable=unnecessary-lambda
   lambda mode: gv.codeUnitScratchpadHelper.getEffectiveNames( mode ),
   # pylint: enable=unnecessary-lambda
   helpdesc='Unique name to identify the RCF code unit',
   pattern='[a-zA-Z_][a-zA-Z0-9_-]*',
   helpname='WORD' )

# The following items are now defined under the CliDynamicPlugin directory in order
# to facilitate dynamic loading of RCF CLI commands:
# * The control-functions mode
# * The definitions and handlers of the commands under control-functions
# * The handlers of the RCF show commands that are defined globally
# See https://aid/11131 for more details.

#----------------------------------------------------------------------------------
#                               C O M M A N D S
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# "control-functions" in router general mode
#--------------------------------------------------------------------------------
class ControlFunctionsCmd( CliCommand.CliCommandClass ):
   syntax = 'control-functions'
   noOrDefaultSyntax = syntax
   data = {
      'control-functions': RcfCliTokens.controlFunctionsKwForConfig,
   }

   handler = 'RcfCliHandler.handlerControlFunctionsCmd'
   noOrDefaultHandler = 'RcfCliHandler.noOrDefaultHandlerControlFunctionsCmd'

RouterGeneralMode.addCommandClass( ControlFunctionsCmd )

#--------------------------------------------------------------------------------
# "show router rcf pending"
#--------------------------------------------------------------------------------
class ShowRouterRcfPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router rcf pending'
   data = {
      'router': RcfCliTokens.routerKw,
      'rcf': RcfCliLib.rcfKw,
      'pending': RcfCliTokens.pendingKw,
   }

   cliModel = 'RcfCliModels.ShowRcfPendingModel'
   handler = 'RcfCliHandler.handlerShowRcfPending'

BasicCli.addShowCommandClass( ShowRouterRcfPendingCmd )

#--------------------------------------------------------------------------------
# "show router rcf errors"
#--------------------------------------------------------------------------------
class ShowRcfErrorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router rcf errors [ startup ]'
   # 'startup' keyword is optional and hidden for backwards compatability
   data = {
      'router': RcfCliTokens.routerKw,
      'rcf': RcfCliLib.rcfKw,
      'errors': RcfCliTokens.errorsKw,
      'startup': RcfCliTokens.startupKw,
   }

   cliModel = 'RcfCliModels.ShowRcfErrorsModel'
   handler = 'RcfCliHandler.handlerShowRcfErrorsCmd'

BasicCli.addShowCommandClass( ShowRcfErrorsCmd )

#--------------------------------------------------------------------------------
# "show router rcf code [ unit UNIT_NAME ]"
#--------------------------------------------------------------------------------
class ShowRcfCodeUnitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router rcf code [ unit UNIT_NAME ]'
   data = {
      'router': RcfCliTokens.routerKw,
      'rcf': RcfCliLib.rcfKw,
      'code': RcfCliTokens.committedCodeKw,
      'unit': RcfCliTokens.unitKw,
      'UNIT_NAME': unitNameMatcher,
   }

   cliModel = 'RcfCliModels.ShowRcfCodeUnitModel'
   handler = 'RcfCliHandler.handlerShowRcfCodeUnitsCmd'

BasicCli.addShowCommandClass( ShowRcfCodeUnitsCmd )

# -------------------------------------------------------------------------------
# "show router rcf function location [ ( function FUNC_NAME ) | ( unit UNIT_NAME ) ]"
# ------------------------------------------------------------------------------
class ShowRcfFunctionLocationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show router rcf function location
            [ ( FUNC_KW FUNC_NAME ) | ( unit UNIT_NAME ) ]'''
   data = {
      'router': RcfCliTokens.routerKw,
      'rcf': RcfCliLib.rcfKw,
      'function': RcfCliTokens.functionKw,
      'location': RcfCliTokens.locationFunctionKw,
      'FUNC_KW': RcfCliTokens.functionFilterKw,
      'FUNC_NAME': RcfCliTokens.funcNameMatcher,
      'unit': RcfCliTokens.unitFilterKw,
      'UNIT_NAME': unitNameMatcher
   }

   cliModel = 'RcfCliModels.ShowRcfFunctionLocationModel'
   handler = 'RcfCliHandler.handlerShowRcfFunctionLocationCmd'

BasicCli.addShowCommandClass( ShowRcfFunctionLocationCmd )

def Plugin( entMan ):
   # rcfConfig is used by unitNameMatcher above
   rcfConfig = ConfigMount.mount( entMan, 'routing/rcf/config', 'Rcf::Config', 'wS' )
   gv.rcfConfig = rcfConfig
   gv.codeUnitScratchpadHelper = RcfCliHelpers.CodeUnitScratchpadHelper( rcfConfig )
   gv.openConfigScratchpadHelper = (
         RcfCliHelpers.OpenConfigFunctionScratchpadHelper( rcfConfig ) )
