#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Assert import assertNotIn
from CodeUnitMapping import CodeUnitKey
import RcfTypeFuture as Rcf

class RenderContext:
   """
   A global context for rendering RCF debugging information. This type should
   contain state required across all parts of the rendering process and across
   function invocations in a given RCF evaluation.

   Constructor Arguments:
      rcfCodeUnits (RcfDebugModels.RcfCodeUnitText):
         The CliModel ( CAPI model ) representation of the RCF text during the
         evaluation.
      rcfEval (RcfDebugModels.RcfDebugEvaluation):
         The CliModel ( CAPI model ) representation of the RCF evaluation details.
   """
   def __init__( self, rcfCodeUnits, rcfEval ):
      self.rcfCodeUnits = rcfCodeUnits
      self.rcfEval = rcfEval

   def getCodeLine( self, lineNumber, functionName, codeUnitKey ):
      """
      Retrieve a line of RCF code from the RcfDebugModels.RcfCodeUnitText.

      Return: (str) The line of code

      Args:
         lineNumber (int):
            The line number of the code to be retrieved.
         functionName (str):
            The name of the function the line is in.
         codeUnitKey (CodeUnitKey):
            The key identifying the code unit the line is in, and thus the line
            number is relative to.
      """
      # We should never attempt to fetch a code line for anything but a user defined
      # code unit
      assert codeUnitKey.isUserDefined(), "Unexpected domain for codeUnitKey"
      codeUnitName = codeUnitKey.codeUnitName
      functionText = self.rcfCodeUnits[ codeUnitName ].functions[ functionName ]
      return functionText.lines[ lineNumber ]

   def getInvocation( self, functionName, invocationIndex ):
      """
      Retrieve a CLI model ( CAPI model) representation of the RCF debugging
      information for a particular invocation of a function.

      Return: (RcfDebugModels.RcfDebugFunctionInvocation) The CLI model.

      Args:
         functionName (str):
            The name of the function that was invoked.
         invocationIndex (int):
            A function can be invoked multiple times during RCF evaluation. The
            evaluation of the called function may differ each time. As such a
            function may have multiple invocations present. This index (from an
            RcfDebugFunctionInvocationReference) represents which invocation is
            being requested.
      """
      functionEvaluation = self.rcfEval.functionEvaluations[ functionName ]
      return functionEvaluation.invocations[ invocationIndex ]

   def getEntryPointName( self ):
      """
      Retrieve the details of the initial/top level/main/entry point RCF function
      that was invoked.

      Return: ( str ) The function name.
      """
      return self.rcfEval.entryPoint

   def getEntryPointResult( self ):
      """
      Retrieve the overall result of the RCF evaluation.

      Return: ( RcfDebugModels.RcfDebugResult ) The result.
      """
      return self.rcfEval.result

   def getCodeUnitKeyForFunction( self, functionName ):
      """
      Retrieve the code unit key the function is defined in.

      Return: ( CodeUnitKey ) The code unit key.
      """
      codeUnitName = self.rcfEval.functionEvaluations[ functionName ].codeUnitName
      return CodeUnitKey( Rcf.Metadata.FunctionDomain.USER_DEFINED, codeUnitName )

   def isHiddenFunction( self, functionName ):
      """
      Retrieve whether the function should be hidden from the user.

      Return: bool
      """
      return ( self.rcfEval.functionEvaluations[ functionName ].functionDomain
               != Rcf.Metadata.FunctionDomain.USER_DEFINED )

class InvocationContext:
   """
   A function invocation specific context for rendering RCF debugging information.
   This type should contain state required across the various objects used to render
   a specific invocations of a function.

   Constructor Arguments:
      functionName (str):
         The name of the function this is the context for.
      codeUnitKey (CodeUnitKey):
         The key identifying the domain and name of the code unit where the function
         is defined.
      invocationIndex (int):
         The index of the invocations this is the context for.
      callStackDepth (int):
         How deep in the call stack this invocation is.
      renderContext (RenderContext):
         The global rendering context, provides access to the CliModel.
   """
   def __init__( self, functionName, codeUnitKey, invocationIndex,
                 callStackDepth, renderContext ):
      self.functionName = functionName
      self.codeUnitKey = codeUnitKey
      self.invocationIndex = invocationIndex
      self.callStackDepth = callStackDepth
      self.renderContext = renderContext

class InvocationCodeManager:
   """
   This type is responsible for providing access to the RCF text of a function and
   maintaining the state of where the renderer is in the functions text.

   The expectation is that the caller will retrieve chunks of code for a given line
   by lengths (popCode), fragment lengths (popCodeForFragment), or up to a specific
   column (popLine(upToOffset=..)) which is likely the start of the next fragment.

   The caller MUST call popLine() without arguments at the end to retrieve the
   remainder of the line if any. Otherwise the codeManager will assert when advancing
   to a new line as the previous line was not fully read. This ensures we don't miss
   trailing ends of code in the output.

   Constructor Arguments:
      context (InvocationContext):
         Context object for a specific invocation of a function, used to acquire a
         given line of code.
   Attributes:
      currentLine (str):
         The current line of RCF code being rendered.
      currentLineNumber (int):
         The line number of code being rendered.
      currentColumnOffset (int):
         The columnar position of the renderer within the current line of code.
         Once the position reaches the end of the line it is set to None. This
         is used to help ensure the same code is not rendered twice.
         Starts at 0.

   """
   def __init__( self, context ):
      self.context = context
      self.currentLine = ""
      self.currentLineNumber = 0
      self.currentColumnOffset = None

   def getCodeLine( self, lineNumber ):
      return self.context.renderContext.getCodeLine( lineNumber,
                                                     self.context.functionName,
                                                     self.context.codeUnitKey )

   def advanceLineNumber( self, lineNumber ):
      """
      Advance to the following line in the RCF text.

      Note: getCode() must have been invoked for any previous line
            before advancing to handle trailing characters.

      Args:
         lineNumber (int):
            line of text to advance to.
      """
      assert self.currentColumnOffset is None, "Entire previous line must be used"
      assert lineNumber > self.currentLineNumber, ( "Line numbers must increase "
                                                    "within a function" )
      self.currentLineNumber = lineNumber
      self.currentColumnOffset = 0
      self.currentLine = self.getCodeLine( lineNumber )

   def popCode( self, length ):
      """
      Retrieve the next N (length) of characters after the current column number
      from the current line.

      Return: (str) The chars/part of the current line.

      Args:
         length (int):
            Number of characters to retrieve.
      """
      assert length is not None and length > 0
      assert self.currentColumnOffset is not None
      assert self.currentColumnOffset >= 0
      newColumnNumber = self.currentColumnOffset + length
      assert newColumnNumber <= len( self.currentLine )
      chunk = self.currentLine[ self.currentColumnOffset : newColumnNumber ]
      self.currentColumnOffset = newColumnNumber
      return chunk

   def popCodeForFragment( self, fragment ):
      """
      Helper method to retrieve the code chunk for a given fragment.

      Return: (str) The chars/part of the current line related to the fragment.

      Args:
         fragment (RcfDebugModels.RcfDebugFragment):
            The fragment for which to retrive code.
      """
      assert fragment.location.line == self.currentLineNumber
      return self.popCode( fragment.location.length )

   def popLine( self, upToOffset=None ):
      """
      Retrieve all characters in the range of the current column number to the
      specified column number from the current line.

      Return: (str) The chars/part of the current line.

      Args:
         upToOffset (int):
            The column offset to retrieve character up to. The end of the line if
            None.
      """
      assert self.currentColumnOffset >= 0
      if upToOffset is None:
         remainder = len( self.currentLine ) - self.currentColumnOffset
         chunk = self.popCode( remainder ) if remainder else ""
         # This ensures renderer must now advance a line before accessing code again
         self.currentColumnOffset = None
      elif upToOffset == self.currentColumnOffset:
         chunk = ""
      elif upToOffset < self.currentColumnOffset:
         assert False, "codeManager has already advanced beyond this offset"
      else:
         chunk = self.popCode( upToOffset - self.currentColumnOffset )
      return chunk

   def peekAheadInLine( self, length ):
      """
      Retrieve the next n characters in a line without advancing the position of the
      current column number such that it can be read again. Used by fragment handlers
      to peek at the future.

      Return: (str) The chars/part of the current line.

      Args:
         length (int):
            The number of characters to retrieve.
      """
      assert length > 0
      return self.currentLine[ self.currentColumnOffset :
                               self.currentColumnOffset + length ]

class Annotations:
   """
   Collection of types used to represent various annotations that can be rendered as
   part of the RCF debug output. These annotations correspond to some fragment of
   RCF text. The getText method on each type accepts a length which would be the
   length of the fragment of RCF text.
   """

   class AnnotationBase:
      @staticmethod
      def getText( length ):
         raise NotImplementedError

   class LineBar( AnnotationBase ):
      lineChar = None

      @classmethod
      def getLine( cls, length ):
         assert length >= 0
         return cls.lineChar * length

      @classmethod
      def getText( cls, length ):
         return cls.getLine( length )

   class BlankBar( LineBar ):
      lineChar = " "

   class EvalBar( LineBar ):
      lineChar = "-"

   class PrefixedBar( LineBar ):
      prefix = None
      @classmethod
      def getText( cls, length ):
         assert length > 0
         return cls.prefix + cls.getLine( length - 1 )

   class TrueBar( PrefixedBar ):
      prefix = "T"
      lineChar = "-"

   class FalseBar( PrefixedBar ):
      prefix = "F"
      lineChar = "-"

   class UnknownBar( PrefixedBar ):
      prefix = "U"
      lineChar = "-"

   resultBarMapping = {
      "true": TrueBar,
      "false": FalseBar,
      "unknown": UnknownBar,
   }

   class ExitBar( PrefixedBar ):
      prefix = "X"
      lineChar = "-"

   @staticmethod
   def fromResult( result ):
      if result is None:
         return Annotations.ExitBar
      return Annotations.resultBarMapping[ result.value ]

   class ContinuationBar( PrefixedBar ):
      prefix = "\\"
      lineChar = "-"

   class OpenBracketAnnotation( AnnotationBase ):
      @staticmethod
      def getText( length ):
         assert length > 0
         return Annotations.BlankBar.getText( length - 1 ) + "("

   class CloseBracketAnnotation( AnnotationBase ):
      @staticmethod
      def getText( length ):
         assert length > 0
         return ")" + Annotations.BlankBar.getText( length - 1 )

class Scratchpad:
   """
   Type representing a single line of code or annotations in the InvocationPrinter,
   passed to ScratchpadRenderer when the line is complete.

   Note: newlines must NOT be written to the scratchpad.

   Attributes:
      contents_ (str):
         The text of code or annotations in the scratchpad. The attribute is private
         and text written to it should not be modified later, only appended to.
      renderEvenIfEmpty (str):
         Typically a blank line of annotations is not rendered. Explicit
         NewlineRenderer objects are used to insert spacing. However in rare cases a
         fragment may require the annotation line should be rendered even if it is
         blank.
   """

   def __init__( self ):
      self._contents = ""
      self.renderEvenIfEmpty = False

   def write( self, rhs ):
      assertNotIn( '\n', rhs )
      self._contents += rhs
      return self

   def trimTrailingChar( self ):
      self._contents = self._contents[ : -1 ]
      return self

   def read( self ):
      return self._contents

   def empty( self ):
      return self._contents is None or not self._contents.strip()

   def __len__( self ):
      return len( self._contents )
