#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Dict, Str, Enum, Int
from ReflectorTypes import ( directionToString, macActionToString,
                             hwPrgmStatusToString )

class ReflectorInterfaceModel( Model ):
   _sortOrder = Int( help='Private: lexicographical sort order for text show' )
   direction = Enum( values=( directionToString.values() ),
                     help='Direction of the reflector interface' )
   macAction = Enum( values=( macActionToString.values() ),
                     help='MAC action of the reflector interface' )
   status = Enum( values=( hwPrgmStatusToString.values() ),
                  help='Status of the reflector interface' )
   statusReason = Str( help='Reason for the status of the reflector interface' )

   # Need method to access private attribute _sortOrder
   # Has to be a method, because CliModel:Model does not allow @property
   def sortOrder( self ):
      return self._sortOrder

class ReflectorModel( Model ):
   reflectorInterfaces = Dict(
      keyType=str, valueType=ReflectorInterfaceModel,
      help='Mapping between reflector interface identifiers and their '
           'config and status' )

   def render( self ):
      spacer = False

      for key, reflectorInterface in sorted( self.reflectorInterfaces.items(),
                                             key=lambda kv: kv[ 1 ].sortOrder() ):
         # Emit a blank line between multiple reflector interface shows
         if spacer:
            print()
         else:
            spacer = True

         # pylint: disable-next=consider-using-f-string
         print( 'Ethernet reflector interface: %s' % key )
         # pylint: disable-next=consider-using-f-string
         print( '   Direction: %s' % reflectorInterface.direction )
         # pylint: disable-next=consider-using-f-string
         print( '   MAC action: %s' % reflectorInterface.macAction )
         if reflectorInterface.statusReason:
            # pylint: disable-next=consider-using-f-string
            reason = ' (%s)' % reflectorInterface.statusReason
         else:
            reason = ''
         print( f'   Status: {reflectorInterface.status}{reason}' )
