# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

import os
import time
import Cell
import LazyMount
from CliModel import Bool, Float, List, Model, Str
import CliPlugin.TechSupportCli
from CliPlugin.ReloadCauseCliLib import (
   numLines,
   panicBreadCrumbsFile,
   debugFileLines,
   usingMSRC
)
import ReloadConsts as RC

entityManager = None
multiSourceReloadCause = None
fileBasedResetCauseHistory = None

# ------------------------------------------------------
# The "show reload cause" command, in "unpriv" mode
#
# The full syntax of this command is:
#
#    show reload cause
#-------------------------------------------------------

class ReloadCauses( Model ):
   class ReloadCause( Model ):
      description = Str( help="Description of the reset cause" )
      timestamp = Float( help="Time of the reload", optional=True )
      recommendedAction = Str( help="Action that is recommended", optional=True )
      debugInfo = List( valueType=str, help="Information from debug file",
                        optional=True )
      debugInfoIsDir = Bool( help="True if debug info is a directory",
                             default=False, optional=True )

   resetCauses = List( valueType=ReloadCause, help="List of reload causes" )
   full = Bool( help="Partial or full log", default=False, optional=True )
   kernelCrashData = List( valueType=str, help="Kernel Panic EEPROM crash data",
                           optional=True )

   def renderSystemDebugInfo( self ):
      if self.full and self.kernelCrashData:
         print( "" )
         print( "Kernel Panic EEPROM crash data:" )
         print( "-------------------------------" )
         for line in self.kernelCrashData:
            print( line )

   def render( self ):
      if not self.resetCauses:
         print( "Reload Cause:" )
         print( "-------------" )
         print( RC.unknownError )
         print( "" )
         print( "Debugging Information:" )
         print( "----------------------" )
         for line in debugFileLines( full=self.full ):
            print( line )
         self.renderSystemDebugInfo()
         return

      i = 0
      for resetCause in self.resetCauses:
         if i != 0:
            print( "" )
         i += 1
         print( "Reload Cause %d:" % i ) # pylint: disable=consider-using-f-string
         print( "-------------------" )
         print( resetCause.description )
         print( "" )
         print( "Reload Time:" )
         print( "------------" )
         timestampStr = ""
         if resetCause.timestamp:
            try:
               # pylint: disable-next=consider-using-f-string
               timestampStr = "Reload occurred at %s." % time.strftime(
                              "%a %b %d %H:%M:%S %Y %Z",
                              time.localtime( resetCause.timestamp ) )
            except OverflowError:
               timestampStr = "Not available."
         else:
            timestampStr = "Not available."
         print( timestampStr )
         print( "" )
         print( "Recommended Action:" )
         print( "-------------------" )
         print( resetCause.recommendedAction or "None available." )
         print( "" )
         print( "Debugging Information:" )
         print( "-------------------------------" )
         if not resetCause.debugInfo:
            print( "None available." )
         else:
            if not resetCause.debugInfoIsDir and not self.full:
               # pylint: disable-next=consider-using-f-string
               print( "NOTE: Displaying last %d lines of crash log" % numLines )

            print( "\n".join( resetCause.debugInfo ) )

      self.renderSystemDebugInfo()

def newResetCauseModel( reloadCauseDetails, full ):
   resetCauseModel = ReloadCauses.ReloadCause()
   resetCauseModel.description = reloadCauseDetails.description
   resetCauseModel.timestamp = reloadCauseDetails.timestamp
   resetCauseModel.recommendedAction = reloadCauseDetails.recommendedAction

   if reloadCauseDetails.debugInfoFilename:
      try:
         # if the filename is a directory, just print out the directory
         debugInfoFile = reloadCauseDetails.debugInfoFilename
         if os.path.isdir( debugInfoFile ):
            resetCauseModel.debugInfo = [ debugInfoFile ]
            resetCauseModel.debugInfoIsDir = True
         else:
            resetCauseModel.debugInfo = debugFileLines( full=full,
                                                        path=debugInfoFile )
      except OSError:
         resetCauseModel.debugInfo = None
   else:
      resetCauseModel.debugInfo = None

   return resetCauseModel

def mergeResetCauses( model ):
   userReload = watchdogReload = None
   resetCauseModel = None
   thermalShutdown = None
   for resetCause in model.resetCauses:
      # Any thermal related shutdown should supersede any other reload cause.
      if resetCause.description == RC.thermoInsufficientFansDescription or \
         resetCause.description == RC.overTempDescription:
         thermalShutdown = resetCause
      if resetCause.description == RC.reloadDescription:
         userReload = resetCause
      # 2RU systems append an origin to indicate whether the upper or lower board
      # generated the reload cause. Remove an origin from description
      # if it's present.
      if resetCause.description.split( ' on the' )[ 0 ] in \
            ( RC.wDescription, RC.wkDescription ):
         watchdogReload = resetCause

   if thermalShutdown:
      # when reload cause is thermalShutdown, ignore other reload causes
      del model.resetCauses[:]
      model.resetCauses.append( thermalShutdown )
   elif userReload and watchdogReload:
      resetCauseModel = ReloadCauses.ReloadCause()
      resetCauseModel.description = RC.wuDescription
      resetCauseModel.timestamp = userReload.timestamp
      resetCauseModel.recommendedAction = RC.noRecommendedAction
      resetCauseModel.debugInfo = None
      model.resetCauses.remove( userReload )
      model.resetCauses.remove( watchdogReload )
      model.resetCauses.append( resetCauseModel )

def doShowReloadCause( mode, args ):
   full = args.get( 'full' )
   retModel = ReloadCauses( full=bool( full ) )

   if usingMSRC( entityManager ):
      if not multiSourceReloadCause.description:
         return retModel
      resetCauseModel = newResetCauseModel( multiSourceReloadCause, full )
      retModel.resetCauses.append( resetCauseModel )
   else:
      rebootHistory = fileBasedResetCauseHistory.rebootHistory
      resetCauses = rebootHistory.get( 0 )
      if not resetCauses:
         return retModel
      for resetCause in resetCauses.causeQ.values():
         resetCauseModel = newResetCauseModel( resetCause, full )
         retModel.resetCauses.append( resetCauseModel )
      mergeResetCauses( retModel )

   if os.path.exists( panicBreadCrumbsFile ) and full:
      retModel.kernelCrashData = debugFileLines( full=full,
                                                 path=panicBreadCrumbsFile )
   else:
      retModel.kernelCrashData = None

   return retModel

#-------------------------------------------------------------
# Register show reload cause command into "show tech-support".
#-------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:01:00',
   cmds=[ 'show reload cause full' ],
   summaryCmds=[ 'show reload cause full' ] )

def Plugin( em ):
   global fileBasedResetCauseHistory
   global multiSourceReloadCause
   global entityManager
   entityManager = em
   # BUG24704 - should look at reload causes from all cells
   fileBasedResetCauseHistory = LazyMount.mount(
         em, f"cell/{Cell.cellId()}/sys/reset/history",
         "System::ResetCauseHistory", "ri" )
   multiSourceReloadCause = LazyMount.mount(
         em, f"cell/{Cell.cellId()}/sys/reload/cause",
         "System::ReloadCause", "ri" )
