# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import time
import LazyMount
from CliModel import Float, List, Dict, Model, Str, Bool
from CliPlugin.ReloadCauseCliLib import debugFileLines

resetCauseHistory = None
numHistory = 10

# ------------------------------------------------------
# The "show reload cause history" command, in "unpriv" mode
#
# The full syntax of this command is:
#
#    show reload cause history
#-------------------------------------------------------

class ReloadCausesHistory( Model ):

   class ReloadCauseList( Model ):
      class ReloadCause( Model ):
         description = Str( help="Description of the reset cause" )
         timestamp = Float( help="Time of the reload", optional=True )
         recommendedAction = Str( help="Action that is recommended", optional=True )
         debugInfo = List( valueType=str, help="Information from debug file",
                           optional=True )
         debugInfoIsDir = Bool( help="True if debug info is a directory",
                                default=False, optional=True )
      resetCauses = List( valueType=ReloadCause, help="List of reload causes" )

   resetHistory = Dict( keyType=int,
                        valueType=ReloadCauseList,
                        help="History of reload causes" )

   def render( self ):
      if not self.resetHistory:
         print( "Unknown Reload History:" )
         print( "----------------------" )
         debugLines = debugFileLines()
         if debugLines:
            print( "" )
            print( "Debugging Information:" )
            print( "----------------------" )
            for line in debugLines:
               print( line )
         return

      for k, resetCauseList in self.resetHistory.items():
         #print "Reboot Instance %d:" % k
         #print "--------------------"
         i = 0
         for resetCause in resetCauseList.resetCauses:
            #if i != 0:
            print( "" )
            i += 1
            # pylint: disable-next=consider-using-f-string
            print( "Reboot Instance %d: Reload Cause %d:" % ( k, i ) )
            print( "------------------------------------" )
            print( resetCause.description )
            print( "" )
            print( "Reload Time:" )
            print( "------------" )
            timestampStr = ""
            if resetCause.timestamp:
               # pylint: disable-next=consider-using-f-string
               timestampStr = "Reload occurred at %s." % time.strftime(
                           "%a %b %d %H:%M:%S %Y %Z",
                           time.localtime( resetCause.timestamp ) )
            else:
               timestampStr = "Not available."
            print( timestampStr )
            print( "" )
            print( "Recommended Action:" )
            print( "-------------------" )
            print( resetCause.recommendedAction or "None available." )
            print( "" )
            print( "Debugging Information:" )
            print( "----------------------" )
            ln = "None available." if not resetCause.debugInfo else \
                                    "\n".join( resetCause.debugInfo )
            print( ln )

def doShowReloadCauseHistory( mode, args ):
   rebootHistory = resetCauseHistory.rebootHistory
   retModel = ReloadCausesHistory()

   if not rebootHistory:
      return retModel

   for key, reboot in rebootHistory.items():
      for cause in reboot.causeQ.values():
         resetCauseModel = ReloadCausesHistory.ReloadCauseList.ReloadCause()
         resetCauseModel.description = cause.description
         resetCauseModel.timestamp = cause.timestamp
         resetCauseModel.recommendedAction = cause.recommendedAction
         if cause.debugInfoFilename:
            try:
               # if the filename is a directory, just print out the directory
               if os.path.isdir( cause.debugInfoFilename ):
                  resetCauseModel.debugInfo = [ cause.debugInfoFilename ]
                  resetCauseModel.debugInfoIsDir = True
               else:
                  resetCauseModel.debugInfo = debugFileLines(
                        path=cause.debugInfoFilename )
            except OSError:
               resetCauseModel.debugInfo = None
         else:
            resetCauseModel.debugInfo = None

         resetCauseListModel = ReloadCausesHistory.ReloadCauseList()

         if key in retModel.resetHistory:
            resetCauseListModel = retModel.resetHistory[key]
         else:
            resetCauseListModel.resetCauses = [ ]

         resetCauseListModel.resetCauses.append(resetCauseModel)
         retModel.resetHistory[key] = resetCauseListModel

   return retModel


#-------------------------------------------------------------
# Register show reload cause history command into "show tech-support".
#-------------------------------------------------------------
import CliPlugin.TechSupportCli # pylint: disable=wrong-import-position

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd( 
   '2010-01-01 00:01:00',
   cmds=[ 'show reload cause history' ] )

def Plugin( entityManager ):
   global resetCauseHistory
   import Cell # pylint: disable=import-outside-toplevel
   resetCauseHistory = LazyMount.mount(
         # pylint: disable-next=consider-using-f-string
         entityManager, "cell/%d/sys/reset/history" % Cell.cellId(),
         "System::ResetCauseHistory", "ri" )
