# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function

from datetime import datetime, timedelta
import os
import re
import sys
import time

import BasicCli
import BasicCliUtil
import Cell
import CEosHelper
import CliCommand
import CliMatcher
from CliOptimizeSwiLib import maybeOptimizeSwi
from CliModel import (
   Bool,
   Float,
   Model,
   Str
)
import CliParser
import DateTimeRule
import FileUrl
from FpgaUtil import printToConsole
import LazyMount
import SimpleConfigFile
import ShowCommand
import Tac
import Tracing
import Url
# Dummy import to get the FlashUrl dependency which is needed by Url.localBootConfig
# BUG122265
import UrlPlugin.FlashUrl # pylint: disable-msg=W0611

__defaultTraceHandle__ = Tracing.Handle( "ReloadCli" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

config = None
status = None
powerFuseConfig = None
sysdbRedSupStatus = None

# A simple script that sets up environment to run boot0

#  For ASU+/ASU2, most of kernel args are inherited from prevoius
#  boot/aboot ( from /proc/cmdline ), functions used by boot0 in aboot
#  context are defined as dummy. kernel args per EOS image are setup by
#  boot0 which is invokded by Aboot or ASU boot.
# 
bootScriptTemplate = """
#!/bin/sh
qparseconfig() {
   return
}

ifget() {
   return
}

writebootconfig() {
   return
}

export arch=i386
export swipath=%s

. /tmp/boot0
"""

#-------------------------------------------------------
# The "reload" command, in "enable" mode
#  reload [ power ] [ now ] 
#  reload [ power ] in hhh:mm|mmm [ reason text ]    
#  reload [ power ] at hh:mm [ month day ] [ reason text ]
#  show reload
#  reload cancel
#  [ no|default ] reload [ text ]
#
# On modular system reload can be qualified with
# two additional keywords "all" and "peer"
#  Ex reload [ all | peer ] [ above variants ]
#     reload peer now 
#-------------------------------------------------------
def notCeosGuard( mode, token ):
   if CEosHelper.isCeos():
      return CliParser.guardNotThisPlatform
   return None

# This is used by AsuCli
reloadKwNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      "reload",
      helpdesc='Reboot the system' ),
   guard=notCeosGuard )

reloadShowKwNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      "reload",
      helpdesc='Display system reload status' ),
   guard=notCeosGuard )

forceKwMatcher = CliMatcher.KeywordMatcher(
   "force",
   helpdesc="Perform action immediately without prompting" )

powerCycleMatcher = CliMatcher.KeywordMatcher(
   "power",
   helpdesc="Power-cycle the system" )

def modularActiveSupeGuard( mode, token ):
   if Cell.cellType() == "supervisor":
      if Cell.cellId() == Cell.activeCell():
         return None
      else:
         return CliParser.guardNotThisSupervisor
   else:
      return CliParser.guardNotThisPlatform

def modularActiveSupeOrFixedGuard( mode, token ):
   if Cell.cellType() == "supervisor":
      return modularActiveSupeGuard( mode, token )
   return None

allKwNode = CliCommand.Node(
   CliMatcher.KeywordMatcher( "all",
                              helpdesc="Reload all supervisors" ),
   guard=modularActiveSupeOrFixedGuard )

peerKwNode = CliCommand.Node(
   CliMatcher.KeywordMatcher( "peer",
                              helpdesc="Reload peer supervisor" ),
   guard=modularActiveSupeGuard )

# Hook for functions to run before reloading. If any of these functions
# return false, cancel the reload. Priorities in the list are ordered
# from running first to running last.
_reloadPriorities = [ "RUN_FIRST", "FILE_MODIFICATION", "RUN_LAST" ]
_reloadPrioritiesExtension = [ "RUN_AFTER_CONFIRMATION" ]
_reloadHook = dict( ( priority, [] ) for priority in _reloadPriorities +
                                                     _reloadPrioritiesExtension )

# Register a function to be called before reloading. Filtering for
# tests will be performed on the name given.
#
# Hooks are normally not called with "reload now/force", unless the hook
# is registered with always=True.
def registerReloadHook( hook, name, priority, always=False ):
   if "RELOAD_PLUGIN_REGEX" in os.environ:
      if re.match( os.environ[ "RELOAD_PLUGIN_REGEX" ], name ):
         _reloadHook[ priority ].append( ( hook, always ) )
   else:
      _reloadHook[ priority ].append( ( hook, always ) )

# Return a list of extensions with the most important coming first
def getReloadHooks( reloadConfirmed=False, force=False ):
   priorities = None
   if not reloadConfirmed:
      priorities = _reloadPriorities
   else:
      priorities = _reloadPrioritiesExtension
   exts = []
   for priority in priorities:
      for ext, always in _reloadHook[ priority ]:
         if always or not force:
            exts.append( ext )
   return exts

def answerMatches( answer, choice ):
   # allow the answer to be a prefix of the choice, e.g. y is allowed for yes,
   # but the prefix can't be empty
   if not answer:
      return False
   if choice.startswith( answer.lower() ):
      return True
   return False

def isReloadScheduled():
   return status.reloadTime != Tac.endOfTime

def isPeerState( expPeerState ):
   return sysdbRedSupStatus.peerState == expPeerState

def getImagePath( mode ):
   # pylint: disable-msg=E1103
   url = FileUrl.localBootConfig( mode.entityManager, mode.session_.disableAaa_ )
   bootConfigFilename = url.realFilename_
   simpleConfigFile = SimpleConfigFile.SimpleConfigFileDict( bootConfigFilename,
                                                             autoSync=True )
   softwareImage = simpleConfigFile.get( 'SWI', '(not set)' )
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   imageFile = Url.parseUrl( softwareImage, context )

   return imageFile, softwareImage

# Returns the number of seconds since epoch, reload should actually happen
# Takes schedule which is "at hh:mm [ month day ]" 
# as tuple ( "at", ( hh, mm ) , ( month, day ) )
# If there is problem such as invalid time then -1 is returned
def getTime( schedule ):
   hours, minutes, year, month, day = 0, 0, 0, 0, 0

   if schedule[ 0 ] == "at":
      hours = schedule[ 1 ][ 0 ]
      minutes = schedule[ 1 ][ 1 ]
      year = datetime.now().year
      if schedule[ 2 ] is not None:
         month = schedule[ 2 ][ 0 ]
         day = schedule[ 2 ][ 1 ]
         if month < datetime.now().month:
            year = year + 1
            # Use case of incrementing year 
            # Current date is December 30 and reload should be scheduled on Jan 1  
      else:    # current day
         day = datetime.now().day
         month = datetime.now().month
         
      if hours >= 24 or minutes >= 60:
         return -1
      try:   
         datetime_time = datetime( year, month, day, hours, minutes, 0 )
      except ValueError:
         return -1

      if year == datetime.now().year and month == datetime.now().month:
         if day < datetime.now().day:
            return -1
         elif( day == datetime.now().day and
                 ( hours < datetime.now().hour or
                    ( hours == datetime.now().hour and 
                      minutes < datetime.now().minute ) ) ):
            # reload is scheduled for current day 
            # but time specified is less than current time 
            # we schedule reload for tomorrow at same time
            datetime_time = datetime_time + timedelta( days=1 )   
      return time.mktime( datetime_time.timetuple() )
   else:
      return -1 # ctrl comes here if grammer fails

def reloadPeerSupervisor( quiet=False ):
   import Fru # pylint: disable=import-outside-toplevel
   name = "Supervisor%d" % ( 3 - Fru.slotId() )

   if isPeerState( "notInserted" ) or isPeerState( "unknownPeerState" ):
      if not quiet:
         print( "Peer supervisor is not present." )
      return

   try:
      powerFuseConfig.powerOffRequested[ name ] = "user configuration"
      Tac.waitFor( lambda: isPeerState( "poweredOff" ),
                   description="peer supervisor to be powered off",
                   sleep=True, timeout=60 )
   except Tac.Timeout as timeout:
      if quiet:
         raise timeout
      print( "Unable to power off peer supervisor" )
      del powerFuseConfig.powerOffRequested[ name ]
      return

   try:
      del powerFuseConfig.powerOffRequested[ name ]
      Tac.waitFor( lambda: isPeerState( "poweringOn" ) or isPeerState( "inserted" ),
                   description="peer supervisor to be powered on",
                   sleep=True, timeout=60 )
   except Tac.Timeout as timeout:
      if quiet:
         raise timeout
      print( "Unable to power on peer supervisor." )
      return

   if not quiet:
      print( "Peer supervisor has been restarted." )

def isReloadNow( *args, **kwargs ):
   if "now" in kwargs and kwargs[ "now" ]:
      return True
   if "force" in kwargs and kwargs[ "force" ]:
      return True
   return False

def isFlashVfat():
   try:
      Tac.run( [ "df", "-t", "vfat", f"{Url.fsRoot()}/flash" ],
               stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      return True
   except Tac.SystemCommandError:
      return False

def writeToPreReloadSaveErrorFile( filename, error ):
   try:
      with open( filename, "w" ) as errorFile:
         errorFile.write( str( error ) + "\n" )
   except Exception as e: # pylint: disable-msg=broad-except
      try:
         printToConsole( str( error ) + "\n" )
         printToConsole( str( e ) + "\n" )
      except: # pylint: disable-msg=bare-except
         pass

def getIntFromEnv( envVar, default ):
   try:
      return int( os.environ.get( envVar, default ) )
   except ValueError:
      return int( default )

def deletePreReloadLogs( errorPath, savePath ):
   Tac.run( [ "timeout", "-k", "3", "3", "rm", "-f",
               errorPath, savePath ],
            asRoot=True, ignoreReturnCode=True,
            stdout=Tac.DISCARD, stderr=Tac.DISCARD )

def savePreReloadLogs( prefix="pre_reload" ):
   # see BUG323324 for reasoning behind saving some logging information
   # memory values in this pre-reload area of code should be expressed in kibibytes
   filesToSync = set()
   try:
      preReloadLogsErrorPath = \
         f"{Url.fsRoot()}/flash/debug/{prefix}_logs_save_errors"
      preReloadLogsSaveErrorFile = os.environ.get(
         f"{prefix.upper()}_LOGS_ERROR_FILE", f"{prefix}_logs_save_errors" )
      preReloadLogsSaveFile = os.environ.get(
         f"{prefix.upper()}_LOGS_SAVE_FILE", f"{prefix}_logs.tgz" )
      preReloadLogsDestDir = os.environ.get(
         f"{prefix.upper()}_LOGS_DEST_DIR", f"{Url.fsRoot()}/flash/debug" )
      try:
         os.makedirs( preReloadLogsDestDir )
      except OSError as e:
         if not os.path.isdir( preReloadLogsDestDir ):
            raise e
      preReloadLogsErrorPath = os.path.join( preReloadLogsDestDir,
                                             preReloadLogsSaveErrorFile )
      preReloadLogsSavePath = os.path.join( preReloadLogsDestDir,
                                            preReloadLogsSaveFile )
      deletePreReloadLogs( preReloadLogsErrorPath, preReloadLogsSavePath )

      preReloadSaveBuffer = getIntFromEnv( "PRE_RELOAD_SAVE_BUFFER", "1024" )

      out = Tac.run( [ "df", "--block-size=1024", preReloadLogsDestDir ],
                     asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      available = int( out.split( "\n" )[ 1 ].split()[ 3 ] )
      preReloadSaveMinSize = getIntFromEnv( "PRE_RELOAD_SAVE_MIN_SIZE", "512" )
      saveSize = ( available - preReloadSaveBuffer )
      if saveSize < preReloadSaveMinSize:
         filesToSync.add( preReloadLogsErrorPath )
         errMsg = ( f"{available}KiB available on {Url.fsRoot()}/flash, "
                    f"too little space to save {prefix} logs" )
         writeToPreReloadSaveErrorFile( preReloadLogsErrorPath, errMsg )
      else:
         preReloadLogs = "/var/log/messages"
         if not isFlashVfat():
            # avoid writing too much on VFAT before reload
            preReloadLogs += " /var/log/agents/ /var/log/qt"
         # include account logs
         logFiles = Tac.run( [ "ls", "/var/log" ], stdout=Tac.CAPTURE ).split()
         for f in logFiles:
            if f.startswith( "account" ):
               preReloadLogs += " " + f
         preReloadLogs = os.environ.get( "PRE_RELOAD_LOGS", preReloadLogs )
         preReloadSaveTimeout = getIntFromEnv( "PRE_RELOAD_SAVE_TIMEOUT", "9" )
         preReloadSaveCmdZero = ( "timeout -k %d %d bash -c" %
                                  ( preReloadSaveTimeout, preReloadSaveTimeout ) )
         preReloadSaveCmdOne = ( "tar -zvc %s | head --bytes=%dK > %s" %
                                 ( preReloadLogs, saveSize, preReloadLogsSavePath ) )
         # Do this before running the command since it might raise an exception
         filesToSync.add( preReloadLogsSavePath )
         Tac.run( preReloadSaveCmdZero.split() + [ preReloadSaveCmdOne ],
                  asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
   except Exception as e: # pylint: disable-msg=broad-except
      filesToSync.add( preReloadLogsErrorPath )
      writeToPreReloadSaveErrorFile( preReloadLogsErrorPath, e )
   # end pre-reload log save

   # We just wrote a file to flash before reload; make sure we fsync the file
   # to avoid corruption.
   filesToSync = [ fname for fname in filesToSync if os.path.isfile( fname ) ]
   if filesToSync:
      Tac.run( [ 'SyncFile' ] + filesToSync, asRoot=True,
               ignoreReturnCode=True )

# Silly pylint, don't complain about space after = in decorators:
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@BasicCliUtil.EapiIncompatible( overrideFn=isReloadNow,
                            additionalMsg="To reload the machine over the API, "
                            "please include the `force` keyword." )
def doReload( mode, supe=None, power=False, now=False, schedule=None,
              reloadReason=None, force=False ):
   # reload and reload power, both just do a linux
   # reboot. During the reboot shutdown sequence, after some
   # minimal cleanup, we power cycle the system by writing
   # the system reset register of the scd.
   interval = -1 
   seconds = -1
   reason = ""

   # either 'now' or 'force' disable prompting
   unconditionally = now or force

   # Reset to ignore the supe parameter if this is not a modular system - the
   # user might have used the 'reset all' variant on a fixed system.
   if Cell.cellType() != 'supervisor':
      supe = None

   if reloadReason is not None:
      reason = reloadReason

   imageFile, softwareImage = getImagePath( mode )
   if imageFile.localFilename() and os.path.exists( imageFile.realFilename_ ):
      # We want to optimize swi image as soon as possible. This way any checks
      # are done against optimized image, which is the one that will be booted.
      maybeOptimizeSwi( imageFile.realFilename_, mode )
      Tac.run( [ 'SyncFile', imageFile.realFilename_ ], asRoot=True )

   if schedule is not None:  
      if isReloadScheduled():
         print( "Reload is already scheduled, please cancel it" )
         return
      # we can schedule reload
      hhh, minute = 0, 0
      if schedule[ 0 ] == "in":
         minute = schedule[ 1 ][ -1 ] # last element is mmm or mm
         if len( schedule[ 1 ] ) > 1:
            hhh = schedule[ 1 ][ 0 ]
            if minute >= 60: # hhh:mm rule imposes 60 minute limit for mm
               print( "Invalid time/date" )
               return  
         interval = ( hhh * 3600 ) + ( minute * 60 )
      else:
         seconds = getTime( schedule )
         if seconds < 0:
            print( "Invalid time/date" )
            return

   # Assume that we can reboot. If any extensions return false, abort reboot
   for ext in getReloadHooks( force=unconditionally ):
      if not ext( mode, power, unconditionally ):
         print( "Cannot shutdown gracefully, use the `force` keyword to bypass" )
         return

   sys.stdout.flush()

   if not unconditionally: # either reload is scheduled or bare reload is invoked
      # pylint: disable-msg=E1103
      if not imageFile.exists():
         if not BasicCliUtil.confirm( mode, "The boot image (%s) is not present. " 
                                  "Are you sure you want to reload? [confirm]" 
                                  % softwareImage ):
            return

      if not BasicCliUtil.confirm( mode, "Proceed with reload? [confirm]" ):
         return

   if supe and not schedule: # User has entered either "all" or "peer" keyword 
      reloadPeerSupervisor()
      if supe != "all":
         # Only peer must be reloaded 
         return

   if schedule is None:
      for ext in getReloadHooks( reloadConfirmed=True, force=unconditionally ):
         if not ext( mode, power, unconditionally ):
            if not unconditionally:
               print( "Cannot shutdown gracefully, "
                      "use the `force` keyword to bypass" )
               return

   if schedule is not None:  
      config.reason = reason
      config.all = bool( supe )
      if seconds < 0:
         config.reloadTime = interval + Tac.now()
      else:
         config.reloadTime = ( seconds - time.time() ) + Tac.now()
         # This will be negative when we schedule reload to happen in
         # in next few minutes and user takes lot of time to answer prompts
         # ( It rarely ever happens ). But we just reload immediately. 
      try:
         Tac.waitFor( isReloadScheduled,
                      description="reload to get scheduled",
                      sleep=True
                    )
         # In case we timeout or user presses Ctrl+C while we are waiting
         # we are not sure whether reload got scheduled or not.
         # It is left to user to decide further action
         # As a special case, if Fru restarts after config.reloadTime is changed
         # but before reactor fires, handleInitialized which handles Fru restart 
         # will retain present state( means no reload is scheduled ).
         # We don't look at config entity inside handleInitialized as we are 
         # not sure whether user is still waiting or has terminated the
         # cmdline with Ctrl+C or has waitFor timedout. This is very rare case and
         # is best handled by user. Same applies for doCancel as well
      except Tac.Timeout:
         print( "Reload schedule could not be ascertained" )
         return
      doShowReload( mode ).render()
      return

   shutdownCmd = os.environ.get( "SHUTDOWN_CMD", "shutdown -r now" )
   # we want to sleep after we issue the shutdown command so as to not
   # return a prompt back so quickly
   postShutdownSleep = int( os.environ.get( "POST_SHUTDOWN_SLEEP", "600" ) )
   # see BUG5953 for a discussion of why we stop the watchdog here
   watchdogCmd = os.environ.get( "WATCHDOGKILL_CMD", "watchdog -k" )

   if os.path.exists( f"{Url.fsRoot()}/flash/NO_PRE_RELOAD_LOGS" ):
      print( "pre_reload_logs is disabled." )
      sys.stdout.flush()
   else:
      savePreReloadLogs()

   # Set in case any agent needs to process (or not process) events in a different
   # way if we are in the middle of a reload
   if hasattr( config, 'reloadInProgress' ):
      config.reloadInProgress = True

   # BUG219569: we want to make sure that this process runs, and it is not
   # interrupted by a SIGHUP. So we daemonize with the stdout/err going to 
   # our current stdout.
   Tac.run( [ "daemonize", "-l", "-", "sh", "-c",
              "wall 'The system is going down for reboot NOW!'; sleep 0.25; "
              "%s; %s" % ( shutdownCmd, watchdogCmd ) ],
              asRoot=True, ignoreReturnCode=True, stdout=sys.stdout,
              stderr=sys.stderr )

   # sleep so to give the reload command time to take effect (and to block the
   # prompt in a real switch)
   time.sleep( postShutdownSleep )

# doReload function ends

def doCancel( mode ):
   if isReloadScheduled():
      config.reason = ""
      config.all = False
      config.reloadTime = Tac.endOfTime 
      try:
         Tac.waitFor( lambda: not isReloadScheduled(),
                      description="reload to get cancelled",
                      sleep=True
                    )
      except Tac.Timeout:
         print( "Reload cancel could not be ascertained" )
         return
      print( "Scheduled reload has been cancelled" )
   else:
      print( "No reload is scheduled " )

class ShowReloadCmdModel( Model ):
   scheduled = Bool( help="A future reload is scheduled" )
   scheduledTime = Float( help="Time of the future reload (UTC)", optional=True )
   reason = Str( help="Reason why a reload has been scheduled", optional=True )
   allSupervisors = Bool( help="All supervisors will be reloaded",
                          optional=True )
   _timeLeft = Float( help="Time left", optional=True )

   def render( self ):
      if self.scheduled:
         hhmmStr = ""
         if self._timeLeft > 0:
            mm = int( self._timeLeft / 60 )
            hhmm = divmod( mm, 60 )
            if hhmm[ 0 ] < 24:
               hhmmStr = "(in %d hours %d minutes)" % hhmm
            # Let the user know, if reload is close by
            print ( "Reload scheduled "
                    "for %s%s %s " % ( time.ctime( self.scheduledTime ),
                    " on all supervisors" if self.allSupervisors else "", hhmmStr ) )

         if self.reason:
            print( "Reload reason:", self.reason )
      else:
         print( "No reload is scheduled " )

def doShowReload( mode, args=None ):

   reloadTime = None
   timeLeft = None
   reloadReason = None
   reloadScheduled = isReloadScheduled()
   allSupervisors = status.all
   if reloadScheduled:
      timeLeft = ( status.reloadTime - Tac.now() )
      reloadTime = time.time() + timeLeft
      reloadReason = status.reason

   return ShowReloadCmdModel( scheduled=reloadScheduled,
                              scheduledTime=reloadTime,
                              reason=reloadReason,
                              allSupervisors=allSupervisors
                              if allSupervisors else None,
                              _timeLeft=timeLeft ) 

class ShowReloadCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show reload"
   data = {
      'reload': reloadShowKwNode
      }
   cliModel = ShowReloadCmdModel
   handler = doShowReload

BasicCli.addShowCommandClass( ShowReloadCommand )

class ReloadCommand( CliCommand.CliCommandClass ):
   syntax = "reload [ all | peer ] [ power ] [ now ] [ force ]"
   data = {
      'reload' : reloadKwNode,
      'all' : allKwNode,
      'peer' : peerKwNode,
      'power' : powerCycleMatcher,
      'now' : "Perform action immediately without prompting",
      'force' : forceKwMatcher
      }

   syncAcct = True
   @staticmethod
   def handler( mode, args ):
      supe = args.get( 'all' ) or args.get( 'peer' )
      power = 'power' in args
      now = 'now' in args
      force = 'force' in args
      doReload( mode, supe=supe, power=power, now=now, force=force )

BasicCli.EnableMode.addCommandClass( ReloadCommand )

mmmMatcher = DateTimeRule.TimeMatcher( name="mmm", numPattern="mmm",
                                       helpdesc="Delay in minutes" )
hhhmmMatcher = DateTimeRule.TimeMatcher( name="hhh:mm", numPattern="hhh:mm",
                   helpdesc="Delay in hours and minutes mm<0-59>" )
hhmmMatcher = DateTimeRule.TimeMatcher( name="hh:mm", numPattern="hh:mm",
                   helpdesc="Time in hours and minutes hh<0-23> mm<0-59>" )

class ReloadLaterCommand( CliCommand.CliCommandClass ):
   syntax = "reload [ all | peer ] [ power ] " \
            "( in MMM | HHHMM ) | " \
            "  ( at HHMM [ ( MONTH DAY ) | ( DAY MONTH ) ] ) " \
            "[ force ] [ reason REASON ]"
   data = {
      'reload' : reloadKwNode,
      'all' : allKwNode,
      'peer' : peerKwNode,
      'power' : powerCycleMatcher,
      'force' : forceKwMatcher,
      'in' : "Perform the reload after specified delay",
      'at' : "Perform reload at specified time",
      'MMM' : mmmMatcher,
      'HHHMM' : hhhmmMatcher,
      'HHMM' : hhmmMatcher,
      'MONTH' : DateTimeRule.monthMatcher,
      'DAY' : DateTimeRule.dayMatcher,
      'reason' : 'String to display explaining the purpose of the reload',
      'REASON' : CliMatcher.StringMatcher( helpname="TEXT",
                  helpdesc='String to display explaining the purpose of the reload' )
      }
   syncAcct = True

   @staticmethod
   def handler( mode, args ):
      supe = args.get( 'all' ) or args.get( 'peer' )
      power = 'power' in args
      now = 'now' in args
      force = 'force' in args
      reloadReason = args.get( 'REASON' )
      if 'in' in args:
         schedule = ( 'in', args.get( 'MMM' ) or args[ 'HHHMM' ] )
      else:
         assert 'at' in args
         if 'MONTH' in args:
            monthDay = ( args[ 'MONTH' ], args[ 'DAY' ] )
         else:
            monthDay = None
         schedule = ( 'at', args[ 'HHMM' ], monthDay )

      doReload( mode, supe=supe, power=power, now=now, force=force,
                schedule=schedule, reloadReason=reloadReason )

BasicCli.EnableMode.addCommandClass( ReloadLaterCommand )

class ReloadCancelCommand( CliCommand.CliCommandClass ):
   syntax = "reload cancel"
   noOrDefaultSyntax = "reload ..."
   data = {
      'reload' : reloadKwNode,
      'cancel' : "Cancel the existing reload command"
      }
   syncAcct = True

   @staticmethod
   def handler( mode, args ):
      doCancel( mode )

   noOrDefaultHandler = handler

BasicCli.EnableMode.addCommandClass( ReloadCancelCommand )

def Plugin( entityManager ):
   global config
   global status
   global powerFuseConfig
   global sysdbRedSupStatus
   config = LazyMount.mount( entityManager, Cell.path( "sys/reload/config" ),
                             "System::Reload::Config", "fw" )
   status = LazyMount.mount( entityManager, Cell.path( "sys/reload/status" ),
                             "System::Reload::Status", "r" )
   powerFuseConfig = LazyMount.mount( entityManager, "power/fuse/config/admin",
                             "Power::SoftwareFuse", "w" )
   sysdbRedSupStatus = LazyMount.mount( entityManager, 
                                        Cell.path( "redundancy/status" ),
                                        "Redundancy::RedundancyStatus", "r" )

