#!/usr/bin/env python3
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import errno

import BasicCliUtil
from CliPlugin import ReloadCli
import FileUrl
import Url

UCE_UNKNOWN_STATE = 0
UCE_UNSAVED_CHANGES = 1
UCE_NO_UNSAVED_CHANGES = 2

def _unsavedChangesExist( mode ):
   # read running-config and startup-config and compare them

   runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   try:
      runningConfigFile = runningConfigUrl.open()
      try:
         runningConfigValue = runningConfigFile.read()
      finally:
         runningConfigFile.close()
   except EnvironmentError:
      # if we can not get the contents of the file, there could be
      # unsaved changes
      return UCE_UNKNOWN_STATE

   startupConfigUrl = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   startupConfigValue = ""
   try:
      startupConfigFile = startupConfigUrl.open()
   except IOError as ex:
      if ex.errno == errno.ENOENT:
         # don't consider a missing startup config to be an error
         return UCE_UNSAVED_CHANGES
      raise
   try:
      startupConfigValue = startupConfigFile.read()
   finally:
      startupConfigFile.close()

   def stripLeadingComments( lines ):
      while lines and lines[ 0 ].startswith( "!" ):
         lines.pop( 0 )

   # Need to strip comments because the first two comment lines
   # (thanks to EosRunningConfigHeader.py) change every time you look.
   # Very quantum-mechanical!
   startupLines = startupConfigValue.splitlines()
   stripLeadingComments( startupLines )

   runningLines = runningConfigValue.splitlines()
   stripLeadingComments( runningLines )

   if len( startupLines ) != len( runningLines ):
      return UCE_UNSAVED_CHANGES
   
   # There was a bug where trailing whitespaces would cause a diff to be detected,
   # but when asked to display, the `diff` cli strips them, causing confusion.
   for index, startupLine in enumerate( startupLines ):
      if startupLine.rstrip() != runningLines[ index ].rstrip():
         return UCE_UNSAVED_CHANGES

   return UCE_NO_UNSAVED_CHANGES

# AAA note:
# There is the potential for viewing the running config and saving the
# running config to fail due to the user not being authorized to run
# those commands.
def doConfigSaveBeforeReload( mode, power, now, noMeansFailure=False, supe=None ):
   # compare configurations and
   # if the configurations are different, ask the user about saving
   # based on the user's answer, save, reload, or cancel.
   # the user can use now to avoid being asked questions
   couldReload = True

   configFile = mode.session_.configFile
   if configFile is not None or now:
      # Don't prompt the user in non-interactive mode, although we don't expect
      # reload to be in a config file. Prompts are also not shown if the user
      # specified 'now' when reloading.
      return True

   # Running interactively with the user
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   if redundancyStatus and redundancyStatus.mode == 'active':
      unsavedChanges = _unsavedChangesExist( mode )
   else:
      # Do not prompt on standby
      unsavedChanges =  UCE_NO_UNSAVED_CHANGES

   if unsavedChanges == UCE_UNKNOWN_STATE:
      promptMessage = "System configuration may have been modified."
   elif unsavedChanges == UCE_UNSAVED_CHANGES:
      promptMessage = "System configuration has been modified."
   elif unsavedChanges == UCE_NO_UNSAVED_CHANGES:
      promptMessage = None

   if promptMessage is not None:
      while True:
         answer = BasicCliUtil.getChoice( mode,
               promptMessage + " Save? ",
               [ 'yes', 'no', 'cancel', 'diff' ], 'no' )

         if ReloadCli.answerMatches( answer, "yes" ):
            try:
               # use runCmd to avoid having to re-implement copy and to get
               # authorization
               mode.session_.runCmd( "copy running-config startup-config" )
            except Exception as ex: # pylint: disable=broad-except
               print( f"Error saving configuration: {ex}" )
               # let the user try again
               continue
         elif ReloadCli.answerMatches( answer, "no" ):
            # Treat "no" as failure. Used in hitless and fastboot reloads
            # to prevent reloading if the configuration is not saved
            if noMeansFailure:
               couldReload = False
         elif ReloadCli.answerMatches( answer, "cancel" ):
            couldReload = False
         elif ReloadCli.answerMatches( answer, "diff" ):
            try:
               mode.session_.runCmd( "diff startup-config running-config" )
            except Exception as ex: # pylint: disable=broad-except
               print( f"Error in diff: {ex}" )

            # the user should see the prompt again after the diff
            continue
         else:
            # we don't understand the user's answer so prompt again
            print( "Invalid response" )
            continue

         # if we got here, we don't want to show the prompt again
         break
   return couldReload

# Register doConfigSaveBeforeReload to be run before a reload occurs.
# Register as a file modification operation (run before plugins that require
# the file system to be in its final stable state)
ReloadCli.registerReloadHook( doConfigSaveBeforeReload, 'ReloadConfigSaveCli',
                              'FILE_MODIFICATION' )
