# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides an interface to retrieve Rib-internal state
# for all Rib agents (across default and non-default VRFs)
#------------------------------------------------------------------------------------
from __future__ import absolute_import, division, print_function
import BasicCli
import CliMatcher
from CliPlugin import TechSupportCli
from CliPlugin.RoutingCommon import GATED_COMMON_SH_TECH_TS
from CliPlugin.VrfCli import VrfExprFactory
import LazyMount
from FileCliUtil import registerPersistentConfiguration
from IpLibTypes import ProtocolAgentModelType
import os
from ShowCommand import ShowCliCommandClass

l3ProtocolAgentModelStatus = None

matcherRibd = CliMatcher.KeywordMatcher( 'ribd',
    helpdesc='Show aggregated status and configuration details for the Rib agent' )

matcherVrfName = VrfExprFactory( helpdesc="VRF name", inclAllVrf=True,
                            inclDefaultVrf=True )

class ShowTechSupportRibd( ShowCliCommandClass ):
   syntax = "show tech-support AGENT [ debug ] [ ribout ] [ verbose ] [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'AGENT': matcherRibd,
           'debug': 'Show debug information for ribd',
           'ribout': 'Show debug information about RibOuts',
           'verbose': 'Enable verbose output',
           'vrfName': matcherVrfName
           }
   handler = "RibShowTechCliHandler.handlerShowTechSupportRibd"

BasicCli.addShowCommandClass( ShowTechSupportRibd )

tokenRunningConfig = CliMatcher.KeywordMatcher( 'running-config',
                     helpdesc='Show running configuration of ribd' )

class ShowTechSupportRibdRunningConfig( ShowCliCommandClass ):
   syntax = "show tech-support ribd running-config [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'ribd': matcherRibd,
           'running-config': tokenRunningConfig,
           'vrfName': matcherVrfName
           }
   handler = "RibShowTechCliHandler.handlerShowTechSupportRibdRunningConfig"

BasicCli.addShowCommandClass( ShowTechSupportRibdRunningConfig )

def _gatedShowCommandsGuard():
   def isSingleAgentMode():
      return ( l3ProtocolAgentModelStatus.protocolAgentModel ==
               ProtocolAgentModelType.ribd )

   return isSingleAgentMode()

TechSupportCli.registerShowTechSupportCmd(
   GATED_COMMON_SH_TECH_TS,
   cmds=[ 'show tech-support ribd running-config',
          'show rib ready vrf all',
          'bash echo "show memory" | cliribd' ],
   cmdsGuard=_gatedShowCommandsGuard,
)

agentsNames = [ 'ospf', 'ospfv3', 'isis', 'rip' ]
agentsEnumDict = { agent: "Show agregated status and configuration "
                           "details for " + agent for agent in agentsNames }

class ShowTechSupportExtendedAgents( ShowCliCommandClass ):
   syntax = "show tech-support extended AGENT [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'extended': TechSupportCli.extendedKwMatcher,
           'AGENT': CliMatcher.EnumMatcher( agentsEnumDict ),
           'vrfName': matcherVrfName,
           }
   handler = "RibShowTechCliHandler.handlerShowTechSupportExtendedAgents"

BasicCli.addShowCommandClass( ShowTechSupportExtendedAgents )

def deleteRibStartup( mode, delete ):
   if delete and os.path.exists( "/mnt/flash/rib-startup" ):
      os.unlink( "/mnt/flash/rib-startup" )

def deleteRibAmi( mode, delete ):
   if delete and os.path.exists( "/persist/sys/ribd.ami" ):
      os.unlink( "/persist/sys/ribd.ami" )

def deleteRibConfig( mode, delete ):
   deleteRibAmi( mode, delete )
   deleteRibStartup( mode, delete )

ribConfigDesc = """config file for Ribd configuration commands that
                 are not managed by EOS."""

registerPersistentConfiguration( "rib-config", deleteRibConfig,
                                ribConfigDesc, default=True )

def Plugin( entityManager ):
   global l3ProtocolAgentModelStatus

   l3ProtocolAgentModelStatus = LazyMount.mount( entityManager,
      'l3/status/protocolAgentModelStatus',
      'L3::ProtocolAgentModelStatus', 'r' )
