# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

from ArnetLib import asnStrToNum
import CliParser
from CliParserCommon import MatchResult, noMatch
import CliMatcher

class RdDistinguisherMatcher( CliMatcher.Matcher ):
   # we're matching:
   # case 1: <16-bit ASN>:<32-bit local number> OR
   # case 2: <16-bit ASN>:<dotted IPv4 address> OR
   # case 3: <dotted IPv4 address>:<16-bit local number>
   # case 4: <32-bit ASN>:<16-bit local number>
   # case 5: <32-bit ASN in Asdot>:<16-bit local number>
   valueRe = r'\d+'
   asdotRe = r'(\d+)\.(\d+)'
   ipAddrRe = r'(\d+)\.(\d+)\.(\d+)\.(\d+)'

   # pylint: disable-next=consider-using-f-string
   rdPattern = r'((%s)|(%s)|(%s)):((%s)|(%s))$' % ( valueRe, asdotRe, ipAddrRe,
                                                    valueRe, ipAddrRe )

#   this expands to the following RE pattern:
#   rdPattern = '((\d+)|((\d+)\.(\d+))|((\d+)\.(\d+)\.(\d+)\.(\d+))):' \
#       '((\d+)|((\d+)\.(\d+)\.(\d+)\.(\d+)))$'

   regExp = re.compile( rdPattern )
   # groups in this RE:
   #   0: entire value before colon
   #   1: ASN if present
   #   2: ASN in Asdot format if present
   #   3: first hex of 2
   #   4: second hex of 2
   #   5: IP addr before colon if present
   #   6: first octet of 5
   #   7: second octet of 5
   #   8: third octet of 5
   #   9: fourth octet of 5
   #  10: entire value after colon
   #  11: either 16 or 32 bit local number, if present
   #  12: IP addr after colon if present
   #  13: first octet of 9
   #  14: second octet of 9
   #  15: third octet of 9
   #  16: fourth octet of 9

   completionRegExp = re.compile( CliParser.makePartialPattern( rdPattern ) )

   def match( self, mode, context, token ):
      m = self.regExp.match( token )
      if m is None:
         return noMatch

      g = m.groups()
      admin = g[ 0 ]
      local = g[ 10 ]
      if g[ 1 ]:
         # only digits before colon, so we are in case 1, case 2 or
         # case 4.
         if int( g[ 1 ] ) > 0xffff:
            # Case 4, 32 bit ASN & 16-bit local number
            if int( g[ 1 ] ) > 0xffffffff:
               return noMatch
            if g[ 12 ] is not None:
               # Cannot have a 4-byte ASN and an IP address
               return noMatch
            if int( g[ 11 ] ) > 0xffff:
               # Cannot have a 4-byte ASN and a 32 bit local number
               return noMatch

            # now distinguish cases 1 and 2 by looking for an IP addr
         elif g[ 12 ] is None:
            # no IP addr, so we're in case 1.  This can be up to 32 bits
            if int( g[ 11 ] ) > 0xffffffff:
               return noMatch
         else:
            # this means we have an IP addr, so we're in case 2.  As
            # Validate that the dotted values are in range
            if ( int( g[ 13 ] ) > 255 or int( g[ 14 ] ) > 255 or
                 int( g[ 15 ] ) > 255 or int( g[ 16 ] ) > 255 ):
               return noMatch
      elif g[2]:
         # this means we matched ASN value in ASdot format before colon,
         # so we are in Case 5
         if int( g[ 3 ] ) > 0xffff or int( g[ 4 ] ) > 0xffff:
            return noMatch
         if g[ 12 ] is not None:
            # Cannot have a 4-byte ASN and an IP address
            return noMatch
         if int( g[ 11 ] ) is None or int( g[ 11 ] ) > 0xffff:
            # Cannot have a 4-byte ASN and a 32 bit local number
            return noMatch
         # Convert Asn from Asdot format to Asplain
         admin = str( asnStrToNum( g[ 2 ] ) )
      elif g[5]:
         # this means we matched an IP address before the colon, so
         # we're in case 3.  Validate the dotted values are in
         # range
         if ( int( g[ 6 ] ) > 255 or int( g[ 7 ] ) > 255 or
              int( g[ 8 ] ) > 255 or int( g[ 9 ] ) > 255 ):
            return noMatch
         # now validate that we don't have an IP address after the
         # colon, and that it's in range for a 16-bit number
         if g[ 11 ] is None or int( g[ 11 ] ) > 0xffff:
            return noMatch
      token = admin + ':' + local
      return MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.completionRegExp.match( token )
      if m is None:
         return []

      return [ CliParser.Completion( '<admin>:<local assignment>', self.helpdesc_,
                                                                            False ) ]

   def __str__( self ):
      return '<admin>:<local number>'
