# Copyright (c) 2009, 2010, 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=too-many-nested-blocks
# pylint: disable=consider-using-from-import
# pylint: disable=protected-access
# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in
# pylint: disable=simplifiable-if-statement

from enum import Enum
import hashlib
import re

import Bunch
from ArnetLib import ( bgpFormatAsn,
      asnStrToNum, U32_MAX_VALUE, U16_MAX_VALUE )
import ConfigMount
import BasicCli
from BasicCliUtil import anyCaseRegex
import BasicCliModes
import CliCommand
from CliDynamicSymbol import LazyCallback
import CliMatcher
from CliMode.RouteMap import Ipv6PrefixListCliMode, RouteMapCliMode
from CliMode.RouteMap import IpPrefixListCliMode, PeerFilterCliMode
from CliMode.RouteMap import generateCommentKey
import CliParser
from CliParser import safeInt, safeFloat
import CliParserCommon
import CliGlobal
from CliPlugin import SwitchIntfCli
import CliPlugin.ConfigConvert
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IraServiceCli as IraServiceCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.LoopbackIntfCli as LoopbackIntfCli
import CliPlugin.SubIntfCli as SubIntfCli
import CliPlugin.VirtualIntfRule as VirtualIntfRule
from CliPlugin.AclCli import aclUnconfReferencesHook
from CliPlugin.IpRibLibCliModels import ResolutionRibProfileConfig
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.RouteMapCliModels import IpLargeCommunityList, PrefixListSummary
from CliPlugin.RouteMapCliModels import IpLargeCommunityLists
from CliPlugin.RouteMapCliModels import IpPrefixLists
from CliPlugin.VlanIntfCli import VlanIntf
from CliPlugin.IpRibLib import ResolutionRibsExprFactory
from CliPlugin.RouteMapCliModels import IpAsPathLists, \
      IpAsPathSummary
from CliPlugin.RouteMapCliModels import IpCommunityListEntry, IpCommunityList, \
      IpExtCommunityList, IpCommunityLists, IpExtCommunityLists
from CliPlugin.RouteMapCliModels import MatchRules, SetRules, NextHop, \
      CommunitySet, \
      RouteMetric, RouteMapEntry, RouteMap, RouteMaps, ConfigSanityRouteMaps, \
      ExpandedRouteMaps, ExpandedRouteMap, ExpandedRouteMapEntry, \
      PeerFilterMatchRules, PeerFilters, AsnRange, \
      CommInstancesRange, AsPathLengthRange, AggregateContributor, DebugInfo, \
      SubRMDebugInfo, AttributeDebug, SubRouteMap, AigpMetric
from CliPlugin.RouteMapCliModels import PolicyEvalRouteMap, PolicyEvalSequence, \
      PolicyEvalRouteMapResult, PolicyEvalSequenceResult
from CliPlugin.VrfCli import getAllPlusReservedVrfNames

import CliSession
import CliToken.Ip
import CliToken.Ipv6
import CliToken.Refresh

from CliToken.Community import CommunityConstExpr
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapCommonTokens
from CliToken.RouteMapCliTokens import (
      ExtCommActionExpression,
      MetricModifierAndNumberMatcher,
      MetricModifierAndFloatMatcher,
      InputRestrictedFloatMatcher
   )
import CliToken.IpRibLibCliTokens
from ConfigConsistencyChecker import UndefinedReferenceChecker
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import LazyMount
import MultiRangeRule
from RouteMapLib import SetActionInfo, MatchAttributes, Printer, extCommType
from RouteMapLib import isMatchAttrEnabled, isSetActionEnabled, LinkBwUnion
from RouteMapLib import matchPermitEnum, metricTypeEnum, \
      isisLevelEnum, isisStyleEnum, SetNextHopArgTransformer
from RouteMapLib import commValueToPrint, isAsdotConfigured
from RouteMapLib import getExtCommValue, getExtCommLbwDisplayValue
from RouteMapLib import getLargeCommValue
from RouteMapLib import getExtCommTypeValue, extCommTypeValueMap, CommunityType
from RouteMapLib import matchRuleCapiName, setRuleCapiName, setMatchRuleValue, \
    getLbwCommValue, listUrlSchemes
from RouteMapLib import getTacAigpMetricFromCliStr, capiMedTypes, capiAigpMetricTypes
from RouteMapLib import ( originAsValidityEnum,
                          getErrorMessageCannotDeleteBecauseUsedBy
)
from RouteMapLib import POLICY_COMM_INST_MAX, POLICY_COMM_INST_MIN, \
      POLICY_ASPATH_LENGTH_MIN, POLICY_ASPATH_LENGTH_MAX
from RouteMapLib import POLICY_ISIS_INST_MIN, POLICY_ISIS_INST_MAX
from RouteMapLib import POLICY_OSPF_INST_MIN, POLICY_OSPF_INST_MAX
from RouteMapLib import matchContribAggAttrOption, matchContribAggAttrRoleStr, \
      matchContribAggAttrContribStr, matchContribAggAttrAggAttrStr
from RouteMapLib import MatchCmdRenderer
from RouteMapLib import AS_PATH_MIN, AS_PATH_MAX
from RouteMapLib import AS_PATH_MAX_LENGTH
from RouteMapLib import PREFIX_SEQNO_MAX_VALUE
from RouteMapLib import RouteMapMatchOption
from RouteMapLib import \
   routeMapMedType, tacMetricValue, routeMapAigpMetricSetType, tacAigpMetricValue
from ShowCommand import ShowCliCommandClass
from Toggles import RouteMapToggleLib
from Toggles.OpenConfigRoutingPolicyToggleLib import (
   toggleAirStreamPolicyDefinitionsEnabled,
)
import Tac
import Tracing
import Url

U32_MAX_VALUE =  0xFFFFFFFF
U64_MAX_VALUE = 0xFFFFFFFFFFFFFFFF
DUP_ENTRY_ERROR_STR = "This entry already exists at sequence number: {}"

gv = CliGlobal.CliGlobal( dict(
   routeMapHelper=None,
) )
mapConfig = None # route map
# Aggregated Map Config from RouteMapConfigAggregatorSm
# The SM is initialized in ConfigAgentPlugin/RouteMap.py
dynamicMapConfig = None

aclConfig = None # standard acl list
aclListConfig = None # as-path list, prefix list
dynPfxListConfigDir = None # dynamic prefix-list
asnConfig = None
peerFilterConfig = None # peer filter
handlerDir = None
inputDir = None
l3Config = None

ExtCommLinkBandwidthSetType = Tac.Type( 'Acl::ExtCommLinkBandwidthSetFlag' )
ExtCommLinkBandwidthSetDeleteType = \
      Tac.Type( 'Acl::ExtCommLinkBandwidthSetDeleteFlag' )

UniqueId = Tac.Type( 'Ark::UniqueId' )
RouteMapEntryKey = Tac.Type( "Acl::RouteMapEntryKey" )

traceHandle = Tracing.Handle( 'RouteMap' )
#pylint: disable-msg=C0321
#pylint: disable-msg=E1103
#pylint: disable-msg=W0702
#pylint: disable-msg=W0703
#pylint: disable=anomalous-backslash-in-string
t0 = traceHandle.trace0
t2 = traceHandle.trace2
t8 = traceHandle.trace8

AMBIGOUS_COMMAND_WARNING = "Ambiguous set community command. Please specify " + \
                           "additive / delete operation."
#------------------------------------------------------------------------
# Simple CLI tokens
#------------------------------------------------------------------------
subRouteMapMatcher = CliMatcher.KeywordMatcher(
   'sub-route-map', helpdesc='Sub route map name' )
communityInstancesMatcher = CliMatcher.KeywordMatcher(
   'instances', helpdesc='Match number of community instances' )
invertSubResultMatcher = CliMatcher.KeywordMatcher(
   'invert-result', helpdesc='Invert sub route map result' )
matcherRouterId = IpAddrMatcher.IpAddrMatcher(
      "OSPF 32-bit router ID as an IP address" )

seqnoMin = 1
seqnoMax = 0x00FFFFFF

# we want to exclude "length" (in any case)
# like LENGTH, or LeNgTh...
excludeAsPathListKeywords = \
   anyCaseRegex( "^(?!{}$)".format( 'length' ) )

# Adding exclusion token to the regex to the as-path list name regular expression
# "length",  cannot be used as as-path-list name
asPathListNameRe = \
   excludeAsPathListKeywords + r'([A-Za-z0-9_:{}\[\]-])([A-Za-z0-9_:{}\.\[\]-]*)'

tagSetNameRe = r'.+'

# we want to exclude additive / delete / instances (in any case)
# like ADDITIVE, or AddiTive...
excludeCommListKeywords = \
   anyCaseRegex( "^(?!{}$|{}$|{}$|{}$|{}$|{}$|{}$)".format(
      'additive',
      'delete',
      'instances',
      'exact-match',
      'regexp',
      'or-results',
      'filter'
   ) )

excludeRouteMapKeywords = anyCaseRegex(
   "^(?!{}$|{}$|{}$|{}$)".format( 'config-sanity',
                                  'expanded',
                                  'standard',
                                  'pruned' ) )

# Allowing = + . to be used in route-map, community-list and prefix-list names
rtMapCommListPreListRe = r'([A-Za-z0-9_:{}\[\]=\+-])([A-Za-z0-9_:{}\.\[\]=\+-]*)'
rtMapCommListPreListRe = excludeRouteMapKeywords + rtMapCommListPreListRe

# Adding exclusion token to the regex to the community-list name regular expression
# additive and delete are used to specify an operation type, they cannot be used
# as community-list name
communityListNameRe = excludeCommListKeywords + rtMapCommListPreListRe

commRegexRuleStr = '[a-zA-Z0-9^$_*+.|()?\\\\[\\]\\-{},:]+'

showLargeCommunityDesc = "Large community value (ASN:Local-part-1:Local-part-2)"

u8Max = 255
u16Max = 65535
u32Max = 4294967295
ufMin = 0.0
uf32Max = 4294967295.0

# This function gets called for auto completion by the new cli parser
# with mode as the first argument, even though we don't need it
def getRouteMapNames( mode ):
   '''return list of configured Route Map names'''
   return mapConfig.routeMap

def getTagSetNames( mode ):
   return aclListConfig.tagSet

class ColorExtCommunityExpression( CliCommand.CliExpression ):
   expression = "color COLOR-VALUE [ color-only ( exact-match | " \
                "( endpoint-match ( any | null ) ) ) ]"
   data = {
      'color': 'Color extended community',
      'COLOR-VALUE': CliMatcher.IntegerMatcher(
         0, 4294967295, helpdesc="Color value" ),
      'color-only': 'Color-Only flags for BGP destination steering',
      'exact-match': 'Exact match of the SR-TE policy (default)',
      'endpoint-match': 'Endpoint match of the SR-TE policy',
      'any': 'Any endpoint match of the SR-TE policy',
      'null': 'Null endpoint match of the SR-TE policy'
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      colorVals = []
      coBits = ':00'
      colorVal = ''
      for arg in argsList:
         if arg[ 0 ] == 'COLOR-VALUE':
            # We already have one stored, save what we have
            if colorVal:
               colorVals.append( 'color ' + colorVal + coBits )
               coBits = ':00'
            colorVal = str( arg[ 1 ] )
         if arg[ 0 ] == 'any':
            coBits = ':10'
         if arg[ 0 ] == 'null':
            coBits = ':01'
      # Save the last one configured
      if colorVal:
         colorVals.append( 'color ' + colorVal + coBits )
      args[ 'COLOR_VAL' ] = colorVals

class EvpnExtCommunityExpression( CliCommand.CliExpression ):
   expression = "evpn es-import rt ESI_VAL"
   data = {
      'evpn': 'EVPN extended communities',
      'es-import': 'EVPN ES Import',
      'rt': 'Route Target extended community',
      'ESI_VAL': MacAddrMatcher( helpdesc='Low-order 6 bytes of '
                                          'ES-Import Route Target' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      val = args.get( 'ESI_VAL', [] )
      esiVal = [ 'esi ' + x for x in val ]
      args[ 'ESI_VAL' ] = esiVal

class OspfExtCommunityExpression( CliCommand.CliExpression ):
   expression = "ospf router-id ROUTER_ID"
   data = {
      'ospf': 'OSPF extended communities',
      'router-id': 'Router ID for the OSPF instance',
      'ROUTER_ID': matcherRouterId,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      routerIds = args.get( 'ROUTER_ID', [] )
      routerIdValue = [ 'router-id ' + x for x in routerIds ]
      args[ 'ROUTER_ID' ] = routerIdValue

class CommonTokens:
   '''
   This class is intended to keep common tokens that may be shared across several
   commands with the new CLI parser.
   '''
   class PermitDenyExpression( CliCommand.CliExpression ):
      expression = '( permit | deny )'
      data = {
         'permit': CliMatcher.KeywordMatcher( 'permit',
            helpdesc='Specify community to accept' ),
         'deny': CliMatcher.KeywordMatcher( 'deny',
            helpdesc='Specify community to reject' )
      }
   equal = CliMatcher.KeywordMatcher( '=',
         helpdesc='equal' )
   andToken = CliMatcher.KeywordMatcher( 'and',
         helpdesc='and' )
   lowerThanEqual = CliMatcher.KeywordMatcher( '<=',
         helpdesc='lesser than or equal' )
   greaterThanEqual = CliMatcher.KeywordMatcher( '>=',
         helpdesc='greater than or equal' )

   matchIpv4 = CliMatcher.KeywordMatcher( 'ip',
         helpdesc='Match IP specific information' )
   matchIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
         helpdesc='Match IPv6 specific information' )
   matchIpAddress = CliMatcher.KeywordMatcher( 'address',
         helpdesc='Match ip address' )
   matchNextHop = CliMatcher.KeywordMatcher( 'next-hop',
         helpdesc='Route next hop' )
   matchResolvedNextHop = CliMatcher.KeywordMatcher( 'resolved-next-hop',
         helpdesc='Route resolved next hop' )
   matchAccessList = CliMatcher.KeywordMatcher( 'access-list',
         helpdesc='IP access list' )
   matchPrefixList = CliMatcher.KeywordMatcher( 'prefix-list',
         helpdesc='Prefix list' )

   matchAll = CliMatcher.KeywordMatcher( 'all',
         helpdesc='Match the entire AS path' )
   accessList = CliMatcher.KeywordMatcher(
         'access-list', helpdesc='Specify access list name' )
   asPathList = CliMatcher.KeywordMatcher( 'as-path',
         helpdesc='BGP AS path list' )
   asPathListName = CliMatcher.DynamicNameMatcher(
         lambda mode: aclListConfig.pathList,
         'AS path list name', pattern=asPathListNameRe )
   asPathPrependReplace = CliMatcher.KeywordMatcher( 'as-path',
         helpdesc='Prepend/replace AS path' )
   bgpAs = CliMatcher.KeywordMatcher( 'as', helpdesc='BGP AS number' )
   asPathLengthMatcher = CliMatcher.IntegerMatcher(
         POLICY_ASPATH_LENGTH_MIN,
         POLICY_ASPATH_LENGTH_MAX,
         helpdesc='Length of as-path'
   )
   asPathFilter = CliMatcher.KeywordMatcher( 'as-path',
         helpdesc='BGP autonomous system path filter' )
   asPathLength = CliMatcher.KeywordMatcher( 'length', 'Specify as-path length' )
   asPath = CliMatcher.KeywordMatcher( 'as-path',
         helpdesc='Prepend/replace AS path' )
   asRepeat = CliMatcher.IntegerMatcher( 2, AS_PATH_MAX_LENGTH,
         helpdesc='The number of times to repeat the AS-Path' )
   asRepeatLastAs = CliMatcher.IntegerMatcher(
         SetActionInfo[ 'asPathPrependLastAs' ][ 'min' ],
         SetActionInfo[ 'asPathPrependLastAs' ][ 'max' ],
         helpdesc='The number of times to prepend the last AS number' )
   asAuto = CliMatcher.KeywordMatcher(
         'auto', helpdesc='Use peer AS number for inbound '
         'and local AS for outbound' )
   communityNone = CliMatcher.KeywordMatcher(
      'none',
      helpdesc='No community attribute' )
   communityPermitOrDeny = CliMatcher.EnumMatcher( {
      'permit': 'Specify community to accept',
      'deny': 'Specify community to reject',
   } )
   commListRegexMatcher = CliMatcher.PatternMatcher(
      pattern=commRegexRuleStr,
      helpname='WORD',
      helpdesc='A regular expression to match communities.'
      ' Use "Ctrl-v ?" to enter literal "?"' )
   commRegexp = CliMatcher.KeywordMatcher(
      'regexp',
      helpdesc='Add an expanded community-list entry' )
   communityListName = CliMatcher.DynamicNameMatcher(
         lambda mode: aclListConfig.communityList,
         "Community list name", pattern=communityListNameRe )
   communityListNameMatcher = CliMatcher.PatternMatcher( communityListNameRe,
                                                 helpname='NAME',
                                                 helpdesc='Name of community-list' )
   extCommunityListNameMatcher = CliMatcher.PatternMatcher(
                                          communityListNameRe,
                                          helpname='NAME',
                                          helpdesc='Extended community list name' )
   largeCommunityListNameMatcher = CliMatcher.PatternMatcher(
      communityListNameRe,
      helpname='NAME',
      helpdesc='Name of large community list' )
   communityList = CliMatcher.KeywordMatcher(
      'community-list',
      helpdesc='Add a community list entry' )
   communityListFilter = CliMatcher.KeywordMatcher(
      'filter',
      helpdesc='Filter by community list' )
   communityListFilterAll = CliMatcher.KeywordMatcher(
      'all',
      helpdesc='Filter all community types' )
   setExtCommunity = CliMatcher.KeywordMatcher(
      'extcommunity',
      helpdesc='BGP extended community attribute' )
   setCommAdditiveDelete = CliMatcher.EnumMatcher( {
         'additive': 'Add communities to those already present',
         'delete': 'Delete matching communities',
   } )
   extCommunityRt = CliMatcher.KeywordMatcher(
      'rt',
      helpdesc='Route Target extended community' )
   extCommunitySoo = CliMatcher.KeywordMatcher(
      'soo',
      helpdesc='Site-of-Origin extended community' )
   extCommunityLbw = CliMatcher.KeywordMatcher(
      'lbw',
      helpdesc='Link Bandwidth extended community (bits/sec)' )
   communityDelete = CliMatcher.KeywordMatcher(
      'delete',
      helpdesc='Delete matching communities' )
   extCommunityLbwDivide = CliMatcher.KeywordMatcher(
      'divide',
      helpdesc='Divide Link Bandwidth' )
   extCommLbwDivideEqualRatio = CliMatcher.EnumMatcher( {
      'equal': 'Equal Link Bandwidth division',
      'ratio': 'Ratio based Link Bandwidth division'
   } )
   extCommunityLbwAggregate = CliMatcher.KeywordMatcher(
      'aggregate',
      helpdesc='Aggregate Link Bandwidth' )
   extCommunityList = CliMatcher.KeywordMatcher(
      'extcommunity-list',
      helpdesc='Add an extended community list entry' )
   matchCommunity = CliMatcher.KeywordMatcher(
      'community',
      helpdesc='Match BGP community list' )
   matchExtCommunity = CliMatcher.KeywordMatcher(
      'extcommunity',
      helpdesc='Match BGP/VPN extended community list' )
   matchLargeCommunity = CliMatcher.KeywordMatcher(
      'large-community',
      helpdesc='Match BGP large community list' )
   invertResult = CliMatcher.KeywordMatcher(
      'invert-result',
      helpdesc='Invert match result' )
   orResults = CliMatcher.KeywordMatcher(
      'or-results',
      helpdesc='Do OR operation to results of matching community lists' )
   exactMatch = CliMatcher.KeywordMatcher(
      'exact-match',
      helpdesc='Do exact matching of communities' )
   largeCommunityList = CliMatcher.KeywordMatcher(
      'large-community-list',
      helpdesc='Add a large community list entry' )
   match = CliMatcher.KeywordMatcher(
      'match',
      helpdesc='Route map match rule' )
   matchAsPath = CliMatcher.KeywordMatcher(
      'match',
      helpdesc='Match AS path to edit' )
   none = CliMatcher.KeywordMatcher( 'none',
      helpdesc='Remove matching AS numbers' )
   originAs = CliMatcher.KeywordMatcher(
      'origin-as',
      helpdesc='Originating autonomous system' )
   originAsValidityEnum = CliMatcher.EnumMatcher( {
      'valid': 'Valid origin AS',
      'invalid': 'Invalid origin AS',
      'not-found': 'No RPKI validation available for this route',
   } )
   prepend = CliMatcher.KeywordMatcher( 'prepend',
         helpdesc='Prepend to the as-path' )
   routeMap = CliMatcher.KeywordMatcher( 'route-map',
                                         helpdesc='Route map information' )
   lastAs = CliMatcher.KeywordMatcher( 'last-as',
      helpdesc='Use the last AS number in the AS path to prepend' )
   repeat = CliMatcher.KeywordMatcher( 'repeat',
      helpdesc='Repeat the entered AS path' )
   routeMapName = CliMatcher.DynamicNameMatcher(
      getRouteMapNames,
      'Route map name',
      helpname='WORD',
      pattern=rtMapCommListPreListRe )
   setAction = CliMatcher.KeywordMatcher(
      'set',
      helpdesc='Set route attribute' )
   setOrigin = CliMatcher.KeywordMatcher(
      'origin',
      helpdesc='Set BGP origin',
   )
   setOriginEnum = CliMatcher.EnumMatcher( {
      'egp': 'Set BGP origin to EGP',
      'igp': 'Set BGP origin to IGP',
      'incomplete': 'Set BGP origin to Incomplete',
   } )
   setIp = CliMatcher.KeywordMatcher(
      'ip',
      helpdesc='Set IP specific information',
   )
   setIpv6 = CliMatcher.KeywordMatcher(
      'ipv6',
      helpdesc='Set IPv6 specific information',
   )
   setEvpn = CliMatcher.KeywordMatcher(
      'evpn',
      helpdesc='Set EVPN specific information',
   )
   setOspf = CliMatcher.KeywordMatcher(
      'ospf',
      helpdesc='Set OSPF specific information',
   )
   tagToken = CliMatcher.KeywordMatcher(
         'tag', helpdesc='Route tag' )
   setToken = CliMatcher.KeywordMatcher(
         'set', helpdesc='Route tag set' )
   tagSetNameMatcher = CliMatcher.DynamicNameMatcher(
         getTagSetNames,
         'Tag set name', pattern=tagSetNameRe )
   setOspfv3 = CliMatcher.KeywordMatcher(
      'ospfv3',
      helpdesc='Set OSPFv3 specific information',
   )
   setIsis = CliMatcher.KeywordMatcher(
      'isis',
      helpdesc='Set IS-IS specific information',
   )
   nextHop = CliMatcher.KeywordMatcher(
      'next-hop',
      helpdesc='Route next hop',
   )
   ospfBit = CliMatcher.KeywordMatcher(
      'bit',
      helpdesc='Set an OSPF protocol bit',
   )
   ospfv3Bit = CliMatcher.KeywordMatcher(
      'bit',
      helpdesc='Set an OSPFv3 protocol bit',
   )
   nextHopPeerAddr = CliMatcher.KeywordMatcher(
      'peer-address',
      helpdesc='Use BGP peering addr as next-hop',
   )
   nextHopUnchanged = CliMatcher.KeywordMatcher(
      'unchanged',
      helpdesc='Keep the next hop when advertising to eBGP peers',
   )
   bitDn = CliMatcher.KeywordMatcher(
      'dn',
      helpdesc='DN bit to prevent routing loops',
   )
   community = CliMatcher.KeywordMatcher(
      'community',
      helpdesc='BGP community attribute' )
   communityNone = CliMatcher.KeywordMatcher(
      'none',
      helpdesc='No community attribute' )
   validity = CliMatcher.KeywordMatcher(
      'validity',
      helpdesc='RPKI origin validation' )
   setAdditiveDelete = CliMatcher.EnumMatcher( {
         'additive': 'Add to the existing community',
         'delete': 'Delete matching communities',
   } )
   setLargeCommunity = CliMatcher.KeywordMatcher(
      'large-community',
      helpdesc='BGP large community attribute' )
   routeTagRange = CliMatcher.IntegerMatcher(
                        MatchAttributes[ 'matchTag' ][ 'min' ],
                        MatchAttributes[ 'matchTag' ][ 'max' ],
                        helpdesc='Route tag' )
   copyOrRenameMatcher = CliMatcher.EnumMatcher( {
      'copy': 'Copy configuration from',
      'rename': 'Rename configuration from' } )
   overwrite = CliMatcher.KeywordMatcher(
      'overwrite', helpdesc='Overwrite existing configuration' )
   prefixListMatcher = CliMatcher.KeywordMatcher(
      'prefix-list', helpdesc='Add a prefix-list entry' )
   prefixListSeq = CliMatcher.KeywordMatcher(
      'seq', helpdesc='Specify Index in the sequence' )
   prefixListSeqnoMatcher = CliMatcher.IntegerMatcher( 0, PREFIX_SEQNO_MAX_VALUE,
      helpdesc='Index in the sequence' )
   prefixListRemark = CliMatcher.KeywordMatcher(
      'remark', helpdesc="Specify comment" )
   prefixListComment = CliMatcher.StringMatcher(
      helpname='Line', helpdesc='Comment, up to 100 characters' )
   prefixListPrefix = IpAddrMatcher.ipPrefixExpr( 'IP Prefix',
      'IP Prefix mask', 'IP address with mask len',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   prefixListPermitOrDeny = CliMatcher.EnumMatcher( {
      'permit': 'Specify prefix-list to accept',
      'deny': 'Specify prefix-list prefix to reject' } )
   prefixListEq = CliMatcher.KeywordMatcher(
      'eq', helpdesc='Specify prefix-list equal to bounds' )
   prefixListGe = CliMatcher.KeywordMatcher(
      'ge',
      helpdesc='Specify prefix-list prefix greater than or equal to bounds' )
   prefixListLe = CliMatcher.KeywordMatcher(
      'le',
      helpdesc='Specify prefix-list prefix less than or equal to bounds' )
   prefixListMaskLenMatcher = CliMatcher.IntegerMatcher( 1, 32,
      helpdesc='Mask length' )
   prefixListSource = CliMatcher.KeywordMatcher(
      'source', helpdesc='Source URL of prefix-list entries' )
   prefixListDuplicateToken = CliMatcher.KeywordMatcher(
      'duplicate', helpdesc='Duplicate handling option' )
   prefixListDuplicateMatcher = CliMatcher.EnumMatcher( {
      'ignore': 'Ignore duplicate prefixes',
      'override': 'Duplicate prefixes override previous ones'
      } )
   prefixListUrlMatcher = Url.UrlMatcher(
      lambda fs: fs.scheme in listUrlSchemes,
      "URL to load prefix-list entries" )
   prefixListResequenceToken = CliMatcher.KeywordMatcher(
      'resequence', helpdesc='Resequence the list' )
   prefixListStartSeq = CliMatcher.IntegerMatcher( 0, PREFIX_SEQNO_MAX_VALUE,
      helpdesc='Starting sequence number (default 10)' )
   prefixIncSeqRule = CliMatcher.IntegerMatcher( 1, PREFIX_SEQNO_MAX_VALUE,
      helpdesc='Step to increment the sequence number (default 10)' )
   replacement = CliMatcher.KeywordMatcher( 'replacement',
      helpdesc='Replacement of the matched AS path' )
   aggregateRole = CliMatcher.KeywordMatcher( matchContribAggAttrRoleStr,
      helpdesc="Role in BGP contributor-aggregate relation" )
   contributor = CliMatcher.KeywordMatcher( matchContribAggAttrContribStr,
      helpdesc="BGP aggregate's contributor" )
   aggregateAttr = CliMatcher.KeywordMatcher( matchContribAggAttrAggAttrStr,
      helpdesc="Route map to apply against the aggregate route" )
   isis = CliMatcher.KeywordMatcher( 'isis', 'IS-IS' )
   instance = CliMatcher.KeywordMatcher( 'instance', 'Route instance' )
   ospf = CliMatcher.KeywordMatcher( 'ospf', 'OSPF' )

class RtSooExtCommCliMatcher( CliMatcher.Matcher ):
   def __init__( self, helpdesc, acceptLongAsn=True, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )
      self.acceptLongAsn = acceptLongAsn
      completionTxt = 'ASN(asplain):nn{} or IP-address:nn'.format(
         ' or ASN(asdot):nn' if acceptLongAsn else '' )
      # note: completion_ is dynamic (RtPrefixCliMatcher overrides it)
      self.completion_ = CliParser.Completion( completionTxt, helpdesc, False )
      self.extCommRe_ = re.compile(
          r'(^(\d+\.\d+):(\d+)$)|'
           r'(^(\d+)(l|L)?:(\d+)$)|'
           r'(^(\d+)\.(\d+)\.(\d+)\.(\d+):(\d+)$)' )

      self.asplainCompletionRe_ = re.compile( r"^(\d+)(l|L)?(?::(\d*))?$" )
      self.asdotCompletionRe_ = re.compile( r"^(\d+\.\d*)(?::(\d*))?$" )
      self.ipCompletionRe_ = re.compile( r"^(\d+)\.(\d*)\.?(\d*)\.?(\d*):?(\d*)$" )

   def match( self, mode, context, token ):
      # must be AA:nn where AA:nn is one of
      #    AS(16-bit):nn(32-bit) or
      #    AS(32-bit):nn(16-bit) or
      #    ASDOT(32-bit):nn(16-bit) or
      #    IP(32-bit):nn(16-bit)
      m = self.extCommRe_.match( token )
      if m is None:
         return CliParserCommon.noMatch
      validExtComm = []
      for group in m.groups():
         if group is not None and ":" not in group:
            validExtComm.append( group )

      # check if optional ASN qualifier 'L' is present
      isLongAsn = False
      if ( len( validExtComm ) == 3 ) and ( validExtComm[ 1 ].upper() == 'L' ):
         validExtComm = validExtComm[ :1 ] + validExtComm[ 2: ]
         isLongAsn = True

      # if 2 elements, then ASN:nn where ASN may be 16-bit, 32-bit or ASDOT
      if len( validExtComm ) == 2:
         asn = asnStrToNum( validExtComm[ 0 ], minValue=0 )
         admin = safeInt( validExtComm[ 1 ] )
         if ( asn is None or asn > u32Max or
              admin is None or admin < 0 or admin > u32Max ):
            return CliParserCommon.noMatch

         # check if ASN value exceeds 2-octet limit
         if not isLongAsn and asn > u16Max:
            isLongAsn = True

         if isLongAsn and ( admin > u16Max or not self.acceptLongAsn ):
            return CliParserCommon.noMatch

      # 5 elements, must be IpAddr:nn
      elif len( validExtComm ) == 5:
         if ( int( validExtComm[ 0 ] ) > u8Max or
              int( validExtComm[ 1 ] ) > u8Max or
              int( validExtComm[ 2 ] ) > u8Max or
              int( validExtComm[ 3 ] ) > u8Max or
              int( validExtComm[ 4 ] ) > u16Max ):
            return CliParserCommon.noMatch
      # don't support anything else
      else:
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      # Match asplain, asdot and ip formats one by one. If there is a valid match
      # then return the completion text. The completion fails when nothing matches.

      if not token:
         return [ self.completion_ ]

      def validAsnComm( asn, admin, isLongAsn ):
         if isLongAsn and not self.acceptLongAsn:
            return False
         if asn is None or asn > u32Max:
            return False
         if admin and ( admin > u32Max or ( isLongAsn and admin > u16Max ) ):
            return False
         return True

      m = self.asplainCompletionRe_.match( token )
      if m:
         # [asn] [L] [nn]
         #   0    1    2
         groups = m.groups( '' )
         asn = int( groups[ 0 ] )
         longAsnQualifier = bool( groups[ 1 ] )
         admin = int( groups[ 2 ] ) if groups[ 2 ] else None
         isLongAsn = ( asn > u16Max or longAsnQualifier )
         if validAsnComm( asn, admin, isLongAsn ):
            return [ self.completion_ ]

      # Don't bother if the input has no dot character
      if "." not in token:
         return []

      if self.acceptLongAsn:
         m = self.asdotCompletionRe_.match( token )
         if m:
            # [as.dot] [nn]
            #    0       1
            groups = m.groups( '' )

            asn = None
            high, low = groups[ 0 ].split( "." )
            if not low and ":" in token:
               pass # incomplete input (NN.:NN)
            else:
               high = int( high )
               low = int( low ) if low else 0
               # higher bytes can't be zero to match the behavior in
               # asnStrToNum (used by match()).
               # pylint: disable-next=chained-comparison
               if ( high >= 1 and high <= u16Max and low >= 0 and low <= u16Max ):
                  asn = ( ( high << 16 ) | low )

            admin = int( groups[ 1 ] ) if groups[ 1 ] else None
            if validAsnComm( asn, admin, isLongAsn=True ):
               return [ self.completion_ ]

      # Not a valid IP format. This is checked early so we know the address
      # elements are entered from left to right with some non-empty values.
      if ".." in token:
         return []

      def validIpComm( ipElems, admin ):
         ipComplete = True
         for elem in ipElems:
            if elem is None:
               ipComplete = False
            elif elem > u8Max:
               return False

         if ":" in token and not ipComplete:
            # Delimiter seen but the ip address is not complete
            return False

         if admin is not None:
            if not ipComplete or admin > u16Max:
               return False

         return True

      m = self.ipCompletionRe_.match( token )
      if m:
         # [ip1] [ip2] [ip3] [ip4] [nn]
         #    0     1     2     3    4
         groups = m.groups( '' )
         ipElems = [ ( int( group ) if group else None ) for group in groups[ 0:4 ] ]
         admin = int( groups[ 4 ] ) if groups[ 4 ] else None
         if validIpComm( ipElems, admin ):
            return [ self.completion_ ]

      return []

   def __str__( self ):
      return 'ASN:nn or IP-address:nn'

class LinkBandwidthFormatCliMatcher( CliMatcher.Matcher ):
   def __init__( self, helpdesc, **kargs ):
      super().__init__(
            helpdesc=helpdesc, **kargs )
      self.extCommRe_ = re.compile( r'^(\d+):(\d+|\d+\.\d+)(K|M|G)?$' )
      self.extCommCompletionRe_ = re.compile( r'^(\d+)?:?(\d+\.\d+)?(K|M|G)?$' )

   def match( self, mode, context, token ):
      m = self.extCommRe_.match( token )
      if m is None:
         return CliParserCommon.noMatch
      validExtComm = []
      for group in m.groups():
         if group is not None and ":" not in group:
            validExtComm.append( group )
      # must be AA:nn.nn
      if len( validExtComm ) == 2 or len( validExtComm ) == 3:
         if safeInt( validExtComm[ 0 ] ) > u16Max or \
               safeInt( validExtComm[ 0 ] ) < 0:
            return CliParserCommon.noMatch
         lbw = LinkBwUnion()
         lbw.bw_f = safeFloat( validExtComm[ 1 ] )
         if lbw.bw_f is None:
            return CliParserCommon.noMatch
         if len( validExtComm ) == 3:
            multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
            lbw.bw_f *= multiplier[validExtComm[2].strip()]
         if lbw.bw_f < ufMin or lbw.bw_u >= u32Max:
            return CliParserCommon.noMatch
      # don't support anything else
      else:
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.extCommCompletionRe_.match( token )
      if m is None:
         return []
      lastValue = 0
      for group in m.groups( '0' ):
         if group == "" or group == "0":
            continue
         if lastValue == 1:
            if safeFloat( group ) is None:
               return []
            if safeFloat( group ) > uf32Max:
               return []
         elif lastValue == 2:
            if group not in ( 'K', 'M', 'G' ):
               return []
         else:
            if safeInt( group ) > u16Max:
               return []
         lastValue = lastValue + 1
      return [ CliParser.Completion( 'ASN:0.0-4294967295.0 or ASN:nn.nn(K|M|G)',
                                      self.helpdesc_, False ) ]

   def __str__( self ):
      return 'ASN:0.0-4294967295.0(K|M|G)'

class LinkBandwidthValueCliMatcher( CliMatcher.Matcher ):
   def __init__( self, helpdesc, **kargs ):
      super().__init__(
            helpdesc=helpdesc, **kargs )
      self.extCommRe_ = re.compile( r'^(\d+|\d+\.\d+)(K|M|G)?$' )
      self.extCommCompletionRe_ = re.compile( r'^(\d+|\d+\.\d+)?(K|M|G)?$' )

   def match( self, mode, context, token ):
      m = self.extCommRe_.match( token )
      if m is None:
         return CliParserCommon.noMatch
      validExtComm = []
      for group in m.groups():
         if group is not None:
            validExtComm.append( group )
      # must be nn.nn
      if len( validExtComm ) == 1 or len( validExtComm ) == 2:
         lbw = LinkBwUnion()
         lbw.bw_f = safeFloat( validExtComm[ 0 ] )
         if lbw.bw_f is None:
            return CliParserCommon.noMatch
         if len( validExtComm ) == 2:
            multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
            lbw.bw_f *= multiplier[validExtComm[1].strip()]
         if lbw.bw_f < ufMin or lbw.bw_u >= u32Max:
            return CliParserCommon.noMatch
      # don't support anything else
      else:
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.extCommCompletionRe_.match( token )
      if m is None:
         return []
      lastValue = 0
      for group in m.groups( '0' ):
         if group == "" or group == "0":
            continue
         if lastValue == 0:
            if safeFloat( group ) is None:
               return []
            if safeFloat( group ) > uf32Max:
               return []
         elif lastValue == 1:
            if group not in ( 'K', 'M', 'G' ):
               return []
         lastValue = lastValue + 1
      return [ CliParser.Completion( '0.0-4294967295.0 or nn.nn(K|M|G)',
                                     self.helpdesc_, False ) ]

   def __str__( self ):
      return '0.0-4294967295.0(K|M|G)'

class LinkBandwidthExp( CliCommand.CliExpression ):
   expression = "lbw ( any-lbw | ( range FIRST SECOND ) | LBW_VAL )"
   data = {
      'lbw': 'Link Bandwidth extended community (bits/sec)',
      'any-lbw': CliMatcher.KeywordMatcher(
         'any', helpdesc='Any route with a link bandwidth value' ),
      'range': 'Routes with link bandwidths within the range',
      'LBW_VAL': LinkBandwidthFormatCliMatcher(
         "Link Bandwidth extended community value as AS:bits/second" ),
      'FIRST': LinkBandwidthFormatCliMatcher(
         "Link Bandwidth extended community value as AS:bits/second" ),
      'SECOND': LinkBandwidthFormatCliMatcher(
         "Link Bandwidth extended community value as AS:bits/second" )
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      lbwAny = args.pop( 'any-lbw', False )
      lbwRange = args.pop( 'range', False )
      lbwVal = args.pop( 'LBW_VAL', False )
      if lbwAny:
         args[ 'LBW_VAL' ] = [ 'lbw any' ]
      elif lbwRange:
         first = args.pop( 'FIRST' )
         second = args.pop( 'SECOND' )
         lbwVal = "lbw range " + first + " " + second
         if isValidRange( lbwVal ):
            args[ 'LBW_VAL' ] = [ lbwVal ]
      elif lbwVal:
         args[ 'LBW_VAL' ] = [ 'lbw ' + lbwVal ]

class GenericExtCommsExp( CliCommand.CliExpression ):
   expression = "( rt RT_VAL ) | ( soo SOO_VAL ) | COLOR_EXP | EVPN_EXP"
   data = {
      'rt': 'Route Target extended community',
      'soo': 'Site-of-Origin extended community',
      'RT_VAL': RtSooExtCommCliMatcher( "VPN extended community" ),
      'SOO_VAL': RtSooExtCommCliMatcher( "VPN extended community" ),
      'COLOR_EXP': ColorExtCommunityExpression,
      'EVPN_EXP': EvpnExtCommunityExpression,
   }
   expression += " | OSPF_EXP"
   data[ 'OSPF_EXP' ] = OspfExtCommunityExpression

   @staticmethod
   def adapter( mode, args, argsList ):
      rtVals = args.pop( 'RT_VAL', [] )
      args[ 'RT_VAL' ] = [ 'rt ' + rt for rt in rtVals ]
      sooVals = args.pop( 'SOO_VAL', [] )
      args[ 'SOO_VAL' ] = [ 'soo ' + soo for soo in sooVals ]

def isSameMatchValue( rule, value ):
   valueType = MatchAttributes[ rule.option ][ 'type' ]
   if valueType == 'int':
      return rule.intValue == int( value )
   if valueType == 'string':
      return rule.strValue == str( value )
   if valueType == 'enum':
      return rule.intValue == \
         MatchAttributes[ rule.option ][ 'values' ][ value ]
   if valueType == 'ipaddr':
      return rule.ipAddrValue == str( value )
   if valueType == 'ip6addr':
      return rule.ip6AddrValue == value
   if valueType == 'u32range':
      return rule.u32Range.low == value.start and rule.u32Range.high == value.end
   if valueType == 'range':
      return rule.intValue == value.start and rule.int2Value == value.end
   if valueType == 'set':
      return set( value ) == set( rule.commListSet.commListSet )
   assert False

# Copy match value from one match rule to another
def updateMatchRule( destMatch, srcMatch ):
   valueType = MatchAttributes[ destMatch.option ][ 'type' ]
   if valueType == 'string':
      destMatch.strValue = srcMatch.strValue
   elif valueType == 'int':
      destMatch.intValue = srcMatch.intValue
   elif valueType == 'ipaddr':
      destMatch.ipAddrValue = srcMatch.ipAddrValue
   elif valueType == 'ip6addr':
      destMatch.ip6AddrValue = srcMatch.ip6AddrValue
   elif valueType == 'enum':
      destMatch.intValue = srcMatch.intValue
   elif valueType == 'range':
      destMatch.intValue = srcMatch.intValue
      destMatch.int2Value = srcMatch.int2Value
   elif valueType == 'u32range':
      destMatch.u32Range = srcMatch.u32Range
   elif valueType == 'set':
      destMatch.commListSet = srcMatch.commListSet
   else:
      assert False

# Compare the content of a CommunitySet to another
def isSameCommunitySet( destCommSet, srcCommSet ):
   if destCommSet is None and srcCommSet is None:
      return True
   elif ( destCommSet is None ) ^ ( srcCommSet is None ):
      return False

   if ( destCommSet.flag != srcCommSet.flag or
        destCommSet.useCommunityList != srcCommSet.useCommunityList or
        list( destCommSet.community ) != list( srcCommSet.community ) or
        list( destCommSet.communityListNameSet ) !=
        list( srcCommSet.communityListNameSet ) or
        destCommSet.commType != srcCommSet.commType ):
      return False

   return True

# Copy the content of a CommunitySet to another
def copyCommunitySet( destCommSet, srcCommSet ):
   destCommSet.community.clear()
   destCommSet.flag = srcCommSet.flag
   destCommSet.useCommunityList = srcCommSet.useCommunityList
   destCommSet.commType = srcCommSet.commType

   if not srcCommSet.useCommunityList:
      for value in srcCommSet.community:
         destCommSet.community[ value ] = True
   else:
      for name in srcCommSet.communityListNameSet:
         destCommSet.communityListNameSet[ name ] = True

def isSameMapEntry( entry1, entry2 ):
   '''
      Compares 2 MapEntry entries. If they are the same return True else
      return False.
   '''
   if entry1 is None and entry2 is None:
      return True
   if entry1 is None or entry2 is None:
      return False
   assert type( entry1 ) == type( entry2 ) # pylint: disable=unidiomatic-typecheck

   tacType = entry1.tacType
   # Tacc Introspection is used to iterate over the attributes
   for tacAttr in tacType.attributeQ:
      # This indicates the attribute is stateful and not aliases to another
      # attribute so if it is false we don't want to compare
      if not tacAttr.isIndependentDomainAttr:
         continue
      if tacType.isEntity and ( tacAttr.name == 'parent' ):
         continue
      if tacAttr.name == 'versionId':
         continue
      entry1Attr = getattr( entry1, tacAttr.name )
      entry2Attr = getattr( entry2, tacAttr.name )

      if tacAttr.isQueueCollection or tacAttr.isStackCollection or \
              tacAttr.isDynArray or tacAttr.isStaticArray:
         assert False, \
         "queue/stack/dynarray/static array collections are not supported"

      # For Collections we want to compare elements one by one
      elif tacAttr.isCollection:
         # Compare the set of collection keys
         if set( entry1Attr ) != set( entry2Attr ):
            return False
         # If tacAttr.isSet is True then would have been covered by above
         if not tacAttr.isSet:
            for k in entry1Attr:
               entry1Member = entry1Attr[ k ]
               entry2Member = entry2Attr[ k ]
               attrCmp = tacAttrCompare( tacAttr, entry1Member, entry2Member )
               if attrCmp is False:
                  return False
            # Check for elements of entry2Attr not in entry1Attr
      else:
         attrCmp = tacAttrCompare( tacAttr, entry1Attr, entry2Attr )
         if not attrCmp:
            return False
   return True

def tacAttrCompare( tacAttr, member1, member2 ):
   '''
      This function will compare the tacc attributes/members of a MapEntry and
      if the attribute is an instantiating collection we call isSameMapEntry
      again with the attributes to recursively compare nested entities.
   '''
   if tacAttr.instantiating:
      return isSameMapEntry( member1, member2 )
   elif tacAttr.memberType.isValue:
      return member1 == member2
   else:
      assert False, "non-instantiating pointer attributes are not supported"

def updateIfDifferent( src, dst ):
   if list( src.values() ) != list( dst.values() ):
      dst.clear()
      dst.update( src )
      return True
   return False

def copyAsPathPrepend( srcEntry, destEntry ):
   changed = updateIfDifferent( srcEntry.asPathPrepend, destEntry.asPathPrepend )
   if destEntry.asPathPrependRepeat != srcEntry.asPathPrependRepeat:
      changed = True
      destEntry.asPathPrependRepeat = srcEntry.asPathPrependRepeat
   if changed:
      destEntry.asPathPrependUpdateCount += 1

def copyAsPathReplace( srcEntry, destEntry ):
   changed = updateIfDifferent( srcEntry.asPathReplace, destEntry.asPathReplace )
   if destEntry.asPathReplaceRepeat != srcEntry.asPathReplaceRepeat:
      changed = True
      destEntry.asPathReplaceRepeat = srcEntry.asPathReplaceRepeat
   if changed:
      destEntry.asPathReplaceUpdateCount += 1

# Copy content of one map entry to another
def copyMapEntry( destEntry, srcEntry, routeCtx=True ):
   destEntry.permit = srcEntry.permit
   destEntry.statementName = srcEntry.statementName
   destEntry.statementNameFromCli = srcEntry.statementNameFromCli

   # copy description lines
   if srcEntry.description:
      destEntry.description.clear()
      for index, desc in enumerate( srcEntry.description.values() ):
         destEntry.description[ index ] = desc
   else:
      destEntry.description.clear()

   # copy sub route-map.
   if routeCtx:
      SubRMap = Tac.Type( "Routing::RouteMap::SubRouteMap" )
      destSubRouteMap = SubRMap( srcEntry.subRouteMap.name,
                                 srcEntry.subRouteMap.invert )
      destEntry.subRouteMap = destSubRouteMap

   # copy Match Rules
   for op in srcEntry.matchRule:
      srcMatch = srcEntry.matchRule[ op ]
      if op in destEntry.matchRule:
         destMatch = Tac.nonConst( destEntry.matchRule[ op ] )
      else:
         destMatch = Tac.Value( "Routing::RouteMap::MatchRule", op )
      updateMatchRule( destMatch, srcMatch )
      destMatch.exact = srcMatch.exact
      destMatch.invert = srcMatch.invert
      destMatch.orResults = srcMatch.orResults
      destEntry.matchRule.addMember( destMatch )
   destEntry.matchTagSetName = srcEntry.matchTagSetName

   for op in destEntry.matchRule:
      if not op in srcEntry.matchRule:
         del destEntry.matchRule[ op ]

   # copy Set Actions
   if not routeCtx:
      return
   destEntry.setFlags = srcEntry.setFlags
   for attr in SetActionInfo: # pylint: disable=consider-using-dict-items
      if attr == "asPathPrepend":
         copyAsPathPrepend( srcEntry, destEntry )
      elif attr == "asPathReplace":
         copyAsPathReplace( srcEntry, destEntry )
      elif "community" in attr.lower():
         # Following code will be executed for attributes, in a MapEntry:
         #   - communityAddReplace
         #   - communityDelete
         #   - extCommunity
         #   - largeCommunity
         destCommSet = getattr( destEntry, attr )
         srcCommSet = getattr( srcEntry, attr )
         # If the community value hasn't changed then continue
         if isSameCommunitySet( destCommSet, srcCommSet ):
            continue

         setattr( destEntry, attr, None )
         if srcCommSet is not None:
            # Initialize the attribute
            setattr( destEntry, attr, ( attr, ) )
            destCommSet = getattr( destEntry, attr )
            copyCommunitySet( destCommSet, srcCommSet )

         # Fetch "communityAddReplaceVersion" or "communityDeleteVersion" or
         # "extCommunityVersion" or "largeCommunityVersion" or
         # "communityFilterVersion" or "extCommunityFilterVersion" or
         # "largeCommunityFilterVersion" attribute and
         # increment it by '1' to indicate the change in config
         version = getattr( destEntry, attr + 'Version' )
         setattr( destEntry, attr + 'Version', version + 1 )
         # `versionId` is updated in `Context.commit`
      elif SetActionInfo[ attr ][ 'enabled' ]:
         # TACC doesn't generate these mutators for *nextHop* fields
         if attr not in SetNextHopArgTransformer.tacAttrToArgTag:
            setattr( destEntry, attr, getattr( srcEntry, attr ) )
   destEntry.lbwDisplay = srcEntry.lbwDisplay
   destEntry.linkBandwidthRefDisplay = srcEntry.linkBandwidthRefDisplay
   destEntry.linkBandwidthRefExtComm = srcEntry.linkBandwidthRefExtComm
   destEntry.extCommLinkBandwidthSetFlag = srcEntry.extCommLinkBandwidthSetFlag
   destEntry.linkBandwidthDeleteAsnDisplay = srcEntry.linkBandwidthDeleteAsnDisplay
   destEntry.extCommLinkBandwidthSetDeleteFlag = \
         srcEntry.extCommLinkBandwidthSetDeleteFlag
   destEntry.setNextHopArg = srcEntry.setNextHopArg

def copyTacEntity( destination, source, copyHandler ):
   """
   Function to create a copy of the given Tac::Entity 'source'
   to destination.
   """
   entityCopy = Tac.newInstance( "Cli::Session::EntityCopy" )
   entityCopy.handler = Tac.newInstance(
       "Cli::Session::EntityCopyGlobalSessionHandler" )
   entityCopy.handler.handlerDir = handlerDir
   entityCopy.handler.handlerTable = handlerDir.inCopyHandlerTable
   copyHandler.handleEntity( entityCopy, destination, source, "" )

asdotNumMatcher = CliMatcher.DottedIntegerMatcher(
      AS_PATH_MIN, U16_MAX_VALUE,
      helpname='<1-65535>.<0-65535>',
      helpdesc='AS Number in asdot notation',
      subLbound=0, subUbound=U16_MAX_VALUE )

# Vaild range is <1-65535>.<0-65535>
asdotRangeMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( U16_MAX_VALUE + 1, U32_MAX_VALUE ),
      noSingletons=False,
      maxRanges=1,
      helpdesc='AS number or range of AS numbers in Asdot notation',
      numberFormat=MultiRangeRule.NumberFormat.DOTTED,
      value=lambda mode, grList:
         AsnRange( start=grList.ranges()[ 0 ][ 0 ],
                  end=grList.ranges()[ 0 ][ 1 ] ) )

asplainNumMatcher = CliMatcher.IntegerMatcher( AS_PATH_MIN, AS_PATH_MAX,
                                               helpdesc='Autonomous system number' )

asplainNumRangeMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( 1, U32_MAX_VALUE ),
      noSingletons=False,
      maxRanges=1,
      helpdesc='AS number or range of AS numbers in numeric form',
      value=lambda mode, grList:
         AsnRange( start=grList.ranges()[ 0 ][ 0 ],
                     end=grList.ranges()[ 0 ][ 1 ] ) )


asdotReplaceMatcher = CliMatcher.DottedIntegerMatcher(
      AS_PATH_MIN, U16_MAX_VALUE,
      helpname='<1-65535>.<0-65534>',
      helpdesc='AS Number in asdot notation',
      subLbound=0, subUbound=U16_MAX_VALUE - 1 )

asplainReplaceMatcher = CliMatcher.IntegerMatcher( AS_PATH_MIN, AS_PATH_MAX - 1,
                                       helpdesc='Autonomous system number' )

class AsNumCliExpr( CliCommand.CliExpression ):
   expression = 'AS_PLAIN_NUM | AS_DOT_NUM'
   data = { 'AS_PLAIN_NUM': asplainNumMatcher,
            'AS_DOT_NUM': asdotNumMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AS_DOT_NUM' in args:
         args[ 'AS_NUM' ] = asnStrToNum( args[ 'AS_DOT_NUM' ] )
      elif 'AS_PLAIN_NUM' in args:
         args[ 'AS_NUM' ] = args[ 'AS_PLAIN_NUM' ]

class AsNumWithAutoCliExpr( CliCommand.CliExpression ):
   expression = 'AS_DOT_NUM | AS_PLAIN_NUM | auto'
   data = { 'AS_PLAIN_NUM': asplainNumMatcher,
            'AS_DOT_NUM': asdotNumMatcher,
            'auto': CommonTokens.asAuto
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      mode.processAsArgs( args, argsList )

class AsReplaceWithAutoCliExpr( CliCommand.CliExpression ):
   expression = 'AS_DOT_NUM | AS_PLAIN_NUM | auto'
   data = { 'AS_PLAIN_NUM': asplainReplaceMatcher,
            'AS_DOT_NUM': asdotReplaceMatcher,
            'auto': CommonTokens.asAuto
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      mode.processAsArgs( args, argsList )

class AsNumRangeCliExpr( CliCommand.CliExpression ):
   expression = 'AS_DOT_NUM | AS_PLAIN_NUM '
   data = { 'AS_PLAIN_NUM': asplainNumRangeMatcher,
            'AS_DOT_NUM': asdotRangeMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AS_DOT_NUM' in args:
         args[ 'AS_RANGE' ] = args[ 'AS_DOT_NUM' ]
      elif 'AS_PLAIN_NUM' in args:
         args[ 'AS_RANGE' ] = args[ 'AS_PLAIN_NUM' ]


# Functions for dynamic name rules
def getStandardAccessListNames( mode ):
   names = []
   ipConfig = aclConfig.config['ip'].acl
   for name in ipConfig:
      if ipConfig[ name ].standard:
         names.append( name )
   return names

def getPrefixListNames( mode ):
   return aclListConfig.prefixList

def getIpv6PrefixListNames( mode ):
   return aclListConfig.ipv6PrefixList

def getAsPathListNames( mode ):
   return aclListConfig.pathList

def getCommunityListNames( mode ):
   return aclListConfig.communityList

def getSubRouteMapNames( mode ):
   mapNames = set( mapConfig.routeMap )
   routeMapName = mode.routeMapContext.mapName()
   mapNames.discard( routeMapName )
   return mapNames

#------------------------------------------------------------------------
# Editing context
#
# A context is created for each child mode. The context either copies
# an existing sequence entry or creates new entry.
#
# The editedEntries_ hold all sequences that are either new or edited.
# If a "no" command was issued, the "match" or "set" is removed from
# the entry. At commit the old entry is completely replaced with the
# entry in context.
#
# Note that a "no route-map" to remove a sequence is outside of the
# route-map config mode and handled in deleteRouteMapMode.
#------------------------------------------------------------------------
class Context:
   def __init__( self, mapName ):
      self.map_ = None
      self.mapName_ = mapName
      # NOTE: It appers that following two attributes describe the same thing
      # and are used interchangeably. Perhaps one of them can be removed.
      self.editedEntries_ = {}
      self.currentEntry_ = None  # This is the entry that gets edited by CLI
      # This is backup copy we check against the currentEntry_  when we commit
      # for changes that were made to the entry by the CLI
      self.originalEntry_ = None
      self.copyEntry = False

      if mapName in mapConfig.routeMap:
         self.map_ = mapConfig.routeMap[ mapName ]

   def copyEditEntry( self, entry, permit ):
      # Add an edit entry by copying a sysdb entry
      newEntry = Tac.newInstance(
         'Routing::RouteMap::MapEntry', entry.seqno )
      copyMapEntry( newEntry, entry )
      # Even if we enter and exit without making changes we changed permit
      if entry.permit != permit:
         self.copyEntry = True
      newEntry.permit = permit
      self.editedEntries_[ entry.seqno ] = newEntry
      self.currentEntry_ = newEntry
      # Make a copy of entry to compare against when we commit to check for changes
      self.originalEntry_ = Tac.newInstance( 'Routing::RouteMap::MapEntry',
                                             entry.seqno )
      copyMapEntry( self.originalEntry_, entry )
      self.originalEntry_.permit = permit
      return newEntry

   def newEditEntry( self, permit, seqno, statementName, statementNameFromCli ):
      # Add an edit entry by creating a new entry
      entry = Tac.newInstance(
         'Routing::RouteMap::MapEntry', seqno )
      entry.permit = permit
      entry.statementName = statementName
      entry.statementNameFromCli = statementNameFromCli
      self.editedEntries_[ seqno ] = entry
      self.currentEntry_ = entry
      # Make a copy of entry to compare against when we commit to check for changes
      self.originalEntry_ = Tac.newInstance( 'Routing::RouteMap::MapEntry',
                                             entry.seqno )
      self.originalEntry_.permit = permit
      self.originalEntry_.statementName = statementName
      self.originalEntry_.statementNameFromCli = statementNameFromCli
      self.copyEntry = True
      return entry

   def removeEntry( self, seqno ):
      if not seqno in self.editedEntries_:
         return
      del self.editedEntries_[ seqno ]

   def mapName( self ):
      return self.mapName_

   def routeMap( self ):
      return self.map_ # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      # Commit current map. Create a new map if not exist.
      if self.map_ is None:
         if self.mapName_ in mapConfig.routeMap:
            self.map_ = mapConfig.routeMap[ self.mapName_ ]
         else:
            self.map_ = mapConfig.routeMap.newMember( self.mapName_ )
      # Update map_ if needed and update version number if there were changes
      for seqno in self.editedEntries_: # pylint: disable=consider-using-dict-items
         # if there has been a new entry or change in permit we want to update map_
         # and also if the mapEntry's aren't the same
         if self.copyEntry or \
            not isSameMapEntry( self.currentEntry_, self.originalEntry_ ):
            if seqno in self.map_.mapEntry:
               entry = self.map_.mapEntry[ seqno ]
            else:
               entry = self.map_.mapEntry.newMember( seqno )
            updateMatchTagSetCache( entry, self.editedEntries_[ seqno ],
                                    self.mapName_, seqno )
            copyMapEntry( entry, self.editedEntries_[ seqno ] )
            if toggleAirStreamPolicyDefinitionsEnabled():
               entry.bumpVersionId()
               t8( 'Assigned new version', entry.versionId.val,
                   'to the map', self.mapName_, 'entry', seqno )
            self.map_.version += 1
         else:
            t8( "Entered and exited Route-Map %s  without making changes" %
                  ( self.mapName_ ) )
      t0( 'Commited map %s, edited entries %s' % (
         self.mapName(), len( self.editedEntries_ ) ) )

def maybeClearTagSetCache( tagSetName, mapName, seqno ):
   tagSet = aclListConfig.tagSet.get( tagSetName )
   if tagSet:
      entryKey = RouteMapEntryKey( mapName, seqno )
      del tagSet.routeMapEntryCache[ entryKey ]
      # If the cache and tagValue are both empty, delete the tagSet entry
      if len( tagSet.routeMapEntryCache ) == 0 and \
         len( tagSet.tagValue ) == 0:
         del aclListConfig.tagSet[ tagSetName ]

def updateMatchTagSetCache( oldEntry, newEntry, mapName, seqno ):
   oldTag = oldEntry.matchTagSetName
   newTag = newEntry.matchTagSetName
   if oldTag != newTag:
      if oldTag:
         # Clean up old tag
         maybeClearTagSetCache( oldTag, mapName, seqno )
      if newTag:
         # Update new or existing TagSet cache
         tagSet = aclListConfig.tagSet.get( newTag )
         if not tagSet:
            tagSet = aclListConfig.newTagSet( newTag )
         tagSet.routeMapEntryCache.add( RouteMapEntryKey( mapName, seqno ) )

#------------------------------------------------------------------------
# route-map mode
#------------------------------------------------------------------------
def hasRouteMapCfgCycle( routeMapName, refRouteMapName ):
   """ Detect route map cycles/loops

   Args:
   routeMapName - current route map in which other route map will be referenced
   refRouteMapName - the referenced route map

   Return:
   True if loop detected, else False

   Examples of configuration loops:
   1.
    route-map A permit 1
        sub-route-map A
   2.
    route-map A permit 1
        sub-route-map B
    route-map B permit 1
        sub-route-map A
   3.
    route-map A permit 1
        match aggregate-role contributor aggregate-attributes A
   4.
    route-map A permit 1
        match aggregate-role contributor aggregate-attributes B
    route-map B permit 1
        sub-route-map A
   """
   if routeMapName == refRouteMapName:
      return True

   queue = [ refRouteMapName ]
   # BFS walk from subRouteMap to see if there exists a path from subRouteMap
   # back to its referring routemap.
   while queue:
      mapName = queue.pop( 0 )
      if mapName in mapConfig.routeMap:
         for mapEntry in mapConfig.routeMap[ mapName ].mapEntry.values():
            # this match rule holds a reference to a route-map - check it too
            hasMatchContribAggAttr = matchContribAggAttrOption in mapEntry.matchRule
            if not mapEntry.subRouteMap.name and not hasMatchContribAggAttr:
               continue
            if mapEntry.subRouteMap.name == routeMapName:
               return True
            if hasMatchContribAggAttr:
               rmName = mapEntry.matchRule[ matchContribAggAttrOption ].strValue
               if rmName == routeMapName:
                  return True
               queue.append( rmName )
            queue.append( mapEntry.subRouteMap.name )
   return False

subRmCfgCycleErrMsgBase = 'Sub-route-map results in cycle'

class RouteMapMode( RouteMapCliMode, BasicCli.ConfigModeBase ):
   name = "Route Map Configuration"
   policyConstructNameByMatchRuleOption = {
      RouteMapMatchOption.matchCommunity: 'community-list',
      RouteMapMatchOption.matchExtCommunity: 'extcommunity-list',
      RouteMapMatchOption.matchLargeCommunity: 'large-community-list',
      RouteMapMatchOption.matchAsPathList: 'as-path list',
      RouteMapMatchOption.matchAccessList: 'access-list',
      RouteMapMatchOption.matchIpAddrDynamicPrefixList: 'dynamic prefix-list',
      RouteMapMatchOption.matchPrefixList: 'prefix-list',
      RouteMapMatchOption.matchNextHopPrefixList: 'prefix-list',
      RouteMapMatchOption.matchResolvedNextHopPrefixList: 'prefix-list',
      RouteMapMatchOption.matchIpv6AddrDynamicPrefixList: 'ipv6 dynamic prefix-list',
      RouteMapMatchOption.matchIpv6PrefixList: 'ipv6 prefix-list',
      RouteMapMatchOption.matchIpv6NextHopPrefixList: 'ipv6 prefix-list',
      RouteMapMatchOption.matchIpv6ResolvedNextHopPrefixList: 'ipv6 prefix-list',
      RouteMapMatchOption.matchRouterId: 'prefix-list',
   }
   communityListTypeByMatchRuleOption = {
      RouteMapMatchOption.matchCommunity: CommunityType.communityTypeStandard,
      RouteMapMatchOption.matchExtCommunity: CommunityType.communityTypeExtended,
      RouteMapMatchOption.matchLargeCommunity: CommunityType.communityTypeLarge
   }

   # Each mode object has a session object. We associate the routemap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      RouteMapCliMode.__init__( self, context.mapName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.routeMapContext = context
      self.commentKeyCache = ""

   # Because the long-mode-key for RouteMapMode is
   # "route-map-mapName", not unique to each of individual mode, we
   # have to override filterExp() and commentKey() to give the
   # complete unique string for 'show active' and 'show comment'.
   def filterExp( self ):
      theEntry = self.routeMapContext.currentEntry()
      return "^route-map %s %s %s$" % ( re.escape( self.routeMapContext.mapName() ),
                                        matchPermitEnum[ theEntry.permit ],
                                        theEntry.seqno )

   def commentKey( self ):
      if self.routeMapContext is None:
         return self.commentKeyCache

      theEntry = self.routeMapContext.currentEntry()
      return generateCommentKey( self.routeMapContext.mapName(),
                                 matchPermitEnum[ theEntry.permit ],
                                 theEntry.seqno )

   def onExit( self ):
      t0( 'RouteMapMode onExit...' )

      # Pre-saving the commentKeyCache here to handle the config replace use cases.
      # It is saved here because the routeMapContext could be modified after initial
      # context creation. Ref - gotoRouteMapMode(), evaluatedRouteMap(), BUG #536882
      if self.routeMapContext is not None:
         theEntry = self.routeMapContext.currentEntry()
         self.commentKeyCache = generateCommentKey( self.routeMapContext.mapName(),
                                    matchPermitEnum[ theEntry.permit ],
                                    theEntry.seqno )

      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.routeMapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.routeMapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.routeMapContext
      self.routeMapContext = None
      context.commit()

   def setSubRouteMap( self, subRouteMapName, subInvert=False ):
      context = self.routeMapContext

      if subRouteMapName and \
         hasRouteMapCfgCycle( context.mapName(), subRouteMapName ):
         self.addError( subRmCfgCycleErrMsgBase )
         return

      if subInvert and \
         getEffectiveProtocolModel( self ) != ProtoAgentModel.multiAgent:
         self.addWarning( "Routing protocol model multi-agent must be"
                          " configured for sub-route-map invert-result" )

      # In case of no or default variant of cmd empty string will be passed
      # and there is no need for any checks.
      entry = context.currentEntry()

      SubRMap = Tac.Type( "Routing::RouteMap::SubRouteMap" )
      subRouteMap = SubRMap( subRouteMapName, subInvert )
      entry.subRouteMap = subRouteMap

   def setMatchValue( self, no, name, value, invertMatch=False ):
      context = self.routeMapContext
      entry = context.currentEntry()
      if name in entry.matchRule:
         invertMatchChanged = False
         exactMatchChanged = False
         orResultsChanged = False
         rule = Tac.nonConst( entry.matchRule[ name ] )
         if name in ( "matchCommunity",
                      "matchExtCommunity",
                      "matchLargeCommunity",
                      matchContribAggAttrOption ):
            invertMatchChanged = rule.invert != ( "invert-result" in value[ 1: ] )
            rule.invert = "invert-result" in value[ 1: ]
            exactMatchChanged = rule.exact != ( "exact-match" in value[ 1: ] )
            rule.exact = "exact-match" in value[ 1: ]
            orResultsChanged = rule.orResults != "or-results" in value[ 1 : ]
            rule.orResults = "or-results" in value[ 1 : ]
            value = value[ 0 ]
         elif name == 'matchPrefixList' or name == 'matchIpv6PrefixList':
            invertMatchChanged = rule.invert != invertMatch
            rule.invert = invertMatch
         elif name == 'matchCommunityInstances' or name == 'matchAsPathLength':
            invertMatchChanged = rule.invert != value.invert
            rule.invert = value.invert
         elif invertMatch:
            invertMatchChanged = rule.invert != invertMatch
            rule.invert = invertMatch

         if no:
            # delete only if the value is the same
            if isSameMatchValue( rule, value ):
               del entry.matchRule[ name ]
            else:
               return
         elif exactMatchChanged or \
                  invertMatchChanged or \
                  orResultsChanged or \
                  not isSameMatchValue( rule, value ):
            if self.checkMatchIsUnconfigured( rule, value ):
               setMatchRuleValue( rule, value )
               entry.matchRule.addMember( rule )
         return
      elif no:
         return
      # A route-map entry can contain only matchPrefixList or
      # matchIpv6PrefixList rule, but not both together
      elif name == 'matchPrefixList':
         del entry.matchRule[ 'matchIpv6PrefixList' ]
      elif name == 'matchIpv6PrefixList':
         del entry.matchRule[ 'matchPrefixList' ]
      # A route-map entry can contain only matchNextHop or
      # matchIpv6NextHop rule, but not both together
      elif name == 'matchNextHop':
         del entry.matchRule[ 'matchIpv6NextHop' ]
      elif name == 'matchIpv6NextHop':
         del entry.matchRule[ 'matchNextHop' ]
      # A route-map entry can contain only matchIpAddressDynamicPrefixList or
      # matchIpv6AddressDynamicPrefixList rule, but not both together
      elif name == 'matchIpAddrDynamicPrefixList':
         del entry.matchRule[ 'matchIpv6AddrDynamicPrefixList' ]
      elif name == 'matchIpv6AddrDynamicPrefixList':
         del entry.matchRule[ 'matchIpAddrDynamicPrefixList' ]
      elif name == matchContribAggAttrOption:
         del entry.matchRule[ matchContribAggAttrOption ]

      rule = Tac.Value( "Routing::RouteMap::MatchRule", name )
      if name in ( "matchCommunity",
                   "matchExtCommunity",
                   "matchLargeCommunity",
                   matchContribAggAttrOption ):
         rule.invert = "invert-result" in value[ 1: ]
         rule.exact = "exact-match" in value[ 1: ]
         rule.orResults = "or-results" in value[ 1 : ]
         value = value[ 0 ]
      elif name == 'matchCommunityInstances' or name == 'matchAsPathLength':
         rule.invert = value.invert
      elif invertMatch:
         rule.invert = invertMatch
      # Warning message if used in gated
      elif name == 'matchOriginAs':
         if getEffectiveProtocolModel( self ) == ProtoAgentModel.ribd:
            self.addWarning( "command `match origin-as` only supported in "
                             "multi-agent mode" )
      if self.checkMatchIsUnconfigured( rule, value ):
         setMatchRuleValue( rule, value )
         entry.matchRule.addMember( rule )

   def checkMatchIsUnconfigured( self, rule, values ):
      """ @brief Checks whether a match rule refers to an existing policy construct.
                 In case of error, it prints a error message on the CLI.
          @param [in] rule     The matching rule (contains match statement data).
          @param [in] values   The values (references) to match.
          @return True if the check passes, False otherwise.
          @note The check control is configured by a knob.
          @see Config//RouteMap.tac for the knob.
      """
      if ( not mapConfig.routeMapPolicyReferenceUnconfiguredError or
           not self.session_.isInteractive() ):
         return True

      unconfiguredValues = []
      # Check for community-list, extcommunity-list, large-community-list
      if rule.option in ( RouteMapMatchOption.matchCommunity,
                          RouteMapMatchOption.matchExtCommunity,
                          RouteMapMatchOption.matchLargeCommunity ):
         unconfiguredValues = getUnconfiguredCommunityLists(
            values, self.communityListTypeByMatchRuleOption[ rule.option ] )
      elif rule.option == RouteMapMatchOption.matchAsPathList:
         unconfiguredValues = getUnconfiguredAsPathLists( [ values ] )
      elif rule.option == RouteMapMatchOption.matchAccessList:
         unconfiguredValues = getUnconfiguredAccessLists( [ values ] )
      elif rule.option in ( RouteMapMatchOption.matchPrefixList,
                            RouteMapMatchOption.matchNextHopPrefixList,
                            RouteMapMatchOption.matchResolvedNextHopPrefixList,
                            RouteMapMatchOption.matchRouterId ):
         unconfiguredValues = getUnconfiguredPrefixLists( [ values ],
                                                          ipv6=False,
                                                          dynamic=False )
      elif rule.option == RouteMapMatchOption.matchIpAddrDynamicPrefixList:
         unconfiguredValues = getUnconfiguredPrefixLists( [ values ],
                                                          ipv6=False,
                                                          dynamic=True )
      elif rule.option in ( RouteMapMatchOption.matchIpv6PrefixList,
                            RouteMapMatchOption.matchIpv6NextHopPrefixList,
                            RouteMapMatchOption.matchIpv6ResolvedNextHopPrefixList ):
         unconfiguredValues = getUnconfiguredPrefixLists( [ values ],
                                                          ipv6=True,
                                                          dynamic=False )
      elif rule.option == RouteMapMatchOption.matchIpv6AddrDynamicPrefixList:
         unconfiguredValues = getUnconfiguredPrefixLists( [ values ],
                                                          ipv6=True,
                                                          dynamic=True )

      if unconfiguredValues:
         self.addError( '{} "{}" is not configured'.format(
            self.policyConstructNameByMatchRuleOption[ rule.option ],
            ', '.join( sorted( unconfiguredValues ) ) ) )
         return False
      return True

   # pylint: disable-next=inconsistent-return-statements
   def getCurrentRmapEntry( self, attr ):
      '''
      Gets the current entry in RouteMapContext
      '''
      context = self.routeMapContext
      if context is None:
         t0( 'Error: setActionValue %s has no context' % attr )
         return

      return context.currentEntry()

   def currentFlagEqualsSetFlag( self, entry, currFlag, commType ):
      '''
      Returns true if the current flag present in configuration (e.g. if
      configuration is "set extcommunity rt 1:2 delete", currFlag will be
      'delete') is same as the saved set flag (e.g. entry.extCommunity.flag)
      '''
      setFlag = getattr( entry, commType ).flag
      if not currFlag.startswith( "commSet" ):
         currFlag = "commSet%s" % currFlag.capitalize()
      return currFlag == setFlag

   def currentFlagNoneOrNotEqualsSetFlag( self, entry, currFlag, commType ):
      '''
      Returns true if the current flag present in configuration (e.g. if
      configuration is "set extcommunity rt 1:2 delete", currFlag will be
      'delete') is None or different from the saved set flag (e.g.
      entry.extCommunity.flag) i.e. additive/delete or delete/additive
      '''
      setFlag = getattr( entry, commType ).flag
      if not currFlag.startswith( "commSet" ):
         currFlag = "commSet%s" % currFlag.capitalize()
      return ( currFlag == "commSetNone" or
               ( currFlag != setFlag and
                 ( currFlag != "commSetDefault" and setFlag != "commSetDefault" ) ) )

   # pylint: disable-next=inconsistent-return-statements
   def setActionValueLargeComm( self, no, entry, value ):
      '''
      Handlers '[no] set large-community ...' configuration commands
      '''
      if no:
         # In this case, we only want to unset the *specified* values, rather than
         # the entire largeCommunity attribute.
         toDelete = value
         # For example, `no set large-community 123:456:789 additive' would result
         # in `toDelete' being ['123:456:789', 'additive']. So, for each
         # item in `toDelete', we specifically unset that item (if it is set).
         for comm in toDelete:
            if comm in ( 'none', 'additive', 'delete' ):
               if self.currentFlagEqualsSetFlag( entry, comm, 'largeCommunity' ):
                  entry.largeCommunity.flag = 'commSetDefault'
               continue

            cval = getLargeCommValue( comm )
            del entry.largeCommunity.community[ cval ]

         if ( entry.largeCommunity.community or
              entry.largeCommunity.flag != 'commSetDefault' ):
            # Some settings remain, so don't clean up completely.
            return False
         return True
      else:
         if entry.largeCommunity is None:
            entry.largeCommunity = ( 'largeCommunity', )
         if entry.largeCommunity.flag == 'commSetNone':
            flag = 'commSetDefault'
         else:
            flag = entry.largeCommunity.flag
         commValues = []
         for comm in value:
            if comm in ( 'none', 'additive', 'delete' ):
               flag = 'commSet%s' % comm.capitalize()
               if comm == 'none':
                  break
            else:
               cval = getLargeCommValue( comm )
               commValues.append( cval )
         if self.currentFlagNoneOrNotEqualsSetFlag( entry, flag, 'largeCommunity' ):
            entry.largeCommunity.community.clear()
         if entry.largeCommunity.useCommunityList:
            entry.largeCommunity.communityListNameSet.clear()
            entry.largeCommunity.useCommunityList = False
         for value in commValues: # pylint: disable=redefined-argument-from-local
            entry.largeCommunity.community[ value ] = True
         entry.largeCommunity.flag = flag

   def resetLbwDelete( self, entry, lbwDeleteAny=False,
                       lbwDeleteAsn=False, lbwDeleteAsnLbw=False ):
      '''
      Based on delete type (i.e. one of 'lbw delete', 'lbw asn 10 delete',
      'lbw 10:100G delete'), this function resets the
      extCommLinkBandwidthSetDeleteFlag and linkBandwidthDeleteAsnDisplay
      to default.
      '''
      deleteType = ""
      if lbwDeleteAny:
         # definitely 'lbw delete' case
         deleteType = "extCommLinkBandwidthSetDeleteAny"
      elif lbwDeleteAsn:
         # definitely 'lbw asn 10 delete' case
         deleteType = "extCommLinkBandwidthSetDeleteAsn"
      elif lbwDeleteAsnLbw:
         # definitely 'lbw 10:100G delete' case
         deleteType = "extCommLinkBandwidthSetDeleteAsnLbw"
      else:
         # if we get here, we shouldn't make any change
         return

      # We only reset the flag if 'no' config being currently
      # parsed matches with the saved configuration. For example,
      # say cli-saved configuration is
      # 'set extcommunity lbw asn 10 delete'. In that case, the
      # entry.extCommLinkBandwidthSetDeleteFlag will be
      # extCommLinkBandwidthSetDeleteAsn. We only should set back the
      # flag to default if we now have
      # 'no set extcommunity lbw asn 10 delete'. If we rather have
      # 'no set extcommunity lbw delete', we shouldn't set flag to
      # default as it should be no-op.
      if ( entry.extCommLinkBandwidthSetDeleteFlag ==
           getattr( ExtCommLinkBandwidthSetDeleteType, deleteType ) ):
         entry.extCommLinkBandwidthSetDeleteFlag = \
            ExtCommLinkBandwidthSetDeleteType.\
            extCommLinkBandwidthSetDeleteDefault
         entry.linkBandwidthDeleteAsnDisplay = ""

   def resetLbwRegeneration( self, entry, comm ):
      '''
      Based on regeneration type (i.e. aggregate, aggregate ref, divide equal,
      divide ratio), this function resets the extCommLinkBandwidthSetFlag,
      linkBandwidthRefDisplay and linkBandwidthRefExtComm to defaults.
      '''
      valueSplit = comm.split()
      regenType = ""
      if valueSplit[ 1 ] == 'aggregate':
         # E.g. 'lbw aggregate', or 'lbw aggregate 1G'.
         if len( valueSplit ) == 2:
            regenType = 'extCommLinkBandwidthSetAggregate'
         elif len( valueSplit ) == 3:
            if entry.linkBandwidthRefDisplay == valueSplit[ 2 ]:
               regenType = 'extCommLinkBandwidthSetAggregateRef'
      elif valueSplit[ 1 ] == 'divide':
         # Either 'lbw divide equal' or 'lbw divide ratio'.
         if valueSplit[ 2 ] == 'equal':
            regenType = 'extCommLinkBandwidthSetDivideEqual'
         elif valueSplit[ 2 ] == 'ratio':
            regenType = 'extCommLinkBandwidthSetDivideRatio'

      if ( entry.extCommLinkBandwidthSetFlag ==
           getattr( ExtCommLinkBandwidthSetType, regenType ) ):
         entry.extCommLinkBandwidthSetFlag = \
            ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDefault
         entry.linkBandwidthRefDisplay = ""
         entry.linkBandwidthRefExtComm = ufMin

   def isNotLbwRegenerationConfig( self, comm ):
      '''
      Returns True if this is a lbw config but not regeneration config.
      '''
      valueSplit = comm.split()
      return ( ( valueSplit[ 0 ] == 'lbw' ) and
             ( ( len( valueSplit ) == 1 ) or
             ( len( valueSplit ) == 3 and valueSplit[ 1 ] == 'asn' ) or
             ( valueSplit[ 1 ] not in ( 'aggregate', 'divide' ) ) ) )

   def removeLbwComm( self, entry ):
      '''
      Removes lbw extcomm from extcommunity community set.
      '''
      entryExtCommKeys = list( entry.extCommunity.community )
      for entryExtComm in entryExtCommKeys:
         if extCommType( entryExtComm ) == \
               extCommTypeValueMap[ 'lbw' ]:
            del entry.extCommunity.community[ entryExtComm ]
            break

   def getLbwDeleteOptions( self, comm ):
      '''
      Returns possible value of lbwDeleteOptions based on the
      configuration.
      '''
      valueSplit = comm.split()
      lbwDeleteOptions = ExtCommLinkBandwidthSetDeleteType.\
            extCommLinkBandwidthSetDeleteDefault
      if len( valueSplit ) == 1:
         # could be 'lbw delete'
         lbwDeleteOptions = ExtCommLinkBandwidthSetDeleteType.\
               extCommLinkBandwidthSetDeleteAny
      elif valueSplit[ 1 ] == 'asn':
         # could be 'lbw asn <asn> delete'
         lbwDeleteOptions = ExtCommLinkBandwidthSetDeleteType.\
               extCommLinkBandwidthSetDeleteAsn
      elif valueSplit[ 1 ] not in ( 'aggregate', 'divide' ):
         # At this point, we are just weighing on a possibility that
         # it could be 'lbw <asn>:<val> delete'. However, we don't
         # know this for sure because we may not have seen 'delete'
         # token yet (it could also be 'lbw <val>').
         # We still set lbwDeleteOptions to
         # extCommLinkBandwidthSetDeleteAsnLbw which is later used
         # to set extCommLinkBandwidthSetDeleteFlag in case we
         # encounter 'delete' token at which point we would be
         # sure that it is 'lbw <asn>:<val> delete' config.
         lbwDeleteOptions = ExtCommLinkBandwidthSetDeleteType.\
               extCommLinkBandwidthSetDeleteAsnLbw
      return lbwDeleteOptions

   def setLbwDelActionValue( self, entry, lbwDeleteType, lbwDeleteAsnDisplay ):
      '''
      Handles "set extcommunity lbw [asn] [ <asn>[:<val>] ] delete" case.
      '''
      entry.extCommLinkBandwidthSetDeleteFlag = lbwDeleteType
      entry.linkBandwidthDeleteAsnDisplay = lbwDeleteAsnDisplay
      # We reset lbwDisplay if lbwDeleteOptions flag is not
      # extCommLinkBandwidthSetDeleteAsnLbw i.e. the configuration
      # is not 'set extcommunity lbw <asn>:<val>'. Otherwise,
      # in case of 'set extcommunity lbw <asn>:<val>' config,
      # we'll need to retain lbwDisplay as it would carry
      # lbw value that will be shown in cli-save output
      if lbwDeleteType != ExtCommLinkBandwidthSetDeleteType.\
            extCommLinkBandwidthSetDeleteAsnLbw:
         entry.lbwDisplay = ""

   def setLbwRegenerationActionValue( self, entry, comm ):
      '''
      Handles lbw regeneration configuration.
      '''
      valueSplit = comm.split()
      if valueSplit[ 1 ] == 'aggregate':
         if len( valueSplit ) == 2:
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetAggregate
         elif len( valueSplit ) == 3:
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetAggregateRef
            entry.linkBandwidthRefDisplay = \
                valueSplit[ 2 ]
            entry.linkBandwidthRefExtComm = \
                getLbwCommValue( valueSplit[ 2 ] )
      elif valueSplit[ 1 ] == 'divide':
         if valueSplit[ 2 ] == 'equal':
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetDivideEqual
         elif valueSplit[ 2 ] == 'ratio':
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetDivideRatio

   def setActionValueExtComm( self, no, entry, value ):
      '''
      Handles '[no] set extcommunity ...' configuration commands.
      '''
      if no:
         # We use the following lbwDelete.* variables if while looping over
         # the tokens, we *think* if they could be one of the lbw delete cases
         lbwDeleteAnyLikely = False
         lbwDeleteAsnLikely = False
         lbwDeleteAsnDisplay = ""
         lbwDeleteAsnLbwLikely = False
         # In this case, we only want to unset the *specified* values, rather than
         # the entire extCommunity attribute.
         toDelete = value
         # For example, `no set extcommunity rt 1:1 soo 2:2 additive' would result
         # in `toDelete' being ['rt 1:1', 'soo 2:2', 'additive']. So, for each
         # item in `toDelete', we specifically unset that item (if it is set).
         for comm in toDelete:
            if comm in ( 'none', 'additive', 'delete' ):
               if self.currentFlagEqualsSetFlag( entry, comm, 'extCommunity' ):
                  entry.extCommunity.flag = 'commSetDefault'
               # If one of the lbwDelete.* is set and we encounter a 'delete'
               # token, we now definitely know that it is one of the lbw delete
               # cases. We thus need to set
               # entry.extCommLikBandwidthSetDeleteFlag accordingly
               lbwDeleteCaseLikely = lbwDeleteAnyLikely or lbwDeleteAsnLikely or \
                     lbwDeleteAsnLbwLikely
               if comm == 'delete' and lbwDeleteCaseLikely:
                  # E.g. definitely one of 'lbw delete' or 'lbw asn 10 delete' or
                  # 'lbw 10:100G delete'
                  self.resetLbwDelete( entry, lbwDeleteAnyLikely,
                        lbwDeleteAsnLikely, lbwDeleteAsnLbwLikely )
               continue

            cval = getExtCommValue( comm )
            if extCommType( cval ) != extCommTypeValueMap[ 'lbw' ]:
               # E.g. 'rt 1:1', or 'soo 2:2'.
               del entry.extCommunity.community[ cval ]
               # We can just continue in the loop because no special
               # setting of any flags is needed unlike with lbw
               continue

            # At this point, we know that it is lbw config since we continue
            # the loop otherwise as seen just above
            if self.isNotLbwRegenerationConfig( comm ):
               valueSplit = comm.split()
               if len( valueSplit ) == 1:
                  # E.g. could be 'lbw delete'
                  lbwDeleteAnyLikely = True
               elif len( valueSplit ) == 3 and valueSplit[ 1 ] == 'asn':
                  # E.g. could be 'lbw asn 10 delete'
                  lbwDeleteAsnLikely = True
                  lbwDeleteAsnDisplay = valueSplit[ 2 ]
               else:
                  # E.g. could 'lbw 1G', or 'lbw 2:2G delete'
                  lbwDeleteAsnLbwLikely = True
               entry.lbwDisplay = ""
               del entry.extCommunity.community[ cval ]
               # Since its not lbw regeneration config, we can continue in the
               # loop as either it lbw delete case for which resetting the flags
               # is handled whenever we encounter 'delete' token or its lbw
               # setter/default case (e.g. set extcommunity lbw 100G) for which
               # we don't need to play with any special flags
               continue

            # The remaining possibilites are certain lbw values that aren't
            # stored in `entry.extCommunity.community', but rather in custom
            # fields of `entry'
            self.resetLbwRegeneration( entry, comm )
         if ( entry.extCommunity.community or entry.linkBandwidthRefDisplay or
              entry.extCommunity.flag != 'commSetDefault' or
              entry.linkBandwidthDeleteAsnDisplay ):
            # Some settings remain, so don't clean up completely.
            return False
      else:
         if entry.extCommunity is None:
            entry.extCommunity = ( 'extCommunity', )
         if entry.extCommunity.flag == 'commSetNone':
            flag = 'commSetDefault'
         else:
            flag = entry.extCommunity.flag
         commValues = []
         linkBwSeen = False
         lbwDeleteOptions = ExtCommLinkBandwidthSetDeleteType.\
               extCommLinkBandwidthSetDeleteDefault
         lbwDeleteAsnDisplay = ""
         setLbwOptionsDefault = False
         setLbwDeleteOptionsDefault = False
         for comm in value:
            if comm in ( 'none', 'additive', 'delete' ):
               if comm == 'none': # pylint: disable=no-else-break
                  flag = 'commSetNone'
                  break
               elif comm == 'delete':
                  # if we have a 'set delete' clause with non-lbw sub-type
                  # (e.g. set extcommunity rt 3:4 delete) and saved config
                  # flag is default, we delete lbw clause from the saved config
                  # to avoid bad syntax such as 'set extcommunity lbw 100G delete').
                  # The non-lbw sub-types in saved config will simply acquire delete
                  # flag.
                  # Note that if the saved config flag is additive, then irrespective
                  # of whether the current configured sub-type is lbw or not,
                  # all the saved clauses would get dropped. This happens later in
                  # code after this for loop ends
                  # Complete example:
                  #  set extcommunity lbw 100G
                  #  set extcommunity soo 1:2
                  #  followed by
                  #  set extcommunity rt 2:3 delete
                  #  results in
                  #  set extcommunity soo 1:2 rt 2:3 delete
                  if commValues and extCommType( commValues[ -1 ] ) != \
                        extCommTypeValueMap[ 'lbw' ] and flag == 'commSetDefault':
                     self.removeLbwComm( entry )
                  # If lbwDeleteOptions has been changed from default, and we have
                  # 'delete' token, that means we are dealing with one of the lbw
                  # delete cases. In this case, we set the
                  # entry.extCommLinkBandwidthSetDeleteFlag
                  elif lbwDeleteOptions != ExtCommLinkBandwidthSetDeleteType.\
                        extCommLinkBandwidthSetDeleteDefault:
                     self.setLbwDelActionValue( entry, lbwDeleteOptions,
                                                lbwDeleteAsnDisplay )
                     # Make sure to unset this so that we don't later reset
                     # the extCommLinkBandwidthSetDeleteFlag
                     setLbwDeleteOptionsDefault = False

                     # Note that transitioning from a link-bw regeneration
                     # config to lbw delete case (e.g. set extcommunity
                     # lbw aggregate followed by set extcommunity lbw delete)
                     # will get rid of the saved link-bw regeneration config
                     # and hence we should make sure that
                     # extCommLinkBandwidthSetFlag is set to default
                     setLbwOptionsDefault = True
               flag = 'commSet%s' % comm.capitalize()
            else:
               cval = getExtCommValue( comm )
               commValues.append( cval )
               if extCommType( cval ) == extCommTypeValueMap[ 'lbw' ]:
                  linkBwSeen = True
                  valueSplit = comm.split()
                  # if we have a 'set' clause and it is not a link-bw
                  # regeneration config (e.g. set extcommunity lbw 100G)
                  # and saved config flag is delete, we get rid of saved configs.
                  # This is to treat 'set extcomm lbw 100G' has implicit additive
                  # otherwise if we treat it like default, we would end up getting
                  # wrong 'set extcommunity lbw 100G delete' in saved config.
                  # Note that if it were a link-bw regeneration config, we
                  # shouldn't have to delete the existing saved configs since
                  # link-bw regeneration configs can co-exist with other extcomm
                  # sub-type configs except lbw config which we delete if present
                  if value[ -1 ] == comm and flag == 'commSetDelete':
                     # Complete example:
                     #  set extcommunity soo 1:2 delete
                     #  followed by
                     #  set extcommunity lbw 100G
                     #  results in
                     #  set extcommunity lbw 100G
                     if valueSplit[ 1 ] not in ( 'aggregate', 'divide' ):
                        entry.extCommunity.community.clear()
                     # Complete example:
                     #  set extcommunity soo 1:2 delete
                     #  set extcommunity lbw 10:100G delete
                     #  followed by
                     #  set extcommunity lbw aggregate
                     #  results in
                     #  set extcommunity soo 1:2 delete
                     #  set extcommunity lbw aggregate
                     else:
                        self.removeLbwComm( entry )
                     flag = 'commSetDefault'
                  lbwDeleteOptions = self.getLbwDeleteOptions( comm )
                  if lbwDeleteOptions == ExtCommLinkBandwidthSetDeleteType.\
                        extCommLinkBandwidthSetDeleteAsn:
                     lbwDeleteAsnDisplay = valueSplit[ 2 ]

                  # if lbwDeleteOptions wasn't set in getLbwDeleteOptions to
                  # non-default or was set to extCommLinkBandwidthSetDeleteAsnLbw,
                  # that means we likely have regeneration config if we have
                  # aggregate/divide in configuration or it could be lbw-value
                  # that has an optional ASN
                  if lbwDeleteOptions == ExtCommLinkBandwidthSetDeleteType.\
                        extCommLinkBandwidthSetDeleteDefault or \
                     lbwDeleteOptions == ExtCommLinkBandwidthSetDeleteType.\
                        extCommLinkBandwidthSetDeleteAsnLbw:
                     if valueSplit[ 1 ] in ( 'aggregate', 'divide' ):
                        # `comm' is (the string representation of) an lbw-value
                        # stored in special fields directly on the `entry'. For
                        # example, 'lbw aggregate 1G', or 'lbw divide ratio'.
                        commValues.remove( cval )
                        self.setLbwRegenerationActionValue( entry, comm )
                     else:
                        # `comm' is an lbw-value that has a bandwidth and an optional
                        # ASN, e.g. 'lbw 1G', or 'lbw 2:2G'.
                        setLbwOptionsDefault = True
                        entry.lbwDisplay = \
                            getExtCommLbwDisplayValue( comm )

                        # Note that if we have 'lbw 2:2G', we would already have
                        # lbwDeleteOptions flag appropriately set above in
                        # getLbwDeleteOptions.
                        # Since we don't know yet if it is delete clause or not,
                        # we set this to make sure extCommLinkBandwidthSetDeleteFlag
                        # is reset. In case we encounter 'delete' token later, we'll
                        # unset this.
                        setLbwDeleteOptionsDefault = True
         if self.currentFlagNoneOrNotEqualsSetFlag( entry, flag, 'extCommunity' ):
            entry.extCommunity.community.clear()
         if entry.extCommunity.useCommunityList:
            entry.extCommunity.communityListNameSet.clear()
            entry.extCommunity.useCommunityList = False
         if linkBwSeen:
            # There can be only one link bandwidth extended community
            for _v in entry.extCommunity.community:
               if extCommType( _v ) == extCommTypeValueMap[ 'lbw' ]:
                  del entry.extCommunity.community[ _v ]
         if setLbwOptionsDefault:
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetDefault
            if setLbwDeleteOptionsDefault:
               entry.extCommLinkBandwidthSetDeleteFlag = \
                   ExtCommLinkBandwidthSetDeleteType.\
                   extCommLinkBandwidthSetDeleteDefault
               entry.linkBandwidthDeleteAsnDisplay = ""
            entry.linkBandwidthRefDisplay = ""
            entry.linkBandwidthRefExtComm = ufMin
         for value in commValues: # pylint: disable=redefined-argument-from-local
            entry.extCommunity.community[ value ] = True
         entry.extCommunity.flag = flag
      return True

   def setActionValueMetric( self, entry, value ):
      setattr( entry, "metric", value )

   def setActionValueTag( self, entry, value ):
      if value == 'auto':
         if getEffectiveProtocolModel( self ) == ProtoAgentModel.ribd:
            self.addWarning( "command `set tag auto` only supported in "
                             "multi-agent mode" )
         else:
            # auto tag is represented max MaybeU63 value
            # from Ark/MaybeTypes.tac
            value = 0x7fffffffffffffff
            setattr( entry, "tag", value )
      else:
         setattr( entry, "tag", value )

   def setActionValueAigpMetric( self, entry, value ):
      setattr( entry, "aigpMetric", getTacAigpMetricFromCliStr( value ) )

   def setActionValueAsPathPrepend( self, entry, value ):
      entry.asPathPrepend.clear()
      for index, v in enumerate( value ):
         # If the value to be prepended is auto, set the value to 0
         if v == 'auto':
            v = 0
         elif v == 'repeat':
            entry.asPathPrependRepeat = int( value[ index + 1 ] )
            return
         entry.asPathPrepend[ int( index ) ] = int( v )
      entry.asPathPrependRepeat = 1

   def setActionValueAsPathReplace( self, entry, value ):
      entry.asPathReplace.clear()
      if 'none' in value:
         entry.asPathReplace[ 0 ] = AS_PATH_MAX
         return
      for index, v in enumerate( value ):
         # If the value to be replaced is auto, set the value to 0
         if v == 'auto':
            v = 0
         elif v == 'repeat':
            entry.asPathReplaceRepeat = int( value[ index + 1 ] )
            return
         entry.asPathReplace[ index ] = int( v )
      entry.asPathReplaceRepeat = 1

   def setActionValueSetNextHop( self, attr, entry, value ):
      SetFlags = Tac.Type( "Routing::RouteMap::MapEntry::SetFlags" )
      # at most one of "has*NextHop*" flags should be set at any given time
      entry.setFlags = SetFlags( entry.setFlags.value
                                 # pylint: disable-next=invalid-unary-operand-type
                                 & ~SetNextHopArgTransformer.flagsMask
                                 | SetActionInfo[ attr ][ 'flag' ] )

      entry.setNextHopArg = SetNextHopArgTransformer.tacSetNextHopArg(
         attr, value )

   def setActionValueIgpMetric( self, entry, value ):
      setattr( entry, "igpMetric", U32_MAX_VALUE if value == "max-metric"
                                   else value )

   def noSetActionValue( self, attr, value ):
      entry = self.getCurrentRmapEntry( attr )
      info = SetActionInfo[ attr ]
      assert info[ 'enabled' ]

      SetFlags = Tac.Type( "Routing::RouteMap::MapEntry::SetFlags" )
      if ( attr == 'largeCommunity' ) and entry.largeCommunity and value:
         # if return value is false, we do not want complete cleanup
         # and so we return
         if not self.setActionValueLargeComm( True, entry, value ):
            return

      if ( attr == 'extCommunity' ) and entry.extCommunity and value:
         # if return value is false, we do not want complete cleanup
         # and so we return
         if not self.setActionValueExtComm( True, entry, value ):
            return

      # From here, we unset the entire attribute, ignoring any specified values.
      entry.setFlags = SetFlags( entry.setFlags.value & ~info[ 'flag' ] )
      if attr == 'asPathPrepend':
         entry.asPathPrepend.clear()
         entry.asPathPrependRepeat = 1
      elif attr == 'asPathReplace':
         entry.asPathReplace.clear()
         entry.asPathReplaceRepeat = 1
      elif attr in SetNextHopArgTransformer.tacAttrToArgTag:
         if ( entry.setNextHopArg.tag ==
              SetNextHopArgTransformer.tacAttrToArgTag[ attr ] ):
            entry.setNextHopArg = SetNextHopArgTransformer.tacSetNextHopArgNone()
      else:
         if attr == 'extCommunity':
            entry.extCommLinkBandwidthSetFlag = \
                ExtCommLinkBandwidthSetType.\
                extCommLinkBandwidthSetDefault
            entry.extCommLinkBandwidthSetDeleteFlag = \
                ExtCommLinkBandwidthSetDeleteType.\
                extCommLinkBandwidthSetDeleteDefault
            entry.linkBandwidthRefDisplay = ""
            entry.linkBandwidthDeleteAsnDisplay = ""
            entry.linkBandwidthRefExtComm = ufMin
         setattr( entry, attr, info[ 'default' ] )

   def setActionValue( self, attr, value ):
      entry = self.getCurrentRmapEntry( attr )
      info = SetActionInfo[ attr ]
      assert info[ 'enabled' ]

      # Warning message if used in gated
      if getEffectiveProtocolModel( self ) == ProtoAgentModel.ribd:
         if attr == 'originAsValidity':
            self.addWarning( "command `set origin-as` only supported in "
                             "multi-agent mode" )
         elif attr == 'aigpMetric':
            self.addWarning( "command `set aigp-metric` only supported in "
                             "multi-agent mode" )
         elif attr == 'resolutionRibProfileConfig':
            self.addWarning( "command `set resolution ribs` only supported in "
                             "multi-agent mode" )

      SetFlags = Tac.Type( "Routing::RouteMap::MapEntry::SetFlags" )
      if attr == 'continueSeq':
         if value is None:
            value = 0
         elif value <= entry.seqno:
            self.addError( "Loop in the routemap" )
            return

      entry.setFlags = SetFlags( entry.setFlags.value | info[ 'flag' ] )

      if attr == "metric":
         self.setActionValueMetric( entry, value )
      elif attr == 'tag':
         self.setActionValueTag( entry, value )
      elif attr == "aigpMetric":
         self.setActionValueAigpMetric( entry, value )
      elif attr == 'asPathPrepend':
         self.setActionValueAsPathPrepend( entry, value )
      elif attr == 'asPathReplace':
         self.setActionValueAsPathReplace( entry, value )
      elif attr == 'largeCommunity':
         self.setActionValueLargeComm( False, entry, value )
      elif attr == 'extCommunity':
         self.setActionValueExtComm( False, entry, value )
      elif attr in SetNextHopArgTransformer.tacAttrToArgTag:
         self.setActionValueSetNextHop( attr, entry, value )
      elif attr == "igpMetric":
         self.setActionValueIgpMetric( entry, value )
      else:
         setattr( entry, attr, value )
#------------------------------------------------------------------------------------
# standard communities CLI
#
# The following code concerns the manipulation of the standard communities TAC model
# by the communities CLI. The representation of standard communities in a routeMap
# entry is handled by the setFlags field in an entry and the 2 communitySet objects
# communityAddReplace and communityDelete which have the following structure:
#   community:            dictionary holding community values (if any)
#   communityListNameSet: dictionary holding community lists (if any)
#   useCommunityFlag:     boolean indicating which dictionary is being used (both
#                         can't be active in a communitySet at the same time)
#   flag:                 string indicating what action the communties will perform
#
# A few notes on the communities system:
#   1. setFlags can be thought of as a binary number where bits are flipped on if the
#      communitySet object to which they correspond exists ( i.e is not none ).
#   2. CommunityAddReplace can have 3 different flags. "commSetAdditive" adds the
#      set's community/communityList values to a matching route. commSetReplace
#      replaces all community / communityList values in a matching route with the
#      ones in the set. commSetNone indicates that nothing is to be done to routes.
#      The last 2 flags also mean that communityDelete must be none
#   3. communityDelete can only have one flag, "commSetDelete" which indicates that
#      any community / communityList found in a matching route should be removed
#      before being sent forward. CommunityDelete can exist (i.e. not be none) if
#      communityAddReplace is None or has its flag set to commSetAdditive.
# The purpose of the functions below is to handle the various interactions between
# the various parts of this system; changing setFlags, useCommunityFlag, or flag,
# adding / removing elements from the communityDictionaries, and creating / removing
# communitySets as needed.
#------------------------------------------------------------------------------------

   def createCommunitySet( self, communitySetName ):
      """
      Handles the creation of community sets as well as setting the setFlags field
      Inputs:
          communitySetName: Name of the communitySet to be created
      """
      entry = self.getEntry()

      if getattr( entry, communitySetName ) is None:
         setattr( entry, communitySetName, ( communitySetName, ) )

         # Using bitwise or, change setFlags to reflect that there is a community set
         SetFlags = Tac.Type( "Routing::RouteMap::MapEntry::SetFlags" )
         entry.setFlags = SetFlags( entry.setFlags.value |
                                 SetActionInfo[ communitySetName ][ "flag" ] )

   def removeCommunitySet( self, communitySetName ):
      """
      Handles removing a communitySet and switiching off the relevant setFlags
      Inputs:
         communitySetName: Name of the communitySet to be removed
      """
      entry = self.getEntry()

      if getattr( entry, communitySetName ):

         # By bitwise and, change the flags to reflect that there is no community set
         SetFlags = Tac.Type( "Routing::RouteMap::MapEntry::SetFlags" )
         entry.setFlags = SetFlags( entry.setFlags.value &
                                 ~SetActionInfo[ communitySetName ][ "flag" ] )

         # Remove the communitySet
         setattr( entry, communitySetName, None )

   def addCommunityFilterType( self, communityFilterType ):
      commType = Tac.Value( 'Acl::CommunityFilterType' )
      for communityType in communityFilterType:
         if communityType == "all":
            commType.allCommTypes = True
         elif communityType == "rt":
            commType.routeTarget = True
         elif communityType == "soo":
            commType.siteOfOrigin = True
         elif communityType == "lbw":
            commType.linkBandwidth = True
         elif communityType == "color":
            commType.color = True
      return commType

   def alterModeCommunityDict( self, communitySetName, communityDictName,
                               setOperation, modifierSet, communityFlag=None,
                               communityFilterType=None, removeEmpty=False ):
      """
      Alters the community dictionaries for community sets in routeMap entries.
      Inputs:
       communitySetName:  Name of the communitySet ( either communityAddReplace or
                          communityDelete ) to be changed.
       communityDictName: Name of the dictionary in the communitySet to be changed
                          ( either community or communityListNameSet ).
       setOperation:      Function with 2 sets as input ( usually union, difference,
                          or intersection ) that will combine the current and new
                          community values.
       modifierSet:       The communitys that are to be add / removed from the entry.
       communityFlag:     New value of community set's flag field (will be either
                          commSet{Default, Additive, Delete, None} or None if the
                          flag is not to be changed).
       removeEmpty:       Boolean indicating if caller wants the communitySet set
                          to None if it holds no communities after operating on it.
      """
      entry = self.getEntry()
      communitySetObject = getattr( entry, communitySetName )

      # We may be operating on a communitySet that doesn't yet exist, so we will
      # need to create it if it hasn't been (it may get deleted at the end of this
      # method if empty and removeEmpty ). We will track the initial state of the
      # communitySet for the purposes of flag setting
      if communitySetObject is None:
         self.createCommunitySet( communitySetName )
      communitySetObject = getattr( entry, communitySetName )
      communityDict = getattr( communitySetObject, communityDictName )

      # Converting the communityDict into a set enables easier addition/deletion of
      # the current communities and those input by the user.
      # pylint: disable-next=unnecessary-comprehension
      communitySet = { key for key in communityDict }

      # Perform the given set operation and convert the result into the dictionary
      # format used by the communitySet TAC object for community/communityListNameSet
      newCommunitySet = setOperation( communitySet, modifierSet )
      newCommunityDict = { key : True for key in newCommunitySet }

      # Change the value of communityField and entry to be the new dict, unless it is
      # empty and the user specified to remove an empty communitySet (e.g it isn't
      # being used for a set none command), in which case set it to none.
      self.replaceCommunityDict( communitySetObject, communityDictName,
                                 newCommunityDict )

      if communityFilterType:
         communitySetObject.commType =\
                                 self.addCommunityFilterType( communityFilterType )
      if removeEmpty and \
         not communitySetObject.community and \
         not communitySetObject.communityListNameSet:
         self.removeCommunitySet( communitySetName )
      else:

         # We may want to change the flag of our communitySet, if the user specified
         # one to change it to.
         if communityFlag:
            communitySetObject.flag = communityFlag

   def setCommunityListFlag( self, flagOn, fieldName ):
      """
      Since we can switch between using communities and communityLists, we will need
      to set the useCommunityList flags in our communitySets from time to time.
          input:
          flagOn:    Boolean indicating the desired value of the communityList flag.
          fieldName: Name of the field being changed (community{AddReplace, Delete})
      """
      entry = self.getEntry()
      communitySet = getattr( entry, fieldName )
      if communitySet:
         setattr( getattr( entry, fieldName ), "useCommunityList", flagOn )

   def replaceCommunityDict( self, commSet, dictName, newDict ):
      """
      Replace old community/communitylist dictionaries with new ones by element wise
      replacement, because direct assignment of dictionaries to TACC bindings was not
      supported by python at the time this was written (July 2018).
          Inputs:
          commSet:  CommunitySet being manipulated (either communityAddReplace or
                    communityDelete).
          dictName: Name of the dictionary attribute of the communitySet (either
                    community or communityListNameSet).
          newDict:  The new dictionary to replace the old one.
      """
      communityDict = getattr( commSet, dictName )

      # Swap out the new elements in communitySet readable style
      communityDict.clear()
      for newKey in newDict:
         communityDict[ newKey ] = True

   # pylint: disable-next=inconsistent-return-statements
   def noCommunityActionPossible( self, action, communityType="standard" ):
      """
      Checks if a specific action is possible given current status of the routeMap
      entry (e.g. the action "no set community delete" doesn't make sense if the
      entry has no deleting communities).
         Inputs:
         action: String (or none) indictating the no action the user wants to take.
      """
      entry = self.getEntry()

      if communityType == "standard":
         addReplace = entry.communityAddReplace
         delete = entry.communityDelete
         commFilter = entry.communityFilter
      elif communityType == "extended":
         addReplace = entry.extCommunity
         delete = entry.extCommunity
         commFilter = entry.extCommunityFilter
      elif communityType == "large":
         addReplace = entry.largeCommunity
         delete = entry.largeCommunity
         commFilter = entry.largeCommunityFilter

      # If the user didn't specify an action, then we can delete a community set if
      # and only if only one is defined ( otherwise, we do nothing ).
      if not action:
         if communityType == "standard":
            return ( ( addReplace is not None ) ^ ( delete is not None ) ^
                     ( commFilter is not None ) )
         else:
            return ( ( addReplace is not None and delete is not None ) ^
                     ( commFilter is not None ) )

      # If an action was given, we want to check that the communitySet that manages
      # those communities exists and has the appropriate flags.
      if action == "additive":
         return addReplace and addReplace.flag == "commSetAdditive"
      elif action == "delete":
         return delete and delete.flag == "commSetDelete"
      elif action == "filter":
         return commFilter and commFilter.flag == "commSetFilter"
      elif action == "none":
         return addReplace and addReplace.flag == "commSetNone"

      else:
         # This error should never be issued unless an input was incorrectly given
         self.addError( "Unknown action " + str( action ) + " encountered." )

   # pylint: disable-next=inconsistent-return-statements
   def communityActionToFlag( self, action, communityType="standard" ):
      """
      Figures out the appropriate flag (commSetDefault, commSetNone, etc.) for a
      given action, as the correct one can change depending on the state of an entry.
          input:
          action: String of the action given by the user at the command line or None.
      """
      # action being None means that the user did not specify an action, so we need
      # to figure out what it is based on the entry.
      if not action:
         entry = self.getEntry()
         if communityType == "standard":
            addReplace = entry.communityAddReplace
            delete = entry.communityDelete
         elif communityType == "extended":
            addReplace = entry.extCommunity
            delete = entry.extCommunity
         elif communityType == "large":
            addReplace = entry.largeCommunity
            delete = entry.largeCommunity

         # If both the additive and delete sets are defined, then the user needs to
         # specify which set they want altered.
         if addReplace and delete and addReplace != delete:
            self.addWarning( AMBIGOUS_COMMAND_WARNING )
            return None
         elif delete and delete.flag == "commSetDelete":
            return "commSetDelete"
         elif addReplace and addReplace.flag == "commSetAdditive":
            return "commSetAdditive"

         # We can return commSetDefault in 3 cases 1. The only communities are
         # default/replace communities 2. The communitySet is in the None state and
         # the user gave no specification of the communities type they want to use
         # 3. addReplace and delete are both none.
         elif addReplace and addReplace.flag in [ "commSetDefault", "commSetNone" ]:
            return "commSetDefault"
         elif addReplace is None and delete is None:
            return "commSetDefault"

         # Assuming a well programmed CLI, one of the previous options should have
         # triggered. However if none were, then something has gone wrong and the
         # user ( or hopefully programmer ) needs to be notified.
         else:
            self.addError( "Unknown route-map configuration encountered." )

      # If the user specified an action, then we can return the flag for that action
      elif action in ( 'additive', 'delete', 'filter', 'none' ):
         return 'commSet' + action.title()
      else:
         self.addError( 'Unknown action %r encountered.' % action )
         return None

   def usingCommunityList( self, communitySet ):
      """
      Evaluates whether the current entry is using communityLists or communities.
      Input:
         communitySet: The communitySet being checked
      """
      if communitySet is None:
         return False
      else:
         return communitySet.useCommunityList

   def addAsNum( self, args, arg ):
      if 'AS_NUM' not in args:
         args[ 'AS_NUM' ] = []
      args[ 'AS_NUM' ].append( arg )

   def processAsArgs( self, args, argsList ):
      for arg in argsList:
         if 'AS_DOT_NUM' in arg:
            self.addAsNum( args, asnStrToNum( arg[ 1 ] ) )
         elif 'AS_PLAIN_NUM' in arg:
            self.addAsNum( args, arg[ 1 ] )
         elif 'auto' in arg:
            self.addAsNum( args, arg[ 1 ] )

   def getEntry( self ):
      """
      Many methods will need to get a routeMap entry, but it may not exist, so
      this method handles fetching it.
      """
      if self.routeMapContext is None:
         self.addError( "Attempted to access a non-existent routeMap." )
      return self.routeMapContext.currentEntry()

   def handleDefaultToDelete( self, communityDictName ):
      """
      If the user initially doesn't specify an action following a community/commList
      in a command (e.g. set commandity <x>), those communities are treated as a
      replacing communities. But if the users then sets delete communities in a
      routemap that only holds default/replace communities, then those communities
      will be moved over to the communityDelete communitySet as part of the execution
      of that set community <x> delete command.
          input:
          communityDictName: Name of dictionary of community attributes to be moved.
      """
      entry = self.getEntry()

      # We only want to move default communities to communityDelete if
      # communityAddReplace exists and is holding default/replace communities and not
      # additive communities
      if entry.communityAddReplace and ( not entry.communityDelete ) and \
         entry.communityAddReplace.flag == "commSetDefault":
         self.createCommunitySet( "communityDelete" )
         entry.communityDelete.flag = "commSetDelete"

         # We will want to move all the entries from the default set over to the
         # delete set and remove communityAddReplace
         communityDict = getattr( entry.communityAddReplace, communityDictName )
         self.replaceCommunityDict( entry.communityDelete, communityDictName,
                                    communityDict )
         self.removeCommunitySet( "communityAddReplace" )

   def exitNone( self ):
      """
      This function handles altering the communityAddReplace if it is none.
      """
      entry = self.getEntry()

      if entry.communityAddReplace and \
         entry.communityAddReplace.flag == "commSetNone":

         # We want to make sure the CLI sets communities in commSetNone mode properly
         if len( entry.communityAddReplace.community ) != 0 or \
            len( entry.communityAddReplace.communityListNameSet ) != 0:
            self.addError( "Attempted to exit none in a configuration with " +
                           "communities or community-lists." )
         if entry.communityDelete:
            self.addError( "Attempted to exit none in a configuration with " +
                           "delete communities or community-lists." )
         self.removeCommunitySet( "communityAddReplace" )

   def showCurrentRouteMap( self, args ):
      ctx = self.routeMapContext
      rtmap = ctx.routeMap()
      if rtmap is None:
         print( "\n" )
      else:
         printer = Printer( mapConfig )
         printer.printRouteMap( rtmap )

def getUnconfiguredCommunityLists( values, communityType ):
   """ @brief Checks whether a community-list set of values exists or not.
       @param [in] values           A list of community-list names to check.
       @param [in] communityType    The community type common to all values.
       @return a list (subset of @param 'values') of all those community-lists
               which are not configured in the current running config.
               In case all values are valid, returns an emtpy list.
   """
   def ComputeAllCommunityLists():
      """ @return all community lists defined in the current running config.
          @note the returned type is a dictonary of dictonaries.
                that allows fast lookup.
      """
      allCommLists = {}
      for communityType in CommunityType:
         allCommLists[ communityType ] = {}
         for name in aclListConfig.communityList:
            if communityListByName( aclListConfig, communityType, name ):
               allCommLists[ communityType ][ name ] = {}
      return allCommLists

   allCommLists = ComputeAllCommunityLists()
   return [ x for x in values if x not in allCommLists[ communityType ] ]

def getUnconfiguredAsPathLists( values ):
   """ @brief Checks whether an AS-Path list of values exists or not.
       @param [in] values    A list values of AS-Path names.
       @return a set (subset of @param 'values') of all those AS-Path lists
               which are not configured in the current running config.
               In case all values are valid, returns an empty set.
   """
   allAsPathLists = set( aclListConfig.pathList )
   return set( values ) - allAsPathLists

def getUnconfiguredAccessLists( values ):
   """ @brief Checks whether an access-list of values exists or not.
       @param [in] values    A list values of access-list names.
       @return a list (subset of @param 'values') of all those Access lists
               which are not configured in the current running config.
               In case all values are valid, returns an empty list.
   """
   # pylint: disable-next=unnecessary-comprehension
   allAccessLists = { name for name in aclConfig.config[ 'ip' ].acl }
   # pylint: disable-next=unnecessary-comprehension
   allAccessLists |= { name for name in aclConfig.config[ 'ipv6' ].acl }
   # pylint: disable-next=unnecessary-comprehension
   allAccessLists |= { name for name in aclConfig.config[ 'mac' ].acl }
   return set( values ) - allAccessLists

def getUnconfiguredPrefixLists( values, ipv6, dynamic ):
   """ @brief Checks whether a prefix-list of values exists or not.
       @param [in] values   A list of prefix-list names.
       @param [in] ipv6     Specify ipv6 (True or False).
       @param [in] dynamic  Specify whether or not the prefix-list is
                            dynamic.
       @return a list (subset of @param 'values') of all those Prefix lists
               which are not configured in the current running config.
   """
   prefixLists = set()
   if dynamic:
      prefixLists.update( dynPfxListConfigDir.dynamicPrefixList )
   else:
      if ipv6:
         prefixLists.update( aclListConfig.ipv6PrefixList )
      else:
         prefixLists.update( aclListConfig.prefixList )
   return set( values ) - prefixLists
matchActions = Enum( 'matchActions', 'commListSet strValue' )
rule2Actions = {
   RouteMapMatchOption.matchCommunity: matchActions.commListSet,
   RouteMapMatchOption.matchExtCommunity: matchActions.commListSet,
   RouteMapMatchOption.matchLargeCommunity: matchActions.commListSet,
   RouteMapMatchOption.matchAsPathList: matchActions.strValue,
   RouteMapMatchOption.matchAccessList: matchActions.strValue,
   RouteMapMatchOption.matchPrefixList: matchActions.strValue,
   RouteMapMatchOption.matchNextHopPrefixList: matchActions.strValue,
   RouteMapMatchOption.matchResolvedNextHopPrefixList: matchActions.strValue,
   RouteMapMatchOption.matchIpv6PrefixList: matchActions.strValue,
   RouteMapMatchOption.matchIpv6NextHopPrefixList: matchActions.strValue,
   RouteMapMatchOption.matchIpv6ResolvedNextHopPrefixList: matchActions.strValue,
   RouteMapMatchOption.matchRouterId: matchActions.strValue,
}

def routeMapsMatchingOnPolicyConstruct( identifier, matchOption ):
   """ @brief  Checks if a policy-construct is used by a `match clause`
               by any route-map configured in the running-config so far.
               Essentially, this function performs an iteration over all
               route-maps defined and checks whether the opportune
               match-entry contains the identifier or not.
       @param [in]    identifier   The identifier string of the policy
                                   construct (e.g.; communityName).
       @param [in]    matchOption  The match option in the matching rule
                                   related to the policy construct type.
       @return A set of route-maps (names, seqno) which use the
               policy-construct.
       @note   The runtime complexity is O(N) where N is the number of
               all route-map entries defined.
   """
   def aggregateMatchOption( matchOption ):
      """ @brief Some 'match options' can refer to the same policy constructs
                 category. For instance, although they are two different match
                 rules, `match next-hop prefix-list` and
                 `match resolved next-hop prefix-list` target a prefix-list
                 construct.
                 This helper function allows aggregating match rules that
                 address the same policy construct.
      """
      aggregationPrefixList = [
         RouteMapMatchOption.matchPrefixList,
         RouteMapMatchOption.matchNextHopPrefixList,
         RouteMapMatchOption.matchResolvedNextHopPrefixList,
         RouteMapMatchOption.matchRouterId ]
      aggregationPrefixListIpv6 = [
         RouteMapMatchOption.matchIpv6PrefixList,
         RouteMapMatchOption.matchIpv6NextHopPrefixList,
         RouteMapMatchOption.matchIpv6ResolvedNextHopPrefixList ]

      aggregations = [ aggregationPrefixList, aggregationPrefixListIpv6 ]
      for agg in aggregations:
         if matchOption in agg:
            return agg
      return [ matchOption ]

   routeMapsDep = set()
   for mapName, routeMap in mapConfig.routeMap.items():
      for mapSeqno, mapEntry in routeMap.mapEntry.items():
         for match in aggregateMatchOption( matchOption ):
            # For all match commands relative to that policy-construct
            action = rule2Actions[ match ]
            matchRule = mapEntry.matchRule.get( match )
            if matchRule:
               if action == matchActions.commListSet:
                  if identifier in matchRule.commListSet.commListSet:
                     routeMapsDep.add( ( mapName, mapSeqno ) )
               elif action == matchActions.strValue:
                  if matchRule.strValue == identifier:
                     routeMapsDep.add( ( mapName, mapSeqno ) )
   return routeMapsDep

def blockAccessListListDeleteIfInUse( mode, aclName, *args, **kwargs ):
   """ @brief Checks if an 'access-list' can be safely deleted without breaking
             any dependency with a route-map.
             Moreover, it prints an error message on the mode (passed as argument)
             in case of dependency.
       @return 'True' if the policy construct is used by a route-map and should
               not be deleted.
               'False' if ok to delete.
   """
   if mapConfig.routeMapPolicyReferenceUnconfiguredError:
      routeMapsDep = routeMapsMatchingOnPolicyConstruct(
         aclName, RouteMapMatchOption.matchAccessList )
      if routeMapsDep:
         mode.addError( getErrorMessageCannotDeleteBecauseUsedBy( 'access-list',
                                                                  aclName,
                                                                  routeMapsDep ) )
         raise CliParser.AlreadyHandledError

def getConfigSanityMatchRules( entry, allCommLists, allPfxLists, allAsPathLists,
                               allTagSets, allRouteMaps, routeCtx=True ):
   matchRule = entry.matchRule
   matchRules = MatchRules()
   matchRules.debugInfo = DebugInfo()
   populatedFields = []
   matchCommands = []
   sortedMatchOptions = list( matchRule )
   sortedMatchOptions.sort( key=lambda op: MatchAttributes[ op ][ 'enum' ] )
   for op in sortedMatchOptions:
      # BUG971584: override matchTag with matchTagSet
      if op == 'matchTag' and entry.matchTagSetName:
         continue
      capiName = matchRuleCapiName( op )
      if capiName is None:
         continue
      attr = MatchAttributes[ op ]
      rule = matchRule[ op ]
      value = None
      if attr[ 'type' ] == 'string':
         if rule.option == 'matchContributorAggregateAttributes':
            if rule.strValue:
               value = AggregateContributor(
                           aggregateRole=matchContribAggAttrContribStr,
                           aggregateAttributes=rule.strValue )
            else:
               value = AggregateContributor(
                           aggregateRole=matchContribAggAttrContribStr )
         else:
            value = rule.strValue
         # pylint: disable-next=no-else-continue
         if capiName == 'prefixList' and value in allPfxLists[ 'v4' ]:
            continue
         elif capiName == 'accessList' and value in allPfxLists[ 'v4access' ]:
            continue
         elif capiName == 'ipAddrDynamicPrefixList' and value in allPfxLists[ 'v4' ]:
            continue
         elif capiName == 'ipv6PrefixList' and value in allPfxLists[ 'v6' ]:
            continue
         elif capiName == 'ipv6AddrDynamicPrefixList' and \
              value in allPfxLists[ 'v6' ]:
            continue
         elif capiName == 'asPathList' and value in allAsPathLists:
            continue
         elif capiName == 'aggregation' and \
              value.aggregateAttributes in allRouteMaps:
            continue
         elif capiName == 'routerIdPrefixList' and value in allPfxLists[ 'v4' ]:
            continue

      elif attr[ 'type' ] == 'set':
         commListSet = rule.commListSet.commListSet
         undefinedCommList = [ name for name in commListSet if
                               name not in allCommLists[ capiName ] ]
         if undefinedCommList:
            undefinedCommunity = AttributeDebug()
            undefinedCommunity.undefined = ' '.join( undefinedCommList )
            setattr( matchRules.debugInfo, capiName, undefinedCommunity )
            value = " ".join( commListSet )
      # Skip disabled entries.  (New check for route maps.)
      if not attr[ 'enabled' if routeCtx else 'peerfilter' ]:
         continue

      setattr( matchRules, capiName, value )
      populatedFields.append( capiName )
      command = attr[ 'command' ]
      if isinstance( command, MatchCmdRenderer ):
         matchCommands.append( command.render( rule ) )
      else:
         matchCommands.append( " ".join( command ) )

   # BUG971584: inject matchTagSet
   if entry.matchTagSetName:
      tagSet = allTagSets.get( entry.matchTagSetName )
      if not tagSet or len( tagSet.tagValue ) == 0:
         capiName = 'routeTagSet'
         command = [ 'tag', 'set' ]
         value = entry.matchTagSetName

         setattr( matchRules, capiName, value )
         populatedFields.append( capiName )
         matchCommands.append( " ".join( command ) )

   matchRules._populatedFields = populatedFields
   matchRules._matchCommands = matchCommands
   return matchRules

def getMatchRules( entry, routeCtx=True ):
   matchRule = entry.matchRule
   populatedFields = []
   matchCommands = []
   matchRules = MatchRules() if routeCtx else PeerFilterMatchRules()
   asdotConfigured = isAsdotConfigured( asnConfig )
   sortedMatchOptions = list( matchRule )
   sortedMatchOptions.sort( key=lambda op: MatchAttributes[ op ][ 'enum' ] )
   for op in sortedMatchOptions:
      rule = matchRule[ op ]
      attr = MatchAttributes[ op ]
      capiName = matchRuleCapiName( op )
      # BUG971584: override matchTag with matchTagSet
      if op == 'matchTag' and entry.matchTagSetName:
         continue
      # Skip disabled entries.  (New check for route maps.)
      if not attr[ 'enabled' if routeCtx else 'peerfilter' ]:
         continue
      if capiName is None:
         continue
      populatedFields.append( capiName )
      command = attr[ 'command' ]
      if isinstance( command, MatchCmdRenderer ):
         matchCommands.append( command.render( rule ) )
      else:
         matchCommands.append( " ".join( command ) )

      value = None
      if attr[ 'type' ] == 'string':
         if rule.option == 'matchContributorAggregateAttributes':
            if rule.strValue:
               value = AggregateContributor(
                           aggregateRole=matchContribAggAttrContribStr,
                           aggregateAttributes=rule.strValue )
            else:
               value = AggregateContributor(
                           aggregateRole=matchContribAggAttrContribStr )
         else:
            value = rule.strValue
      elif attr[ 'type' ] == 'int':
         if op == 'matchPeerAs' or op == 'matchOriginAs':
            value = bgpFormatAsn( rule.intValue, asdotConfigured )
         else:
            value = rule.intValue
      elif attr[ 'type' ] == 'ipaddr':
         value = str( rule.ipAddrValue )
      elif attr[ 'type' ] == 'ip6addr':
         value = rule.ip6AddrValue.stringValue
      elif attr[ 'type' ] == 'enum':
         # This API is expected to return the capi form of the command.
         # If this matchAttr has a separate dictionary of capiValues use
         # that, otherwise use the common dictionary of values.
         value = attr.get( 'capiValues', attr[ 'values' ] )[ rule.intValue ]
      elif attr [ 'type' ] == 'range':
         if rule.option == 'matchCommunityInstances':
            value = CommInstancesRange( start=rule.intValue,
                                        end=rule.int2Value )
         else:
            value = AsnRange( start=rule.intValue, end=rule.int2Value )
            matchRules._isAsdotConfigured = asdotConfigured
      elif attr[ 'type' ] == 'u32range':
         value = AsPathLengthRange( start=rule.u32Range.low,
                                    end=rule.u32Range.high )
      elif attr[ 'type' ] == 'set':
         value = " ".join( rule.commListSet.commListSet )
      setattr( matchRules, capiName, value )
      if rule.exact:
         matchRules.communityExactMatch = True
      if rule.orResults:
         matchRules.communityOrResults = True
      if rule.invert:
         if rule.option == 'matchPrefixList':
            matchRules.prefixListInvertResult = True
         elif rule.option == 'matchIpv6PrefixList':
            matchRules.ipv6PrefixListInvertResult = True
         elif rule.option == 'matchCommunity':
            matchRules.communityInvertResult = True
         elif rule.option == 'matchExtCommunity':
            matchRules.extCommunityInvertResult = True
         elif rule.option == 'matchLargeCommunity':
            matchRules.largeCommunityInvertResult = True
         elif rule.option == 'matchCommunityInstances':
            matchRules.communityInstancesInvertResult = True
         elif rule.option == 'matchAsPathLength':
            matchRules.asPathLengthInvertResult = True
         elif rule.option == matchContribAggAttrOption:
            matchRules.aggregationInvertResult = True
         elif rule.option == 'matchOriginAsValidity':
            matchRules.originAsValidityInvertResult = True
         else:
            # Unexepected rule.option!!!
            # You MUST implement new matchRules.<variable>
            # for this unmatched rule.option!
            assert False

   # BUG971584: inject matchTagSet
   if entry.matchTagSetName:
      command = [ 'tag', 'set' ]
      value = entry.matchTagSetName
      capiName = 'routeTagSet'

      matchCommands.append( " ".join( command ) )
      populatedFields.append( capiName )
      setattr( matchRules, capiName, value )

   matchRules._populatedFields = populatedFields
   matchRules._matchCommands = matchCommands
   return matchRules

def getCommTypeModelList( commType ):
   communityFilterType = []
   if commType.allCommTypes:
      communityFilterType.append( 'all' )
   if commType.routeTarget:
      communityFilterType.append( 'rt' )
   if commType.siteOfOrigin:
      communityFilterType.append( 'soo' )
   if commType.linkBandwidth:
      communityFilterType.append( 'lbw' )
   if commType.color:
      communityFilterType.append( 'color' )
   return communityFilterType

# this function sets all types of community lists
def getConfigSanitySetRules( entry, allCommLists, allPfxLists, allAsPathLists ):
   populatedFields = []
   setCommands = []
   setRules = SetRules()
   setRules.debugInfo = DebugInfo()
   for attr in SetActionInfo: # pylint: disable=consider-using-dict-items
      if not entry.setFlags.value & SetActionInfo[ attr ][ 'flag' ]:
         continue
      capiName = setRuleCapiName( attr )
      if capiName is None:
         continue
      if attr in ( 'communityAddReplace', 'communityDelete', 'communityFilter',
                   'extCommunity', 'extCommunityFilter', 'largeCommunity',
                   'largeCommunityFilter' ):
         populatedFields.append( capiName )
         setCommands.append( " ".join( SetActionInfo[ attr ][ 'command' ] ) )
         value = getattr( entry, attr )
         op = 'default'
         communityLists = None
         communityFilterType = None
         if value is not None:
            if value.flag != 'none':
               if value.useCommunityList:
                  communityLists = []
                  for name in value.communityListNameSet:
                     communityLists.append( name )
            if value.flag == 'commSetAdditive':
               op = 'add'
            elif value.flag == 'commSetDelete':
               op = 'delete'
            elif value.flag == 'commSetFilter':
               op = 'filter'
               communityFilterType = getCommTypeModelList( value.commType )
            elif value.flag == 'commSetNone':
               op = 'none'

         value = CommunitySet( operationType=op,
                               communityLists=communityLists,
                               communityValues=None,
                               communityFilterTypes=communityFilterType )
         setattr( setRules, capiName, value )
         ignored = None
         if attr in ( 'communityAddReplace', 'communityDelete', 'communityFilter' ):
            if capiName == 'communityAddOrReplace' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'community' ] )
               ignored = " ".join(
                    name for name in communityLists if
                    name in allCommLists[ 'community' ] and
                    'hasRegexpEntries' in allCommLists[ 'community' ][ name ] )
               if undefined or ignored:
                  setRules.debugInfo.communityAddOrReplace = AttributeDebug()
                  setRules.debugInfo.communityAddOrReplace.undefined = \
                                                                  undefined or None
                  setRules.debugInfo.communityAddOrReplace.ignored = ignored or None
            elif capiName == 'communityDelete' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'community' ] )
               if undefined:
                  setRules.debugInfo.communityDelete = AttributeDebug()
                  setRules.debugInfo.communityDelete.undefined = undefined
                  setRules.debugInfo.communityDelete.ignored = None
            elif capiName == 'communityFilter' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'community' ] )
               if undefined:
                  setRules.debugInfo.communityFilter = AttributeDebug()
                  setRules.debugInfo.communityFilter.undefined = undefined
         elif attr in ( 'extCommunity', 'extCommunityFilter' ):
            if capiName == 'extCommunity' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'extCommunity' ] )
               if op in ( 'add', 'default' ):
                  ignored = " ".join(
                       name for name in communityLists if
                       name in allCommLists[ 'extCommunity' ] and
                       'hasRegexpEntries' in allCommLists[ 'extCommunity' ][ name ] )
               if undefined or ignored:
                  setRules.debugInfo.extCommunity = AttributeDebug()
                  setRules.debugInfo.extCommunity.undefined = undefined or None
                  setRules.debugInfo.extCommunity.ignored = ignored or None
            elif capiName == 'extCommunityFilter' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'extCommunity' ] )
               if undefined:
                  setRules.debugInfo.extCommunityFilter = AttributeDebug()
                  setRules.debugInfo.extCommunityFilter.undefined = undefined
         elif attr in ( 'largeCommunity', 'largeCommunityFilter' ):
            if capiName == 'largeCommunity' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'largeCommunity' ] )
               if op in ( 'add', 'default' ):
                  ignored = " ".join(
                    name for name in communityLists if
                    name in allCommLists[ 'largeCommunity' ] and
                    'hasRegexpEntries' in allCommLists[ 'largeCommunity' ][ name ] )
               if undefined or ignored:
                  setRules.debugInfo.largeCommunity = AttributeDebug()
                  setRules.debugInfo.largeCommunity.undefined = undefined or None
                  setRules.debugInfo.largeCommunity.ignored = ignored or None
            elif capiName == 'largeCommunityFilter' and communityLists is not None:
               undefined = " ".join( name for name in communityLists if
                                     name not in allCommLists[ 'largeCommunity' ] )
               if undefined:
                  setRules.debugInfo.largeCommunityFilter = AttributeDebug()
                  setRules.debugInfo.largeCommunityFilter.undefined = undefined

   setRules._populatedFields = populatedFields
   setRules._setCommands = setCommands
   return setRules

def getSetRules( entry ):
   populatedFields = []
   setCommands = []
   setRules = SetRules()
   asdotConfigured = isAsdotConfigured( asnConfig )
   for attr in SetActionInfo: # pylint: disable=consider-using-dict-items
      if not entry.setFlags.value & SetActionInfo[ attr ][ 'flag' ]:
         continue
      capiName = setRuleCapiName( attr )
      if capiName is None:
         continue
      populatedFields.append( capiName )
      setCommands.append( " ".join( SetActionInfo[attr]['command'] ) )
      value = getattr( entry, attr )
      if attr == 'metricType':
         value = metricTypeEnum[ value ]
      elif attr == 'level':
         value = isisLevelEnum[ value ]
      elif attr == 'isisMetricStyle':
         value = isisStyleEnum[ value ]
      elif attr == 'asPathPrepend':
         values = []
         for k in entry.asPathPrepend:
            value = entry.asPathPrepend[ k ]
            valueStr = bgpFormatAsn( value, asdotConfigured ) if value else 'auto'
            values.append( valueStr )
         if entry.asPathPrependRepeat > 1:
            values.extend( [ 'repeat', str( entry.asPathPrependRepeat ) ] )
         value = " ".join( values )
      elif attr == 'asPathReplace':
         values = []
         for k in entry.asPathReplace:
            value = entry.asPathReplace[ k ]
            if value == AS_PATH_MAX:
               valueStr = 'none'
            elif value == 0:
               valueStr = 'auto'
            else:
               valueStr = bgpFormatAsn( value, asdotConfigured )
            values.append( valueStr )
         if entry.asPathReplaceRepeat > 1:
            values.extend( [ 'repeat', str( entry.asPathReplaceRepeat ) ] )
         value = " ".join( values )
      elif attr == 'nextHopPeerAddr':
         value = NextHop( nextHopType='peerAddress' )
      elif attr == 'ipv6NextHopPeerAddr':
         value = NextHop( nextHopType='ipv6PeerAddress' )
      elif attr == 'nextHopUnchanged':
         value = NextHop( nextHopType='unchanged' )
      elif attr == 'ipv6NextHopUnchanged':
         value = NextHop( nextHopType='ipv6Unchanged' )
      elif attr == 'evpnNextHopUnchanged':
         value = NextHop( nextHopType='evpnUnchanged' )
      elif attr == 'ospfBitDn':
         value = None
      elif attr == 'ospfv3BitDn':
         value = None
      elif attr == 'nextHop':
         value = NextHop( nextHopType='ipAddress', ipAddress=value )
      elif attr == 'ipv6NextHop':
         value = NextHop( nextHopType='ipAddress', ipAddress=value.stringValue )
      elif attr in ( 'communityAddReplace', 'communityDelete', 'communityFilter',
                     'extCommunity', 'extCommunityFilter', 'largeCommunity',
                     'largeCommunityFilter' ):
         op = 'default'
         communityLists = None
         communityValues = None
         communityFilterType = None
         if attr == 'largeCommunity' or attr == 'largeCommunityFilter':
            commType = CommunityType.communityTypeLarge
         elif attr == 'extCommunity' or attr == 'extCommunityFilter':
            commType = CommunityType.communityTypeExtended
         elif attr == 'communityAddReplace' or attr == 'communityDelete' or\
              attr == 'communityFilter':
            commType = CommunityType.communityTypeStandard
         else:
            assert False, "Unrecognized community set attr: %s" % ( attr )
         # add in moving commType to list for model
         if value is not None:
            if value.flag != 'none':
               if value.useCommunityList:
                  communityLists = []
                  for name in value.communityListNameSet:
                     communityLists.append( name )
               elif value.community:
                  communityValues = []
                  for val in value.community:
                     communityValues.append( commValueToPrint( val,
                                             commType=commType,
                                             entry=entry,
                                             asdotConfigured=asdotConfigured ) )
            if value.flag == 'commSetAdditive':
               op = 'add'
            elif value.flag == 'commSetDelete':
               op = 'delete'
            elif value.flag == 'commSetFilter':
               op = 'filter'
               communityFilterType = getCommTypeModelList( value.commType )
            elif value.flag == 'commSetNone':
               op = 'none'

         if commType == CommunityType.communityTypeExtended and \
            entry.extCommLinkBandwidthSetFlag != \
                ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDefault:
            if communityValues is None:
               communityValues = []
            lbwComm = getExtCommTypeValue( 'lbw' )
            communityValues.append( commValueToPrint( lbwComm,
                                                      commType=commType,
                                                      entry=entry ) )
         value = CommunitySet( operationType=op,
                               communityLists=communityLists,
                               communityValues=communityValues,
                               communityFilterTypes=communityFilterType )
      elif attr == 'metric':
         for capiMedTypeStr, medType in capiMedTypes.items():
            if value.medType == medType:
               value = RouteMetric( value=value.metricValue,
                                    metricType=capiMedTypeStr )
               break
      elif attr == 'aigpMetric':
         for capiAigpMetricTypeStr, aigpMetricType in \
         capiAigpMetricTypes.items():
            if value.aigpMetricSetType == aigpMetricType:
               value = AigpMetric( value=value.metric,
                                   metricType=capiAigpMetricTypeStr )
               break
      elif attr == 'originAsValidity':
         value = originAsValidityEnum[ value ]
      elif attr == 'resolutionRibProfileConfig':
         value = ResolutionRibProfileConfig.fromTac( value )

      setattr( setRules, capiName, value )
   setRules._populatedFields = populatedFields
   setRules._setCommands = setCommands
   return setRules

def getSubRouteMap( entry ):
   subRouteMap = SubRouteMap( name=entry.subRouteMap.name,
                              invert=entry.subRouteMap.invert )
   return subRouteMap

def routeMapByName( routeMap, configSanity=False, routeMapConfig=None,
                    allCommLists=None, allPfxLists=None,
                    allAsPathLists=None, allTagSets=None, expanded=False ):
   routeMapEntries = {}
   sequences = sorted( routeMap.mapEntry )
   for seqno in sequences:
      entry = routeMap.mapEntry[ seqno ]
      permit = matchPermitEnum[ entry.permit ]
      if configSanity:
         matchRules = getConfigSanityMatchRules( entry, allCommLists,
                                                 allPfxLists, allAsPathLists,
                                                 allTagSets,
                                                 routeMapConfig.routeMap )
         setRules = getConfigSanitySetRules( entry, allCommLists, allPfxLists,
                                             allAsPathLists )
      else:
         matchRules = getMatchRules( entry )
         setRules = getSetRules( entry )
      continueClause = None
      continueSeqno = None
      if entry.setFlags.value & SetActionInfo[ 'continueSeq' ][ 'flag' ]:
         value = entry.continueSeq
         continueClause = True
         if value > 0:
            continueSeqno = value

      statementName = None
      if entry.statementName and entry.statementNameFromCli:
         statementName = entry.statementName

      subRouteMap = getSubRouteMap( entry )
      if configSanity:
         # subroutemap is undefined if it has a value set which is not a defined
         # route-map (empty-string indicates unset for b/w compatability)
         if not subRouteMap.name or subRouteMap.name in routeMapConfig.routeMap:
            subRouteMap = None
         else:
            subRouteMap.debugInfo = SubRMDebugInfo()
            subRouteMap.debugInfo.name = AttributeDebug()
            subRouteMap.debugInfo.name.undefined = subRouteMap.name

         sequences = sorted( routeMap.mapEntry )
         # we only want a continue clause if the continue sequence number is not
         # defined or if continue next is used in the last route map sequence
         if ( continueSeqno and continueSeqno in sequences ) or \
            ( not continueSeqno and seqno != sequences[ -1 ] ):
            continueSeqno = None
            continueClause = None

      description = []
      undefinedReference = configSanity and \
                           ( continueClause or
                             ( subRouteMap and
                               subRouteMap.debugInfo.name and
                               subRouteMap.debugInfo.name.undefined ) or
                             ( matchRules.debugInfo.community and
                               matchRules.debugInfo.community.undefined ) or
                             ( matchRules.debugInfo.extCommunity and
                               matchRules.debugInfo.extCommunity.undefined ) or
                             ( matchRules.debugInfo.largeCommunity and
                               matchRules.debugInfo.largeCommunity.undefined ) or
                             ( setRules.debugInfo.communityAddOrReplace and
                               setRules.debugInfo.communityAddOrReplace.
                                                                     undefined ) or
                             ( setRules.debugInfo.communityDelete and
                               setRules.debugInfo.communityDelete.undefined ) or
                             ( setRules.debugInfo.communityFilter and
                               setRules.debugInfo.communityFilter.undefined ) or
                             ( setRules.debugInfo.extCommunity and
                               setRules.debugInfo.extCommunity.undefined ) or
                             ( setRules.debugInfo.extCommunityFilter and
                               setRules.debugInfo.extCommunityFilter.undefined ) or
                             ( setRules.debugInfo.largeCommunity and
                               setRules.debugInfo.largeCommunity.undefined ) or
                             ( setRules.debugInfo.largeCommunityFilter and
                               setRules.debugInfo.largeCommunityFilter.undefined ) or
                             matchRules.prefixList or
                             matchRules.accessList or
                             matchRules.ipAddrDynamicPrefixList or
                             matchRules.ipv6PrefixList or
                             matchRules.ipv6AddrDynamicPrefixList or
                             matchRules.asPathList or
                             matchRules.routerIdPrefixList or
                             matchRules.routeTag or
                             matchRules.routeTagSet or
                             matchRules.aggregation
                             )

      ignoredReference = configSanity and \
                         ( ( setRules.debugInfo.communityAddOrReplace and
                             setRules.debugInfo.communityAddOrReplace.ignored ) or
                           ( setRules.debugInfo.communityDelete and
                             setRules.debugInfo.communityDelete.ignored ) or
                           ( setRules.debugInfo.communityFilter and
                             setRules.debugInfo.communityFilter.ignored ) or
                           ( setRules.debugInfo.extCommunity and
                             setRules.debugInfo.extCommunity.ignored ) or
                           ( setRules.debugInfo.extCommunityFilter and
                             setRules.debugInfo.extCommunityFilter.ignored ) or
                           ( setRules.debugInfo.largeCommunity and
                             setRules.debugInfo.largeCommunity.ignored ) or
                           ( setRules.debugInfo.largeCommunityFilter and
                             setRules.debugInfo.largeCommunityFilter.ignored ) )

      if expanded:
         for descKey in entry.description:
            description.append( entry.description[ descKey ] )
         routeMapEntries[ seqno ] = ExpandedRouteMapEntry( filterType=permit,
            statementName=statementName, matchRules=matchRules,
            subRouteMap=subRouteMap, setRules=setRules,
            continueClause=continueClause, continueSeqno=continueSeqno,
            description=description )
      elif not configSanity:
         for descKey in entry.description:
            description.append( entry.description[ descKey ] )
         routeMapEntries[ seqno ] = RouteMapEntry( filterType=permit,
            statementName=statementName, matchRules=matchRules,
            subRouteMap=subRouteMap, setRules=setRules,
            continueClause=continueClause, continueSeqno=continueSeqno,
            description=description )
      elif undefinedReference or ignoredReference:
         routeMapEntries[ seqno ] = RouteMapEntry( filterType=permit,
            statementName=statementName, matchRules=matchRules,
            subRouteMap=subRouteMap, setRules=setRules,
            continueClause=continueClause, continueSeqno=continueSeqno,
            description=description )
   if expanded:
      return ExpandedRouteMap( entries=routeMapEntries )
   return RouteMap( entries=routeMapEntries )

def routeMapContainerModel( routeMapConfig, mapName=None ):
   routeMaps = {}
   if mapName is None:
      for name in routeMapConfig.routeMap:
         routeMaps[ name ] = routeMapByName( routeMapConfig.routeMap[ name ] )
   elif mapName in routeMapConfig.routeMap:
      routeMaps[ mapName ] = routeMapByName( routeMapConfig.routeMap[ mapName ] )
   return RouteMaps( routeMaps=routeMaps )

def evaluatedRouteMap( routeMapName, log, discardSetRules=False ):
   """
   This function returns a PolicyEvalRouteMap model using the config of the route map
   based on a PolicyEvaluationLog (log). It creates the necessary debugInfo and
   result models in a PolicyEvalRouteMap.

   discardSetRules is set to True if the calling route map result is deny. This is
   then passed to sub route maps to indicate that their setters should be discarded
   from the model, even if that sub route map result is permit, since if the calling
   route map is deny then the setters are discarded.
   """
   if not log[ 'result' ]:
      # we set this in the case of a route map being deny. It is a one way
      # switch that should never be unset.
      discardSetRules = True
   routeMap = mapConfig.routeMap[ routeMapName ]
   matchingSequences = []
   evaluatedSequences = {}

   # get the evaluated sequences from the log, so only evaluated ones can
   # be present in the output. Make sure to sort them as integers and
   # not alphabetically.
   seqs = sorted( log[ 'seqs' ].keys(), key=int )
   for seqno in seqs:
      seqLog = log[ 'seqs' ][ seqno ]
      entry = routeMap.mapEntry[ int( seqno ) ]
      permit = matchPermitEnum[ entry.permit ]

      descriptions = list( entry.description.values() ) or None

      result = 'permit'
      if not seqLog[ 'result' ]:
         if seqLog[ 'matchActionDeny' ]:
            result = 'deny'
         else:
            result = 'fallThrough'
      else:
         matchingSequences.append( int( seqno ) )

      seqResults = PolicyEvalSequenceResult( result=result )

      matcherResults = []
      newEntry = Tac.newInstance( 'Routing::RouteMap::MapEntry', entry.seqno )
      failureEncountered = False
      # The matchRule dict is unorded. Order the keys in the dict to be in the order
      # they are evaluated. This is the order of ascending MatchOptions enum values.
      # These enum values are stored in the 'enum' field for each matcher in
      # MatchAttributes.
      sortedMatchOptions = list( entry.matchRule )
      sortedMatchOptions.sort( key=lambda op: MatchAttributes[ op ][ 'enum' ] )
      for op in sortedMatchOptions:
         # we need to copy the entry over to a temporary one and clear all but one
         # matcher at a time to create the list of matchRules where there
         # is one match rule and one corresponding debuginfo.
         copyMapEntry( newEntry, entry )
         tmp = newEntry.matchRule[ op ]
         newEntry.matchRule.clear()
         newEntry.matchRule.addMember( tmp )

         if "passedMatchers" in seqLog and op in seqLog[ "passedMatchers" ]:
            matcher = getMatchRules( newEntry )
            matcher.debugInfo = DebugInfo( matchResult="matched" )
            matcherResults.append( matcher )
         elif 'failingMatcher' in seqLog and op == seqLog[ 'failingMatcher' ]:
            matcher = getMatchRules( newEntry )
            matcher.debugInfo = DebugInfo( matchResult="failed" )
            matcherResults.append( matcher )
            failureEncountered = True
            # once we hit a failure we're not interested in any more matchers.
            break
         else:
            # some matchers are not supported in every context. To determine this
            # if the matcher is not in the passing matchers in the log and a
            # failure has not occured then the matcher is unsupported.
            matcher = getMatchRules( newEntry )
            matcher.debugInfo = DebugInfo( matchResult="unsupported" )
            matcherResults.append( matcher )
      if not matcherResults:
         # to prevent an empty list appearing in the output set it back to None
         matcherResults = None

      setRules = None
      if discardSetRules:
         # if discardSetRules is True then the setters in this route map were never
         # applied due to the result of this route map or one that called it, etc,
         # so we can skip adding them to the model
         pass
      elif seqLog[ 'result' ]:
         # Only add set rules if the sequence matched
         setRules = getSetRules( entry )
         if not setRules._populatedFields:
            # to prevent an empty dict appearing in the output set it back to None
            setRules = None

      subRouteMapName = None
      subRouteMap = None
      subRouteMapInvertResult = None
      # in the event of a failing matcher we do not add the sub route map
      if not failureEncountered and entry.subRouteMap:
         if 'subRouteMap' in seqLog:
            subRouteMapLog = seqLog[ 'subRouteMap' ]
            subRouteMapName = subRouteMapLog[ 'rmname' ]
            subRouteMap = evaluatedRouteMap( subRouteMapName,
                                             seqLog[ 'subRouteMap' ],
                                             discardSetRules=discardSetRules )
            subRouteMapInvertResult = seqLog[ 'subRouteMapInvertResult' ]
         elif entry.subRouteMap.name not in mapConfig.routeMap:
            # the sub route map is undefined as it is not present in the config
            # and the result must have been permit.
            subRouteMapName = entry.subRouteMap.name
            subRouteMapResult = PolicyEvalRouteMapResult(
               result="permit", permittingSeqnos=None )
            subRouteMap = PolicyEvalRouteMap( routeMapName=subRouteMapName,
                                              routeMapResult=subRouteMapResult,
                                              sequences=None,
                                              notFound=True )
            subRouteMapInvertResult = entry.subRouteMap.invert
         else:
            # A cycle must have occured in the evaluated sub route maps, for example
            #     route map A -> sub route map B -> sub route map A
            # such that the sub route map A is configured in a sequene in B, but not
            # evaluated to avoid this loop.
            subRouteMapName = entry.subRouteMap.name
            subRouteMapResult = PolicyEvalRouteMapResult(
               result="permit", permittingSeqnos=None )
            subRouteMap = PolicyEvalRouteMap( routeMapName=subRouteMapName,
                                              routeMapResult=subRouteMapResult,
                                              sequences=None,
                                              routeMapCycleDetected=True )
            subRouteMapInvertResult = entry.subRouteMap.invert

      continueNext = None
      continueSeqno = None
      missingContinueTarget = None
      if seqLog[ "result" ]:
         if entry.setFlags.value & SetActionInfo[ 'continueSeq' ][ 'flag' ]:
            # get the configured continue statement
            value = entry.continueSeq
            if value > 0:
               continueSeqno = value
            else:
               continueNext = True

            # determine if it was in the log, if not it was undefined
            if 'continue' in seqLog and seqLog[ 'continue' ]:
               seqResults.continueSeqno = seqLog[ 'continue' ]
            else:
               missingContinueTarget = True

      evaluatedSequences[ seqno ] = PolicyEvalSequence(
         filterType=permit, matcherResults=matcherResults,
         subRouteMap=subRouteMap,
         subRouteMapInvertResult=subRouteMapInvertResult,
         continueNextSequence=continueNext, continueSeqno=continueSeqno,
         missingContinueTarget=missingContinueTarget,
         setRules=setRules, descriptions=descriptions, sequenceResult=seqResults )

   result = "permit"
   if not log[ 'result' ]:
      if log[ 'defaultDeny' ]:
         result = "noMatchingSequenceDeny"
      else:
         result = "explicitDeny"
      # if the result of the route map was deny then we can
      # drop the matching sequences
      matchingSequences = None

   routeMapResult = PolicyEvalRouteMapResult( result=result,
                                              permittingSeqnos=matchingSequences )

   return PolicyEvalRouteMap( routeMapName=routeMapName,
                              sequences=evaluatedSequences,
                              routeMapResult=routeMapResult )

def createPolicyEvalRouteMapModel( result ):
   if 'log' in result:
      # the route map was evaluated by the debugger, form a PolicyEvalRouteMap
      # based off the generated log
      log = result[ 'log' ]
      rmName = log[ 'rmname' ]
      return evaluatedRouteMap( rmName, log )
   elif 'missingPolicy' in result:
      # the route map was undefined or had missing policy and the
      # debugger didn't evaluate the route map as a result.
      rmName = result[ 'missingPolicy' ]
      missingPolicyResult = "explicitDeny"
      if result[ 'permit' ]:
         missingPolicyResult = "permit"
      routeMapResult = PolicyEvalRouteMapResult( result=missingPolicyResult,
                                                 permittingSeqnos=None )
      return PolicyEvalRouteMap( routeMapName=rmName,
                                 sequences=None,
                                 routeMapResult=routeMapResult,
                                 notFound=True )
   else:
      # No route map was applied to the peer
      return None

#------------------------------------------------------------------------
# Tag Set
# [no] tag set <name> <value>
# ------------------------------------------------------------------------
class TagSetCmd( CliCommand.CliCommandClass ):
   syntax = 'tag set TAG_SET TAG_VALUE'
   noOrDefaultSyntax = 'tag set TAG_SET ...'
   data = {
         'tag': CommonTokens.tagToken,
         'set': CommonTokens.setToken,
         'TAG_SET': CommonTokens.tagSetNameMatcher,
         'TAG_VALUE': CliMatcher.IntegerMatcher(
            MatchAttributes[ 'matchTag' ][ 'min' ],
            MatchAttributes[ 'matchTag' ][ 'max' ],
            helpdesc='Route tag value' )
   }
   handler = 'RouteMapCliHandler.handlerTagSet'
   noOrDefaultHandler = handler

if RouteMapToggleLib.toggleSingleMatchTagListEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( TagSetCmd )

# ------------------------------------------------------------------------
# Register top-level CLI commands
#------------------------------------------------------------------------
# [no] service routing configuration route-map set-operations
#                                       [merged | sequential]
class SetOperationsCmd( CliCommand.CliCommandClass ):
   syntax = '''service routing configuration route-map set-operations
               ( merged | sequential )'''
   noOrDefaultSyntax = 'service routing configuration route-map set-operations ...'
   data = {
      'service': CliToken.Service.serviceMatcherForConfig,
      'routing': IraServiceCli.routingMatcherForConfig,
      'configuration': CliToken.Service.configMatcherForAfterService,
      'route-map': 'Route map configuration',
      'set-operations': 'Route map set operations',
      'merged': 'Process additions before deletions of cumulative path attributes',
      'sequential': 'Process clauses in sequence order'
   }
   handler = "RouteMapCliHandler.handlerSetOperationsCmd"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( SetOperationsCmd )

# [no] service routing configuration route-map no-set-community explicit-members
class NoSetCommExplicitCmd( CliCommand.CliCommandClass ):
   syntax = """service routing configuration route-map no-set-community
               explicit-members"""
   noOrDefaultSyntax = syntax
   data = {
      "service": "Configure service parameters",
      "routing": "Routing service configuration",
      "configuration": "Configuration settings",
      "route-map": "Route map configuration",
      "no-set-community": 'Configure "no set community ..." command behavior',
      "explicit-members": '"no set community ..." command only removes specified'
      ' communities',
      }
   handler = "RouteMapCliHandler.handlerNoSetCommExplicitCmd"
   noOrDefaultHandler = "RouteMapCliHandler.noOrDefaultHandlerNoSetCommExplicitCmd"

BasicCli.GlobalConfigMode.addCommandClass( NoSetCommExplicitCmd )

mapNameMatcher = CommonTokens.routeMapName

statementNameMatcher = CliMatcher.PatternMatcher( '[A-Za-z0-9_-]+',
   helpname='WORD',
   helpdesc='Statement Name' )
routeMapPermitOrDeny = CliMatcher.EnumMatcher( {
   'permit': 'Specify route-map sequence number to allow',
   'deny': 'Specify route-map sequence number to reject',
} )
sequenceMatcher = CliMatcher.IntegerMatcher( seqnoMin,
   seqnoMax,
   helpdesc='Index in the sequence' )
routeMapNameSrc = CliMatcher.DynamicNameMatcher(
   getRouteMapNames,
   'Source route map name',
   helpname='WORD',
   pattern=rtMapCommListPreListRe )

class GoToRouteMapMode( CliCommand.CliCommandClass ):
   syntax = 'route-map MAP [ statement STATEMENT ] [ ACTION [ SEQUENCE ] ]'
   noOrDefaultSyntax = syntax
   data = {
         'route-map': 'route-map',
         'MAP': mapNameMatcher,
         'statement': 'statement',
         'STATEMENT': statementNameMatcher,
         'ACTION': routeMapPermitOrDeny,
         'SEQUENCE': sequenceMatcher,
         }
   handler = "RouteMapCliHandler.gotoRouteMapMode"
   noOrDefaultHandler = "RouteMapCliHandler.deleteRouteMapMode"

BasicCli.GlobalConfigMode.addCommandClass( GoToRouteMapMode )

# route-map <destination_name> { copy|rename } <source_name> [ overwrite ]
class CopyOrRenameRouteMapCmd( CliCommand.CliCommandClass ):
   syntax = 'route-map DESTINATION ACTION SOURCE [ overwrite ]'
   data = {
         'route-map': 'route-map',
         'DESTINATION': mapNameMatcher,
         'ACTION': CommonTokens.copyOrRenameMatcher,
         'SOURCE': routeMapNameSrc,
         'overwrite': CommonTokens.overwrite,
         }
   handler = "RouteMapCliHandler.copyOrRenameRouteMapMode"

BasicCli.GlobalConfigMode.addCommandClass( CopyOrRenameRouteMapCmd )

# show route-map config-sanity
class ShowRouteMapConfigSanityCmd( ShowCliCommandClass ):
   syntax = 'show route-map config-sanity'
   data = {
      'route-map': CommonTokens.routeMap,
      'config-sanity': 'Check for undefined route map elements'
   }
   cliModel = ConfigSanityRouteMaps
   handler = "RouteMapCliHandler.handlerShowRouteMapConfigSanityCmd"

BasicCli.addShowCommandClass( ShowRouteMapConfigSanityCmd )

# show route-map expanded
class ShowRouteMapExpandedCmd( ShowCliCommandClass ):
   syntax = 'show route-map expanded [NAME] [pruned]'
   data = {
      'route-map': CommonTokens.routeMap,
      'expanded': 'Display sub route map definitions inline',
      'NAME': CommonTokens.routeMapName,
      'pruned': 'Avoid reprinting common subtrees',
   }
   cliModel = ExpandedRouteMaps
   handler = "RouteMapCliHandler.handlerShowRouteMapExpandedCmd"

BasicCli.addShowCommandClass( ShowRouteMapExpandedCmd )

# show route-map <name>
class ShowRouteMapCmd( ShowCliCommandClass ):
   syntax = 'show route-map [ NAME ]'
   data = {
      'route-map': CommonTokens.routeMap,
      'NAME': CommonTokens.routeMapName
   }
   cliModel = RouteMaps
   handler = "RouteMapCliHandler.handlerShowRouteMapCmd"

BasicCli.addShowCommandClass( ShowRouteMapCmd )

class RouteMapModeShowCmd( ShowCliCommandClass ):
   syntax = "show [ pending | diff ]"
   data = {
      'pending': 'Show pending list in this route map entry',
      'diff': 'Show the difference between active and pending list'
   }
   handler = "RouteMapCliHandler.handlerRouteMapModeShowCmd"

RouteMapMode.addShowCommandClass( RouteMapModeShowCmd )

class RouteMapModeAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
         'abort': 'Exit route-map configuration without committing changes'
         }
   handler = "RouteMapCliHandler.handlerRouteMapModeAbortCmd"

RouteMapMode.addCommandClass( RouteMapModeAbortCmd )

matcherDescription = CliMatcher.KeywordMatcher( 'description',
            helpdesc='Add a description for this peer to the configuration' )
matcherDescriptionString = CliMatcher.StringMatcher( helpname='DESCRIPTION',
            helpdesc='Description of the Peer' )

class RouteMapModeDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
         'description': matcherDescription,
         'DESCRIPTION': matcherDescriptionString
         }
   handler = "RouteMapCliHandler.setDescription"
   noOrDefaultHandler = "RouteMapCliHandler.noDescription"

RouteMapMode.addCommandClass( RouteMapModeDescriptionCmd )

#-----------------------------------------------------------------------
# Subroutemap binding rules
#-----------------------------------------------------------------------
subRouteMapListRule = CliMatcher.DynamicNameMatcher(
      getSubRouteMapNames,
      'Sub route map',
      helpname='WORD',
      pattern=rtMapCommListPreListRe )

class SubRouteMapCmd( CliCommand.CliCommandClass ):
   syntax = 'sub-route-map [ invert-result ] NAME'
   noOrDefaultSyntax = 'sub-route-map ...'
   data = {
         'sub-route-map': subRouteMapMatcher,
         'invert-result': invertSubResultMatcher,
         'NAME': subRouteMapListRule,
         }
   handler = "RouteMapCliHandler.handlerSubRouteMapCmd"
   noOrDefaultHandler = handler

RouteMapMode.addCommandClass( SubRouteMapCmd )

#------------------------------------------------------------------------
# Match and set value binding rules
#------------------------------------------------------------------------
prefixListDynamicNameMatcher = CliMatcher.DynamicNameMatcher(
         getPrefixListNames, 'Prefix list', helpname='WORD',
         pattern=rtMapCommListPreListRe )
prefixListNameMatcher = CliMatcher.DynamicNameMatcher( getPrefixListNames,
         "Prefix list", pattern=rtMapCommListPreListRe )
ipv6PrefixListDynamicNameMatcher = CliMatcher.DynamicNameMatcher(
         getIpv6PrefixListNames, 'IPv6 Prefix list',
         helpname='WORD', pattern=rtMapCommListPreListRe )
ipv6PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv6PrefixListNames,
         "IPv6 Prefix list", pattern=rtMapCommListPreListRe )
srcPrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getPrefixListNames,
         "Source prefix list name", pattern=rtMapCommListPreListRe )
srcIpv6PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv6PrefixListNames,
         "Source IPv6 prefix list name", pattern=rtMapCommListPreListRe )

# match ip access-list
if isMatchAttrEnabled( 'matchAccessList' ):
   aclMatcher = CliMatcher.DynamicNameMatcher(
      getStandardAccessListNames,
      helpdesc='Access list',
      helpname='WORD' )

   class MatchAccessList( CliCommand.CliCommandClass ):
      syntax = 'match ip address access-list ACL'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ip': CommonTokens.matchIpv4,
         'address': CommonTokens.matchIpAddress,
         'access-list': CommonTokens.matchAccessList,
         'ACL': aclMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchAccessList"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchAccessList )

# match ip prefix-list
if isMatchAttrEnabled( 'matchPrefixList' ):
   class MatchPrefixList( CliCommand.CliCommandClass ):
      if RouteMapToggleLib.toggleMatchPrefixListInvertResultEnabled():
         syntax = 'match [ invert-result ] ip address prefix-list NAME'
      else:
         syntax = 'match ip address prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'invert-result': CommonTokens.invertResult,
         'ip': CommonTokens.matchIpv4,
         'address': CommonTokens.matchIpAddress,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': prefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchPrefixList"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchPrefixList )

# match ipv6 prefix-list
if isMatchAttrEnabled( 'matchIpv6PrefixList' ):
   class MatchPrefixListV6( CliCommand.CliCommandClass ):
      if RouteMapToggleLib.toggleMatchPrefixListInvertResultEnabled():
         syntax = 'match [ invert-result ] ipv6 address prefix-list NAME'
      else:
         syntax = 'match ipv6 address prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'invert-result': CommonTokens.invertResult,
         'ipv6': CommonTokens.matchIpv6,
         'address': CommonTokens.matchIpAddress,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': ipv6PrefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchPrefixListV6"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchPrefixListV6 )

# match ip gateway
class MatchIpGateway( CliCommand.CliCommandClass ):
   syntax = 'match ip gateway GATEWAY'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'ip': CommonTokens.matchIpv4,
      'gateway': 'IP gateway',
      'GATEWAY': IpAddrMatcher.IpAddrMatcher( helpdesc='Gateway' ),
   }
   handler = "RouteMapCliHandler.handlerMatchIpGateway"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchGateway' ):
   RouteMapMode.addCommandClass( MatchIpGateway )

# match ip next-hop <addr>
if isMatchAttrEnabled( 'matchNextHop' ):
   nextHopHiddenMatcher = CliCommand.Node(
      matcher=IpAddrMatcher.IpAddrMatcher(
         helpdesc='Route next hop IP address' ), hidden=True )

   class MatchNextHop( CliCommand.CliCommandClass ):
      syntax = 'match ip next-hop NEXTHOP'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ip': CommonTokens.matchIpv4,
         'next-hop': CommonTokens.matchNextHop,
         'NEXTHOP': nextHopHiddenMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchNextHop"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchNextHop )

# match ipv6 next-hop <addr>
if isMatchAttrEnabled( 'matchIpv6NextHop' ):
   ipv6NextHopHiddenMatcher = CliCommand.Node(
      matcher=Ip6AddrMatcher.Ip6AddrMatcher(
         helpdesc='IPv6 Next hop' ), hidden=True )

   class MatchNextHopV6( CliCommand.CliCommandClass ):
      syntax = 'match ipv6 next-hop NEXTHOP'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ipv6': CommonTokens.matchIpv6,
         'next-hop': CommonTokens.matchNextHop,
         'NEXTHOP': ipv6NextHopHiddenMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchNextHopV6"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchNextHopV6 )

# match ip next-hop prefix-list <prefix-list>
if isMatchAttrEnabled( 'matchNextHopPrefixList' ):
   class MatchNextHopPfxList( CliCommand.CliCommandClass ):
      syntax = 'match ip next-hop prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ip': CommonTokens.matchIpv4,
         'next-hop': CommonTokens.matchNextHop,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': prefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchNextHopPfxList"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchNextHopPfxList )

# match ipv6 next-hop prefix-list <prefix-list>
if isMatchAttrEnabled( 'matchIpv6NextHopPrefixList' ):
   class MatchNextHopPfxListV6( CliCommand.CliCommandClass ):
      syntax = 'match ipv6 next-hop prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ipv6': CommonTokens.matchIpv6,
         'next-hop': CommonTokens.matchNextHop,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': ipv6PrefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchNextHopPfxListV6"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchNextHopPfxListV6 )

# match ip resolved-next-hop prefix-list <prefix-list>
if isMatchAttrEnabled( 'matchResolvedNextHopPrefixList' ):
   class MatchResolvedNextHopPfxList( CliCommand.CliCommandClass ):
      syntax = 'match ip resolved-next-hop prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ip': CommonTokens.matchIpv4,
         'resolved-next-hop': CommonTokens.matchResolvedNextHop,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': prefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchResolvedNextHopPfxList"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchResolvedNextHopPfxList )

# match ipv6 resolved-next-hop prefix-list <prefix-list>
if isMatchAttrEnabled( 'matchIpv6ResolvedNextHopPrefixList' ):
   class MatchResolvedNextHopPfxListV6( CliCommand.CliCommandClass ):
      syntax = 'match ipv6 resolved-next-hop prefix-list NAME'
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'ipv6': CommonTokens.matchIpv6,
         'resolved-next-hop': CommonTokens.matchResolvedNextHop,
         'prefix-list': CommonTokens.matchPrefixList,
         'NAME': ipv6PrefixListDynamicNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchResolvedNextHopPfxListV6"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchResolvedNextHopPfxListV6 )

# match AS path regex
class MatchAsPathName( CliCommand.CliCommandClass ):
   syntax = 'match as-path-name NAME'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'as-path-name': 'BGP AS path name',
      'NAME': CliMatcher.StringMatcher( helpname='WORD', helpdesc='AS path regex' )
   }
   handler = "RouteMapCliHandler.handlerMatchAsPathName"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchAsPath' ):
   RouteMapMode.addCommandClass( MatchAsPathName )

if isMatchAttrEnabled( matchContribAggAttrOption ):
   class MatchAggregateAttributes( CliCommand.CliCommandClass ):
      syntax = (
         'match [ invert-result ] aggregate-role contributor '
         '[ aggregate-attributes MAP ]'
      )
      noOrDefaultSyntax = syntax
      data = {
         'match': CommonTokens.match,
         'invert-result': invertSubResultMatcher,
         'aggregate-role': CommonTokens.aggregateRole,
         'contributor': CommonTokens.contributor,
         'aggregate-attributes': CommonTokens.aggregateAttr,
         'MAP': mapNameMatcher,
      }
      handler = "RouteMapCliHandler.handlerMatchAggregateAttributes"
      noOrDefaultHandler = handler
   RouteMapMode.addCommandClass( MatchAggregateAttributes )

class MatchCommunityCmd( CliCommand.CliCommandClass ):
   syntax = (
      'match [invert-result] community [or-results] {COMMUNITIES} [exact-match]'
   )
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'community': CommonTokens.matchCommunity,
      'or-results': CommonTokens.orResults,
      'COMMUNITIES': CliMatcher.DynamicNameMatcher(
         getCommunityListNames,
         'Community list name',
         pattern=communityListNameRe,
      ),
      'exact-match': CommonTokens.exactMatch,
   }
   handler = "RouteMapCliHandler.handlerMatchCommunityCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchCommunity' ):
   RouteMapMode.addCommandClass( MatchCommunityCmd )

class MatchExtCommunityCmd( CliCommand.CliCommandClass ):
   syntax = (
      'match [invert-result] extcommunity [or-results] {COMMUNITIES} [exact-match]'
   )
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'extcommunity': CommonTokens.matchExtCommunity,
      'or-results': CommonTokens.orResults,
      'COMMUNITIES': CliMatcher.DynamicNameMatcher(
         getCommunityListNames,
         'Extended community list name',
         pattern=communityListNameRe,
      ),
      'exact-match': CommonTokens.exactMatch,
   }
   handler = "RouteMapCliHandler.handlerMatchExtCommunityCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchExtCommunity' ):
   RouteMapMode.addCommandClass( MatchExtCommunityCmd )

class MatchLargeCommunityCmd( CliCommand.CliCommandClass ):
   syntax = (
      'match [invert-result] large-community [or-results] {COMMUNITIES} '
      '[exact-match]'
   )
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'large-community': CommonTokens.matchLargeCommunity,
      'or-results': CommonTokens.orResults,
      'COMMUNITIES': CliMatcher.DynamicNameMatcher(
         getCommunityListNames,
         'Large community list name',
         pattern=communityListNameRe,
      ),
      'exact-match': CommonTokens.exactMatch,
   }
   handler = "RouteMapCliHandler.handlerMatchLargeCommunityCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchLargeCommunity' ):
   RouteMapMode.addCommandClass( MatchLargeCommunityCmd )

communityInstanceCountMatcher = CliMatcher.IntegerMatcher(
      POLICY_COMM_INST_MIN, POLICY_COMM_INST_MAX,
      helpdesc='Number of communities'
)

class MatchCommunityInstances( CliCommand.CliCommandClass ):
   syntax = ( 'match [ invert-result ] community instances '
                  '( ( <= LE [ and >= GE ] ) | '
                     '( = EQ ) | '
                    '( >= GE [ and <= LE ] ) )' )
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'community': CommonTokens.matchCommunity,
      'instances': communityInstancesMatcher,
      '<=': CommonTokens.lowerThanEqual,
      '=': CommonTokens.equal,
      '>=': CommonTokens.greaterThanEqual,
      'LE': communityInstanceCountMatcher,
      'GE': communityInstanceCountMatcher,
      'EQ': communityInstanceCountMatcher,
      'and': CommonTokens.andToken
   }
   handler = "RouteMapCliHandler.handlerMatchCommunityInstances"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchCommunityInstances' ):
   RouteMapMode.addCommandClass( MatchCommunityInstances )

class MatchDistanceCmd( CliCommand.CliCommandClass ):
   # match distance
   syntax = 'match distance DIST'
   noOrDefaultSyntax = syntax

   data = {
      'match': CommonTokens.match,
      'distance': 'Protocol independent distance',
      'DIST': CliMatcher.IntegerMatcher(
                  SetActionInfo[ 'distance' ][ 'min' ],
                  SetActionInfo[ 'distance' ][ 'max' ],
                  helpdesc='Route distance' )
   }
   handler = "RouteMapCliHandler.handlerMatchDistanceCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchDistance' ):
   RouteMapMode.addCommandClass( MatchDistanceCmd )

interfaceMatcher = VirtualIntfRule.IntfMatcher()
interfaceMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
interfaceMatcher |= SwitchIntfCli.SwitchIntf.matcher
interfaceMatcher |= SubIntfCli.subMatcher
interfaceMatcher |= VlanIntf.matcher
interfaceMatcher |= LoopbackIntfCli.LoopbackIntf.matcher
interfaceMatcher |= LagIntfCli.EthLagIntf.matcher
interfaceMatcher |= LagIntfCli.subMatcher

class MatchInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'match interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'interface': 'Interface',
      'INTF': interfaceMatcher,
   }
   handler = "RouteMapCliHandler.handlerMatchInterfaceCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchInterface' ):
   RouteMapMode.addCommandClass( MatchInterfaceCmd )

class MatchLocalPrefCmd( CliCommand.CliCommandClass ):
   # match local-preference
   syntax = 'match local-preference BGP_LOCAL_PREF'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'local-preference': 'BGP local preference',
      'BGP_LOCAL_PREF': CliMatcher.IntegerMatcher(
                            MatchAttributes[ 'matchLocalPref' ][ 'min' ],
                            MatchAttributes[ 'matchLocalPref' ][ 'max' ],
                            helpdesc='BGP local preference' )
   }
   handler = "RouteMapCliHandler.handlerMatchLocalPrefCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchLocalPref' ):
   RouteMapMode.addCommandClass( MatchLocalPrefCmd )

class MatchMedCmd( CliCommand.CliCommandClass ):
   syntax = 'match med BGP_MED'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'med': 'BGP MED',
      'BGP_MED': CliMatcher.IntegerMatcher(
                    MatchAttributes[ 'matchMed' ][ 'min' ],
                    MatchAttributes[ 'matchMed' ][ 'max' ],
                    helpdesc='BGP MED' )
   }
   handler = "RouteMapCliHandler.handlerMatchMedCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchMed' ):
   RouteMapMode.addCommandClass( MatchMedCmd )

class MatchMetricCmd( CliCommand.CliCommandClass ):
   # match metric
   syntax = 'match metric RT_METRIC'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'metric': 'Route metric',
      'RT_METRIC': CliMatcher.IntegerMatcher(
                       MatchAttributes[ 'matchMetric' ][ 'min' ],
                       MatchAttributes[ 'matchMetric' ][ 'max' ],
                       helpdesc='Route metric' )
   }
   handler = "RouteMapCliHandler.handlerMatchMetricCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchMetric' ):
   RouteMapMode.addCommandClass( MatchMetricCmd )

class MatchMetricTypeCmd( CliCommand.CliCommandClass ):
   # match metric-type
   # OSPF type-1 and type-2 metric types
   syntax = 'match metric-type OSPF_TYPE'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'metric-type': 'Route metric type',
      'OSPF_TYPE': CliMatcher.EnumMatcher( {
                       'type-1': 'OSPF type 1',
                       'type-2': 'OSPF type 2',
                    } )
   }
   handler = "RouteMapCliHandler.handlerMatchMetricTypeCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchMetricType' ):
   RouteMapMode.addCommandClass( MatchMetricTypeCmd )

class MatchIsisLevelCmd( CliCommand.CliCommandClass ):
   # match isis level
   syntax = 'match isis level ISIS_LVL'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'isis': CommonTokens.isis,
      'level': 'Route level',
      'ISIS_LVL': CliMatcher.EnumMatcher( {
                      'level-1': 'IS-IS level 1',
                      'level-2': 'IS-IS level 2',
                   } )
   }
   handler = "RouteMapCliHandler.handlerMatchIsisLevelCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchIsisLevel' ):
   RouteMapMode.addCommandClass( MatchIsisLevelCmd )

matchProtocolMatcher = CliMatcher.EnumMatcher( {
      'attached-host': 'ARP generated host routes',
      'bgp': 'Border Gateway Protocol',
      'connected': 'Connected interface routes',
      'dynamic': 'Dynamic policy routes',
      'isis': 'IS-IS',
      'ospf': 'OSPF protocol',
      'ospfv3': 'Open Shortest Path First (OSPF) version 3',
      'rip': 'Routing Information Protocol',
      'static': 'Static routes'
} )

class MatchProtocolCmd( CliCommand.CliCommandClass ):
   # match protocol
   syntax = 'match source-protocol PROTOCOL'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'source-protocol': 'Source routing protocol',
      'PROTOCOL': matchProtocolMatcher
   }
   handler = "RouteMapCliHandler.handlerMatchProtocolCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchProtocol' ):
   RouteMapMode.addCommandClass( MatchProtocolCmd )

class MatchTagCmd( CliCommand.CliCommandClass ):
   # match tag ( ROUTE_TAG | set TAG_SET )
   if RouteMapToggleLib.toggleSingleMatchTagListEnabled():
      syntax = 'match tag ( ROUTE_TAG | ( set TAG_SET ) )'
   else:
      syntax = 'match tag ROUTE_TAG'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'tag': CommonTokens.tagToken,
      'ROUTE_TAG': CliMatcher.IntegerMatcher(
                       MatchAttributes[ 'matchTag' ][ 'min' ],
                       MatchAttributes[ 'matchTag' ][ 'max' ],
                       helpdesc='Route tag' ),
      'set': CommonTokens.setToken,
      'TAG_SET': CommonTokens.tagSetNameMatcher,
   }
   handler = "RouteMapCliHandler.handlerMatchTagCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchTag' ):
   RouteMapMode.addCommandClass( MatchTagCmd )

class MatchRouterIdCmd( CliCommand.CliCommandClass ):
   # match router id
   syntax = 'match router-id prefix-list PFX_LIST'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'router-id': 'Router ID',
      'prefix-list': 'Prefix list',
      'PFX_LIST': prefixListNameMatcher
   }
   handler = "RouteMapCliHandler.handlerMatchRouterIdCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchRouterId' ):
   RouteMapMode.addCommandClass( MatchRouterIdCmd )

routeTypeMatcherkeywordsHelpDescMapping = {
   'confederation-external': 'BGP confederation external route',
   'external': 'External route',
   'internal': 'Internal route',
   'local': 'Local route',
   'vpn': 'VPN route',
   }

routeTypeMatcher = CliMatcher.EnumMatcher( routeTypeMatcherkeywordsHelpDescMapping )

class MatchRouteTypeCmd( CliCommand.CliCommandClass ):
   # match route type
   syntax = 'match route-type RT_TYPE'
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'route-type': 'Route Type',
      'RT_TYPE': routeTypeMatcher
   }
   handler = "RouteMapCliHandler.handlerMatchRouteTypeCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchRouteType' ):
   RouteMapMode.addCommandClass( MatchRouteTypeCmd )

# match origin-as
class MatchOriginAsCmd( CliCommand.CliCommandClass ):
   syntax = "match origin-as AS_NUM"
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'origin-as': CommonTokens.originAs,
      'AS_NUM': AsNumCliExpr,
   }
   handler = "RouteMapCliHandler.handlerMatchOriginAsCmd"
   noOrDefaultHandler = handler

# match origin-as validity
class MatchOriginAsValidityCmd( CliCommand.CliCommandClass ):
   syntax = "match [invert-result] origin-as validity VALIDITY"
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'origin-as': CommonTokens.originAs,
      'validity': CliCommand.Node( CommonTokens.validity ),
      'VALIDITY': CommonTokens.originAsValidityEnum,
   }
   handler = "RouteMapCliHandler.handlerMatchOriginAsValidityCmd"
   noOrDefaultHandler = handler

# match as
class MatchAsCmd( CliCommand.CliCommandClass ):
   syntax = "match as AS_NUM"
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'as': CommonTokens.bgpAs,
      'AS_NUM': AsNumCliExpr,
   }
   handler = "RouteMapCliHandler.handlerMatchAsCmd"
   noOrDefaultHandler = handler

# match as-path
class MatchAsPath( CliCommand.CliCommandClass ):
   syntax = "match as-path AS_PATH_LIST"
   noOrDefaultSyntax = syntax

   data = {
      'match': CommonTokens.match,
      'as-path': CommonTokens.asPathFilter,
      'AS_PATH_LIST': CommonTokens.asPathListName,
   }
   handler = "RouteMapCliHandler.handlerMatchAsPath"
   noOrDefaultHandler = handler

class AsPathLengthRangeExpr( CliCommand.CliExpression ):
   fmt_ = {
      'eqExpr': '= VALUE',
      'leqExpr': '<= END [and >= START]',
      'geqExpr': '>= START [and <= END]',
   }
   expression = '({eqExpr}) | ({leqExpr}) | ({geqExpr})'.format( **fmt_ )

   data = {
      '=': CommonTokens.equal,
      '<=': CommonTokens.lowerThanEqual,
      '>=': CommonTokens.greaterThanEqual,
      'and': CommonTokens.andToken,
      'VALUE': CommonTokens.asPathLengthMatcher,
      'START': CommonTokens.asPathLengthMatcher,
      'END': CommonTokens.asPathLengthMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      value = Bunch.Bunch()
      if '=' in args:
         value.start = value.end = args[ 'VALUE' ]
      else:
         value.start = args.get( 'START', POLICY_ASPATH_LENGTH_MIN )
         value.end = args.get( 'END', POLICY_ASPATH_LENGTH_MAX )

      if value.start > value.end:
         mode.addErrorAndStop( "Invalid match clause" )

      args[ 'AS_PATH_LENGTH_EXPR' ] = value

class MatchAsPathLength( CliCommand.CliCommandClass ):
   syntax = "match [invert-result] as-path length AS_PATH_LENGTH_EXPR"
   noOrDefaultSyntax = syntax
   data = {
      'match': CommonTokens.match,
      'invert-result': CommonTokens.invertResult,
      'as-path': CommonTokens.asPathFilter,
      'length': CommonTokens.asPathLength,
      'AS_PATH_LENGTH_EXPR': AsPathLengthRangeExpr,
   }
   handler = "RouteMapCliHandler.handlerMatchAsPathLength"
   noOrDefaultHandler = handler

# match isis instance
class MatchIsisInstance( CliCommand.CliCommandClass ):
   syntax = "match isis instance INSTANCE_ID"
   noOrDefaultSyntax = syntax

   data = {
      'match': CommonTokens.match,
      'isis': CommonTokens.isis,
      'instance': CommonTokens.instance,
      'INSTANCE_ID': CliMatcher.IntegerMatcher( POLICY_ISIS_INST_MIN,
                                                POLICY_ISIS_INST_MAX,
                              helpdesc='IS-IS instance identifier' ),
   }
   handler = "RouteMapCliHandler.handlerMatchIsisInstance"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchAsPathList' ):
   RouteMapMode.addCommandClass( MatchAsPath )

if isMatchAttrEnabled( 'matchPeerAs' ):
   RouteMapMode.addCommandClass( MatchAsCmd )

if isMatchAttrEnabled( 'matchAsPathLength' ):
   RouteMapMode.addCommandClass( MatchAsPathLength )

if isMatchAttrEnabled( 'matchOriginAs' ):
   RouteMapMode.addCommandClass( MatchOriginAsCmd )
   RouteMapMode.addCommandClass( MatchOriginAsValidityCmd )

if isMatchAttrEnabled( 'matchIsisInstance' ):
   RouteMapMode.addCommandClass( MatchIsisInstance )

# match ospf instance
class MatchOspfInstance( CliCommand.CliCommandClass ):
   syntax = "match ospf instance INSTANCE_ID"
   noOrDefaultSyntax = syntax

   data = {
      'match': CommonTokens.match,
      'ospf': CommonTokens.ospf,
      'instance': CommonTokens.instance,
      'INSTANCE_ID': CliMatcher.IntegerMatcher( POLICY_OSPF_INST_MIN,
                                                POLICY_OSPF_INST_MAX,
                              helpdesc='OSPF instance identifier' ),
   }
   handler = "RouteMapCliHandler.handlerMatchOspfInstance"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchOspfInstance' ):
   RouteMapMode.addCommandClass( MatchOspfInstance )
#------------------------------------------------------------------------
# Set commands
#------------------------------------------------------------------------
class SetOriginCmd( CliCommand.CliCommandClass ):
   # set origin egp | igp | incomplete
   syntax = 'set origin ACTION'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'origin': CommonTokens.setOrigin,
      'ACTION': CommonTokens.setOriginEnum,
   }
   handler = "RouteMapCliHandler.handlerSetOriginCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'origin' ):
   RouteMapMode.addCommandClass( SetOriginCmd )

# set origin-validity
class SetOriginAsValidityCmd( CliCommand.CliCommandClass ):
   syntax = "set origin-as validity VALIDITY"
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'origin-as': CommonTokens.originAs,
      'validity': CommonTokens.validity,
      'VALIDITY': CommonTokens.originAsValidityEnum,
   }
   handler = "RouteMapCliHandler.handlerSetOriginAsValidityCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'originAsValidity' ):
   RouteMapMode.addCommandClass( SetOriginAsValidityCmd )

class SetDistanceCmd( CliCommand.CliCommandClass ):
   # set distance
   syntax = 'set distance DIST'
   noOrDefaultSyntax = syntax

   data = {
      'set': CommonTokens.setAction,
      'distance': 'Set protocol independent distance',
      'DIST': CliMatcher.IntegerMatcher(
                  SetActionInfo[ 'distance' ][ 'min' ],
                  SetActionInfo[ 'distance' ][ 'max' ],
                  helpdesc='Protocol independent distance' )
   }
   handler = "RouteMapCliHandler.handlerSetDistanceCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'distance' ):
   RouteMapMode.addCommandClass( SetDistanceCmd )

# as-path prepend
class AsPathPrependCmd( CliCommand.CliCommandClass ):
   syntax = 'set as-path prepend { AS_NUM } [ repeat REPS ]'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'as-path': CommonTokens.asPathPrependReplace,
      'prepend': CommonTokens.prepend,
      'AS_NUM': AsNumWithAutoCliExpr,
      'repeat': CommonTokens.repeat,
      'REPS': CommonTokens.asRepeat
   }
   handler = "RouteMapCliHandler.handlerAsPathPrependCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'asPathPrepend' ):
   RouteMapMode.addCommandClass( AsPathPrependCmd )

class AsPathReplaceCmd( CliCommand.CliCommandClass ):
   syntax = '''set as-path match all replacement
               ( none | ( { AS_NUM } [ repeat REPS ] ) )'''
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'as-path': CommonTokens.asPathPrependReplace,
      'match': CommonTokens.matchAsPath,
      'all': CommonTokens.matchAll,
      'replacement': CommonTokens.replacement,
      'none': CommonTokens.none,
      'AS_NUM': AsReplaceWithAutoCliExpr,
      'repeat': CommonTokens.repeat,
      'REPS': CommonTokens.asRepeat
   }
   handler = "RouteMapCliHandler.handlerAsPathReplaceCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'asPathReplace' ):
   RouteMapMode.addCommandClass( AsPathReplaceCmd )

# set as-path prepend last-as
class LastAsCmd( CliCommand.CliCommandClass ):
   syntax = 'set as-path prepend last-as REPS'
   noOrDefaultSyntax = syntax
   data = {
         'set': CommonTokens.setAction,
         'as-path': CommonTokens.asPathPrependReplace,
         'prepend': CommonTokens.prepend,
         'last-as': CommonTokens.lastAs,
         'REPS': CommonTokens.asRepeatLastAs
         }
   handler = "RouteMapCliHandler.handlerLastAsCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'asPathPrependLastAs' ):
   RouteMapMode.addCommandClass( LastAsCmd )

class SetIpNextHopCmd( CliCommand.CliCommandClass ):
   syntax = 'set ip next-hop NEXTHOP'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ip': CommonTokens.setIp,
      'next-hop': CommonTokens.nextHop,
      'NEXTHOP': IpAddrMatcher.ipAddrMatcher,
   }
   handler = "RouteMapCliHandler.handlerSetIpNextHopCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'nextHop' ):
   RouteMapMode.addCommandClass( SetIpNextHopCmd )

class SetIpv6NextHopCmd( CliCommand.CliCommandClass ):
   syntax = 'set ipv6 next-hop NEXTHOP'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ipv6': CommonTokens.setIpv6,
      'next-hop': CommonTokens.nextHop,
      'NEXTHOP': Ip6AddrMatcher.ip6AddrMatcher,
   }
   handler = "RouteMapCliHandler.handlerSetIpv6NextHopCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'ipv6NextHop' ):
   RouteMapMode.addCommandClass( SetIpv6NextHopCmd )

class SetIpNextHopPeerAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'set ip next-hop peer-address'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ip': CommonTokens.setIp,
      'next-hop': CommonTokens.nextHop,
      'peer-address': CommonTokens.nextHopPeerAddr,
   }
   handler = "RouteMapCliHandler.handlerSetIpNextHopPeerAddrCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'nextHopPeerAddr' ):
   RouteMapMode.addCommandClass( SetIpNextHopPeerAddrCmd )

class SetIpv6NextHopPeerAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'set ipv6 next-hop peer-address'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ipv6': CommonTokens.setIpv6,
      'next-hop': CommonTokens.nextHop,
      'peer-address': CommonTokens.nextHopPeerAddr,
   }
   handler = "RouteMapCliHandler.handlerSetIpv6NextHopPeerAddrCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'ipv6NextHopPeerAddr' ):
   RouteMapMode.addCommandClass( SetIpv6NextHopPeerAddrCmd )

class SetIpNextHopUnchangedCmd( CliCommand.CliCommandClass ):
   syntax = 'set ip next-hop unchanged'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ip': CommonTokens.setIp,
      'next-hop': CommonTokens.nextHop,
      'unchanged': CommonTokens.nextHopUnchanged,
   }
   handler = "RouteMapCliHandler.handlerSetIpNextHopUnchangedCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'nextHopUnchanged' ):
   RouteMapMode.addCommandClass( SetIpNextHopUnchangedCmd )

class SetIpv6NextHopUnchangedCmd( CliCommand.CliCommandClass ):
   syntax = 'set ipv6 next-hop unchanged'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ipv6': CommonTokens.setIpv6,
      'next-hop': CommonTokens.nextHop,
      'unchanged': CommonTokens.nextHopUnchanged,
   }
   handler = "RouteMapCliHandler.handlerSetIpv6NextHopUnchangedCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'ipv6NextHopUnchanged' ):
   RouteMapMode.addCommandClass( SetIpv6NextHopUnchangedCmd )

class SetEvpnNextHopUnchangedCmd( CliCommand.CliCommandClass ):
   syntax = 'set evpn next-hop unchanged'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'evpn': CommonTokens.setEvpn,
      'next-hop': CommonTokens.nextHop,
      'unchanged': CommonTokens.nextHopUnchanged,
   }
   handler = "RouteMapCliHandler.handlerSetEvpnNextHopUnchangedCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'evpnNextHopUnchanged' ):
   RouteMapMode.addCommandClass( SetEvpnNextHopUnchangedCmd )

class SetOspfBitDnCmd( CliCommand.CliCommandClass ):
   syntax = 'set ospf bit dn'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ospf': CommonTokens.setOspf,
      'bit': CommonTokens.ospfBit,
      'dn': CommonTokens.bitDn,
   }
   handler = "RouteMapCliHandler.handlerSetOspfBitDnCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'ospfBitDn' ):
   RouteMapMode.addCommandClass( SetOspfBitDnCmd )

class SetOspfv3BitDnCmd( CliCommand.CliCommandClass ):
   syntax = 'set ospfv3 bit dn'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'ospfv3': CommonTokens.setOspfv3,
      'bit': CommonTokens.ospfv3Bit,
      'dn': CommonTokens.bitDn,
   }
   handler = "RouteMapCliHandler.handlerSetOspfv3BitDnCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'ospfv3BitDn' ):
   RouteMapMode.addCommandClass( SetOspfv3BitDnCmd )

class SetResolutionRibProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'set next-hop resolution ribs ( RIBS | system-default )'
   noOrDefaultSyntax = 'set next-hop resolution ribs ...'
   data = {
      'set': CommonTokens.setAction,
      'next-hop': RouteMapCommonTokens.nextHop,
      'resolution': CliToken.IpRibLibCliTokens.matcherResolution,
      'ribs': CliToken.IpRibLibCliTokens.matcherRibs,
      'RIBS': ResolutionRibsExprFactory(),
      'system-default': CliToken.IpRibLibCliTokens.matcherSystemDefault,
   }
   handler = "RouteMapCliHandler.handlerSetResolutionRibProfileCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'resolutionRibProfileConfig' ):
   RouteMapMode.addCommandClass( SetResolutionRibProfileCmd )

class SetLocalPrefCmd( CliCommand.CliCommandClass ):
   # set local-preference
   syntax = 'set local-preference BGP_LOCAL_PREF'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'local-preference': 'BGP local preference',
      'BGP_LOCAL_PREF': CliMatcher.IntegerMatcher(
                            MatchAttributes[ 'matchLocalPref' ][ 'min' ],
                            MatchAttributes[ 'matchLocalPref' ][ 'max' ],
                            helpdesc='BGP local preference' )
   }
   handler = "RouteMapCliHandler.handlerSetLocalPrefCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'localPref' ):
   RouteMapMode.addCommandClass( SetLocalPrefCmd )

# set weight
weightMatcher = CliMatcher.IntegerMatcher( 0, 65535,
   helpdesc='BGP weight' )

class SetWeightCmd( CliCommand.CliCommandClass ):
   syntax = 'set weight WEIGHT'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'weight': 'BGP weight',
      'WEIGHT': weightMatcher,
      }
   handler = "RouteMapCliHandler.handlerSetWeightCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'weight' ):
   RouteMapMode.addCommandClass( SetWeightCmd )

# set bgp bestpath as-path weight
bgpBestPathWeight = CliMatcher.IntegerMatcher( 1, 65535,
   helpdesc='BGP AS path multipath weight' )

class SetBgpBestpathAsPathWeight( CliCommand.CliCommandClass ):
   syntax = 'set bgp bestpath as-path weight WEIGHT'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'bgp': 'Border Gateway Protocol',
      'bestpath': 'BGP bestpath',
      'as-path': CommonTokens.asPathFilter,
      'weight': 'BGP bestpath AS path weight',
      'WEIGHT': bgpBestPathWeight,
      }
   handler = "RouteMapCliHandler.handlerSetBgpBestpathAsPathWeight"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'bpAsPathWeight' ):
   RouteMapMode.addCommandClass( SetBgpBestpathAsPathWeight )

def setMetricMultiplierGuard( mode, token ):
   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent:
      return None
   return 'only supported in multi-agent mode'

tokenMetricValue = CliMatcher.IntegerMatcher( 0, 2**32 - 1,
                                              helpdesc='Set metric value' )

class SetMetricExpression( CliCommand.CliExpression ):
   expression = ( 'METRIC | ADD | SUBTRACT | MULTIPLY | '
                  'igp-metric | igp-nexthop-cost | '
                  '+igp-metric | +igp-nexthop-cost | '
                  '( METRIC +igp-metric ) | ( METRIC +igp-nexthop-cost )' )
   data = {
      'METRIC': tokenMetricValue,
      'ADD': MetricModifierAndNumberMatcher(
                  matcher=CliMatcher.IntegerMatcher( 0, 2**32 - 1 ),
                  helpname='+<0-4294967295>',
                  helpdesc='Set additive metric value',
                  modifier='+' ),
      'SUBTRACT': MetricModifierAndNumberMatcher(
                        matcher=CliMatcher.IntegerMatcher( 0, 2**32 - 1 ),
                        helpname='-<0-4294967295>',
                        helpdesc='Set subtractive metric value',
                        modifier='-' ),
      'MULTIPLY': CliCommand.Node( MetricModifierAndFloatMatcher(
                           matcher=InputRestrictedFloatMatcher( 0, 2**16,
                                                   precisionString='%.2f',
                                                   maxPrecision=4 ),
                           helpname='*<0.0000-65536.0000>',
                           helpdesc='Set multiplicative metric value',
                           modifier='*' ),
                        guard=setMetricMultiplierGuard
                  ),
      'igp-metric': 'IGP metric',
      'igp-nexthop-cost': 'IGP nexthop cost',
      '+igp-metric': 'Add IGP metric',
      '+igp-nexthop-cost': 'Add IGP nexthop cost'
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'METRIC' in args:
         if '+igp-metric' in args:
            args[ 'METRIC' ] = tacMetricValue( args[ 'METRIC' ],
                                    routeMapMedType.medValueAddIgpMetric )
         elif '+igp-nexthop-cost' in args:
            args[ 'METRIC' ] = tacMetricValue( args[ 'METRIC' ],
                                    routeMapMedType.medValueAddIgpNexthopCost )
         else:
            args[ 'METRIC' ] = tacMetricValue( args[ 'METRIC' ],
                                               routeMapMedType.medNormal )
      elif 'ADD' in args:
         args[ 'METRIC' ] = tacMetricValue( args[ 'ADD' ],
                                          routeMapMedType.medAdditive )
      elif 'SUBTRACT' in args:
         args[ 'METRIC' ] = tacMetricValue( args[ 'SUBTRACT' ],
                                          routeMapMedType.medSubtractive )
      elif 'MULTIPLY' in args:
         args[ 'MULTIPLY' ] = int( round( args[ 'MULTIPLY' ] * 10000 ) )
         args[ 'METRIC' ] = tacMetricValue( args[ 'MULTIPLY' ],
                                       routeMapMedType.medMultiplicative )
      elif 'igp-metric' in args:
         args[ 'METRIC' ] = tacMetricValue( 0, routeMapMedType.medIgpMetric )
      elif 'igp-nexthop-cost' in args:
         args[ 'METRIC' ] = tacMetricValue( 0, routeMapMedType.medIgpNexthopCost )
      elif '+igp-metric' in args:
         args[ 'METRIC' ] = tacMetricValue( 0, routeMapMedType.medAddIgpMetric )
      elif '+igp-nexthop-cost' in args:
         args[ 'METRIC' ] = tacMetricValue( 0, routeMapMedType.medAddIgpNexthopCost )

# set metric
class SetMetric( CliCommand.CliCommandClass ):
   syntax = ( 'set metric METRIC [ source openconfig YANG_SOURCE ]' )
   noOrDefaultSyntax = 'set metric ...'
   data = {
      'set': CommonTokens.setAction,
      'metric': 'Route metric',
      'METRIC': SetMetricExpression,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source', '' ), hidden=True ),
      'openconfig': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'openconfig', '' ), hidden=True ),
      'YANG_SOURCE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( { 'bgp-actions': '',
                                           'ospf-actions': '',
                                           'isis-actions': '' } ),
         hidden=True ),
   }
   handler = "RouteMapCliHandler.handlerSetMetric"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'metric' ):
   RouteMapMode.addCommandClass( SetMetric )

# set metric-type
class SetMetricType( CliCommand.CliCommandClass ):
   syntax = ( 'set metric-type TYPE' )
   noOrDefaultSyntax = 'set metric-type ...'
   data = {
      'set': CommonTokens.setAction,
      'metric-type': 'Route metric type',
      'TYPE': CliMatcher.EnumMatcher( {
         'type-1': 'OSPF type 1',
         'type-2': 'OSPF type 2'
      } )
   }
   handler = "RouteMapCliHandler.handlerSetMetricType"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'metricType' ):
   RouteMapMode.addCommandClass( SetMetricType )

aigpMetricRangeMatcher = CliMatcher.IntegerMatcher( 0, U64_MAX_VALUE - 1,
                                                    helpdesc='AIGP metric value' )
aigpMetricIgpEnumMatcher = CliMatcher.EnumMatcher( {
   'metric': 'IGP metric',
   'next-hop-cost': 'IGP next-hop-cost',
   } )

class SetAigpMetricExpression( CliCommand.CliExpression ):
   expression = ( 'AIGP_METRIC | ADD | SUBTRACT ' )
   data = {
      'AIGP_METRIC': aigpMetricRangeMatcher,
      'ADD': MetricModifierAndNumberMatcher(
                  matcher=CliMatcher.IntegerMatcher( 0, U64_MAX_VALUE - 1 ),
                  helpname='+<0-18446744073709551614>',
                  helpdesc='Set additive aigp-metric value',
                  modifier='+' ),
      'SUBTRACT': MetricModifierAndNumberMatcher(
                        matcher=CliMatcher.IntegerMatcher( 0, U64_MAX_VALUE - 1 ),
                        helpname='-<0-18446744073709551614>',
                        helpdesc='Set subtractive aigp-metric value',
                        modifier='-' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AIGP_METRIC' in args:
         args[ 'AIGP_METRIC' ] = tacAigpMetricValue( args[ 'AIGP_METRIC' ],
                                    routeMapAigpMetricSetType.aigpMetricSetNormal )
      elif 'ADD' in args:
         args[ 'AIGP_METRIC' ] = tacAigpMetricValue( args[ 'ADD' ],
                                    routeMapAigpMetricSetType.aigpMetricSetAdditive )
      elif 'SUBTRACT' in args:
         args[ 'AIGP_METRIC' ] = tacAigpMetricValue( args[ 'SUBTRACT' ],
                                    routeMapAigpMetricSetType
                                       .aigpMetricSetSubtractive )

class SetNormalAigpMetricExpression( CliCommand.CliExpression ):
   expression = ( 'AIGP_METRIC ' )
   data = {
      'AIGP_METRIC': aigpMetricRangeMatcher
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AIGP_METRIC' in args:
         args[ 'AIGP_METRIC' ] = tacAigpMetricValue( args[ 'AIGP_METRIC' ],
                                    routeMapAigpMetricSetType.aigpMetricSetNormal )

class SetAigpMetricCmd( CliCommand.CliCommandClass ):
   syntax = ( 'set aigp-metric ( AIGP_METRIC | '
              '( igp IGP_SET_OPTION ) )' )
   noOrDefaultSyntax = 'set aigp-metric ...'
   data = {
      'set': CommonTokens.setAction,
      'aigp-metric': 'AIGP metric for the route',
      'AIGP_METRIC': SetAigpMetricExpression
                     if RouteMapToggleLib.toggleSetAigpMetricAddSubEnabled()
                     else SetNormalAigpMetricExpression,
      'igp': 'Interior Gateway Protocol',
      'IGP_SET_OPTION': aigpMetricIgpEnumMatcher,
   }
   handler = "RouteMapCliHandler.handlerSetAigpMetricCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'aigpMetric' ):
   RouteMapMode.addCommandClass( SetAigpMetricCmd )

class SetIsisLevelCmd( CliCommand.CliCommandClass ):
   # set isis level
   syntax = 'set isis level ISIS_LVL'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'isis': CommonTokens.setIsis,
      'level': 'Route level',
      'ISIS_LVL': CliMatcher.EnumMatcher( {
                      'level-1': 'IS-IS level 1',
                      'level-2': 'IS-IS level 2',
                      'level-1-2': 'IS-IS level 1 and level 2',
                   } )
   }
   handler = "RouteMapCliHandler.handlerSetIsisLevelCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'level' ):
   RouteMapMode.addCommandClass( SetIsisLevelCmd )

# set isis metric style
class SetIsisMetricStyleCmd( CliCommand.CliCommandClass ):
   syntax = 'set isis metric style wide'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'isis': CommonTokens.setIsis,
      # the rest of the command is to be used only in OC context, hence it's hidden
      'metric': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                                         'metric',
                                                         helpdesc="IS-IS metric" ),
                                                         hidden=True ),
      'style': 'metric style',
      'wide': CliMatcher.EnumMatcher( { 'wide': 'WIDE_METRIC style', } )
   }
   handler = "RouteMapCliHandler.handlerSetIsisMetricStyleCmd"
   noOrDefaultHandler = handler

RouteMapMode.addCommandClass( SetIsisMetricStyleCmd )

# set tag
class SetTagCmd( CliCommand.CliCommandClass ):
   syntax = "set tag ( auto | VALUE )"
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'tag': CommonTokens.tagToken,
      'auto': 'Use remote peer AS number as tag',
      'VALUE': CommonTokens.routeTagRange,
   }
   handler = "RouteMapCliHandler.handlerSetTagCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'tag' ):
   RouteMapMode.addCommandClass( SetTagCmd )

class SetCommunityNoneCmd( CliCommand.CliCommandClass ):
   syntax = 'set community none'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'none': CommonTokens.communityNone,
   }
   handler = "RouteMapCliHandler.handlerSetCommunityNoneCmd"
   noOrDefaultHandler = "RouteMapCliHandler.noOrDefaultHandlerSetCommunityNoneCmd"

RouteMapMode.addCommandClass( SetCommunityNoneCmd )

class NoSetCommunityCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'set community [ ACTION ]'
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'ACTION': CommonTokens.setAdditiveDelete,
   }
   noOrDefaultHandler = "RouteMapCliHandler.noOrDefaultHandlerNoSetCommunityCmd"

RouteMapMode.addCommandClass( NoSetCommunityCmd )

# [no] set community community-list LIST1 [LIST2, ...] [action]
class SetCommunityListCmd( CliCommand.CliCommandClass ):
   syntax = 'set community community-list { LIST } [ ACTION ]'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'community-list': 'Community-list',
      'LIST': CommonTokens.communityListNameMatcher,
      'ACTION': CommonTokens.setAdditiveDelete
   }
   handler = "RouteMapCliHandler.handlerSetCommunityListCmd"
   noOrDefaultHandler = handler

RouteMapMode.addCommandClass( SetCommunityListCmd )

# [no] set community community-list LIST1 [LIST2, ...] filter all
class SetCommunityListFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'set community community-list { LIST } filter all'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'community-list': 'Community-list',
      'LIST': CommonTokens.communityListNameMatcher,
      'filter': CommonTokens.communityListFilter,
      'all': CommonTokens.communityListFilterAll,
   }
   handler = "RouteMapCliHandler.handlerSetCommunityListFilterCmd"
   noOrDefaultHandler = handler

RouteMapMode.addCommandClass( SetCommunityListFilterCmd )

# [no] set community comm-const [comm-const, ...] [action]
class SetCommunityCmd( CliCommand.CliCommandClass ):
   syntax = 'set community { COMM } [ ACTION ]'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'COMM': CommunityConstExpr,
      'ACTION': CommonTokens.setAdditiveDelete,
   }
   handler = "RouteMapCliHandler.handlerSetCommunityCmd"
   noOrDefaultHandler = handler

RouteMapMode.addCommandClass( SetCommunityCmd )

# set community remove comm-const [comm-const, ...] [action]
class SetCommunityRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'set community remove { COMM } [ ACTION ]'
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'remove': 'Remove specified communities from route-map',
      'COMM': CommunityConstExpr,
      'ACTION': CommonTokens.setAdditiveDelete,
   }
   handler = "RouteMapCliHandler.handlerSetCommunityRemoveCmd"

RouteMapMode.addCommandClass( SetCommunityRemoveCmd )

# set community remove community-list LIST1 [LIST2, ...] [action]
class SetCommunityListRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'set community remove community-list {LIST} [ ACTION ]'
   data = {
      'set': CommonTokens.setAction,
      'community': CommonTokens.community,
      'remove': 'Remove specified communities from route-map',
      'community-list': 'Community-list',
      'LIST': CommonTokens.communityListNameMatcher,
      'ACTION': CommonTokens.setAdditiveDelete,
   }
   handler = "RouteMapCliHandler.handlerSetCommunityListRemoveCmd"

RouteMapMode.addCommandClass( SetCommunityListRemoveCmd )

# set large-communities to send
class LargeCommCliMatcher( CliMatcher.Matcher ):
   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )
      self.largeCommRe_ = re.compile(
          r'(^(\d+)(l|L)?:(\d+):(\d+)$)|'
           r'(^(\d+\.\d+):(\d+):(\d+)$)' )
      self.largeCommCompletionRe_ = re.compile(
          r'(^(\d+?\.?\d+?):?(\d+)?:?(\d+)?$)|'
           r'(^(\d+)?(l|L)?:?(\d+)?:?(\d+)?$)' )

   def match( self, mode, context, token ):
      # must be aa:nn:nn where aa:nn:nn is one of
      #    ASN(32-bit):nn(32-bit):nn(32-bit) or
      #    ASDOT(32-bit):nn(32-bit):nn(32-bit)
      m = self.largeCommRe_.match( token )
      if m is None:
         return CliParserCommon.noMatch
      validLargeComm = []
      for group in m.groups():
         if group is not None and ":" not in group:
            validLargeComm.append( group )

      # check if optional ASN qualifier 'L' is present
      if ( len( validLargeComm ) == 4 ) and ( validLargeComm[ 1 ].upper() == 'L' ):
         validLargeComm = validLargeComm[ : 1 ] + validLargeComm[ 2 : ]

      if len( validLargeComm ) == 3:
         asn = asnStrToNum( validLargeComm[ 0 ], minValue=0 )
         local1 = safeInt( validLargeComm[ 1 ] )
         local2 = safeInt( validLargeComm[ 2 ] )
         if ( asn is None or asn > u32Max or
              local1 is None or local1 < 0 or local1 > u32Max or
              local2 is None or local2 < 0 or local2 > u32Max ):
            return CliParserCommon.noMatch

      # don't support anything else
      else:
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.largeCommCompletionRe_.match( token )
      if m is None:
         return []
      currElement = 0
      asn = None
      local1 = None
      local2 = None

      # 2 formats acceptable: ASDOT:nn:nn, ASN:nn:nn
      # Also 2 potential encodings for ASN:nn:nn depending on the values
      # iterate through groups checking against reg-exp positions
      # [in] [asdot] [local1] [local2] [in] [asn] [L] [local1] [local2]
      #  0      1       2        3      4     5    6     7        8
      for group in m.groups( '' ):
         if group == '' or ":" in group:
            currElement += 1
            continue

         if currElement == 1:
            # ASDOT value
            asn = asnStrToNum( group, minValue=0 )
            if asn is None:
               return []
         elif currElement == 5:
            # ASN integer value
            if ( not group.isdigit() ): # pylint: disable=superfluous-parens
               return []
            asn = safeInt( group )
         elif currElement == 2 or currElement == 7:
            local1 = safeInt( group )
         elif currElement == 3 or currElement == 8:
            local2 = safeInt( group )
         currElement += 1

      # post-process checks on ASN value to ensure ok
      if ( asn is not None ) and ( local1 is not None ) and ( local2 is not None ):
         if ( ( asn > u32Max ) or
              ( local1 < 0 ) or ( local1 > u32Max ) or
              ( local2 < 0 ) or ( local2 > u32Max ) ):
            return []

      return [ CliParser.Completion( 'ASN:nn:nn', self.helpdesc_, False ) ]

   def __str__( self ):
      return 'ASN:nn:nn'

# [no] set large-community comm-const [comm-const, ...] [action]
# no set large-community [action]
# [no] set large-community none
class SetLargeCommunityCmd( CliCommand.CliCommandClass ):
   syntax = 'set large-community ( ( { COMM } [ ACTION ] ) | none )'
   noOrDefaultSyntax = 'set large-community [ ( [ { COMM } ] [ ACTION ] ) | none ]'
   data = {
      'set': CommonTokens.setAction,
      'large-community': CommonTokens.setLargeCommunity,
      'COMM': LargeCommCliMatcher( showLargeCommunityDesc ),
      'ACTION': CommonTokens.setCommAdditiveDelete,
      'none': CommonTokens.communityNone,
   }
   handler = "RouteMapCliHandler.handlerSetLargeCommunityCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'largeCommunity' ):
   RouteMapMode.addCommandClass( SetLargeCommunityCmd )
# [no] set large-community large-community-list LIST1 [LIST2, ...] [action]
class SetLargeCommunityListCmd( CliCommand.CliCommandClass ):
   syntax = 'set large-community large-community-list { LIST } [ ACTION ]'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'large-community': CommonTokens.setLargeCommunity,
      'large-community-list': CommonTokens.largeCommunityList,
      'LIST': CommonTokens.largeCommunityListNameMatcher,
      'ACTION': CommonTokens.setAdditiveDelete,
   }
   handler = "RouteMapCliHandler.handlerSetLargeCommunityListCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'largeCommunity' ):
   RouteMapMode.addCommandClass( SetLargeCommunityListCmd )

# [no] set large-community large-community-list LIST1 [LIST2, ...] filter all
class SetLargeCommunityListFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'set large-community large-community-list { LIST } filter all'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'large-community': CommonTokens.setLargeCommunity,
      'large-community-list': CommonTokens.largeCommunityList,
      'LIST': CommonTokens.largeCommunityListNameMatcher,
      'filter': CommonTokens.communityListFilter,
      'all': CommonTokens.communityListFilterAll
   }
   handler = "RouteMapCliHandler.handlerSetLargeCommunityListFilterCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'largeCommunity' ):
   RouteMapMode.addCommandClass( SetLargeCommunityListFilterCmd )

# set extended communities to send
if isSetActionEnabled( 'extCommunity' ):
   class SetExtCommunityExpr( CliCommand.CliExpression ):
      expression = 'set extcommunity'
      data = {
         'set': CommonTokens.setAction,
         'extcommunity': CommonTokens.setExtCommunity
      }

   class SetExtCommunityNoneCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity none
      syntax = 'SET_EXTCOMM none'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'none': CommonTokens.communityNone
      }
      handler = "RouteMapCliHandler.handlerSetExtCommunityNoneCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityNoneCmd )

   class NoSetExtCommunityCmd( CliCommand.CliCommandClass ):
      # no set extcommunity [action]
      noOrDefaultSyntax = 'SET_EXTCOMM [ ACTION ]'
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'ACTION': CommonTokens.setCommAdditiveDelete
      }
      noOrDefaultHandler = \
         "RouteMapCliHandler.noOrDefaultHandlerNoSetExtCommunityCmd"

   RouteMapMode.addCommandClass( NoSetExtCommunityCmd )

   class SetExtCommunityRtSooCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity comm-const [comm-const, ...] [action]
      syntax = 'SET_EXTCOMM { ( rt RT_VAL ) | ( soo SOO_VAL ) } [ ACTION ]'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'rt': CommonTokens.extCommunityRt,
         'RT_VAL': RtSooExtCommCliMatcher( "VPN extended community" ),
         'soo': CommonTokens.extCommunitySoo,
         'SOO_VAL': RtSooExtCommCliMatcher( "VPN extended community" ),
         'ACTION': CommonTokens.setCommAdditiveDelete
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         rtVals = args.pop( 'RT_VAL', [] )
         args[ 'RT_VAL' ] = [ 'rt ' + rt for rt in rtVals ]
         sooVals = args.pop( 'SOO_VAL', [] )
         args[ 'SOO_VAL' ] = [ 'soo ' + soo for soo in sooVals ]

      handler = "RouteMapCliHandler.handlerSetExtCommunityRtSooCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityRtSooCmd )

   class SetExtCommunityLbwDelCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity lbw AS:bits/second delete
      # [no] set extcommunity lbw delete ( deletes any lbw extcomm )
      # [no] set extcommunity lbw asn AS delete ( deletes lbw extcomm with given AS )
      # NOTE: 'delete' is required for this format of the command
      syntax = 'SET_EXTCOMM ( lbw [ LBW_VAL | ( asn AS_NUM ) ] ) delete'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'lbw': CommonTokens.extCommunityLbw,
         'LBW_VAL': LinkBandwidthFormatCliMatcher(
            'Link Bandwidth extended community as AS:bits/second' ),
         'asn' : 'BGP AS number',
         'AS_NUM': CliMatcher.IntegerMatcher( AS_PATH_MIN, U16_MAX_VALUE,
                                              helpdesc='Autonomous system number' ),
         'delete': CommonTokens.communityDelete
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         # e.g. "set extcommunity lbw 10:100G delete"
         if 'LBW_VAL' in args:
            args[ 'LBW_VAL' ] = 'lbw ' + args[ 'LBW_VAL' ]
         # e.g. "set extcommunity lbw asn 10 delete"
         elif 'AS_NUM' in args:
            args[ 'LBW_VAL' ] = 'lbw asn ' + str( args[ 'AS_NUM' ] )
         # "set extcommunity lbw delete"
         else:
            args[ 'LBW_VAL' ] = 'lbw'

      handler = "RouteMapCliHandler.handlerSetExtCommunityLbwDelCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityLbwDelCmd )

   class SetExtCommunityLbwCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity lbw bits/second
      syntax = 'SET_EXTCOMM lbw LBW_VAL'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'lbw': CommonTokens.extCommunityLbw,
         'LBW_VAL': LinkBandwidthValueCliMatcher(
            'Link Bandwidth extended community value as bits/second' ),
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         args[ 'LBW_VAL' ] = 'lbw ' + args[ 'LBW_VAL' ]

      handler = "RouteMapCliHandler.handlerSetExtCommunityLbwCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityLbwCmd )

   class SetExtCommunityLbwDivideCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity lbw bits/second
      syntax = 'SET_EXTCOMM lbw divide OPTION'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'lbw': CommonTokens.extCommunityLbw,
         'divide': CommonTokens.extCommunityLbwDivide,
         'OPTION': CommonTokens.extCommLbwDivideEqualRatio
      }

      handler = "RouteMapCliHandler.handlerSetExtCommunityLbwDivideCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityLbwDivideCmd )

   class SetExtCommunityLbwAggregateCmd( CliCommand.CliCommandClass ):
      # [no] set extcommunity lbw bits/second
      syntax = 'SET_EXTCOMM lbw aggregate [ LBW_VAL ]'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'lbw': CommonTokens.extCommunityLbw,
         'aggregate': CommonTokens.extCommunityLbwAggregate,
         'LBW_VAL': LinkBandwidthValueCliMatcher(
            'Link Bandwidth extended community value as bits/second' ),
      }
      handler = "RouteMapCliHandler.handlerSetExtCommunityLbwAggregateCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityLbwAggregateCmd )

   class SetExtCommunityColor( CliCommand.CliCommandClass ):
      syntax = 'set extcommunity { COLOR_EXP } [ ACTION_EXP ]'
      noOrDefaultSyntax = syntax
      data = {
         'set': 'Set route attribute',
         'extcommunity': 'BGP extended community attribute',
         'COLOR_EXP': ColorExtCommunityExpression,
         'ACTION_EXP': ExtCommActionExpression
      }
      handler = "RouteMapCliHandler.setExtCommColorHandlerCommon"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityColor )

   # [no] set extcommunity extcommunity-list LIST1 [LIST2, ...] [action]
   class SetExtCommunityListCmd( CliCommand.CliCommandClass ):
      syntax = 'set extcommunity extcommunity-list { LIST } [ ACTION ]'
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'extcommunity-list': CommonTokens.extCommunityList,
         'LIST': CommonTokens.extCommunityListNameMatcher,
         'ACTION': CommonTokens.setCommAdditiveDelete
      }
      handler = "RouteMapCliHandler.handlerSetExtCommunityListCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityListCmd )

   # [no] set extcommunity extcommunity-list LIST1 [LIST2, ...]
   # filter ( all | COMM_TYPES )
   class SetExtCommunityListFilterCmd( CliCommand.CliCommandClass ):
      syntax = '''set extcommunity extcommunity-list { LIST }
                  filter ( all | COMM_TYPES )'''
      noOrDefaultSyntax = syntax
      data = {
         'SET_EXTCOMM': SetExtCommunityExpr,
         'extcommunity-list': CommonTokens.extCommunityList,
         'LIST': CommonTokens.extCommunityListNameMatcher,
         'filter': CommonTokens.communityListFilter,
         'all': CommonTokens.communityListFilterAll,
         'COMM_TYPES': CliCommand.SetEnumMatcher( {
            'rt': 'Filter Route Target community types',
            'soo': 'Filter Site-Of-Origin community types',
            'lbw': 'Filter Link Bandwidth community types',
            'color': 'Filter Color community types',
         } )
      }
      handler = "RouteMapCliHandler.handlerSetExtCommunityListFilterCmd"
      noOrDefaultHandler = handler

   RouteMapMode.addCommandClass( SetExtCommunityListFilterCmd )

# set segment-index
indexMatcher = CliMatcher.IntegerMatcher(
   0,
   0xFFFFFFFF,
   helpdesc='Segment Index' )
segmentIndex = CliMatcher.KeywordMatcher( 'segment-index',
   helpdesc='MPLS Segment-routing Segment Index' )

class SetSegmentIndexCmd( CliCommand.CliCommandClass ):
   syntax = 'set segment-index INDEX'
   noOrDefaultSyntax = syntax
   data = {
      'set': CommonTokens.setAction,
      'segment-index': segmentIndex,
      'INDEX': indexMatcher,
      }
   handler = "RouteMapCliHandler.handlerSetSegmentIndexCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'segmentIndex' ):
   RouteMapMode.addCommandClass( SetSegmentIndexCmd )

# continue
continueMatcher = CliMatcher.KeywordMatcher( 'continue',
   helpdesc='Continue to a different entry within the route map' )
sequenceMatcher = CliMatcher.IntegerMatcher(
   seqnoMin,
   seqnoMax,
   helpdesc='Route map entry sequence number' )

class ContinueSeqCmd( CliCommand.CliCommandClass ):
   syntax = 'continue [ SEQUENCE ]'
   noOrDefaultSyntax = syntax
   data = {
      'continue': continueMatcher,
      'SEQUENCE': sequenceMatcher,
      }
   handler = "RouteMapCliHandler.handlerContinueSeqCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'continueSeq' ):
   RouteMapMode.addCommandClass( ContinueSeqCmd )

# set igp-metric
metricValue = CliMatcher.IntegerMatcher(
   1,
   U32_MAX_VALUE,
   helpdesc='Set IGP metric value' )
maxMetricValue = CliMatcher.KeywordMatcher( 'max-metric',
   'Set max IGP metric value' )

class SetIgpMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'set next-hop igp-metric ( max-metric | VALUE )'
   noOrDefaultSyntax = syntax
   data = {
         'set': CommonTokens.setAction,
         'next-hop': RouteMapCommonTokens.nextHop,
         'igp-metric': 'IGP metric',
         'max-metric': maxMetricValue,
         'VALUE': metricValue,
         }
   handler = "RouteMapCliHandler.handlerSetIgpMetricCmd"
   noOrDefaultHandler = handler

if isSetActionEnabled( 'igpMetric' ):
   RouteMapMode.addCommandClass( SetIgpMetricCmd )

#-------------------------------------------------------------------------
# AS path regex mode
#
# [no] ip as-path regex-mode { asn | character }
#-------------------------------------------------------------------------
asPathMatchType = CliMatcher.EnumMatcher( {
   'asn': 'Match AS path as AS numbers',
   'string': 'Match AS path as a string'
} )
asPathRegexMode = CliMatcher.KeywordMatcher( 'regex-mode',
                                             helpdesc='Regex Mode' )

class AsPathRegexMode( CliCommand.CliCommandClass ):
   syntax = 'ip as-path regex-mode TYPE'
   noOrDefaultSyntax = 'ip as-path regex-mode ...'
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'as-path': CommonTokens.asPathFilter,
         'regex-mode': asPathRegexMode,
         'TYPE': asPathMatchType
         }

   handler = "RouteMapCliHandler.setRegexMode"
   noOrDefaultHandler = "RouteMapCliHandler.noRegexMode"

BasicCli.GlobalConfigMode.addCommandClass( AsPathRegexMode )

#-------------------------------------------------------------------------
# AS path list
#
# [no] ip as-path access-list <name> <permit|deny> <regex> [<origin>]
#-------------------------------------------------------------------------
# NOTE: when adding new non-digit symbols into the below
#       asPathRule PatternRule, please don't forget to add the symbol
#       into the char *delim string inside str_asdot_to_u32() function
#       in file gated/gated-ctk/src/aspath/routemap_regex_helper.c

asPathAccessListUrlMatcher = Url.UrlMatcher(
      lambda fs: fs.scheme in listUrlSchemes,
      "URL to load as-path list entries" )
asPathACLSource = CliMatcher.KeywordMatcher(
      'source', helpdesc='Source URL of as-path access-list' )

asPathDuplicate = CliMatcher.KeywordMatcher(
      'duplicate', helpdesc='Duplicate handling option' )

duplicateHandlingOption = CliMatcher.EnumMatcher( {
   'ignore': 'Ignore duplicate handling option',
   'override': ' Override duplicate handling option' } )

asPathPermitOrDeny = CliMatcher.EnumMatcher( {
   'permit': 'Specify as-path to accept',
   'deny': 'Specify as-path to reject',
} )
asPathOrigin = CliMatcher.EnumMatcher( {
   'any': "Any BGP origin",
   'egp': "EGP origin",
   'igp': "IGP origin",
   'incomplete': "Incomplete origin",
} )
asPathRegexMatcher = CliMatcher.PatternMatcher(
   pattern=r'.+',
   helpdesc='A regular expression to match '
            'AS paths. Use "ctrl-v" to enter'
            ' literal "?"',
   helpname="WORD" )

class AsPathAclPermitOrDenyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip as-path access-list NAME ACTION REGEX [ ORIGIN ]'
   noOrDefaultSyntax = 'ip as-path access-list NAME ACTION REGEX ...'
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'as-path': CommonTokens.asPathFilter,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName,
         'ACTION': asPathPermitOrDeny,
         'REGEX': asPathRegexMatcher,
         'ORIGIN': asPathOrigin
         }
   handler = "RouteMapCliHandler.asPathListHelper"
   noOrDefaultHandler = "RouteMapCliHandler.asPathEntryRemover"

BasicCli.GlobalConfigMode.addCommandClass( AsPathAclPermitOrDenyCmd )

class RemoveAsPathAclCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip as-path access-list NAME'
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'as-path': CommonTokens.asPathFilter,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName
         }

   noOrDefaultHandler = "RouteMapCliHandler.removeAsPathList"

BasicCli.GlobalConfigMode.addCommandClass( RemoveAsPathAclCmd )


# show asPath list
class ShowIpAsPathList( ShowCliCommandClass ):
   syntax = "show ip as-path access-list [ NAME ]"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'as-path': CommonTokens.asPathList,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName
   }

   cliModel = IpAsPathLists
   handler = "RouteMapCliHandler.showAsPathList"

BasicCli.addShowCommandClass( ShowIpAsPathList )

class ShowIpAsPathListDetail( ShowCliCommandClass ):
   syntax = "show ip as-path access-list NAME detail"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'as-path': CommonTokens.asPathList,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName,
         'detail': 'detail',
   }

   cliModel = IpAsPathLists
   handler = "RouteMapCliHandler.showAsPathAccessListDetail"

BasicCli.addShowCommandClass( ShowIpAsPathListDetail )

class ShowIpAsPathListSummary( ShowCliCommandClass ):
   syntax = "show ip as-path access-list NAME summary"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'as-path': CommonTokens.asPathList,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName,
         'summary': 'summary',
   }

   cliModel = IpAsPathSummary
   handler = "RouteMapCliHandler.showAsPathAccessListSummary"

BasicCli.addShowCommandClass( ShowIpAsPathListSummary )

def getPrefixEntryHashVal( prefix, permit, geLen, leLen ):
   '''
      This method returns the hash-value for a prefix list entry, which
      is used to detect existing entries in a prefix-list config.
   '''
   m = hashlib.md5()
   m.update( str( prefix ).encode() )
   m.update( str( permit ).encode() )
   m.update( str( geLen ).encode() )
   m.update( str( leLen ).encode() )
   return m.hexdigest()

class IpPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = "ip prefix-list NAME [ seq SEQNO ] ACTION PREFIX"\
            "[ ( eq EQLEN ) "\
            "| ( ge GELEN [ le LELEN ] )"\
            "| ( le LELEN [ ge GELEN ] ) ]"
   noOrDefaultSyntax = "ip prefix-list NAME"\
                       "[ ( seq SEQNO ... )"\
                       "| ( ACTION PREFIX [ ( eq EQLEN )"\
                                         "| ( ge GELEN [ le LELEN ] )"\
                                         "| ( le LELEN [ ge GELEN ] ) ] ) ]"
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         'seq': CommonTokens.prefixListSeq,
         'SEQNO': CommonTokens.prefixListSeqnoMatcher,
         'ACTION': CommonTokens.prefixListPermitOrDeny,
         'PREFIX': CommonTokens.prefixListPrefix,
         'ge': CommonTokens.prefixListGe,
         'le': CommonTokens.prefixListLe,
         'eq': CommonTokens.prefixListEq,
         'GELEN': CommonTokens.prefixListMaskLenMatcher,
         'LELEN': CommonTokens.prefixListMaskLenMatcher,
         'EQLEN': CommonTokens.prefixListMaskLenMatcher
         }
   handler = "RouteMapCliHandler.managePrefixListHandler"
   noOrDefaultHandler = "RouteMapCliHandler.removePrefixListEntryHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpPrefixListCmd )

allVrfWithNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllPlusReservedVrfNames, 'VRF name' )

class IpPrefixListLoadURLCmd( CliCommand.CliCommandClass ):
   syntax = "ip prefix-list NAME source URL [ vrf VRFNAME ] [ duplicate ACTION ]"
   noOrDefaultSyntax = syntax
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         'source': CommonTokens.prefixListSource,
         'URL': CommonTokens.prefixListUrlMatcher,
         'vrf': 'VRF name',
         'VRFNAME': allVrfWithNameMatcher,
         'duplicate': CommonTokens.prefixListDuplicateToken,
         'ACTION': CommonTokens.prefixListDuplicateMatcher
         }
   handler = "RouteMapCliHandler.loadPrefixListUrlHandler"
   noOrDefaultHandler = "RouteMapCliHandler.removePrefixListEntryHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpPrefixListLoadURLCmd )

# ip prefix-list  <destination_name> { copy|rename } <source_name> [ overwrite ]
class IpPrefixListCopyOrRenameCmd( CliCommand.CliCommandClass ):
   syntax = "ip prefix-list DESTINATION ACTION SOURCE [ overwrite ]"
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'DESTINATION': prefixListNameMatcher,
         'ACTION': CommonTokens.copyOrRenameMatcher,
         'SOURCE': srcPrefixListNameMatcher,
         'overwrite': CommonTokens.overwrite
         }
   handler = "RouteMapCliHandler.copyOrRenamePrefixListHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpPrefixListCopyOrRenameCmd )

# ipv6 prefix-list  <destination_name> { copy|rename } <source_name> [ overwrite ]
class Ipv6PrefixListCopyOrRenameCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 prefix-list DESTINATION ACTION SOURCE [ overwrite ]"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'DESTINATION': ipv6PrefixListNameMatcher,
         'ACTION': CommonTokens.copyOrRenameMatcher,
         'SOURCE': srcIpv6PrefixListNameMatcher,
         'overwrite': CommonTokens.overwrite
         }
   handler = "RouteMapCliHandler.copyOrRenameIpv6PrefixList"

BasicCli.GlobalConfigMode.addCommandClass( Ipv6PrefixListCopyOrRenameCmd )

class ShowIpPrefixListCmd( ShowCliCommandClass ):
   syntax = "show ip prefix-list [ NAME ]"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         }
   cliModel = IpPrefixLists
   handler = "RouteMapCliHandler.showPrefixList"

BasicCli.addShowCommandClass( ShowIpPrefixListCmd )

class ShowIpPrefixListDetailCmd( ShowCliCommandClass ):
   syntax = "show ip prefix-list NAME detail"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         'detail': 'detail',
         }
   cliModel = IpPrefixLists
   handler = "RouteMapCliHandler.showPrefixListDetail"

BasicCli.addShowCommandClass( ShowIpPrefixListDetailCmd )

class ShowIpPrefixListSummaryCmd( ShowCliCommandClass ):
   syntax = "show ip prefix-list NAME summary"
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         'summary': 'summary',
         }
   cliModel = PrefixListSummary
   handler = "RouteMapCliHandler.showPrefixListSummary"

BasicCli.addShowCommandClass( ShowIpPrefixListSummaryCmd )

class RefreshIpPrefixList( CliCommand.CliCommandClass ):
   syntax = 'refresh ip prefix-list [ LISTNAME ]'
   data = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'ip': CliToken.Ip.ipMatcherForRefresh,
      'prefix-list': CommonTokens.matchPrefixList,
      'LISTNAME': prefixListDynamicNameMatcher,
   }
   handler = "RouteMapCliHandler.handlerRefreshIpPrefixList"

BasicCli.EnableMode.addCommandClass( RefreshIpPrefixList )

# pylint: disable-next=inconsistent-return-statements
def communityListByName( config, commType, listName ):
   assert type( commType ) == CommunityType # pylint: disable=unidiomatic-typecheck
   communityList = config.communityList.get( listName )
   if communityList is None:
      return
   asdotConfigured = isAsdotConfigured( asnConfig )
   entries = []
   for entry in communityList.communityListEntry.values():
      if entry.version == 0:
         continue
      if ( entry.listType == 'communityStandard' and
           commType == CommunityType.communityTypeStandard ) or \
         ( entry.listType == 'extCommunityStandard' and
           commType == CommunityType.communityTypeExtended ) or \
         ( entry.listType == 'largeCommunityStandard' and
           commType == CommunityType.communityTypeLarge ):
         listType = 'standard'
      elif ( entry.listType == 'communityExpanded' and
             commType == CommunityType.communityTypeStandard ) or \
           ( entry.listType == 'extCommunityExpanded' and
             commType == CommunityType.communityTypeExtended ) or \
           ( entry.listType == 'largeCommunityExpanded' and
             commType == CommunityType.communityTypeLarge ):
         listType = 'expanded'
      else:
         continue
      commValues = None
      commRegex = None
      if listType == 'standard':
         commValues = []
         lbwRange = False
         for value in entry.community:
            #check if range, choose the right display
            if commType == CommunityType.communityTypeExtended and \
               extCommType( value ) == 36:
               commValues.append( commValueToPrint(
                  value, commType=commType, lbwDisplay=entry.lbwDisplay,
                  lbwRange=lbwRange, asdotConfigured=asdotConfigured ) )
               lbwRange = True
            else:
               commValues.append( commValueToPrint(
                  value, commType=commType, lbwDisplay=entry.lbwDisplay,
                  asdotConfigured=asdotConfigured ) )
      else:
         commRegex = entry.commRegexDisplay
      ipCommunityListEntry = IpCommunityListEntry(
         filterType=matchPermitEnum[ entry.permit ], listType=listType,
         communityValues=commValues, communityRegex=commRegex )
      entries.append( ipCommunityListEntry )

   if not entries:
      return None
   if commType == CommunityType.communityTypeStandard:
      return IpCommunityList( entries=entries )
   elif commType == CommunityType.communityTypeExtended:
      return IpExtCommunityList( entries=entries )
   elif commType == CommunityType.communityTypeLarge:
      return IpLargeCommunityList( entries=entries )
   else:
      assert False, "Unrecognized community type: %s" % ( commType )

#-------------------------------------------------------------------------
# Community list
#
# # Deprecated Version.
# [no] ip community-list expanded <name> <permit|deny> <regex>
# [no] ip community-list standard <name> <permit|deny> [<1-4294967295>
#                                                       <aa:nn>
#                                                       local-as
#                                                       no-advertise
#                                                       no-export]
#
# # New Version.
# [no] ip community-list regexp NAME ( permit | deny ) REGEX
# [no] ip community-list <name> <permit|deny> [<1-4294967295>
#                                              <aa:nn>
#                                              local-as
#                                              no-advertise
#                                              no-export]
#-------------------------------------------------------------------------
# Deprecated version
class IpCommunityListDeprecatedCmd( CliCommand.CliCommandClass ):
   syntax = 'ip community-list standard NAME ACTION { COMM_VALUES }'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'community-list': CommonTokens.communityList,
      'standard': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'standard',
            helpdesc='Add a standard community-list entry' ),
         deprecatedByCmd='ip community-list' ),
      'regexp': CommonTokens.commRegexp,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'COMM_VALUES': CommunityConstExpr
   }
   handler = "RouteMapCliHandler.ipCommunityListCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipCommunityListCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpCommunityListDeprecatedCmd )

# New Version
class IpCommunityListCmd( CliCommand.CliCommandClass ):
   syntax = 'ip community-list NAME ACTION { COMM_VALUES }'
   noOrDefaultSyntax = 'ip community-list NAME [ ACTION { COMM_VALUES } ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'community-list': CommonTokens.communityList,
      'regexp': CommonTokens.commRegexp,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'COMM_VALUES': CommunityConstExpr
   }
   handler = "RouteMapCliHandler.ipCommunityListCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipCommunityListCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpCommunityListCmd )

# Deprecated version
class IpCommunityListRegexpDeprecatedCmd( CliCommand.CliCommandClass ):
   syntax = 'ip community-list expanded NAME ACTION REGEX'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'community-list': CommonTokens.communityList,
      'expanded': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'expanded',
            helpdesc='Add an expanded community-list entry' ),
         deprecatedByCmd='ip community-list regexp' ),
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'REGEX': CommonTokens.commListRegexMatcher,
   }
   handler = "RouteMapCliHandler.ipCommunityListRegexpCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipCommunityListRegexpCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpCommunityListRegexpDeprecatedCmd )

# New Version
class IpCommunityListRegexpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip community-list regexp NAME ACTION REGEX'
   noOrDefaultSyntax = 'ip community-list regexp NAME [ ACTION REGEX ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'community-list': CommonTokens.communityList,
      'regexp': CommonTokens.commRegexp,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'REGEX': CommonTokens.commListRegexMatcher,
   }
   handler = "RouteMapCliHandler.ipCommunityListRegexpCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipCommunityListRegexpCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpCommunityListRegexpCmd )

# show community list
class ShowIpCommunityListCmd( ShowCliCommandClass ):
   syntax = 'show ip community-list [ NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'community-list': CommonTokens.communityList,
      'NAME': CommonTokens.communityListName,
   }
   cliModel = IpCommunityLists
   handler = "RouteMapCliHandler.handlerShowIpCommunityListCmd"

BasicCli.addShowCommandClass( ShowIpCommunityListCmd )

#-------------------------------------------------------------------------
# Extended Community list
#
# Deprecated version
# [no] ip extcommunity-list expanded <name> <permit|deny> <regex>
# [no] ip extcommunity-list standard <name> <permit|deny> <rt|soo> ASN:nn
#
# New
# [no] ip extcommunity-list regexp <name> <permit|deny> <regex> (expanded commlist)
# [no] ip extcommunity-list <name> <permit|deny> <rt|soo> ASN:nn (standard commlist)
#-------------------------------------------------------------------------
def isRange( lbwVal ):
   valueSplit = lbwVal.split()
   if len( valueSplit ) == 4 and valueSplit[ 0 ] == 'lbw' and \
      valueSplit[ 1 ] == 'range':
      return True
   else:
      return False

def isValidRange( lbwVal ):
   valueSplit = lbwVal.split()
   if ':' in valueSplit[ 2 ] and ':' in valueSplit[ 3 ]:
      lbw1Split = valueSplit[ 2 ].split( ':' )
      lbw2Split = valueSplit[ 3 ].split( ':' )
      # check for K,M,G
      bwre = re.compile( r'(\d+|\d+\.\d+)(K|M|G)?$' )
      m1 = bwre.match( lbw1Split[ 1 ] )
      m2 = bwre.match( lbw2Split[ 1 ] )
      multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
      if m1.groups()[1] and m1.groups()[1] in multiplier:
         lbw1_f = float( m1.groups()[0] ) * multiplier[m1.groups()[1]]
      else:
         lbw1_f = float( m1.groups()[0] )
      if m2.groups()[1] and m2.groups()[1] in multiplier:
         lbw2_f = float( m2.groups()[0] ) * multiplier[m2.groups()[1]]
      else:
         lbw2_f = float( m2.groups()[0] )
      if lbw1Split[ 0 ] != lbw2Split[ 0 ]:
         return False
      elif lbw1_f >= lbw2_f:
         return False
      else:
         return True
   else:
      return False

class ExtCommsListExp( CliCommand.CliExpression ):
   expression = "( { EXTCOMMS } | ( [ { EXTCOMMS } ] LBW_EXP [ { EXTCOMMS } ] ) )"
   data = {
      'EXTCOMMS': GenericExtCommsExp,
      'LBW_EXP': LinkBandwidthExp,
   }

class IpExtCommunityListDeprecatedCmd( CliCommand.CliCommandClass ):
   syntax = "ip extcommunity-list standard NAME ACTION EXTCOMMS"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'extcommunity-list': CommonTokens.extCommunityList,
      'standard': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'standard',
            helpdesc='Add a standard extcommunity-list entry' ),
         deprecatedByCmd='ip extcommunity-list' ),
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'EXTCOMMS': ExtCommsListExp,
   }
   handler = "RouteMapCliHandler.ipExtCommunityListCmdHandler"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( IpExtCommunityListDeprecatedCmd )

# ip extcommunity-list {...}
# new syntax of the command.
class IpExtCommunityListCmd( CliCommand.CliCommandClass ):
   syntax = "ip extcommunity-list NAME ACTION EXTCOMMS"
   noOrDefaultSyntax = "ip extcommunity-list NAME [ ACTION EXTCOMMS ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'extcommunity-list': CommonTokens.extCommunityList,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'EXTCOMMS': ExtCommsListExp,
   }
   handler = "RouteMapCliHandler.ipExtCommunityListCmdHandler"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( IpExtCommunityListCmd )

# extended expanded community list( Deprecated )
# ip extcommunity-list expanded <name> <permit|deny> <regex>
class IpExtCommunityListRegexDeprecatedCmd( CliCommand.CliCommandClass ):
   syntax = "ip extcommunity-list expanded NAME ACTION REGEX"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'extcommunity-list': CommonTokens.extCommunityList,
      'expanded': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'expanded',
            helpdesc='Add an expanded extcommunity-list entry' ),
         deprecatedByCmd='ip extcommunity-list regexp' ),
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'REGEX': CommonTokens.commListRegexMatcher,
   }
   handler = "RouteMapCliHandler.ipExtCommunityListRegexpCmdHandler"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( IpExtCommunityListRegexDeprecatedCmd )

# extended expanded community list( New Syntax )
# ip extcommunity-list regexp <name> <permit|deny> <regex>
class IpExtCommunityListRegexCmd( CliCommand.CliCommandClass ):
   syntax = "ip extcommunity-list regexp NAME ACTION REGEX"
   noOrDefaultSyntax = "ip extcommunity-list regexp NAME [ ACTION REGEX ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'extcommunity-list': CommonTokens.extCommunityList,
      'regexp': CommonTokens.commRegexp,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'REGEX': CommonTokens.commListRegexMatcher,
   }
   handler = "RouteMapCliHandler.ipExtCommunityListRegexpCmdHandler"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( IpExtCommunityListRegexCmd )

# show ext community list
class ShowIpExtCommunityListCmd( ShowCliCommandClass ):
   syntax = 'show ip extcommunity-list [ NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'extcommunity-list': CommonTokens.extCommunityList,
      'NAME': CommonTokens.communityListName,
   }
   cliModel = IpExtCommunityLists
   handler = "RouteMapCliHandler.handlerShowIpExtCommunityListCmd"

BasicCli.addShowCommandClass( ShowIpExtCommunityListCmd )

#-------------------------------------------------------------------------
# Large Community list
#
# [no] ip large-community-list <name> <permit|deny> { ASN:nn:nn }
# [no] ip large-community-list regexp <name> <permit|deny> <regex>
#-------------------------------------------------------------------------
class IpLargeCommunityListCmd( CliCommand.CliCommandClass ):
   syntax = "ip large-community-list NAME ACTION { LARGECOMMS }"
   noOrDefaultSyntax = "ip large-community-list NAME [ ACTION { LARGECOMMS } ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'large-community-list': CommonTokens.largeCommunityList,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'LARGECOMMS': LargeCommCliMatcher( showLargeCommunityDesc ),
   }
   handler = "RouteMapCliHandler.ipLargeCommunityListCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipLargeCommunityListCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpLargeCommunityListCmd )

class IpLargeCommunityListRegexCmd( CliCommand.CliCommandClass ):
   syntax = "ip large-community-list regexp NAME ACTION REGEX"
   noOrDefaultSyntax = "ip large-community-list regexp NAME [ ACTION REGEX ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'large-community-list': CommonTokens.largeCommunityList,
      'regexp': CommonTokens.commRegexp,
      'NAME': CommonTokens.communityListName,
      'ACTION': CommonTokens.communityPermitOrDeny,
      'REGEX': CommonTokens.commListRegexMatcher,
   }
   handler = "RouteMapCliHandler.ipLargeCommunityListRegexpCmdHandler"
   noOrDefaultHandler = "RouteMapCliHandler.ipLargeCommunityListRegexpCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( IpLargeCommunityListRegexCmd )

# show large community list
class ShowIpLargeCommunityListCmd( ShowCliCommandClass ):
   syntax = 'show ip large-community-list [ NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'large-community-list': CommonTokens.largeCommunityList,
      'NAME': CommonTokens.communityListName,
   }
   cliModel = IpLargeCommunityLists
   handler = "RouteMapCliHandler.handlerShowIpLargeCommunityListCmd"

BasicCli.addShowCommandClass( ShowIpLargeCommunityListCmd )

#------------------------------------------------------------------------
# Prefix list mode
#
# [no] ip prefix-list <name>
#     [no] [<seqno>] <permit|deny> <prefix> [<ge> <len> <le> <len>]
#------------------------------------------------------------------------
class IpPrefixListMode( IpPrefixListCliMode, BasicCli.ConfigModeBase ):
   prompt = "(config-ip-pfx)#"
   name = 'IP prefix list configuration'

   def __init__ ( self, parent, session, listName ):
      # name of this prefix-list
      self.listName  = listName
      # used to abandon changes if the user decides to abort
      self.abort_ = False
      # stores the config changes prior to commiting
      self.pendingConfig = None
      self.pendingRemarkConfig = None
      # contains the list of modified sequence nos for this prefixList
      self.modSeqnos = set()
      self.modRemarkSeqnos = set()
      # set for keeping track of sequence numbers
      self.allRemarkSeqnos = set()
      self.allPrefixSeqnos = set()
      # contains the list of prefix-entry hash values which are
      # useful for duplicate check
      self.prefixEntryHashVals = set()
      self.config = aclListConfig.prefixList.get( listName )
      self.remarkList = aclListConfig.remarkCollection.get( listName )

      self.pendingConfig = Tac.newInstance( "Acl::PrefixList", listName)
      self.pendingRemarkList = Tac.newInstance( "Acl::RemarkList", listName )
      # make a copy of the prefix list entries
      if self.config:
         self.version = self.config.version
         for entry in self.config.prefixEntry.values():
            self.pendingConfig.prefixEntry.addMember( entry )
            self.allPrefixSeqnos.add( entry.seqno )
            hashVal = getPrefixEntryHashVal( entry.prefix,
                                             entry.permit,
                                             entry.ge,
                                             entry.le )
            t0( 'Adding hashVal %s' % hashVal )
            self.prefixEntryHashVals.add( hashVal )
      else:
         self.version = 1

      # make a copy of the remark list entries
      if self.remarkList:
         for seq, remark in self.remarkList.remarkEntries.items():
            self.pendingRemarkList.remarkEntries[ seq ] = remark
            self.allRemarkSeqnos.add( seq )

      IpPrefixListCliMode.__init__( self, listName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitPrefixListCfg( self ):
      if self.abort_:
         return

      pfxList = aclListConfig.prefixList
      if self.listName not in pfxList:
         if self.version != 1:
            # if the prefix list does not exist in sysdb, then the version in
            # the state must be 1. If not, the sysdb state for this prefix
            # list was removed by another session
            self.addError( "Error: There was a commit from "\
                           " another session. Ignoring the changes " )
            return

         if self.pendingConfig is not None:
            # create an entry for the prefixListName if not present
            pfxListEntry = pfxList.newMember( self.listName )
         else:
            # Both the local state and the sysdb state for this prefix list
            # name is empty. So just return
            return
      else:
         pfxListEntry =  pfxList.get( self.listName )
         if pfxListEntry.version > self.version:
            # There was a commit from another session. abort the changes
            self.addError( "Error: There was a commit from "\
                           " another session. Ignoring the changes " )
            return
         # if the prefix-list was removed as part of this config mode, remove
         # its entry from sysdb
         if self.pendingConfig is None:
            del pfxList[ self.listName ]
            return

      if not pfxListEntry.inPrefixListMode:
         pfxListEntry.inPrefixListMode = True
         aclListConfig.prefixListModeCount += 1
      #delete any stale entries in sysdb
      for entry in pfxListEntry.prefixEntry:
         if entry not in self.pendingConfig.prefixEntry:
            del pfxListEntry.prefixEntry[ entry ]

      # Check the modified sequence numbers and update them in sysdb
      for seq in self.modSeqnos:
         pfxEntry = self.pendingConfig.prefixEntry.get( seq )
         pfxListEntry.prefixEntry.addMember( pfxEntry )

      # Notify that prefix list mode changes are committed to sysdb
      pfxListEntry.version = self.version + 1

   def commitRemarkList( self ):
      if self.abort_:
         return

      remarkCollection = aclListConfig.remarkCollection
      remarkList = None
      # instantiate new remark list if list name is new
      if self.listName not in remarkCollection:
         # if there are no new remarks and no previous remarks, return
         if not self.pendingRemarkList:
            return
         remarkList = remarkCollection.newMember( self.listName )
      else:
         # if remarkList has been removed
         if self.pendingRemarkList is None:
            del remarkCollection[ self.listName ]
            return
         remarkList = remarkCollection.get( self.listName )

      # delete stale sysdb remark entries
      for entry in remarkList.remarkEntries:
         if entry not in self.pendingRemarkList.remarkEntries:
            del remarkList.remarkEntries[ entry ]

      # update modified seq nos
      for seq in self.modRemarkSeqnos:
         remarkEntry = self.pendingRemarkList.remarkEntries.get( seq )
         remarkList.remarkEntries[ seq ] = remarkEntry

   def onExit( self ):
      self.commitPrefixListCfg( )
      self.commitRemarkList( )
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.pendingConfig = None
      self.abort_ = True
      self.session_.gotoParentMode()

class GoToIpPrefixListModeCmd ( CliCommand.CliCommandClass ):
   syntax = "ip prefix-list NAME"
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': prefixListNameMatcher,
         }
   handler = "RouteMapCliHandler.gotoIpPfxListMode"

BasicCli.GlobalConfigMode.addCommandClass( GoToIpPrefixListModeCmd )

class IpShowPendingPfxListCmd( ShowCliCommandClass ):
   syntax = "show pending"
   data = {
         'pending': 'Show the pending prefix-list information'
         }
   handler = "RouteMapCliHandler.handlerIpShowPendingPfxListCmd"

IpPrefixListMode.addShowCommandClass( IpShowPendingPfxListCmd )

class IpShowDiffPfxListCmd( ShowCliCommandClass ):
   syntax = "show diff"
   data = {
         'diff': 'Show the difference between active and pending filter'
         }
   handler = "RouteMapCliHandler.handlerIpShowDiffPfxListCmd"

IpPrefixListMode.addShowCommandClass( IpShowDiffPfxListCmd )

class IpPrefixListModeCmd ( CliCommand.CliCommandClass ):
   syntax = "[ seq SEQNO ] ACTION PREFIX"\
               "[ ( eq EQLEN ) "\
               "| ( ge GELEN [ le LELEN ] ) "\
               "| (le LELEN [ ge GELEN ] ) ]"
   noOrDefaultSyntax = "( seq SEQNO ) | ( ACTION PREFIX"\
                                            "[ ( eq EQLEN )"\
                                            "| ( ge GELEN [ le LELEN ] )"\
                                            "| ( le LELEN [ ge GELEN ] ) ] )"
   data = {
         'seq': CommonTokens.prefixListSeq,
         'SEQNO': CommonTokens.prefixListSeqnoMatcher,
         'ACTION': CommonTokens.prefixListPermitOrDeny,
         'PREFIX': CommonTokens.prefixListPrefix,
         'eq': CommonTokens.prefixListEq,
         'ge': CommonTokens.prefixListGe,
         'le': CommonTokens.prefixListLe,
         'EQLEN': CommonTokens.prefixListMaskLenMatcher,
         'GELEN': CommonTokens.prefixListMaskLenMatcher,
         'LELEN': CommonTokens.prefixListMaskLenMatcher
         }
   handler = "RouteMapCliHandler.managePrefixListHandler"
   noOrDefaultHandler = "RouteMapCliHandler.removePrefixListEntryHandler"

IpPrefixListMode.addCommandClass( IpPrefixListModeCmd )


class IpPrefixListModeResequenceCmd( CliCommand.CliCommandClass ):
   syntax = "resequence [ START [ INC ] ]"
   data = {
         'resequence': CommonTokens.prefixListResequenceToken,
         'START': CommonTokens.prefixListStartSeq,
         'INC': CommonTokens.prefixIncSeqRule
        }
   handler = "RouteMapCliHandler.resequencePrefixListHandler"

IpPrefixListMode.addCommandClass( IpPrefixListModeResequenceCmd )

class IpPrefixListModeAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
         'abort': 'Exit ip prefix-list mode config without committing changes'
         }
   handler = "RouteMapCliHandler.handlerIpPrefixListModeAbortCmd"

IpPrefixListMode.addCommandClass( IpPrefixListModeAbortCmd )

#Register convertInPrefixListMode via "config convert new-syntax"
CliPlugin.ConfigConvert.registerConfigConvertCallback(
   LazyCallback( "RouteMapCliHandler.convertInPrefixListMode" ) )

#------------------------------------------------------------------------
# Ipv6 Prefix list
#
# [no] ipv6 prefix-list <name> <permit|deny> <prefix> [<ge> <len> |  <le> <len>
#                                       <ge> <le]
#------------------------------------------------------------------------
class Ipv6PrefixListMode( Ipv6PrefixListCliMode, BasicCli.ConfigModeBase ):
   prompt = "(config-ipv6-pfx)#"
   name = 'IPv6 prefix list configuration'

   def __init__ ( self, parent, session, ipv6PfxListName ):
      self.pfxListName  = ipv6PfxListName
      self.abort_ = False
      self.pendingConfig = None
      # contains the list of modified sequence nos for this prefixList
      self.modSeqnos = set()
      self.modRemarkSeqnos = set()
      # sets for keeping track of sequence numbers
      self.allRemarkSeqnos = set()
      self.allPrefixSeqnos = set()
      # contains the list of prefix-entry hash values which are
      # useful for duplicate check
      self.prefixEntryHashVals = set()
      self.config = aclListConfig.ipv6PrefixList.get( ipv6PfxListName )
      self.remark6List = aclListConfig.ipv6RemarkCollection.get( ipv6PfxListName )

      self.pendingConfig = Tac.newInstance( "Acl::Ipv6PrefixList",
                                            ipv6PfxListName )
      self.pendingRemarkList = Tac.newInstance( "Acl::RemarkList", ipv6PfxListName )
      # make a copy of the prefix list entries
      if self.config:
         self.version = self.config.version
         for entry in self.config.ipv6PrefixEntry.values():
            self.pendingConfig.ipv6PrefixEntry.addMember( entry )
            self.allPrefixSeqnos.add( entry.seqno )
            hashVal = getPrefixEntryHashVal( entry.ip6Prefix, entry.permit,
                                             entry.ge, entry.le )
            t0( 'Adding hashVal %s' % hashVal )
            self.prefixEntryHashVals.add( hashVal )
      else:
         self.version = 1
      
      # make a copy of the remark list entries
      if self.remark6List:
         for seq, remark in self.remark6List.remarkEntries.items():
            self.pendingRemarkList.remarkEntries[ seq ] = remark
            self.allRemarkSeqnos.add( seq )

      Ipv6PrefixListCliMode.__init__( self, ipv6PfxListName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitIpv6PrefixListCfg( self ):
      if self.abort_:
         return

      ipv6PfxList = aclListConfig.ipv6PrefixList
      if self.pfxListName not in ipv6PfxList:
         if self.version != 1:
            # if the prefix list does not exist in sysdb, then the version in
            # the state must be 1. If not, the sysdb state for this prefix
            # list was removed by another session
            self.addError( "There was a commit from another session. "
                           "Ignoring the changes " )
            return

         if self.pendingConfig is not None:
            # create an entry for the prefixListName if not present
            pfxListEntry =  ipv6PfxList.newMember( self.pfxListName )
         else:
            # Both the local state and the sysdb state for this prefix list
            # name is empty. So just return
            return
      else:
         pfxListEntry =  ipv6PfxList.get( self.pfxListName )
         if pfxListEntry.version > self.version:
            # There was a commit from another session. abort the changes
            self.addError( "There was a commit from another session. "
                           "Ignoring the changes " )
            return
         # if the prefix-list was removed as part of this config mode, remove
         # its entry from sysdb
         if self.pendingConfig is None:
            del ipv6PfxList[ self.pfxListName ]
            return

      #delete any stale entries in sysdb
      for entry in pfxListEntry.ipv6PrefixEntry:
         if entry not in self.pendingConfig.ipv6PrefixEntry:
            del pfxListEntry.ipv6PrefixEntry[ entry ]

      # Check the modified sequence numbers and update them in sysdb
      for seq in self.modSeqnos:
         pfxEntry = self.pendingConfig.ipv6PrefixEntry.get( seq )
         pfxListEntry.ipv6PrefixEntry.addMember( pfxEntry )

      # Notify that prefix list mode changes are committed to sysdb
      pfxListEntry.version = self.version + 1
   
   def commitIpv6RemarkList( self ):
      if self.abort_:
         return
      
      remarkCollection = aclListConfig.ipv6RemarkCollection
      remarkList = None
      # instantiate new remark list if list name is new
      if self.pfxListName not in remarkCollection:
         # if there are no new remarks and no previous remarks, return
         if not self.pendingRemarkList:
            return
         remarkList = remarkCollection.newMember( self.pfxListName )
      else:
         # if remarkList has been removed
         if self.pendingRemarkList is None:
            del remarkCollection[ self.pfxListName ]
            return
         remarkList = remarkCollection.get( self.pfxListName )

      # delete stale sysdb remark entries
      for entry in remarkList.remarkEntries:
         if entry not in self.pendingRemarkList.remarkEntries:
            del remarkList.remarkEntries[ entry ]

      # update modified seq nos
      for seq in self.modRemarkSeqnos:
         remarkEntry = self.pendingRemarkList.remarkEntries.get( seq )
         remarkList.remarkEntries[ seq ] = remarkEntry


   def onExit( self ):
      self.commitIpv6PrefixListCfg( )
      self.commitIpv6RemarkList( )
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.pendingConfig = None
      self.abort_ = True
      self.session_.gotoParentMode()

ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixValidMatcher( "Ipv6 Prefix",
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
ipv6prefixListMaskLenMatcher = CliMatcher.IntegerMatcher( 1, 128,
   helpdesc='Mask length' )

class GoToIPv6PfxListModeCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 prefix-list NAME"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher
         }
   handler = "RouteMapCliHandler.gotoIpv6PfxListMode"

BasicCli.GlobalConfigMode.addCommandClass( GoToIPv6PfxListModeCmd )

class Ipv6PrefixListLoadUrlCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 prefix-list NAME SOURCE URL [ vrf VRFNAME ] [ duplicate ACTION ]"
   noOrDefaultSyntax = syntax
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher,
         'SOURCE': CommonTokens.prefixListSource,
         'URL': CommonTokens.prefixListUrlMatcher,
         'vrf': 'VRF name',
         'VRFNAME': allVrfWithNameMatcher,
         'duplicate': CommonTokens.prefixListDuplicateToken,
         'ACTION': CommonTokens.prefixListDuplicateMatcher
         }
   handler = "RouteMapCliHandler.loadIpv6PrefixListUrlHandler"
   noOrDefaultHandler = "RouteMapCliHandler.removeIpv6PrefixListEntryHandler"

BasicCli.GlobalConfigMode.addCommandClass( Ipv6PrefixListLoadUrlCmd )

class Ipv6ShowPendingPfxListCmd( ShowCliCommandClass ):
   syntax = "show pending"
   data = {
         'pending': 'Show the pending prefix-list information'
         }
   handler = "RouteMapCliHandler.handlerIpv6ShowPendingPfxListCmd"

Ipv6PrefixListMode.addShowCommandClass( Ipv6ShowPendingPfxListCmd )

class Ipv6ShowDiffPfxListCmd( ShowCliCommandClass ):
   syntax = "show diff"
   data = {
         'diff': 'Show the difference between active and pending filter'
         }
   handler = "RouteMapCliHandler.handlerIpv6ShowDiffPfxListCmd"

Ipv6PrefixListMode.addShowCommandClass( Ipv6ShowDiffPfxListCmd )

class Ipv6PrefixListModeCmd ( CliCommand.CliCommandClass ):
   syntax = "[ seq SEQNO ] ACTION PREFIX"\
                  "[ ( eq EQLEN )"\
                  "| ( ge GELEN [ le LELEN ] )"\
                  "| ( le LELEN [ ge GELEN ] ) ]"
   noOrDefaultSyntax = "( seq SEQNO )"\
                       "| ( ACTION PREFIX [ ( eq EQLEN )"\
                                         "| ( ge GELEN [ le LELEN ] )"\
                                         "| ( le LELEN [ ge GELEN ] ) ] )"
   data = {
         'seq': CommonTokens.prefixListSeq,
         'SEQNO': CommonTokens.prefixListSeqnoMatcher,
         'ACTION': CommonTokens.prefixListPermitOrDeny,
         'PREFIX': ipv6PrefixMatcher,
         'eq': CommonTokens.prefixListEq,
         'ge': CommonTokens.prefixListGe,
         'le': CommonTokens.prefixListLe,
         'EQLEN': ipv6prefixListMaskLenMatcher,
         'GELEN': ipv6prefixListMaskLenMatcher,
         'LELEN': ipv6prefixListMaskLenMatcher
         }
   handler = "RouteMapCliHandler.manageIpv6PrefixListHandler"
   noOrDefaultHandler = "RouteMapCliHandler.removeIpv6PrefixListEntryHandler"

Ipv6PrefixListMode.addCommandClass( Ipv6PrefixListModeCmd )

class Ipv6PrefixListModeResequenceCmd( CliCommand.CliCommandClass ):
   syntax = "resequence [ START [ INC ] ]"
   data = {
         'resequence': CommonTokens.prefixListResequenceToken,
         'START': CommonTokens.prefixListStartSeq,
         'INC': CommonTokens.prefixIncSeqRule
        }
   handler = "RouteMapCliHandler.resequenceIpv6PrefixListHandler"

Ipv6PrefixListMode.addCommandClass( Ipv6PrefixListModeResequenceCmd )

class RefreshIpv6PrefixList( CliCommand.CliCommandClass ):
   syntax = 'refresh ipv6 prefix-list [ LISTNAME ]'
   data = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'ipv6': CliToken.Ipv6.ipv6MatcherForRefresh,
      'prefix-list': CommonTokens.matchPrefixList,
      'LISTNAME': ipv6PrefixListDynamicNameMatcher,
   }
   handler = "RouteMapCliHandler.handlerRefreshIpv6PrefixList"

BasicCli.EnableMode.addCommandClass( RefreshIpv6PrefixList )

class Ipv6PrefixListModeAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
         'abort': 'Exit IPv6 prefix-list mode config without committing changes'
         }
   handler = "RouteMapCliHandler.handlerIpv6PrefixListModeAbortCmd"

Ipv6PrefixListMode.addCommandClass( Ipv6PrefixListModeAbortCmd )

class Ipv6PrefixListModeRemoveListNameCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = "ipv6 prefix-list NAME"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher
         }
   noOrDefaultHandler = "RouteMapCliHandler.removeIpv6PrefixListHandler"

BasicCli.GlobalConfigMode.addCommandClass( Ipv6PrefixListModeRemoveListNameCmd )

class ShowIpv6PrefixListCmd( ShowCliCommandClass ):
   syntax = "show ipv6 prefix-list [ NAME ]"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher,
         }
   cliModel = IpPrefixLists
   handler = "RouteMapCliHandler.showIpv6PrefixList"

BasicCli.addShowCommandClass( ShowIpv6PrefixListCmd )

class ShowIpv6PrefixListDetailCmd( ShowCliCommandClass ):
   syntax = "show ipv6 prefix-list NAME detail"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher,
         'detail': 'detail',
         }
   cliModel = IpPrefixLists
   handler = "RouteMapCliHandler.showIpv6PrefixListDetail"

BasicCli.addShowCommandClass( ShowIpv6PrefixListDetailCmd )

class ShowIpv6PrefixListSummaryCmd( ShowCliCommandClass ):
   syntax = "show ipv6 prefix-list NAME summary"
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
         'prefix-list': CommonTokens.prefixListMatcher,
         'NAME': ipv6PrefixListNameMatcher,
         'summary': 'summary',
         }
   cliModel = PrefixListSummary
   handler = "RouteMapCliHandler.showIpv6PrefixListSummary"

BasicCli.addShowCommandClass( ShowIpv6PrefixListSummaryCmd )

# Implements Remarks within IP and IPv6 Environments
class RemarkCmd( CliCommand.CliCommandClass ):
   syntax = "[seq SEQNO] remark COMMENT"
   data = {
      'seq': CommonTokens.prefixListSeq,
      'SEQNO': CommonTokens.prefixListSeqnoMatcher,
      'remark': CommonTokens.prefixListRemark,
      'COMMENT': CommonTokens.prefixListComment
   }
   handler = "RouteMapCliHandler.managePrefixListRemarksHandler"

remarksEnabled = RouteMapToggleLib.togglePrefixModeRemarksEnabled()
if remarksEnabled:
   IpPrefixListMode.addCommandClass( RemarkCmd )
   Ipv6PrefixListMode.addCommandClass( RemarkCmd )

# There is not a no/default handler for remarks, since the prefix handler already
# defines functionality for the syntax "no/default seq [SEQNO]". Since it would be
# unintuitive to not delete an entry at a sequence number if it's a remark,
# the functionality for deleting a remark at a sequence number is located within
# the prefix no/default handler

#------------------------------------------------------------------------
# Peer Filter mode
#
# [no] peer-filter <name>
#     [no] [<seqno>] match ... result {accept|reject}
#------------------------------------------------------------------------
class PeerFilterMode( PeerFilterCliMode, BasicCli.ConfigModeBase ):
   name = 'Peer Filter Configuration'
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, filterName ):
      self.filterName = filterName     # filter name
      self.abort_ = False              # abort requested?
      self.pendingConfig = None        # config with uncommitted changes
      self.modSeqnos = set()       # list of seqnos modified

      # Copy Sysdb config into pendingConfig
      self.pendingConfig = Tac.newInstance( 'Routing::RouteMap::Map', filterName )
      self.pendingConfig.version = 1   # version read from config

      filterConfig = peerFilterConfig.peerFilter.get( filterName )
      if filterConfig:
         self.pendingConfig.version = filterConfig.version
         if filterConfig.description:
            for index, desc in enumerate( filterConfig.description.values() ):
               self.pendingConfig.description[ index ] = desc
         for entry in filterConfig.mapEntry.values():
            newEntry = self.pendingConfig.mapEntry.newMember( entry.seqno )
            copyMapEntry( newEntry, entry, routeCtx=False )

      PeerFilterCliMode.__init__( self, filterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitConfig( self ):
      if self.abort_:
         return

      if self.filterName not in peerFilterConfig.peerFilter:
         # filter not in Sysdb

         if self.pendingConfig is None:
            # No pending changes
            return

         if self.pendingConfig.version != 1:
            # Filter was removed by another session
            self.addError( 'Error: There was a commit from '\
                           ' another session. Discarding the changes.' )
            return

         # Create new filter
         pFilter = peerFilterConfig.peerFilter.newMember( self.filterName )

      else:
         pFilter = peerFilterConfig.peerFilter.get( self.filterName )
         if pFilter.version > self.pendingConfig.version:
            # There was a commit from another session!
            self.addError( 'Error: There was a commit from '\
                           ' another session. Discarding the changes.' )
            return

      # Update description lines
      pFilter.description.clear()
      if self.pendingConfig.description:
         for index, desc in enumerate( self.pendingConfig.description.values() ):
            pFilter.description[ index ] = desc

      # Delete stale entries from Sysdb
      for seq in pFilter.mapEntry:
         if seq not in self.pendingConfig.mapEntry:
            del pFilter.mapEntry[ seq ]

      # Check the modified sequence numbers and update them in sysdb
      changed = False
      for seq in self.modSeqnos:
         if seq in pFilter.mapEntry:
            entry = pFilter.mapEntry[ seq ]
         else:
            entry = pFilter.mapEntry.newMember( seq )

         same = isSameMapEntry( entry, self.pendingConfig.mapEntry.get( seq ) )
         if not same:
            copyMapEntry( entry, self.pendingConfig.mapEntry.get( seq ),
                          routeCtx=False )
            changed = True

      # Bump version
      if changed:
         pFilter.version = self.pendingConfig.version + 1

   def onExit( self ):
      self.commitConfig()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pendingConfig = None
      self.abort_ = True
      self.session_.gotoParentMode()

peerFilterNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: peerFilterConfig.peerFilter,
   'Peer filter name' )

class GotoPeerFilterModeCmd( CliCommand.CliCommandClass ):
   syntax = 'peer-filter NAME'
   data = {
         'peer-filter': 'peer-filter',
         'NAME': peerFilterNameMatcher
         }

   handler = "RouteMapCliHandler.gotoPeerFilterMode"

BasicCli.GlobalConfigMode.addCommandClass( GotoPeerFilterModeCmd )

filterSeqnoMatcher = CliMatcher.IntegerMatcher( 0, 65535,
      helpdesc='Index in the sequence' )

class ManagePeerFilterCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ ] match as-range AS_RANGE result RESULT'
   data = {
      'SEQ': filterSeqnoMatcher,
      'match': 'Specify match rule',
      'as-range': 'ASN range',
      'AS_RANGE': AsNumRangeCliExpr,
      'result': 'Specify resulting action',
      'RESULT': CliMatcher.EnumMatcher( {
         'accept': 'Accept matching peer',
         'reject': 'Reject matching peer',
      } )
   }
   handler = "RouteMapCliHandler.handlerManagePeerFilterCmd"

if isMatchAttrEnabled( 'matchAsRange', routeCtx=False ):
   PeerFilterMode.addCommandClass( ManagePeerFilterCmd )

class SeqCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'SEQ'
   data = {
      'SEQ': filterSeqnoMatcher,
   }

   noOrDefaultHandler = "RouteMapCliHandler.peerFilterEntryRemover"

PeerFilterMode.addCommandClass( SeqCmd )

class PeerFilterModeConfigCmd( ShowCliCommandClass ):
   syntax = 'show ( active | pending | diff )'
   data = {
      'active': BasicCliModes.showActiveNode,
      'pending': 'Show the pending filter in this session',
      'diff': 'Show the difference between active and pending filter',
   }
   handler = "RouteMapCliHandler.handlerPeerFilterModeConfigCmd"

PeerFilterMode.addShowCommandClass( PeerFilterModeConfigCmd )

peerFilterDescMatcher = CliMatcher.StringMatcher( helpname='STRING',
      helpdesc='Description for the peer-filter' )

class PeerFilterDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
         'description': 'description',
         'DESCRIPTION': peerFilterDescMatcher,
         }

   handler = "RouteMapCliHandler.setPeerFilterDescription"
   noOrDefaultHandler = "RouteMapCliHandler.noPeerFilterDescription"

PeerFilterMode.addCommandClass( PeerFilterDescriptionCmd )

class PeerFilterModeAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
         'abort': 'Exit peer-filter configuration without committing changes'
         }
   handler = "RouteMapCliHandler.handlerPeerFilterModeAbortCmd"

PeerFilterMode.addCommandClass( PeerFilterModeAbortCmd )

class RemovePeerFilterCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'peer-filter NAME'
   data = {
         'peer-filter': 'peer-filter',
         'NAME': peerFilterNameMatcher,
         }
   noOrDefaultHandler = "RouteMapCliHandler.removePeerFilter"

BasicCli.GlobalConfigMode.addCommandClass( RemovePeerFilterCmd )

class PeerFilterShowCmd( ShowCliCommandClass ):
   syntax = 'show peer-filter [ NAME ]'
   data = {
         'peer-filter': 'peer-filter',
         'NAME': peerFilterNameMatcher,
         }

   cliModel = PeerFilters
   handler = "RouteMapCliHandler.showPeerFilter"

BasicCli.addShowCommandClass( PeerFilterShowCmd )

class LoadAsPathAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'ip as-path access-list NAME source URL [ duplicate ACTION ]'
   noOrDefaultSyntax = syntax
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'as-path': CommonTokens.asPathFilter,
         'access-list': CommonTokens.accessList,
         'NAME': CommonTokens.asPathListName,
         'source': asPathACLSource,
         'URL': asPathAccessListUrlMatcher,
         'duplicate': asPathDuplicate,
         'ACTION': duplicateHandlingOption
         }
   handler = "RouteMapCliHandler.handlerLoadAsPathAccessListCmd"
   noOrDefaultHandler = "RouteMapCliHandler.handlerNoLoadAsPathAccessListCmd"

BasicCli.GlobalConfigMode.addCommandClass( LoadAsPathAccessListCmd )

class RefreshAsPathAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ip as-path access-list [ NAME ]'
   data = {
          'refresh': CliToken.Refresh.refreshMatcherForExec,
          'ip': CliToken.Ip.ipMatcherForRefresh,
          'as-path': CommonTokens.asPathFilter,
          'access-list': CommonTokens.accessList,
          'NAME': CommonTokens.asPathListName
          }

   handler = "RouteMapCliHandler.refreshAsPathAccessListUrl"

BasicCli.EnableMode.addCommandClass( RefreshAsPathAccessListCmd )

aclUnconfReferencesHook.addExtension( blockAccessListListDeleteIfInUse )

#-------------------------------------------------------------------------------
# Definitions to prefix list configurations gatherers
#-------------------------------------------------------------------------------
pfxListTypes = { "IP prefix list": 'ip', "IPv6 prefix list": 'ipv6' }

class PfxListDefinitionGatherer:
   """This class gathers prefix list definitions."""
   @staticmethod
   def gather( feature ):
      pfxListType = pfxListTypes[ feature ]
      ipReferences = set( aclListConfig.prefixList )
      ipv6References = set( aclListConfig.ipv6PrefixList )
      return ipReferences if pfxListType == 'ip' else ipv6References

UndefinedReferenceChecker.addDefinitionGatherer( pfxListTypes,
                                                 PfxListDefinitionGatherer )

#-------------------------------------------------------------------------------
# Definitions to community list configurations gatherer
#-------------------------------------------------------------------------------
commListTypes = { "IP community list": 'ip' }

class CommunityListDefinitionGatherer:
   """This class gathers community list definitions."""
   @staticmethod
   def gather( feature ):
      return set( aclListConfig.communityList )

UndefinedReferenceChecker.addDefinitionGatherer( commListTypes,
                                                 CommunityListDefinitionGatherer )

#-------------------------------------------------------------------------------
# Route-map references to access, prefix, community list configurations gatherer
#-------------------------------------------------------------------------------
matchOptions = { "IP access list": RouteMapMatchOption.matchAccessList,
                 "IP prefix list": RouteMapMatchOption.matchPrefixList,
                 "IPv6 prefix list": RouteMapMatchOption.matchIpv6PrefixList,
                 "IP community list": RouteMapMatchOption.matchCommunity }

class RouteMapReferenceGatherer:
   """
   This class gathers prefix, access, community list references used by route
   maps.
   """
   @staticmethod
   def gather( feature ):
      matchOption = matchOptions[ feature ]
      references = set()
      routeMaps = mapConfig.routeMap
      for routeMapName in routeMaps:
         routeMap = routeMaps[ routeMapName ]
         mapEntries = routeMap.mapEntry
         for seqNo in mapEntries:
            mapEntry = mapEntries[ seqNo ]
            if matchOption in mapEntry.matchRule:
               valueType = MatchAttributes[ matchOption ][ 'type' ]
               rule = mapEntry.matchRule[ matchOption ]
               if valueType == 'string':
                  references.add( rule.strValue )
               elif valueType == 'set':
                  references.update( rule.commListSet.commListSet )
               else:
                  assert False, "RouteMapReferenceGatherer only supports " \
                                "access lists, prefix lists and community lists"
            if matchOption == RouteMapMatchOption.matchCommunity:
               for communitySet in ( mapEntry.communityAddReplace,
                                     mapEntry.communityDelete,
                                     mapEntry.communityFilter ):
                  if communitySet is not None:
                     references.update( communitySet.communityListNameSet )
      return references

UndefinedReferenceChecker.addReferenceGatherer( matchOptions,
                                                RouteMapReferenceGatherer )

routeMapTypes = [ "Route map" ]

class RouteMapNameDefinitionGatherer:
   """
   This class gathers definitions of route maps.
   """
   @staticmethod
   def gather( feature ):
      return set( mapConfig.routeMap.members() )

UndefinedReferenceChecker.addDefinitionGatherer( routeMapTypes,
                                                 RouteMapNameDefinitionGatherer )

class DynPfxListRouteMapReferenceGatherer:
   """
   This class gathers references to route maps in dynamic prefix list config.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      dynPfxLists = dynPfxListConfigDir.dynamicPrefixList
      for dynPfxListName in dynPfxLists:
         dynPfxListEntry = dynPfxLists[ dynPfxListName ]
         routeMap = dynPfxListEntry.matchMap
         if routeMap:
            references.add( routeMap )
      return references

UndefinedReferenceChecker.addReferenceGatherer( routeMapTypes,
                                                DynPfxListRouteMapReferenceGatherer )

#------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------
def Plugin( entityManager ):
   global mapConfig, dynamicMapConfig
   global aclConfig, aclListConfig, asnConfig, peerFilterConfig
   global handlerDir
   global inputDir
   global dynPfxListConfigDir
   global l3Config

   mapConfig = ConfigMount.mount(
      entityManager, 'routing/routemap/config', 'Routing::RouteMap::Config', 'w' )
   asnConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/asn/config', 'Routing::AsnConfig', 'w' )
   aclListConfig = ConfigMount.mount(
      entityManager, "routing/acl/config", "Acl::AclListConfig", "wS" )
   peerFilterConfig = ConfigMount.mount(
      entityManager, 'routing/peerfilter/config',
      'Routing::RouteMap::PeerFilterConfig', 'w' )
   aclConfig = LazyMount.mount( entityManager, "acl/config/cli",
                                "Acl::Input::Config", "r" )
   dynPfxListConfigDir = LazyMount.mount( entityManager,
                                          'routing/dynPfxList/config',
                                          'Routing::DynamicPrefixList::Config', 'r' )
   handlerDir = CliSession.handlerDir( entityManager )
   inputDir = CliSession.inputDir( entityManager )
   l3Config = LazyMount.mount( entityManager, 'l3/config', 'L3::Config', 'r' )

   dynamicMapConfig = Tac.newInstance( 'Routing::RouteMap::Config', 'output' )

   gv.routeMapHelper = Tac.newInstance( 'Routing::RouteMap::Helper' )
