#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
from Toggles import ( IpRibLibToggleLib, NexthopGroupToggleLib, RoutingLibToggleLib,
                      RouterGeneralCliToggleLib)
from CliPlugin import (
   IpAddrMatcher as IpAddr,
   Ip6AddrMatcher as Ip6Addr,
   VrfCli,
   IraServiceCli,
   IraIpCli,
   RcfCliLib,
)
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher
from CliPlugin.IraNexthopGroupCli import nexthopGroupSupportedGuard

from CliPlugin.UcmpLibCliLib import ucmpSizeRange
import BasicCli
import BasicCliModes
import Tracing
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.RouterGeneral import (
   RouterGeneralAutoVersionBaseMode,
   RouterGeneralBaseMode,
   RouterGeneralMetricProfileBaseMode,
   RouterGeneralRoutePriorityBaseMode,
   RouterGeneralVrfBaseMode,
   RouterGeneralVrfRoutePriorityBaseMode,
   RouterGeneralVrfFilteredRibBaseMode,
)
import CliParser
import CliToken.Clear
from CliToken.Router import routerMatcherForConfig
from CliToken.UcmpLibCliTokens import ucmpLibTokenUcmp, ucmpDevRangeMatcher
from CliParserCommon import namePattern

from IpLibTypes import ProtocolAgentModelType
import LazyMount
import ConfigMount
import Tac
from TypeFuture import TacLazyType
from Arnet.NsLib import DEFAULT_NS

traceHandle = Tracing.Handle( 'RouterGeneralCli' )
t5 = traceHandle.trace5 # Info
# NOTE:
# This is expected to be the dumpyard for routing config knobs that
# do not fit into any of the existing protocols (so a whole lot
# of not so related code is expected to come in here).
#
# Configuration knobs:
# rtr# router general
#      ### config applicable to all vrfs goes here
#      ### Example:
#      ### [no|default] next-hops fec dedicated
#
# Vrf submode - All vrf specific configuration will go into this mode
# NOTE: default is also considered as a vrf, and user is expected to
# input 'vrf default' to configure for default vrf.
#
# rtr# router general
# router-general# vrf <vrfname>
#                 ### Example:
#                 ### routes dynamic prefix-list <name>
#
# To simplify the code organization the following is the approach
# (similar ip commands which is present is a whole lot of places).
#
# - Any configuration which is a single knob can be added in
#   RouterGeneral package (this file itself). This is a straight
#   forward case and nothing special is required.
#   The config is mounted at : routing/general/config/global
#
# - Any considerable configuration, which may need its own sm to
#   to sync configuration to gated, should prefferably be in its
#   own package (at the very least we could have separate CliPlugin
#   and tacc SM files).
#   To support config-replace and no/default for the above case the
#   new package will have to register callback handlers (using
#   CliExtensions.CliHook() ).
#   The config is mounted at : routing/general/config/<somename>
#

defaultRouterIdV4 = Tac.Value( 'Arnet::IpAddr' ).ipAddrZero
defaultRouterIdV6 = Tac.Value( 'Arnet::Ip6Addr' )
RoutingGeneralConfig = TacLazyType( 'Routing::General::Config' )

#----------------------------------------------------------------------------------
#       Support for registering cleanup handlers
#----------------------------------------------------------------------------------
routerGeneralCleanupHook = CliExtensions.CliHook()
routerGeneralVrfCleanupHook = CliExtensions.CliHook()

#----------------------------------------------------------------------------------
#                           T A C     I N S T A N C E S
#----------------------------------------------------------------------------------
generalConfig = None  # Routing::General::Config
routingHwStatus = None  # Routing::Hardware::Status
prioritizationConfig = None  # Routing::VrfPrioritizationConfigAll
ucmpStaticVrfConfigDir = None # Routing::Ucmp::VrfUcmpConfigDir
autoVersionConfig = None # Routing::AutoVersion::Config

#----------------------------------------------------------------------------------
#       Helpers
#----------------------------------------------------------------------------------
def getRouterConfig( vrfName ):
   return generalConfig.vrfConfig.get( vrfName, generalConfig )

def orderedEcmpSupportedGuard( mode, token ):
   if routingHwStatus.orderedEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ucmpSupportedGuard( mode, token ):
   if routingHwStatus.ucmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def hwSupportedGuard( mode, token ):
   # Guards against enabling cli in single-agent mode and checks routingSupported
   if IraServiceCli.getEffectiveProtocolModel( mode ) == \
         ProtocolAgentModelType.multiAgent and routingHwStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

def hierarchicalFecSupportedGuard( mode, token ):
   if routingHwStatus.hierarchicalFecSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def addIpv4RoutableReservedRebootWarning( mode ):
   mode.addWarning( "Changes are only ensured after switch reboot" )

def getIpv4Routable240ClassE():
   return generalConfig.ipv4Routable240ClassE

IraIpCli.ipv4Routable240ClassEHook.addExtension( getIpv4Routable240ClassE )

def getIpv4Routable0ClassA():
   return generalConfig.ipv4Routable0ClassA

IraIpCli.ipv4Routable0ClassAHook.addExtension( getIpv4Routable0ClassA )

#----------------------------------------------------------------------------------
#       Cli Nodes
#----------------------------------------------------------------------------------
ucmpNode = CliCommand.Node( matcher=ucmpLibTokenUcmp,
                            guard=ucmpSupportedGuard )

#----------------------------------------------------------------------------------
#                                CLI MODES
# - Global mode applicable to all vrfs
# - Per vrf mode, specific to vrfs
#   (default vrf config will be under 'vrf default' mode)
#----------------------------------------------------------------------------------
class RouterGeneralMode( RouterGeneralBaseMode, BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.
   '''
   name = 'Protocol independent routing configuration'

   def __init__( self, parent, session ):
      RouterGeneralBaseMode.__init__( self, param='generalconfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vrfName = ''

   def deleteModeConfig( self ):
      for vrfName in generalConfig.vrfConfig:
         routerGeneralVrfCleanupHook.notifyExtensions( vrfName )
         del generalConfig.vrfConfig[ vrfName ]

      # delete all the vrf ucmp configs
      ucmpStaticVrfConfigDir.vrfConfig.clear()

      generalConfig.metricProfile.clear()
      prioritizationConfig.globalPriority.protocolPriority.clear()
      prioritizationConfig.vrfConfig.clear()

class RouterGeneralVrfMode( RouterGeneralVrfBaseMode, BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.
   '''
   name = 'Protocol independent routing vrf configuration'

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      RouterGeneralVrfBaseMode.__init__( self, self.vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      '''
      This allows default and non-default VRFs to have a separate parse-tree,
      and we can leverage this to add modelets which supports/restricts some
      commands on default/non-default VRFs.
      '''
      return 'default-vrf' if ( self.vrfName == DEFAULT_NS ) else \
             'non-default-vrf'

   def getOrCreateVrfConfig( self, vrfName ):
      return generalConfig.vrfConfig.newMember( vrfName )

   def deleteVrfConfig( self, vrfName ):
      # If therere any registed cleanup handlers call them
      routerGeneralVrfCleanupHook.notifyExtensions( vrfName )
      del ucmpStaticVrfConfigDir.vrfConfig[ vrfName ]
      del generalConfig.vrfConfig[ vrfName ]
      del prioritizationConfig.vrfConfig[ vrfName ]

def getOrCreateMetricProfile( profileName ):
   return generalConfig.metricProfile.newMember( profileName )

def delMetricProfile( profileName ):
   del generalConfig.metricProfile[ profileName ]

class RouterGeneralMetricProfileMode( RouterGeneralMetricProfileBaseMode,
                                      BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.

   Currently, only OSPF2 supports the router general metric profiles. The variable
   PROFILE_RULE_LENGTH in /src/gated/gated-ctk/src/new_ospf/new_ospf_main.h
   is depended on the length of longest possible metric rule. As such, if the
   router general metric profiles are integrated into another protocol with higher
   MAX_METRIC than 65535, the value of PROFILE_RULE_LENGTH may need to be adjusted
   accordingly.
   '''
   name = 'Protocol independent routing metric profile configuration'

   def __init__( self, parent, session, profileName, metricProfileConfig ):
      self.profileName = profileName
      self.metricProfileConfig = metricProfileConfig
      RouterGeneralMetricProfileBaseMode.__init__( self, self.profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setBaseMetric( self, args ):
      self.metricProfileConfig.baseMetric = args.get( 'METRIC',
            self.metricProfileConfig.metricDefault )

   def addMetricThresholdRule( self, args ):
      metric = args[ 'METRIC' ]
      speed = args[ 'SPEED' ]
      speedUnit = args[ 'UNIT' ]
      newMetricThesholdRule = Tac.Value( "Routing::General::MetricRule", metric,
                                         speed, speedUnit )
      actualSpeed = newMetricThesholdRule.actualSpeed
      if CliCommand.isNoOrDefaultCmd( args ):
         self._delMetricThresholdRule( metric, speed, speedUnit, actualSpeed )
      else :
         oldRule = self.metricProfileConfig.metricRules.get( actualSpeed, None )
         if oldRule:
            # pylint: disable-next=consider-using-f-string
            warning = "Multiple rules provided with same speed %d" % actualSpeed
            self.addWarning( warning )
            self._delMetricThresholdRule( oldRule.metric, oldRule.speed,
                                          oldRule.speedUnit, actualSpeed )
         self.metricProfileConfig.metricRules.addMember( newMetricThesholdRule )

   def _delMetricThresholdRule( self, metric, speed, speedUnit, actualSpeed ):
      rule = self.metricProfileConfig.metricRules.get( actualSpeed )
      if ( rule and rule.metric == metric and rule.speed == speed and
           rule.speedUnit == speedUnit ):
         del self.metricProfileConfig.metricRules[ actualSpeed ]

   def setMetricRatio( self, args ):
      speed = args[ 'SPEED' ]
      speedUnit = args[ 'UNIT' ]
      ratio = Tac.Value( "Routing::General::MetricRatio", speed, speedUnit )
      self.metricProfileConfig.metricRatio = ratio

   def delMetricRatio( self, args ):
      ratio = Tac.Value( "Routing::General::MetricRatio" )
      self.metricProfileConfig.metricRatio = ratio

class RoutePriorityHelper:
   def __init__( self, config, vrfSubMode ):
      self.config = config
      self.globalConfig = prioritizationConfig.globalPriority
      self.vrfSubMode = vrfSubMode
      self.defaultPriority = Tac.Type(
         'Routing::VrfPrioritizationConfig' ).defaultPriority().protocolPriority

   def addOrUpdateProtocol( self, args ):
      proto = args[ 'PROTO_NAME' ]
      mappedPriority = {
         'low': 'routePriorityLow',
         'medium': 'routePriorityMedium',
         'high': 'routePriorityHigh'
      }[ args[ 'PRIORITY_VALUE' ] ]
      d = self.defaultPriority[ proto ]
      if mappedPriority == d and not self.vrfSubMode:
         # Trying to set the priority to default value should only be
         # allowed in VRF submode
         del self.config.protocolPriority[ proto ]
      else:
         self.config.protocolPriority[ proto ] = \
            Tac.enumValue( 'Routing::RoutePriority', mappedPriority )

   def delProtocol( self, args ):
      proto = args[ 'PROTO_NAME' ]
      del self.config.protocolPriority[ proto ]

class RouterGeneralRoutePriorityMode( RouterGeneralRoutePriorityBaseMode,
                                      RoutePriorityHelper,
                                      BasicCli.ConfigModeBase ):
   name = 'Route priority configuration'

   def __init__( self, parent, session, config ):
      RouterGeneralRoutePriorityBaseMode.__init__( self, param='priorityConfig' )
      RoutePriorityHelper.__init__( self, config, vrfSubMode=False )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterGeneralVrfRoutePriorityMode( RouterGeneralVrfRoutePriorityBaseMode,
                                         RoutePriorityHelper,
                                         BasicCli.ConfigModeBase ):
   name = 'Route priority configuration in VRF submode'

   def __init__( self, parent, session, config, vrfName ):
      self.vrfName = vrfName
      RouterGeneralVrfRoutePriorityBaseMode.__init__( self, self.vrfName )
      RoutePriorityHelper.__init__( self, config, vrfSubMode=True )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def deleteVrfConfig( self, vrfName ):
      del prioritizationConfig.vrfConfig[ vrfName ]

def getOrCreateFilteredRibCfg( vrfName, addrFamily, filteredRibName, rcfName=None ):
   vrfConfig = generalConfig.vrfConfig.get( vrfName )
   filteredRib = vrfConfig.filteredRib.get( filteredRibName )
   if ( not filteredRib or
        filteredRib.af != addrFamily or
        ( rcfName is not None and filteredRib.rcfName != rcfName ) ):
      # Create a new filteredRib if at least one of the attributes does not match
      rcfName = rcfName or '' # Use empty rcfName if rcfName is None
      filteredRib = Tac.Value( "Routing::General::FilteredRibConfig",
                               filteredRibName, addrFamily, rcfName )
      vrfConfig.filteredRib.addMember( filteredRib )
   return filteredRib

class RouterGeneralVrfFilteredRibMode( RouterGeneralVrfFilteredRibBaseMode,
                                       BasicCli.ConfigModeBase ):
   name = 'Filtered RIB configuration in VRF submode'

   def __init__( self, parent, session, vrfName, addrFamily, filteredRibName ):
      self.vrfName = vrfName
      self.addrFamily = addrFamily
      self.filteredRibName = filteredRibName
      param = ( vrfName, addrFamily, filteredRibName )
      RouterGeneralVrfFilteredRibBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def updateRcfName( self, rcfName ):
      getOrCreateFilteredRibCfg( self.vrfName, self.addrFamily, self.filteredRibName,
                                 rcfName )

class RouterGeneralAutoVersionMode( RouterGeneralAutoVersionBaseMode,
                                    BasicCli.ConfigModeBase ):
   name = "Auto Version Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RouterGeneralAutoVersionBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#----------------------------------------------------------------------------------
#                                CLI MODELET
# A modelet for global / per vrf modes each.
#----------------------------------------------------------------------------------
class RouterGeneralBaseModelet( CliParser.Modelet ):
   pass

class RouterGeneralModelet( RouterGeneralBaseModelet ):
   pass

class RouterGeneralVrfModelet( RouterGeneralBaseModelet ):
   pass

class RouterGeneralMetricProfileModelet( RouterGeneralBaseModelet ):
   pass

class RouterGeneralRoutePriorityModelet( RouterGeneralBaseModelet ):
   pass

class RouterGeneralVrfRoutePriorityModelet( RouterGeneralVrfModelet ):
   pass

class RouterGeneralVrfFilteredRibModelet( RouterGeneralVrfModelet ):
   pass

class RouterGeneralAutoVersionModelet( RouterGeneralBaseModelet ):
   pass

#----------------------------------------------------------------------------------
#                                K E Y W O R D S
#----------------------------------------------------------------------------------
matcherGeneral = CliMatcher.KeywordMatcher( 'general',
   'Protocol independent routing configuration' )
matcherNexthops  = CliMatcher.KeywordMatcher( 'next-hops',
   'Next hop configuration' )
matcherFec       = CliMatcher.KeywordMatcher( 'fec',
   'Forward Equivalence Class configuration' )
matcherDedicated = CliMatcher.KeywordMatcher( 'dedicated',
   'Use dedicated FEC per next hop' )
matcherHierarchical = CliMatcher.KeywordMatcher( 'hierarchical',
   'Use IP over IP Hierarchical FEC' )
matcherResolvingProtocol = CliMatcher.KeywordMatcher( 'protocol',
   'Protocol of resolving route' )
matcherIpv4Config = CliMatcher.KeywordMatcher( 'ipv4',
    'IPv4 configuration commands' )
nodeIpv4Config = CliCommand.Node( matcher=matcherIpv4Config,
                                  guard=hwSupportedGuard )
matcherIpv6Config = CliMatcher.KeywordMatcher( 'ipv6',
    'IPv6 configuration commands' )
matcherRoutable = CliMatcher.KeywordMatcher( "routable",
    'Enable reserved-address IPv4 routing' )
matcher240_4 = CliMatcher.KeywordMatcher( "240.0.0.0/4",
    'Enable reserved-address IPv4 routing for 240.0.0.0/4' )
matcher0_8 = CliMatcher.KeywordMatcher( "0.0.0.0/8",
    'Enable reserved-address IPv4 routing for 0.0.0.0/8' )
matcherRouterId = CliMatcher.KeywordMatcher( 'router-id',
   'Configure a general router ID for all routing processes' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
   'General router ID in IP address format' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
   'General router ID in IPv6 address format' )
matcherV4Addr = IpAddr.IpAddrMatcher(
   "General router ID in IP address format" )
matcherV6Addr = Ip6Addr.Ip6AddrMatcher(
   "General router ID in IPv6 address format" )
matcherResolution = CliMatcher.KeywordMatcher(
   'resolution', helpdesc='Next hop recursive resolution' )
matcherRoute = CliMatcher.KeywordMatcher( 'route',
   'Route commands' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
   'Static routes' )
matcherTrack = CliMatcher.KeywordMatcher(
   'track', 'Track route programming for a prefix' )
matcherNexthopGroup = CliMatcher.KeywordMatcher( 'nexthop-group',
   'Nexthop groups' )
matcherUnresolved = CliMatcher.KeywordMatcher( 'unresolved',
   'Unreachable destinations' )
matcherInvalid = CliMatcher.KeywordMatcher( 'invalid',
   'Do not install in routing table' )
matcherSoftware = CliMatcher.KeywordMatcher( 'software',
   'Software configuration' )
matcherForwarding = CliMatcher.KeywordMatcher( 'forwarding',
   'Configure packet forwarding' )
matcherHardware = CliMatcher.KeywordMatcher( 'hardware',
   'Configure hardware forwarding of software packets')
matcherOffload = CliMatcher.KeywordMatcher( 'offload',
   'Configure hardware offload of software packets')
matcherMtu = CliMatcher.KeywordMatcher( 'mtu',
   'Configure MTU value used for software forwarded packets that use hardware '
   'offload' )
matcherPreference = CliMatcher.KeywordMatcher( 'preference',
   'Administrative distance for static routes' )
matcherPreferenceVal = CliMatcher.IntegerMatcher( 1, 255,
   helpdesc='Admin-distance/Preference value' )
matcherProfileName = CliMatcher.PatternMatcher( '.{1,100}',
   helpname='WORD', helpdesc='General metric profile name' )
matcherMetric = CliMatcher.KeywordMatcher( 'metric',
   helpdesc='Sets metric for Metric Profile' )
matcherMetricValue = CliMatcher.IntegerMatcher( 1, 65535,
   helpdesc='Value of the route metric' )
matcherInterfaceSpeed = CliMatcher.IntegerMatcher( 1, 4294967295,
   helpdesc='Interface speed' )
matcherSpeedUnit = CliMatcher.EnumMatcher( {
   'mbps' : 'Megabits per second',
   'gbps' : 'Gigabits per second',
})
matcherFilteredRibName = CliMatcher.PatternMatcher( namePattern,
   helpname='WORD', helpdesc='Name of the unicast RIB' )

#----------------------------------------------------------------------------------
#                               C O M M A N D S
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# "[ no | default ] next-hops fec dedicated"
# in "router general" mode
#--------------------------------------------------------------------------------
class NexthopsFecDedicatedCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops fec dedicated'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'fec' : matcherFec,
      'dedicated' : matcherDedicated,
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecPerNexthop = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecPerNexthop = False

RouterGeneralModelet.addCommandClass( NexthopsFecDedicatedCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp emulated"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibEcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp emulated'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'emulated': 'Emulate',
   }

   @staticmethod
   def handler( mode, args):
      generalConfig.fecEcmpEmulated = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpEmulated = False

RouterGeneralModelet.addCommandClass( RibFibEcmpCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp ordered"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibOrderedCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp ordered'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'ordered': CliCommand.guardedKeyword( 'ordered',
                                            'Enforce order of next hops in FEC',
                                            guard=orderedEcmpSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecEcmpOrdered = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpOrdered = False

RouterGeneralModelet.addCommandClass( RibFibOrderedCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec hierarchical resolution protocol static"
# in "router general" mode
#--------------------------------------------------------------------------------
class HfecOnResolvingStaticCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec hierarchical resolution protocol static'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'hierarchical' : matcherHierarchical,
      'resolution' : matcherResolution,
      'protocol' : matcherResolvingProtocol,
      'static' : 'Use HFEC for resolving routes over static'
   }
   handler = ( 'RouterGeneralCliHandler.setHfecOnResolvingStatic' )
   noOrDefaultHandler = ( 'RouterGeneralCliHandler.noHfecOnResolvingStatic' )

if IpRibLibToggleLib.toggleHfecSupportEnabled():
   RouterGeneralModelet.addCommandClass( HfecOnResolvingStaticCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec hierarchical resolution"
# in "router general" mode
#--------------------------------------------------------------------------------
class IpOverIpHfecCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec hierarchical resolution'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'hierarchical' : CliCommand.Node(
                           matcherHierarchical,
                           guard=hierarchicalFecSupportedGuard ),
      'resolution' : matcherResolution,
   }
   handler = ( 'RouterGeneralCliHandler.setIpOverIpHfec' )
   noOrDefaultHandler = ( 'RouterGeneralCliHandler.noIpOverIpHfec' )

if RoutingLibToggleLib.toggleIpOverIpHfecEnabled():
   RouterGeneralModelet.addCommandClass( IpOverIpHfecCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] route static bfd map admin-down down"
# in "router general" mode
#--------------------------------------------------------------------------------
class BfdMapAdminDownDownCmd( CliCommand.CliCommandClass ):
   syntax = 'route static bfd map admin-down down'
   noOrDefaultSyntax = syntax
   data = {
      'route': 'Route commands',
      'static': 'Static routes',
      'bfd': 'Configure BFD parameters',
      'map' : 'Treat the first session state change as the second one',
      'admin-down' : 'Local or remote session Admin-Down state',
      'down' : 'Local session Down state',
   }
   handler = ( 'RouterGeneralCliHandler.bfdMapAdminDownDownHandler' )
   noOrDefaultHandler = ( 'RouterGeneralCliHandler.noBfdMapAdminDownDownHandler' )

RouterGeneralModelet.addCommandClass( BfdMapAdminDownDownCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] route static ucmp forwarding \
#                   [ fec maximum-size MAXUCMP [ deviation DEVIATION percent ] ]"
# in "router general vrf" mode
#--------------------------------------------------------------------------------
class StaticUcmpForwardingCmd( CliCommand.CliCommandClass ):
   syntax = 'route static ucmp forwarding \
             [ fec maximum-size MAXUCMP [ deviation DEVIATION percent ] ]'
   noOrDefaultSyntax = 'route static ucmp forwarding ...'
   data = {
      'route': 'Route commands',
      'static': 'Static routes',
      'ucmp': ucmpNode,
      'forwarding': 'UCMP forwarding to FIB',
      'fec': 'FEC maximum size',
      'maximum-size': 'Value for total number of UCMP nexthops',
      'MAXUCMP': ucmpSizeRange,
      'deviation': 'Oversubscription deviation allowed',
      'DEVIATION': ucmpDevRangeMatcher,
      'percent': 'deviation as a percentage',
   }
   handler = "RouterGeneralCliHandler.ucmpModeCmdHandler"
   noOrDefaultHandler = "RouterGeneralCliHandler.ucmpModeCmdNoDefaultHandler"
if RouterGeneralCliToggleLib.toggleStaticUcmpLocalForwardingEnabled():
   RouterGeneralVrfModelet.addCommandClass( StaticUcmpForwardingCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ipv4 routable 240.0.0.0/4"
# in "router general" mode
#--------------------------------------------------------------------------------
class Ipv4ReservedClassEAddressingCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv4 routable 240.0.0.0/4'''
   noOrDefaultSyntax = syntax

   data = { "ipv4": nodeIpv4Config,
            "routable": matcherRoutable,
            "240.0.0.0/4": matcher240_4,
          }

   @staticmethod
   def handler( mode, args ):
      generalConfig.ipv4Routable240ClassE = True
      addIpv4RoutableReservedRebootWarning( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.ipv4Routable240ClassE = False
      addIpv4RoutableReservedRebootWarning( mode )

RouterGeneralModelet.addCommandClass( Ipv4ReservedClassEAddressingCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ipv4 routable 0.0.0.0/8"
# in "router general" mode
#--------------------------------------------------------------------------------
class Ipv4Reserved0ClassAAddressingCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv4 routable 0.0.0.0/8'''
   noOrDefaultSyntax = syntax

   data = { "ipv4": nodeIpv4Config,
            "routable": matcherRoutable,
            "0.0.0.0/8": matcher0_8,
          }

   @staticmethod
   def handler( mode, args ):
      generalConfig.ipv4Routable0ClassA = True
      addIpv4RoutableReservedRebootWarning( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.ipv4Routable0ClassA = False
      addIpv4RoutableReservedRebootWarning( mode )

RouterGeneralModelet.addCommandClass( Ipv4Reserved0ClassAAddressingCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] router-id ipv4 <ip-address>"
# in "router general/ vrf" mode
#--------------------------------------------------------------------------------
class RouterIdIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 IP_ADDR'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id': matcherRouterId,
      'ipv4': matcherIpv4,
      'IP_ADDR': matcherV4Addr,
   }

   @staticmethod
   def handler( mode, args ):
      routerIdV4 = args[ 'IP_ADDR' ]
      if routerIdV4 == defaultRouterIdV4:
         # pylint: disable-next=consider-using-f-string
         mode.addError( '%s is not a valid router ID' % routerIdV4 )
         return
      routerConfig = getRouterConfig( mode.vrfName )
      routerConfig.routerIdV4 = routerIdV4

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerConfig = getRouterConfig( mode.vrfName )
      routerConfig.routerIdV4 = defaultRouterIdV4

RouterGeneralModelet.addCommandClass( RouterIdIpv4Cmd )
RouterGeneralVrfModelet.addCommandClass( RouterIdIpv4Cmd )

#--------------------------------------------------------------------------------
# "[ no | default ] router-id ipv6 <ipv6-address>" command in both "router general"
# in "router general/ vrf" mode
#--------------------------------------------------------------------------------
class RouterIdIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 IP_ADDR'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id': matcherRouterId,
      'ipv6': matcherIpv6,
      'IP_ADDR': matcherV6Addr,
   }

   @staticmethod
   def handler( mode, args ):
      routerIdV6 = args[ 'IP_ADDR' ]
      if routerIdV6 == defaultRouterIdV6:
         # pylint: disable-next=consider-using-f-string
         mode.addError( '%s is not a valid router ID' % routerIdV6 )
         return
      routerConfig = getRouterConfig( mode.vrfName )
      routerConfig.routerIdV6 = routerIdV6

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerConfig = getRouterConfig( mode.vrfName )
      routerConfig.routerIdV6 = defaultRouterIdV6

RouterGeneralModelet.addCommandClass( RouterIdIpv6Cmd )
RouterGeneralVrfModelet.addCommandClass( RouterIdIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] router general
#--------------------------------------------------------------------------------
class RouterGeneralCmd( CliCommand.CliCommandClass ):
   syntax = 'router general'
   noOrDefaultSyntax = 'router general ...'
   data = {
      'router': routerMatcherForConfig,
      'general': 'Protocol independent routing configuration',
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go from 'config' mode to 'router general' mode
      '''
      childMode = mode.childMode( RouterGeneralMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecPerNexthop = False
      generalConfig.fecEcmpEmulated = False
      generalConfig.fecEcmpOrdered = False
      generalConfig.fecEcmpMixTunnelIp = False
      generalConfig.fwdDevMtu = RoutingGeneralConfig.defaultFwdDevMtu
      generalConfig.hfecOnResolvingProto.clear()
      generalConfig.ipOverIpHfec = False
      generalConfig.bfdMapAdminDownDown = False
      generalConfig.routerIdV4 = defaultRouterIdV4
      generalConfig.routerIdV6 = defaultRouterIdV6
      generalConfig.resolveOverAggregates = False
      generalConfig.aigpStaticAccumulate = False
      generalConfig.aigpRouteInputAccumulate = False
      generalConfig.resolveNexthopGroupVias = False
      generalConfig.ipv4Routable240ClassE = False
      generalConfig.ipv4Routable0ClassA = False
      generalConfig.prefixAutoExplicitNull = False
      generalConfig.routeStaticGlobalPreferencePresent = False
      generalConfig.ucmpEligible = False
      generalConfig.routeStaticGlobalPreference = Tac.Value(
         "Routing::RoutePreference" )
      routerGeneralCleanupHook.notifyExtensions( mode=mode )
      autoVersionConfig.nhgAutoVersion = False
      childMode = mode.childMode( RouterGeneralMode )
      childMode.deleteModeConfig()

BasicCliModes.GlobalConfigMode.addCommandClass( RouterGeneralCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfCli.VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                    inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go from 'router general' mode to vrf submode
      '''
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( RouterGeneralVrfMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateVrfConfig( vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( RouterGeneralVrfMode, vrfName=vrfName )
      childMode.deleteVrfConfig( vrfName=vrfName )

RouterGeneralMode.addCommandClass( VrfCmd )

RouterGeneralMode.addModelet( RouterGeneralModelet )
RouterGeneralVrfMode.addModelet( RouterGeneralVrfModelet )

#--------------------------------------------------------------------------------
# [ no | default ] metric profile PROFILE_NAME
#--------------------------------------------------------------------------------
_rtrGeneralMetricDesc = 'Metric related configuration'
_metricProfileDesc = 'Define metric profile'
class MetricProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'metric profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'metric' : _rtrGeneralMetricDesc,
      'profile' : _metricProfileDesc,
      'PROFILE_NAME' : matcherProfileName
   }

   @staticmethod
   def handler( mode, args ):
      '''Function to go from 'router general' mode to metric profile mode
      '''
      profileName = args[ 'PROFILE_NAME' ]
      metricProfileConfig = getOrCreateMetricProfile( profileName )
      childMode = mode.childMode( RouterGeneralMetricProfileMode,
                                  profileName=profileName,
                                  metricProfileConfig=metricProfileConfig)
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      delMetricProfile( profileName )

RouterGeneralMode.addCommandClass( MetricProfileCmd )
RouterGeneralMetricProfileMode.addModelet( RouterGeneralMetricProfileModelet )

#--------------------------------------------------------------------------------
# [ no | default ] next-hops resolution bgp aggregates allowed
#--------------------------------------------------------------------------------
class NexthopResAggsAllowedCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops resolution bgp aggregates allowed'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'resolution': matcherResolution,
      'bgp': 'Recursive resolution over BGP aggregates',
      'aggregates': 'Recursive resolution over aggregates',
      'allowed': 'Recursive resolution over aggregates',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.resolveOverAggregates = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.resolveOverAggregates = False

RouterGeneralMode.addCommandClass( NexthopResAggsAllowedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] next-hops resolution static aigp-metric accumulated
#--------------------------------------------------------------------------------
class NexthopResStaticAigpCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops resolution static aigp-metric accumulated'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'resolution': matcherResolution,
      'static': 'Next hop resolution behavior of static routes',
      'aigp-metric': 'AIGP distance accumulation over static route next hops',
      'accumulated': 'AIGP distance accumulation over static route next hops',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.aigpStaticAccumulate = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.aigpStaticAccumulate = False

RouterGeneralMode.addCommandClass( NexthopResStaticAigpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] next-hops resolution user aigp-metric accumulated
#--------------------------------------------------------------------------------
class NexthopResUserAigpCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops resolution user aigp-metric accumulated'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'resolution': matcherResolution,
      'user': 'Next hop resolution behavior of user-generated SDK routes',
      'aigp-metric': 'AIGP distance accumulation over SDK route next hops',
      'accumulated': 'AIGP distance accumulation over SDK route next hops',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.aigpRouteInputAccumulate = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.aigpRouteInputAccumulate = False

RouterGeneralMode.addCommandClass( NexthopResUserAigpCmd )


#--------------------------------------------------------------------------------
# [ no | default ] route static nexthop-group unresolved invalid
#--------------------------------------------------------------------------------
class RouteStaticNexthopGroupUnresolvedCmd( CliCommand.CliCommandClass ):
   syntax = 'route static nexthop-group unresolved invalid'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'static': matcherStatic,
      'nexthop-group': CliCommand.Node(
                           matcherNexthopGroup,
                           guard=nexthopGroupSupportedGuard ),
      'unresolved': matcherUnresolved,
      'invalid': matcherInvalid,
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.resolveNexthopGroupVias = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.resolveNexthopGroupVias = False

RouterGeneralMode.addCommandClass( RouteStaticNexthopGroupUnresolvedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] route static preference <PREFERENCE>
#--------------------------------------------------------------------------------
class RouteStaticGlobalPreferenceCmd( CliCommand.CliCommandClass ):
   syntax = 'route static ipv4 ipv6 preference PREFERENCE'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'static': matcherStatic,
      'ipv4': matcherIpv4,
      'ipv6': matcherIpv6,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceVal,
   }

   @staticmethod
   def handler( mode, args ):
      if 'PREFERENCE' in args:
         generalConfig.routeStaticGlobalPreference = Tac.Value(
            "Routing::RoutePreference", args[ 'PREFERENCE' ] )
         generalConfig.routeStaticGlobalPreferencePresent = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.routeStaticGlobalPreference = Tac.Value(
         "Routing::RoutePreference" )
      generalConfig.routeStaticGlobalPreferencePresent = False

RouterGeneralMode.addCommandClass( RouteStaticGlobalPreferenceCmd )

def mtuRangeFn( mode, context ):
   mtuMin = 68
   mtuMax = 65535
   try:
      cmd = [ "ip", "link", "show", 'txfwd' ]
      out = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.DISCARD, )
      mtuS = re.search( r'mtu (?P<mtu>\d+)', out )
      if mtuS:
         mtu = int( mtuS.group( 'mtu' ) )
         if mtu > mtuMin:
            mtuMax = mtu
   except Tac.SystemCommandError:
      t5( "mtuRangeFn txfwd interface not found" )

   return mtuMin, mtuMax

#--------------------------------------------------------------------------------
# "[ no | default ] software forwarding hardware offload mtu <MTU value>" command
# in both "router general" and "router general/vrf" mode
#--------------------------------------------------------------------------------
class softwareForwardingDeviceMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'software forwarding hardware offload mtu MTU'
   noOrDefaultSyntax = 'software forwarding hardware offload mtu ...'
   data = {
      'software' : matcherSoftware,
      'forwarding': matcherForwarding,
      'hardware' : matcherHardware,
      'offload' : matcherOffload,
      'mtu' : matcherMtu,
      'MTU' : CliMatcher.DynamicIntegerMatcher( rangeFn=mtuRangeFn,
                                                helpdesc='MTU value' )
   }

   @staticmethod
   def handler( mode, args ):
      mtu = args[ 'MTU' ]
      routerConfig = getRouterConfig( mode.vrfName )
      routerConfig.fwdDevMtu = mtu

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerConfig = getRouterConfig( mode.vrfName )
      # Vrf config has to be reset to 0 to indicate that mtu value
      # has to be inherited from global config.
      inherited = 0
      default = RoutingGeneralConfig.defaultFwdDevMtu
      routerConfig.fwdDevMtu = inherited if mode.vrfName else default

RouterGeneralModelet.addCommandClass( softwareForwardingDeviceMtuCmd )
RouterGeneralVrfModelet.addCommandClass( softwareForwardingDeviceMtuCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp compatible tunnel ip"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibEcmpMixTunnelIpCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp compatible tunnel ip'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'tunnel': 'Tunnel Vias',
      'compatible' : 'Compatible',
      'ip': 'IP Vias',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecEcmpMixTunnelIp = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpMixTunnelIp = False

RouterGeneralModelet.addCommandClass( RibFibEcmpMixTunnelIpCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib ucmp eligibility mode link-bandwidth all "
# in "router general" mode
#--------------------------------------------------------------------------------
class RibUcmpEligibleCmd( CliCommand.CliCommandClass ):
   syntax = 'rib ucmp eligibility ( mode link-bandwidth all )'
   noOrDefaultSyntax = 'rib ucmp eligibility ...'
   data = {
      'rib': 'Routing table',
      'ucmp': ucmpNode,
      'eligibility': 'Eligibility',
      'mode': 'UCMP Mode',
      'link-bandwidth': 'Enforce UCMP based on link-bandwidth of next hops',
      'all': 'All',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.ucmpEligible = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.ucmpEligible = False

if IpRibLibToggleLib.toggleUcmpEligibleEnabled():
   RouterGeneralModelet.addCommandClass( RibUcmpEligibleCmd )


#--------------------------------------------------------------------------------
# "[ no | default ] route forwarding label explicit-null tunnel af-mismatch
# in "router general" mode
#--------------------------------------------------------------------------------
class RoutePrefixAutoExplicitNullCmd( CliCommand.CliCommandClass ):
   syntax = 'route forwarding label explicit-null tunnel af-mismatch'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'forwarding' : 'Route forwarding options',
      'label' : 'Push an MPLS label',
      'explicit-null' : 'Push Explicit NULL label based on route\'s address family',
      'tunnel' : 'Apply to routes resolving with next hop resolving via a tunnel',
      'af-mismatch' : 'Apply if the route\'s address family is different from' +
                      ' the tunnel\'s address family'
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.prefixAutoExplicitNull = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.prefixAutoExplicitNull = False

RouterGeneralModelet.addCommandClass( RoutePrefixAutoExplicitNullCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] metric <METRIC>
# in "router general metric profile" mode
#--------------------------------------------------------------------------------
class MetricProfileBaseMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC'
   noOrDefaultSyntax = 'metric ...'
   data = {
      'metric' : matcherMetric,
      'METRIC' : matcherMetricValue
   }
   handler = RouterGeneralMetricProfileMode.setBaseMetric
   noOrDefaultHandler = handler

RouterGeneralMetricProfileMode.addCommandClass( MetricProfileBaseMetricCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] metric <metric> if speed <= <speed> <mbps|gbps>
# in "router general metric profile" mode
#--------------------------------------------------------------------------------
_metricRuleIfDesc = 'Use given metric when interface matches condition'
_metricRuleSpeedDesc = 'Use given metric when interface speed matches condition'
_lessThanOrEqualToDesc = 'Less than or equal to'

class MetricProfileMetricThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC if speed <= SPEED UNIT'
   noOrDefaultSyntax = 'metric METRIC if speed <= SPEED UNIT'
   data = {
      'metric' : matcherMetric,
      'METRIC' : matcherMetricValue,
      'if' : _metricRuleIfDesc,
      'speed' : _metricRuleSpeedDesc,
      '<=' : _lessThanOrEqualToDesc,
      'SPEED' : matcherInterfaceSpeed,
      'UNIT' : matcherSpeedUnit,
   }
   handler = RouterGeneralMetricProfileMode.addMetricThresholdRule
   noOrDefaultHandler = handler

RouterGeneralMetricProfileMode.addCommandClass( MetricProfileMetricThresholdCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] metric ratio <speed> <mbps|gbps>
# in "router general metric profile" mode
#--------------------------------------------------------------------------------
_metricRuleDesc = 'Given metric is used when interface speed matches condition'

class MetricProfileMetricRatioCmd( CliCommand.CliCommandClass ):
   syntax = 'metric ratio SPEED UNIT'
   noOrDefaultSyntax = 'metric ratio ...'
   data = {
      'metric' : matcherMetric,
      'ratio' : 'Sets metric inversely proportional to given <speed>',
      'SPEED' : matcherInterfaceSpeed,
      'UNIT' : matcherSpeedUnit,
   }
   handler = RouterGeneralMetricProfileMode.setMetricRatio
   noOrDefaultHandler = RouterGeneralMetricProfileMode.delMetricRatio

RouterGeneralMetricProfileMode.addCommandClass( MetricProfileMetricRatioCmd )

matcherPriority = CliMatcher.KeywordMatcher( 'priority',
   'Enter route priority sub-mode' )

#--------------------------------------------------------------------------------
# [ no | default ] route priority
#--------------------------------------------------------------------------------
class RoutePriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'route priority'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'priority': matcherPriority,
   }

   @staticmethod
   def handler( mode, args ):
      '''Function to go from 'router general' mode to route priority
      mode'''
      childMode = mode.childMode( RouterGeneralRoutePriorityMode,
                                  config=prioritizationConfig.globalPriority )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      prioritizationConfig.globalPriority.protocolPriority.clear()

if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
   RouterGeneralMode.addCommandClass( RoutePriorityCmd )
   RouterGeneralRoutePriorityMode.addModelet( RouterGeneralRoutePriorityModelet )

#--------------------------------------------------------------------------------
# [ no | default ] route priority
# in "router general" VRF submode
#--------------------------------------------------------------------------------
class RoutePriorityVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'route priority'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'priority': matcherPriority,
   }

   @staticmethod
   def handler( mode, args ):
      '''Function to go from 'router general' VRF submode to route
      priority mode
      '''
      childMode = mode.childMode(
         RouterGeneralVrfRoutePriorityMode,
         config=prioritizationConfig.vrfConfig.newMember( mode.vrfName ),
         vrfName=mode.vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del prioritizationConfig.vrfConfig[ mode.vrfName ]

if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
   RouterGeneralVrfMode.addCommandClass( RoutePriorityVrfCmd )
   RouterGeneralVrfRoutePriorityMode.addModelet(
      RouterGeneralVrfRoutePriorityModelet )

#--------------------------------------------------------------------------------
# "[ no | default ] protocol PROTO_NAME PRIORITY_VALUE
# in "router general / vrf route priority" mode
#--------------------------------------------------------------------------------
class RoutePriorityProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol PROTO_NAME PRIORITY_VALUE'
   noOrDefaultSyntax = 'protocol PROTO_NAME ...'
   _defaultPriorityEntity = Tac.Type(
      'Routing::VrfPrioritizationConfig' ).defaultPriority()
   _routePrioProtocols = _defaultPriorityEntity.protocolPriority.keys()
   data = {
      'protocol' : CliMatcher.KeywordMatcher(
         'protocol', 'Protocol specific configuration' ),
      'PROTO_NAME' : CliMatcher.EnumMatcher(
         { proto : f'Configure default priority for {proto}'
           for proto in _routePrioProtocols } ),
      'PRIORITY_VALUE' : CliMatcher.EnumMatcher( {
         'low': 'Treat as normal priority, use for most routes',
         'medium': 'Treat at elevated priority, use for infrastructure routes',
         'high': 'Treat as highest priority, use for critical routes',
      } ),
   }
   handler = RoutePriorityHelper.addOrUpdateProtocol
   noOrDefaultHandler = RoutePriorityHelper.delProtocol

if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
   RouterGeneralRoutePriorityMode.addCommandClass( RoutePriorityProtocolCmd )
   RouterGeneralVrfRoutePriorityMode.addCommandClass( RoutePriorityProtocolCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] unicast-rib ( ipv4 | ipv6 ) RIB_NAME"
# in "router general" VRF submode
#--------------------------------------------------------------------------------
class FilteredRibCmd( CliCommand.CliCommandClass ):
   syntax = 'unicast-rib ( ipv4 | ipv6 ) RIB_NAME'
   noOrDefaultSyntax = syntax
   data = {
     'unicast-rib': 'Custom unicast RIB',
     'ipv4': matcherIpv4Config,
     'ipv6': matcherIpv6Config,
     'RIB_NAME': matcherFilteredRibName,
   }
   handler = 'RouterGeneralCliHandler.filteredRibHandler'
   noOrDefaultHandler = 'RouterGeneralCliHandler.filteredRibNoCmdHandler'

if IpRibLibToggleLib.toggleIsisConditionalAdvertiseEnabled():
   RouterGeneralVrfMode.addCommandClass( FilteredRibCmd )
   RouterGeneralVrfFilteredRibMode.addModelet(
      RouterGeneralVrfFilteredRibModelet )

#--------------------------------------------------------------------------------
# "[ no | default ] match rcf FUNCTION
# in "router general" VRF unicast-rib submode
#--------------------------------------------------------------------------------
class FilteredRibMatchRcfCmd( CliCommand.CliCommandClass ):
   syntax = 'match rcf FUNCTION'
   noOrDefaultSyntax = 'match rcf ...'
   data = {
      'match': 'Unicast-rib match rule',
      'rcf': 'Routing control function',
      'FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = 'RouterGeneralCliHandler.filteredRibMatchRcfHandler'
   noOrDefaultHandler = handler

RouterGeneralVrfFilteredRibModelet.addCommandClass( FilteredRibMatchRcfCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] route event track vrf VRF <prefix>
# in "router general" mode
#--------------------------------------------------------------------------------
class RouteTrackPrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'route event track VRF PREFIX'
   noOrDefaultSyntax = 'route event track ...'
   hidden = True
   data = {
      'route' : matcherRoute,
      'event' : CliMatcher.KeywordMatcher(
         'event', 'Track events associated with the prefix' ),
      'track' : matcherTrack,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='VRF name',
                                     inclDefaultVrf=True, inclAllVrf=True ),
      'PREFIX': IpGenPrefixMatcher( helpdesc='Prefix to match',
                                    helpdesc4='IPv4 prefix to match',
                                    helpdesc6='IPv6 prefix to match' ),
   }
   handler = 'RouterGeneralCliHandler.routeTrackHandler'
   noOrDefaultHandler = 'RouterGeneralCliHandler.routeTrackNoCmdHandler'

if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
   RouterGeneralModelet.addCommandClass( RouteTrackPrefixCmd )

#--------------------------------------------------------------------------------
# "clear route event track"
# in enable mode
#--------------------------------------------------------------------------------
class ClearRouteTrackCmd( CliCommand.CliCommandClass ):
   syntax = 'clear route event track'
   hidden = True
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'route' : matcherRoute,
      'event' : CliMatcher.KeywordMatcher(
         'event', 'Clear events associated with the prefix until now' ),
      'track' : matcherTrack
   }
   handler = 'RouterGeneralCliHandler.clearTrackingHandler'

if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
   BasicCli.EnableMode.addCommandClass( ClearRouteTrackCmd )

#--------------------------------------------------------------------------------
# "auto-version"
#--------------------------------------------------------------------------------
class AutoVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'auto-version'
   noOrDefaultSyntax = syntax
   data = {
      'auto-version': 'Auto version configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RouterGeneralAutoVersionMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      autoVersionConfig.nhgAutoVersion = False

RouterGeneralMode.addCommandClass( AutoVersionCmd )
RouterGeneralAutoVersionMode.addModelet( RouterGeneralAutoVersionModelet )

#--------------------------------------------------------------------------------
# "nexthop-group"
# in "auto-version" mode under "router general" mode
#--------------------------------------------------------------------------------
class NhgAutoVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'nexthop-group'
   noOrDefaultSyntax = syntax
   data = {
      'nexthop-group': CliCommand.guardedKeyword( 'nexthop-group',
                              'Enable nexthop group auto-versioning',
                              guard=nexthopGroupSupportedGuard )
   }
   hidden = not NexthopGroupToggleLib.toggleNhgAutoVersioningEnabled()

   @staticmethod
   def handler( mode, args ):
      autoVersionConfig.nhgAutoVersion = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      autoVersionConfig.nhgAutoVersion = False

RouterGeneralAutoVersionMode.addCommandClass( NhgAutoVersionCmd )

t5( 'Loaded RouterGeneralCli' )

def Plugin( entityManager ):
   global generalConfig
   global routingHwStatus
   global prioritizationConfig
   global ucmpStaticVrfConfigDir
   global autoVersionConfig

   generalConfig = ConfigMount.mount( entityManager, 'routing/general/config/global',
                                      'Routing::General::Config', 'w' )
   ucmpStaticVrfConfigDir = ConfigMount.mount( entityManager,
         'routing/ucmp/static/vrf/config', 'Routing::Ucmp::VrfUcmpConfigDir', 'w' )
   autoVersionConfig = ConfigMount.mount( entityManager,
         'routing/autoVersion/config', 'Routing::AutoVersion::Config', 'w' )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   prioritizationConfig = LazyMount.mount(
      entityManager,
      'routing/general/config/vrfPrioritization',
      'Routing::VrfPrioritizationConfigAll', 'w' )
