# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# CliPlugin module for BGP configuration commands

from __future__ import absolute_import, division, print_function

from itertools import product
from CliPlugin.RouteMapCli import (
   getPrefixListNames,
   getIpv6PrefixListNames,
)
from CliPlugin.IraServiceCli import (
   routingMatcherForConfig,
   getEffectiveProtocolModel,
)
from CliPlugin import (
   IpAddrMatcher,
   Ip6AddrMatcher,
   Ip6LlAddr,
   IraVrfCli,
)
from CliPlugin.BgpCliConstants import orrPositionNameValidRegex
from IpLibTypes import ProtocolAgentModelType
import Tac
import LazyMount
import ConfigMount
import CliCommand
import CliMatcher
import BasicCli
from BasicCliUtil import anyCaseRegex
import CliParser
from ArnetLib import asnStrToNum, bgpFormatAsn
from TypeFuture import TacLazyType
import CliExtensions
from CliMode.BgpCommon import (
   RoutingBgpBaseMode,
   RoutingBgpVrfMode,
   BgpFieldSetMappingsVrfMode,
   FieldSetIpv4BaseMode,
   FieldSetIpv6BaseMode,
   RoutingBgpBaseAfMode,
   RoutingBgpVrfAfMode,
   BgpOrrPositionMode,
   BgpRfdPolicyMode,
   BgpRouteDistinguisherMode,
)
from RouteMapLib import isAsdotConfigured
from BgpLib import (
   PeerConfigKey,
   cleanupBgpConfig,
   cleanupUcmpConfig,
   vpnTypeAttrMap,
   vpnTypeAttrMapInv,
   vpnAfTypeMapInv,
   NoOrDefault,
)
import BgpLib
from IpLibConsts import VRFNAMES_RESERVED
from IpLibConsts import DEFAULT_VRF
import AclCliLib
from socket import IPPROTO_TCP
import CliParserCommon
import CliToken.Service
import CliToken.IpRibLibCliTokens
import six
from Bunch import Bunch

bgpConfig = None
bgpVrfConfigDir = None
rdConfigInputDir = None
routingHardwareStatus = None
bgpVrfClearReqDir = None
BGP_PORT = 179
addPathSendAcceptOneApp = True
asnConfig = None
aclConfig = None
aclCpConfig = None
ucmpConfig = None
ucmpVrfConfigDir = None

redistInternalStateEnum = Tac.Type( "Routing::Bgp::RedistInternalState" )
MetricOutStateEnum = Tac.Type( "Routing::Bgp::MetricOutState" )
SendCommunityLinkBandwidthStateEnum = \
    Tac.Type( "Routing::Bgp::SendCommunityLinkBandwidthState" )
ExtendedNextHopCapabilityEnum = Tac.Type( "Routing::Bgp::ExtendedNextHopCapability" )
policyActionEnum = Tac.Type( "Routing::Bgp::PolicyActionType" )

mpDirOptions = {
   'in': 'Missing policy inbound direction',
   'out': 'Missing policy outbound direction'
}
mpActionOptions = {
   'permit': 'Permit all routes for specified direction with missing policy '
             'configuration',
   'deny': 'Deny all routes for specified direction with missing policy '
           'configuration',
   'deny-in-out': 'Deny all routes for both in/out directions with missing '
                  'policy configuration in specified direction'
}
mpActionValues = {
   'permit': policyActionEnum.policyActionPermit,
   'deny': policyActionEnum.policyActionDeny,
   'deny-in-out': policyActionEnum.policyActionDenyBoth
}

def rfdPolicyNameFn( mode ):
   # pylint: disable-next=unnecessary-comprehension
   return { policyName for policyName in bgpConfig.rfdPolicy.keys() }

# Pattern name matcher: please keep these alphabetically sorted
class RfdPolicyNameMatcher( CliMatcher.DynamicNameMatcher ):
   ''' Accept standard rfd policy names, and allow for certain keywords to be
   excluded as a match.'''
   def __init__( self ):
      self.excludeKeywords = [ 'disabled' ]
      excludePattern = ''.join( anyCaseRegex( "(?!" + k + "$)" )
            for k in self.excludeKeywords )
      pattern = excludePattern + r'[A-Za-z0-9_:{}\[\]-]*'
      CliMatcher.DynamicNameMatcher.__init__( self,
            rfdPolicyNameFn,
            pattern=pattern, helpname='WORD',
            helpdesc='Route flap damping policy name' )

rfdPolicyNameMatcher = RfdPolicyNameMatcher()

def mpIncludeOptions():
   includes = [] # Add includes alphabetically
   includes.append( 'community-list' )
   includes.append( 'prefix-list' )
   includes.append( 'sub-route-map' )
   return includes

# Helper class to parse missing-policy Cli input arguments
class MissingPolicyConfig( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, args, af=False ):
      if 'ACT' in args:
         self.actionType = mpActionValues[ args[ 'ACT' ] ]
      else:
         self.actionType = None
      self.addressFamilyAll = 'address-family' in args
      self.includeSubRm = 'sub-route-map' in args
      self.includePfxList = 'prefix-list' in args
      self.includeCommList = 'community-list' in args
      direction = args[ 'DIR' ].capitalize()
      afStr = 'Af' if af else ''
      self.actionAttr = 'missingPolicyAction' + afStr + direction
      self.addrFamilyAllAttr = 'missingPolicyAddressFamilyAll' + afStr + direction
      self.includeSubAttr = 'missingPolicyIncludeSubRouteMap' + afStr + direction
      self.includePfxListAttr = 'missingPolicyIncludePrefixList' + afStr + direction
      self.includeCommListAttr = 'missingPolicyIncludeCommList' + afStr + direction

#--------------------------------------------------------------------------------
# AfModeExtensionHook to provide support for registering CLI config for new
# address families in their own package in such a way that unwanted package
# dependencies are not introduced
#--------------------------------------------------------------------------------
class AfModeExtensionHook( object ): # pylint: disable=useless-object-inheritance
   """A hook to allow for extensions for the new address families created for
   multi-agent mode in a manner that is independent of package dependencies."""
   def __init__( self ):
      """Dictionary that contains address family mode extensions.
      @key : address family string
      @value : address family submode for the address family's configuration
      Example : { 'vpn-ipv4' : RouterBgpBaseAfMplsVpnMode }
      This will be populated by the CliPlugin of the corresponding af's package.
      Please see Plugin() in /src/MplsVpn/CliPlugin/MplsVpnCli.py.
      """
      self.afModeExtension = {}
      self.vrfAfModeExtension = {}

   def addAfModeExtension( self, afStr, afSubMode ):
      """Add extension @afSubMode for address family @afStr. These extensions will
      be used to go to the specific submode when an address family is configured.
      Please see RouterBgpBaseMode.gotoRouterBgpBaseAfMplsVpnMode() for usage.
      """
      # Each address family extension should only be registered once.  However,
      # in cohab Cli breadth tests it is possible for plugins to be reloaded,
      # therefore it's not safe to assert that the afStr has not already been
      # registered.
      self.afModeExtension[ afStr ] = afSubMode

   def afModeExtensions( self ):
      """Return list of all address families that have currently provided an
      afMode extension.
      """
      return list( self.afModeExtension )

   def addVrfAfModeExtension( self, afStr, afSubMode ):
      """Add extension @afSubMode for address family @afStr with vrf mode as parent.
      These extensions will be used to go to the specific submode when an address
      family is configured under a vrf.
      Please see RouterBgpVrfMode.addCommandClass( EnterBgpFlowspecVrfAfMode ) for
      usage.
      """
      # Each address family extension should only be registered once.  However,
      # in cohab Cli breadth tests it is possible for plugins to be reloaded,
      # therefore it's not safe to assert that the afStr has not already been
      # registered.
      self.vrfAfModeExtension[ afStr ] = afSubMode

   def vrfAfModeExtensions( self ):
      """Return list of all address families that have currently provided an
      afMode extension under a given vrf.
      """
      return list( self.vrfAfModeExtension )

afModeExtensionHook = AfModeExtensionHook()

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------
def routingSupportedGuard( mode, token ):
   if routingHardwareStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

def hwSupportedGuard( mode, token ):
   # guards against enabling cli in single-agent mode and checks routingsupported
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.multiAgent and \
         routingHardwareStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without next-hop
# labeled-unicast originate LFIB backup IP forwarding support
#-------------------------------------------------------------------------------
def nhLuOrigLfibBackupIpFwdSupportedGuard( mode, token ):
   if routingHardwareStatus.nhLuOrigLfibBackupIpFwdSupported:
      return None
   return CliParser.guardNotThisPlatform

def deleteVrfHook( vrfName ):
   if vrfName and vrfName in bgpVrfClearReqDir.clearRequestDir:
      del bgpVrfClearReqDir.clearRequestDir[ vrfName ]
   return ( True, None )

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( vrfName )
#
# vrfName is the name of the relevant vrf
#
# and has no expected return values
deleteRouterBgpVrfHook = CliExtensions.CliHook()
deleteRouterBgpMacVrfHook = CliExtensions.CliHook()
deleteRouterBgpUcmpVrfHook = CliExtensions.CliHook()
deleteRouterBgpAfiSafiHook = CliExtensions.CliHook()

def deleteBgpVrfConfigHook( vrfName ):
   if not vrfName:
      return
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         cleanupBgpConfig( bgpVrfConfigDir.vrfConfig[ vrfName ] )
         del bgpVrfConfigDir.vrfConfig[ vrfName ]
      bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
      del bgpRouteDistinguisherInput.routeDistinguisher[ vrfName ]
      rdRemoteDomainConfig = getBgpRouteDistinguisherRemoteDomainInput()
      del rdRemoteDomainConfig.routeDistinguisher[ vrfName ]
   else:
      # Cleanup config under default-vrf mode
      deleteBgpDefaultVrfConfig()
      # Cleanup global BGP config
      cleanupBgpConfig( bgpConfig )
      bgpConfig.asNumber = 0
      IraVrfCli.removeAgentVrfEntry( DEFAULT_VRF, "Bgp" )
      cleanupUcmpConfig( ucmpConfig )

def deleteBgpDefaultVrfConfig():
   """Method to clean config under router-bgp-vrf-default mode"""

   # Currently, only VPN and Traffic Policy mappings config is present under
   # this mode. Remove them.

   # Delete route-distinguisher config
   bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
   del bgpRouteDistinguisherInput.routeDistinguisher[ DEFAULT_VRF ]
   rdRemoteDomainConfig = getBgpRouteDistinguisherRemoteDomainInput()
   del rdRemoteDomainConfig.routeDistinguisher[ DEFAULT_VRF ]
   bgpConfig.routeDistinguisher = 'INVALID'
   bgpConfig.rdAll = False

   # Delete route-target config
   for attrName in [ 'routeTargetImport', 'routeTargetExport' ]:
      # Reset route-target config
      getattr( bgpConfig, attrName ).clear()
      # Also reset the per VPN route-target config
      for vpnType in vpnTypeAttrMap:
         vpnTypeAttrName = attrName + vpnType
         getattr( bgpConfig, vpnTypeAttrName ).clear()
         vpnTypeAttrName = attrName + 'RemoteDomain' + vpnType
         getattr( bgpConfig, vpnTypeAttrName, {} ).clear()
         vpnTypeAttrName = attrName + 'All' + vpnType
         getattr( bgpConfig, vpnTypeAttrName, {} ).clear()

   for attrName in [ 'routeMapImport', 'routeMapExport',
                     'allowImportedRouteToExport' ]:
      getattr( bgpConfig, attrName ).clear()
      for importAf in [ 'AfV4', 'AfV6' ]:
         vpnTypeAttrName = attrName + importAf
         getattr( bgpConfig, vpnTypeAttrName ).clear()

   for attrName in [ 'rcfImport' ]:
      getattr( bgpConfig, attrName ).clear()
      for importAf in [ 'AfV4', 'AfV6' ]:
         vpnTypeAttrName = attrName + importAf
         getattr( bgpConfig, vpnTypeAttrName ).clear()

   for attrName in [ 'rcfExport' ]:
      getattr( bgpConfig, attrName ).clear()
      for importAf in [ 'AfV4', 'AfV6' ]:
         vpnTypeAttrName = attrName + importAf
         getattr( bgpConfig, vpnTypeAttrName ).clear()

   bgpConfig.routeTargetExportFilterDisabled.clear()

   attrName = 'defaultExport'
   getattr( bgpConfig, attrName ).clear()
   for af in [ 'AfV4', 'AfV6' ]:
      afAttrName = attrName + af
      getattr( bgpConfig, afAttrName ).clear()

   # Delete traffic-policy config
   bgpConfig.tpFieldSetMappingDisabled = 'isInvalid'
   bgpConfig.tpFieldSetMappingBindingsIpv4.clear()
   bgpConfig.tpFieldSetMappingBindingsIpv6.clear()

   # Delete VRF selection policy config
   bgpConfig.vspFieldSetMappingDisabled = 'isInvalid'
   bgpConfig.vspFieldSetMappingBindingsIpv4.clear()
   bgpConfig.vspFieldSetMappingBindingsIpv6.clear()

def deleteUcmpVrfConfigHook( vrfName ):
   if not vrfName:
      return
   if vrfName != DEFAULT_VRF:
      if vrfName in ucmpVrfConfigDir.vrfConfig:
         cleanupUcmpConfig( ucmpVrfConfigDir.vrfConfig[ vrfName ] )
         del ucmpVrfConfigDir.vrfConfig[ vrfName ]
   else:
      cleanupUcmpConfig( ucmpConfig )

def configForVrf( vrfName ):
   config = bgpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         config = bgpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

def ucmpConfigForVrf( vrfName ):
   config = ucmpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in ucmpVrfConfigDir.vrfConfig:
         config = ucmpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

def bgpNeighborConfig( neighborKey, vrfName, create=True ):
   # Create/look-for peerGroup types in default vrf config
   if neighborKey.type == 'peerGroup':
      config = bgpConfig
   else:
      config = configForVrf( vrfName )
   neighConfig = config.neighborConfig.get( neighborKey )
   if not neighConfig and create:
      neighConfig = config.neighborConfig.newMember( neighborKey )
   return neighConfig

def delNeighborConfigIfDefault( peerKey, vrfName ):
   '''Function to delete neighborConfig if all attributes have default
   values

   If all the attributes of a neighborConfig are default, then the
   neighbor should be deleted. This can happen as the result of a
   setAttr call, if the value it is set to is the default. In other
   words, it's possible for a set call to have the effect of deleting
   a neighbor config.

   This should be called after any set or clear operation on a
   NeighborConfig attribute.
   '''
   config = configForVrf( vrfName )
   peerConfig = bgpNeighborConfig( peerKey, vrfName, create=False )
   if not peerConfig or peerConfig.isPeerGroup or peerConfig.isPeerGroupPeer:
      return
   allAttrs = { i.name for i in peerConfig.tacType.attributeQ if i.writable }
   attrsWithPresent = { i for i in allAttrs if hasattr( peerConfig, i + 'Present' ) }
   attrsWithPresent.add( 'timers' )
   attrsWithPresent.add( 'metricOut' )
   for attr in attrsWithPresent:
      if attr == 'metricOut':
         if ( getattr( peerConfig, 'metricOutState' ) !=
              getattr( peerConfig, 'metricOutStateDefault' ) ):
            return
      elif getattr( peerConfig, attr + 'Present' ):
         return
   if ( peerConfig.apSend or peerConfig.apSendIPv4Uni or
        peerConfig.apSendIPv6Uni or
        peerConfig.apSendIPv4LabeledUni or
        peerConfig.apSendIPv6LabeledUni or
        peerConfig.apSendEvpn or
        peerConfig.apSendVpnV4 or peerConfig.apSendVpnV6 or
        peerConfig.apSendDps ):
      return
   if peerConfig.maxRoutesAfiSafiConfig:
      return
   if peerConfig.maxAcceptedRoutesAfiSafiConfig:
      return
   if peerConfig.maxAdvRoutesAfiSafiConfig:
      return
   if ( peerConfig.routeTargetExportFilterDisabled or
        peerConfig.routeTargetExportFilterDisabledAf ):
      return
   if ( peerConfig.missingPolicyActionIn !=
        getattr( peerConfig, 'missingPolicyActionInvalid' ) or
        peerConfig.missingPolicyActionOut !=
        getattr( peerConfig, 'missingPolicyActionInvalid' ) or
        peerConfig.missingPolicyActionAfIn or
        peerConfig.missingPolicyActionAfOut ):
      return
   if ( peerConfig.rtMembershipDefaultRouteTarget !=
        peerConfig.rtMembershipDefaultRouteTargetDefault or
        peerConfig.rtMembershipDefaultRouteTargetEncoding !=
        peerConfig.rtMembershipDefaultRouteTargetEncodingDefault ):
      return
   if peerConfig.rtMembershipNlriSupport is not None:
      return
   del config.neighborConfig[ peerConfig.key ]

class RouterBgpBaseMode( RoutingBgpBaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP configuration'

   def __init__( self, parent, session, asNumber ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseMode.__init__( self, asNumber )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   # 'show active' command extracts and displays the configuration under
   # 'router bgp <ASN>' in a way similar to how 'show running-configuration |
   # section router bgp <ASN>' works.
   # In BasicCli.py the function showRunningConfigWithMode gets the entire
   # configuration and filters it out using the filterExp provided by parent
   # mode (RoutingBgpBaseMode). The 'filterExp' by default uses parent Mode's
   # enterCmd (check ConfigModeBase in BasicCli.py).
   # when 'bgp asn notation asdot' is configured under 'router bgp <ASN>', we need
   # to filter router bgp configuration dynamically. This is because if <ASN>
   # is >65535, with 'bgp asn notation asdot' we represent <ASN> in dotted
   # notation.
   def filterExp( self ):
      return "^router bgp %s$" % bgpFormatAsn( self.param_, \
                                              isAsdotConfigured( asnConfig ) )

class RouterBgpDefaultVrfMode( RoutingBgpVrfMode, BasicCli.ConfigModeBase ):
   """BGP default VRF config mode. This mode only has VPN commands currently"""

   name = "BGP Default VRF configuration"

   def __init__( self, parent, session, vrfName ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = DEFAULT_VRF
      RoutingBgpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfMode( RoutingBgpVrfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF configuration'

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      self.maybeMakeVrf()
      RoutingBgpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def maybeMakeVrf( self ):
      assert self.vrfName not in VRFNAMES_RESERVED
      config = bgpVrfConfigDir.vrfConfig.get( self.vrfName )
      if not config:
         bgpVrfConfigDir.vrfConfig.newMember( self.vrfName )
      if not ucmpVrfConfigDir.vrfConfig.get( self.vrfName ):
         ucmpVrfConfigDir.vrfConfig.newMember( self.vrfName )

class TrafficPolicyFieldSetMappingsMode( BgpFieldSetMappingsVrfMode,
                                         BasicCli.ConfigModeBase ):
   name = 'BGP traffic policy field set mappings configuration'

   def __init__( self, parent, session, mapCtx, vrfName ):
      self.mapCtx = mapCtx
      self.vrfName = vrfName
      BgpFieldSetMappingsVrfMode.__init__( self, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TpFieldSetIpv4Mode( FieldSetIpv4BaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP traffic policy field set ipv4 configuration'

   def __init__( self, parent, session, fsName, mapCtx, vrfName ):
      FieldSetIpv4BaseMode.__init__( self, fsName, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TpFieldSetIpv6Mode( FieldSetIpv6BaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP traffic policy field set ipv6 configuration'

   def __init__( self, parent, session, fsName, mapCtx, vrfName ):
      FieldSetIpv6BaseMode.__init__( self, fsName, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Vrf selection policy field-set mode(s)
class VrfSelectionPolicyFieldSetMappingsMode(
      BgpFieldSetMappingsVrfMode,
      BasicCli.ConfigModeBase ):
   name = 'BGP VRF selection policy field set mappings configuration'

   def __init__( self, parent, session, mapCtx, vrfName ):
      assert vrfName == DEFAULT_VRF
      self.mapCtx = mapCtx
      self.vrfName = vrfName
      BgpFieldSetMappingsVrfMode.__init__( self, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VspFieldSetIpv4Mode( FieldSetIpv4BaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF selection policy field set ipv4 configuration'

   def __init__( self, parent, session, fsName, mapCtx, vrfName ):
      FieldSetIpv4BaseMode.__init__( self, fsName, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VspFieldSetIpv6Mode( FieldSetIpv6BaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF selection policy field set ipv6 configuration'

   def __init__( self, parent, session, fsName, mapCtx, vrfName ):
      FieldSetIpv6BaseMode.__init__( self, fsName, mapCtx, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpUniMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Unicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpv6UniMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IPV6 Unicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "address-family ipv4 labeled-unicast" config mode
#-------------------------------------------------------------------------------
class RouterBgpBaseAfLabelV4Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family ipv4 labeled unicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-label' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "address-family ipv6 labeled-unicast" config mode
#-------------------------------------------------------------------------------
class RouterBgpBaseAfLabelV6Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family ipv6 labeled unicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-label' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# This modelet has all the commands which are shared between ipv4 and ipv6
# labeled-unicast config mode.
#-------------------------------------------------------------------------------
class RouterBgpAfLabelSharedModelet( CliParser.Modelet ):
   pass

RouterBgpBaseAfLabelV4Mode.addModelet( RouterBgpAfLabelSharedModelet )
RouterBgpBaseAfLabelV6Mode.addModelet( RouterBgpAfLabelSharedModelet )

class RouterBgpBaseAfIpMulticastMode( RoutingBgpBaseAfMode,
                                      BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Multicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpv6MulticastMode( RoutingBgpBaseAfMode,
                                      BasicCli.ConfigModeBase ):
   name = 'BGP address family IPv6 Multicast configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpMulticastMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Multicast configuration'

   def __init__( self, parent, session, addrFamily, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpv6MulticastMode( RoutingBgpVrfAfMode,
      BasicCli.ConfigModeBase ):
   name = 'BGP address family IPv6 Multicast configuration'

   def __init__( self, parent, session, addrFamily, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "address-family <ipv4|ipv6> sr-te" config mode
#-------------------------------------------------------------------------------
class RouterBgpBaseAfSrTeMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Segment Routing Traffic Engineering configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-srte' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# XXX_FIX_BGP_CORE_CLI_EVPN_CONFIG: The Evpn CLI mode and callback functions
# should eventually be moved out of the Bgp package. But we've decided to defer
# doing so until later on. Note: Defining this directly in the
# RouterBgpBaseMode, so this command is not available in Nd-VRF submode.
class RouterBgpBaseAfEvpnMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family EVPN configuration'

   def __init__( self, parent, session, addrFamily='evpn' ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfLinkStateMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Link State configuration'

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, 'link-state' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "address-family path-selection" config mode
#-------------------------------------------------------------------------------
class RouterBgpBaseAfDpsMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Dynamic-Path-Selection configuration'

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, 'path-selection' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF IPv4 configuration'

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIp6Mode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF IPv6 configuration'

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpDefaultVrfAfMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   """BGP default VRF address family config mode. This mode only has VPN commands
   currently.
   """

   name = "BGP Default VRF address family configuration"

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfSharedModelet( CliParser.Modelet ):
   """This modelet has all the commands which are shared between default and
   non-default VRFs e.g. VPN configuration commands
   """

RouterBgpDefaultVrfMode.addModelet( RouterBgpVrfSharedModelet )
RouterBgpVrfMode.addModelet( RouterBgpVrfSharedModelet )

# In the default VRF we don't CliSave attributes that match default, so return
# 'isInvalid' (meaning the default value is set). In a non-default VRF, we
# CliSave all explicitly-configured attributes, so return the actual value.
# This method should only be used in the handler for the command that's setting
# the value of the attribute that matches the default value.
def getExplicitDefaultTristate( mode, attr ):
   if mode.vrfName == DEFAULT_VRF:
      return 'isInvalid'
   config = configForVrf( mode.vrfName )
   meaningOfInvalid = getattr( config, attr + 'Default' )
   return 'isTrue' if meaningOfInvalid else 'isFalse'

def getExplicitDefaultValue( mode, attr, attrValue, useUcmpConfig=False ):
   # In the default VRF we don't CliSave attributes that match default, so return
   # the value of attr + 'Invalid'. In a non-default VRF, we CliSave all
   # explicitly-configured attributes, so return the actual value. This method
   # should only be used in the handler for the command that's setting the value of
   # the attribute that matches the default value.
   config = configForVrf( mode.vrfName ) if not useUcmpConfig \
      else ucmpConfigForVrf( mode.vrfName )
   if attr != 'maxEcmp':
      defValue = getattr( config, attr + 'Default' )
   else:
      # The maxEcmp attribute doesn't have a default in the bgp instance.
      # The default is set by the platform and stored in Routing:Hardware::Status
      defValue = routingHardwareStatus.maxLogicalProtocolEcmp
   if mode.vrfName == DEFAULT_VRF and attrValue == defValue:
      return getattr( config, attr + 'Invalid' )
   return attrValue

class RouterBgpSharedModelet( CliParser.Modelet ):
   """This modelet is used to add instance-level BGP commands to both default
   and vrf modes."""

RouterBgpBaseMode.addModelet( RouterBgpSharedModelet )
RouterBgpVrfMode.addModelet( RouterBgpSharedModelet )

def attrIsPresent( config, attr ):
   if getattr( config, attr + 'Present' ):
      return True
   return False

def configModeCmdForAf( addrFamily ):
   if addrFamily == 'all':
      modeCmd = 'router bgp'
   else:
      modeCmd = 'address-family %s' % addrFamily
   return modeCmd

def configModeKeyForVrfAf( addrFamily, vrfName=DEFAULT_VRF ):
   modeKey = 'router-bgp-vrf-%s' % vrfName
   if addrFamily != 'all':
      modeKey += '-af'
   return modeKey

def getConflictingAttrsForAf( attrs, addrFamily ):
   if addrFamily == 'all':
      conflictingAttrs = [ ( af, c ) for af, c in six.iteritems( attrs )
                           if af != 'all' ]
   else:
      allAttr = attrs.get( 'all', '' )
      # For some groups of attributes, a global version of the attribute does not
      # exist.
      conflictingAttrs = [ ( 'all', allAttr ) ] if allAttr else []
   return conflictingAttrs

def haveConflictingRedistribute( mode, config, proto, addrFamily ):
   """ Print an error and returns True if redistribution is configured in
       conflicting mode"""

   attrs = BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   protocolNameMap = { 'protoIsis'      : 'isis',
                       'protoOspf3'     : 'ospfv3',
                       'protoOspf3Nssa' : 'ospfv3 nssa',
                       'protoOspf3Ase'  : 'ospfv3 external' }
   for otherAf, c in conflictingAttrs:
      if proto in getattr( config, c ):
         errorMsg = "Cannot configure redistribute %s in mode '%s' " \
                    "while it is configured in mode '%s'" \
                    % ( protocolNameMap[ proto ],
                        configModeCmdForAf( addrFamily ),
                        configModeCmdForAf( otherAf ) )
         mode.addError( errorMsg )
         return True
   return False

def maybeWarnConflictingRedist( mode, config, proto, addrFamily ):
   """ Print a warning if redistribution is configured in conflicting mode
       in ribd mode"""

   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.multiAgent:
      return
   attrs = BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   protocolNameMap = { 'protoIsis': 'isis',
                       'protoOspf3': 'ospfv3',
                       'protoOspf3Nssa': 'ospfv3 nssa',
                       'protoOspf3Ase': 'ospfv3 external' }

   for otherAf, c in conflictingAttrs:
      if proto in getattr( config, c ):
         warningMsg = "Configuring 'redistribute %s' in mode '%s' while it is " \
                      "configured in mode '%s' is only supported in " \
                      "multi-agent model" \
                      % ( protocolNameMap[ proto ],
                          configModeCmdForAf( addrFamily ),
                          configModeCmdForAf( otherAf ) )
         mode.addWarning( warningMsg )
         return

def redistributeConfig( addrFamily, config ):
   attr = BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ].get( addrFamily )
   return getattr( config, attr )

def checkForConflictingPolicy( config, attrName, policyName, addrFamily,
                               policyType ):
   """
   Get any conflicting policy (RCF or route maps) and return errors associated
   with them.
   """
   assert policyType in [ "route-map", "rcf" ]
   attrs = BgpLib.peerConfigAttrsAfMap[ attrName ]
   conflictingPolicyAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, conflictingPolicy in conflictingPolicyAttrs:
      if attrIsPresent( config, conflictingPolicy ) and \
            getattr( config, conflictingPolicy ):
         errorMsg = ( f"Cannot configure {policyType} '%s' in mode '%s' "
                      "while '%s' is configured in mode '%s'"
                      % ( policyName, configModeCmdForAf( addrFamily ),
                          getattr( config, conflictingPolicy ),
                          configModeCmdForAf( otherAf ) ) )
         return errorMsg
   return None

def checkForConflictingRouteMaps( config, attrName, mapName, addrFamily ):
   """Get any conflicting route maps and return errors associated with them."""
   return checkForConflictingPolicy( config, attrName, mapName, addrFamily,
                                     "route-map" )

def haveConflictingRouteMaps( mode, config, attrName, mapName, addrFamily ):
   """ Print an error and returns True if a conflicting route-map is already set """
   errorMsg = checkForConflictingRouteMaps( config, attrName, mapName, addrFamily )
   if errorMsg:
      mode.addError( errorMsg )
      return True
   return False

def haveConflictingWeight( mode, config, addrFamily ):
   """
   Print an error and return True if a conflicting weight config is
   already present.
   """
   attrs = BgpLib.peerConfigAttrsAfMap[ 'weight' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if attrIsPresent( config, c ):
         errMsg = "Cannot configure weight in mode '%s' while it is "\
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )
         mode.addError( errMsg )
         return True
   return False

def addPathSendConfig( config, addrFamily ):
   attr = BgpLib.bgpConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   return getattr( config, attr )

def haveConflictingRedistInternal( mode, config, addrFamily ):
   """ Print an error and returns True if conflicting
       redistribute-internal option is set """
   afMode = addrFamily in ( 'ipv4', 'ipv6' )
   routerBgpMode = 'router-bgp' in mode.longModeKey and not afMode
   vrfStr = ''
   if mode.vrfName != DEFAULT_VRF:
      vrfStr = '-vrf-%s' % mode.vrfName
   if routerBgpMode and config.bgpRedistInternalAfV4 == config.bgpRedistInternalAfV6:
      return False
   if afMode and \
         config.bgpRedistInternal != redistInternalStateEnum.disableRedistInt:
      return False
   otherMode = { True : 'router-bgp%s-af', False : 'router-bgp%s' }
   # Print error message as there is conflicting redistribute-internal config
   errorMsg = "Cannot configure command in mode '%s' when " \
         "'no bgp redistribute-internal' has been configured " \
         "in mode '%s'" % ( mode.longModeKey, otherMode[ routerBgpMode ] % vrfStr )
   mode.addError( errorMsg )
   return True

def switchRedistInternalConfigToAfMode( config ):
   # switch redistribute-internal config to addrFamily mode config
   allAfState = config.bgpRedistInternal
   if config.bgpRedistInternalAfV4 == redistInternalStateEnum.notConfigured and \
         config.bgpRedistInternalAfV6 == redistInternalStateEnum.notConfigured:
      assert allAfState != redistInternalStateEnum.notConfigured

      config.bgpRedistInternalAfV4 = allAfState
      config.bgpRedistInternalAfV6 = allAfState
      config.bgpRedistInternal = redistInternalStateEnum.notConfigured

def switchRedistInternalConfigToAllAfMode( config ):
   # switch redistribute-internal config to router-bgp mode config
   allAfState = config.bgpRedistInternal
   if allAfState == redistInternalStateEnum.notConfigured:
      assert config.bgpRedistInternalAfV4 == config.bgpRedistInternalAfV6

      afState = config.bgpRedistInternalAfV4
      config.bgpRedistInternalAfV4 = redistInternalStateEnum.notConfigured
      config.bgpRedistInternalAfV6 = redistInternalStateEnum.notConfigured
      config.bgpRedistInternal = afState

def conflictingRouteInstallMapConfigs( mode, routeInstallMapName, config,
                                       addrFamily ):
   # possible conflicts occur when routeInstallMap has been configured
   # at af level and global level. We need to prevent that.
   attrs = BgpLib.bgpConfigAttrsAfMap[ 'routeInstallMap' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if getattr( config, c ) != getattr( config, c + 'Default' ):
         errMsg = "Cannot configure route install-map in mode '%s' while it is " \
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )
         mode.addError( errMsg )
         return True
   return False

def conflictingRouteTargetConfigs( mode, config, addrFamily='all', vpnType=None ):
   vpnAf = vpnTypeAttrMapInv.get( vpnType, '' )
   impExpList = ( 'Import', 'Export' )
   domains = ( '', 'RemoteDomain' )
   conflictingAttrs = []
   for domain, impExp in product( domains, impExpList ):
      attrName = 'routeTarget' + impExp + domain + vpnAf
      attrs = BgpLib.bgpConfigAttrsAfMap.get( attrName, {} )
      conflictingAttrs.extend( getConflictingAttrsForAf( attrs, addrFamily ) )
   for otherAf, c in conflictingAttrs:
      if getattr( config, c, False ):
         errMsg = ( "Cannot configure route-target in mode '%s' while it is "
                    "configured in mode '%s'" % (
                       configModeKeyForVrfAf( addrFamily, mode.vrfName ),
                       configModeKeyForVrfAf( otherAf, mode.vrfName ) ) )
         mode.addError( errMsg )
         return True
   return False

def vpnTypeTokenAllowedInMode( vpnType, mode ):
   ''' This function checks if the vpnType token is allowed in the mode '''
   tokenAllowed = True
   if isinstance(
            mode,
            ( RouterBgpVrfAfIpMode,
              RouterBgpVrfAfIp6Mode,
              RouterBgpDefaultVrfAfMode )
   ):
      if ( vpnType == 'vpn-ipv4' and mode.addrFamily != 'ipv4' ) or (
           vpnType == 'vpn-ipv6' and mode.addrFamily != 'ipv6' ):
         tokenAllowed = False
   return tokenAllowed

def validateRouteTargetCommand( mode, vpnType, warn=True ):
   ''' This function validates the route target command to check if the vpnType
       is allowed in the mode '''
   # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
   valid = True
   if not vpnType and warn:
      mode.addWarning( 'This form of the route-target command is deprecated.'
                       ' Please specify one of %s.' % (
                       ', '.join( sorted( vpnTypeAttrMapInv.keys() ) ) ) )

   if not vpnTypeTokenAllowedInMode( vpnType, mode ):
      mode.addError( '%s argument is not allowed under address-family %s' % (
                     vpnType, mode.addrFamily ) )
      valid = False
   return valid

def getVpnTypeAttr( config, attrName, addrFamily, vpnType ):
   ''' Returns the vpnTypeAttr from the config corresponding to
       attrName, addrFamily, vpnType '''
   attrName = BgpLib.bgpConfigAttrsAfMap.get( attrName, {} ).get( addrFamily, '' )
   vpnTypeAttrName = attrName + vpnTypeAttrMapInv.get( vpnType, '' )
   vpnTypeAttr = getattr( config, vpnTypeAttrName, None )
   return vpnTypeAttr

def haveConflictingOutDelay( mode, config, addrFamily ):
   """
   Print an error and return True if a conflicting outDelay config is
   already present.
   """
   attrs = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if attrIsPresent( config, c ):
         errMsg = "Cannot configure outDelay in mode '%s' while it is "\
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )

         mode.addError( errMsg )
         return True
   return False

def conflictingRouteMapConfigs( mode, config, addrFamily, vpnType ):
   ''' Print an error and returns True if import/export route-map is configured in
       conflicting mode '''

   importAttrs = BgpLib.bgpConfigAttrsAfMap[ 'routeMapImport' ]
   exportAttrs = BgpLib.bgpConfigAttrsAfMap[ 'routeMapExport' ]
   conflictingAttrs = getConflictingAttrsForAf( importAttrs, addrFamily )
   conflictingAttrs.extend( getConflictingAttrsForAf( exportAttrs, addrFamily ) )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   for otherAf, c in conflictingAttrs:
      if vpnAf in getattr( config, c ):
         errorMsg = ( "Cannot configure route-map %s in mode '%s' "
                      "while it is configured in mode '%s'"
                      % ( vpnType, configModeCmdForAf( addrFamily ),
                        configModeCmdForAf( otherAf ) ) )
         mode.addError( errorMsg )
         return True
   return False

def conflictingRcfConfigs( mode, config, addrFamily, vpnType ):
   ''' Print an error and returns True if import/export rcf is configured in
       conflicting mode '''

   importAttrs = BgpLib.bgpConfigAttrsAfMap[ 'rcfImport' ]
   exportAttrs = BgpLib.bgpConfigAttrsAfMap[ 'rcfExport' ]
   conflictingAttrs = getConflictingAttrsForAf( importAttrs, addrFamily )
   conflictingAttrs.extend( getConflictingAttrsForAf( exportAttrs, addrFamily ) )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   for otherAf, c in conflictingAttrs:
      if vpnAf in getattr( config, c ):
         errorMsg = ( "Cannot configure rcf %s in mode '%s' "
                      "while it is configured in mode '%s'"
                      % ( vpnType, configModeCmdForAf( addrFamily ),
                          configModeCmdForAf( otherAf ) ) )
         mode.addError( errorMsg )
         return True
   return False

def setServiceAclCommon( mode, aclType='ip', no=False, aclName=None ):
   if no:
      AclCliLib.noServiceAcl( mode, 'bgpsacl', aclConfig, aclCpConfig,
                              aclName, aclType, mode.vrfName )
   else:
      assert aclName
      AclCliLib.setServiceAcl( mode, 'bgpsacl', IPPROTO_TCP,
                               aclConfig, aclCpConfig, aclName, aclType,
                               mode.vrfName, port=[ BGP_PORT ], sport=[ BGP_PORT ] )

bgpMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'bgp',
                                          helpdesc='Border Gateway Protocol' ),
                                       guard=routingSupportedGuard )

#
# Add 'router bgp <asn>' rule to the list of global configs so that when
# the user types 'router bgp ?', the current AS number (if configured) will
# show up in the help list.  Also, if the user types 'router bgp [tab]', and
# BGP is already configured with an AS number, that AS number will be
# autocompleted.
#
# This is all for convenience/user-friendliness:
#
#   switch(config)# router bgp ?
#      <1-4294967295>  AS Number
#
#   switch(config)# router bgp 100       << configure BGP AS 100
#   switch(config-router-bgp)#
#   switch(config-router-bgp)# exit
#   switch(config)#
#   switch(config)#
#   switch(config)# router bgp ?
#      100        Configure BGP AS 100   << AS 100 shows up in the help list
#      <1-4294967295>  AS Number
#
#   switch(config)#
#
def getCurrentBgpAs( mode ):
   asn = bgpConfig.asNumber
   asdotConfigured = isAsdotConfigured( asnConfig )
   if asn == 0:
      return {}
   return { bgpFormatAsn( asn, asdotConfigured ): \
         "Configure BGP AS %s" % bgpFormatAsn( asn, asdotConfigured ) }

def getBgpAsValue( mode, match=None ):
   return asnStrToNum( match )

GrHelperRestartTimeClassEnum = TacLazyType(
   "Routing::Bgp::GrHelperRestartTimeClass" )

def getTimeInSeconds( time, timeClass ):
   if timeClass is GrHelperRestartTimeClassEnum.days:
      return time * 60 * 60 * 24
   elif timeClass is GrHelperRestartTimeClassEnum.hours:
      return time * 60 * 60
   elif timeClass is GrHelperRestartTimeClassEnum.minutes:
      return time * 60
   else:
      return time

#-------------------------------------------------------------------------------
# "[no|default] service routing configuration bgp no-equals-default" command,
# in "config" mode.
#-------------------------------------------------------------------------------
class NedModeKwMatcher( CliMatcher.KeywordMatcher ):
   """Keyword Matcher for no-equals-default mode. Returns a Matcher instance."""

   def __init__( self, keyword, helpdesc ):
      CliMatcher.KeywordMatcher.__init__( self, keyword, helpdesc=helpdesc )

   def match( self, mode, context, token ):
      if bgpConfig.noEqualsDefaultMode:
         return CliMatcher.KeywordMatcher.match( self, mode, context, token )
      else:
         return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      if bgpConfig.noEqualsDefaultMode:
         return CliMatcher.KeywordMatcher.completions( self, mode, context, token )
      else:
         return []

# In general, commands that configure a boolean value will have a new 'disabled'
# token, while commands that set a numerical value will interpret configuring the
# default value explicitly the way the 'no' version of the command works outside
# of no-equals-default mode. See AID3714 for a full list of all affected commands.
noEqualsDefaultMatcherForConfig = CliMatcher.KeywordMatcher( 'no-equals-default',
      helpdesc='Interpret all \'no neighbor\' commands ' +
               'as \'default neighbor\' commands' )

configMode = BasicCli.GlobalConfigMode

class ConfigBgpNoEqualsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'service routing configuration bgp no-equals-default'
   noOrDefaultSyntax = 'service routing configuration bgp no-equals-default'
   data = { 'service': CliToken.Service.serviceMatcherForConfig,
            'routing': routingMatcherForConfig,
            'configuration': CliToken.Service.configMatcherForAfterService,
            'bgp': bgpMatcherForConfig,
            'no-equals-default': noEqualsDefaultMatcherForConfig
          }
   handler = "RoutingBgpCliHandler.handlerConfigBgpNoEqualsDefaultCmd"
   noOrDefaultHandler = "RoutingBgpCliHandler." \
      "noOrDefaultHandlerConfigBgpNoEqualsDefaultCmd"

configMode.addCommandClass( ConfigBgpNoEqualsDefaultCmd )

def getAfPrefixListNames( mode ):
   if mode.addrFamily == 'ipv4':
      return getPrefixListNames( mode )
   else:
      return getIpv6PrefixListNames( mode )

afPrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getAfPrefixListNames,
      'Name of the prefix list' )

# Peergroup names can be a combination of letters, digits, '-', '.' and '_'.
# But, if it includes a '.', it must begin with a letter
# Cannot be the (case-insensitive) keywords in blackListedPeerGroupNames.
validNameRegex = r'(([A-Za-z][-A-Za-z0-9_.]+)|([-A-Za-z0-9_]+))'
disallowRegex = anyCaseRegex( "(default|interface)" )
peergroupNameRegex = "^(?!%s$)%s$" % ( disallowRegex, validNameRegex )

def peergroupNameFn( mode ):
   return ( x.group for x in bgpConfig.neighborConfig if x.type == 'peerGroup' )
peergroupNameMatcher = CliMatcher.DynamicNameMatcher( peergroupNameFn,
                                                      'Name of the peer-group',
                                                      helpname='WORD',
                                                      pattern=peergroupNameRegex )

def ancestorPeerGroupNameFn( mode ):
   return ( x.group for x in bgpConfig.neighborConfig if x.type == 'peerGroup' and
            not bgpConfig.neighborConfig[ x ].isPeerGroupWithAncestors )

ancestorPeerGroupNameMatcher = CliMatcher.DynamicNameMatcher(
   ancestorPeerGroupNameFn, 'Name of the ancestor peer-group',
   helpname='WORD', pattern=peergroupNameRegex )
ipv4PeerMatcher = IpAddrMatcher.IpAddrMatcher( "Neighbor IPv4 address" )
ipv6PeerMatcher = Ip6AddrMatcher.Ip6AddrMatcher( "Neighbor IPv6 address" )
ipv6LlPeerMatcher = Ip6LlAddr.Ip6LlAddrMatcher( "Neighbor IPv6 link-local address" )

class PeerCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP_PEER | IP6_PEER | IP6_LL_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in
                            ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER', 'GROUP' )
                            if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

# PeerSet names can be a combination of letters, digits, '-', '.' and '_'.
# But, if it includes a '.', it must begin with a letter
# Cannot be the (case-insensitive) keywords in blackListedPeerGroupNames.
validPeerSetNameRegex = r'(([A-Za-z][-A-Za-z0-9_.]+)|([-A-Za-z0-9_]+))'
disallowPeerSetNameRegex = anyCaseRegex( "(default)" )
peerSetNameRegex = "^(?!%s$)%s$" % ( disallowPeerSetNameRegex,
                                     validPeerSetNameRegex )

def peerSetNameFn( mode ):
   return ( x.group for x in bgpConfig.neighborConfig if x.type == 'peerSet' )

epePeerSetNameMatcher = CliMatcher.DynamicNameMatcher( peerSetNameFn,
                                                    pattern=peerSetNameRegex,
                                                    helpname='WORD',
                                                    helpdesc='EPE peer set name' )

def orrPositionNameFn( mode ):
   return { positionName: 'BGP optimal route reflection position %s' % positionName
            for positionName in bgpConfig.orrNamedPosition }

# Position names can be a combination of letters, digits, '-', '.', ':'
# and '_'.
# But it must start with a letter or digit.
# Cannot start with (case-insensitive) 'peer' to avoid conflicting
# with auto generated position names for peer, as well as the
# 'peer-address' keyword.
orrPositionNameInvalidRegex = anyCaseRegex( 'peer' )
orrPositionNameRegex = "^(?!%s)%s$" % ( orrPositionNameInvalidRegex,
                                        orrPositionNameValidRegex )

class OrrPositionNameExpression( CliCommand.CliExpression ):
   expression = 'EXISTING_POSITION_NAME | NEW_POSITION_NAME'
   data = { 'EXISTING_POSITION_NAME':
            CliMatcher.DynamicKeywordMatcher( orrPositionNameFn ),
            'NEW_POSITION_NAME':
            CliMatcher.DynamicNameMatcher( orrPositionNameFn,
                                           'BGP optimal route reflection ' +
                                           'position',
                                           helpname='WORD',
                                           pattern=orrPositionNameRegex ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      positionName = args.get( 'EXISTING_POSITION_NAME' ) or \
                     args.get( 'NEW_POSITION_NAME' )
      if positionName is not None:
         args[ 'POSITION_NAME' ] = positionName

def emitWarningIfOrrNotSupportedInRibdMode( mode ):
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.multiAgent:
      return
   msg = 'BGP optimal route reflection is only supported in multi-agent mode.'
   mode.addWarning( msg )

def emitWarningIfRfdNotSupportedInRibdMode( mode ):
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.multiAgent:
      return
   msg = 'BGP route flap damping is only supported in multi-agent mode.'
   mode.addWarning( msg )

class V4OrPeerGroupCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if any( x in V4OrPeerGroupCliExpression.data for x in args ):
         addr = args[ next( peer for peer in
                      ( 'IP_PEER', 'GROUP' )
                      if peer in args ) ]
         args[ 'PEER' ] = PeerConfigKey( addr )

class V6OrPeerGroupCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP6_PEER | IP6_LL_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if any( x in V6OrPeerGroupCliExpression.data for x in args ):
         addr = args[ next( peer for peer in
                      ( 'GROUP', 'IP6_PEER', 'IP6_LL_PEER' )
                      if peer in args ) ]
         args[ 'PEER' ] = PeerConfigKey( addr )

class V4V6PeerKeyCliExpression( CliCommand.CliExpression ):
   expression = 'IP_PEER | IP6_PEER | IP6_LL_PEER'
   data = { 'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in
                   ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER' )
                   if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

# Defines an expression that matches an IPv4/IPv6 address or a peer group name,
# but *not* an IPv6 link-local address.
class V4OrV6OrPeerGroupCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP_PEER | IP6_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in ( 'IP_PEER', 'IP6_PEER', 'GROUP' )
                            if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

class RouterBgpAfSharedModelet( CliParser.Modelet ):
   """This modelet is used to add commands common to multiple address-family modes.
   It is added to unicast, multicast, and evpn submodes."""

class RouterBgpAfUnsupportedPrefixListSharedModelet( CliParser.Modelet ):
   """This modelet is used to add commands common to multiple address-family modes,
   which don't support prefix-lists. It is added to all submodes except
   ipv4/6 uni/multicast."""

class RouterBgpAfSharedVrfModelet( CliParser.Modelet ):
   """This modelet is used to add commands common to address-family modes for
   default and non-default VRFs e.g. VPN configuration commands.
   It is only added to unicast address-family modes.
   """

class RouterBgpAfVpnModelet( CliParser.Modelet ):
   """This modelet is used to add commands common to vpn modes (evpn, vpn-ipv4,
   vpn-ipv6)"""

#-------------------------------------------------------------------------------
# "address-family [ipv4 | ipv6] [multicast]" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpUniAndMcastSharedModelet( CliParser.Modelet ):
   """This modelet is added to the unicast and multicast address-families,
   in default and non-default vrfs."""

#-------------------------------------------------------------------------------
# "address-family [ipv4 | ipv6]" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpUniSharedModelet( CliParser.Modelet ):
   """This modelet is added to the unicast address-family,
   in default and non-default vrfs."""

#-------------------------------------------------------------------------------
# "address-family ipv4|ipv6 multicast" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpMcastSharedModelet( CliParser.Modelet ):
   """This modelet is added to the multicast address-family,
   in default and non-default vrfs."""

RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfVpnModelet )

# Add RouterBgpAfSharedVrfModelet to both default and non-default VRF modes
RouterBgpDefaultVrfAfMode.addModelet( RouterBgpAfSharedVrfModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfSharedVrfModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfSharedVrfModelet )

RouterBgpVrfAfIpMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfIpUniSharedModelet )

RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )

RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )

RouterBgpBaseAfLinkStateMode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfDpsMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpVrfMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfSrTeMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfLabelV4Mode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfLabelV6Mode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )

#-------------------------------------------------------------------------------
def getBgpRouteDistinguisherInput():
   bgpRouteDistinguisherInput = rdConfigInputDir.get( 'bgp' )
   assert bgpRouteDistinguisherInput
   return bgpRouteDistinguisherInput

def getBgpRouteDistinguisherRemoteDomainInput():
   return rdConfigInputDir.get( 'bgpRemoteDomain' )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global routingHardwareStatus
   global rdConfigInputDir
   global bgpVrfConfigDir
   global bgpVrfClearReqDir
   global asnConfig
   global aclConfig
   global aclCpConfig
   global ucmpConfig
   global ucmpVrfConfigDir

   bgpConfig = ConfigMount.mount( entityManager, 'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   bgpVrfConfigDir = ConfigMount.mount( entityManager, 'routing/bgp/vrf/config',
                                        'Routing::Bgp::VrfConfigDir', 'w' )
   asnConfig = ConfigMount.mount( entityManager, 'routing/bgp/asn/config',
                                  'Routing::AsnConfig', 'w' )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   rdConfigInputDir = \
      ConfigMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/config',
                         'Tac::Dir', 'w' )

   bgpVrfClearReqDir = LazyMount.mount( entityManager,
                                        'routing/bgp/clear/vrf/request',
                                        'Routing::Bgp::VrfClearRequestDir', 'w' )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   ucmpConfig = ConfigMount.mount( entityManager, 'routing/ucmp/bgp/config',
                                   'Routing::Ucmp::UcmpConfig', 'w' )
   ucmpVrfConfigDir = ConfigMount.mount( entityManager,
                                         'routing/ucmp/bgp/vrf/config',
                                         'Routing::Ucmp::VrfUcmpConfigDir',
                                         'w' )
   IraVrfCli.canDeleteVrfHook.addExtension( deleteVrfHook )

   deleteRouterBgpVrfHook.addExtension( deleteBgpVrfConfigHook )
   deleteRouterBgpUcmpVrfHook.addExtension( deleteUcmpVrfConfigHook )

# 'no-equals-default' mode, or NED mode, is described in AID3792.
# Refer to the AID for more details.
#
# The 'BgpNEDCmdBaseClass' calls an handler based on the form of the
# command executed and on the mode:
#
#  Mode          | CLI Command foom      | Handler triggered
# ------------------------------------------------------------
#  ANY mode      | <cmd> (positive form) | _handleNormal()
# ------------------------------------------------------------
#  ANY mode      | default <cmd>         | _handleNoOrDefault( noOrDefault=DEFAULT )
# ------------------------------------------------------------
#  normal mode   | no <cmd>              | _handleNoOrDefault( noOrDefault=NO )
#  NED mode      | no <cmd>              | _handleNoOrDefault( noOrDefault=DEFAULT )
# ------------------------------------------------------------
#  NED mode only | <cmd> disabled        | _handleNoOrDefault( noOrDefault=NO )
#
# Calls to _handleNoOrDefault are passed an enum noOrDefault which indicates if the
# command in question is being explicitly disabled or set to default settings.
#
# For a feature being disabled by default, the feature is implicitly disabled,
# meaning that the 'Present' attribute is False.
#
# As an optimization, explicit disables for non peer-group peers are treated the same
# as implicit disables. This is to avoid unnecessarily rendering the "no" form of the
# command when it has no practical effect. Explicit disable only makes a practical
# effect (compared to implicit disable) if there is some config to inherit from (e.g.
# a peer-group's config for a peer-group peer).
#
# Examples of features disabled by default are 'neighbor <peer> default-originate'
#
#  non-peer-group peer or peer group
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit disable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | implicit disable
#  NED mode      | no <cmd>       | implicit disable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | implicit disable
#
# peer group peer
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit disable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | explicit disable
#  NED mode      | no <cmd>       | implicit disable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | explicit disable
#
# For a feature that is enabled by default, an example is
# 'neighbor <peer> additional-path receive'. The behavior is the following:
#
#  non-peer-group peer or peer group or peer group peer
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit enable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | explicit disable
#  NED mode      | no <cmd>       | implicit enable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | explicit disable
#
class BgpCmdBaseClass( CliCommand.CliCommandClass ):
   '''
   Base class for commands where NED mode is not dependent on the running
   configuration (config is always in NED mode).

   All NEW bgp cli commands (unless otherwise stated
   by cli-review) should derive from this class and are always in NED mode.
   Additionally, commands that are not setting a numeric attribute MUST provide a
   `disabled` form of the command.

   Please be sure to check in with cli-review@arista.com if implementing a new
   command in either NED mode or always-NED mode.
   '''
   _disabledKeyword = 'disabled'
   _disabledKeywordHelpDesc = "Explicitly disable this configuration"
   _disabledMatcher = CliMatcher.KeywordMatcher(
         _disabledKeyword,
         helpdesc=_disabledKeywordHelpDesc
   )
   data = {
      'disabled': _disabledMatcher
   }

   @classmethod
   def _createSyntaxData( cls, dataDict ):
      '''
      Helper method for performing the requisite parent class data dictionary copy,
      and update with the passed args.  This is done to streamline the process of
      creating new commands, and to help prevent errors e.g. where the base class
      data dictionary is not copied before updating.  The data dictionary may still
      be created/modified manually.
      '''
      data = cls.data.copy()
      data.update( dataDict )
      return data

   @staticmethod
   def callNoOrDefaultHandler( handleNoOrDefault, mode, args ):
      handleNoOrDefault( mode, args, NoOrDefault.DEFAULT )

   @staticmethod
   def callHandler( handleNormal, handleNoOrDefault, mode, args ):
      if 'disabled' in args:
         handleNoOrDefault( mode, args, NoOrDefault.NO )
      else:
         handleNormal( mode, args )

class BgpNEDCmdBaseClass( BgpCmdBaseClass ):
   '''
   Base class for commands where NED mode is dependent on the running
   configuration (config is not always in NED mode).  If adding a brand new command,
   chances are that NED mode is always "on" for your command, so use BgpCmdBaseClass
   instead.

   This base class should only be used for Bgp commands which were added (NOT
   converted) prior to November of 2019.
   '''
   data = {
      'disabled': NedModeKwMatcher(
            BgpCmdBaseClass._disabledKeyword,
            BgpCmdBaseClass._disabledKeywordHelpDesc
      )
   }

   @staticmethod
   def callNoOrDefaultHandler( handleNoOrDefault, mode, args ):
      # This needs to override the parent class as NED mode in this case is
      # dependent on the running config of the device.
      if CliCommand.isNoCmd( args ) and not bgpConfig.noEqualsDefaultMode:
         noOrDefault = NoOrDefault.NO
      else:
         noOrDefault = NoOrDefault.DEFAULT
      handleNoOrDefault( mode, args, noOrDefault )

#
# optimal-route-reflection position POSITION_NAME
#
class RouterBgpOrrPositionMode( BgpOrrPositionMode, BasicCli.ConfigModeBase ):
   name = 'Named optimal-route-reflection position'

   def __init__( self, parent, session, orrPositionName ):
      BgpOrrPositionMode.__init__( self, orrPositionName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpRfdPolicyMode( BgpRfdPolicyMode, BasicCli.ConfigModeBase ):
   name = 'BGP route flap damping configuration'

   def __init__( self, parent, session, rfdPolicyName ):
      # used to abandon changes if the user decides to abort
      self.abort_ = False
      # stores the config changes prior to committing
      self.pendingConfig = Bunch()
      # Create an instance of rfdPolicyParams to get default values
      self.rfdDefaultParams = Tac.Value( "Routing::Bgp::RfdPolicyParams" )
      BgpRfdPolicyMode.__init__( self, rfdPolicyName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitRfdPolicyConfig( self ):
      if self.abort_:
         return
      if self.rfdPolicyName not in bgpConfig.rfdPolicy:
         bgpConfig.rfdPolicy.newMember( self.rfdPolicyName )
      rfdPolicy = bgpConfig.rfdPolicy[ self.rfdPolicyName ]
      rfdPolicyParams = Tac.nonConst( rfdPolicy.rfdPolicyParams )
      if 'suppressionIgnoredPrefixListV4Uni' in self.pendingConfig:
         rfdPolicy.suppressionIgnoredPrefixListV4Uni = \
            self.pendingConfig.pop( 'suppressionIgnoredPrefixListV4Uni' )
      if 'suppressionIgnoredPrefixListV6Uni' in self.pendingConfig:
         rfdPolicy.suppressionIgnoredPrefixListV6Uni = \
            self.pendingConfig.pop( 'suppressionIgnoredPrefixListV6Uni' )
      for key, value in self.pendingConfig.items():
         setattr( rfdPolicyParams, key, value )
      rfdPolicy.rfdPolicyParams = rfdPolicyParams

   def onExit( self ):
      # Commit the configuration on exit
      self.commitRfdPolicyConfig()

   def abort( self ):
      self.pendingConfig = None
      self.abort_ = True
      self.session_.gotoParentMode()

class RouterBgpRouteDistinguisherMode( BgpRouteDistinguisherMode,
                                       BasicCli.ConfigModeBase ):
   name = 'BGP route distinguisher configuration'

   def __init__( self, parent, session ):
      BgpRouteDistinguisherMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
