# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from BasicCli import addShowCommandClass
from CliCommand import Node
from CliMatcher import IntegerMatcher, EnumMatcher
from ShowCommand import ShowCliCommandClass

from CliPlugin.RouteMapCli import CommonTokens as RouteMapCommonTokens
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RoutingBgpShowCli import (
      allVrfExprFactory,
      routeSummaryVrfModel,
      summaryVrfModel,
   )
from CliToken.RoutingBgpShowCliTokens import (
      AddrFamilyAfterShowBgp,
      bgpAfterShow,
      color,
      communityExact,
      detail,
      distinguisher,
      endpoint,
      ipGenAddrMatcher,
      neighbors,
      showCommunity,
      showCommunityList,
      srTeAfterShowBgp,
      summary,
   )
from CliToken.Community import CommunityConstExpr

# This file provides the CLI parser handling for BGP show commands that are
# specific to the SR-TE SAFI

srTePolicyMatcherNode = Node(
   matcher=EnumMatcher( {
      'policies': 'Displays all SR TE policies received and accepted',
      'received-policies': (
         'Displays all SR TE received policies (both accepted and rejected)' ),
      'advertised-policies': 'Displays all SR TE advertised policies',
   } ), maxMatches=1 )

#----------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] summary [vrf <vrfName>]
#----------------------------------------------------------
class ShowBgpSrTeSummary( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] summary [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "summary": summary,
         "VRF": allVrfExprFactory,
      }
   cliModel = summaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler.showBgpSrTeSummaryHandler"

addShowCommandClass( ShowBgpSrTeSummary )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] [ detail ] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTe( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] [ detail ] [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler.showBgpSrTeHandler"

addShowCommandClass( ShowBgpSrTe )

#---------------------------------------------------------------------------------
# "show bgp sr-te endpoint <> color <> distinguisher <> [detail] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTeEndpointColorDistinguisher( ShowCliCommandClass ):
   syntax = """show bgp sr-te
               endpoint ENDPOINT_VALUE
               color COLOR_VALUE
               distinguisher DISTINGUISHER_VALUE
               [ detail ] [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "endpoint": endpoint,
         "ENDPOINT_VALUE": ipGenAddrMatcher,
         "color": color,
         "COLOR_VALUE": IntegerMatcher( 0, 0xFFFFFFFF,
            helpdesc='Policy color value' ),
         "distinguisher": distinguisher,
         "DISTINGUISHER_VALUE": IntegerMatcher( 0, 0xFFFFFFFF,
            helpdesc='Distinguisher value' ),
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler."  \
             "showBgpSrTeEndpointColorDistinguisherHandler"

addShowCommandClass( ShowBgpSrTeEndpointColorDistinguisher )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community [<com>] [exact] [detail] [vrf <name>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTeCommunity( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] community
               { COMM_VALUE }
               [ exact ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "community": showCommunity,
         "COMM_VALUE": CommunityConstExpr, # communityValue
         "exact": communityExact,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler.showBgpSrTeCommunityHandler"

addShowCommandClass( ShowBgpSrTeCommunity )

#-------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community-list <community-list name>
#       [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpSrTeCommunityList( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] community-list
               COMM_LIST_NAME
               [ exact ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "community-list": showCommunityList,
         "COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "exact": communityExact,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler.showBgpSrTeCommunityListHandler"

addShowCommandClass( ShowBgpSrTeCommunityList )

#-----------------------------------------------------------------------------
# "show bgp neighbors <neigh_addr> [ ipv4 | ipv6 ] sr-te
# ( policies | received-policies | advertised-policies ) [ detail ] [vrf <vrfName>]"
#-----------------------------------------------------------------------------
class ShowBgpNeighborsSrTePolicies( ShowCliCommandClass ):
   syntax = """show bgp neighbors PEER_ADDR [ AF_AFTER_BGP ] sr-te
               POLICY_FILTER
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "sr-te": srTeAfterShowBgp,
         "POLICY_FILTER": srTePolicyMatcherNode,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   handler = "RoutingBgpShowCliSrTeHandler.showBgpNeighborsSrTePoliciesHandler"

addShowCommandClass( ShowBgpNeighborsSrTePolicies )
