# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from CliModel import (
   Enum,
   Model,
   Str,
   Int,
)

class NlriModel( Model ):
   """
   Represent BGP Network Layer Reachability Information (NLRI)

   This class is meant to represent any BGP NLRI. As such, most fields will be
   optional. The nlriType field should be used to determine the fields and format
   for rendering.
   """

   # The NLRI type values come from Routing::Bgp::NlriType::NlriTypeEnum. We do not
   # directly import and reuse the Tac enum for two reasons:
   #    1. The 'nlriType' field in this CAPI model should only support NLRI types for
   #       which the necessary fields are present in this model and the rendering of
   #       the NLRI type is supported in the renderNlri() function below.
   #    2. Dependency reasons largely stemming from BgpCommon (where the Tac enum is
   #       defined) depending on both RcfLib/ and RoutingLib/ packages.
   nlriType = Enum( values=( "ipv4Unicast", "ipv6Unicast", "evpnType5Ipv4",
                             "evpnType5Ipv6", "mplsVpnIpv4Unicast",
                             "mplsVpnIpv6Unicast", "ipv4Flowspec",
                             "ipv6Flowspec", "ipv4FlowspecVpn",
                             "ipv6FlowspecVpn", "rtMembership" ),
                    help="Type of BGP NLRI" )
   ipPrefix = IpGenericPrefix( optional=True, help='IPv4 or IPv6 prefix' )
   rd = Str( optional=True, help='Route distinguisher' )
   pathId = Int( optional=True, help='Path ID of the route' )
   flowspecRule = Str( optional=True, help='Bgp flowspec rule' )
   originAs = Str( optional=True, help='Rt membership origin as' )
   routeTarget = Str( optional=True, help='Rt membership route-target prefix' )

   def renderNlri( self ):
      nlriStr = None
      if self.nlriType in [ 'evpnType5Ipv4', 'evpnType5Ipv6' ]:
         # EVPN type 5 NLRI
         if self.rd is not None and self.ipPrefix is not None:
            nlriStr = f"RD: {self.rd} ip-prefix {self.ipPrefix}"
         else:
            assert False, "EVPN NLRIs must have RD and ipPrefix"
      elif self.nlriType in [ 'mplsVpnIpv4Unicast', 'mplsVpnIpv6Unicast' ]:
         # MplsVpn IPv4/IPv6 Unicast NLRI
         if self.rd is not None and self.ipPrefix is not None:
            if len( str( self.ipPrefix ).split( '.' ) ) == 4:
               addrIpType = 4
            else:
               addrIpType = 6
            nlriStr = "RD: {} IPv{} prefix {}".format(
                  self.rd, addrIpType, self.ipPrefix )
         else:
            assert False, "MplsVpn NLRIs must have RD and ipPrefix"
      elif self.nlriType in [ 'ipv4Unicast', 'ipv6Unicast' ]:
         # IPv4/IPv6 Unicast NLRI
         if self.ipPrefix is not None:
            nlriStr = str( self.ipPrefix )
         else:
            assert False, "IPv4/IPv6 Unicast NLRIs must have ipPrefix"
      elif self.nlriType in [ 'ipv4Flowspec', 'ipv6Flowspec',
                              'ipv4FlowspecVpn', 'ipv6FlowspecVpn' ]:
         # Flowspec IPv4/IPv6 Unicast NLRI
         if self.flowspecRule is not None:
            nlriStr = self.flowspecRule
         else:
            assert False, "Flowspec NLRI not present"
      elif self.nlriType == "rtMembership":
         assert self.originAs and self.routeTarget
         nlriStr = f"Origin AS: {self.originAs} Route target: {self.routeTarget}"
      else:
         assert False, "Unexpected NLRI type"
      if self.pathId is not None:
         return nlriStr + f", Rx Path ID: 0x{self.pathId:x}"
      return nlriStr

def createNlriModel( nlriType, nlriAttrs ):
   return NlriModel( nlriType=nlriType, **nlriAttrs )
