# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libRsvp.so*

import Toggles.RsvpToggleLib

import AclCliLib
import AclLib
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMode.Rsvp import (
      RsvpMode,
      GrHelperMode,
      GrSpeakerMode,
      HitlessRestartMode,
      RsvpP2mpMode,
      )
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
import CliPlugin.MplsCli as MplsCli # pylint: disable=consider-using-from-import
import CliPlugin.IntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliToken.Rsvp
import CliParser
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ReversibleSecretCli
from RsvpLib import (
      bandwidthBitsToBytes,
      matcherAdminGroupExc,
      matcherAdminGroupInc,
      matcherAdminGrpLists,
      parseAdminGroupConstraints,
      )
import Tac
import Tracing
from TypeFuture import TacLazyType

t0 = Tracing.trace0

RsvpCliConfig = Tac.Type( 'Rsvp::RsvpCliConfig' )
config = None
routingHardwareRouteStatus = None
aclConfig = None
aclCpConfig = None

EthType = TacLazyType( 'Arnet::EthType' )

FrrMode = TacLazyType( 'Rsvp::Cli::FrrMode' )
LabelLocalTerminationMode = TacLazyType( 'Rsvp::Cli::LabelLocalTerminationMode' )
P2mpMode = TacLazyType( 'Rsvp::Cli::P2mpMode' )
SrlgMode = TacLazyType( 'Rsvp::Cli::SrlgMode' )

RsvpConstants = TacLazyType( 'Rsvp::RsvpConstants' )
RsvpLerAdminGroupConstraints = TacLazyType( 'Rsvp::RsvpLerAdminGroupConstraints' )
RsvpLerAdminGroupConstraintsWithNames = \
   TacLazyType( 'Rsvp::RsvpLerAdminGroupConstraintsWithNames' )
RsvpLspRequestId = TacLazyType( 'Rsvp::RsvpLspRequestId' )
RsvpP2mpCliConfig = TacLazyType( 'Rsvp::RsvpP2mpCliConfig' )

def rsvpSupportedIntfsGuard( mode, token ):
   if mode.intf.routingSupported():
      return None
   return CliParser.guardNotThisPlatform

class RsvpConfigMode( RsvpMode, BasicCli.ConfigModeBase ):
   name = 'Rsvp Configuration'

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName
      RsvpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoRsvpHitlessRestartMode( self, args ):
      childMode = self.childMode( RsvpHitlessRestartConfigMode )
      childMode.enable()
      self.session_.gotoChildMode( childMode )

   def noRsvpHitlessRestartMode( self, args ):
      childMode = self.childMode( RsvpHitlessRestartConfigMode )
      childMode.disable()
      childMode.recoveryIs( {} )

   def gotoRsvpGrHelperMode( self, args ):
      childMode = self.childMode( RsvpGrHelperConfigMode )
      childMode.helperIs()
      self.session_.gotoChildMode( childMode )

   def noRsvpGrHelperMode( self, args ):
      childMode = self.childMode( RsvpGrHelperConfigMode )
      childMode.noHelper()
      childMode.noMaximumRestart( {} )
      childMode.noMaximumRecovery( {} )

   def gotoRsvpGrSpeakerMode( self, args ):
      childMode = self.childMode( RsvpGrSpeakerConfigMode )
      childMode.speakerIs()
      self.session_.gotoChildMode( childMode )

   def noRsvpGrSpeakerMode( self, args ):
      childMode = self.childMode( RsvpGrSpeakerConfigMode )
      childMode.noSpeaker()
      childMode.restartIs( {} )
      childMode.recoveryIs( {} )

# ------------------------------------------------------------------------------
# mpls rsvp hitless-restart role config mode
# Hitless-restart is disabled by default.
# ------------------------------------------------------------------------------
class RsvpHitlessRestartConfigMode( HitlessRestartMode, BasicCli.ConfigModeBase ):
   name = "Mpls Rsvp Graceful restart Speaker Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      HitlessRestartMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enable( self ):
      config.hitlessRestartEnabled = True

   def disable( self ):
      config.hitlessRestartEnabled = False

   def recoveryIs( self, args ):
      config.hitlessRestartRecoveryTime = args.get( 'RECOVERY',
                                           config.hitlessRestartRecoveryTimeDefault )

# ------------------------------------------------------------------------------
# mpls rsvp graceful-restart role (helper|speaker) config mode
# Graceful-restart is disabled by default.
# Enabling speaker implies support for helper as well. The role configurations
# are not mutually exclusive, as the helper-specific timer configurations are under
# the helper submode.
# ------------------------------------------------------------------------------
class RsvpGrHelperConfigMode( GrHelperMode, BasicCli.ConfigModeBase ):
   name = "Mpls Rsvp Graceful restart Helper Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      GrHelperMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def helperIs( self ):
      config.grHelperModeEnabled = True

   def noHelper( self ):
      config.grHelperModeEnabled = False

   def maximumRestartIs( self, args ):
      config.grHelperMaximumRestart = args[ 'MAX_RESTART' ]

   def noMaximumRestart( self, args ):
      self.maximumRestartIs( { 'MAX_RESTART' : None } )

   def maximumRecoveryIs( self, args ):
      config.grHelperMaximumRecovery = args[ 'MAX_RECOVERY' ]

   def noMaximumRecovery( self, args ):
      self.maximumRecoveryIs( { 'MAX_RECOVERY' : None } )

class RsvpGrSpeakerConfigMode( GrSpeakerMode, BasicCli.ConfigModeBase ):
   name = "Mpls Rsvp Graceful restart Speaker Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      GrSpeakerMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def speakerIs( self ):
      config.grSpeakerModeEnabled = True

   def noSpeaker( self ):
      config.grSpeakerModeEnabled = False

   def restartIs( self, args ):
      config.grSpeakerRestart = args.get( 'RESTART', config.grSpeakerRestartDefault )

   def recoveryIs( self, args ):
      config.grSpeakerRecovery = args.get( 'RECOVERY',
                                           config.grSpeakerRecoveryDefault )

#------------------------------------------------------------------------------------
# Cli Tokens
#------------------------------------------------------------------------------------
authKwMatcher = CliMatcher.KeywordMatcher( 'authentication',
                        helpdesc='Configure cryptographic authentication' )
matcherFastReroute = CliMatcher.KeywordMatcher( 'fast-reroute',
                        helpdesc='Configure fast reroute' )
neighborKwMatcher = CliMatcher.KeywordMatcher( 'neighbor',
                        helpdesc='Neighbor-specific configuration' )

# Constants
MAX_INDEX = 2**32 - 1
MAX_INTERVAL = 2**16 - 1
MAX_MUL = 2**8 - 1
MAX_WINDOW = 2**8 - 1
MAX_PREEMPTION_TIMER = 65535
MAX_MAX_SUBGROUP_SIZE = 1000

#--------------------------------------------------------------------------------
# [ no | default ] mpls rsvp
#--------------------------------------------------------------------------------
class MplsRsvpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls rsvp'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MplsCli.mplsNodeForConfig,
      'rsvp': CliToken.Rsvp.rsvpMatcherForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      rsvpMode = mode.childMode( RsvpConfigMode )
      mode.session_.gotoChildMode( rsvpMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for aclType in [ 'ip', 'ipv6' ]:
         if aclType in aclCpConfig.cpConfig:
            _noServiceAcl( mode, aclType )
      config.reset()

BasicCliModes.GlobalConfigMode.addCommandClass( MplsRsvpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable RSVP',
   }

   @staticmethod
   def handler( mode, args ):
      config.enabled = False

   @staticmethod
   def noHandler( mode, args ):
      config.enabled = True

   # default is the to be shutdown
   defaultHandler = handler

RsvpConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mtu signaling
#--------------------------------------------------------------------------------
class MtuSignalingCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu signaling'
   noOrDefaultSyntax = syntax
   data = {
      'mtu' : 'MTU parameters',
      'signaling' : 'Enable MTU signaling',
   }

   @staticmethod
   def handler( mode, args ):
      config.mtuSignalingEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.mtuSignalingEnabled = False

RsvpConfigMode.addCommandClass( MtuSignalingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] refresh interval INTERVAL
#--------------------------------------------------------------------------------
class RefreshIntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh interval INTERVAL'
   noOrDefaultSyntax = 'refresh interval ...'
   data = {
      'refresh' : 'Configure neighbor state refresh',
      'interval' : 'Time between refreshes',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, MAX_INTERVAL,
         helpdesc='Interval in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      config.refreshInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.refreshInterval = config.refreshIntervalDefault

RsvpConfigMode.addCommandClass( RefreshIntervalIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] refresh method METHOD
#--------------------------------------------------------------------------------
class RefreshMethodCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh method METHOD'
   noOrDefaultSyntax = 'refresh method ...'
   data = {
      'refresh': 'Configure neighbor state refresh',
      'method': 'Neighbor refresh mechanism',
      'METHOD': CliMatcher.EnumMatcher( {
         'explicit': 'Send each message individually',
         'bundled': 'Refresh states using message identifier lists',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'METHOD' ] == 'explicit':
         config.refreshMethod = 'refreshExplicit'
      else:
         config.refreshMethod = 'refreshBundled'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.refreshMethod = config.refreshMethodDefault

RsvpConfigMode.addCommandClass( RefreshMethodCmd )

secondUnitMatcher = CliMatcher.KeywordMatcher( 'seconds',
      helpdesc='Value in seconds' )

#--------------------------------------------------------------------------------
# [ no | default ] Bypass interval INTERVAL
#--------------------------------------------------------------------------------
class BypassIntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute bypass tunnel optimization interval INTERVAL seconds'
   noOrDefaultSyntax = 'fast-reroute bypass tunnel optimization interval ...'
   data = {
      'fast-reroute' : matcherFastReroute,
      'bypass' : 'Fast-reroute bypass configuration',
      'tunnel' : 'Bypass tunnel configuration',
      'optimization' : 'Configure bypass optimization interval',
      'interval' : 'Interval between each re-optimization attempt',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1,
         RsvpConstants.maxOptimizationInterval,
         helpdesc='Interval in units of seconds' ),
      'seconds' : secondUnitMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      config.bypassOptimizationInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.bypassOptimizationInterval = config.bypassOptimizationIntervalDefault

RsvpConfigMode.addCommandClass( BypassIntervalIntervalCmd )

# Admin group Cli syntax:
# rsvp bypass administrative-group include all <list> include any <list> exclude
# <list>
#
# where 'include all', 'include any', and 'exclude' can appear at most
# only once. Since the 'include' keyword is repeated twice, and it can not
# appear twice in the 'syntax' below, it is referred to as 'include_all'
# and 'include_any'.
class RsvpPathSpecBypassAdminGrpCmd( CliCommand.CliCommandClass ):
   syntax = ( '''rsvp bypass administrative-group
                { ( include_all all INCL_ALL_MIXED_LISTS ) |
                  ( include_any any INCL_ANY_MIXED_LISTS ) |
                  ( exclude EXCL_MIXED_LISTS ) }''' )
   noOrDefaultSyntax = 'rsvp bypass administrative-group'
   data = {
      'rsvp' : CliCommand.guardedKeyword( 'rsvp',
                  helpdesc='Resource reservation protocol',
                  guard=rsvpSupportedIntfsGuard ),
      'bypass' : 'Bypass LSPs for this interface',
      'administrative-group' : 'Administrative groups for these paths',
      'include_all' : CliCommand.Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'all' : 'Include all of the following admin groups',
      'INCL_ALL_MIXED_LISTS' : matcherAdminGrpLists,
      'include_any' : CliCommand.Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'any' : 'Include any of the following admin groups',
      'INCL_ANY_MIXED_LISTS' : matcherAdminGrpLists,
      'exclude' : CliCommand.Node( matcher=matcherAdminGroupExc, maxMatches=1 ),
      'EXCL_MIXED_LISTS' : matcherAdminGrpLists,
   }

   @staticmethod
   def handler( mode, args ):
      config.bypassAdminGroupConstraintsForIntf[ mode.intf.name ] = \
         parseAdminGroupConstraints( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del config.bypassAdminGroupConstraintsForIntf[ mode.intf.name ]

class RsvpIntfCleaner( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( f'setDefault {self.intf_.name}' )
      del config.bypassAdminGroupConstraintsForIntf[ self.intf_.name ]

CliPlugin.IntfCli.IntfConfigMode.addCommandClass( RsvpPathSpecBypassAdminGrpCmd )
CliPlugin.IntfCli.Intf.registerDependentClass( RsvpIntfCleaner )

#--------------------------------------------------------------------------------
# [ no | default ] hello interval INTERVAL multiplier MULTIPLIER
#--------------------------------------------------------------------------------
class HelloIntervalMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'hello interval INTERVAL multiplier MULTIPLIER'
   noOrDefaultSyntax = 'hello interval ...'
   data = {
      'hello': 'Configure hello messages',
      'interval': 'Time between hello messages',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 60,
         helpdesc='Interval in units of seconds' ),
      'multiplier': 'Timeout multiplier',
      'MULTIPLIER': CliMatcher.IntegerMatcher( 1, 255,
         helpdesc='Number of missed hellos after which the neighbor is expired' ),
   }
   @staticmethod
   def handler( mode, args ):
      config.helloInterval = args[ 'INTERVAL' ]
      config.helloMultiplier = args[ 'MULTIPLIER' ]

   @staticmethod
   def noHandler( mode, args ):
      config.helloInterval = config.helloIntervalDisabled
      config.helloMultiplier = config.helloMultiplierDefault

   @staticmethod
   def defaultHandler( mode, args ):
      config.helloInterval = config.helloIntervalDefault
      config.helloMultiplier = config.helloMultiplierDefault

RsvpConfigMode.addCommandClass( HelloIntervalMultiplierCmd )

#--------------------------------------------------------------------------------
# [ no | default ] authentication type AUTH_TYPE
#--------------------------------------------------------------------------------
class AuthenticationTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication type AUTH_TYPE'
   noOrDefaultSyntax = 'authentication type ...'
   data = {
      'authentication': authKwMatcher,
      'type': 'Authentication mechanism',
      'AUTH_TYPE': CliMatcher.EnumMatcher( {
         'none': 'No Authentication mechanism',
         'md5': 'MD5 Hash',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      config.authType = args[ 'AUTH_TYPE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.authType = config.authTypeDefault

RsvpConfigMode.addCommandClass( AuthenticationTypeCmd )

#--------------------------------------------------------------------------------
#  [ no | default ] authentication sequence-number window WINDOW
#--------------------------------------------------------------------------------
class AuthenticationSequenceNumberWindowWindowCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication sequence-number window WINDOW'
   noOrDefaultSyntax = 'authentication sequence-number window ...'
   data = {
      'authentication': authKwMatcher,
      'sequence-number': 'Index in the sequence',
      'window': 'Reorder window size',
      'WINDOW': CliMatcher.IntegerMatcher( 1, MAX_WINDOW,
         helpdesc='Size of reorder window' ),
   }

   @staticmethod
   def handler( mode, args ):
      config.authSeqWindowSize = args[ 'WINDOW' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.authSeqWindowSize = config.authSeqWindowSizeDefault

RsvpConfigMode.addCommandClass( AuthenticationSequenceNumberWindowWindowCmd )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] authentication index INDEX password PASSWORD
#------------------------------------------------------------------------------------
authIndexKwMatcher = CliMatcher.KeywordMatcher( 'index',
                                                helpdesc='Password index' )
authIndexIntegerMatcher = CliMatcher.IntegerMatcher( 1, MAX_INDEX,
                                                     helpdesc='Password index' )

class AuthCommand( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX password PASSWORD'
   noOrDefaultSyntax = 'authentication index INDEX password ...'
   data = {
      'authentication': authKwMatcher,
      'index': authIndexKwMatcher,
      'INDEX': authIndexIntegerMatcher,
      'password': 'Password',
      'PASSWORD' : ReversibleSecretCli.defaultReversiblePwdCliExpr,
     }

   @staticmethod
   def handler( mode, args ):
      index = args[ 'INDEX' ]
      password = args[ 'PASSWORD' ]
      if( index in config.encodedPassword and
        config.encodedPassword[ index ].clearTextEqual( password.getClearText() ) ):
         # configure the same password, nothing to do
         return
      config.encodedPassword[ index ] = password

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del config.encodedPassword[ args[ 'INDEX' ] ]

RsvpConfigMode.addCommandClass( AuthCommand )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] authentication index INDEX active
#------------------------------------------------------------------------------------
class ActiveAuthCommand( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX active'
   noOrDefaultSyntax = syntax
   data = {
      'authentication': authKwMatcher,
      'index': authIndexKwMatcher,
      'INDEX': authIndexIntegerMatcher,
      'active': 'Use index as active password'
     }

   @staticmethod
   def handler( mode, args ):
      index = args[ 'INDEX' ]
      if index not in config.encodedPassword:
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( 'Password with index %s has not been configured' % index )
      config.activePasswordIndex = index

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.activePasswordIndex = 0

RsvpConfigMode.addCommandClass( ActiveAuthCommand )

hitlessRestartMatcher = CliMatcher.KeywordMatcher( 'hitless-restart',
   helpdesc='RSVP hitless restart' )
hrTimerMatcher = CliMatcher.KeywordMatcher( 'timer',
   helpdesc='RSVP hitless restart timer' )
hitlessRestartRecoveryMatcher = CliMatcher.IntegerMatcher( 1,
   RsvpConstants.hitlessRestartMaximumRecoveryValue,
   helpdesc='Time stale states will be preserved after restart' )
gracefulRestartMatcher = CliMatcher.KeywordMatcher( 'graceful-restart',
   helpdesc='RSVP graceful restart' )
roleMatcher = CliMatcher.KeywordMatcher( 'role',
   helpdesc='RSVP graceful restart role' )
gracefulRestartHelperMatcher = CliMatcher.KeywordMatcher( 'helper',
   helpdesc='RSVP graceful restart helper role' )
gracefulRestartSpeakerMatcher = CliMatcher.KeywordMatcher( 'speaker',
   helpdesc='RSVP graceful restart speaker role' )
grTimerMatcher = CliMatcher.KeywordMatcher( 'timer',
   helpdesc='RSVP graceful restart timer' )
grMaximumMatcher = CliMatcher.KeywordMatcher( 'maximum',
   helpdesc='Maximum value' )
grRestartMatcher = CliMatcher.KeywordMatcher( 'restart',
   helpdesc='Restart timer' )
grSpeakerRestartMatcher = CliMatcher.IntegerMatcher( 1,
   RsvpConstants.maximumTxRestartValue,
   helpdesc='Time neighbors should keep this node alive during restart' )
grHelperMaxRestartMatcher = CliMatcher.IntegerMatcher( 1,
   RsvpConstants.maximumAcceptedRxRestartValue,
   helpdesc='Maximum restart timer value' )
restartRecoveryMatcher = CliMatcher.KeywordMatcher( 'recovery',
   helpdesc='Recovery timer' )
grSpeakerRecoveryMatcher = CliMatcher.IntegerMatcher( 1,
   RsvpConstants.maximumTxRecoveryValue,
   helpdesc='Time neighbors should preserve stale states from this '
            'node after restart' )
grHelperMaxRecoveryMatcher = CliMatcher.IntegerMatcher( 1,
   RsvpConstants.maximumAcceptedRxRecoveryValue,
   helpdesc='Maximum recovery timer value' )

# -----------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] histless-restart
# -----------------------------------------------------------------------------------
class RsvpHitlessRestartCommand( CliCommand.CliCommandClass ):
   syntax = 'hitless-restart'
   noOrDefaultSyntax = syntax
   data = {
      'hitless-restart' : hitlessRestartMatcher,
     }

   handler = RsvpConfigMode.gotoRsvpHitlessRestartMode
   noOrDefaultHandler = RsvpConfigMode.noRsvpHitlessRestartMode

RsvpConfigMode.addCommandClass( RsvpHitlessRestartCommand )

# Hitless Restart timers
# --------------------------------------------------------------------------------
# [ no | default ] timer recovery RECOVERY seconds
# --------------------------------------------------------------------------------
class HitlessRestartRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'timer recovery RECOVERY seconds'
   noOrDefaultSyntax = 'timer recovery ...'
   data = {
      'timer' : hrTimerMatcher,
      'recovery' : restartRecoveryMatcher,
      'RECOVERY' : hitlessRestartRecoveryMatcher,
      'seconds' : secondUnitMatcher,
   }

   handler = RsvpHitlessRestartConfigMode.recoveryIs
   noOrDefaultHandler = handler

RsvpHitlessRestartConfigMode.addCommandClass( HitlessRestartRecoveryCmd )

# -----------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] graceful-restart role helper
# -----------------------------------------------------------------------------------

class GracefulRestartHelperCommand( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart role helper'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart' : gracefulRestartMatcher,
      'role' : roleMatcher,
      'helper' : gracefulRestartHelperMatcher,
     }

   handler = RsvpConfigMode.gotoRsvpGrHelperMode
   noOrDefaultHandler = RsvpConfigMode.noRsvpGrHelperMode

RsvpConfigMode.addCommandClass( GracefulRestartHelperCommand )

# GR Helper timers
# --------------------------------------------------------------------------------
# [ no | default ] timer restart maximum MAX_RESTART seconds
# --------------------------------------------------------------------------------
class GrHelperMaximumRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'timer restart maximum MAX_RESTART seconds'
   noOrDefaultSyntax = 'timer restart maximum ...'
   data = {
      'timer' : grTimerMatcher,
      'restart' : grRestartMatcher,
      'maximum' : grMaximumMatcher,
      'MAX_RESTART' : grHelperMaxRestartMatcher,
      'seconds' : secondUnitMatcher,
   }

   handler = RsvpGrHelperConfigMode.maximumRestartIs
   noOrDefaultHandler = RsvpGrHelperConfigMode.noMaximumRestart

RsvpGrHelperConfigMode.addCommandClass( GrHelperMaximumRestartCmd )

# --------------------------------------------------------------------------------
# [ no | default ] timer recovery maximum MAX_RECOVERY seconds
# --------------------------------------------------------------------------------
class GrHelperMaximumRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'timer recovery maximum MAX_RECOVERY seconds'
   noOrDefaultSyntax = 'timer recovery maximum ...'
   data = {
      'timer' : grTimerMatcher,
      'recovery' : restartRecoveryMatcher,
      'maximum' : grMaximumMatcher,
      'MAX_RECOVERY' : grHelperMaxRecoveryMatcher,
      'seconds' : secondUnitMatcher,
   }

   handler = RsvpGrHelperConfigMode.maximumRecoveryIs
   noOrDefaultHandler = RsvpGrHelperConfigMode.noMaximumRecovery

RsvpGrHelperConfigMode.addCommandClass( GrHelperMaximumRecoveryCmd )

# -----------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] graceful-restart role speaker
# -----------------------------------------------------------------------------------

class GracefulRestartSpeakerCommand( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart role speaker'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart' : gracefulRestartMatcher,
      'role' : roleMatcher,
      'speaker' : gracefulRestartSpeakerMatcher,
     }

   handler = RsvpConfigMode.gotoRsvpGrSpeakerMode
   noOrDefaultHandler = RsvpConfigMode.noRsvpGrSpeakerMode

RsvpConfigMode.addCommandClass( GracefulRestartSpeakerCommand )

# GR Speaker timers
# --------------------------------------------------------------------------------
# [ no | default ] timer restart RESTART seconds
# --------------------------------------------------------------------------------
class GrSpeakerRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'timer restart RESTART seconds'
   noOrDefaultSyntax = 'timer restart ...'
   data = {
      'timer' : grTimerMatcher,
      'restart' : grRestartMatcher,
      'RESTART' : grSpeakerRestartMatcher,
      'seconds' : secondUnitMatcher,
   }

   handler = RsvpGrSpeakerConfigMode.restartIs
   noOrDefaultHandler = handler

RsvpGrSpeakerConfigMode.addCommandClass( GrSpeakerRestartCmd )

# --------------------------------------------------------------------------------
# [ no | default ] timer recovery RECOVERY seconds
# --------------------------------------------------------------------------------
class GrSpeakerRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'timer recovery RECOVERY seconds'
   noOrDefaultSyntax = 'timer recovery ...'
   data = {
      'timer' : grTimerMatcher,
      'recovery' : restartRecoveryMatcher,
      'RECOVERY' : grSpeakerRecoveryMatcher,
      'seconds' : secondUnitMatcher,
   }

   handler = RsvpGrSpeakerConfigMode.recoveryIs
   noOrDefaultHandler = handler

RsvpGrSpeakerConfigMode.addCommandClass( GrSpeakerRecoveryCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACL_NAME [ in ]
# [ no | default ] ipv6 access-group ACL_NAME [ in ]
#--------------------------------------------------------------------------------
def _setServiceAcl( mode, aclName, aclType='ip' ):
   AclCliLib.setServiceAcl( mode, 'rsvp', AclLib.IPPROTO_RSVP,
                            aclConfig, aclCpConfig, aclName, aclType )

def _noServiceAcl( mode, aclType='ip' ):
   AclCliLib.noServiceAcl( mode, 'rsvp', aclConfig, aclCpConfig, None, aclType )

class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ in ]'
   data = {
      'ip': AclCli.ipKwForServiceAclMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL_NAME': AclCli.ipAclNameMatcher,
      'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      _setServiceAcl( mode, args[ 'ACL_NAME' ], aclType='ip' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _noServiceAcl( mode, aclType='ip' )

RsvpConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

class Ipv6AccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ in ]'
   data = {
      'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL_NAME': AclCli.ip6AclNameMatcher,
      'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      _setServiceAcl( mode, args[ 'ACL_NAME' ], aclType='ipv6' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _noServiceAcl( mode, aclType='ipv6' )

RsvpConfigMode.addCommandClass( Ipv6AccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] fast-reroute mode FRR_MODE
#--------------------------------------------------------------------------------
class FastRerouteModeCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute mode FRR_MODE'
   noOrDefaultSyntax = 'fast-reroute mode ...'
   data = {
      'fast-reroute': matcherFastReroute,
      'mode': 'Fast reroute mode',
      'FRR_MODE': CliMatcher.EnumMatcher( {
         'link-protection': 'Protect against failure of the next link',
         'none': 'Disable fast reroute',
         'node-protection': 'Protect against failure of the next node',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      frrStrToMode = {
         'node-protection' : FrrMode.frrNodeProtection,
         'link-protection' : FrrMode.frrLinkProtection,
         'none' : FrrMode.frrNone,
      }
      frrMode = args[ 'FRR_MODE' ]
      frrModeValue = frrStrToMode.get( frrMode )
      if frrModeValue != FrrMode.frrNone and \
         not routingHardwareRouteStatus.hierarchicalFecsEnabled:
         mode.addWarning( 'Fast Re-Route support is not '
                          'available without Hierarchical FECs' )
      config.frrMode = frrModeValue

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.frrMode = config.frrModeDefault

RsvpConfigMode.addCommandClass( FastRerouteModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] srlg [ strict ]
#--------------------------------------------------------------------------------
class SrlgModeCmd( CliCommand.CliCommandClass ):
   syntax = 'srlg [ strict ]'
   noOrDefaultSyntax = 'srlg ...'
   data = {
      'srlg': 'Select SRLG behavior',
      'strict': 'Apply strict SRLG constraint'
   }

   @staticmethod
   def handler( mode, args ):
      srlgMode = SrlgMode.srlgStrict if 'strict' in args else SrlgMode.srlgLoose
      config.srlgMode = srlgMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.srlgMode = config.srlgModeDefault

RsvpConfigMode.addCommandClass( SrlgModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] fast-reroute reversion REVERSION_MODE
#--------------------------------------------------------------------------------
class FastRerouteReversionCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute reversion REVERSION_MODE'
   noOrDefaultSyntax = 'fast-reroute reversion ...'
   data = {
      'fast-reroute': matcherFastReroute,
      'reversion': 'Select reversion behavior',
      'REVERSION_MODE': CliMatcher.EnumMatcher( {
         'local': 'Local revertive repair',
         'global': 'Global revertive repair',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'REVERSION_MODE' ] == 'local':
         config.reversionMode = 'reversionLocal'
      else:
         config.reversionMode = 'reversionGlobal'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.reversionMode = config.reversionModeDefault

RsvpConfigMode.addCommandClass( FastRerouteReversionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] preemption method METHOD [ timer TIMER ]
#--------------------------------------------------------------------------------
class PreemptionTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'preemption method METHOD [ timer TIMER ]'
   noOrDefaultSyntax = 'preemption method ...'
   data = {
      'preemption' : 'Configure preemption',
      'method' : 'Select preemption method',
      'METHOD' : CliMatcher.EnumMatcher( {
         'soft' : 'Soft preemption',
         'hard' : 'Hard preemption',
      } ),
      'timer' : 'Time limit for LSP teardown',
      'TIMER' : CliMatcher.IntegerMatcher( 1, MAX_PREEMPTION_TIMER,
         helpdesc='Timer value in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'METHOD' ] == 'soft':
         config.preemptionTimer = config.preemptionTimerDefault
         if 'timer' in args:
            config.preemptionTimer = args[ 'TIMER' ]
      else:
         config.preemptionTimer = config.preemptionTimerDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.preemptionTimer = config.preemptionTimerDefault

RsvpConfigMode.addCommandClass( PreemptionTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] label local-termination { implicit-null | explicit-null }
#--------------------------------------------------------------------------------
class LabelLocalTerminationCmd( CliCommand.CliCommandClass ):
   syntax = 'label local-termination ( implicit-null | explicit-null )'
   noOrDefaultSyntax = 'label local-termination ...'
   data = {
      'label' : 'Label to be advertised',
      'local-termination' : 'Local termination label to be advertised',
      'implicit-null' : 'Advertise implicit null',
      'explicit-null' : 'Advertise explicit null'
   }

   @staticmethod
   def handler( mode, args ):
      termMode = LabelLocalTerminationMode.explicitNull if 'explicit-null' in args \
                    else LabelLocalTerminationMode.implicitNull
      config.labelLocalTerminationMode = termMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.labelLocalTerminationMode = config.labelLocalTerminationModeDefault

RsvpConfigMode.addCommandClass( LabelLocalTerminationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor <IP> authentication type AUTH_TYPE
#--------------------------------------------------------------------------------
# TODO BUG883684: add a way to remove all neighbor authentication
class NeighborAuthTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor NEIGHBOR_IP authentication type AUTH_TYPE'
   noOrDefaultSyntax = 'neighbor NEIGHBOR_IP authentication type ...'
   data = {
      'neighbor' : neighborKwMatcher,
      'NEIGHBOR_IP' : IpGenAddrMatcher.IpGenAddrMatcher(
                        helpdesc="Neighbor's interface IPv4 or IPv6 address" ),
      'authentication' : authKwMatcher,
      'type' : 'Authentication mechanism',
      'AUTH_TYPE' : CliMatcher.EnumMatcher( {
         'none' : 'No Authentication mechanism',
         'md5' : 'MD5 Hash',
      } )
   }
   @staticmethod
   def handler( mode, args ):
      neighborIp = args[ 'NEIGHBOR_IP' ]
      config.authTypeForNeighbor[ neighborIp ] = args[ 'AUTH_TYPE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      neighborIp = args[ 'NEIGHBOR_IP' ]
      del config.authTypeForNeighbor[ neighborIp ]

RsvpConfigMode.addCommandClass( NeighborAuthTypeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor <IP> authentication index INDEX active
#--------------------------------------------------------------------------------
class NeighborAuthIndexCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor NEIGHBOR_IP authentication index INDEX active'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : neighborKwMatcher,
      'NEIGHBOR_IP' : IpGenAddrMatcher.IpGenAddrMatcher(
                        helpdesc="Neighbor's interface IPv4 or IPv6 address" ),
      'authentication' : authKwMatcher,
      'index' : authIndexKwMatcher,
      'INDEX' : authIndexIntegerMatcher,
      'active' : 'Use index as active password',
   }
   @staticmethod
   def handler( mode, args ):
      neighborIp = args[ 'NEIGHBOR_IP' ]
      index = args[ 'INDEX' ]
      if index not in config.encodedPassword:
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( 'Password with index %s has not been configured' % index )
      config.activePasswordIndexForNeighbor[ neighborIp ] = index

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      neighborIp = args[ 'NEIGHBOR_IP' ]
      del config.activePasswordIndexForNeighbor[ neighborIp ]

RsvpConfigMode.addCommandClass( NeighborAuthIndexCmd )

# -----------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] p2mp
# -----------------------------------------------------------------------------------
class RsvpP2mpConfigMode( RsvpP2mpMode, BasicCli.ConfigModeBase ):
   name = "Mpls Rsvp P2mp Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      RsvpP2mpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

p2mpKeywordMatcher = CliMatcher.KeywordMatcher( 'p2mp',
   helpdesc='P2MP configuration' )

class P2mpCommand( CliCommand.CliCommandClass ):
   syntax = 'p2mp'
   noOrDefaultSyntax = syntax
   data = {
      'p2mp' : p2mpKeywordMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # If the mode has not been previously configured, enable P2MP when entering it.
      if config.p2mpConfig.mode == P2mpMode.p2mpModeNone:
         # If mode was already none, config was already default.
         # Create a nominal with defaults and mode set to entered.
         config.p2mpConfig = RsvpP2mpCliConfig( P2mpMode.p2mpModeEntered )

      childMode = mode.childMode( RsvpP2mpConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Clear existing config, create a nominal with defaults and mode set to none.
      config.p2mpConfig = RsvpP2mpCliConfig( P2mpMode.p2mpModeNone )
      config.p2mpDebugLspRequestColl.lspRequest.clear()

RsvpConfigMode.addCommandClass( P2mpCommand )

# --------------------------------------------------------------------------------
# (config-mpls-rsvp-p2mp) # [ no | default ] sub-group limit <1-1000> sub-lsps
# --------------------------------------------------------------------------------
subLspUnitMatcher = CliMatcher.KeywordMatcher( 'sub-lsps',
                        helpdesc='Value in sub-LSPs' )

def setSubGroupLimit( subGroupMaxSize ):
   p2mpConfig = Tac.nonConst( config.p2mpConfig )
   p2mpConfig.subGroupMaxSize = subGroupMaxSize
   config.p2mpConfig = p2mpConfig

class RsvpP2mpSubGroupLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'sub-group limit SIZE sub-lsps'
   noOrDefaultSyntax = 'sub-group limit ...'
   data = {
      'sub-group' : 'Sub-group configuration',
      'limit' : 'Limit of the maximum number of sub-LSPs per P2MP sub-group',
      'SIZE' : CliMatcher.IntegerMatcher( 1, MAX_MAX_SUBGROUP_SIZE,
         helpdesc='Number of sub-LSPs' ),
      'sub-lsps' : subLspUnitMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      setSubGroupLimit( args[ 'SIZE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setSubGroupLimit( RsvpP2mpCliConfig.subGroupMaxSizeDefault )

if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
   RsvpP2mpConfigMode.addCommandClass( RsvpP2mpSubGroupLimitCmd )

# --------------------------------------------------------------------------------
# (config-mpls-rsvp-p2mp) # [ no | default ] disabled
# --------------------------------------------------------------------------------
def setP2mpMode( p2mpMode ):
   p2mpConfig = Tac.nonConst( config.p2mpConfig )
   p2mpConfig.mode = p2mpMode
   config.p2mpConfig = p2mpConfig

class RsvpP2mpDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled' : 'Disable P2MP',
   }

   @staticmethod
   def handler( mode, args ):
      setP2mpMode( P2mpMode.p2mpModeExplicitDisabled )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setP2mpMode( P2mpMode.p2mpModeEntered )

RsvpP2mpConfigMode.addCommandClass( RsvpP2mpDisabledCmd )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] debug tunnel id <id>
#                                                sub-id <subId>
#                                                src <srcIp
#                                                dst <dstIp>
#                                                ether-type <ethType>
#                                                lpd <node|link|off>
#                                                session-name <name>
#                                                [bandwidth <0-1000>
#                                                (bps|kbps|mbps|gbps)]
#                                                [preemption <soft|hard>]
#                                                [setup-pri <0-7>]
#                                                [hold-pri <0-7>]
#                                                hops <ip1> [<ip2 ...]
#------------------------------------------------------------------------------------
MIN_TUNNEL_ID = 0
MAX_TUNNEL_ID = 2**16 - 1
MIN_TUNNEL_SUBID = 0
MAX_TUNNEL_SUBID = 2**16 - 1
MIN_PRI = 0
MAX_PRI = 7

matcherPriority = CliMatcher.IntegerMatcher( MIN_PRI, MAX_PRI, helpdesc='Priority' )

class DebugTunnelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'debug tunnel id REQ_ID sub-id REQ_SUB_ID src SRC_IP dst DST_IP '
              ' ether-type ETH_TYPE lpd LPD session-name SESSION_NAME '
              ' [ bandwidth BANDWIDTH UNIT ] [ preemption PREEMPTION ] '
              ' [ setup-prio SETUP_PRIO ] [ hold-prio HOLD_PRIO ] '
              ' [ mtu-signaling ] hops { HOPS }' )
   noOrDefaultSyntax = 'debug tunnel [ id REQ_ID ] [ sub-id REQ_SUB_ID ] ...'
   data = {
      'debug': 'Debug configuration',
      'tunnel': 'Tunnel headend setup',
      'id': 'Tunnel identifier',
      'REQ_ID': CliMatcher.IntegerMatcher( MIN_TUNNEL_ID, MAX_TUNNEL_ID,
         helpdesc='Tunnel ID integer' ),
      'sub-id': 'Tunnel sub-ID',
      'REQ_SUB_ID': CliMatcher.IntegerMatcher( MIN_TUNNEL_SUBID, MAX_TUNNEL_SUBID,
         helpdesc='Tunnel sub-ID integer' ),
      'src': 'Tunnel source IP address',
      'SRC_IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' ),
      'dst': 'Tunnel desination IP address',
      'DST_IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' ),
      'ether-type': 'Tunnel EtherType',
      'ETH_TYPE': CliMatcher.EnumMatcher( {
         'ip': 'IP EtherType (0x800)',
         'ip6': 'IPv6 EtherType (0x86DD)',
         'mpls': 'MPLS EtherType (0x8847)',
      } ),
      'lpd': 'Tunnel local protection desired',
      'LPD': CliMatcher.EnumMatcher( {
         'node': 'Local protection on with node protection',
         'link': 'Local protection on with link protection',
         'off': 'Local protection off',
      } ),
      'session-name': 'Tunnel Session Name',
      'SESSION_NAME': CliMatcher.QuotedStringMatcher( helpdesc='Tunnel SessionName',
         pattern=CliParser.namePattern ),
      'bandwidth': 'Bandwidth characteristics of the flow',
      'BANDWIDTH': CliMatcher.FloatMatcher( 0, 1000,
         helpdesc='Bandwidth requested', precisionString='%.2f' ),
      'UNIT': CliMatcher.EnumMatcher( {
         'bps' : 'Maximum reservable bandwidth in bits per second',
         'kbps' : 'Maximum reservable bandwidth in kilobits per second',
         'mbps' : 'Maximum reservable bandwidth in megabits per second',
         'gbps' : 'Maximum reservable bandwidth in gigabits per second',
      } ),
      'preemption' : 'Desired LSP preemption mode',
      'PREEMPTION' : CliMatcher.EnumMatcher( {
         'soft' : 'Soft preemption',
         'hard' : 'Hard preemption',
      } ),
      'setup-prio' : 'Setup priority',
      'SETUP_PRIO' : matcherPriority,
      'hold-prio' : 'Hold priority',
      'HOLD_PRIO' : matcherPriority,
      'mtu-signaling' : 'Enable MTU Signaling',
      'hops': 'Tunnel Hops',
      'HOPS': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      reqId = args[ 'REQ_ID' ]
      reqSubId = args[ 'REQ_SUB_ID' ]
      srcIp = args[ 'SRC_IP' ]
      dstIp = args[ 'DST_IP' ]
      ethType = args[ 'ETH_TYPE' ]
      lpd = args[ 'LPD' ]
      sessionName = args[ 'SESSION_NAME' ]
      bandwidth = None
      if 'bandwidth' in args:
         bandwidth = ( args[ 'BANDWIDTH' ], args[ 'UNIT' ] )
      spd = args.get( 'PREEMPTION' )
      setupPriority = args.get( 'SETUP_PRIO', 7 )
      holdPriority = args.get( 'HOLD_PRIO', 0 )
      mtuSignalingEnabled = 'mtu-signaling' in args
      hops = args[ 'HOPS' ]
      # pylint: disable-msg=W0102, dangerous-default-value
      lspReqId = RsvpLspRequestId( reqId, reqSubId )

      # Convert ethType to Enum value
      ethTypes = { 'ip' : EthType.ethTypeIp,
                   'ip6' : EthType.ethTypeIp6,
                   'mpls' : EthType.ethTypeMpls }
      ethType = ethTypes[ ethType ]

      # Identify if nodeProtectionDesired should be set
      npd = lpd == 'node'

      # Convert lpd to Bool
      lpd = ( lpd == 'node' or lpd == 'link' ) # pylint: disable=consider-using-in

      # Convert Bandwidth to Bps
      bwMultiplier = { 'bps': 1, 'kbps': 1e3, 'mbps': 1e6, 'gbps': 1e9 }

      if bandwidth is None:
         bandwidthBps = 0
      else:
         unit = bandwidth[ 1 ]
         bw = int( bandwidth[ 0 ] * bwMultiplier[ unit ] )
         if bw % 8 > 0:
            mode.addWarning( "Bandwidth will be rounded to whole bytes" )
         bandwidthBps = bandwidthBitsToBytes( bw )

      # Convert spd to Bool
      spd = ( spd == 'soft' )

      oldReq = config.debugLspRequestColl.debugLspReq.get( lspReqId )
      # If this ID already has a request, and it has not changed, don't do anything
      if oldReq is not None:
         if ( oldReq.srcIp == srcIp
              and oldReq.dstIp == dstIp
              and oldReq.ethType == ethType
              and oldReq.setupPriority == setupPriority
              and oldReq.holdPriority == holdPriority
              and oldReq.localProtectionDesired == lpd
              and oldReq.nodeProtectionDesired == npd
              and oldReq.softPreemptionDesired == spd
              and oldReq.mtuSignalingEnabled == mtuSignalingEnabled
              and oldReq.sessionName == sessionName
              and oldReq.bandwidth == bandwidthBps
              and list( oldReq.hop.items() ) == list( enumerate( hops ) ) ):
            t0( 'Debug tunnel request has not changed' )
            return
         t0( 'Debug tunnel request has changed' )
      else:
         t0( 'New debug tunnel request' )

      # Request is new or has changed
      req = config.debugLspRequestColl.debugLspReq.newMember( lspReqId )
      req.srcIp = srcIp
      req.dstIp = dstIp
      req.ethType = ethType
      req.setupPriority = setupPriority
      req.holdPriority = holdPriority
      req.localProtectionDesired = lpd
      req.nodeProtectionDesired = npd
      req.softPreemptionDesired = spd
      req.mtuSignalingEnabled = mtuSignalingEnabled
      req.sessionName = sessionName
      req.bandwidth = bandwidthBps
      req.hop.clear()
      for i, hop in enumerate( hops ):
         req.hop[ i ] = hop
      req.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      reqId = args.get( 'REQ_ID' )
      reqSubId = args.get( 'REQ_SUB_ID' )

      if reqId is not None and reqSubId is not None:
         lspReqId = RsvpLspRequestId( reqId, reqSubId )
         del config.debugLspRequestColl.debugLspReq[ lspReqId ]
      else:
         # Wipe all requests
         config.debugLspRequestColl.debugLspReq.clear()

RsvpConfigMode.addCommandClass( DebugTunnelCmd )

def Plugin( entityManager ):
   global config
   config = ConfigMount.mount( entityManager, RsvpCliConfig.mountPath,
                               'Rsvp::RsvpCliConfig', 'w' )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                           'routing/hardware/route/status',
                           'Routing::Hardware::RouteStatus',
                           'r' )

   global aclConfig, aclCpConfig
   aclConfig = ConfigMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'w' )

   aclCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                    'Acl::Input::CpConfig', 'w' )
