# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import difflib
from collections import namedtuple

import BasicCli
import CliCommand
from CliCommand import Node
from CliMode.RsvpLer import (
      RsvpTeMode,
      RsvpPathSpecMode,
      RsvpP2mpLeafMode,
      RsvpP2mpTunnelProfileMode,
      RsvpP2mpTunnelSpecMode,
      RsvpTunnelBaseMode,
      RsvpTunnelProfileMode,
      RsvpTunnelSpecMode,
      )
import CliMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin import IntfCli
from CliPlugin import TeCli
from CliPlugin.TeCli import (
      RouterGlobalTeMode,
      )
import CliToken.RsvpLer
import ConfigMount
import LazyMount
from RsvpLib import (
      bandwidthBitsToBytes,
      dictToTaccForCli,
      dictToTaccNominalForCli,
      matcherAdminGroupExc,
      matcherAdminGroupInc,
      matcherAdminGrpLists,
      parseAdminGroupConstraints,
      )
import ShowCommand
import Tac
from TypeFuture import TacLazyType
import Tracing
import Toggles.RsvpToggleLib

t0 = Tracing.trace0

def IpGenAddr( *args, **kwargs ):
   # Need constness for dict keys
   return Tac.ValueConst( 'Arnet::IpGenAddr', *args, **kwargs )

CspfConstraintIdx = TacLazyType( "Cspf::ConstraintAttrIndex" )

FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
FeatureState = TacLazyType( 'Ale::FlexCounter::ConfigState' )
TristateU32 = TacLazyType( 'Ark::TristateU32' )

RsvpLerAdminGroupConstraintsWithNames = \
   TacLazyType( 'Rsvp::RsvpLerAdminGroupConstraintsWithNames' )
RsvpLerAutoBwParam = TacLazyType( 'Rsvp::RsvpLerAutoBwParam' )
RsvpLerCliConfig = TacLazyType( 'Rsvp::RsvpLerCliConfig' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLibConstants = TacLazyType( 'Rsvp::RsvpLibConstants' )
RsvpFrrMode = TacLazyType( 'Rsvp::Cli::FrrMode' )
RsvpLerLdpTunnelingConfig = TacLazyType( 'Rsvp::RsvpLerLdpTunnelingConfig' )
RsvpLerLspSelfPingMode = TacLazyType( 'Rsvp::RsvpLerLspSelfPingMode' )
RsvpLerMetricConfig = TacLazyType( 'Rsvp::RsvpLerMetricConfig' )
RsvpLerPathSpec = TacLazyType( 'Rsvp::RsvpLerPathSpec' )
RsvpLerPathSpecHop = TacLazyType( 'Rsvp::RsvpLerPathSpecHop' )
RsvpLerPathSpecId = TacLazyType( 'Rsvp::RsvpLerPathSpecId' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )
RsvpLerSplitBwParam = Tac.Type( 'Rsvp::RsvpLerSplitBwParam' )
RsvpLerSplitTunnelConfig = Tac.Type( 'Rsvp::RsvpLerSplitTunnelConfig' )
RsvpLerTunnelBandwidthConfig = TacLazyType( 'Rsvp::RsvpLerTunnelBandwidthConfig' )
RsvpLerTunnelPriority = TacLazyType( 'Rsvp::RsvpLerTunnelPriority' )
RsvpLerTunnelSpecId = TacLazyType( 'Rsvp::RsvpLerTunnelSpecId' )
RsvpLerTunnelProfileId = TacLazyType( 'Rsvp::RsvpLerTunnelProfileId' )
RsvpSessionType = TacLazyType( 'Rsvp::RsvpSessionType' )

lspSelfPingNone = RsvpLerLspSelfPingMode.lspSelfPingNone
lspSelfPingEnabled = RsvpLerLspSelfPingMode.lspSelfPingEnabled
pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType
rsvpLerSubTunnel = TacLazyType( "Tunnel::TunnelTable::TunnelType" ).rsvpLerSubTunnel
tunnelSourceCli = TacLazyType( 'Rsvp::RsvpLerTunnelSource' ).tunnelSourceCli
p2mpLspTunnel = RsvpSessionType.p2mpLspTunnel
p2pLspTunnel = RsvpSessionType.p2pLspTunnel

def makePathSpecHop( hop ):
   pathSpecHop = RsvpLerPathSpecHop( hop.ip, hop.loose )
   pathSpecHop.node = hop.node
   return pathSpecHop


Hop = namedtuple( 'Hop', [ 'ip', 'loose', 'node' ], defaults=[ False, False ] )

class UniqueHopList:
   '''Keeps a list of hops, where each hop is of type Hop. The list is guaranteed
   to have all `ip` values of the hops unique. This is achieved by removing
   a previously existing `ip` if a new hop is added that contains the same `ip`.

   To keep this guarantee, users should only modify the list with the provided
   accessors. They ensure that the internal data structures are kept consistent.
   '''
   def __init__( self ):
      self._hopList = []
      self._ipSet = set()

   def __len__( self ):
      return len( self._hopList )

   def getHopList( self ):
      return self._hopList[ : ]

   def getIpSet( self ):
      return self._ipSet.copy()

   def removeIp( self, ipToRemove ):
      if ipToRemove not in self._ipSet:
         return
      self._hopList = [ hop for hop in self._hopList if hop.ip != ipToRemove ]
      self._ipSet.remove( ipToRemove )

   def findIpIndex( self, ipToFind ):
      return next( i for i, hop in enumerate( self._hopList )
                     if hop.ip == ipToFind )

   def insertHopBeforeIp( self, ip, hopToInsert ):
      '''Insert hopToInsert before position of ip in hop list.
      Expects that ip is actually already in the list.
      '''
      assert ip in self._ipSet

      ipToInsert = hopToInsert.ip
      prevIndex = self.findIpIndex( ip )
      if ipToInsert == ip:
         # IP is replacing itself
         self._hopList[ prevIndex ] = hopToInsert
         return
      self.removeIp( ipToInsert )
      self._hopList.insert( prevIndex, hopToInsert )
      self._ipSet.add( ipToInsert )

   def insertHopAfterIp( self, ip, hopToInsert ):
      '''Insert hopToInsert after position of ip in hop list.
      Expects that ip is actually already in the list.
      '''
      assert ip in self._ipSet

      ipToInsert = hopToInsert.ip
      prevIndex = self.findIpIndex( ip )
      if ipToInsert == ip:
         # IP is replacing itself
         self._hopList[ prevIndex ] = hopToInsert
         return
      self.removeIp( ipToInsert )
      self._hopList.insert( prevIndex + 1, hopToInsert )
      self._ipSet.add( ipToInsert )

   def appendHop( self, hop ):
      self.removeIp( hop.ip )
      self._hopList.append( hop )
      self._ipSet.add( hop.ip )

   def appendPathSpecHop( self, hop ):
      self.appendHop( Hop( hop.hopIp, loose=hop.loose, node=hop.node ) )

lerClient = None
config = None
lerConfig = None
fcFeatureConfigDir = None
tunnelCountersPriorityTable = None
teConfig = None

class RsvpTeConfigMode( RsvpTeMode, BasicCli.ConfigModeBase ):
   name = "RSVP LER Configuration"

   def __init__( self, parent, session ):
      RsvpTeMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#---------------------------------
# [ no | default ] rsvp
#---------------------------------
class CfgRsvpCmd( CliCommand.CliCommandClass ):
   syntax = 'rsvp'
   noOrDefaultSyntax = syntax
   data = {
      'rsvp': CliToken.RsvpLer.rsvpNodeForLerConfig,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RsvpTeConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.reset()
      lerClient.tunnelProfile.clear()
      lerClient.tunnelSpec.clear()
      lerClient.pathSpec.clear()
      teConfig.rsvpEcmpLeastFill = True

RouterGlobalTeMode.addCommandClass( CfgRsvpCmd )

#---------------------------------------------------------------
# Remove all rsvp configs when the parent is removed
# i.e., "no router traffic-engineering" in config mode.
#---------------------------------------------------------------
class TeSubMode( TeCli.TeDependentBase ):
   def setDefault( self ):
      CfgRsvpCmd.noOrDefaultHandler( None, None )

#--------------------------------------------------------------------------------
# [ no | default ] local-interface INTF
#--------------------------------------------------------------------------------
class LocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': 'Local interface',
      'INTF': IntfCli.Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      intfId = intf.name
      config.localIntf = intfId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.localIntf = ''

RsvpTeConfigMode.addCommandClass( LocalInterfaceCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bandwidth auto tunnel ...
# --------------------------------------------------------------------------------
matcherGlobalBwKeyword = CliMatcher.KeywordMatcher(
   'bandwidth', helpdesc='Global bandwidth settings' )
matcherGlobalAutoBwKeyword = CliMatcher.KeywordMatcher(
   'auto', helpdesc='Global auto bandwidth settings' )
matcherGlobalAutoBwTunnelKeyword = CliMatcher.KeywordMatcher(
   'tunnel', helpdesc='Tunnel-related auto bandwidth settings' )
# --------------------------------------------------------------------------------
# bandwidth auto tunnel failure computed-bandwidth (preserved|reset)
# --------------------------------------------------------------------------------
matcherTunnelFailureKeyword = CliMatcher.KeywordMatcher(
   'failure', helpdesc='Auto bandwidth behavior on tunnel failure' )
matcherTunnelFailureBwKeyword = CliMatcher.KeywordMatcher(
   'computed-bandwidth',
   helpdesc='Last computed bandwidth behavior on tunnel failure' )
matcherTunnelFailureBwModeKeyword = CliMatcher.EnumMatcher( {
   'preserved': 'Last computed bandwidth value is preserved',
   'reset': 'Computed bandwidth value is reset to minimum',
} )

class GlobalAutoBandwidthTunnelFailureCmd( CliCommand.CliCommandClass ):
   syntax = 'bandwidth auto tunnel failure computed-bandwidth PRESERVE_MODE'
   noOrDefaultSyntax = 'bandwidth auto tunnel failure computed-bandwidth ...'
   data = {
      'bandwidth': matcherGlobalBwKeyword,
      'auto': matcherGlobalAutoBwKeyword,
      'tunnel': matcherGlobalAutoBwTunnelKeyword,
      'failure': matcherTunnelFailureKeyword,
      'computed-bandwidth': matcherTunnelFailureBwKeyword,
      'PRESERVE_MODE': matcherTunnelFailureBwModeKeyword,
   }

   @staticmethod
   def handler( mode, args ):
      preserve = args[ 'PRESERVE_MODE' ] == 'preserved'
      config.lastBwPreserved = preserve

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.lastBwPreserved = True

RsvpTeConfigMode.addCommandClass( GlobalAutoBandwidthTunnelFailureCmd )

# --------------------------------------------------------------------------------
# [ no | default ] optimization interval INTERVAL seconds
# --------------------------------------------------------------------------------
MAX_OPTIMIZATION_INTERVAL = 2**32 - 1
matcherOptimizationKeyword = CliMatcher.KeywordMatcher( 'optimization',
   helpdesc='Configure periodic tunnel optimization' )
matcherIntervalKeyword = CliMatcher.KeywordMatcher( 'interval',
   helpdesc='Time between tunnel optimizations' )
matcherInterval = CliMatcher.IntegerMatcher( 1, MAX_OPTIMIZATION_INTERVAL,
   helpdesc='Interval value' )
matcherSecondsKeyword = CliMatcher.KeywordMatcher( 'seconds',
   helpdesc='Optimization interval to be specified as seconds' )

class RsvpOptimizationCmd( CliCommand.CliCommandClass ):
   syntax = 'optimization interval INTERVAL seconds'
   noOrDefaultSyntax = 'optimization ...'
   data = {
      'optimization': matcherOptimizationKeyword,
      'interval': matcherIntervalKeyword,
      'INTERVAL': matcherInterval,
      'seconds': matcherSecondsKeyword,
   }

   @staticmethod
   def handler( mode, args ):
      config.optimizationInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.optimizationInterval = RsvpLerConstants.optimizationIntervalDisabled

RsvpTeConfigMode.addCommandClass( RsvpOptimizationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] path <name> ( explicit | dynamic )
#--------------------------------------------------------------------------------
class RsvpPathSpecConfigMode( RsvpPathSpecMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, pathSpecName, pathSpecType, p2mp ):
      param = ( pathSpecName, pathSpecType, p2mp )
      RsvpPathSpecMode.__init__( self, param=param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.pathSpecId = RsvpLerPathSpecId( self.pathSpecName, tunnelSourceCli )
      self.pathSpec = None
      self.p2mp = p2mp
      self.pendingConfig = self.buildPendingConfigFromPathSpec()

   def getActivePathSpec( self, createIfNotExist=False ):
      pathSpec = None
      if createIfNotExist:
         sessionType = p2mpLspTunnel if self.p2mp else p2pLspTunnel
         pathSpec = lerClient.pathSpec.newMember( self.pathSpecId, sessionType )
      else:
         pathSpec = lerClient.pathSpec.get( self.pathSpecId )
      return pathSpec

   def getPendingPathSpec( self ):
      hops = self.pendingConfig[ 'hops' ].getHopList()
      leaf = self.pendingConfig[ 'leaf' ]
      excludeHops = self.pendingConfig[ 'excludeHops' ]
      adminGroupConstraints = self.pendingConfig[ 'adminGroupConstraints' ]
      pathSpec = self.buildPathSpec( hops, leaf=leaf, excludeHops=excludeHops,
         adminGroupConstraints=adminGroupConstraints )
      return pathSpec

   def buildPathSpec( self, hops, leaf=None, excludeHops=None,
         adminGroupConstraints=RsvpLerAdminGroupConstraintsWithNames(),
         pathSpec=None ):
      '''
      If pathSpec is not provided, a new PathSpec (orphan) will be created.
      The pathSpec is then populated using the arguments passed to this function.
      Being able to create a temporary pathSpec is useful for `show pending` and
      `show diff`, when we need to check the difference between two objects,
      including one that is only 'pending' from now, i.e only exists through a
      pending configuration (self.pendingConfig).
      '''
      # Set default values
      excludeHops = excludeHops or set()

      # Create actual pathSpec if needed
      if pathSpec is None:
         pathId = RsvpLerPathSpecId( self.pathSpecName, tunnelSourceCli )
         sessionType = p2mpLspTunnel if self.p2mp else p2pLspTunnel
         pathSpec = RsvpLerPathSpec( pathId, sessionType )

      # Create leaves
      leafs = []
      if leaf is not None:
         for dstIp in leaf:
            oldPath = pathSpec.leaf.get( dstIp,
                        Tac.newInstance( 'Rsvp::RsvpLerP2mpLeaf', dstIp ) )
            newPath = Tac.newInstance( 'Rsvp::RsvpLerP2mpLeaf', dstIp )
            # Copy hops
            for hop in leaf[ dstIp ][ 'hops' ].getHopList():
               newPath.includeHop.push( makePathSpecHop( hop ) )
            # Set version based on old version
            newPath.version = oldPath.version
            if oldPath.includeHop.values() != newPath.includeHop.values():
               newPath.version += 1
            leafs.append( ( dstIp, newPath ) )

      # Build a config dict to use dictToTaccForCli
      includeHopList = []
      for i, hop in enumerate( hops ):
         includeHopList.append( ( i, makePathSpecHop( hop ) ) )
      excludeHopList = []
      for excludeHop in excludeHops:
         excludeHopList.append( ( excludeHop, True ) )

      configDict = {
            'pathSpecType': self.pathSpecType,
            'leaf': leafs,
            'includeHop': includeHopList,
            'excludeHop': excludeHopList,
            'adminGroupConstraints': adminGroupConstraints,
      }
      changed = dictToTaccForCli( configDict, pathSpec )
      if changed:
         pathSpec.changeCount = Tac.now()

      return pathSpec

   def buildPendingConfigFromPathSpec( self ):
      pendingConfig = {
         'hops': UniqueHopList(), 'leaf': {}, 'excludeHops': set(),
         'adminGroupConstraints': RsvpLerAdminGroupConstraintsWithNames(),
      }
      pathSpec = self.getActivePathSpec( createIfNotExist=False )
      if pathSpec is None:
         return pendingConfig

      for dstIp in pathSpec.leaf:
         pendingConfig[ 'leaf' ][ dstIp ] = {}
         pendingConfig[ 'leaf' ][ dstIp ][ 'hops' ] = UniqueHopList()
         # Include hops
         for hop in pathSpec.leaf[ dstIp ].includeHop.values():
            pendingConfig[ 'leaf' ][ dstIp ][ 'hops' ].appendPathSpecHop( hop )
      for pathSpecHop in pathSpec.includeHop.values():
         pendingConfig[ 'hops' ].appendPathSpecHop( pathSpecHop )
      for excludeHop in pathSpec.excludeHop:
         pendingConfig[ 'excludeHops' ].add( excludeHop )

      pendingConfig[ 'adminGroupConstraints' ] = pathSpec.adminGroupConstraints
      return pendingConfig

   def commitContext( self ):
      if self.pendingConfig is None:
         # Abort
         return
      leaf = self.pendingConfig[ 'leaf' ]
      hops = self.pendingConfig[ 'hops' ].getHopList()
      excludeHops = self.pendingConfig[ 'excludeHops' ]
      adminGroupConstraints = self.pendingConfig[ 'adminGroupConstraints' ]
      self.pathSpec = self.getActivePathSpec( createIfNotExist=True )
      self.buildPathSpec( hops, leaf=leaf, excludeHops=excludeHops,
         adminGroupConstraints=adminGroupConstraints,
         pathSpec=self.pathSpec )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      subPrefix = '   '
      pathSpec = self.getPendingPathSpec()
      cliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )
      print( self.enterCmd() )
      for cliCmd in cliCmds:
         print( subPrefix + cliCmd )
      for dstIp in pathSpec.leaf:
         leafMode = RsvpP2mpLeafMode( ( self, str( dstIp ) ) )
         print( subPrefix + leafMode.enterCmd() )
         for cmd in leafMode.leafToCliCmds( pathSpec.leaf[ dstIp ],
                                            saveAll=False ):
            print( subPrefix * 2 + cmd )

   def showDiff( self, args ):
      # Pending configuration
      pathSpec = self.getPendingPathSpec()
      pendingCliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )
      for dstIp in pathSpec.leaf:
         leafMode = RsvpP2mpLeafMode( ( self, str( dstIp ) ) )
         pendingCliCmds.append( leafMode.enterCmd() )
         for cmd in leafMode.leafToCliCmds( pathSpec.leaf[ dstIp ],
                                                  saveAll=False ):
            pendingCliCmds.append( '   ' + cmd )

      # Current configuration
      pathSpec = self.getActivePathSpec( createIfNotExist=False )
      activeCliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )
      if pathSpec:
         for dstIp in pathSpec.leaf:
            leafMode = RsvpP2mpLeafMode( ( self, str( dstIp ) ) )
            activeCliCmds.append( leafMode.enterCmd() )
            for cmd in leafMode.leafToCliCmds( pathSpec.leaf[ dstIp ],
                                               saveAll=False ):
               activeCliCmds.append( '   ' + cmd )

      # Generate diff
      diff = difflib.unified_diff( activeCliCmds, pendingCliCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

class RsvpPathSpecExplicitConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER Explicit Path Specification"

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecExplicitType,
                                       p2mp=False )

class RsvpP2mpPathSpecExplicitConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER P2MP Explicit Path Specification"

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecExplicitType, p2mp=True )

class RsvpPathSpecDynamicConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER Dynamic Path Specification"

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecDynamicType, p2mp=False )

class RsvpP2mpPathSpecDynamicConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER Dynamic P2MP Path Specification"

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecDynamicType,
                                       p2mp=True )

def pathSpecNameList( mode, p2mp=None ):
   p2mp = p2mp if p2mp is not None else getattr( mode, 'p2mp', None )
   if p2mp is not None:
      sessionType = p2mpLspTunnel if p2mp else p2pLspTunnel
      return [ pathSpec.pathSpecName for pathSpec in lerClient.pathSpec.values()
               if pathSpec.sessionType == sessionType ]
   else:
      return [ pathSpec.pathSpecName for pathSpec in lerClient.pathSpec.values() ]

def p2pPathSpecNameList( mode ):
   return pathSpecNameList( mode, p2mp=False )

def p2mpPathSpecNameList( mode ):
   return pathSpecNameList( mode, p2mp=True )

class RsvpTreeSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'tree TREE_NAME TREESPEC_MODE'
   noOrDefaultSyntax = 'tree TREE_NAME ...'
   data = {
      'tree': 'Enter the configuration mode for a new p2mp tree specification',
      'TREE_NAME': CliMatcher.DynamicNameMatcher( p2mpPathSpecNameList,
                helpdesc='Tree specification name', pattern=r'.+' ),
   }
   if Toggles.RsvpToggleLib.toggleRsvpP2mpLerDynamicTreeSpecEnabled():
      data[ 'TREESPEC_MODE' ] = CliMatcher.EnumMatcher( {
         'explicit': 'Explicit route specification',
         'dynamic': 'Dynamic route specification',
      } )
   else:
      data[ 'TREESPEC_MODE' ] = CliMatcher.EnumMatcher( {
         'explicit': 'Explicit route specification' } )

   @staticmethod
   def handler( mode, args ):
      treeSpecType = args[ 'TREESPEC_MODE' ]
      treeSpecName = args[ 'TREE_NAME' ]
      if treeSpecType == 'explicit':
         childMode = mode.childMode( RsvpP2mpPathSpecExplicitConfigMode,
                                     pathSpecName=treeSpecName )
      elif treeSpecType == 'dynamic':
         childMode = mode.childMode( RsvpP2mpPathSpecDynamicConfigMode,
                                     pathSpecName=treeSpecName )
      else:
         mode.addError( 'Tree specification type not supported' )
         return

      # Check already existing entries with different treeSpecType
      pathSpecId = RsvpLerPathSpecId( treeSpecName, tunnelSourceCli )
      existingPathSpec = lerClient.pathSpec.get( pathSpecId )
      if existingPathSpec and (
            existingPathSpec.pathSpecType != childMode.pathSpecType
            or existingPathSpec.sessionType != p2mpLspTunnel ):
         if existingPathSpec.sessionType == p2mpLspTunnel:
            oldType = 'tree'
         elif existingPathSpec.sessionType == p2pLspTunnel:
            oldType = 'path'
         else:
            oldType = existingPathSpec.sessionType
         mode.addError( f'Changing the type of an existing {oldType} specification '
                        'is not allowed' )
         return

      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      treeSpecName = args[ 'TREE_NAME' ]
      pathSpecId = RsvpLerPathSpecId( treeSpecName, tunnelSourceCli )

      if pathSpecId in lerClient.pathSpec:
         del lerClient.pathSpec[ pathSpecId ]

if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
   RsvpTeConfigMode.addCommandClass( RsvpTreeSpecCmd )

class RsvpPathSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'path PATH_NAME PATH_MODE'
   noOrDefaultSyntax = 'path PATH_NAME ...'
   data = {
      'path': 'Enter the configuration mode for a new path specification',
      'PATH_NAME': CliMatcher.DynamicNameMatcher( p2pPathSpecNameList,
                helpdesc='Path Specification name', pattern=r'.+' ),
      'PATH_MODE': CliMatcher.EnumMatcher( {
         'explicit': 'Explicit route specification',
         'dynamic': 'Dynamic route specification',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      pathSpecType = args[ 'PATH_MODE' ]
      pathSpecName = args[ 'PATH_NAME' ]
      if pathSpecType == 'explicit':
         childMode = mode.childMode( RsvpPathSpecExplicitConfigMode,
                                     pathSpecName=pathSpecName )
      elif pathSpecType == 'dynamic':
         childMode = mode.childMode( RsvpPathSpecDynamicConfigMode,
                                     pathSpecName=pathSpecName )
      else:
         mode.addError( 'Path specification type not supported' )
         return

      # Check already existing entries with different pathSpecType
      pathSpecId = RsvpLerPathSpecId( pathSpecName, tunnelSourceCli )
      existingPathSpec = lerClient.pathSpec.get( pathSpecId )
      if existingPathSpec and (
            existingPathSpec.pathSpecType != childMode.pathSpecType
            or existingPathSpec.sessionType == p2mpLspTunnel ):
         if existingPathSpec.sessionType == p2mpLspTunnel:
            oldType = 'tree'
         elif existingPathSpec.sessionType == p2pLspTunnel:
            oldType = 'path'
         else:
            oldType = existingPathSpec.sessionType
         mode.addError( f'Changing the type of an existing {oldType} specification '
                        'is not allowed' )
         return

      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pathSpecName = args[ 'PATH_NAME' ]
      pathSpecId = RsvpLerPathSpecId( pathSpecName, tunnelSourceCli )

      if pathSpecId in lerClient.pathSpec:
         del lerClient.pathSpec[ pathSpecId ]

RsvpTeConfigMode.addCommandClass( RsvpPathSpecCmd )

matcherHopBefore = CliMatcher.KeywordMatcher( 'before',
   helpdesc='Specify the hop which this hop should be placed directly before' )
matcherHopAfter = CliMatcher.KeywordMatcher( 'after',
    helpdesc='Specify the hop which this hop should be placed directly after' )
matcherHopExclude = CliMatcher.KeywordMatcher( 'exclude',
      helpdesc='Exclude this hop from the dynamic path computation' )
matcherHopLoose = CliMatcher.KeywordMatcher( 'loose',
      helpdesc='Define this hop as a loose hop' )
matcherHop = IpAddrMatcher( helpdesc='IPv4 Address' )
matcherHopKeyword = CliMatcher.KeywordMatcher( 'hop',
                    helpdesc='Hop along the path to include or exclude' )
matcherHopIncludeKeyword = CliMatcher.KeywordMatcher( 'hop',
                    helpdesc='Hop along the path to include' )
matcherHopExcludeKeyword = CliMatcher.KeywordMatcher( 'hop',
                    helpdesc='Hop along the path to exclude' )
matcherHopNode = CliMatcher.KeywordMatcher( 'node',
   helpdesc='The specified hop address can be any IP address on the node '
            'or its TE router ID' )

# --------------------------------------------------------------------------------
# [ no | default ] leaf <IpGenAddr>
# --------------------------------------------------------------------------------
class RsvpP2mpLeafConfigMode( RsvpP2mpLeafMode, BasicCli.ConfigModeBase ):
   name = "RSVP LER P2MP Leaf Specification"

   def __init__( self, parent, session, dstIp, pathSpecType ):
      param = ( parent, dstIp )
      RsvpP2mpLeafMode.__init__( self, param=param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.parentPathSpecConfigMode = parent
      self.parentPathSpecId = self.parentPathSpecConfigMode.pathSpecId
      self.dstIp = IpGenAddr( dstIp )
      self.pathSpecType = pathSpecType
      # Create the pendingConfig using the parent mode's existing pendingConfig
      # for this leaf
      self.pendingConfig = { 'hops': UniqueHopList() }
      parentLeaf = self.parentPathSpecConfigMode.pendingConfig[ 'leaf' ].get(
                                                                         self.dstIp )
      if parentLeaf:
         self.pendingConfig[ 'hops' ] = parentLeaf[ 'hops' ]

   def getActiveLeaf( self ):
      pathSpec = lerClient.pathSpec.get( self.parentPathSpecId,
           Tac.newInstance( 'Rsvp::RsvpLerPathSpec', self.parentPathSpecId,
                            p2mpLspTunnel ) )
      if self.dstIp in pathSpec.leaf:
         return pathSpec.leaf[ self.dstIp ]
      else:
         return Tac.newInstance( 'Rsvp::RsvpLerP2mpLeaf', self.dstIp )

   def buildLeaf( self, configDict, leaf=None ):
      if leaf is None:
         leaf = Tac.newInstance( 'Rsvp::RsvpLerP2mpLeaf', self.dstIp )
      hopList = configDict[ 'hops' ].getHopList()
      hopDict = { i: makePathSpecHop( hop ) for i, hop in enumerate( hopList ) }

      nominalDict = { 'includeHop': hopDict }
      return dictToTaccNominalForCli( nominalDict, leaf )

   def commitContext( self ):
      # If this was an explicit path and there is no hop, just remove that leaf
      # from the parent pathSpec
      if self.pendingConfig is None:
         return
      parentLeafs = self.parentPathSpecConfigMode.pendingConfig[ 'leaf' ]
      explicit = self.pathSpecType == pathSpecExplicitType
      if not self.pendingConfig[ 'hops' ] and explicit:
         if self.dstIp in parentLeafs:
            del parentLeafs[ self.dstIp ]
         return

      if self.dstIp not in parentLeafs:
         parentLeafs[ self.dstIp ] = {}
      parentLeafs[ self.dstIp ].update( self.pendingConfig )
      # TODO Increase the version?

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      print( self.enterCmd() )
      leaf = self.buildLeaf( self.pendingConfig )
      cmds = self.leafToCliCmds( leaf, saveAll=False )
      for cmd in cmds:
         print( '   ' + cmd )

   def showDiff( self, args ):
      leaf = self.buildLeaf( self.pendingConfig )
      pendingCmds = self.leafToCliCmds( leaf, saveAll=False )

      activeLeaf = self.getActiveLeaf()
      activeCmds = self.leafToCliCmds( activeLeaf, saveAll=False )
      # Generate diff
      diff = difflib.unified_diff( activeCmds, pendingCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

class RsvpP2mpLeafExplicitConfigMode( RsvpP2mpLeafConfigMode ):
   name = "RSVP LER P2MP Explicit Leaf Specification"

   def __init__( self, parent, session, dstIp ):
      RsvpP2mpLeafConfigMode.__init__( self, parent, session, dstIp,
                                       pathSpecType=pathSpecExplicitType )

class RsvpP2mpLeafDynamicConfigMode( RsvpP2mpLeafConfigMode ):
   name = "RSVP LER P2MP Dynamic Leaf Specification"

   def __init__( self, parent, session, dstIp ):
      RsvpP2mpLeafConfigMode.__init__( self, parent, session, dstIp,
                                       pathSpecType=pathSpecDynamicType )

#--------------------------------------------------------------------------------
# Common Path (Explicit/Dynamic) commands
#--------------------------------------------------------------------------------

class RsvpP2mpPathSpecLeafCmd( CliCommand.CliCommandClass ):
   syntax = 'leaf IP_ADDR'
   noOrDefaultSyntax = 'leaf [ IP_ADDR ]'
   data = {
      'leaf': 'Path for a sub-LSP',
      'IP_ADDR': IpAddrMatcher( helpdesc='IPv4 Address' ),
   }

   @staticmethod
   def handler( mode, args ):
      leafDstIp = args[ 'IP_ADDR' ]
      if mode.pathSpecType == pathSpecExplicitType:
         childMode = mode.childMode( RsvpP2mpLeafExplicitConfigMode,
                                     dstIp=leafDstIp )
      else:
         childMode = mode.childMode( RsvpP2mpLeafDynamicConfigMode,
                                     dstIp=leafDstIp )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      leafDstIp = args.get( 'IP_ADDR' )
      if leafDstIp is None:
         mode.pendingConfig[ 'leaf' ] = {}
         return
      leafIpGenAddr = IpGenAddr( leafDstIp )
      if leafIpGenAddr in mode.pendingConfig[ 'leaf' ]:
         del mode.pendingConfig[ 'leaf' ][ leafIpGenAddr ]

RsvpP2mpPathSpecExplicitConfigMode.addCommandClass( RsvpP2mpPathSpecLeafCmd )
RsvpP2mpPathSpecDynamicConfigMode.addCommandClass( RsvpP2mpPathSpecLeafCmd )

class RsvpPathSpecAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit path configuration without committing changes'
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

RsvpPathSpecExplicitConfigMode.addCommandClass( RsvpPathSpecAbortCmd )
RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecAbortCmd )
RsvpP2mpPathSpecExplicitConfigMode.addCommandClass( RsvpPathSpecAbortCmd )
RsvpP2mpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecAbortCmd )

class RsvpLeafAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit leaf configuration without committing changes'
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

RsvpP2mpLeafExplicitConfigMode.addCommandClass( RsvpLeafAbortCmd )
RsvpP2mpLeafDynamicConfigMode.addCommandClass( RsvpLeafAbortCmd )

class RsvpPathSpecShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show the pending path specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showPending( args )

RsvpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )
RsvpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )
RsvpP2mpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )
RsvpP2mpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )

class RsvpLeafShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show the pending leaf specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showPending( args )

RsvpP2mpLeafExplicitConfigMode.addCommandClass( RsvpLeafShowPendingCmd )
RsvpP2mpLeafDynamicConfigMode.addCommandClass( RsvpLeafShowPendingCmd )

class RsvpPathSpecShowDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show a diff between active and pending path specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showDiff( args )

RsvpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )
RsvpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )
RsvpP2mpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )
RsvpP2mpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )

class RsvpLeafShowDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show a diff between active and pending leaf specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showDiff( args )

RsvpP2mpLeafExplicitConfigMode.addCommandClass( RsvpLeafShowDiffCmd )
RsvpP2mpLeafDynamicConfigMode.addCommandClass( RsvpLeafShowDiffCmd )

class RsvpPathSpecGenericHopCmd( CliCommand.CliCommandClass ):

   @staticmethod
   def handler( mode, args ):
      pendingConfigHops = mode.pendingConfig[ 'hops' ]
      ip = IpGenAddr( args[ 'HOP_ADDR' ] )
      maxHop = CspfConstraintIdx.max - CspfConstraintIdx.min + 1
      exclude = 'exclude' in args
      if exclude:
         # Marking a hop as excluded
         pendingConfigExcludeHops = mode.pendingConfig[ 'excludeHops' ]
         if len( pendingConfigExcludeHops ) >= maxHop:
            mode.addWarning(
               'Can not configure more than %d exclude hops, ignoring...' % maxHop )
            return
         pendingConfigExcludeHops.add( ip )
      else:
         node = 'node' in args
         loose = 'loose' in args
         hop = Hop( ip, loose=loose, node=node )
         # Marking a hop as included
         if len( pendingConfigHops ) >= maxHop:
            mode.addWarning(
               'Can not configure more than %d hops, ignoring...' % maxHop )
            return
         after = 'after' in args
         before = 'before' in args
         extHopAddr = IpGenAddr( args.get( 'EXT_HOP_ADDR' ) or '' )
         existingHops = pendingConfigHops.getIpSet()
         if extHopAddr and extHopAddr not in existingHops:
            mode.addWarning( '{} has not been configured for this path, '
                             'ignoring...'.format( extHopAddr ) )
            return
         if after:
            pendingConfigHops.insertHopAfterIp( extHopAddr, hop )
         elif before:
            pendingConfigHops.insertHopBeforeIp( extHopAddr, hop )
         else:
            pendingConfigHops.appendHop( hop )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'HOP_ADDR' not in args:
         mode.pendingConfig[ 'hops' ] = UniqueHopList()
         mode.pendingConfig.get( 'excludeHops', {} ).clear()
      else:
         ip = IpGenAddr( args[ 'HOP_ADDR' ] )
         mode.pendingConfig[ 'hops' ].removeIp( ip )
         if 'excludeHops' in mode.pendingConfig and\
               ip in mode.pendingConfig[ 'excludeHops' ]:
            mode.pendingConfig[ 'excludeHops' ].remove( ip )

#--------------------------------------------------------------------------------
# Explicit Path Mode commands
#--------------------------------------------------------------------------------

class RsvpPathSpecExplicitHopCmd( RsvpPathSpecGenericHopCmd ):
   # When the feature toggle is removed, this class can be merged with
   # RsvpP2mpLeafExplicitHopCmd
   if Toggles.RsvpToggleLib.toggleRsvpP2pExplicitNodeHopsEnabled():
      syntax = 'hop HOP_ADDR [ node ] [ ( before | after ) EXT_HOP_ADDR ]'
   else:
      syntax = 'hop HOP_ADDR [ ( before | after ) EXT_HOP_ADDR ]'
   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopIncludeKeyword,
      'HOP_ADDR': matcherHop,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }
   if Toggles.RsvpToggleLib.toggleRsvpP2pExplicitNodeHopsEnabled():
      data[ 'node' ] = matcherHopNode

RsvpPathSpecExplicitConfigMode.addCommandClass( RsvpPathSpecExplicitHopCmd )

class RsvpP2mpLeafExplicitHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = 'hop HOP_ADDR [ node ] [ ( before | after ) EXT_HOP_ADDR ]'
   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopIncludeKeyword,
      'HOP_ADDR': matcherHop,
      'node': matcherHopNode,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }


RsvpP2mpLeafExplicitConfigMode.addCommandClass( RsvpP2mpLeafExplicitHopCmd )

#--------------------------------------------------------------------------------
# Dynamic Path Mode commands
#--------------------------------------------------------------------------------

class RsvpPathSpecDynamicHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = (
      '''hop HOP_ADDR
         [ exclude | ( [ loose ] [ ( before | after ) EXT_HOP_ADDR ] ) ]''' )

   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopKeyword,
      'HOP_ADDR': matcherHop,
      'exclude': matcherHopExclude,
      'loose': matcherHopLoose,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }

RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecDynamicHopCmd )

class RsvpP2mpLeafSpecDynamicHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = (
      '''hop HOP_ADDR [ [ loose ] [ ( before | after ) EXT_HOP_ADDR ] ]''' )

   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopIncludeKeyword,
      'HOP_ADDR': matcherHop,
      'loose': matcherHopLoose,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }

RsvpP2mpLeafDynamicConfigMode.addCommandClass( RsvpP2mpLeafSpecDynamicHopCmd )

class RsvpP2mpPathSpecDynamicExcludeHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = (
      '''hop HOP_ADDR exclude''' )

   noOrDefaultSyntax = 'hop [ HOP_ADDR exclude ]'
   data = {
      'hop': matcherHopExcludeKeyword,
      'HOP_ADDR': matcherHop,
      'exclude': matcherHopExclude,
   }

RsvpP2mpPathSpecDynamicConfigMode.addCommandClass(
      RsvpP2mpPathSpecDynamicExcludeHopCmd )

# Admin group Cli syntax:
# administrative-group include all <list> include any <list> exclude <list>
#
# where 'include all', 'include any', and 'exclude' can appear at most
# only once. Since the 'include' keyword is repeated twice, and it can not
# appear twice in the 'syntax' below, it is referred to as 'include_all'
# and 'include_any'.
class RsvpPathSpecDynamicAdminGrpCmd( CliCommand.CliCommandClass ):
   syntax = ( '''administrative-group
                { ( include_all all INCL_ALL_MIXED_LISTS ) |
                  ( include_any any INCL_ANY_MIXED_LISTS ) |
                  ( exclude EXCL_MIXED_LISTS ) }''' )
   noOrDefaultSyntax = 'administrative-group'
   data = {
      'administrative-group': 'Administrative groups for that path',
      'include_all': Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'all': 'Include all of the following admin groups',
      'INCL_ALL_MIXED_LISTS': matcherAdminGrpLists,
      'include_any': Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'any': 'Include any of the following admin groups',
      'INCL_ANY_MIXED_LISTS': matcherAdminGrpLists,
      'exclude': Node( matcher=matcherAdminGroupExc, maxMatches=1 ),
      'EXCL_MIXED_LISTS': matcherAdminGrpLists,
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'adminGroupConstraints' ] = \
         parseAdminGroupConstraints( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'adminGroupConstraints' ] = \
         RsvpLerAdminGroupConstraintsWithNames()

RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecDynamicAdminGrpCmd )
RsvpP2mpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecDynamicAdminGrpCmd )

class RsvpTunnelSpecBaseConfigMode( RsvpTunnelBaseMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, tunnelSpecId, tacType, p2mp ):

      RsvpTunnelBaseMode.__init__( self, param=tunnelSpecId.tunnelName )

      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.tunnelSpecId = tunnelSpecId
      self.tacType = tacType
      self.pendingConfig = self.buildPendingConfigFromTunnelSpec()
      self.p2mp = p2mp
      self.tunnelSpec = None

   def getActiveTunnelSpec( self, create=False ):
      raise NotImplementedError

   def getPendingTunnelSpec( self ):
      tunnelSpec = self.buildTunnelSpec( pendingConfig=self.pendingConfig )
      return tunnelSpec

   def buildTunnelSpec( self, pendingConfig=None, tunnelSpec=None ):
      '''
      If tunnelSpec is not provided, a new TunnelSpec (orphan) will be created.
      The tunnelSpec is then populated using the arguments passed to this function.
      Being able to create a temporary tunnelSpec is useful for `show pending` and
      `show diff`, when we need to check the difference between two objects,
      including one that is only 'pending' from now, i.e only exists through a
      pending configuration (self.pendingConfig).
      '''

      sessionType = p2mpLspTunnel if self.p2mp else p2pLspTunnel

      if tunnelSpec is None:
         if self.tacType == 'Rsvp::RsvpLerTunnelSpec':
            tunnelSpec = Tac.newInstance( self.tacType, self.tunnelSpecId,
                                          sessionType )
         else:
            print( self.tacType )
            assert False, f"unknown type {self.tacType}"

      changed = dictToTaccForCli( pendingConfig, tunnelSpec )
      if changed:
         tunnelSpec.changeCount = Tac.now()

      return tunnelSpec

   def defaultConfigDict( self ):
      raise NotImplementedError

   def buildPendingConfigFromTunnelSpec( self ):
      pendingConfig = self.defaultConfigDict()
      tunnelSpec = self.getActiveTunnelSpec( create=False )
      if tunnelSpec is None:
         return pendingConfig

      for pendingAttr in pendingConfig:
         tunnelSpecVal = getattr( tunnelSpec, pendingAttr )
         if Tac.isCollection( tunnelSpecVal ):
            pendingConfig[ pendingAttr ].update( tunnelSpecVal )
         else:
            pendingConfig[ pendingAttr ] = tunnelSpecVal

      return pendingConfig

   def commitContext( self ):
      if self.pendingConfig is None:
         # Abort
         return

      # Note: here we are assigning self.tunnelSpec with a tunnel Spec
      # that immediately after is filled with the pending config.
      self.tunnelSpec = self.getActiveTunnelSpec( create=True )
      self.buildTunnelSpec( tunnelSpec=self.tunnelSpec,
                            pendingConfig=self.pendingConfig )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      tunnelSpec = self.getPendingTunnelSpec()
      cliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelSpec, config, False )
      print( self.enterCmd() )
      for cliCmd in cliCmds:
         print( '   ' + cliCmd )

   def showDiff( self, args ):
      tunnelSpec = self.getPendingTunnelSpec()
      pendingCliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelSpec, config, False )
      # Current configuration
      tunnelSpec = self.getActiveTunnelSpec( create=False )
      activeCliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelSpec, config, False )

      # Generate diff
      diff = difflib.unified_diff( activeCliCmds, pendingCliCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

class RsvpTunnelProfileBaseConfigMode( RsvpTunnelBaseMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, tunnelProfileId, tacType, p2mp ):

      RsvpTunnelBaseMode.__init__( self, param=tunnelProfileId.tunnelProfileName )

      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.tunnelProfileId = tunnelProfileId
      self.tacType = tacType
      self.tunnelProfile = None
      self.pendingConfig = self.buildPendingConfigFromTunnelProfile()
      self.p2mp = p2mp

   def getActiveTunnelProfile( self, create=False ):
      raise NotImplementedError

   def getPendingTunnelProfile( self ):
      tunnelProfile = self.buildTunnelProfile( pendingConfig=self.pendingConfig )
      return tunnelProfile

   def buildTunnelProfile( self, pendingConfig=None, tunnelProfile=None ):
      '''
      If tunnelProfile is not provided, a new RsvpLerTunnelProfile (orphan)
      will be created.
      The tunnelProfile is then populated using the arguments passed
      to this function. Being able to create a temporary tunnelProfile is useful
      for `show pending` and `show diff`, when we need to check the difference
      between two objects, including one that is only 'pending' from now,
      i.e only exists through a pending configuration (self.pendingConfig).
      '''
      if tunnelProfile is None:
         if self.tacType == 'Rsvp::RsvpLerTunnelProfile':
            sessionType = p2mpLspTunnel if self.p2mp else p2pLspTunnel
            tunnelProfile = Tac.newInstance( self.tacType, self.tunnelProfileId,
                                             sessionType )
         else:
            assert False, f"unknown type {self.tacType}"

      changed = dictToTaccForCli( pendingConfig, tunnelProfile )
      if changed:
         tunnelProfile.changeCount = Tac.now()

      return tunnelProfile

   def defaultConfigDict( self ):
      raise NotImplementedError

   def buildPendingConfigFromTunnelProfile( self ):
      pendingConfig = self.defaultConfigDict()
      tunnelProfile = self.getActiveTunnelProfile( create=False )
      if tunnelProfile is None:
         return pendingConfig

      for pendingAttr in pendingConfig:
         tunnelProfileVal = getattr( tunnelProfile, pendingAttr )
         if Tac.isCollection( tunnelProfileVal ):
            pendingConfig[ pendingAttr ].update( tunnelProfileVal )
         else:
            pendingConfig[ pendingAttr ] = tunnelProfileVal

      return pendingConfig

   def commitContext( self ):
      if self.pendingConfig is None:
         # Abort
         return

      self.tunnelProfile = self.getActiveTunnelProfile( create=True )

      self.buildTunnelProfile( tunnelProfile=self.tunnelProfile,
                               pendingConfig=self.pendingConfig )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      tunnelProfile = self.getPendingTunnelProfile()
      cliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelProfile, config, False )
      print( self.enterCmd() )
      for cliCmd in cliCmds:
         print( '   ' + cliCmd )

   def showDiff( self, args ):
      tunnelProfile = self.getPendingTunnelProfile()
      pendingCliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelProfile, config,
                                                          False )
      # Current configuration
      tunnelProfile = self.getActiveTunnelProfile( create=False )
      activeCliCmds = self.tunnelSpecOrProfileToCliCmds( tunnelProfile, config,
                                                         False )

      # Generate diff
      diff = difflib.unified_diff( activeCliCmds, pendingCliCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

# --------------------------------------------------------------------------------
# [ no | default ] tunnel <name> [ p2mp ]
# --------------------------------------------------------------------------------
matcherTunnelKeyword = CliMatcher.KeywordMatcher(
   'tunnel', helpdesc='Enter the configuration mode for a new tunnel specification' )

class RsvpTunnelSpecConfigMode( RsvpTunnelSpecBaseConfigMode, RsvpTunnelSpecMode ):
   name = "RSVP LER Tunnel Specification"

   def __init__( self, parent, session, tunnelSpecName ):
      RsvpTunnelSpecMode.__init__( self, tunnelSpecName )
      RsvpTunnelSpecBaseConfigMode.__init__(
         self, parent, session,
         tunnelSpecId=RsvpLerTunnelSpecId( tunnelSpecName, tunnelSourceCli ),
         tacType='Rsvp::RsvpLerTunnelSpec', p2mp=False )

   def getActiveTunnelSpec( self, create=False ):
      if create:
         # Create new tunnel spec, and CLI tunnel spec table if it doesn't exist.
         return lerClient.tunnelSpec.newMember( self.tunnelSpecId, p2pLspTunnel )
      else:
         # Get existing tunnel spec if the CLI client exists.
         return lerClient.tunnelSpec.get( self.tunnelSpecId ) \
            if lerClient else None

   def defaultConfigDict( self ):
      defaultConfig = {
            'enabled': False,
            'dstIp': set(),
            'extraTep': set(),
            'color': None,
            'fastRerouteConfig': None,
            'metric': None,
            'primaryPath': RsvpLerPathSpecId(),
            'secondaryPath': RsvpLerPathSpecId(),
            'secondaryPathPreSignaled': False,
            'bandwidthConfig': None,
            'splitTunnelConfig': None,
            'signalBw': None,
            'tunnelPriority': None,
            'ldpTunnelingConfig': None,
            'eligibleForIgpShortcut': None,
            'optimizationInterval': None,
            'profileId': RsvpLerTunnelProfileId(),
      }
      return defaultConfig

class RsvpP2mpTunnelSpecConfigMode( RsvpTunnelSpecBaseConfigMode,
                                    RsvpP2mpTunnelSpecMode ):
   name = "RSVP LER P2MP Tunnel Specification"

   def __init__( self, parent, session, tunnelSpecName ):
      RsvpP2mpTunnelSpecMode.__init__( self, tunnelSpecName )
      RsvpTunnelSpecBaseConfigMode.__init__(
         self, parent, session,
         tunnelSpecId=RsvpLerTunnelSpecId( tunnelSpecName, tunnelSourceCli ),
         tacType='Rsvp::RsvpLerTunnelSpec', p2mp=True )

   def getActiveTunnelSpec( self, create=False ):
      if create:
         # Create new tunnel spec, and CLI tunnel spec table if it doesn't exist.
         return lerClient.tunnelSpec.newMember( self.tunnelSpecId, p2mpLspTunnel )
      else:
         # Get existing tunnel spec if the CLI client exists.
         return lerClient.tunnelSpec.get( self.tunnelSpecId ) \
                if lerClient else None

   def defaultConfigDict( self ):
      defaultConfig = {
            'enabled': False,
            'dstIp': set(),
            'color': None,
            'metric': None,
            'primaryPath': RsvpLerPathSpecId(),
            'secondaryPath': RsvpLerPathSpecId(),
            'secondaryPathPreSignaled': False,
            'bandwidthConfig': None,
            'splitTunnelConfig': None,
            'signalBw': None,
            'tunnelPriority': None,
            'optimizationInterval': None,
            'profileId': RsvpLerTunnelProfileId(),
      }
      return defaultConfig

def tunSpecNameList( mode ):
   if lerClient is None:
      return []
   return [ tunSpec.tunnelName for tunSpec in lerClient.tunnelSpec.values() ]

matcherP2mpKeyword = CliMatcher.KeywordMatcher(
   'p2mp', helpdesc='Point to Multipoint' )

class RsvpTunnelSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNEL_NAME'
   if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
      syntax += ' [ p2mp ]'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnelKeyword,
      'TUNNEL_NAME': CliMatcher.DynamicNameMatcher( tunSpecNameList,
                     helpdesc='Tunnel name', pattern=r'.+' ),
   }
   if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
      data[ 'p2mp' ] = matcherP2mpKeyword

   @staticmethod
   def handler( mode, args ):
      tunnelSpecName = args[ 'TUNNEL_NAME' ]
      p2mp = 'p2mp' in args
      if p2mp:
         childMode = mode.childMode( RsvpP2mpTunnelSpecConfigMode,
                                     tunnelSpecName=tunnelSpecName )
         sessionType = p2mpLspTunnel
      else:
         childMode = mode.childMode( RsvpTunnelSpecConfigMode,
                                     tunnelSpecName=tunnelSpecName )
         sessionType = p2pLspTunnel

      activeTunnelSpec = childMode.getActiveTunnelSpec()
      if activeTunnelSpec and activeTunnelSpec.sessionType != sessionType:
         mode.addErrorAndStop( 'Changing the type of an existing tunnel '
                               'specification is not allowed' )
         return
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if lerClient is None:
         return
      tunnelSpecName = args[ 'TUNNEL_NAME' ]
      tunnelSpecId = RsvpLerTunnelSpecId( tunnelSpecName, tunnelSourceCli )
      del lerClient.tunnelSpec[ tunnelSpecId ]

RsvpTeConfigMode.addCommandClass( RsvpTunnelSpecCmd )

# --------------------------------------------------------------------------------
# [ no | default ] tunnel profile <name> [ p2mp ]
# --------------------------------------------------------------------------------
class RsvpP2pTunnelProfileConfigMode( RsvpTunnelProfileBaseConfigMode,
                                      RsvpTunnelProfileMode ):
   name = "RSVP LER Tunnel Profile"

   def __init__( self, parent, session, tunnelProfileName ):
      RsvpTunnelProfileMode.__init__( self, tunnelProfileName )
      RsvpTunnelProfileBaseConfigMode.__init__(
         self, parent, session,
         tunnelProfileId=RsvpLerTunnelProfileId( tunnelProfileName,
                                                 tunnelSourceCli ),
         tacType='Rsvp::RsvpLerTunnelProfile',
         p2mp=False )

   def getActiveTunnelProfile( self, create=False ):
      if create:
         # Create new tunnel profile.
         return lerClient.tunnelProfile.newMember( self.tunnelProfileId,
                                                   p2pLspTunnel )
      else:
         # Get existing tunnel profile.
         return lerClient.tunnelProfile.get( self.tunnelProfileId ) \
            if lerClient else None

   def defaultConfigDict( self ):
      defaultConfig = {
         'color': TristateU32.valueInvalid(),
         'fastRerouteConfig': None,
         'metric': RsvpLerMetricConfig(),
         'bandwidthConfig': RsvpLerTunnelBandwidthConfig(),
         'signalBw': RsvpLerConstants.defaultSignalBw,
         'tunnelPriority': RsvpLerTunnelPriority(),
         'ldpTunnelingConfig': RsvpLerLdpTunnelingConfig(),
         'eligibleForIgpShortcut': RsvpLerConstants.defaultEligibleForIgpShortcut,
         'optimizationInterval': None,
         'splitTunnelConfig': RsvpLerSplitTunnelConfig(),
      }
      return defaultConfig

class RsvpP2mpTunnelProfileConfigMode( RsvpTunnelProfileBaseConfigMode,
                                       RsvpP2mpTunnelProfileMode ):
   name = "RSVP LER p2mp Tunnel Profile"

   def __init__( self, parent, session, tunnelProfileName, ):
      RsvpP2mpTunnelProfileMode.__init__( self, tunnelProfileName )
      RsvpTunnelProfileBaseConfigMode.__init__(
         self, parent, session,
         tunnelProfileId=RsvpLerTunnelProfileId( tunnelProfileName,
                                                 tunnelSourceCli ),
         tacType='Rsvp::RsvpLerTunnelProfile',
         p2mp=True )

   def getActiveTunnelProfile( self, create=False ):
      if create:
         # Create new tunnel profile.
         return lerClient.tunnelProfile.newMember( self.tunnelProfileId,
                                                   p2mpLspTunnel )
      else:
         # Get existing tunnel profile.
         return lerClient.tunnelProfile.get( self.tunnelProfileId ) \
            if lerClient else None

   def defaultConfigDict( self ):
      defaultConfig = {
            'primaryPath': RsvpLerPathSpecId(),
      }
      return defaultConfig

# pylint: disable=dangerous-default-value
def tunProfileNameList( mode, sessionTypeFilter={ p2pLspTunnel, p2mpLspTunnel } ):
   return [ profile.profileName for profile in lerClient.tunnelProfile.values()
            if profile.sessionType in sessionTypeFilter ]

matcherProfileKeyword = CliMatcher.KeywordMatcher(
   'profile', helpdesc='Enter the configuration mode for a new tunnel profile' )
matcherProfileName = CliMatcher.DynamicNameMatcher(
   tunProfileNameList, helpdesc='Tunnel profile name', pattern=r'.+' )

class RsvpTunnelProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel profile PROFILE_NAME'
   if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
      syntax += ' [ p2mp ]'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnelKeyword,
      'profile': matcherProfileKeyword,
      'PROFILE_NAME': matcherProfileName,
   }
   if Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
      data[ 'p2mp' ] = matcherP2mpKeyword

   @staticmethod
   def handler( mode, args ):
      tunnelProfileName = args[ 'PROFILE_NAME' ]
      p2mp = 'p2mp' in args
      if p2mp:
         childMode = mode.childMode( RsvpP2mpTunnelProfileConfigMode,
                                     tunnelProfileName=tunnelProfileName )
         sessionType = p2mpLspTunnel
      else:
         childMode = mode.childMode( RsvpP2pTunnelProfileConfigMode,
                                     tunnelProfileName=tunnelProfileName )
         sessionType = p2pLspTunnel

      activeTunnelProfile = childMode.getActiveTunnelProfile()
      if activeTunnelProfile and activeTunnelProfile.sessionType != sessionType:
         mode.addErrorAndStop( 'Changing the type of an existing tunnel profile '
                               'is not allowed' )
         return
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if lerClient is None:
         return
      tunnelProfileName = args[ 'PROFILE_NAME' ]
      profileId = RsvpLerTunnelProfileId( tunnelProfileName, tunnelSourceCli )
      del lerClient.tunnelProfile[ profileId ]

RsvpTeConfigMode.addCommandClass( RsvpTunnelProfileCmd )

# --------------------------------------------------------------------------------
# Tunnel Mode commands
# --------------------------------------------------------------------------------

class RsvpTunnelSpecAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit tunnel configuration without committing changes'
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecAbortCmd )
RsvpP2mpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecAbortCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecAbortCmd )
RsvpP2mpTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecAbortCmd )

class RsvpTunnelSpecShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show the pending tunnel specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showPending( args )

RsvpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowPendingCmd )
RsvpP2mpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowPendingCmd )
RsvpP2pTunnelProfileConfigMode.addShowCommandClass(
   RsvpTunnelSpecShowPendingCmd )
RsvpP2mpTunnelProfileConfigMode.addShowCommandClass( RsvpTunnelSpecShowPendingCmd )

class RsvpTunnelSpecShowDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show a diff between active and pending tunnel specification'
   }

   @staticmethod
   def handler( mode, args ):
      mode.showDiff( args )

RsvpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowDiffCmd )
RsvpP2mpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowDiffCmd )
RsvpP2pTunnelProfileConfigMode.addShowCommandClass( RsvpTunnelSpecShowDiffCmd )
RsvpP2mpTunnelProfileConfigMode.addShowCommandClass( RsvpTunnelSpecShowDiffCmd )

class RsvpTunnelSpecShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable this tunnel',
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'enabled' ] = False

   @staticmethod
   def noHandler( mode, args ):
      mode.pendingConfig[ 'enabled' ] = True

   # default is the tunnel to be shutdown
   defaultHandler = handler

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecShutdownCmd )
RsvpP2mpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecShutdownCmd )

class RsvpTunnelSpecDstIpCmd( CliCommand.CliCommandClass ):
   syntax = 'destination ip IP_ADDR'
   noOrDefaultSyntax = 'destination ip ...'
   data = {
      'destination': 'Tunnel destination',
      'ip': 'IP address of the destination',
      'IP_ADDR': IpAddrMatcher( helpdesc='IPv4 Address' ),
   }

   @staticmethod
   def handler( mode, args ):
      dstIp = IpGenAddr( args[ 'IP_ADDR' ] )
      mode.pendingConfig[ 'dstIp' ] = { dstIp, }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'dstIp' ] = set()

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecDstIpCmd )

class RsvpP2mpTunnelSpecDstIpCmd( CliCommand.CliCommandClass ):
   syntax = 'destination ip IP_ADDR'
   noOrDefaultSyntax = 'destination ip [ IP_ADDR ]'
   data = {
      'destination': 'Tunnel destination',
      'ip': 'IP address of the destination',
      'IP_ADDR': IpAddrMatcher( helpdesc='IPv4 Address' ),
   }

   @staticmethod
   def handler( mode, args ):
      dstIp = IpGenAddr( args[ 'IP_ADDR' ] )
      mode.pendingConfig[ 'dstIp' ].add( dstIp )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dstIp = args.get( 'IP_ADDR' )
      if dstIp is None:
         mode.pendingConfig[ 'dstIp' ] = set()
         return
      dstIpAddr = IpGenAddr( dstIp )
      if dstIpAddr in mode.pendingConfig[ 'dstIp' ]:
         mode.pendingConfig[ 'dstIp' ].remove( dstIpAddr )


RsvpP2mpTunnelSpecConfigMode.addCommandClass( RsvpP2mpTunnelSpecDstIpCmd )

MAX_COLOR = 2 ** 32 - 1
matcherColorValue = CliMatcher.IntegerMatcher( 0, MAX_COLOR, helpdesc="Color value" )

class RsvpTunnelSpecColorCmd( CliCommand.CliCommandClass ):
   syntax = 'color ( COLOR | disabled )'
   noOrDefaultSyntax = 'color ...'
   data = {
      'color': 'Color',
      'COLOR': matcherColorValue,
      'disabled': 'Tunnel color disabled',
   }

   @staticmethod
   def handler( mode, args ):
      color = args.get( 'COLOR', None )
      mode.pendingConfig[ 'color' ] = \
         TristateU32.valueSet( color ) if color is not None else \
         TristateU32.valueInvalid()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'color' ] = mode.defaultConfigDict()[ 'color' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecColorCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecColorCmd )

matcherFastReroute = CliMatcher.KeywordMatcher( 'fast-reroute',
                        helpdesc='Configure fast reroute' )

class FastRerouteModeCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute mode FRR_MODE'
   noOrDefaultSyntax = 'fast-reroute mode [...]'
   data = {
         'fast-reroute': matcherFastReroute,
         'mode': 'Fast reroute mode',
         'FRR_MODE': CliMatcher.EnumMatcher( {
            'link-protection': 'Protect against failure of the next link',
            'none': 'Disable fast reroute',
            'node-protection': 'Protect against failure of the next node',
            } )
         }

   @staticmethod
   def handler( mode, args ):
      frrStrToMode = {
            'node-protection': RsvpFrrMode.frrNodeProtection,
            'link-protection': RsvpFrrMode.frrLinkProtection,
            'none': RsvpFrrMode.frrNone,
            }
      frrMode = args[ 'FRR_MODE' ]
      frrModeValue = frrStrToMode.get( frrMode )
      mode.pendingConfig[ 'fastRerouteConfig' ] = frrModeValue

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      parameter = 'fastRerouteConfig'
      mode.pendingConfig[ parameter ] = mode.defaultConfigDict()[ parameter ]

RsvpTunnelSpecConfigMode.addCommandClass( FastRerouteModeCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( FastRerouteModeCmd )

MAX_METRIC = 2 ** 32 - 1
matcherMetricValue = CliMatcher.IntegerMatcher(
   0, MAX_METRIC, helpdesc='Tunnel static metric value' )

class RsvpTunnelSpecMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'metric ( METRIC | igp )'
   data = {
      'metric': 'Metric for this tunnel',
      'METRIC': matcherMetricValue,
      'igp': 'Use the IGP SPF metric',
   }
   noOrDefaultSyntax = 'metric ...'


   @staticmethod
   def handler( mode, args ):
      useDynamicMetric = 'igp' in args
      staticMetric = args.get( 'METRIC', 0 )
      mode.pendingConfig[ 'metric' ] = RsvpLerMetricConfig(
         staticMetric, useDynamicMetric )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'metric' ] = mode.defaultConfigDict()[ 'metric' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecMetricCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecMetricCmd )

class RsvpTunnelSpecPathCmd( CliCommand.CliCommandClass ):
   syntax = 'path PATH_NAME [ secondary [ pre-signaled ] ]'
   noOrDefaultSyntax = 'path [ PATH_NAME ] [ secondary ] ...'
   data = {
      'path': 'Primary path specification name',
      'PATH_NAME': CliMatcher.DynamicNameMatcher( p2pPathSpecNameList,
                   helpdesc='Path Specification name', pattern=r'.+' ),
      'secondary': 'Secondary path specification name',
      'pre-signaled': 'Make this secondary path pre-signaled',
   }

   @staticmethod
   def handler( mode, args ):
      # Create pathSepcId from passed path name
      path = args[ 'PATH_NAME' ]
      pathSpecId = RsvpLerPathSpecId( path, tunnelSourceCli )
      existingPathSpec = lerClient.pathSpec.get( pathSpecId )
      if existingPathSpec and existingPathSpec.sessionType == p2mpLspTunnel:
         mode.addWarning( 'The specified path is a point-to-multipoint tree '
                          'specification, ignoring path configuration' )
         return
      if 'secondary' in args:
         mode.pendingConfig[ 'secondaryPath' ] = pathSpecId
         mode.pendingConfig[ 'secondaryPathPreSignaled' ] = 'pre-signaled' in args
      else:
         mode.pendingConfig[ 'primaryPath' ] = pathSpecId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      path = args.get( 'PATH_NAME' )
      if path is not None:
         # Create pathSepcId from passed path name and remove primary if
         # if secondary keyword is not passed and path spec matches
         # the existing primary spec. Likewise if secondary keyword is
         # passed remove secondary if path spec matches that of
         # existing secondary.
         pathSpecId = RsvpLerPathSpecId( path, tunnelSourceCli )
         if not 'secondary' in args and pathSpecId == \
               mode.pendingConfig[ 'primaryPath' ]:
            mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()
         if 'secondary' in args and pathSpecId == \
               mode.pendingConfig[ 'secondaryPath' ]:
            mode.pendingConfig[ 'secondaryPath' ] = RsvpLerPathSpecId()
            mode.pendingConfig[ 'secondaryPathPreSignaled' ] = False
      else:
         # No path name specified, remove all primary and secondary paths.
         mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()
         mode.pendingConfig[ 'secondaryPath' ] = RsvpLerPathSpecId()
         mode.pendingConfig[ 'secondaryPathPreSignaled' ] = False

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecPathCmd )

class RsvpP2mpTunnelSpecTreeCmd( CliCommand.CliCommandClass ):
   syntax = 'tree TREE_NAME'
   noOrDefaultSyntax = 'tree [ TREE_NAME ]'
   data = {
      'tree': 'P2MP tree specification name',
      'TREE_NAME': CliMatcher.DynamicNameMatcher( p2mpPathSpecNameList,
                   helpdesc='Tree Specification name', pattern=r'.+' ),
   }

   @staticmethod
   def handler( mode, args ):
      # Create pathSepcId from passed tree name
      tree = args[ 'TREE_NAME' ]
      pathSpecId = RsvpLerPathSpecId( tree, tunnelSourceCli )
      existingPathSpec = lerClient.pathSpec.get( pathSpecId )
      if existingPathSpec and not existingPathSpec.sessionType == p2mpLspTunnel:
         mode.addWarning( 'The specified tree is a point-to-point path specification'
                          ', ignoring tree configuration' )
         return
      mode.pendingConfig[ 'primaryPath' ] = pathSpecId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tree = args.get( 'TREE_NAME' )
      if tree is not None:
         pathSpecId = RsvpLerPathSpecId( tree, tunnelSourceCli )
         if pathSpecId == mode.pendingConfig[ 'primaryPath' ]:
            mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()
      else:
         # No tree name specified
         mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()

RsvpP2mpTunnelSpecConfigMode.addCommandClass( RsvpP2mpTunnelSpecTreeCmd )
RsvpP2mpTunnelProfileConfigMode.addCommandClass( RsvpP2mpTunnelSpecTreeCmd )

MAX_ADJUSTMENT_PERIOD = 2**32 - 1
MIN_SENSITIVITY = 1
MAX_SENSITIVITY = 10
matcherBandwidthKeyword = CliMatcher.KeywordMatcher( 'bandwidth',
                    helpdesc='Bandwidth requirement of the tunnel' )
matcherBandwidthValue = CliMatcher.FloatMatcher(
      0,
      1000,
      helpdesc=\
         'Bandwidth value, which will be subject to IEEE floating point rounding',
      precisionString='%.2f' )
matcherBandwidthUnit = CliMatcher.EnumMatcher( {
                       'bps': 'bandwidth in bits per second',
                       'kbps': 'bandwidth in kilobits per second',
                       'mbps': 'bandwidth in megabits per second',
                       'gbps': 'bandwidth in gigabits per second',
                       } )
class RsvpTunnelSpecBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = '''bandwidth (
                  ( FIXED_BW FIXED_BW_UNIT ) |
                  ( auto min MIN_BW MIN_BW_UNIT max MAX_BW MAX_BW_UNIT
                       [ adjustment-period ADJUSTMENT_PERIOD ]
                       [ sensitivity ( SENSITIVITY | dynamic ) ] ) )'''
   noOrDefaultSyntax = '''bandwidth [
                             ( FIXED_BW FIXED_BW_UNIT ) |
                             ( auto ... ) ]'''
   data = {
      'bandwidth': matcherBandwidthKeyword,
      'FIXED_BW': matcherBandwidthValue,
      'FIXED_BW_UNIT': matcherBandwidthUnit,
      'auto': 'Auto bandwidth',
      'min': 'Minimum bandwidth requirement of the tunnel',
      'MIN_BW': matcherBandwidthValue,
      'MIN_BW_UNIT': matcherBandwidthUnit,
      'max': 'Maximum bandwidth requirement of the tunnel',
      'MAX_BW': matcherBandwidthValue,
      'MAX_BW_UNIT': matcherBandwidthUnit,
      'adjustment-period': 'Minimal time between bw adjustments',
      'ADJUSTMENT_PERIOD': CliMatcher.IntegerMatcher( 1, MAX_ADJUSTMENT_PERIOD,
         helpdesc='Interval in units of seconds' ),
      'sensitivity': "Algorithm's sensitivity to flow rate changes",
      'SENSITIVITY': CliMatcher.IntegerMatcher( MIN_SENSITIVITY, MAX_SENSITIVITY,
         helpdesc='Integer %d (least sensitive) to %d (most sensitive)' %
               ( MIN_SENSITIVITY, MAX_SENSITIVITY ) ),
      'dynamic': 'Set sensitivity dynamically'
   }

   @staticmethod
   def handler( mode, args ):
      # Convert Bandwidth to Bps
      bwMultiplier = { 'bps': 1, 'kbps': 1e3, 'mbps': 1e6, 'gbps': 1e9 }
      if 'auto' in args:
         if FeatureId.MplsTunnel not in fcFeatureConfigDir.feature or \
            fcFeatureConfigDir.feature[ FeatureId.MplsTunnel ] == \
               FeatureState.disabled:
            # Warn the user if mpls tunnel counter feature is not enabled
            mode.addWarning( "Autobandwidth support is not "
              "available when the mpls tunnel counter feature is disabled" )

         # Create autoBwParam
         adjustmentPeriod = args.get( 'ADJUSTMENT_PERIOD', 0 )
         minBw = int( args[ 'MIN_BW' ] * bwMultiplier[ args[ 'MIN_BW_UNIT' ] ] )
         if minBw % 8 > 0:
            mode.addWarning( "Minimal bandwidth will be rounded to whole bytes" )
         minBw = bandwidthBitsToBytes( minBw ) # convert to bytes-per-sec
         maxBw = int( args[ 'MAX_BW' ] * bwMultiplier[ args[ 'MAX_BW_UNIT' ] ] )
         if maxBw % 8 > 0:
            mode.addWarning( "Maximal bandwidth will be rounded to whole bytes" )
         maxBw = bandwidthBitsToBytes( maxBw ) # convert to bytes-per-sec
         if minBw > maxBw:
            mode.addWarning(
            "Minimal bandwidth configured to be greater than maximal bandwidth." +
            " Will be set to {} {}.".format( args[ 'MAX_BW' ],
                  args[ 'MAX_BW_UNIT' ] ) )
         minBw = min( minBw, maxBw )
         autoBwParam = RsvpLerAutoBwParam( adjustmentPeriod, minBw, maxBw )
         if 'dynamic' in args:
            autoBwParam.adjAlgorithmSensitivity = 0
         else:
            autoBwParam.adjAlgorithmSensitivity = \
               args.get( 'SENSITIVITY',
                         RsvpLerAutoBwParam().adjAlgorithmSensitivity )

         mode.pendingConfig[ 'bandwidthConfig' ] = \
            RsvpLerTunnelBandwidthConfig( True, 0, autoBwParam )
      else:
         explicitBw = \
            int( args[ 'FIXED_BW' ] * bwMultiplier[ args[ 'FIXED_BW_UNIT' ] ] )
         if explicitBw % 8 > 0:
            mode.addWarning( "Bandwidth will be rounded to whole bytes" )
         explicitBw = bandwidthBitsToBytes( explicitBw ) # convert to bytes-per-sec
         mode.pendingConfig[ 'bandwidthConfig' ] = \
            RsvpLerTunnelBandwidthConfig( False, explicitBw, RsvpLerAutoBwParam() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'bandwidthConfig' ] = \
         mode.defaultConfigDict()[ 'bandwidthConfig' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecBandwidthCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecBandwidthCmd )

signalingHelpDesc = 'Enable signaling the bandwidth'
signalingKeyword = CliMatcher.KeywordMatcher(
   'signaling', helpdesc=signalingHelpDesc )
signalingKeywordWithTypo = CliCommand.Node(
   CliMatcher.KeywordMatcher(
      'signalling', helpdesc=signalingHelpDesc ),
   hidden=True )
signalingDisabledKeyword = CliMatcher.KeywordMatcher(
   'disabled', helpdesc='Bandwidth signaling disabled' )

class RsvpTunnelBaseBandwidthSignalingCmd( CliCommand.CliCommandClass ):
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'signalBw' ] = mode.defaultConfigDict()[ 'signalBw' ]

class RsvpTunnelSpecBandwidthSignalingCmd( RsvpTunnelBaseBandwidthSignalingCmd ):
   syntax = 'bandwidth ( signaling | signalling ) [ disabled ]'
   noOrDefaultSyntax = 'bandwidth ( signaling | signalling ) ...'
   data = {
      'bandwidth': matcherBandwidthKeyword,
      'signaling': signalingKeyword,
      'signalling': signalingKeywordWithTypo,
      'disabled': signalingDisabledKeyword,
   }

   @staticmethod
   def handler( mode, args ):
      signalBw = 'disabled' not in args
      mode.pendingConfig[ 'signalBw' ] = signalBw
      config.bwSignalingTypoFix = bool(
         'signalling' not in args or config.bwSignalingTypoFix )

class RsvpTunnelProfileBandwidthSignalingCmd( RsvpTunnelBaseBandwidthSignalingCmd ):
   syntax = 'bandwidth signaling [ disabled ]'
   noOrDefaultSyntax = 'bandwidth signaling ...'
   data = {
      'bandwidth': matcherBandwidthKeyword,
      'signaling': signalingKeyword,
      'disabled': signalingDisabledKeyword,
   }

   @staticmethod
   def handler( mode, args ):
      signalBw = 'disabled' not in args
      mode.pendingConfig[ 'signalBw' ] = signalBw

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecBandwidthSignalingCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass(
   RsvpTunnelProfileBandwidthSignalingCmd )

matcherIntervalUnitKeyword = CliMatcher.EnumMatcher( {
   'seconds': 'Reduction delay in seconds',
   'minutes': 'Reduction delay in minutes',
   'hours': 'Reduction delay in hours',
} )

class RsvpTunnelSpecSplitTunnelCmd( CliCommand.CliCommandClass ):
   syntax = '''split-tunnel
               ( ( quantum QUANTUM QUANTUM_UNIT [ sub-tunnels limit LIMIT ] )
               | ( min MIN_BW MIN_BW_UNIT max MAX_BW MAX_BW_UNIT
                    [ sub-tunnels limit LIMIT ]
                    [ reduction-delay DELAY DELAY_UNIT ] )
               | disabled )'''
   noOrDefaultSyntax = 'split-tunnel ...'
   data = {
      'split-tunnel': 'Enable split-tunnel feature for the tunnel',
      'quantum': 'Quantum bandwidth to reserve for each sub-tunnel',
      'QUANTUM': matcherBandwidthValue,
      'QUANTUM_UNIT': matcherBandwidthUnit,
      'min': 'Minimum bandwidth requirement of each sub-tunnel',
      'MIN_BW': matcherBandwidthValue,
      'MIN_BW_UNIT': matcherBandwidthUnit,
      'max': 'Maximum bandwidth requirement of each sub-tunnel',
      'MAX_BW': matcherBandwidthValue,
      'MAX_BW_UNIT': matcherBandwidthUnit,
      'sub-tunnels': 'Sub-tunnels for the tunnel',
      'limit': 'Upper limit on the number of sub-tunnels for the tunnel',
      'LIMIT': CliMatcher.IntegerMatcher(
                  1, RsvpLibConstants.maxSubTunnelLimit, helpdesc='Limit value' ),
      'reduction-delay': 'Time delay before extra sub-tunnels are removed',
      'DELAY': CliMatcher.IntegerMatcher( 1, 9999, helpdesc='Time delay value' ),
      'DELAY_UNIT': matcherIntervalUnitKeyword,
      'disabled': 'Split-tunnel disabled',
   }

   @staticmethod
   def handler( mode, args ):
      if 'disabled' in args:
         mode.pendingConfig[ 'splitTunnelConfig' ] = RsvpLerSplitTunnelConfig()
         return

      if rsvpLerSubTunnel not in tunnelCountersPriorityTable.tunnelTypePriority:
         # Warn the user if sub-tunnel counters are not enabled
         mode.addWarning( "Split-tunnel does not work with autobandwidth when "
            "sub-tunnel counters are not enabled" )
      # Convert Bandwidth to Bps
      bwMultiplier = { 'bps': 1, 'kbps': 1e3, 'mbps': 1e6, 'gbps': 1e9 }
      if 'quantum' in args:
         quantumBw = \
            int( args[ 'QUANTUM' ] * bwMultiplier[ args[ 'QUANTUM_UNIT' ] ] )
         if quantumBw % 8 > 0:
            mode.addWarning( "Quantum bandwidth will be rounded to whole bytes" )
         quantumBw = bandwidthBitsToBytes( quantumBw ) # convert to bytes-per-sec
         # reset minBw and maxBw since quantum is specified.
         minBw = 0
         maxBw = 0
      else:
         # Warn the user if autobandwidth is not configured
         bandwidthConfig = mode.pendingConfig.get( 'bandwidthConfig', None )
         if bandwidthConfig is None:
            # If bandwidth is not configured for the tunnel, check if there is a
            # profile configured that might configure autobandwidth
            if lerClient is not None:
               profileId = mode.pendingConfig.get( 'profileId',
                                                   RsvpLerTunnelProfileId() )
               profile = lerClient.tunnelProfile.get( profileId, None )
            if profile and profile.bandwidthConfig is not None:
               bandwidthConfig = profile.bandwidthConfig
            else:
               bandwidthConfig = RsvpLerTunnelBandwidthConfig()
         if not bandwidthConfig.autoBw:
            mode.addWarning( "Adaptive split-tunnel does not work if autobandwidth "
               "is not enabled" )

         minBw = int( args[ 'MIN_BW' ] * bwMultiplier[ args[ 'MIN_BW_UNIT' ] ] )
         if minBw % 8 > 0:
            mode.addWarning( "Minimal bandwidth will be rounded to whole bytes" )
         minBw = bandwidthBitsToBytes( minBw ) # convert to bytes-per-sec
         maxBw = int( args[ 'MAX_BW' ] * bwMultiplier[ args[ 'MAX_BW_UNIT' ] ] )
         if maxBw % 8 > 0:
            mode.addWarning( "Maximal bandwidth will be rounded to whole bytes" )
         maxBw = bandwidthBitsToBytes( maxBw ) # convert to bytes-per-sec
         if minBw > maxBw:
            mode.addWarning(
            "Minimal bandwidth configured to be greater than maximal bandwidth." +
            " Will be set to {} {}.".format( args[ 'MAX_BW' ],
                  args[ 'MAX_BW_UNIT' ] ) )
         minBw = min( minBw, maxBw )
         # reset quantum since minBw and maxBw are specified
         quantumBw = 0

      subTunnelLimit = args.get( 'LIMIT', RsvpLerConstants.defaultSubTunnelLimit )
      splitBwParam = RsvpLerSplitBwParam( quantumBw, subTunnelLimit, minBw, maxBw )
      reductionDelay = \
         args.get( 'DELAY',
                   RsvpLibConstants.defaultAdaptiveSubTunnelPreservationTime )
      delayUnit = args.get( 'DELAY_UNIT', 'seconds' )
      delayMultiplier = { 'seconds': 1, 'minutes': 60, 'hours': 3600 }
      reductionDelay *= delayMultiplier[ delayUnit ]
      # Reduction delay cannot be less than a sensible default.
      # Currently is 20 * samplingDelay.
      minReductionDelay = int(
         lerConfig.bwSamplingDelay *
         RsvpLerConstants.adaptiveSubTunnelPreservationTimeSamplingFactor )
      if reductionDelay < minReductionDelay:
         mode.addWarning(
            "Reduction delay is less than the minimum required %s seconds"
            % minReductionDelay )
      splitBwParam.adaptiveSubTunnelPreservationTime = reductionDelay
      mode.pendingConfig[ 'splitTunnelConfig' ] = RsvpLerSplitTunnelConfig(
            True, splitBwParam )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'splitTunnelConfig' ] = \
         mode.defaultConfigDict()[ 'splitTunnelConfig' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecSplitTunnelCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecSplitTunnelCmd )

MIN_PRI = 0
MAX_PRI = 7
matcherPriority = CliMatcher.IntegerMatcher( MIN_PRI, MAX_PRI, helpdesc='Priority' )

class RsvpTunnelSpecPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'priority setup SETUP_PRIORITY hold HOLD_PRIORITY'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': 'Tunnel priority',
      'setup': 'Setup priority',
      'SETUP_PRIORITY': matcherPriority,
      'hold': 'Hold priority',
      'HOLD_PRIORITY': matcherPriority,
   }

   @staticmethod
   def handler( mode, args ):
      setupPriority = args[ 'SETUP_PRIORITY' ]
      holdPriority = args[ 'HOLD_PRIORITY' ]
      if setupPriority < holdPriority:
         mode.addWarning( 'Setup priority should not be more preferred (have a '
                          'smaller value) than hold priority' )
      mode.pendingConfig[ 'tunnelPriority' ] = \
            RsvpLerTunnelPriority( setupPriority, holdPriority )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'tunnelPriority' ] = \
         mode.defaultConfigDict()[ 'tunnelPriority' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecPriorityCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecPriorityCmd )

class RsvpTunnelSpecLdpTunnelingCmd( CliCommand.CliCommandClass ):
   syntax = 'tunneling ldp [ ( ucmp | disabled ) ]'
   noOrDefaultSyntax = 'tunneling ...'
   data = {
      'tunneling': 'Tunneling configuration',
      'ldp': 'Allow RSVP tunnel to be used for LDP tunneling',
      'ucmp': 'UCMP load balance LDP tunneling traffic',
      'disabled': 'LDP tunneling disabled',
   }

   @staticmethod
   def handler( mode, args ):
      ucmp = 'ucmp' in args
      eligible = 'disabled' not in args
      mode.pendingConfig[ 'ldpTunnelingConfig' ] = \
         RsvpLerLdpTunnelingConfig( eligible, ucmp )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'ldpTunnelingConfig' ] = \
         mode.defaultConfigDict()[ 'ldpTunnelingConfig' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecLdpTunnelingCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecLdpTunnelingCmd )

class RsvpTunnelSpecIgpShortcutCmd( CliCommand.CliCommandClass ):
   syntax = 'igp shortcut [ disabled ]'
   noOrDefaultSyntax = 'igp shortcut ...'
   data = {
      'igp': 'IGP configuration',
      'shortcut': 'Allow RSVP tunnel to be used for IGP shortcuts',
      'disabled': 'IGP shortcut disabled',
   }

   @staticmethod
   def handler( mode, args ):
      eligible = 'disabled' not in args
      mode.pendingConfig[ 'eligibleForIgpShortcut' ] = eligible

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'eligibleForIgpShortcut' ] = \
         mode.defaultConfigDict()[ 'eligibleForIgpShortcut' ]

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecIgpShortcutCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecIgpShortcutCmd )

class RsvpTunnelSpecOptimizationCmd( CliCommand.CliCommandClass ):
   syntax = 'optimization ( ( interval INTERVAL seconds ) | disabled )'
   noOrDefaultSyntax = 'optimization ...'
   data = {
      'optimization': matcherOptimizationKeyword,
      'interval': matcherIntervalKeyword,
      'INTERVAL': matcherInterval,
      'seconds': matcherSecondsKeyword,
      'disabled': 'Tunnel optimization disabled',
   }

   @staticmethod
   def handler( mode, args ):
      interval = args.get( 'INTERVAL',
                           RsvpLerConstants.optimizationIntervalDisabled )
      mode.pendingConfig[ 'optimizationInterval' ] = interval

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'optimizationInterval' ] = None

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecOptimizationCmd )
RsvpP2pTunnelProfileConfigMode.addCommandClass( RsvpTunnelSpecOptimizationCmd )

class RsvpTunnelSpecAliasEndpointCmd( CliCommand.CliCommandClass ):
   syntax = 'alias endpoint IP_ADDR'
   noOrDefaultSyntax = 'alias endpoint [ IP_ADDR ]'
   data = {
      'alias': 'Additional tunnel endpoint',
      'endpoint': 'IPv4/IPv6 tunnel endpoint',
      'IP_ADDR': IpGenAddrMatcher( helpdesc='IP Address' ),
   }

   @staticmethod
   def handler( mode, args ):
      pendingConfigExtraTeps = mode.pendingConfig[ 'extraTep' ]
      if len( pendingConfigExtraTeps ) >= RsvpLerConstants.maxTunnelAliasTepEntries:
         mode.addWarning(
            'Can not configure more than %d alias endpoints, ignoring...'
            % RsvpLerConstants.maxTunnelAliasTepEntries )
         return

      extraTep = args[ 'IP_ADDR' ]
      pendingConfigExtraTeps.add( extraTep )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pendingConfigExtraTeps = mode.pendingConfig[ 'extraTep' ]
      ip = args.get( 'IP_ADDR' )
      if ip:
         if ip in pendingConfigExtraTeps:
            pendingConfigExtraTeps.remove( ip )
      else:
         pendingConfigExtraTeps.clear()

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecAliasEndpointCmd )

class RsvpTunnelSpecProfileCmdBase( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE_NAME'
   noOrDefaultSyntax = 'profile ...'

   @staticmethod
   def handler( mode, args ):
      profileName = args.get( 'PROFILE_NAME', '' )
      profileId = RsvpLerTunnelProfileId( profileName, tunnelSourceCli ) \
            if profileName else RsvpLerTunnelProfileId()
      mode.pendingConfig[ 'profileId' ] = profileId

   noOrDefaultHandler = handler

def p2pTunProfileNameList( mode ):
   return tunProfileNameList( mode, sessionTypeFilter={ p2pLspTunnel } )

class RsvpP2pTunnelSpecProfileCmd( RsvpTunnelSpecProfileCmdBase ):
   data = {
      'profile': 'Associate a profile with this tunnel',
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher(
         p2pTunProfileNameList, helpdesc='p2p tunnel profile name', pattern=r'.+' )
   }


RsvpTunnelSpecConfigMode.addCommandClass( RsvpP2pTunnelSpecProfileCmd )

def p2mpTunProfileNameList( mode ):
   return tunProfileNameList( mode, sessionTypeFilter={ p2mpLspTunnel } )

class RsvpP2mpTunnelSpecProfileCmd( RsvpTunnelSpecProfileCmdBase ):
   data = {
      'profile': 'Associate a profile with this tunnel',
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher(
         p2mpTunProfileNameList, helpdesc='p2mp tunnel profile name', pattern=r'.+' )
   }


RsvpP2mpTunnelSpecConfigMode.addCommandClass( RsvpP2mpTunnelSpecProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cspf ecmp tie-break ( least-fill | random )
#--------------------------------------------------------------------------------
class EcmpTiebreakCmd( CliCommand.CliCommandClass ):
   syntax = 'cspf ecmp tie-break ( least-fill | random )'
   noOrDefaultSyntax = 'cspf ecmp tie-break ...'
   data = {
      'cspf': 'Cspf path computation',
      'ecmp': 'ECMP config commands',
      'tie-break': 'Tie-break options',
      'least-fill': 'Use bandwidth least-fill algorithm',
      'random': 'Use random selection',
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.rsvpEcmpLeastFill = 'least-fill' in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.rsvpEcmpLeastFill = True

RsvpTeConfigMode.addCommandClass( EcmpTiebreakCmd )

# --------------------------------------------------------------------------------
# [ no | default ] lsp SELF_PING_MODE
# --------------------------------------------------------------------------------
class RsvpLspSelfPingCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp SELF_PING_MODE'
   noOrDefaultSyntax = 'lsp [ SELF_PING_MODE ]'
   data = {
      'lsp': 'LSP self-ping settings',
      'SELF_PING_MODE': CliMatcher.EnumMatcher( {
         'self-ping': 'Enable self-ping for all LSPs',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      lspSelfPingMode = args[ 'SELF_PING_MODE' ]
      if lspSelfPingMode == 'self-ping':
         config.lspSelfPingMode = lspSelfPingEnabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.lspSelfPingMode = lspSelfPingNone

RsvpTeConfigMode.addCommandClass( RsvpLspSelfPingCmd )

#--------------------------------------------------------------------------------
# Plugin
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lerClient
   lerClient = ConfigMount.mount( entityManager, 'mpls/rsvp/lerClient/cli',
                                  'Rsvp::RsvpLerClient', 'w' )

   global config
   config = ConfigMount.mount( entityManager, RsvpLerCliConfig.mountPath,
                               'Rsvp::RsvpLerCliConfig', 'w' )
   global lerConfig
   lerConfig = LazyMount.mount( entityManager, 'mpls/rsvp/lerConfig',
                                'Rsvp::RsvpLerConfig', 'r' )
   global fcFeatureConfigDir
   fcFeatureConfigDir = LazyMount.mount( entityManager,
                                           'flexCounter/featureConfigDir/cliAgent',
                                           'Ale::FlexCounter::FeatureConfigDir',
                                           'r' )
   global tunnelCountersPriorityTable
   tunnelCountersPriorityTable = LazyMount.mount(
      entityManager, 'hardware/counter/tunnel/tunnelTypePriority',
      'Ale::TunnelCountersPriorityTable', 'r' )

   global teConfig
   teConfig = ConfigMount.mount( entityManager, 'te/config',
                                 'TrafficEngineering::Config', 'w' )

   TeCli.TeModeDependents.registerDependentClass( TeSubMode, priority=20 )
