# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from IntfModels import Interface
from CliModel import ( Bool, DeferredModel, Dict, Enum, Float, Model, Submodel,
                       Int, Str, List )
from TypeFuture import TacLazyType

RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )

class RsvpLerError( Model ):
   code = Int( help="RSVP error code" )
   value = Int( help="RSVP error value" )
   node = IpGenericAddress( help="Router on which the error occurred" )

class RsvpLerTunnelPath( Model ):
   specName = Str( help="Path specification name" )
   specType = Enum( help="Path specification type",
                    values=( 'explicit', 'dynamic', 'unknown' ) )
   state = Enum( help="Path state",
                 values=( 'up', 'down', 'establishing', 'coldStandby', 'unknown' ) )
   cspfPathKeyId = Int( help="CSPF Path ID", optional=True )
   mbb = Bool( help="Path is being re-established with make-before-break" )
   inUse = Bool( help="Path is in use" )
   preSignaled = Bool( help="Path is pre-signaled", optional=True )
   hops = List( help="Path hops", valueType=IpGenericAddress )
   pathErrors = List( help="Errors on this path", valueType=RsvpLerError )
   # MTU signaling attributes (only shown in "detail", therefore optional)
   # Only shown when MTU signaling is enabled
   intfMtu = Int( help="Interface MTU", optional=True )
   pathMtu = Int( help="Path MTU", optional=True )

class RsvpLeaf( Model ):
   state = Enum( help="Leaf state", values=( 'up', 'down' ) )
   hops = List( help="Path hops", valueType=IpGenericAddress, optional=True )

class RsvpLerPathWithChange( Model ):
   changeType = Enum( help="The type of change of a path",
                      values=( 'new', 'changed', 'removed', 'notFound' ) )
   hops = List( help="Path hops", valueType=IpGenericAddress, optional=True )

class RsvpLerTunnelTree( Model ):
   specName = Str( help="Tree specification name" )
   specType = Enum( help="Tree specification type",
                    values=( 'explicit', 'dynamic', 'unknown' ) )
   state = Enum( help="Tree state",
                 values=( 'up', 'down', 'establishing', 'coldStandby', 'unknown' ) )
   intfMtus = Dict( help="Interface MTUs of a P2MP tunnel",
                    keyType=str, valueType=int, optional=True )
   pathMtu = Int( help="Path MTU", optional=True )
   cspfTreeId = Int( help="CSPF Tree ID", optional=True )
   mbb = Bool( help="Tree is being re-established with make-before-break" )
   inUse = Bool( help="Tree is in use" )
   treeErrors = List( help="Errors on this tree", valueType=RsvpLerError )
   treeLeaves = Dict( help="Path and state for each leaf of this tree",
                      keyType=IpGenericAddress, valueType=RsvpLeaf )

class RsvpLerAutoBandwidthParams( Model ):
   adjustmentPeriod = Float(
         help="Bandwidth adjustment period for auto bandwidth" )
   minBandwidth = Int(
         help="Minimum bandwidth for auto bandwidth in bps" )
   maxBandwidth = Int(
         help="Maximum bandwidth for auto bandwidth in bps" )
   lastSampledBandwidth = Int(
         help="Last sampled bandwidth for auto bandwidth in bps" )

class RsvpLerSplitTunnelParams( Model ):
   quantum = Int( help="Quantum bandwidth for split-tunnel in bps" )
   subTunnelLimit = Int( help="Limit on the number of sub-tunnels for split-tunnel" )
   minBandwidth = Int( help="Minimum bandwidth for each sub-tunnel in bps" )
   maxBandwidth = Int( help="Maximum bandwidth for each sub-tunnel in bps" )
   reductionDelay = Int( help="Sub-tunnel reduction delay interval value" )

class RsvpLerLsp( Model ):
   lspId = Int( help="LSP ID" )
   cspfPathKeyId = Int( help="CSPF Path ID", optional=True )
   pathType = Enum( help="Path type",
                    values=( 'primary', 'secondary', 'unknown' ) )
   pathSpecName = Str( help="Path specification name" )
   pathSpecType = Enum( help="Path specification type",
                        values=( 'explicit', 'dynamic', 'unknown' ) )
   bandwidth = Int( help="LSP requested bandwidth in bps" )
   state = Enum( help="LSP state",
                 values=( 'up', 'down', 'establishing', 'unverified', 'unknown' ) )
   inUse = Bool( help="LSP is in use" )
   lspErrors = List( help="Errors on this LSP", valueType=RsvpLerError )
   hops = List( help="Path hops", valueType=IpGenericAddress )

class RsvpLerHistoryEntry( Model ):
   timestamp = Float( help='Timestamp of the event' )
   event = Str( help="History event" )
   hops = List( help="P2P path hops", valueType=IpGenericAddress, optional=True )
   tree = Dict( help="Path and change type for each changed leaf of the P2MP tree",
                keyType=IpGenericAddress, valueType=RsvpLerPathWithChange,
                optional=True )

class RsvpLerHistory( Model ):
   history = List( help="Recent history events", valueType=RsvpLerHistoryEntry )

class RsvpLerSubTunnel( Model ):
   state = Enum( help="Sub tunnel state", values=( 'up', 'down', 'unknown' ) )
   mbb = Bool( help="Sub tunnel is getting re-established with make-before-break" )

   currentBandwidth = Int( help="Sub tunnel bandwidth in bps" )
   sessionId = Int( help="Session ID associated with this sub tunnel",
                    optional=True )
   tunnelIndex = Int( help="Index of this sub tunnel in the tunnel FIB",
                      optional=True )
   lspCount = Int( help="Number of LSPs in this sub tunnel" )
   lspIds = List( help="LSP IDs in this sub tunnel", valueType=int,
                  optional=True )

   activePathType = Enum( help="Active path type",
                          values=( 'primary', 'secondary', 'unknown' ) )

   # Path info (not included in "summary")
   primaryPath = Submodel( valueType=RsvpLerTunnelPath,
                           help="Primary path information",
                           optional=True )
   secondaryPaths = List( help="Secondary paths information",
                          valueType=RsvpLerTunnelPath,
                          optional=True )

   # Optional since only used in "lsp" variant
   lsps = List( help="LSPs part of this sub tunnel", valueType=RsvpLerLsp,
                optional=True )

   subTunnelHistory = Submodel( valueType=RsvpLerHistory,
                                help='Sub tunnel history events', optional=True )

   # Optional since only shown in 'detail' and if non-zero
   lastSampledBandwidth = Int(
         help="Last sampled bandwidth of this sub tunnel in bps", optional=True )

class RsvpLerTunnel( Model ):
   source = IpGenericAddress( help="Destination address" )
   destination = IpGenericAddress( help="Destination address" )
   p2mpId = Int( help="P2MP ID in case of a P2MP tunnel", optional=True )
   configSource = Enum( optional=True, help="Tunnel configuration source",
                        values=( 'unknown', 'cli', 'mvpn', 'frrBypass', 'default' ) )
   profile = Str( optional=True, help="Tunnel profile name" )
   # Alias endpoints (only shown in "detail", therefore optional)
   aliasTeps = List( help="Tunnel endpoint aliases",
                     valueType=IpGenericAddress, optional=True )
   color = Int( optional=True, help="Tunnel color" )
   metric = Int( help="Tunnel metric" )
   metricMode = Enum( help="Tunnel metric mode", values=( 'static', 'dynamic' ) )
   state = Enum( help="Tunnel state", values=( 'up', 'down', 'unknown' ) )
   mbb = Bool( help="Tunnel is getting re-established with make-before-break" )

   # Bandwidth-related attributes
   currentBandwidth = Int( help="Currently reserved bandwidth in bps" )
   requestedBandwidth = Int( help="Requested bandwidth in bps" )
   autoBandwidth = Bool( help="Bandwidth is automatically determined" )
   # Optional since it is not shown in summary output
   signalBandwidth = Bool( help="Bandwidth is signaled", optional=True )
   # Optional since only shown in "detail" and if autoBandwidth==True
   autoBandwidthParams = \
         Submodel( valueType=RsvpLerAutoBandwidthParams,
                   help="Parameters for tuning automagic bandwidth determination",
                   optional=True )
   # Optional since only shown in "detail"
   setupPriority = Int( help="Setup priority", optional=True )
   holdPriority = Int( help="Hold priority", optional=True )

   # MTU signaling attributes (only shown in "detail", therefore optional)
   mtuSignalingEnabled = Bool( help="MTU Signaling is enabled", optional=True )

   sessionId = Int( help="Session ID associated with this tunnel", optional=True )

   tunnelIndex = Int( help="Index of this tunnel in the tunnel FIB", optional=True )

   lspCount = Int( help="Number of LSPs in this tunnel" )
   lspIds = List( help="LSP IDs in this tunnel", valueType=int,
                  optional=True )

   # LDP Tunneling and IGP Shortcut eligibility flags. Optional since only in detail.
   ldpTunneling = Bool( help="Tunnel eligible to be used for LDP tunneling",
                        optional=True )
   ldpTunnelingUcmp = Bool( help="UCMP enabled on tunnel for LDP tunneling",
                            optional=True )
   igpShortcut = Bool( help="Tunnel eligible to be used for IGP shortcuts",
                       optional=True )

   activePathType = Enum( help="Active path type",
                          values=( 'primary', 'secondary', 'unknown', 'split' ) )

   # Path info (not included in "summary")
   primaryPath = Submodel( valueType=RsvpLerTunnelPath,
                           help="Primary path information (P2P)",
                           optional=True )
   secondaryPaths = List( help="Secondary paths information (P2P)",
                          valueType=RsvpLerTunnelPath,
                          optional=True )
   primaryTree = Submodel( valueType=RsvpLerTunnelTree,
                           help="Primary tree information (P2MP)",
                           optional=True )

   # Optional since only used in "history" variant
   tunnelHistory = Submodel( valueType=RsvpLerHistory, help='Tunnel history events',
                             optional=True )

   # Optional since only used in "lsp" variant
   lsps = List( help="LSPs part of this tunnel", valueType=RsvpLerLsp,
                optional=True )

   # Tunnel optimization interval (only shown in "detail", therefore optional)
   optimizationInterval = Int( help="Optimization interval value in seconds",
                               optional=True )

   splitTunnel = Bool( help="Tunnel has split-tunnel enabled" )
   splitTunnelParams = Submodel( valueType=RsvpLerSplitTunnelParams,
                                 help="Parameters for split tunnel", optional=True )

   subTunnels = Dict( help="Sub tunnels part of this tunnel keyed by sub tunnel ID",
                      keyType=int, valueType=RsvpLerSubTunnel, optional=True )

class RsvpLerTunnels( DeferredModel ):
   '''Model for "show traffic-engineering rsvp tunnel"'''
   tunnels = Dict( help="Table of tunnel information keyed by tunnel name",
                   keyType=str,
                   valueType=RsvpLerTunnel )

class RsvpLerState( DeferredModel ):
   '''Model for "show traffic-engineering rsvp" '''
   localIntf = Interface( help="Local interface" )
   localIntfConfigured = Bool( help="Is the local interface configured" )
   localIntfIp = IpGenericAddress( help="Local interface IP address" )
   tunnelCount = Int( help="Number of P2P tunnels" )
   tunnelUpCount = Int( help="Number of P2P tunnels that are up" )
   p2mpTunnelCount = Int( help="Number of P2MP tunnels" )
   p2mpTunnelUpCount = Int( help="Number of P2MP tunnels that are up" )
   tunnelSecondaryEnabledCount = Int(
      help="Number of tunnels with secondary path configured" )
   tunnelSecondaryInUseCount = Int(
      help="Number of tunnels with secondary path in use" )
   lspCount = Int( help="Number of P2P LSPs" )
   lspUpCount = Int( help="Number of P2P LSPs that are up" )
   p2mpLspCount = Int( help="Number of P2MP LSPs" )
   p2mpLspUpCount = Int( help="Number of P2MP LSPs that are up" )
   p2mpSubLspCount = Int( help="Number of P2MP sub-LSPs" )
   p2mpSubLspUpCount = Int( help="Number of P2MP sub-LSPs that are up" )
   optimizationInterval = Int( help="Optimization interval value in seconds" )
   autoBwSupported = Bool( help="Is auto bandwidth supported" )
   splitBwTunnelCount = Int(
      help="Number of tunnels with split bandwidth configured", optional=True )
   subTunnelCount = Int( help="Number of sub tunnels", optional=True )
   subTunnelUpCount = Int( help="Number of sub tunnels that are up", optional=True )
   subTunnelSecondaryInUseCount = Int(
      help="Number of sub tunnels with secondary path in use", optional=True )
