# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libRsvp.so*

import sys

import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import (
   AclCli,
   AclCliModel,
   IntfCli,
   IpGenAddrMatcher,
   MplsCli,
   RsvpModel,
   TechSupportCli,
)
import CliToken.Rsvp
from IpLibConsts import DEFAULT_VRF
import LazyMount
import SharkLazyMount
import ShowCommand
import SmashLazyMount
import Tac
from TypeFuture import TacLazyType

# Type Alias
BandwidthClientStatus = Tac.Type( 'TrafficEngineering::BandwidthClientStatus' )
BandwidthClientUsage = Tac.Type( 'TrafficEngineering::BandwidthClientUsage' )
FrrMode = Tac.Type( 'Rsvp::Cli::FrrMode' )

RsvpLspType = TacLazyType( 'Rsvp::RsvpLspType' )
bypassType = RsvpLspType.bypass

RsvpSessionRole = TacLazyType( 'Rsvp::RsvpSessionRole' )
unknown = RsvpSessionRole.unknownSessionRole
ingress = RsvpSessionRole.ingressSessionRole
transit = RsvpSessionRole.transitSessionRole
egress = RsvpSessionRole.egressSessionRole
bud = RsvpSessionRole.p2mpBudSessionRole

# global
shmemEm = None
routingVrfInfoDir = None
routingHardwareRouteStatus = None
mplsRoutingConfig = None
teConfig = None
rsvpCliConfig = None
rsvpConfig = None
rsvpMessageCounters = None
rsvpMessageCountersSnapshot = None
rsvpErrorCounters = None
rsvpErrorCountersSnapshot = None
rsvpSharkStatus = None
rsvpStatus = None
sysname = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
rsvpBwClientUsage = None
rsvpBwClientStatus = None
mplsHwCapability = None
rsvpSessionHistory = None
rsvpSpHistory = None

# Helpers
def rsvpProtocolAgentRunning():
   return AgentDirectory.agentIsRunning( sysname, 'RsvpProtocol' )

def mplsAgentRunning():
   return AgentDirectory.agentIsRunning( sysname, 'Mpls' )

def showCmdWarnings( mode ):
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   if not rsvpCliConfig.enabled:
      mode.addWarning( "RSVP is not enabled" )
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "MPLS routing is not enabled" )
   if not ( routingInfo and routingInfo.routing ):
      mode.addWarning( "IP routing is not enabled" )
   if not teConfig.enabled or teConfig.routerId == teConfig.routerIdDefault:
      mode.addWarning( "Traffic engineering is not enabled" )
   if not rsvpProtocolAgentRunning():
      mode.addWarning( "Agent RSVP is not running" )
   if not mplsAgentRunning():
      mode.addWarning( "Agent MPLS is not running" )

def showHfecDisabledCmdWarning( mode ):
   if ( rsvpCliConfig.frrMode != FrrMode.frrNone and
        not routingHardwareRouteStatus.hierarchicalFecsEnabled ):
      mode.addWarning( "Fast Re-Route support is not available without "
                       "Hierarchical FECs" )


# Tokens
matcherBandwidth = CliMatcher.KeywordMatcher( 'bandwidth',
      helpdesc='Show RSVP bandwidth information' )
matcherNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='Show RSVP neighbors' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Show summarized information' )
matcherHistory = CliMatcher.KeywordMatcher( 'history',
      helpdesc='Show last 10 history events' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='More comprehensive output' )

# Constants
RsvpCliId = Tac.Type( "Rsvp::RsvpCliId" )
MIN_CLI_ID = RsvpCliId.minValue
MAX_CLI_ID = RsvpCliId.maxValue

def sessionNameList( mode ):
   sessionNames = []
   if spStatusColl := rsvpSharkStatus.spStatusColl:
      sessionNames.extend( [ spStatus.sessionName for spStatus
                             in spStatusColl.spStatus.values() ] )
   if p2mpSpStatusColl := rsvpSharkStatus.p2mpSpStatusColl:
      sessionNames.extend( [ spStatus.sessionAttributeObj.value.sessionName
                             for spStatus in p2mpSpStatusColl.spStatus.values()
                             if spStatus.sessionAttributeObj.hasValue ] )
   return sessionNames

#------------------------------------------------------------------------------------
# show mpls rsvp neighbor[<neighborAddress>]
#------------------------------------------------------------------------------------
def showMplsRsvpNeighbor( mode, args ):
   showCmdWarnings( mode )

   neighborAddr = args.get( 'NEIGHBOR_ADDR' )
   summary = 'summary' in args
   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   if neighborAddr is None:
      neighborAddr = Tac.Value( "Arnet::IpGenAddr" )

   LazyMount.force( rsvpStatus )
   rsvpSharkStatusEntity = SharkLazyMount.force( rsvpSharkStatus )

   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpNeighbor",
                               rsvpStatus,
                               rsvpSharkStatusEntity )
   renderer.render( socket, fmt, neighborAddr, summary )

   if summary:
      return RsvpModel.RsvpNeighborSummary
   else:
      return RsvpModel.RsvpNeighbors

class MplsRsvpNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp neighbor [ NEIGHBOR_ADDR ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'neighbor': matcherNeighbor,
      'NEIGHBOR_ADDR': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP (v4 or v6) address of neighbor' ),
   }
   handler = showMplsRsvpNeighbor
   cliModel = RsvpModel.RsvpNeighbors

BasicCli.addShowCommandClass( MplsRsvpNeighborCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp neighbor summary
#------------------------------------------------------------------------------------
class MplsRsvpNeighborSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp neighbor summary'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'neighbor': matcherNeighbor,
      'summary': matcherSummary,
   }
   handler = showMplsRsvpNeighbor
   cliModel = RsvpModel.RsvpNeighborSummary

BasicCli.addShowCommandClass( MplsRsvpNeighborSummaryCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp
#------------------------------------------------------------------------------------
def roleSetToSessionRole( lspRoleSet ):
   # If no role or any unknown, then unknown role.
   if not lspRoleSet or unknown in lspRoleSet:
      return unknown

   # If ingress, ingress if only ingress, otherwise unknown.
   if ingress in lspRoleSet:
      if lspRoleSet == { ingress }:
         return ingress
      else:
         return unknown

   # If only transit, then transit role.
   if lspRoleSet == { transit }:
      return transit

   # If only egress, then egress role.
   if lspRoleSet == { egress }:
      return egress

   # Return bud role for all other combinations.
   return bud

def populateP2pCounts( model ):
   model.sessionCount = 0
   model.ingressCount = 0
   model.transitCount = 0
   model.egressCount = 0
   model.bypassTunnelCount = 0

   model.lspCount = RsvpModel.LspCount()
   model.lspCount.total = 0
   model.lspCount.operational = 0
   model.lspCount.usingBypassTunnels = 0
   model.lspCount.ingress = 0
   model.lspCount.transit = 0
   model.lspCount.egress = 0

   # Only populate P2P counts if shark collections are initialized.
   sessionStateColl = rsvpSharkStatus.sessionStateColl
   spConfigColl = rsvpSharkStatus.spConfigColl
   spStatusColl = rsvpSharkStatus.spStatusColl
   if not ( sessionStateColl and spConfigColl and spStatusColl ):
      return

   for sessionState in sessionStateColl.sessionState.values():
      model.sessionCount += 1
      if sessionState.sessionRole == 'ingressSessionRole':
         model.ingressCount += 1
      elif sessionState.sessionRole == 'transitSessionRole':
         model.transitCount += 1
      elif sessionState.sessionRole == 'egressSessionRole':
         model.egressCount += 1

      for spId in sessionState.spMember:
         spConfig = spConfigColl.spConfig.get( spId )
         spStatus = spStatusColl.spStatus.get( spId )
         # Skip counting the LSP if spConfig or spStatus are missing.
         if not ( spConfig and spStatus ):
            continue

         model.lspCount.total += 1
         if spConfig.dsFrrInUse:
            model.lspCount.usingBypassTunnels += 1
         if spConfig.sessionRole == 'ingressSessionRole':
            model.lspCount.ingress += 1
         elif spConfig.sessionRole == 'transitSessionRole':
            model.lspCount.transit += 1
         elif spConfig.sessionRole == 'egressSessionRole':
            model.lspCount.egress += 1
         if spStatus.operational:
            model.lspCount.operational += 1
         if spStatus.lspType == bypassType:
            model.bypassTunnelCount += 1

def populateP2mpCounts( model ):
   model.p2mpSessionCount = 0
   model.p2mpLspCount = RsvpModel.P2mpLspCount()
   model.p2mpSubLspCount = RsvpModel.P2mpSubLspCount()

   # P2MP sessions only rendered if P2MP Shark collections are initialized.
   p2mpSessionStateColl = rsvpSharkStatus.p2mpSessionStateColl
   p2mpSpGroupStateColl = rsvpSharkStatus.p2mpSpGroupStateColl
   p2mpSpStatusColl = rsvpSharkStatus.p2mpSpStatusColl
   if not ( p2mpSessionStateColl and
            p2mpSpGroupStateColl and
            p2mpSpStatusColl ):
      return

   # Go through session states, counting number of sessions, LSPs and sub-LSPs.
   for sessionState in p2mpSessionStateColl.sessionState.values():
      model.p2mpSessionCount += 1
      for spGroupId in sessionState.spGroupMember:
         spGroupState = \
            p2mpSpGroupStateColl.spGroupState.get( spGroupId )
         if not spGroupState:
            # Missing groupState, skip.
            continue

         # LSP-wide information to keep track of.
         lspOperational = False
         lspRoleSet = set()
         subLspRoleMap = {}
         subLspOperationalSet = set()
         subLspUsingBypassSet = set()

         for spId in spGroupState.spMember:
            spStatus = p2mpSpStatusColl.spStatus.get( spId )
            if not spStatus:
               # Missing spStatus, skip.
               continue

            lspRoleSet.add( spStatus.spSessionRole )
            if spStatus.operational:
               lspOperational = True

            subLspStateColl = spStatus.subLspStateColl
            if not subLspStateColl:
               # subLspStateColl not initialized, skip counting sub-LSPs.
               continue

            for dstId, subLspState in subLspStateColl.subLspState.items():
               # Get role for sub-LSP. If role differs for the same sub-LSP between
               # sub-groups, then the sub-LSP is unknown role.
               if dstId not in subLspRoleMap:
                  subLspRoleMap[ dstId ] = subLspState.sessionRole
               elif subLspRoleMap[ dstId ] != subLspState.sessionRole:
                  subLspRoleMap[ dstId ] = unknown

               if dstId in subLspStateColl.subLspOperational:
                  subLspOperationalSet.add( dstId )

               if subLspState.dsNeighborId in spGroupState.dsFrrInUse:
                  subLspUsingBypassSet.add( dstId )

         lspRole = roleSetToSessionRole( lspRoleSet )

         # LSP is rendered, increment LSP counts.
         model.p2mpLspCount.total += 1
         if lspOperational:
            model.p2mpLspCount.operational += 1
         if subLspUsingBypassSet:
            model.p2mpLspCount.usingBypassTunnels += 1
         if lspRole == ingress:
            model.p2mpLspCount.ingress += 1
         elif lspRole == transit:
            model.p2mpLspCount.transit += 1
         elif lspRole == egress:
            model.p2mpLspCount.egress += 1
         elif lspRole == bud:
            model.p2mpLspCount.bud += 1

         # Increment sub-LSP counts for this LSP.
         model.p2mpSubLspCount.total += len( subLspRoleMap )
         model.p2mpSubLspCount.operational += len( subLspOperationalSet )
         model.p2mpSubLspCount.usingBypassTunnels += len( subLspUsingBypassSet )
         for dstId, subLspRole in subLspRoleMap.items():
            if subLspRole == ingress:
               model.p2mpSubLspCount.ingress += 1
            elif subLspRole == transit:
               model.p2mpSubLspCount.transit += 1
            elif subLspRole == egress:
               model.p2mpSubLspCount.egress += 1

def getNeighborCount():
   # All unique neighbors in all P2P and P2MP sessions
   uniqueNeighborIdSet = set()

   # P2P
   if neighborStateColl := rsvpStatus.neighborStateColl:
      uniqueNeighborIdSet.update( neighborStateColl.neighborState )

   # P2MP
   p2mpSessionStateColl = rsvpSharkStatus.p2mpSessionStateColl
   p2mpSpGroupStateColl = rsvpSharkStatus.p2mpSpGroupStateColl
   p2mpSpStatusColl = rsvpSharkStatus.p2mpSpStatusColl
   if ( rsvpConfig.p2mpEnabled and
        p2mpSessionStateColl and
        p2mpSpGroupStateColl and
        p2mpSpStatusColl ):

      # Go through P2MP session states
      for sessionState in p2mpSessionStateColl.sessionState.values():
         for spGroupId in sessionState.spGroupMember:
            spGroupState = \
               p2mpSpGroupStateColl.spGroupState.get( spGroupId )
            if not spGroupState:
               # Missing groupState, skip.
               continue
            for spId in spGroupState.spMember:
               spStatus = p2mpSpStatusColl.spStatus.get( spId )
               if not spStatus:
                  # Missing spStatus, skip.
                  continue

               usNeighborStateColl = spStatus.usNeighborStateColl
               dsNeighborStateColl = spStatus.dsNeighborStateColl
               if not usNeighborStateColl or not dsNeighborStateColl:
                  # Neighbor state entities not initialized, skip couting neighbors.
                  continue

               # Get unique neighbor ids from the subgroup
               uniqueNeighborIdSet.update(
                  usNeighborStateColl.usNeighborState )
               uniqueNeighborIdSet.update(
                  dsNeighborStateColl.dsNeighborState )

   return len( uniqueNeighborIdSet )

def showMplsRsvp( mode, args ):
   showCmdWarnings( mode )
   showHfecDisabledCmdWarning( mode )

   ReversionMode = Tac.Type( 'Rsvp::Cli::ReversionMode' )
   model = RsvpModel.RsvpState()
   model.adminState = 'enabled' if rsvpCliConfig.enabled else 'disabled'
   if rsvpConfig.isRsvpOperational():
      model.operationalState = "up"
   else:
      model.operationalState = "down"
   model.refreshInterval = rsvpConfig.refreshPeriod
   model.refreshReduction = rsvpConfig.rorEnabled

   model.helloEnabled = rsvpConfig.helloEnabled
   if model.helloEnabled:
      model.helloInterval = rsvpConfig.helloInterval
      model.helloMultiplier = rsvpConfig.helloTimeoutFactor

   model.fastReroute = rsvpConfig.frrEnabled
   if rsvpConfig.frrEnabled:
      if rsvpConfig.frrMode == FrrMode.frrNodeProtection:
         model.fastRerouteMode = 'nodeProtection'
      elif rsvpConfig.frrMode == FrrMode.frrLinkProtection:
         model.fastRerouteMode = 'linkProtection'
   else:
      model.fastRerouteMode = 'none'
   model.hierarchicalFecsEnabled = routingHardwareRouteStatus.hierarchicalFecsEnabled
   if rsvpConfig.frrEnabled:
      if rsvpConfig.frrLocalReversion:
         model.reversionMode = 'local'
      else:
         model.reversionMode = 'global'
      if rsvpCliConfig.reversionMode == ReversionMode.reversionLocal:
         model.reversionModeConfig = 'local'
      else:
         model.reversionModeConfig = 'global'

   model.bypassOptimizationInterval = rsvpConfig.bypassOptimizationInterval

   if rsvpConfig.authType == 'rsvpAuthMd5':
      model.authentication = 'MD5'
   else:
      model.authentication = 'disabled'

   model.srlgMode = rsvpConfig.srlgMode

   model.preemptionPeriod = rsvpConfig.preemptionPeriod
   model.mtuSignalingEnabled = rsvpConfig.mtuSignalingEnabled

   model.labelLocalTerminationMode = rsvpConfig.labelLocalTerminationMode

   if rsvpConfig.hitlessRestartEnabled:
      model.hitlessRestart = RsvpModel.HitlessRestart()
      model.hitlessRestart.recovery = rsvpConfig.hitlessRestartRecoveryTime

   if rsvpConfig.grHelperModeEnabled:
      model.grHelper = RsvpModel.GrHelper()
      model.grHelper.maximumAcceptedRestart = rsvpConfig.grHelperMaximumRestart
      model.grHelper.maximumAcceptedRecovery = rsvpConfig.grHelperMaximumRecovery

   if rsvpConfig.grSpeakerModeEnabled:
      model.grSpeaker = RsvpModel.GrSpeaker()
      model.grSpeaker.restart = rsvpConfig.grSpeakerRestart
      model.grSpeaker.recovery = rsvpConfig.grSpeakerRecovery

   populateP2pCounts( model )

   model.neighborCount = getNeighborCount()
   if neighborIntfMap := rsvpSharkStatus.neighborIntfMap:
      model.interfaceCount = neighborIntfMap.intfCount()
   else:
      model.interfaceCount = 0

   model.p2mpEnabled = rsvpConfig.p2mpEnabled
   if rsvpConfig.p2mpEnabled:
      populateP2mpCounts( model )

   return model

class MplsRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
   }
   handler = showMplsRsvp
   cliModel = RsvpModel.RsvpState

BasicCli.addShowCommandClass( MplsRsvpCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp session [filters]
# Session Filters:
#  - destination <destination>
#  - id <cliId>
#  - role (transit|ingress|egress)
# Sp Filters:
#  - name <sessionName>
#  - lsp <lsp>
#  - state <up, down, path-only>
#  - bypass
#------------------------------------------------------------------------------------
def createTacFilter( filterArg ):
   tacFilter = Tac.Value( "Rsvp::Cli::ShowRsvpSessionFilter" )
   if filterArg:
      # filterArg [('spFilter', [('sessionName', 'foo')]),
      #            ('spOperationalFilter', [('spOperational', 'up')]),
      #            ('sessionFilter', [('cliId', 2)]),
      #            ('sessionRole', 'ingress'),
      #            ('bypass', True) ]
      for filterType, arg in filterArg:
         if filterType == 'sessionFilter':
            # Cli Filter rule names are same as RsvpSessionFilter attributes
            for filterAttr, filterValue in arg:
               if filterAttr in [ 'cliId', 'dstIp' ]:
                  setattr( tacFilter, filterAttr, filterValue )

         elif filterType == 'sessionRole':
            tacFilter.sessionRole = arg

         elif filterType == 'spFilter':
            # Cli Filter rule names are same as RsvpSessionSpFilter attributes
            for filterAttr, filterValue in arg:
               if filterAttr in [ 'sessionName', 'lsp' ]:
                  setattr( tacFilter, filterAttr, filterValue )

         elif filterType == 'spOperationalFilter':
            # Cli Filter rule name is the same as one of the RsvpSessionSpFilter
            # attributes
            for filterAttr, filterValue in arg:
               if filterAttr == 'spOperational':
                  tacFilter.spOperational = filterValue

         elif filterType == 'bypass':
            tacFilter.bypass = True

   return tacFilter

def showMplsRsvpSession( mode, args ):
   showCmdWarnings( mode )

   filterArg = args.get( 'filterArg' )
   summary = 'summary' in args
   history = 'history' in args
   detail = 'detail' in args

   tacFilter = createTacFilter( filterArg )

   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   requestedRevision = mode.session_.requestedModelRevision()

   LazyMount.force( rsvpStatus )
   rsvpSharkStatusEntity = SharkLazyMount.force( rsvpSharkStatus )

   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpSession", rsvpStatus,
                               rsvpSharkStatusEntity, rsvpSessionHistory,
                               rsvpSpHistory )
   renderer.render( socket, fmt, requestedRevision, summary, history, detail,
                    tacFilter )

   return RsvpModel.RsvpSessions

sessionFilterSharedObj = object()
spFilterSharedObj = object()
roleBypassSharedObj = object()

class MplsRsvpSessionFilterArg( CliCommand.CliExpression ):
   expression = ( '{ '
                  '( ( destination DEST_ADDR ) | ( id ID ) ) | '
                  '( ( name NAME ) | ( lsp LSP ) | bypass ) | '
                  '( role ROLE ) | '
                  '( state STATE ) '
                  '}')
   _roleMap = {
      'transit' : 'Show transit LSPs',
      'ingress' : 'Show locally originated LSPs',
      'egress' : 'Show locally terminating LSPs',
      'bud' : 'Show bud LSPs',
   }
   data = {
      'destination': CliCommand.singleKeyword( 'destination',
         helpdesc='Filter by destination address',
         sharedMatchObj=sessionFilterSharedObj ),
      'DEST_ADDR': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP (v4 or v6) address of destination' ),
      'id': CliCommand.singleKeyword( 'id', helpdesc='Filter by Session ID',
         sharedMatchObj=sessionFilterSharedObj ),
      'ID': CliMatcher.IntegerMatcher( MIN_CLI_ID, MAX_CLI_ID,
         helpdesc='Session number' ),
      'name': CliCommand.singleKeyword( 'name', helpdesc='Filter by session name',
         sharedMatchObj=spFilterSharedObj ),
      'NAME': CliMatcher.DynamicNameMatcher( sessionNameList, 'Session name',
         pattern=r'.+' ),
      'lsp': CliCommand.singleKeyword( 'lsp', helpdesc='Filter by LSP#',
         sharedMatchObj=spFilterSharedObj ),
      'LSP': CliMatcher.IntegerMatcher( MIN_CLI_ID, MAX_CLI_ID, helpdesc='LSP#' ),
      'bypass': CliCommand.singleKeyword( 'bypass',
         helpdesc='Filter for bypass LSPs, only applicable on ingress node',
         sharedMatchObj=roleBypassSharedObj ),
      'role' : CliCommand.singleKeyword( 'role',
         helpdesc='Filter by router role', sharedMatchObj=roleBypassSharedObj ),
      'ROLE' : CliMatcher.EnumMatcher( _roleMap ),
      'state': CliCommand.singleKeyword( 'state', helpdesc='Filter by LSP state' ),
      'STATE': CliMatcher.EnumMatcher( {
         'up': 'LSP state up',
         'down': 'LSP state down',
         'path-only': 'LSP state path-only',
      } )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      filterArg = []
      if 'destination' in args:
         filterArg.append( ( 'sessionFilter', [ ( 'dstIp',
                                                args[ 'DEST_ADDR' ][ 0 ] ) ] ) )
      if 'id' in args:
         filterArg.append( ( 'sessionFilter', [ ( 'cliId', args[ 'ID' ][ 0 ] ) ] ) )
      if 'name' in args:
         filterArg.append( ( 'spFilter',  [ ( 'sessionName',
                                            args[ 'NAME' ][ 0 ] ) ] ) )
      if 'lsp' in args:
         filterArg.append( ( 'spFilter',  [ ( 'lsp', args[ 'LSP' ][ 0 ] ) ] ) )
      if 'bypass' in args:
         filterArg.append( ( 'bypass', True ) )
      if 'role' in args:
         filterArg.append( ( 'sessionRole', args[ 'ROLE' ][ 0 ] ) )
      if 'state' in args:
         filterArg.append( ( 'spOperationalFilter', [ ( 'spOperational',
                                                        args[ 'STATE' ][ 0 ] ) ] ) )

      if filterArg:
         args[ 'filterArg' ] = filterArg

class MplsRsvpSessionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp session [ FILTERS ] [ summary | history | detail ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'session': 'Show RSVP session information',
      'FILTERS': MplsRsvpSessionFilterArg,
      'summary': matcherSummary,
      'history' : matcherHistory,
      'detail' : matcherDetail,
   }
   handler = showMplsRsvpSession
   cliModel = RsvpModel.RsvpSessions

BasicCli.addShowCommandClass( MplsRsvpSessionCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp ip access-list [ <acl> ] [ summary ]
# show mpls rsvp ip6 access-list [ <acl> ] [ summary ]
#------------------------------------------------------------------------------------
class ShowMplsRsvpAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mpls rsvp'
              '('
              ' ( ip access-list [ <ipAclName> ] ) | '
              ' ( ipv6 access-list [ <ipv6AclName> ] ) '
              ')' )
   data = {
            'mpls': MplsCli.mplsNodeForShow,
            'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            '<ipAclName>': AclCli.ipAclNameExpression,
            '<ipv6AclName>': AclCli.ip6AclNameExpression
          }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    aclType,
                                    args[ '<aclNameExpr>' ],
                                    serviceName='rsvp' )

BasicCli.addShowCommandClass( ShowMplsRsvpAclCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp counters [ipv4|ipv6][interface <interfaceRange>]
#------------------------------------------------------------------------------------
def messageTypeFromTacc( tac ):
   if tac == 'rsvpPathMsg':
      return "Path"
   if tac == 'rsvpResvMsg':
      return "Resv"
   if tac == 'rsvpPathErrMsg':
      return "PathErr"
   if tac == 'rsvpResvErrMsg':
      return "ResvErr"
   if tac == 'rsvpPathTearMsg':
      return "PathTear"
   if tac == 'rsvpResvTearMsg':
      return "ResvTear"
   if tac == 'rsvpSrefreshMsg':
      return "Srefresh"
   if tac == 'rsvpUnknownMsg':
      return "Errors"
   else:
      return "Other"

def errorTypeFromTacc( tac ):
   def enumValue( valueStr ):
      return Tac.ValueConst( 'Rsvp::Smash::ErrorCounterKey', valueStr )
   return {
      # TACC enum : json value
      enumValue( 'unknownErrorType' ) : 'unknownError',
      enumValue( 'rsvpDisabledErrorType' ) : 'rsvpDisabledError',
      enumValue( 'mplsDisabledErrorType' ) : 'mplsDisabledError',
      enumValue( 'intfErrorType' ) : 'intfError',
      enumValue( 'arpErrorType' ) : 'arpError',
      enumValue( 'invalidPktFormatErrorType' ) : 'invalidPktFormatError',
      enumValue( 'pktParserGenericFailureErrorType' ) :
         'pktParserGenericFailureError',
      enumValue( 'pktParserBadMsgChecksumErrorType' ) :
         'pktParserBadMsgChecksumError',
      enumValue( 'pktParserIntegrityFailureErrorType' ) :
         'pktParserIntegrityFailureError',
      enumValue( 'pktParserSubLspEroDecompressionErrorType' ) :
         'pktParserSubLspEroDecompressionError',
      enumValue( 'pktParserSubLspRroDecompressionErrorType' ) :
         'pktParserSubLspRroDecompressionError',
      enumValue( 'pktGeneratorGenericFailureErrorType' ) :
         'pktGeneratorGenericFailureError',
      enumValue( 'pktGeneratorAuthIndexNotSetErrorType' ) :
         'pktGeneratorAuthIndexNotSetError',
      enumValue( 'pktGeneratorNoSecretForAuthIndexErrorType' ) :
         'pktGeneratorNoSecretForAuthIndexError',
      enumValue( 'pktGeneratorRroDroppedErrorType' ) : 'pktGeneratorRroDroppedError',
      enumValue( 'pktGeneratorPktTooBigForMtuErrorType' ) :
         'pktGeneratorPktTooBigForMtuError',
      enumValue( 'unknownNeighborErrorType' ) : 'unknownNeighborError',
      enumValue( 'invalidSeqNoErrorType' ) : 'invalidSeqNoError',
      enumValue( 'pktParserUnknownObjClassErrorType' ) :
         'pktParserUnknownObjClassError',
      enumValue( 'pktParserUnknownObjCtypeErrorType' ) :
         'pktParserUnknownObjCtypeError',
   }.get( tac, 'other' )

def showMplsRsvpCounters( mode, args ):
   showCmdWarnings( mode )

   model = RsvpModel.RsvpMessageCounters()

   if intfs := args.get( 'INTFS' ):
      intfSet = set( intfs )
      def accept( counterKey ):
         return counterKey.intfId in intfSet
   else:
      def accept( _ ):
         return True

   afs = args.get( 'AF', ( 'ipv4', 'ipv6' ) )
   if 'ipv4' in afs:
      for counterKey in rsvpMessageCounters.rxIpv4Count:
         if accept( counterKey ):
            msgCount = rsvpMessageCounters.rxIpv4RelativeCount(
                       counterKey, rsvpMessageCountersSnapshot )
            if msgCount > 0:
               intfModel = model.intfModel( counterKey.intfId )
               mType = messageTypeFromTacc( counterKey.mType )
               intfModel.rx.countInc( mType, msgCount )
      for counterKey in rsvpMessageCounters.txIpv4Count:
         if accept( counterKey ):
            msgCount = rsvpMessageCounters.txIpv4RelativeCount(
                       counterKey, rsvpMessageCountersSnapshot )
            if msgCount > 0:
               intfModel = model.intfModel( counterKey.intfId )
               mType = messageTypeFromTacc( counterKey.mType )
               intfModel.tx.countInc( mType, msgCount )

   if 'ipv6' in afs:
      for counterKey in rsvpMessageCounters.rxIpv6Count:
         if accept( counterKey ):
            msgCount = rsvpMessageCounters.rxIpv6RelativeCount(
                       counterKey, rsvpMessageCountersSnapshot )
            if msgCount > 0:
               intfModel = model.intfModel( counterKey.intfId )
               mType = messageTypeFromTacc( counterKey.mType )
               intfModel.rx.countInc( mType, msgCount )
      for counterKey in rsvpMessageCounters.txIpv6Count:
         if accept( counterKey ):
            msgCount = rsvpMessageCounters.txIpv6RelativeCount(
                       counterKey, rsvpMessageCountersSnapshot )
            if msgCount > 0:
               intfModel = model.intfModel( counterKey.intfId )
               mType = messageTypeFromTacc( counterKey.mType )
               intfModel.tx.countInc( mType, msgCount )

   # Error counters
   if not intfs:
      for errorType in rsvpErrorCounters.rxErrorCount:
         errorCount = rsvpErrorCounters.rxErrorRelativeCount(
                         errorType, rsvpErrorCountersSnapshot )
         if errorCount > 0:
            jsonValue = errorTypeFromTacc( errorType )
            errorCounterModel = model.errorCountsModel( jsonValue )
            errorCounterModel.rx = errorCount
      for errorType in rsvpErrorCounters.txErrorCount:
         errorCount = rsvpErrorCounters.txErrorRelativeCount(
                         errorType, rsvpErrorCountersSnapshot )
         if errorCount > 0:
            jsonValue = errorTypeFromTacc( errorType )
            errorCounterModel = model.errorCountsModel( jsonValue )
            errorCounterModel.tx = errorCount

   return model

class MplsRsvpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp counters [ AF ] [ interface INTFS ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'counters': 'Show RSVP message counters',
      'AF': CliMatcher.EnumMatcher( {
         'ipv4': 'Filter by IPv4',
         'ipv6': 'Filter by IPv6',
      } ),
      'interface': 'Filter by interface',
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpCounters
   cliModel = RsvpModel.RsvpMessageCounters

BasicCli.addShowCommandClass( MplsRsvpCountersCmd )

# -----------------------------------------------------------------------------------
# show mpls rsvp bandwidth summary [interface <interfaceRange>]
# -----------------------------------------------------------------------------------
def showMplsRsvpBandwidthSummary( mode, args ):
   showCmdWarnings( mode )

   if intfs := args.get( 'INTFS' ):
      intfs = set( intfs )
   else:
      intfs = set( rsvpBwClientUsage.reservedBandwidth )
      intfs.update( rsvpBwClientStatus.maxReservableBandwidth )

   model = RsvpModel.RsvpBandwidthSummaryModel()
   model.interfaces = {}

   for intf in intfs:
      usage = rsvpBwClientUsage.reservedBandwidth.get( intf )
      maxReservableBandwidth = \
            rsvpBwClientStatus.maxReservableBandwidth.get( intf )

      if usage:
         bw = sum( usage.bandwidth.values(), 0.0 )
      elif maxReservableBandwidth:
         bw = 0.0
      else:
         continue

      intfModel = RsvpModel.RsvpBandwidthIntfSummaryModel()
      # convert bytes-per-sec to bits-per-sec
      intfModel.reservedBandwidth = bw * 8
      if maxReservableBandwidth:
         intfModel.totalBandwidth = maxReservableBandwidth.bandwidth * 8
      else:
         intfModel.totalBandwidth = 0.0

      model.interfaces[ intf ] = intfModel
   return model

class MplsRsvpBandwidthSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp bandwidth summary [ INTFS ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'bandwidth': matcherBandwidth,
      'summary': matcherSummary,
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpBandwidthSummary
   cliModel = RsvpModel.RsvpBandwidthSummaryModel

BasicCli.addShowCommandClass( MplsRsvpBandwidthSummaryCmd )

# -----------------------------------------------------------------------------------
# show mpls rsvp bandwidth [interface <interfaceRange>]
# -----------------------------------------------------------------------------------
def showMplsRsvpBandwidth( mode, args ):
   showCmdWarnings( mode )

   if intfs := args.get( 'INTFS' ):
      intfs = set( intfs )
   else:
      intfs = set( rsvpBwClientUsage.reservedBandwidth )
      intfs.update( rsvpBwClientStatus.maxReservableBandwidth )

   model = RsvpModel.RsvpBandwidthModel()
   model.interfaces = {}

   for intf in intfs:
      usage = rsvpBwClientUsage.reservedBandwidth.get( intf )
      maxReservableBandwidth = \
            rsvpBwClientStatus.maxReservableBandwidth.get( intf )
      if not usage and not maxReservableBandwidth:
         continue

      intfModel = RsvpModel.RsvpBandwidthIntfModel()
      intfModel.reservedBandwidth = {}
      bandwidth = usage.bandwidth if usage else {}
      # convert bytes-per-sec to bits-per-sec
      for priority in range( 0, 8 ):
         intfModel.reservedBandwidth[ priority ] = \
               float( bandwidth.get( priority, 0 ) * 8 )
      if maxReservableBandwidth:
         intfModel.totalBandwidth = maxReservableBandwidth.bandwidth * 8
      else:
         intfModel.totalBandwidth = 0.0

      model.interfaces[ intf ] = intfModel
   return model

class MplsRsvpBandwidthCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp bandwidth [ INTFS ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'bandwidth': matcherBandwidth,
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpBandwidth
   cliModel = RsvpModel.RsvpBandwidthModel

BasicCli.addShowCommandClass( MplsRsvpBandwidthCmd )

#---
# Support for show tech-support
def _showTechGuard():
   return mplsHwCapability.mplsSupported

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd(
   '2018-07-24 16:30:00',
   cmds=[ 'show mpls rsvp',
           'show mpls rsvp ip access-list',
           'show mpls rsvp ipv6 access-list',
           'show mpls rsvp counter',
           'show mpls rsvp neighbor',
           'show mpls rsvp session detail',
           'show mpls rsvp bandwidth',
           'show mpls rsvp session history' ],
   cmdsGuard=_showTechGuard,
   summaryCmds=[ 'show mpls rsvp neighbor', 'show mpls rsvp session summary' ],
   summaryCmdsGuard=_showTechGuard )

def Plugin( entityManager ):
   global routingVrfInfoDir
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                        "routing/vrf/routingInfo/status",
                                        "Tac::Dir", "ri" )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                                 "routing/hardware/route/status",
                                                 "Routing::Hardware::RouteStatus",
                                                 'r' )

   global mplsRoutingConfig
   mplsRoutingConfig = LazyMount.mount( entityManager,
                                        "routing/mpls/config",
                                        "Mpls::Config", 'r' )
   global teConfig
   teConfig = LazyMount.mount( entityManager,
                               "te/config",
                               "TrafficEngineering::Config", 'r' )
   global rsvpCliConfig
   RsvpCliConfig = Tac.Type( "Rsvp::RsvpCliConfig" )
   rsvpCliConfig = LazyMount.mount( entityManager,
                                    RsvpCliConfig.mountPath,
                                    "Rsvp::RsvpCliConfig", 'r' )
   global rsvpConfig
   RsvpConfig = Tac.Type( "Rsvp::RsvpConfig" )
   rsvpConfig = LazyMount.mount( entityManager, RsvpConfig.mountPath,
                                 "Rsvp::RsvpConfig", 'r' )

   global rsvpSharkStatus
   autoUnmount = True
   RsvpSharkStatus = Tac.Type( "Rsvp::RsvpSharkStatus" )
   rsvpSharkStatus = SharkLazyMount.mount(
         entityManager,
      RsvpSharkStatus.mountPath,
      "Rsvp::RsvpSharkStatus",
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount )

   global rsvpStatus
   RsvpSysdbStatus = Tac.Type( "Rsvp::RsvpSysdbStatus" )
   rsvpStatus = LazyMount.mount( entityManager,
                                 RsvpSysdbStatus.mountPath,
                                 "Rsvp::RsvpSysdbStatus", 'rS' )

   global rsvpMessageCounters
   RsvpMessageCounterColl = Tac.Type( "Rsvp::Smash::MessageCounterColl" )
   rsvpMessageCounters = SmashLazyMount.mount(
                           entityManager,
                           RsvpMessageCounterColl.mountPath,
                           "Rsvp::Smash::MessageCounterColl",
                           SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpMessageCountersSnapshot
   RsvpMessageCounterColl = Tac.Type( "Rsvp::Smash::MessageCounterColl" )
   rsvpMessageCountersSnapshot = SmashLazyMount.mount(
                           entityManager,
                           RsvpMessageCounterColl.snapshotMountPath,
                           "Rsvp::Smash::MessageCounterColl",
                           RsvpMessageCounterColl.writerMountInfo() )

   global rsvpErrorCounters
   RsvpErrorCounterColl = Tac.Type( "Rsvp::Smash::ErrorCounterColl" )
   rsvpErrorCounters = SmashLazyMount.mount(
                           entityManager,
                           RsvpErrorCounterColl.mountPath,
                           "Rsvp::Smash::ErrorCounterColl",
                           SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpErrorCountersSnapshot
   RsvpErrorCounterColl = Tac.Type( "Rsvp::Smash::ErrorCounterColl" )
   rsvpErrorCountersSnapshot = SmashLazyMount.mount(
                           entityManager,
                           RsvpErrorCounterColl.snapshotMountPath,
                           "Rsvp::Smash::ErrorCounterColl",
                           RsvpErrorCounterColl.writerMountInfo() )

   global sysname
   sysname = entityManager.sysname()

   global aclCpConfig
   aclCpConfig = LazyMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "r" )

   global aclStatus
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )

   global aclCheckpoint
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )

   BW_CLIENT_NAME = 'rsvpFrr'
   global rsvpBwClientUsage
   rsvpBwClientUsage = SmashLazyMount.mount(
         entityManager,
         BandwidthClientUsage.mountPath( DEFAULT_VRF, BW_CLIENT_NAME ),
         "TrafficEngineering::BandwidthClientUsage",
         SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpBwClientStatus
   rsvpBwClientStatus = SmashLazyMount.mount(
         entityManager,
         BandwidthClientStatus.mountPath( DEFAULT_VRF, BW_CLIENT_NAME ),
         "TrafficEngineering::BandwidthClientStatus",
         SmashLazyMount.mountInfo( 'reader' ) )
   global mplsHwCapability
   mplsHwCapability = LazyMount.mount( entityManager,
                                 "routing/hardware/mpls/capability",
                                 "Mpls::Hardware::Capability", 'r' )

   global rsvpSessionHistory
   RsvpSessionHistory = Tac.Type( "Rsvp::RsvpSessionHistory" )
   rsvpSessionHistory = SmashLazyMount.mount( entityManager,
                                              RsvpSessionHistory.mountPath,
                                              "Rsvp::RsvpSessionHistory",
                                              SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpSpHistory
   RsvpSpHistory = Tac.Type( "Rsvp::RsvpSpHistory" )
   rsvpSpHistory = SmashLazyMount.mount( entityManager, RsvpSpHistory.mountPath,
                                         "Rsvp::RsvpSpHistory",
                                         SmashLazyMount.mountInfo( 'reader' ) )
