#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   BgpCmdBaseClass,
   PeerCliExpression,
   RouterBgpBaseMode,
   RoutingBgpBaseAfMode,
   RouterBgpAfUnsupportedPrefixListSharedModelet,
)
from CliPlugin.RoutingBgpNeighborCli import (
   RouterBgpAfActivateCommand,
   SetNeighborGracefulRestartCmd,
   SetNeighborRcfCmd,
)
from CliPlugin.RoutingBgpInstanceCli import (
   SetGracefulRestartCmd,
)
import CliToken.RoutingBgp as bgpTokens
from BgpLib import (
   updateConfigAttrsAfMap,
)
from IpLibConsts import DEFAULT_VRF
from CliCommand import (
   CliExpression,
   Node )
import CliMatcher
from RtMembershipLib import (
   nlriSupportCliToEnum,
)

#
# Rt Membership configuration
#

nlriMatcher = Node( CliMatcher.KeywordMatcher( 'nlri', helpdesc='NLRI handling' ) )

class NlriSupportExpr( CliExpression ):
   expression = '( 0,96 | 0,32-96 | 32-96 )'
   data = {
      '0,96': 'Only advertise NLRI 0 or 96 bits long',
      '0,32-96': 'Advertise NLRI 0 or 32 to 96 bits long',
      '32-96': 'Only advertise NLRI 32 to 96 bits long' }

   @staticmethod
   def adapter( mode, args, argsList ):
      matchingKeywords = args.keys() & nlriSupportCliToEnum.keys()
      if matchingKeywords:
         [ matchingKeyword ] = matchingKeywords
         nlriSupport = nlriSupportCliToEnum[ matchingKeyword ]
      else:
         nlriSupport = None
      args[ 'NLRI_SUPPORT' ] = nlriSupport

updateConfigAttrsAfMap(
   { 'rt-membership': [ 'RtMembership', 'AfRtMembership' ] } )

class RouterBgpBaseAfRtMembershipMode( RoutingBgpBaseAfMode,
                                       BasicCli.ConfigModeBase ):
   name = 'Route Target VPN route membership address family configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

afModeExtensionHook.addAfModeExtension( 'rt-membership',
                                        RouterBgpBaseAfRtMembershipMode )
RouterBgpBaseAfRtMembershipMode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )

# ---------------------------------------------------------------------------------
# '[ no | default ] address-family rt-membership' config mode
# ---------------------------------------------------------------------------------
class RtMembershipAfModeCmd( BgpCmdBaseClass ):
   syntax = 'address-family rt-membership'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'rt-membership': 'Route Target VPN route membership address family'
   }
   handler = "RtMembershipConfigCliHandler.RtMembershipAfModeCmd_handler"
   noOrDefaultHandler = \
      "RtMembershipConfigCliHandler.RtMembershipAfModeCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RtMembershipAfModeCmd )

for cmd in [ RouterBgpAfActivateCommand,
             SetGracefulRestartCmd,
             SetNeighborGracefulRestartCmd,
             SetNeighborRcfCmd,
]:
   RouterBgpBaseAfRtMembershipMode.addCommandClass( cmd )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor PEER default-route-target [ only | disabled ]
# ---------------------------------------------------------------------------------
class RtMembershipNeighborDefaultRt( BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-route-target [only | disabled]'
   # Do not accept trailing garbage, as this command is a prefix to
   # the "encoding" command below. If we accept trailing garbage, then
   # a no/default command with a partial "encoding" keyword will
   # match.
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'PEER': PeerCliExpression,
        'default-route-target': bgpTokens.defaultRouteTarget,
        'only': 'Only advertise default route target prefix' } )
   handler = "RtMembershipConfigCliHandler.RtMembershipNeighborDefaultRt_handler"
   noOrDefaultHandler = \
      "RtMembershipConfigCliHandler.RtMembershipNeighborDefaultRt_noOrDefaultHandler"

RouterBgpBaseAfRtMembershipMode.addCommandClass(
      RtMembershipNeighborDefaultRt )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor PEER default-route-target encoding origin-as omit
# ---------------------------------------------------------------------------------
class RtMembershipNeighborDefaultRtEncoding( BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-route-target encoding origin-as omit [disabled]'
   noOrDefaultSyntax = 'neighbor PEER default-route-target encoding ...'
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'PEER': PeerCliExpression,
        'default-route-target': bgpTokens.defaultRouteTarget,
        'encoding': 'advertised default route target NLRI encoding',
        'origin-as': 'advertised NLRI origin-AS encoding',
        'omit': 'advertise NLRI with origin-AS unspecified for compatibility' } )
   handler = \
      "RtMembershipConfigCliHandler.RtMembershipNeighborDefaultRtEncoding_handler"
   noOrDefaultHandler = \
      "RtMembershipConfigCliHandler." + \
      "RtMembershipNeighborDefaultRtEncoding_noOrDefaultHandler"

RouterBgpBaseAfRtMembershipMode.addCommandClass(
   RtMembershipNeighborDefaultRtEncoding )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor default nlri support length
#                                     ( disabled | 0,96 | 0,32-96 | 32-96 )
# ---------------------------------------------------------------------------------
class RtMembershipNlriSupport( BgpCmdBaseClass ):
   syntax = 'neighbor default nlri support length ' + \
         '( disabled | 0,96 | 0,32-96 | 32-96 )'
   noOrDefaultSyntax = 'neighbor default nlri support length ...'
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'default': bgpTokens.neighborDefaultGlobal,
        'nlri': nlriMatcher,
        'support': 'NLRI supported by peer',
        'length': 'NLRI lengths supported by peer',
        'disabled': CliMatcher.KeywordMatcher(
                 'disabled',
                 helpdesc='Only advertise NLRI 32 to 96 bits long, '
                          'add origin AS to received 0 bits long NLRI' ),
        'NLRI_SUPPORT': NlriSupportExpr,
       } )
   handler = \
      "RtMembershipConfigCliHandler.RtMembershipNlriSupport_handler"
   noOrDefaultHandler = \
      "RtMembershipConfigCliHandler." + \
      "RtMembershipNlriSupport_noOrDefaultHandler"

RouterBgpBaseAfRtMembershipMode.addCommandClass( RtMembershipNlriSupport )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor PEER nlri support length
#                                     ( disabled | 0,96 | 0,32-96 | 32-96 )
# ---------------------------------------------------------------------------------
class RtMembershipNeighborNlriSupport( BgpCmdBaseClass ):
   syntax = 'neighbor PEER nlri support length ' + \
         '( disabled | 0,96 | 0,32-96 | 32-96 )'
   noOrDefaultSyntax = 'neighbor PEER nlri support length ...'
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'PEER': PeerCliExpression,
        'nlri': nlriMatcher,
        'support': 'NLRI supported by peer',
        'length': 'NLRI lengths supported by peer',
        'disabled': CliMatcher.KeywordMatcher(
                 'disabled',
                 helpdesc='Only advertise NLRI 32 to 96 bits long, '
                          'add origin AS to received 0 bits long NLRI' ),
        'NLRI_SUPPORT': NlriSupportExpr,
       } )
   handler = \
      "RtMembershipConfigCliHandler.RtMembershipNeighborNlriSupport_handler"
   noOrDefaultHandler = \
      "RtMembershipConfigCliHandler." + \
      "RtMembershipNeighborNlriSupport_noOrDefaultHandler"

RouterBgpBaseAfRtMembershipMode.addCommandClass( RtMembershipNeighborNlriSupport )
