# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from CliModel import Model, Enum, Int, Bool, Str
from ArnetModel import Ip4Address, IpGenericAddress, IpGenericPrefix
from IntfModels import Interface
import Tac

# NOTE:
# 'show isis segment-routing ...' & 'show ip ospf segement-routing ...'
# are very similar command hierarchies for Isis and Ospf respectively.
# The data for these commands is mostly fetched from srSmash and srSysdb
# objects.
# This file defines the common models, and the common functions to
# populate the models that can be used by both IGPs.

_srDataPlaneEnum = Tac.Type( "Routing::SegmentRoutingCli::SrDataPlane" )

#------------------------------------------------------------------------------------
# Common models
#------------------------------------------------------------------------------------
reachabilityAlgorithmEnumValues = ( "SPF", "SSPF" )

# Common model, which almost all commands in this hierarchy use
# Isis additionally has systemId
class SrCommonHeader( Model ):
   """Base class for all SR models
   """
   dataPlane = Enum( values=[ "None", "MPLS", "IPV6", "BOTH" ], help="Data-plane" )
   routerId = Ip4Address( help="Router ID", optional=True )

   def render( self ):
      if self.dataPlane == "MPLS":
         print( "SR supported Data-plane: %s\t\t\tSR Router ID: %s" %
                ( self.dataPlane, self.routerId ) )

# The self originated segment statistics displayed under
# - 'show isis segment-routing'
# - 'show ip ospf segement-routing'
class SelfOriginatedSegmentStatisticsModel( Model ):
   """Model to store segment statistics
   """
   nodeSidCount = Int( help="Number of SR node segments" )
   prefixSidCount = Int( help="Number of SR prefix segments" )
   proxyNodeSidCount = Int( help="Number of SR proxy node segments" )
   adjSidCount = Int( help="Number of IS-IS SR adjacency segments" )

   def render( self ):
      print( "" )
      print( "Self-Originated Segment Statistics:" )
      print( "Node-Segments       : %d" % ( self.nodeSidCount ) )
      print( "Prefix-Segments     : %d" % ( self.prefixSidCount ) )
      print( "Proxy-Node-Segments : %d" % ( self.proxyNodeSidCount ) )
      print( "Adjacency Segments  : %d" % ( self.adjSidCount ) )
      print( "" )

npHelpStr = "Set if the penultimate hop MUST NOT pop the Prefix-SID before " \
            "delivering the packet to the node that advertised the Prefix-SID"

class SrPrefixSegmentFlagsModel( Model ):
   """Common flags used in OSPF/ISIS prefix segments
   """
   e = Bool( help="Set if any upstream neighbor of the Prefix-SID originator MUST "
                  "replace the Prefix-SID with a Prefix-SID having an Explicit-NULL "
                  "value before forwarding the packet",
             default=False )
   v = Bool( help="Set if the Adj-SID carries a value", default=False )
   l = Bool( help="Set if the value/index carried by the Prefix-SID has local "
                  "significance",
             default=False )

# Base class used by IsisPrefixSegmentModel and OspfPrefixSegmentModel
class SrPrefixSegmentModel( Model ):
   """Model to store individual prefix segment record
   """
   selfOriginatedPrefix = Bool( help="Prefix is self originated" )
   prefix = IpGenericPrefix( help="Prefix address with netmask" )
   algorithm = Str( help="Algorithm of the prefix segment", optional=True )
   segmentId = Int( help="Segment ID" )
   # label = SRGB base of the advertising router + segmentId
   label = Int( help="Absolute Segment ID", optional=True )
   segmentType = Enum( values=[ "Proxy-Node", "Node", "Prefix" ], help="Type" )
   # SrPartiallyReachable indicates that some IS-IS next-hops are SR-unreachable
   # srReachability is always None in case of OSPF as it doesn't support
   # SR path verification
   srReachability = Enum( values=[ "srReachable", "srPartiallyReachable",
                                   "srUnreachable" ],
                          help="Reachability information for Prefix Segments" )

   def statusCode( self ):
      code = ""
      if self.selfOriginatedPrefix:
         code += "*"
      if self.srReachability == "srReachable":
         code += ""
      elif self.srReachability == "srPartiallyReachable":
         code += "#"
      elif self.srReachability == "srUnreachable":
         code += "!"
      return code

class SrGlobalBlockModel( Model ):
   """Model to store individual global block record, used by OSPF and ISIS
   """
   base = Int( help="SRGB base" )
   size = Int( help="SRGB size" )

class SrAdjacencySegmentFlagsModel( Model ):
   """Model for flags used in adjacency segment
   """
   b = Bool( help="Adj-SID is eligible for protection", default=False )
   v = Bool( help="Adj-SID carries a value", default=False )
   l = Bool( help="Adj-SID has local significance", default=False )

class SrAdjacencySegmentModel( Model ):
   """Model to store individual adjacency segments record
   """
   ipAddress = IpGenericAddress( help="IP Address of the adjacency" )
   localIntf = Interface( help="Local interface" )
   sid = Int( help="Adjacency Segment assigned to this adjacency" )
   lan = Bool( help="This is a LAN-SID" )
   sidOrigin = Enum( values=[ "configured", "dynamic" ],
         help="Indicates if adj-SID is configured manually or assigned dynamically" )
   protection = Enum( values=[ "link", "node", "unprotected" ],
                      help="Protection mode of the local adj-SID" )

#------------------------------------------------------------------------------------
# Helper functions to populate/render the models or fetch some data from smash/sysdb
#------------------------------------------------------------------------------------
def renderSRGB( model ):
   print( "SR Global Block( SRGB ): Base: %s\tSize: %s" %
          ( str( model.srgbBase ).ljust( 16 ), str( model.srgbSize ).ljust( 16 ) ) )

# Common render function for SR algo, srms capability etc
# The models aren't common because the help string are different for Ospf
# and Isis
def renderCommonSummary( model, protocol ):
   reachabilityAlgorithmMapping = dict( [ ( "SPF", 0 ), ( "SSPF", 1 ) ] )

   print( "" )
   # The flags are different for OSPF, and hence don't print them
   if protocol == "IS-IS":
      print( "All Prefix Segments have    : P:0 E:0 V:0 L:0" )
   print( "%s Reachability Algorithm : %s (%d)" % ( protocol,
            model.reachabilityAlgorithm,
            reachabilityAlgorithmMapping[ model.reachabilityAlgorithm ] ) )

   if protocol == "IS-IS":
      if model.inspectProxyAttachedFlag:
         print( "Proxy-node segment attached flag: inspected" )
      else:
         print( "Proxy-node segment attached flag: ignored" )

   if model.mappingServer:
      print( "This is a Segment Routing Mapping Server (SRMS)" )
   print( "" )
   print( "Number of %s segment routing capable nodes excluding self: "\
         "%d" % ( protocol, model.srPeerCount ) )

# Common render function to print prefix segment summary
def renderPrefixSegmentSummary( legend, table, nodeSidCount, prefixSidCount,
                                proxyNodeSidCount ):
   total = "Total Segments: %d" % ( nodeSidCount + prefixSidCount +
                                    proxyNodeSidCount )
   nodeC = "Node: %d" % ( nodeSidCount )
   prxyC = "Proxy-Node: %d" % ( proxyNodeSidCount )
   pfxC = "Prefix: %d" % ( prefixSidCount )

   print( "\n%s  %s  %s   %s" % ( nodeC.ljust( 11 ), prxyC.ljust( 17 ),
                                 pfxC.ljust( 13 ), total ) )
   print()
   print( legend )
   print( table.output() )
   print()

# Populate the SRGB values from sysdb into the model
def _populateSRGBValues( result, srSysdbStatus ):
   result.srgbBase = srSysdbStatus.labelRange.base
   result.srgbSize = srSysdbStatus.labelRange.size

# Populate the attribute of SrCommonHeader
def _populateSrCommonHeader( model, srDataPlane, srSysdbStatus ):
   dataPlaneMapping = dict( [ ( _srDataPlaneEnum.srDataPlaneNone, "NONE" ),
                    ( _srDataPlaneEnum.srDataPlaneMpls, "MPLS" ),
                    ( _srDataPlaneEnum.srDataPlaneIpv6, "IPV6" ),
                    ( _srDataPlaneEnum.srDataPlaneBoth, "BOTH" ) ] )

   model.dataPlane = dataPlaneMapping[ srDataPlane ]
   model.routerId = str( srSysdbStatus.igpRtrId )

# Get the adjacency segment origin/source
def getAdjSidOrigin( adjSeg, rtrId, igpInstanceId ):
   if adjSeg.rtrid.igpInstanceId != igpInstanceId:
      return None
   if adjSeg.staticSid:
      return "configured"
   elif adjSeg.rtrid.stringValue == rtrId:
      return "dynamic"
   else:
      return "remote"

def isSelfOriginated( ownRtrId, igpInstanceId, pfxSegment ):
   if str( pfxSegment.rtrid ) == str( ownRtrId ) and \
      pfxSegment.rtrid.igpInstanceId == igpInstanceId:
      return True
   if pfxSegment.prefixSegmentInfo.anycastSystemID is None:
      return False
   for asRtrid in pfxSegment.prefixSegmentInfo.anycastSystemID.values():
      if str( asRtrid ) == str( ownRtrId ) and \
         asRtrid.igpInstanceId == igpInstanceId:
         return True
   return False

# Populate attributes of SelfOriginatedSegmentStatisticsModel
def _getSelfOriginatedSegmentStatistics( srSmashStatus, srSysdbStatus, rtrId,
                                         igpInstanceId ):
   result = SelfOriginatedSegmentStatisticsModel( )
   result.nodeSidCount = 0
   result.prefixSidCount = 0
   result.proxyNodeSidCount = 0
   result.adjSidCount = 0
   for _, entry in srSmashStatus.prefixSegment.items( ):
      if not isSelfOriginated( rtrId, igpInstanceId, entry ):
         continue
      if entry.flags.proxy:
         result.proxyNodeSidCount += 1
      elif entry.flags.node:
         result.nodeSidCount += 1
      else:
         result.prefixSidCount += 1

   for _, entry in srSysdbStatus.adjacencySegment.items( ):
      sidOrigin = getAdjSidOrigin( entry, rtrId, igpInstanceId )
      if sidOrigin is not None and sidOrigin != "remote":
         result.adjSidCount += len( entry.nexthop )

   return result

def _getReachabilityAlgorithm():
   return "SPF"

def _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus ):
   return len( srSmashStatus.globalBlock )
