#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import glob
import os
import signal
import socket
from threading import Thread
import time

import BasicCli
import CliCommand
import CliMatcher
from CliToken.LogMgr import sendKeywordMatcher
import CmdExtension
import FileCliUtil
import Fru
import Tac
from Tac import SystemCommandError
import Url
from UrlPlugin.FlashUrl import FlashUrl
from UrlPlugin.NetworkUrl import NetworkUrl
from UrlPlugin.SupeUrl import (
   FileReplicationCmds,
   PeerSupeUrl,
)

supportBundleCmds = [
   'bash dmesg -T -x -S',
   'bash lsblk',
   'bash sudo ls -lahRS /var/log',
   'show arp vrf all',
   'show tech-support all',
   'show tech-support extended evpn',
   'show tech-support extended isis vrf all',
   'show tech-support extended ospf vrf all',
   'show tech-support extended ospfv3 vrf all',
   'show tech-support extended rip vrf all',
   'show agent qtrace',
]

# These commands are meant to be formatted with agent names. Running show memory
# allocation commands without specifying an agent can cause quite a memory spike as
# it stores all type info from every agent in memory prior to combining the result
# and printing. Running per agent mitigates this risk, and also gives us more
# information as we know the allocation of every agent.
showMemoryAllocationCmds = [
   'show agent {} memory allocations',
]

# All the memory allocation calls are appended to one file, which is named using
# the following string.
showMemoryAllocationsLabel = 'show-agent-memory-allocations'

supportBundleFiles = [
   '/mnt/flash/debug',
   '/mnt/flash/Fossil',
   '/mnt/flash/schedule/tech-support',
   '/proc/mounts',
   '/var/log/agents',
   '/var/log/canary.log',
   '/var/log/kernel.debug',
   '/var/log/rbfd.log',
   '/var/log/qt',
   '/var/log/startup-config-output',
]

#####################################################################################
# Implementation of the send support-bundle command
# send support-bundle [ exclude cores ] <dst-url> [ case-number <case-number> ]
#####################################################################################

supportBundleUrlMatcher = Url.UrlMatcher(
   lambda fs: ( fs.fsType in [ 'network', 'peer' ] or
                fs.fsType == 'flash' and fs.scheme != 'file:' ),
   helpdesc='Destination URL for the support bundle',
   acceptSimpleFile=False )

class SendSupportBundle( CliCommand.CliCommandClass ):
   syntax = ( 'send support-bundle [ exclude cores ] '
              'DEST_URL [ case-number CASE_NUMBER ]' )
   data = {
      'send': sendKeywordMatcher,
      'support-bundle': 'Create a tech-support bundle',
      'exclude': 'Exclude files from the tech-support bundle',
      'cores': 'Exclude core files',
      'DEST_URL': supportBundleUrlMatcher,
      'case-number': 'Enter a tech-support case number',
      'CASE_NUMBER': CliMatcher.IntegerMatcher(
         0, 9999999, helpdesc='Tech-support case number' ),
   }

   @staticmethod
   def handler( mode, args ):
      durl = args[ 'DEST_URL' ]
      caseNumber = args.get( 'CASE_NUMBER' )

      FileCliUtil.checkUrl( durl )
      if isinstance( durl, NetworkUrl ):
         if not durl.hostname:
            mode.addError( 'Network URL hostname not specified' )
            return

      # Create the support bundle filename, including the case number if specified
      supportBundlePrefix = 'support-bundle'
      localHostname = socket.gethostname()
      timestamp = time.strftime( '%Y_%m_%d-%H_%M_%S' )
      supportBundleNameList = [ supportBundlePrefix ]
      if caseNumber is not None:
         supportBundleNameList.append( 'SR' + str( caseNumber ) )
      supportBundleNameList += [ localHostname, timestamp ]
      supportBundleName = '-'.join( supportBundleNameList )

      # Create temporary directory for cli command named pipes
      tempOutputDir = '/tmp/support-bundle-cmds/'
      try:
         os.mkdir( tempOutputDir, 0o777 )
      except OSError as e:
         mode.addError( 'Error creating directory {} ({})'.format(
            tempOutputDir, e.strerror ) )
         return

      cliCmdExt = CmdExtension.getCmdExtender()

      def cleanUpTempOutputDir():
         cmd = [ 'rm', '-r', tempOutputDir ]
         cliCmdExt.runCmd( cmd, mode.session, asRoot=True )

      def cliCmdExtRun( cmdArgs ):
         return cliCmdExt.runCmd( cmdArgs, mode.session, asRoot=True )


      bashPids = []

      # Run supportBundleCmds and write to named pipes in tempOutputDir.
      # FIFO named pipes are used instead of directly writing the command output
      # to files in /tmp to avoid using up memory on tmpfs. When the zip command
      # runs, the output from the commands will be directed into the zip process
      # and written to the specified destination without being written in /tmp.
      def runCmdToNewPipe( supportBundleCmd ):
         outputPipeName = supportBundleCmd.replace( ' ', '-' ).replace( '/', '-' )
         outputPipePath = \
            f'{tempOutputDir}{localHostname}-{outputPipeName}-{timestamp}.log'

         os.mkfifo( outputPipePath, 0o777 )
         cmd = f'CliShell -Ap15 -c \"{supportBundleCmd}\" >& {outputPipePath}'
         mode.addMessage( 'Running cmd %s' % ( cmd ) )
         try:
            # Start the named pipe command in the background, and save its PID.
            pid = int( Tac.run( [ 'daemonize', '-p', '/dev/stdout', 'bash', '-c',
                                  cmd ], stdout=Tac.CAPTURE ) )
            bashPids.append( pid )
            supportBundleIncludedFiles.append( outputPipePath )
         except OSError as e:
            mode.addError( f'Error running cmd {cmd} ({e.strerror})' )

      showAllocsPipename = \
         f'{tempOutputDir}{localHostname}-{showMemoryAllocationsLabel}'\
         f'-{timestamp}.log'
      showAllocsPipe = None

      # Run a CLI command synchronously and return its output
      def runCmdToStdout( cliCmd ):
         try:
            cmd = [ 'CliShell', '-Ap15', '-c', cliCmd ]
            cmdStr = ' '.join( cmd )
            mode.addMessage( f'Running cmd {cmdStr}' )
            return Tac.run( cmd, stdout=Tac.CAPTURE )
         except OSError as e:
            mode.addError( f'Error running {cliCmd} ({e.strerror})' )
            return ''

      # Returns an array of all the agent names
      def getAllAgents( mode ):
         showAgentCmd = 'show agent names'
         return runCmdToStdout( showAgentCmd ).splitlines()

      # Run the show agent <Agent> memory allocation commands and write to named
      # pipe similar to the above. The show allocation command is run once per
      # agent, but only one pipe is used for all agents to avoid having many files
      # strewn about. Labelling is used to identify the start of each agent's
      # output.
      def runMemoryAllocationCmds():

         # This will be run in the background so that the pipe is
         # coordinated with the zip file which reads from the pipe.
         def doRunCommands():
            agentNames = getAllAgents( mode )

            # Will block until the zip command is reading from pipe.
            with open( showAllocsPipename, 'w' ) as showAllocsPipe:
               for cmdFormat in showMemoryAllocationCmds:
                  for agent in agentNames:
                     formattedCmd = cmdFormat.format( agent )
                     headerStr = f'Result of {formattedCmd}:\n'
                     showAllocsPipe.write( headerStr )

                     out = runCmdToStdout( formattedCmd )
                     if out:
                        showAllocsPipe.write( out + '\n' )
               showAllocsPipe.close()

         try:
            os.mkfifo( showAllocsPipename, 0o777 )

            # Kick off background daemon to collect all of the allocations
            mode.addMessage(
                  f'Piping memory allocation data to '
                  f'{showAllocsPipename}.' )
            showAllocsThread = Thread( target=doRunCommands )
            showAllocsThread.start()
            supportBundleIncludedFiles.append( showAllocsPipename )
         except OSError as e:
            mode.addError( f'Error making fifo ({e.strerror})' )

      try:
         # Include optional files if not excluded
         # Don't modify the global supportBundleFiles list, create copy
         supportBundleIncludedFiles = supportBundleFiles[ : ]

         for path in glob.glob( '/var/log/messages*' ):
            supportBundleIncludedFiles.append( path )
         for path in glob.glob( '/var/log/account*' ):
            supportBundleIncludedFiles.append( path )
         if 'cores' not in args:
            supportBundleIncludedFiles.append( '/var/core' )

         for supportBundleCmd in supportBundleCmds:
            runCmdToNewPipe( supportBundleCmd )

         runMemoryAllocationCmds()

         # Compose the zip command based on whether the dest is remote or local
         supportBundleZip = supportBundleName + '.zip'
         tmpErrorFile = '/tmp/%s-error-log' % supportBundleName

         if isinstance( durl, FlashUrl ):
            cmd = 'sudo zip -FI -qr {}/{} {} 2> {}'.format(
               durl.localFilename(), supportBundleZip,
               ' '.join( supportBundleIncludedFiles ), tmpErrorFile )
            runCmd = Tac.run
         elif isinstance( durl, PeerSupeUrl ):
            supeId = Fru.slotId()
            # pylint: disable=protected-access
            peerIp = FileReplicationCmds._peerIp( supeId )
            sshOptions = FileReplicationCmds._sshOptions( supeId, False, True )
            # pylint: enable=protected-access
            cmd = 'sudo zip -FI -qr - {} | sudo ssh {} {} \"cat > /{}/{}\" 2> {}' \
                  .format( ' '.join( supportBundleIncludedFiles ), sshOptions,
                           peerIp, durl.pathname, supportBundleZip, tmpErrorFile )
            runCmd = Tac.run
         elif isinstance( durl, NetworkUrl ):
            sshUser = durl.username
            sshHost = durl.hostname
            if not sshUser:
               # In case ssh user is None or empty, just set it to empty string
               # If no username is specified for ssh, current system user is used
               sshUser = ''
            else:
               sshUser = sshUser + '@'

            cmd = 'sudo zip -FI -qr - {} | ssh {}{} \"cat > /{}/{}\" 2> {}'.format(
               ' '.join( supportBundleIncludedFiles ), sshUser, sshHost,
               durl.pathname, supportBundleZip, tmpErrorFile )
            runCmd = cliCmdExtRun
         else:
            # With supportBundleUrlMatcher this should never happen, but just in case
            mode.addError(
               'URL type not supported for support bundle: %s' % durl.url )
            return

         # Run the zip command
         mode.addMessage( 'Running cmd %s' % cmd )
         try:
            runCmd( [ 'bash', '-c', cmd ] )
            mode.addMessage( 'Support bundle {} successfully sent to {}'.format(
               supportBundleZip, durl.url ) )
         except SystemCommandError as e:
            with open( tmpErrorFile ) as f:
               mode.addError(
                  'Error sending support bundle to destination {} ({})\n{}'.format(
                     durl.url, e, f.read() ) )
               return
      finally:
         # Clean up all remaining background processes.
         for pid in bashPids:
            try:
               # Kill the process group since the support bundle commands will spawn
               # their own background processes.
               os.kill( -pid, signal.SIGHUP )
            except ProcessLookupError:
               pass

         if showAllocsPipe and not showAllocsPipe.closed:
            showAllocsPipe.close()

         cleanUpTempOutputDir()

BasicCli.EnableMode.addCommandClass( SendSupportBundle )
