#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Int, Model, Str, Float, List
import Tac
from HumanReadable import formatTimeInterval

class TWSummary( Model ):
   '''
   Holds atributes and renders the appropriate prints for command:
      show twserver summary
   '''
   configuredVrfs = List( valueType=str, help='VRFs with servers configured' )
   activeVrfs = List( valueType=str, help='VRFs with servers running' )
   serverStartAttempts = Int( help='Total number of server start attempts' )
   serverStartSuccesses = Int( help='Total number of server start successes' )
   serverStopAttempts = Int( help='Total number of server stop attempts' )
   serverStopSuccesses = Int( help='Total number of server stop successes' )

   def render( self ):
      # Sample output show twserver summary:
      #    Current status
      #    Number of servers configured: 1
      #    VRFs with servers configured: default
      #    Number of servers running: 1
      #    VRFs with servers running: default

      #    Historical status
      #    Total number of server start attempts: 1
      #    Total number of server start successes: 1
      #    Total number of server stop attempts: 0
      #    Total number of server stop successes: 0

      # Sample output of show twserver summary | json:
      # {
      #  "configuredVrfs": [
      #      "default"
      #  ],
      #  "activeVrfs": [
      #      "default"
      #  ],
      #  "serverStartAttempts": 24,
      #  "serverStartSuccesses": 23,
      #  "serverStopAttempts": 20,
      #  "serverStopSuccesses": 18
      # }

      print( 'Current status' )
      print( f'Number of servers configured: '
            f'{len( self.configuredVrfs ) if self.configuredVrfs else 0}' )
      print( 'VRFs with servers configured: %s' %
            ( ', '.join( sorted( self.configuredVrfs ) ) if self.configuredVrfs
            else 'Not available' ) )
      print( f'Number of servers running: '
            f'{len( self.activeVrfs ) if self.activeVrfs else 0}' )
      print( 'VRFs with servers running: %s\n' %
            ( ', '.join( self.activeVrfs ) if self.activeVrfs
             else 'Not available' ) )
      print( 'Historical status' )
      print( f'Total number of server start attempts: {self.serverStartAttempts}' )
      print( f'Total number of server start successes: {self.serverStartSuccesses}' )
      print( f'Total number of server stop attempts: {self.serverStopAttempts}' )
      print( f'Total number of server stop successes: {self.serverStopSuccesses}' )

      if not self.activeVrfs:
         print( 'Server not currently running' )

class TWServer( Model ):
   '''
   Holds atributes and renders the appropriate prints for command:
      show twserver
   '''
   startTime = Float( help='Twserver boot time in epoch', optional=True )
   listenPort = Int( help='Server\'s listening port', optional=True )
   localIpOrHost = Str( help='Local address', optional=True )
   minTestPort = Int( help="Minimum test port", optional=True )
   maxTestPort = Int( help="Maximum test port", optional=True )
   maxControlSessions = Int( help="Maximum control sessions", optional=True )

   def render( self ):
      # Sample output of show twserver:
      #    Server running (up for 12:34:56)
      #    Listen port: 50099
      #    Local address: Any
      #    Min test port: 51000
      #    Max test port: 51100
      #    Max control sessions: unlimited

      # Sample output of show twserver | json:
      # {
      #  "startTime": 1702564485.006124,
      #  "listenPort": 50099,
      #  "localIpOrHost": "0",
      #  "minTestPort": 51000,
      #  "maxTestPort": 51100,
      #  "maxControlSessions": 0
      # }

      if not self.startTime:
         return
      uptime = Tac.utcNow() - self.startTime
      print( f'Server running (up for {formatTimeInterval( uptime )})' )
      if self.listenPort:
         print( f'Listen port: {self.listenPort}' )
      if self.localIpOrHost:
         print( 'Local address: %s' %
               ( 'Any' if self.localIpOrHost == '0' else self.localIpOrHost ) )
      if self.minTestPort:
         print( f'Min test port: {self.minTestPort}' )
      if self.maxTestPort:
         print( f'Max test port: {self.maxTestPort}' )
      if self.maxControlSessions == 0:
         print( 'Max control sessions: unlimited' )
      else:
         print( 'Max control sessions: %d' % self.maxControlSessions )
