# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import io
import tempfile

import ArPyUtils
import Ark
import BasicCli
import BasicCliModes
import BasicCliSession
import ConfigMount
import ConfigSessionCommon
import CliCommand
import CliMatcher
import CliSave
import CliSaveBlock
import CliSession
from CliSession import sessionLock, internalSessionName, validInternalSessionName
import CliToken.Configure
import CliToken.Service
import CliToken.Terminal
import DateTimeRule
import FileCliUtil
import GitCliLib
import LazyMount
import Tac
import Tracing
import Url

t0 = Tracing.trace0

cliSessionStatus = None
cliConfig = None

# Minimal config session commands, needed by Cli. The rest of the Clis are
# implemented in ConfigSession/CliPlugin.
# configure session [<name> ] [description <description>]
# commit [ timer <hours:minutes:seconds> ]
# abort [ name ]
# rollback clean-config
# rollback with name also does a clean-config first, in order
#  to add all the roots to this session (so it replaces the config, not
#  simply adds to it)
# future: update: if there are no conflicts, removes preempted by copying
#         current running-config to ancestor.

class ConfigSessionMode( BasicCli.GlobalConfigMode ):
   modeParseTreeShared = True

   def __init__( self, parent, session ):
      BasicCli.GlobalConfigMode.__init__( self, parent=parent, session=session )
      self.cs = CliSession.currentSession( self.session_.entityManager_ )
      self.userSession = False      # True if started by "configure session" CLI

   def enter( self ):
      pass

   def onExit( self ):
      if self.userSession:
         # TODO: `canCommitSession` is a misnomer;
         # It returns a reason why one can't commit or an empty string if one can.
         # So `not canCommitSession` means "no problem to commit a session".
         if ( self.session_.isInteractive() and
              not self.session_.sessionDataPop( 'skipExitSessionWarning', False )
              and CliSession.currentSessionState( self.entityManager ) == 'pending'
              and not CliSession.canCommitSession( self.entityManager, self.cs ) ):
            cmd = 'configure session ' + self.cs
            warning = ( 'Exiting configuration session without committing changes. '
                        'To get back to the session, type %r.' % cmd )
            self.addWarning( warning )
         CliSession.exitSession( self.session_.entityManager_ )
         ConfigSessionCommon.doLog( self,
               ConfigSessionCommon.SYS_CONFIG_SESSION_EXITED, self.cs,
               # don't log SESSION_EXITED to the terminal
               loggingTerminal=False )
      BasicCli.GlobalConfigMode.onExit( self )

# Not all sessions are meant to be user-visible. We shouldn't syslog
# actions taken on internal sessions ("internal" session are used by
# some other feature internally; the use of a session in those cases
# is just an implementation detail).
def shouldLogSessionActions( sessionName ):
   return not validInternalSessionName( sessionName )

def shouldManageSession( sessionName ):
   return not validInternalSessionName( sessionName )

def timerValueToSecs( mode, timerValue, useDefault=True ):
   if ( timerValue is None and
        mode.session_.isInteractive() and
        not CliSession.isCommitTimerInProgress() and
        useDefault and
        cliConfig.sysdbObj().defaultCommitTimer ):
      # BUG965948: For interactive sessions apply the default commit timeout if it
      # has been specified
      return int( cliConfig.sysdbObj().defaultCommitTimer )

   if timerValue is None:
      return None

   hrs, mins, secs = timerValue
   return hrs * 3600 + mins * 60 + secs

def secsToTimerValue( timerValueSecs ):
   if timerValueSecs is None:
      return None

   hrs = timerValueSecs // 3600
   mins = ( timerValueSecs % 3600 ) // 60
   secs = timerValueSecs - ( hrs * 3600 ) - ( mins * 60 )
   return ( hrs, mins, secs )

def commitSession( mode, timerValue=None, sessionName=None,
                   ignoreMergeCommit=False ):
   if timerValue == 0:
      mode.addError( "Time value cannot be zero." )
      return "Error"
   t0( 'Committing session', sessionName, 'in', mode, 'with timerValue', timerValue )

   em = mode.session_.entityManager_
   # use the values stored in Sysdb rather than the session when figuring out
   # what config knobs to apply
   sysdbCliConfig = cliConfig.sysdbObj()

   commitTimerInProgress = CliSession.isCommitTimerInProgress()
   t0( 'Is commit timer in progress ? ', commitTimerInProgress )
   response = CliSession.canCommitSession( em, sessionName )
   if response:
      mode.addError( response )
      return "Error"

   if sessionName is None:
      sessionName = CliSession.currentSession( em )
      if sessionName is None:
         mode.addError( "Cannot commit session due to missing name." )
         return "Error"

   # this means a commit timer is in progress. 2 things can happen here:
   # 1) The commit is confirmed and we just close out the session
   # 2) The timer is being updated, but session stays unconfirmed
   if commitTimerInProgress:
      if timerValue:
         ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED,
               sessionName, time=secsToTimerValue( timerValue ),
               loggingTerminal=sysdbCliConfig.loggingTerminal )

      # Ask the session manager to commit the session. If None is returned,
      # the session was concurrently deleted. Otherwise a non-empty string
      # means the session could not be committed.
      # NB: This call should never commit a session. It should either extend
      # the timer if timerValue is specified, otherwise it'll just close the
      # session
      response = CliSession.commitSession( em, None, timerValue, sessionName,
                                           mode=mode )
      if response:
         mode.addError( response )
         ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_FAILURE, sessionName,
               loggingTerminal=sysdbCliConfig.loggingTerminal )
         return "Error"

      if response is None:
         return None
      if not timerValue:
         ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_TIMER_COMPLETED,
            sessionName,
            loggingTerminal=sysdbCliConfig.loggingTerminal )

      return None

   if not BasicCliSession.CONFIG_LOCK.canRunConfigureCmds():
      raise ConfigMount.ConfigChangeProhibitedError(
         ConfigMount.ConfigChangeProhibitedError.CONFIG_LOCKED )

   # if we this commit isn't releated to a commit a session that's already been
   # commit and we want a merge commit then perform a merge commit.
   # This is done in 3 stages:
   # 1) Generate the merged config. This can generate a conflict error
   # 2) Roll-back to clean-config
   # 3) Apply all of the commands from the merged-config to the session
   # then we can proceed to merge the session as normal
   pcc = CliSession.sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not ignoreMergeCommit:
      if pcc and pcc.mergeOnCommit:
         # BUG729626: There is a bug with multi-line input and we need to pass
         # in a real file instead of something like StringIO
         with tempfile.TemporaryFile( mode='w+' ) as stream:
            stream.write( 'configure session %s\n' % sessionName )
            try:
               hasChanges = CliSave.saveMergedSessionConfig( em, stream, sessionName,
                                                             localRootsOnly=True )
            except CliSaveBlock.CliMergeConflict as e:
               errStream = io.StringIO()
               e.renderConflictMsg( errStream,
                     'session:/%s-ancestor-config' % sessionName,
                     'system:/running-config',
                     'session:/%s-session-config' % sessionName )
               mode.addError( errStream.getvalue() )
               mode.addError( 'Please use \'commit no-merge\' to replace the '
                              'running-config with the session-config' )

               return 'Error'

            # If the session has changes, then we perform a rollback only
            # for the roots that have been touched, and apply the merged config.
            # Otherwise, apply the session normally.
            if hasChanges:
               response = CliSession.rollbackSession( em, sessionName=sessionName,
                                                      localRootsOnly=True )
               if response:
                  mode.addError( response )
                  return "Error"

               stream.flush()
               stream.seek( 0 )
               # apply the merged config to the current session
               import MainCli # pylint: disable=import-outside-toplevel
               with ConfigMount.ConfigMountDisabler( disable=False ):
                  with ArPyUtils.StdoutAndStderrInterceptor() as out:
                     # All commands should have been authorized and accepted by
                     # now, so we disable AAA and guards.
                     MainCli.loadConfig( stream,
                                         mode.session,
                                         initialModeClass=BasicCli.EnableMode,
                                         disableAaa=True,
                                         disableGuards=True,
                                         skipConfigCheck=True )

               # Print out errors and the commands causing them.
               # Errors come out like this:
               # > configure session foo
               #
               # % Invalid input at line 1
               startIdx = None
               lines = out.contents().splitlines()
               for idx, line in enumerate( lines ):
                  if line.startswith( '>' ):
                     startIdx = idx
                  elif line.startswith( '%' ):
                     if startIdx is not None:
                        print( '\n'.join( lines[ startIdx : idx ] ) )
                     print( line )
                     startIdx = idx + 1

   changeId = None
   if sysdbCliConfig.maxCheckpoints > 0:
      changeId = GitCliLib.savePreCommitRunningConfig( mode, sessionName,
            maxHistorySize=sysdbCliConfig.maxCheckpoints )

   # Checkpoint only the commit timer(not commit) for
   # the first time.
   if timerValue:
      savePreCommitConfig( mode, changeId )

   # Ask the session manager to commit the session. If None is returned,
   # the session was concurrently deleted. Otherwise a non-empty string
   # means the session could not be committed.
   response = CliSession.commitSession( em, None, timerValue, sessionName,
                                        mode=mode )
   if response:
      mode.addError( response )
      ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_FAILURE, sessionName,
            loggingTerminal=sysdbCliConfig.loggingTerminal )
      return "Error"

   if response is None:
      return None

   if timerValue:
      # Log first as invokeOnCommitHandlersOrRevert might CANCEL the timer
      ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED,
            sessionName, time=secsToTimerValue( timerValue ),
            loggingTerminal=sysdbCliConfig.loggingTerminal )

   # Invoke commit handlers and rollback if error
   onCommitStatus = ConfigSessionCommon.invokeOnCommitHandlersOrRevert( mode,
                       sessionName, changeId, sysdbCliConfig.loggingTerminal )
   if onCommitStatus is not None:
      mode.addError( onCommitStatus )
      return None

   # if the commit was successful (and we aren't just continuing/closing an
   # exisiting commit timer) then declare the session as committed
   if sysdbCliConfig.maxCheckpoints > 0:
      GitCliLib.savePostCommitRunningConfig( mode, sessionName,
            description=pcc.userString,
            commitTimeExpiry=timerValue,
            maxHistorySize=sysdbCliConfig.maxCheckpoints )
   if shouldLogSessionActions( sessionName ):
      ConfigSessionCommon.doLog( mode,
         ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_SUCCESS,
         sessionName, description=pcc.userString,
         # only log if there is no timer value AND logging terminal is enabled
         loggingTerminal=not timerValue and sysdbCliConfig.loggingTerminal )
   with ConfigMount.ConfigMountDisabler( disable=True ):
      CliSession.runPostCommitHandlers( mode, sessionName )
   mode.session_.addToHistoryEventTable( "commandLine", "session",
                                         "running", sessionName, "" )
   return None

def savePreCommitConfig( mode, changeId ):
   """Save running-config into a checkpoint file."""
   t0( 'savePreCommitConfig in mode', mode )
   ctx = Url.Context( mode.session_.entityManager_, disableAaa=True,
                      cliSession=mode.session )

   checkpointUrl = None
   if changeId:
      commit = GitCliLib.getCommit( mode.entityManager.sysname(), changeId )
      if commit:
         t0( 'commit hash', commit, 'for change ID', changeId )
         checkpointUrl = Url.parseUrl( f'checkpoint:/{commit["commitHash"]}',
               ctx )
      else:
         t0( 'commit hash not found for change ID', changeId )
   else:
      t0( 'no change ID seen' )

   # Copy the config from either the latest checkpoint or the running config
   # if a checkpoint isn't available. This avoids having to re-generate the
   # running-config which can be slow
   surl = ( checkpointUrl if checkpointUrl else
               Url.parseUrl( 'system:/running-config', ctx ) )
   t0( 'savePreCommitConfig source config url', surl )

   # this could be invoked under different user IDs; so make sure we can write to it
   preCommitConfig = cliSessionStatus.preCommitConfig
   Tac.run( [ 'rm', '-f', preCommitConfig.lstrip( 'file:' ) ], asRoot=True )
   durl = Url.parseUrl( preCommitConfig, ctx )
   FileCliUtil.copyFile( None, mode, surl, durl, commandSource="configure replace" )

#-----------------------------------------------------------------------------------
# [ no | default ] configure session [ <sessionName> ] [ description <description> ]
#-----------------------------------------------------------------------------------
sessionKwForConfig = CliMatcher.KeywordMatcher( 'session',
      helpdesc='Enter configuration session; commands applied only on commit' )
sessionNameMatcher = CliMatcher.PatternMatcher(
      # We want to not match `description` or `pending-timer` for cosmetic reasons.
      # Othwerise we can run `config sess description description description`.
      #                                   ^^^^^^^^^^^ name.
      # Or `config sess pending-timer ?` returns more than just `commit`.
      # Note that the leading [^@] in the regex is there to disallow "@<foo>",
      # because validSessionName(<name>) accepts a leading "@", which is used
      # for some internal sessions
      r'(?!description$)(?!pending-timer$)([^@].*)',
      helpname='WORD',
      helpdesc='Name for session' )
descriptionKwMatcher = CliMatcher.KeywordMatcher( 'description',
      helpdesc='Session description' )
descriptionValueMatcher = CliMatcher.StringMatcher(
      helpdesc='Description message' )

@Ark.synchronized( sessionLock )
def enterConfigSession( mode, args ):
   sessionName = args.get( '<sessionName>' )
   noCommit = 'no-commit' in args
   if shouldManageSession( sessionName ):
      openSessionNum = 0
      for sname, pcs in cliSessionStatus.pendingChangeStatusDir.status.items():
         if sname == sessionName:
            openSessionNum = -1
            break
         if shouldManageSession( sname ) and not pcs.completed:
            openSessionNum = openSessionNum + 1
      if openSessionNum >= cliConfig.maxOpenSessions:
         mode.addError( "Maximum number of pending sessions has been reached. "
                        "Please commit or abort previous sessions "
                        "before creating a new one." )
         return
   em = mode.session_.entityManager_
   if not sessionName:
      sessionName = CliSession.uniqueSessionName( em )

   # `enterSession` also does this check, but we have to pre-check here,
   # because we may have to leave the current config session mode, if in one.
   if error := CliSession.canEnterSession( sessionName, em, noCommit ):
      mode.addErrorAndStop( error )

   # We know we'll be entering config-s mode from exec mode, so
   # in case we are typed in a config mode already, we need to
   # re-enable ConfigMounts and call the onExit() handlers for
   # config modes we've entered.
   with ConfigMount.ConfigMountDisabler( disable=False ):
      mode.session_.commitMode()

   if error := CliSession.enterSession( sessionName, em, noCommit=noCommit,
                                        description=args.get( '<description>' ),
                                        mergeOnCommit=cliConfig.mergeOnCommit ):
      mode.addErrorAndStop( error )

   # add a description to userString if Cli provides one. TODO
   newMode = ConfigSessionMode( mode, mode.session_ )
   # if internal session, then even if we came in through the Cli, this is
   # being driven by a feature that is managing this, so not "userSession"
   newMode.userSession = not internalSessionName( sessionName )
   mode.session_.gotoChildMode( newMode )
   if shouldLogSessionActions( sessionName ):
      ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_SESSION_ENTERED,
            CliSession.currentSession( em ),
            # don't log SESSION_ENTERED to the terminal
            loggingTerminal=False )

class EnterConfigSession( CliCommand.CliCommandClass ):
   syntax = 'configure session [ <sessionName> ] [ description <description> ]'
   noOrDefaultSyntax = 'configure session <sessionName> ...'
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher,
            'description': descriptionKwMatcher,
            '<description>': descriptionValueMatcher,
          }
   handler = enterConfigSession

   @staticmethod
   @Ark.synchronized( sessionLock )
   def noOrDefaultHandler( mode, args ):
      sessionName = args[ '<sessionName>' ] # not optional, so exists
      em = mode.session_.entityManager_

      if ( sessionName ==
         CliSession.sessionStatus.sessionStateDir.commitTimerSessionName ):
         ConfigSessionCommon.commitTimerRollback( mode )

      response = CliSession.deleteSession( em, sessionName )
      if response:
         mode.addError( response )
         return

      if shouldLogSessionActions( sessionName ):
         ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_SESSION_DELETED, sessionName,
               loggingTerminal=cliConfig.loggingTerminal )

BasicCli.EnableMode.addCommandClass( EnterConfigSession )

#-----------------------------------------------------------------------------------
# configure session <sessionName> no-commit [ description <description> ]
#-----------------------------------------------------------------------------------
class EnterConfigNoCommitSession( CliCommand.CliCommandClass ):
   syntax = 'configure session <sessionName> no-commit [ description <description> ]'
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher,
            'no-commit': CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'no-commit',
                                       helpdesc='Per user configure session that '
                                                'disallows commit' ),
                              hidden=True ),
            'description': descriptionKwMatcher,
            '<description>': descriptionValueMatcher,
          }
   handler = enterConfigSession

BasicCli.EnableMode.addCommandClass( EnterConfigNoCommitSession )

#-----------------------------------------------------------------------------------
# configure session <sessionName> | pending-timer commit [ timer TIMER ]
#-----------------------------------------------------------------------------------
timerKwMatcher = CliMatcher.KeywordMatcher(
   'timer',
   helpdesc='commit session with a timeout. '
            'If not committed within this time, config will be reverted.' )
timerValueMatcher = DateTimeRule.TimeMatcher( helpdesc='timeout' )

# ignore commit merge setting for the session
noMergeNode = CliCommand.hiddenKeyword( 'no-merge' )

abortKwMatcher = CliCommand.guardedKeyword( 'abort',
                                            'Abort configuration session',
                                            ConfigSessionCommon.sessionGuard )

pendingTimerDesc = "Commit the session that is pending a commit timer"

commitKwDesc = 'Commit pending session'

def _sessionNameFromArgs( mode, args ):
   if 'pending-timer' in args:
      sessionName = CliSession.commitTimerSessionName()
      if not sessionName:
         mode.addErrorAndStop( 'No session is currently pending a commit timer' )
   else:
      sessionName = args[ '<sessionName>' ]
   return sessionName

class ConfigSessionCommit( CliCommand.CliCommandClass ):
   syntax = ( 'configure session <sessionName> commit '
              '[ timer TIMER ] [ no-merge ]' )
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'session': sessionKwForConfig,
      '<sessionName>': sessionNameMatcher,
      'commit': commitKwDesc,
      'timer': timerKwMatcher,
      'TIMER': timerValueMatcher,
      'no-merge': noMergeNode
   }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      sessionName = _sessionNameFromArgs( mode, args )
      timerValue = timerValueToSecs( mode, args.get( 'TIMER' ) )
      commitSession( mode, timerValue=timerValue,
                     sessionName=sessionName,
                     ignoreMergeCommit='no-merge' in args )

BasicCli.EnableMode.addCommandClass( ConfigSessionCommit )

class ConfigSessionPendingTimerCommit( CliCommand.CliCommandClass ):
   syntax = '''configure session pending-timer commit [ timer TIMER ]'''
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            'pending-timer': pendingTimerDesc,
            'commit': commitKwDesc,
            'timer': timerKwMatcher,
            'TIMER': timerValueMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      sessionName = _sessionNameFromArgs( mode, args )
      timerValue = timerValueToSecs( mode, args.get( 'TIMER' ), useDefault=False )
      commitSession( mode, timerValue=timerValue,
                     sessionName=sessionName )

BasicCli.EnableMode.addCommandClass( ConfigSessionPendingTimerCommit )

#-----------------------------------------------------------------------------------
# configure session <sessionName> abort
#-----------------------------------------------------------------------------------
class ConfigSessionAbort( CliCommand.CliCommandClass ):
   syntax = '''configure session <sessionName> | pending-timer abort'''
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'session': sessionKwForConfig,
      '<sessionName>': sessionNameMatcher,
      'pending-timer': pendingTimerDesc,
      'abort': 'Abort configuration session'
   }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      sessionName = _sessionNameFromArgs( mode, args )
      if not CliSession.isSessionPresent( sessionName ):
         mode.addError( "Cannot abort non-existent session %s" % sessionName )
         return
      if ConfigSessionCommon.abortSession( mode, sessionName ):
         return
      isOkToLog = not CliSession.isSessionPendingCommitTimer( sessionName )
      if isOkToLog and shouldLogSessionActions( sessionName ):
         ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_SESSION_ABORTED, sessionName,
               loggingTerminal=cliConfig.loggingTerminal )

BasicCli.EnableMode.addCommandClass( ConfigSessionAbort )

#-----------------------------------------------------------------------------------
# commit [ timer TIMER ] [ no-merge ]
#-----------------------------------------------------------------------------------
class CommitInConfigureSession( CliCommand.CliCommandClass ):
   syntax = '''commit [ timer TIMER ] [ no-merge ]'''
   data = {
            'commit': CliCommand.guardedKeyword( 'commit',
                                                 'Commit pending session',
                                                 ConfigSessionCommon.sessionGuard ),
            'timer': timerKwMatcher,
            'TIMER': timerValueMatcher,
            'no-merge': noMergeNode,
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      timerValue = timerValueToSecs( mode, args.get( 'TIMER' ) )
      response = commitSession( mode, timerValue=timerValue,
               ignoreMergeCommit='no-merge' in args )
      if response is None:
         # Exit the session if we are running "commit" inside a session.
         #
         # Note if we are inside a session FOO, and run
         # "configure session FOO commit", we should not exit the session,
         # since we'd be left with global config mode instead of enable mode,
         # and it'd break things like "service configuration terminal disabled".
         CliSession.exitSession( mode.session_.entityManager_ )
         mode.session_.gotoParentMode()

ConfigSessionMode.addCommandClass( CommitInConfigureSession )

#-----------------------------------------------------------------------------------
# abort
#-----------------------------------------------------------------------------------
class AbortInConfigureSession( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort': CliCommand.guardedKeyword( 'abort',
                                                'Abort pending session',
                                                ConfigSessionCommon.sessionGuard )
            }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      sessionName = CliSession.currentSession( mode.session_.entityManager_ )
      response = ConfigSessionCommon.abortSession( mode, sessionName )
      mode.session_.gotoParentMode()
      if not response:
         ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_SESSION_ABORTED, sessionName,
               loggingTerminal=cliConfig.sysdbObj().loggingTerminal )

ConfigSessionMode.addCommandClass( AbortInConfigureSession )

#-----------------------------------------------------------------------------------
# rollback clean-config
#-----------------------------------------------------------------------------------
class RollbackSession( CliCommand.CliCommandClass ):
   syntax = 'rollback clean-config'
   data = {
             'rollback': CliCommand.Node(
                                    matcher=CliMatcher.KeywordMatcher( 'rollback',
                                              helpdesc='Clear configuration state' ),
                                    guard=ConfigSessionCommon.sessionGuard ),
             'clean-config': 'Copy config state from clean, default, config',
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      em = mode.session_.entityManager_
      response = CliSession.rollbackSession( em )
      if response:
         mode.addError( response )

ConfigSessionMode.addCommandClass( RollbackSession )

#-----------------------------------------------------------------------------------
# service configuration session max completed <maxSavedSessions>
#-----------------------------------------------------------------------------------
sessionKwMatcher = CliMatcher.KeywordMatcher( 'session',
                                              helpdesc='configure session settings' )
maxKwMatcher = CliMatcher.KeywordMatcher( 'max', helpdesc='set a max limit' )

class ConfigSessionMaxCompleted( CliCommand.CliCommandClass ):
   syntax = 'service configuration session max completed <maxSavedSessions>'
   noOrDefaultSyntax = 'service configuration session max completed ...'
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'max': maxKwMatcher,
            'completed': 'maximum number of completed sessions kept in memory',
            '<maxSavedSessions>': CliMatcher.IntegerMatcher( 0, 20,
                                       helpdesc='Maximum number of saved sessions' )
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.maxSavedSessions = args[ '<maxSavedSessions>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.maxSavedSessions = cliConfig.defaultMaxSavedSessions

BasicCli.GlobalConfigMode.addCommandClass( ConfigSessionMaxCompleted )

#-----------------------------------------------------------------------------------
# service configuration session max pending <maxPendingSessions>
#-----------------------------------------------------------------------------------
class ConfigSessionMaxPending( CliCommand.CliCommandClass ):
   syntax = 'service configuration session max pending <maxPendingSessions>'
   noOrDefaultSyntax = 'service configuration session max pending ...'
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'max': maxKwMatcher,
            'pending': 'maximum number of pending sessions',
            '<maxPendingSessions>': CliMatcher.IntegerMatcher( 1, 20,
                                      helpdesc='Maximum number of pending sessions' )
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.maxOpenSessions = args[ '<maxPendingSessions>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.maxOpenSessions = cliConfig.defaultMaxOpenSessions

BasicCli.GlobalConfigMode.addCommandClass( ConfigSessionMaxPending )

#-----------------------------------------------------------------------------------
# service configuration session commit merge
#-----------------------------------------------------------------------------------
class ConfigSessionMergeOnCommit( CliCommand.CliCommandClass ):
   syntax = 'service configuration session commit merge'
   noOrDefaultSyntax = syntax
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'commit': 'Modify behavior when committing a session',
            'merge': 'Perform a 3-way merge on commit',
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.mergeOnCommit = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.mergeOnCommit = cliConfig.defaultMergeOnCommit

BasicCli.GlobalConfigMode.addCommandClass( ConfigSessionMergeOnCommit )

#-----------------------------------------------------------------------------------
# service configuration terminal disabled
#-----------------------------------------------------------------------------------
class ConfigTerminalDisabled( CliCommand.CliCommandClass ):
   syntax = 'service configuration terminal disabled'
   noOrDefaultSyntax = syntax
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'terminal': CliToken.Terminal.terminalKwForConfig,
            'disabled': 'Disable configuring without a session',
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.configTerminalEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.configTerminalEnabled = cliConfig.defaultConfigTerminalEnabled

BasicCli.GlobalConfigMode.addCommandClass( ConfigTerminalDisabled )

#-----------------------------------------------------------------------------------
# service configuration session commit save startup-config
#-----------------------------------------------------------------------------------
class AutosaveToStartupOnCommit( CliCommand.CliCommandClass ):
   syntax = 'service configuration session commit save startup-config'
   noOrDefaultSyntax = syntax
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'commit': 'Modify behavior when committing a session',
            'save': 'Autosave feature',
            'startup-config': 'Save to startup config',
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.saveToStartupConfigOnCommit = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      default = cliConfig.defaultSaveToStartupConfigOnCommit
      cliConfig.saveToStartupConfigOnCommit = default

BasicCli.GlobalConfigMode.addCommandClass( AutosaveToStartupOnCommit )

#-----------------------------------------------------------------------------------
# service configuration session logging terminal
#-----------------------------------------------------------------------------------
class SessionLoggingTerminal( CliCommand.CliCommandClass ):
   syntax = 'service configuration session logging terminal'
   noOrDefaultSyntax = syntax
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'logging': 'Modify logging behavior',
            'terminal': 'Print to console configuration session state changes'
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.loggingTerminal = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.loggingTerminal = cliConfig.defaultLoggingTerminal

BasicCli.GlobalConfigMode.addCommandClass( SessionLoggingTerminal )

#-----------------------------------------------------------------------------------
# service configuration session commit timer default TIMER interactive
#-----------------------------------------------------------------------------------
class ServiceSessionTimer( CliCommand.CliCommandClass ):
   syntax = 'service configuration session commit timer default TIMER interactive'
   noOrDefaultSyntax = ( 'service configuration session commit timer default '
                         '[ TIMER ] interactive' )
   data = {
      'service': CliToken.Service.serviceKw,
      'configuration': CliToken.Service.configKwAfterService,
      'session': sessionKwMatcher,
      'commit': 'Modify behavior when committing a session',
      'timer': timerKwMatcher,
      'default': 'Default timer value',
      'TIMER': timerValueMatcher,
      'interactive': 'Apply to interactive sessions'
    }

   @staticmethod
   def handler( mode, args ):
      hr, minutes, sec = args[ 'TIMER' ]
      cliConfig.defaultCommitTimer = hr * 3600 + minutes * 60 + sec

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.defaultCommitTimer = 0

BasicCli.GlobalConfigMode.addCommandClass( ServiceSessionTimer )

#-----------------------------------------------------------------------------------
# Hook into the `configure [ terminal ]` command so we can err out if disabled.
#-----------------------------------------------------------------------------------
def configCmdCallbackForDisabledTerminal( mode, args ):
   if not cliConfig or cliConfig.configTerminalEnabled:
      return False

   if 'terminal' in args:
      mode.addErrorAndStop(
         'Configuration without a session has been disabled by the %r command. '
         'You can still use %r.' % (
            'service configuration terminal disabled',
            'configure [ session [ NAME ] ]' ) )
   else:
      # Automatically enter a session.
      # If already in one, supress the "Exiting ..." warning from its `onExit`.
      currentSessionName = CliSession.currentSession( mode.entityManager )
      if currentSessionName:
         # We are reentering a config session w/o specifying its name.
         mode.session_.sessionDataIs( 'skipExitSessionWarning', True )
         args[ '<sessionName>' ] = currentSessionName
         mode.addWarning( 'Already inside a configuration session' )
      enterConfigSession( mode, args )

   return True

BasicCliModes.EnableMode.configSessionHook = configCmdCallbackForDisabledTerminal

#-----------------------------------------------------------------------------------
# Plugin Func
#-----------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig, cliSessionStatus

   cliConfig = ConfigMount.mount( entityManager, "cli/session/input/config",
                                  "Cli::Session::CliConfig", "w" )

   cliSessionStatus = LazyMount.mount( entityManager, "cli/session/status",
                                       "Cli::Session::Status", "r" )
