# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from BasicCli import addShowCommandClass
from CliModel import cliPrinted
from ShowCommand import ShowCliCommandClass
import Cell
import LazyMount
from AgentCommandRequest import runCliPrintSocketCommand
from CliCommand import (
   CliExpression,
   Node,
   singleKeyword,
)
from CliMatcher import (
   DynamicNameMatcher,
)
from CliParser import (
   PRIO_LOW,
)
from CliPlugin.ClassificationCli import (
   applicationKw,
   appCategoryKw,
   appNameKw,
   appServiceKw,
   recognitionKw,
   trafficKw,
   allAppNamesHelpStr,
   categoryNameMatcher,
   serviceNamesMatcher,
)
from CliPlugin import SfeAppShowModel
from CliPlugin import VrfCli

appRecognitionConfig = None
classificationStatus = None
vrfNameStatus = None

debugKw = singleKeyword( 'debug',
                         helpdesc='Supply additional debug information' )

countersKw = singleKeyword( 'counters',
                             helpdesc='Counters for active applications' )

profileKw = singleKeyword( 'profile',
             helpdesc='Active applications matching a specific application profile' )

def appNamesWithUnclassified( mode ):
   apps = [ 'unclassified' ]
   if appRecognitionConfig:
      apps.extend( appRecognitionConfig.app )
   return list( set( apps ) )

# Matches default and custom apps
showAppCounterNamesMatcher = Node(
   matcher=DynamicNameMatcher( appNamesWithUnclassified,
                               allAppNamesHelpStr,
                               priority=PRIO_LOW ),
   maxMatches=1 )

def appProfileNamesWithDefault( mode ):
   profileNames = { 'default' }
   if appRecognitionConfig is None:
      return profileNames
   profileNames.update( appRecognitionConfig.appProfile )
   return profileNames

appProfileNameMatcher = Node(
   matcher=DynamicNameMatcher( appProfileNamesWithDefault,
                               "Application profile name",
                               priority=PRIO_LOW ),
   maxMatches=1 )

def executeCommand( mode, params, model ):
   paramStr = ' '.join( k + '=' + str( v ) for k, v in params.items() )
   return runCliPrintSocketCommand( mode.entityManager,
                                    'Sfe',
                                    'application-counters',
                                    paramStr,
                                    mode,
                                    keepalive=True,
                                    connErrMsg='Sfe agent is inactive',
                                    model=model )

# ------------------------------------------------------------------------------
#    show application traffic recognition counters
#         [ ( name APP ) | ( category CATEGORY ) | ( service SERVICE ) |
#           ( profile PROFILE ) | ( VRF ) ]
# ------------------------------------------------------------------------------
class ShowAppTraficRecCountersFilterExp( CliExpression ):
   expression = '''[ { ( name APP )
                     | ( service SERVICE_NAME )
                     | ( category CATEGORY_NAME )
                     | ( profile PROFILE_NAME )
                     | ( VRF )
                     | debug } ]'''
   data = {
      'name' : appNameKw,
      'APP' : showAppCounterNamesMatcher,
      'service' : appServiceKw,
      'SERVICE_NAME' : serviceNamesMatcher,
      'category' : appCategoryKw,
      'CATEGORY_NAME' : categoryNameMatcher,
      'profile' : profileKw,
      'PROFILE_NAME' : appProfileNameMatcher,
      'debug' : debugKw,
      'VRF' : VrfCli.VrfExprFactory(
                     helpdesc='Active applications in a specific vrf',
                     inclDefaultVrf=True,
                     maxMatches=1 )
   }

class ShowAppTrafficRecCountersCmd( ShowCliCommandClass ):
   syntax = ( '''show application traffic recognition counters [ FILTER ]''' )
   data = {
      'application' : applicationKw,
      'traffic' : trafficKw,
      'recognition' : recognitionKw,
      'counters' : countersKw,
      'FILTER' : ShowAppTraficRecCountersFilterExp
   }

   cliModel = SfeAppShowModel.ShowApplicationCounters

   @staticmethod
   def handler( mode, args ):
      params = {}

      if 'name' in args:
         appName = args[ 'APP' ]
         if appName in classificationStatus.applicationStatus:
            params[ 'app' ] = classificationStatus.applicationStatus[ appName ].appId
         elif appName == 'unclassified':
            params[ 'app' ] = 0
         else:
            mode.addError( f"application {appName} was not found" )
            return None

      if 'category' in args:
         categoryName = args[ 'CATEGORY_NAME' ]
         if categoryName in classificationStatus.categoryStatus:
            params[ 'category' ] = \
                  classificationStatus.categoryStatus[ categoryName ].categoryId
         else:
            mode.addError( f"category {categoryName} was not found" )
            return None

      if 'service' in args:
         serviceName = args[ 'SERVICE_NAME' ]
         if serviceName in classificationStatus.serviceStatus:
            params[ 'service' ] = \
                  classificationStatus.serviceStatus[ serviceName ].serviceId
         else:
            mode.addError( f"service {serviceName} was not found" )
            return None

      if 'profile' in args:
         profileName = args[ 'PROFILE_NAME' ]
         if ( profileName == 'default' ) or \
            ( profileName in appRecognitionConfig.appProfile ):
            params[ 'profile' ] = profileName
         else:
            mode.addError( f"profile name {profileName} was not found" )
            return None

      if 'VRF' in args:
         vrfName = args[ 'VRF' ]
         # Note we're passing vrfName so the agent can convert it to a platform
         # dependent ID
         if vrfNameStatus.vrfIdExists( vrfName ):
            params[ 'vrf' ] = vrfName
         else:
            mode.addError( f"vrf {vrfName} was not found" )
            return None

      if 'debug' in args:
         params[ 'debug' ] = 1

      model = executeCommand( mode, params,
                     SfeAppShowModel.ShowApplicationCounters )
      return cliPrinted( model ) if model else None

addShowCommandClass( ShowAppTrafficRecCountersCmd )

def Plugin( entityManager ):
   global appRecognitionConfig, classificationStatus, vrfNameStatus

   appRecognitionConfig = LazyMount.mount( entityManager,
                                          'classification/app-recognition/config',
                                          'Classification::AppRecognitionConfig',
                                          'r' )
   classificationStatus = LazyMount.mount( entityManager,
                             'classification/app-recognition/status',
                             'Classification::Status',
                             'r' )
   vrfNameStatus = LazyMount.mount( entityManager,
                                       Cell.path( 'vrf/vrfNameStatus' ),
                                       'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
