#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import AgentCommandRequest
from AgentCommandRequest import runCliPrintSocketCommand
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
from CliMatcher import KeywordMatcher
from CliToken.Platform import (
      platformMatcherForClear,
      platformMatcherForConfig,
      platformMatcherForShow,
)
from CliToken.Clear import clearKwNode
from CliToken.Trace import matcherTrace
from CliToken.Ip import ipMatcherForConfig
import ConfigMount
import SmashLazyMount
import LazyMount
import Tac
import SfeAgent
from QosTypes import tacRateUnit
from CliPlugin.ClassificationCliLib import applicationModeGuards
from CliPlugin.SfeCliLib import ( nodeSfe, nodeSfeEos, sfe, matcherStatus,
                                  matcherMemory, matcherPool, isPlatformCblOrInd,
                                  notHwPlatform )
from CliPlugin.SfeL3UnicastShowCliModel import Counter
from CliPlugin.SfeL3UnicastShowCliModel import CounterModel, PlatformModel, NicInfo
from CliPlugin.SfeL3UnicastShowCliModel import Licensing
from CliPlugin.SfeL3UnicastShowCliModel import Qos
from CliPlugin.SfeMemoryPoolShowModel import MemoryPoolShowModel
from CliPlugin import EthIntfCli
from CliMode.BessDpi import BessDpiEngineConfigMode
from CliMode.SfePort import PortConfigModeBase, PortProfileModeBase, PortIntfModeBase
from CliModel import cliPrinted
from CliCommand import guardedKeyword
from MultiRangeRule import MultiRangeMatcher
from Toggles.EosInitToggleLib import toggleSfeRssCaravanEnabled
import ShowCommand
from io import StringIO
import ast
import re

bessCliConfig = None
bessCounterDb = None
bessCounterDbBessMgr = None
bessCounterInfo = None
bessCounterInfoBessMgr = None
bessCounterSnapshot = None
cpuUtilDir = None
veosConfig = None
routingHwStatusCommon = None
sfeAgentInfo = None
sfeDpiConfig = None
capabilityConfig = None
portConfig = None

TEST_MAX_DP_CORES_DEF = 6
profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: portConfig.portProfileConfig, 'Port profile name' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='Create port profile' )

def getDpCpuInfo():
   nDpCores = len( cpuUtilDir.cpuUtil )
   if nDpCores:
      return nDpCores
   else: # btest
      return TEST_MAX_DP_CORES_DEF

def workersRange():
   return ( 0, getDpCpuInfo() - 1 )

def rxQueueRange( mode=None, context=None ):
   return ( 1, getDpCpuInfo() )

matcherNumRxQueues = CliMatcher.DynamicIntegerMatcher(
      rxQueueRange, helpdesc='Number of receive queues' )

modeMatcher = CliMatcher.EnumMatcher( {
      'shared' : 'workers will be shared',
      'exclusive' : 'workers will be exclusive'
      } )

#------------------------------------------------------------------------------
# The "platform sfe debug trace" hidden command, in "config" mode.
#-------------------------------------------------------------------------------
nodeDebug = CliCommand.Node(
            matcher=KeywordMatcher( 'debug',
                    helpdesc='Bess debugging setting commands' ),
            hidden=True )
nodeDpi = CliCommand.Node(
            matcher=KeywordMatcher( 'dpi',
                    helpdesc='DPI engine commands' ),
            hidden=False )
nodePort = CliCommand.Node(
            matcher=KeywordMatcher( 'port',
                    helpdesc='Hardware ethernet network interface' ),
            guard=isPlatformCblOrInd )

def unsetTrace( mode, args ):
   bessCliConfig.traceConfig = ''

def setTrace( mode, args ):
   traceConfig = args[ 'TRACECONFIG' ]
   bessCliConfig.traceConfig = traceConfig

class PlatformSfeDebugTraceCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe debug trace TRACECONFIG'
   noOrDefaultSyntax = 'platform sfe debug trace ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'debug' : nodeDebug,
      'trace' : matcherTrace,
      'TRACECONFIG' : CliMatcher.PatternMatcher( pattern='.+',
                      helpdesc='Trace config in the format of FACILITY/LEVELS,...',
                      helpname='EXPR' ),
   }

   handler = setTrace
   noOrDefaultHandler = unsetTrace

BasicCliModes.GlobalConfigMode.addCommandClass( PlatformSfeDebugTraceCmd )
#------------------------------------------------------------------------------
# The "ip load-sharing sfe hash funcId" command, in "config" mode.
#-------------------------------------------------------------------------------
def setTableIndex( mode, args ):
   bessCliConfig.hashFuncId = args.get( 'FUNC', 0 )

matcherLoadSharing = CliMatcher.KeywordMatcher( 'load-sharing',
        helpdesc='Configure ECMP route selection parameters' )

matcherHash = CliMatcher.KeywordMatcher( 'hash',
        helpdesc='Configure hash function used for load balancing' )

class SfeLoadSharingConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ip load-sharing sfe hash FUNC'
   noOrDefaultSyntax = 'ip load-sharing sfe hash ...'
   data = {
      'ip' : ipMatcherForConfig,
      'load-sharing' : matcherLoadSharing,
      'sfe' : nodeSfe,
      'hash' : matcherHash,
      'FUNC' : CliMatcher.IntegerMatcher( 0, 3, helpdesc='hash function ID' ),
   }

   handler = setTableIndex

   # noOrDefault syntax wil ignore the FUNC and anything after that.
   # hence hanlder will return 0
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( SfeLoadSharingConfigCmd )

#------------------------------------------------------------------------------
# (config)# platform sfe dpi
# -------------------------------------------------------------------------------

class PlatformSfeDpiCommand( CliCommand.CliCommandClass ):
   syntax = 'platform sfe dpi'
   noOrDefaultSyntax = syntax
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'dpi' : nodeDpi,
      }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( BessDpiEngineConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = BessDpiEngineConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( PlatformSfeDpiCommand )

# ------------------------------------------------------------------------------
# The "fastcase" command, in "platform sfe dpi" mode.
# -------------------------------------------------------------------------------

def setFastCaseValue( mode, args ):
   sfeDpiConfig.fastCase = True

def unsetFastCaseValue( mode, args ):
   sfeDpiConfig.fastCase = False

class SfeDpiFastCaseEnable( CliCommand.CliCommandClass ):
   syntax = 'fastcase'
   noOrDefaultSyntax = syntax
   data = {
      'fastcase' : 'Enable fastcase processing',
   }
   handler = setFastCaseValue
   noOrDefaultHandler = unsetFastCaseValue

BessDpiEngineConfigMode.addCommandClass( SfeDpiFastCaseEnable )

# ------------------------------------------------------------------------------
# The "basic-dpi" command, in "platform sfe dpi" mode.
# -------------------------------------------------------------------------------

def setBasicDpiValue( mode, args ):
   sfeDpiConfig.basicDpi = True

def unsetBasicDpiValue( mode, args ):
   sfeDpiConfig.basicDpi = False

class SfeDpiBasicDpiEnable( CliCommand.CliCommandClass ):
   syntax = 'basic-dpi'
   noOrDefaultSyntax = 'basic-dpi ...'
   data = {
      'basic-dpi' : 'Use a fast HTTP parser for improved performance',
   }

   handler = setBasicDpiValue
   noHandler = unsetBasicDpiValue
   defaultHandler = handler

BessDpiEngineConfigMode.addCommandClass( SfeDpiBasicDpiEnable )
# ------------------------------------------------------------------------------
# The "classification cache" command, in "platform sfe dpi" mode.
# -------------------------------------------------------------------------------

def setClassificationCacheValue( mode, args ):
   cache = 2 if 'performance' in args else 1
   sfeDpiConfig.classificationCache = cache

def unsetClassificationCacheValue( mode, args ):
   sfeDpiConfig.classificationCache = 0

class SfeDpiClassificationCache( CliCommand.CliCommandClass ):
   syntax = 'classification cache ( standard | performance )'
   noOrDefaultSyntax = 'classification cache...'
   data = {
      'classification' : 'Enable first packet classification',
      'cache' : 'Enable first packet classification',
      'standard' : 'Enable standard classification caching',
      'performance' : 'Bypass additional packet processing',
   }

   handler = setClassificationCacheValue
   noHandler = unsetClassificationCacheValue
   defaultHandler = handler

BessDpiEngineConfigMode.addCommandClass( SfeDpiClassificationCache )
# ------------------------------------------------------------------------------
# The "offload declassification" command, in "platform sfe dpi" mode.
# -------------------------------------------------------------------------------
def setOffloadDeclassification( mode, args ):
   sfeDpiConfig.offloadDeclassification = True

def unsetOffloadDeclassification( mode, args ):
   sfeDpiConfig.offloadDeclassification = False

offloadHelpString = '''Offload flows with a path containing
the declassification feature'''

class SfeDpiOffloadDeclassification( CliCommand.CliCommandClass ):
   syntax = 'offload declassification'
   noOrDefaultSyntax = syntax
   data = {
      'offload' : offloadHelpString,
      'declassification' : offloadHelpString,
   }
   handler = setOffloadDeclassification
   noOrDefaultHandler = unsetOffloadDeclassification

BessDpiEngineConfigMode.addCommandClass( SfeDpiOffloadDeclassification )

# ------------------------------------------------------------------------------
# The "platform sfe debug vlog" hidden command, in "config" mode.
#-------------------------------------------------------------------------------
def unsetVlog( mode, args ):
   bessCliConfig.vlogConfig = ''

def setVlog( mode, args ):
   vlogConfig = args[ 'VLOGCONFIG' ]
   bessCliConfig.vlogConfig = vlogConfig

class PlatformSfeDebugVlogCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe debug vlog VLOGCONFIG'
   noOrDefaultSyntax = 'platform sfe debug vlog ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'debug' : nodeDebug,
      'vlog' : 'Enable Bess VLog facility',
      'VLOGCONFIG' : CliMatcher.PatternMatcher( pattern='.+',
                     helpdesc='Vlog config in the format of FACILITY=LEVEL,...',
                     helpname='EXPR' ),
   }

   handler = setVlog
   noOrDefaultHandler = unsetVlog

BasicCliModes.GlobalConfigMode.addCommandClass( PlatformSfeDebugVlogCmd )

#---------------------------------------------------------------------------------
# show platform sfe qos
#---------------------------------------------------------------------------------
def doShowSfeQos( mode, args ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "QOS", stringBuff=buff )
   try:
      qosDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return None
   qos = Qos()
   qos.setAttrsFromDict( qosDict )
   return qos

class PlatformSfeQosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe qos'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'qos' : 'Show QOS information',
   }

   handler = doShowSfeQos
   cliModel = Qos
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeQosCmd )

#--------------------------------------------------------------------------------
# show platform sfe counters [ ( module | module-class | type ) [ RULE ] ]
#--------------------------------------------------------------------------------
matcherCounters = KeywordMatcher( 'counters', 'Bess debug counters' )
matcherCounterMatchRule = CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\./_-]+',
                          helpdesc='Counter matching rule', helpname='WORD' )
matcherCounterMatchType = CliMatcher.EnumMatcher( {
                          'module' : 'Name of the module holding the counter',
                          'module-class' : 'Module class of the counter owner',
                          'type' : 'Counter type' } )

def showBessCounter( mode, args ):
   matchingType = args.get( 'MATCHTYPE' )
   if matchingType == 'module-class':
      matchingType = 'moduleClass'
   matchingRule = args.get( 'RULE' )

   counterDict = {}
   ownerDict = {}
   counterOwnerInfoInvMap = \
      { v : k for k, v in bessCounterInfo.counterOwnerInfo.items() }

   for ( counterId, counterInfo ) in bessCounterInfo.counterInfo.items():
      ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
      if not matchingRule or \
         ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
         ( matchingRule == getattr( ownerInfo, matchingType, None ) ):

         smashCount = bessCounterDb.smashCount.get( counterId )
         if smashCount is None:
            continue

         snapshotCount = 0
         snapshotCounter = bessCounterSnapshot.counterCopy.get( counterId )
         if snapshotCounter and snapshotCounter.timestamp > counterInfo.timestamp:
            snapshotCount = snapshotCounter.count
         counter = Counter( counterType=counterInfo.type,
                            unit=counterInfo.unit,
                            count=smashCount.count - snapshotCount )
         if counterInfo.name:
            counter.name = counterInfo.name

         if ownerInfo:
            counter.ownerId = counterInfo.ownerId
            ownerDict.setdefault( counterInfo.ownerId, ownerInfo.module )

         counterDict[ counterId ] = counter

   if bessCounterDbBessMgr and bessCounterInfoBessMgr:
      counterOwnerInfoInvMap = \
         { v : k for k, v in bessCounterInfoBessMgr.counterOwnerInfo.items() }
      for ( counterId, counterInfo ) in bessCounterInfoBessMgr.counterInfo.items():
         ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
         if not matchingRule or \
            ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
            ( matchingRule == getattr( ownerInfo, matchingType, None ) ):
            smashCount = bessCounterDbBessMgr.smashCount.get( counterId )
            if smashCount is None:
               continue
            snapshotCount = 0
            snapshotCounter = bessCounterSnapshot.counterCopy.get( counterId )
            if snapshotCounter and snapshotCounter.timestamp > counterInfo.timestamp:
               snapshotCount = snapshotCounter.count
            counter = Counter( counterType=counterInfo.type,
                               unit=counterInfo.unit,
                               count=smashCount.count - snapshotCount )
            if counterInfo.name:
               counter.name = counterInfo.name
            if ownerInfo:
               counter.ownerId = counterInfo.ownerId
               ownerDict.setdefault( counterInfo.ownerId, ownerInfo.module )
            counterDict[ counterId ] = counter

   return CounterModel( counters=counterDict, owners=ownerDict )

class PlatformSfeCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe counters [ MATCHTYPE [ RULE ] ]'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'counters' : matcherCounters,
      'MATCHTYPE' : matcherCounterMatchType,
      'RULE' : matcherCounterMatchRule,
   }

   handler = showBessCounter
   cliModel = CounterModel

BasicCli.addShowCommandClass( PlatformSfeCountersCmd )

def showSfePlatform( mode, args ):
   cpuUtil = cpuUtilDir.cpuUtil
   nCpCores = min( cpuUtil ) if cpuUtil else 0
   nDpCores = len( cpuUtil )
   bessMem = veosConfig.bessMemoryMb

   platform = sfeAgentInfo.platform
   vrfCount = sfeAgentInfo.vrfCount
   maxVrfs = routingHwStatusCommon.vrfCapability.maxVrfs
   num2MbHugepages = sfeAgentInfo.num2MbHugepages
   num1GbHugepages = sfeAgentInfo.num1GbHugepages
   bessBuffers = sfeAgentInfo.bessBuffers
   cpuGhz = float( sfeAgentInfo.cpuClockSpeed.replace( ' MHz', '' ) ) / 1000
   cpuModel = sfeAgentInfo.cpuModel
   threadsPerCore = sfeAgentInfo.threadsPerCore
   cloudInstanceType = sfeAgentInfo.cloudInstanceType
   isolCpuLow = sfeAgentInfo.isolCpuLow
   isolCpuHigh = sfeAgentInfo.isolCpuHigh
   lastReloadTime = sfeAgentInfo.lastReloadTime
   dpdkVersion = sfeAgentInfo.dpdkVersion.replace( 'DPDK', '' ).strip()

   nicInfoDict = {}
   for nic in sfeAgentInfo.nic:
      nicInfo = sfeAgentInfo.nic[ nic ]
      nicShortName = nic
      if nic[ : 3 ] == 'hv_':
         nicShortName = nic[ 3 : ]
      # For modular interface names
      pattern = re.compile( r"\d_\d" )
      if pattern.search( nic ):
         nicShortName = nic.replace( "_", "/" )
      nicName = Tac.newInstance( "Arnet::IntfId" ).fromShortName(
            nicShortName.capitalize() )
      nicInfoDict[ nicName ] = NicInfo(
            pciAddr=nicInfo.pciAddr, macAddr=nicInfo.macAddr, driver=nicInfo.driver,
            version=nicInfo.version, busInfo=nicInfo.busInfo,
            nicType=nicInfo.nicType, l3SubIntfCapable=nicInfo.l3SubIntfCapable )

   return PlatformModel( platform=platform, nCpCores=nCpCores, nDpCores=nDpCores,
         dpMem=bessMem, cloudInstanceType=cloudInstanceType, cpuGhz=cpuGhz,
         cpuModel=cpuModel, vrfCount=vrfCount, maxVrfs=maxVrfs,
         num2MbHugepages=num2MbHugepages, num1GbHugepages=num1GbHugepages,
         bessBuffers=bessBuffers, threadsPerCore=threadsPerCore,
         isolCpuLow=isolCpuLow, isolCpuHigh=isolCpuHigh,
         lastReloadTime=lastReloadTime, dpdkVersion=dpdkVersion,
         nicInfo=nicInfoDict )

class PlatformSfeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe status'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'status' : matcherStatus,
   }

   handler = showSfePlatform
   cliModel = PlatformModel

BasicCli.addShowCommandClass( PlatformSfeCmd )

# ---------------------------------------------------------------------------------
# show platform sfe memory pool
# ---------------------------------------------------------------------------------

class PlatformSfeMemoryPoolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe memory pool'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'memory' : matcherMemory,
      'pool' : matcherPool
   }
   cliModel = MemoryPoolShowModel

   @staticmethod
   def handler( mode, args ):
      model = runCliPrintSocketCommand( mode.entityManager,
                                        'Sfe',
                                        'memory-pool',
                                        '',
                                        mode,
                                        keepalive=True,
                                        connErrMsg='Sfe agent is inactive',
                                        model=MemoryPoolShowModel )
      return cliPrinted( model ) if model else None

BasicCli.addShowCommandClass( PlatformSfeMemoryPoolCmd )

# ---------------------------------------------------------------------------------
# show platform sfe memory counters
# ---------------------------------------------------------------------------------
counterMemoryMatcher = KeywordMatcher( 'counters', 'Memory used by Bess counters' )

class PlatformShowBessdCountersMemoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe memory counters'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'memory' : matcherMemory,
      'counters' : counterMemoryMatcher
   }
   cliModel = "BessdCountersShowModel.CountersMemShowModel"
   handler = "BessdCountersShowMemCli.showBessdCountersMemHandler"

BasicCli.addShowCommandClass( PlatformShowBessdCountersMemoryCmd )

#---------------------------------------------------------------------------------
# clear platform sfe counters [ ( module | module-class | type ) [ RULE ] ]
#---------------------------------------------------------------------------------
def clearBessCounter( mode, args ):
   matchingType = args.get( 'MATCHTYPE' )
   if matchingType == 'module-class':
      matchingType = 'moduleClass'
   matchingRule = args.get( 'RULE' )

   counterOwnerInfoInvMap = \
      { v : k for k, v in bessCounterInfo.counterOwnerInfo.items() }

   # walk through current counter to do snapshot
   for ( counterId, counterInfo ) in bessCounterInfo.counterInfo.items():
      ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
      if not matchingRule or \
         ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
         ( matchingRule == getattr( ownerInfo, matchingType, None ) ):

         smashCount = bessCounterDb.smashCount.get( counterId )
         if smashCount is None or smashCount.count == 0:
            continue

         bessCounterSnapshot.counterCopy[ counterId ] = \
            Tac.Value( "Sfe::BessCounterCopy", count=smashCount.count,
                       timestamp=Tac.now() )

   # walk through current counter to do snapshot
   if bessCounterDbBessMgr and bessCounterInfoBessMgr:
      counterOwnerInfoInvMap = \
         { v : k for k, v in bessCounterInfoBessMgr.counterOwnerInfo.items() }

      for ( counterId, counterInfo ) in bessCounterInfoBessMgr.counterInfo.items():
         ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
         if not matchingRule or \
            ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
            ( matchingRule == getattr( ownerInfo, matchingType, None ) ):

            smashCount = bessCounterDbBessMgr.smashCount.get( counterId )
            if smashCount is None or smashCount.count == 0:
               continue

            bessCounterSnapshot.counterCopy[ counterId ] = \
               Tac.Value( "Sfe::BessCounterCopy", count=smashCount.count,
                          timestamp=Tac.now() )

   # walk through snapshot to cleanup stale counterIds
   for ( counterId, counterCopy ) in bessCounterSnapshot.counterCopy.items():
      counterInfo = bessCounterInfo.counterInfo.get( counterId )
      counterInfoBessMgr = None
      if bessCounterInfoBessMgr:
         counterInfoBessMgr = bessCounterInfoBessMgr.counterInfo.get( counterId )
      if ( ( counterInfo is None or counterInfo.timestamp >= counterCopy.timestamp )
           and ( counterInfoBessMgr is None or
                 counterInfoBessMgr.timestamp >= counterCopy.timestamp ) ):
         del bessCounterSnapshot.counterCopy[ counterId ]

   buff = StringIO()
   if not matchingType and not matchingRule and capabilityConfig.nat  :
      AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "NATClearCounters", stringBuff=buff )

class ClearPlatformSfeCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear platform sfe counters [ MATCHTYPE [ RULE ] ]'
   data = {
      'clear' : clearKwNode,
      'platform' : platformMatcherForClear,
      'sfe' : nodeSfeEos,
      'counters' : matcherCounters,
      'MATCHTYPE' : matcherCounterMatchType,
      'RULE' : matcherCounterMatchRule,
   }
   hidden = not sfe()

   handler = clearBessCounter

BasicCliModes.EnableMode.addCommandClass( ClearPlatformSfeCountersCmd )

#---------------------------------------------------------------------------------
# clear platform sfe counters session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters module [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters module-class [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters type [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# show platform sfe licensing
#---------------------------------------------------------------------------------
def doShowSfeLicensing( mode, args ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "LICENSE", stringBuff=buff )
   try:
      licDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return None
   lic = Licensing()
   lic.setAttrsFromDict( licDict )
   return lic

class PlatformSfeLicensingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe licensing'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'licensing' : CliCommand.guardedKeyword( 'licensing',
         'Show licensing information', notHwPlatform )
   }

   handler = doShowSfeLicensing
   cliModel = Licensing
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeLicensingCmd )

def applicationModePlugin( token ):
   return sfe() and ( token == "ipv4" ) # enable `application ipv4 APPNAME`

applicationModeGuards.addExtension( applicationModePlugin )

def Plugin( entityManager ):
   global bessCliConfig, bessCounterDb, bessCounterInfo, bessCounterSnapshot
   global bessCounterDbBessMgr, bessCounterInfoBessMgr, capabilityConfig
   global routingHwStatusCommon
   global cpuUtilDir, veosConfig, sfeAgentInfo, sfeDpiConfig
   global portConfig

   bessCliConfig = ConfigMount.mount(
      entityManager, "bess/cli/config", "Sfe::BessdCliConfig", "w" )
   bessCounterDbMountInfo = SmashLazyMount.mountInfo( 'reader' )
   bessCounterDb = SmashLazyMount.mount( entityManager, "bess/counter",
                                         "SfeModules::SmashCounterDbStatus",
                                         bessCounterDbMountInfo )
   bessCounterInfo = LazyMount.mount( entityManager, "bess/counter/info",
                                      "Sfe::BessCounterInfo", "r" )
   bessCounterDbBessMgr = SmashLazyMount.mount(
      entityManager, "bess/counterBessMgr", "SfeModules::SmashCounterDbStatus",
      bessCounterDbMountInfo )
   bessCounterInfoBessMgr = LazyMount.mount(
      entityManager, "bess/counter/infoBessMgr", "Sfe::BessCounterInfo", "r" )
   bessCounterSnapshot = LazyMount.mount( entityManager, "bess/cli/counter/snapshot",
                                          "Sfe::BessCounterSnapshot", "wr" )
   cpuUtilDir = LazyMount.mount( entityManager, "dp/cpu/util",
                                    "Sfe::CpuUtilDir", "r" )
   veosConfig = LazyMount.mount( entityManager, "hardware/sfe/veosConfig",
                                    "Sfe::VeosConfig", "r" )
   routingHwStatusCommon = LazyMount.mount(
      entityManager,
      "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   sfeAgentInfo = LazyMount.mount( entityManager, "sfe/agent/info",
                                    "Sfe::AgentInfo", "r" )
   sfeDpiConfig = ConfigMount.mount( entityManager, "bess/cli/dpi",
                                    "Sfe::BessDpiConfig", "w" )
   capabilityConfig =  LazyMount.mount(
      entityManager, 'hardware/sfe/capability',
      'Sfe::CapabilityConfig', 'r' )
   portConfig = ConfigMount.mount( entityManager, "sfe/port/config",
                                    "Sfe::SfePortConfig", "w" )

#------------------------------------------------------------------------------
# [ no | default ] paltform sfe control-plane receive-limit <value> <kbps|mbps>
#-------------------------------------------------------------------------------
def cpuReceivelimit( mode, args ):
   receiveLimitValue = args[ 'LIMIT' ]
   if 'kbps' in args:
      receiveLimitUnit = tacRateUnit.rateUnitKbps
   else:
      receiveLimitUnit = tacRateUnit.rateUnitMbps
   bessCliConfig.cpuReceiveLimit = Tac.Value(
         "Sfe::CpuReceiveLimitConfig", receiveLimitValue, receiveLimitUnit )

def noCpuReceivelimit( mode, args ):
   bessCliConfig.cpuReceiveLimit = Tac.Value( "Sfe::CpuReceiveLimitConfig" )

class PlatformSfeControlPlaneReceiveLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe control-plane receive-limit LIMIT ( kbps | mbps )'
   noOrDefaultSyntax = 'platform sfe control-plane receive-limit ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'control-plane' : 'Control plane configuration',
      'receive-limit' : 'Rate limit for control plane traffic',
      'LIMIT' : CliMatcher.IntegerMatcher( 1, 10000000,
                                           helpdesc='Rate limit' ),
      'kbps' : 'Rate limit unit kbps',
      'mbps' : 'Rate limit unit mbps',
   }

   handler = cpuReceivelimit
   noOrDefaultHandler = noCpuReceivelimit

BasicCliModes.GlobalConfigMode.addCommandClass(
      PlatformSfeControlPlaneReceiveLimitCmd )

#------------------------------------------------------------------------------
# [ no | default ] platform sfe data-plane cpu maximum <max_cpus>
#-------------------------------------------------------------------------------
reloadMessage = "Please save your config and reload the system to make the changes"\
        " effective"

def getCoreCount():
   return int( os.getenv( 'SFE_FAKE_CORE_COUNT',
         os.sysconf( "SC_NPROCESSORS_ONLN" ) ) )

def dataPlaneCoresCmd( mode, args ):
   maxCoreValue = args.get( 'MAX_DP_CORES_NUM', 0 )
   bessCliConfig.maxDpCores = maxCoreValue
   mode.addMessage( reloadMessage )

class PlatformSfeDataPlaneCoresCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe data-plane cpu allocation maximum MAX_DP_CORES_NUM'
   noOrDefaultSyntax = 'platform sfe data-plane cpu allocation maximum...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'data-plane' : 'Data plane configuration',
      'cpu' : 'CPU related configuration',
      'allocation' : 'Control the number of physical CPUs used for data plane '
              'traffic forwarding',
      'maximum' : 'Maximum number of CPUs assigned for forwarding traffic',
      'MAX_DP_CORES_NUM' : CliMatcher.IntegerMatcher( 1, 128,
                            helpdesc='Maximum number of CPUs assigned for '
                            'forwarding traffic' ),
   }
   handler = dataPlaneCoresCmd
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass(
      PlatformSfeDataPlaneCoresCmd )

def platformSfeIpChecksumDisableHandler( mode, args ):
   bessCliConfig.ipHwChecksumOffload = False
   mode.addWarning( "IP hardware checksum offload disabled. "
                    "Please save the config and restart Sfe "
                    "for this change to take effect." )

def platformSfeIpChecksumEnableHandler( mode, args ):
   bessCliConfig.ipHwChecksumOffload = True
   mode.addWarning( "IP hardware checksum offload enabled. "
                    "Please save the config and restart Sfe "
                    "for this change to take effect." )

# ------------------------------------------------------------------------------
# [no | default] platform sfe ip checksum offload disabled
# -------------------------------------------------------------------------------
class PlatformSfeIpChecksumOffloadDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe ip checksum offload disabled'
   noOrDefaultSyntax = 'platform sfe ip checksum offload ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'ip' : guardedKeyword( 'ip', 'IP configuration',
                             guard=isPlatformCblOrInd ),
      'checksum' : 'checksum configuration',
      'offload' : 'enable/disable IP checksum offload',
      'disabled' : 'disable IP checksum offload'
   }
   handler = platformSfeIpChecksumDisableHandler
   noOrDefaultHandler = platformSfeIpChecksumEnableHandler

BasicCliModes.GlobalConfigMode.addCommandClass(
      PlatformSfeIpChecksumOffloadDisabledCmd )

# ---------------------------------------------------------------------------
# platform sfe port implementation start
# ---------------------------------------------------------------------------
def getProfileConfig( profileName ):
   return portConfig.portProfileConfig.get( profileName )

class PortProfileModeContext:
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profileName = profileName
      self.profileConfig = None

   def addProfile( self ):
      self.profileConfig = portConfig.portProfileConfig.newMember( self.profileName )

   def removeProfile( self ):
      if not getProfileConfig( self.profileName ):
         return
      del portConfig.portProfileConfig[ self.profileName ]
      self.profileConfig = None

class PortProfileIntfContext:
   def __init__( self, mode, intfName ):
      self.mode = mode
      self.intfName = intfName
      self.profileConfig = None
      self.intfConfig = None

   def addProfileIntf( self ):
      mode = self.mode.portProfileModeContext
      self.profileConfig = mode.profileConfig
      intf = Tac.Value( "Arnet::IntfId", self.intfName )
      self.intfConfig = self.profileConfig.profileIntfConfig.newMember( intf )

   def removeProfileIntf( self, add=True ):
      mode = self.mode.portProfileModeContext
      self.profileConfig = mode.profileConfig
      intf = Tac.Value( "Arnet::IntfId", self.intfName )
      if intf in self.profileConfig.profileIntfConfig:
         del self.profileConfig.profileIntfConfig[ intf ]

class PortConfigMode( PortConfigModeBase, BasicCli.ConfigModeBase ):
   name = "Port configuration mode"

   def __init__( self, parent, session ):
      PortConfigModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class PortProfileMode( PortProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Port profile configuration mode"

   def __init__( self, parent, session, context ):
      assert isinstance( parent, PortConfigMode )
      self.portProfileModeContext = context
      self.profileName = context.profileName
      PortProfileModeBase.__init__( self, self.profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class PortIntfMode( PortIntfModeBase, BasicCli.ConfigModeBase ):
   name = "Port profile interfaces configuration mode"

   def __init__( self, parent, session, intfCtx ):
      assert isinstance( parent, PortProfileMode )
      self.intfCtx = intfCtx
      self.intfName = intfCtx.intfName
      PortIntfModeBase.__init__( self, parent.profileName, self.intfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# ------------------------------------------------------------------------------
# (config)# [ no/default ] platform sfe port in GlobalConfig mode
# ------------------------------------------------------------------------------
class PlatformSfePortCommand( CliCommand.CliCommandClass ):
   syntax = 'platform sfe port'
   noOrDefaultSyntax = syntax
   data = {
         'platform' : platformMatcherForConfig,
         'sfe' : nodeSfe,
         'port' : nodePort,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PortConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      portConfig.portProfileConfig.clear()
      portConfig.portProfileApplied = ""

# ------------------------------------------------------------------------------
# (config-platform-sfe-port)#profile <profileName> in platform sfe port mode
# ------------------------------------------------------------------------------
class PortProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'profile <profileName>'
   noOrDefaultSyntax = syntax
   data = {
         'profile' : matcherProfile,
         '<profileName>' : profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ '<profileName>' ]
      context = PortProfileModeContext( mode, profileName )
      context.addProfile()
      childMode = mode.childMode( PortProfileMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ '<profileName>' ]
      context = PortProfileModeContext( mode, profileName )
      context.removeProfile()

# ------------------------------------------------------------------------------
# (config-port-profile-<profileName>)#interface INTFNAME
# ------------------------------------------------------------------------------
class PortProfileIntfCommand( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = 'interface INTF'
   data = {
         'interface' : 'Configure interface parameters',
         'INTF' : EthIntfCli.EthPhyIntf.ethMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      intfCtx = PortProfileIntfContext( mode, intf.name )
      intfCtx.addProfileIntf()
      childMode = mode.childMode( PortIntfMode, intfCtx=intfCtx )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      intfCtx = PortProfileIntfContext( mode, intf.name )
      intfCtx.removeProfileIntf()

class PortIntfRxQueueCommand( CliCommand.CliCommandClass ):
   syntax = 'rx-queue count RXQ'
   noOrDefaultSyntax = 'rx-queue count ...'
   data = {
         'rx-queue' : 'Configure interface receive queue parameters',
         'count' : 'Number of receive queues',
         'RXQ' : matcherNumRxQueues,
   }

   @staticmethod
   def handler( mode, args ):
      mode.intfCtx.intfConfig.numRxQs = args[ 'RXQ' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intfCtx.intfConfig.numRxQs = mode.intfCtx.intfConfig.numRxQsDefault

class PortIntfRxWorkerCommand( CliCommand.CliCommandClass ):
   syntax = 'rx-queue worker WORKERS'
   noOrDefaultSyntax = 'rx-queue worker ...'
   data = {
         'rx-queue' : 'Configure interface receive queue parameters',
         'worker' : 'Assign receive queues to workers',
         'WORKERS' : CliCommand.Node(
                        matcher=MultiRangeMatcher(
                           rangeFn=workersRange,
                           noSingletons=False,
                           helpdesc='list of workers',
                           priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      workerIds = list( args[ 'WORKERS' ].values() ) if 'WORKERS' in args else []
      intfConfig = mode.intfCtx.intfConfig
      intfConfig.rxWorkers.clear()
      for worker in workerIds:
         intfConfig.rxWorkers[ worker ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intfCtx.intfConfig.rxWorkers.clear()

class PortIntfRxModeCommand( CliCommand.CliCommandClass ):
   syntax = 'rx-queue mode MODE'
   noOrDefaultSyntax = 'rx-queue mode ...'
   data = {
         'rx-queue' : 'Configure interface receive queue parameters',
         'mode' : 'Receive queues to worker affinity mode type',
         'MODE' : modeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.intfCtx.intfConfig.rxIntfMode = args.get( 'MODE' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intfCtx.intfConfig.rxIntfMode = mode.intfCtx.intfConfig.rxIntfModeDefault

# ------------------------------------------------------------------------------
# (config-platform-sfe-port)#port profile <profileName>
# ------------------------------------------------------------------------------
class PortProfileApplyCommand( CliCommand.CliCommandClass ):
   syntax = 'port profile <profileName>'
   noOrDefaultSyntax = 'port profile ...'
   data = {
         'port' : 'Apply port profile',
         'profile' : 'profile',
         '<profileName>' : profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Set new profile applied
      portConfig.portProfileApplied = args[ '<profileName>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      portConfig.portProfileApplied = ""

if toggleSfeRssCaravanEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( PlatformSfePortCommand )
   PortConfigMode.addCommandClass( PortProfileCommand )
   PortConfigMode.addCommandClass( PortProfileApplyCommand )
   PortProfileMode.addCommandClass( PortProfileIntfCommand )
   PortIntfMode.addCommandClass( PortIntfRxQueueCommand )
   PortIntfMode.addCommandClass( PortIntfRxWorkerCommand )
   PortIntfMode.addCommandClass( PortIntfRxModeCommand )
# ---------------------------------------------------------------------------
# platform sfe port implementation end
# ---------------------------------------------------------------------------
