#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliCommand
import CliMatcher
import CliParser
import LazyMount
import AgentDirectory
import SmashLazyMount
import Cell
import ConfigMount

import os

entityManager = None
portVifStatus = None
sfePhyStatusDir = None
launcherConfig = None
v4RouteCacheHitbitStatus = None
phyConfigDir = None
bessCliConfig = None
veosConfigDir = None

def sfe():
   # pylint: disable=consider-using-ternary
   return ( ( launcherConfig and 'Sfe' in launcherConfig ) or
            os.getenv( "SIMULATION_SFE" ) )

def sfeGuard( mode, token ):
   if sfe():
      return None
   else:
      return CliParser.guardNotThisPlatform

def sfeEosGuard( mode, token ):
   if sfe() or AgentDirectory.agent( mode.sysname, 'BessMgr' ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def notHwPlatform( mode, token ):
   pcdPath = f'hardware/cell/{Cell.cellId()}/phy/sfe/config'
   pcd = phyConfigDir
   if pcd is None:
      pcd = mode.sysdbRoot.entity[ pcdPath ]

   # If it is hw platform return not-supported right away
   if pcd.isSfeSecondaryForwardingAgent:
      return CliParser.guardNotThisPlatform

   # If it is not hwPlatform, do SFE specific check
   return sfeGuard( mode, token )

def bfnGuard( mode, token ):
   if AgentDirectory.agent( mode.sysname, 'BfnCentral' ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def strataGuard( mode, token ):
   if AgentDirectory.agent( mode.sysname, 'StrataCentral' ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def bfnPlatform( mode, token ):
   pcd = phyConfigDir
   if pcd is None:
      pcdPath = f'hardware/cell/{Cell.cellId()}/phy/sfe/config'
      pcd = mode.sysdbRoot.entity[ pcdPath ]

   # If it is not hw platform return not-supported right away
   if not pcd.isSfeSecondaryForwardingAgent:
      return CliParser.guardNotThisPlatform

   # If it is hwPlatform, do BFN specific check
   return bfnGuard( mode, token )

def strataPlatform( mode, token ):
   pcd = phyConfigDir
   if pcd is None:
      pcdPath = f'hardware/cell/{Cell.cellId()}/phy/sfe/config'
      pcd = mode.sysdbRoot.entity[ pcdPath ]

   # If it is not hw platform return not-supported right away
   if not pcd.isSfeSecondaryForwardingAgent:
      return CliParser.guardNotThisPlatform

   # If it is hwPlatform, do Strata specific check
   return strataGuard( mode, token )

def sfeSecondaryPlatformWithNat( mode, token ):
   return bfnPlatform( mode, token ) and strataPlatform( mode, token )

# Ruby Dfw mode does not have FIB module, hence blocking CLIs related to FIB
def notRubyDfw( mode, token ):
   if veosConfigDir.platformRuby:
      return CliParser.guardNotThisPlatform
   else:
      return None

def isPlatformCblOrInd( mode, token ):
   if veosConfigDir.platformCaravan and veosConfigDir.platformHasNac:
      return None
   else:
      return CliParser.guardNotThisPlatform

matcherSfe = CliMatcher.KeywordMatcher( 'sfe',
                                        'Sfe (Software Forwarding Engine)' )
nodeSfe = CliCommand.Node( matcher=matcherSfe, guard=sfeGuard )
nodeSfeEos = CliCommand.Node( matcher=matcherSfe, guard=sfeEosGuard )

matcherStatus = CliCommand.guardedKeyword( 'status', 'Status', notHwPlatform )
matcherMemory = CliMatcher.KeywordMatcher( 'memory', 'Memory information' )
matcherPool = CliMatcher.KeywordMatcher( 'pool', 'Memory pool' )

def isSfeRunning( sysname ):
   # The SfeCentral agent will launch the SFE slice agents so it can gate
   # if we're a SFE system
   return bool( AgentDirectory.agent( sysname, 'SfeCentral' ) )

def isSfeAgentRunning( sysname, agentName ):
   if not agentName.startswith( "Sfe" ):
      return False
   return bool( AgentDirectory.agent( sysname, agentName ) )

def Plugin( em ):
   global entityManager
   entityManager = em
   global portVifStatus
   global sfePhyStatusDir
   global launcherConfig
   global v4RouteCacheHitbitStatus
   global phyConfigDir
   global bessCliConfig
   global veosConfigDir

   portVifStatus = LazyMount.mount( em, 'hardware/sfe/vif/status/port',
                                    'SfeVif::PortVifStatus', 'r' )
   # take sfe phystatus directly from the hardware tree
   sfePhyStatusDir = LazyMount.mount( em,
         'hardware/phy/status/sfe80Phy/slice/FixedSystem/SfeSlice',
         'SfeSlice::PhySfe80StatusDir', 'r' )

   v4RouteCacheHitbitStatus = SmashLazyMount.mount( entityManager,
         'sfe/routecache/status/ipv4',
         'RouteCache::Ipv4HitBitStatus',
         SmashLazyMount.mountInfo( 'reader' ) )

   phyConfigDir = LazyMount.mount( em,
         f'hardware/cell/{Cell.cellId()}/phy/sfe/config',
         'Hardware::Phy::SfePhyConfigDir', 'r' )

   bessCliConfig = ConfigMount.mount(
      entityManager, "bess/cli/config", "Sfe::BessdCliConfig", "wi" )

   launcherConfig = LazyMount.mount( em, 'hardware/sfe/launcherConfig',
                                     'Tac::Dir', 'ri' )

   veosConfigDir = LazyMount.mount( em, 'hardware/sfe/veosConfig',
                                    'Sfe::VeosConfig', 'r' )
