# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
import BasicCli
import LazyMount
import CliMatcher
import ShowCommand
import Tac
from CliPlugin.WanTEShowCli import (
            getVniAvtId,
            getVrfAvtName,
            getAvtName,
)
from CliPlugin.SfeInternetExitShowCliModel import (
            SfeIeExitKeyModel,
            SfeIeExitOptionStatusModel,
            SfeIeExitGroupStatusModel,
            SfeIeExitGroupInPolicyModel,
            SfeIePolicyStatusModel,
            SfeIePolicyShowModel,
)
from CliToken.AvtCliToken import (
            AvtVrfExprFactory,
            AvtNameExprFactory,
)

gv = CliGlobal.CliGlobal( dict( ieCliConfig=None,
                                iePolicyStatus=None,
                                ieExitGroupStatus=None,
                                ieExitOptionStatus=None,
                                siConnectionStatus=None ) )

ieMatcher = CliMatcher.KeywordMatcher(
   keyword='internet-exit',
   helpdesc='Internet exit',
   autoCompleteMinChars=8
)

policyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda _: gv.ieCliConfig.policyConfig, helpdesc='Name of the policy' )

vrfFilterHelpStr = 'Filter by VRF name'
avtFilterHelpStr = 'Filter by adaptive virtual topology name'

ExitType = Tac.Type( 'SfeInternetExit::ExitType' )
ExitResultType = Tac.Type( 'SfeInternetExit::ExitResultType' )
ExitVniAvtKey = Tac.Type( 'SfeInternetExit::ExitVniAvtKey' )
GroupVniAvtKey = Tac.Type( 'SfeInternetExit::GroupVniAvtKey' )
PolicyVniAvtKey = Tac.Type( 'SfeInternetExit::PolicyVniAvtKey' )
VniAndAvtId = Tac.Type( 'Avt::VniAndAvtId' )

typeToStringMap = {
   ExitType.unknown : 'unknown',
   ExitType.localExit : 'local',
   ExitType.fibDefault : 'fib-default',
   ExitType.remoteExit : 'remote',
}

def getIeExitKeyModel( vniExitKey ):
   exitKeyModel = SfeIeExitKeyModel()
   exitKeyModel.type = typeToStringMap[ vniExitKey.exitKey.type ]
   exitKeyModel.name = vniExitKey.exitKey.name
   return exitKeyModel

def getSfeIeExitOptionStatusModel( optionKey, exitOptionResult ):
   optionModel = SfeIeExitOptionStatusModel()
   optionModel.optionKey = getIeExitKeyModel( optionKey )
   optionModel.isActive = True
   if optionKey.exitKey.type == ExitType.localExit:
      optionModel.intf = exitOptionResult.intfId
      optionModel.vrf = exitOptionResult.vrf
   return optionModel

def getDetailedIeExitOptionStatusModel( optionKey ):
   ieOptionStatus = gv.ieExitOptionStatus.exitOptionStatus
   connectionStatus = gv.siConnectionStatus.connection
   optionModel = SfeIeExitOptionStatusModel()
   optionModel.optionKey = getIeExitKeyModel( optionKey )

   if optionKey in ieOptionStatus:
      optionModel.isActive = True
      exitOptionResult = ieOptionStatus[ optionKey ].resultExitIntf
      if exitOptionResult.resultType == ExitResultType.invalid:
         optionModel.isActive = False

      optionName = optionKey.exitKey.name
      if optionKey.exitKey.type == ExitType.localExit:
         if not exitOptionResult.intfId:
            # get the result from connection status if present
            optionModel.isActive = False
            if optionName in connectionStatus:
               exitOptionResult = connectionStatus[ optionName ].resultConnIntf
         if exitOptionResult.intfId:
            optionModel.intf = exitOptionResult.intfId
            optionModel.vrf = exitOptionResult.vrf

   return optionModel

def getSfeIeExitGroupStatusModel( groupKey ):
   groupModel = SfeIeExitGroupStatusModel()
   groupModel.groupName = groupKey.groupName
   # validate if the groupkey is present in exitGroupStatus
   if groupKey not in gv.ieExitGroupStatus.exitGroupStatus:
      return groupModel
   exitGroupStatus = gv.ieExitGroupStatus.exitGroupStatus[ groupKey ]
   # get the options list populated for the group
   for optionKey, exitOptionResult in exitGroupStatus.result.items():
      groupModel.exitOption.append(
         getSfeIeExitOptionStatusModel( optionKey, exitOptionResult ) )
   return groupModel

def getDetailedIeExitGroupStatusModel( groupKey ):
   groupModel = SfeIeExitGroupStatusModel()
   vniAvtId, groupName = groupKey.vniAvtId, groupKey.groupName
   groupModel.groupName = groupName

   egConfig = gv.ieCliConfig.exitGroupConfig.get( groupName )
   if egConfig is not None:
      for egOption in egConfig.exitOption:
         optionKey = ExitVniAvtKey( vniAvtId, egOption )
         groupModel.exitOption.append( getDetailedIeExitOptionStatusModel(
                                                                        optionKey ) )
   return groupModel

def updatePolicyModel( policyModel, policyKey ):
   policyModel.policyName = policyKey.policyName
   policyModel.vniId = ( policyKey.vniAvtId >> 16 )
   policyModel.avtId = ( policyKey.vniAvtId & 0xFFFF )
   policyModel.vrfName, policyModel.avtName = getVrfAvtName(
                                          policyModel.vniId, policyModel.avtId )
   if policyModel.vniId == 0:
      policyModel.vrfName = 'default( underlay )'
      policyModel.avtName = getAvtName( 'default', policyModel.avtId )

def getIePolicyStatusModel( policyKey ):
   policyModel = SfeIePolicyStatusModel()
   updatePolicyModel( policyModel, policyKey )
   # we know for sure that the policyStatus is present for the policyKey and
   # there is a cliConfig for the policy
   policyStatus = gv.iePolicyStatus.policyStatus[ policyKey ]
   winningExitGroup = policyStatus.winningExitGroup
   if winningExitGroup.groupName in gv.ieCliConfig.exitGroupConfig:
      exitGroupInPolicy = SfeIeExitGroupInPolicyModel()
      exitGroupInPolicy.isWinning = True
      exitGroupInPolicy.exitGroup = getSfeIeExitGroupStatusModel(
                                                               winningExitGroup )
      policyModel.exitGroups.append( exitGroupInPolicy )
   return policyModel

def getDetailedIePolicyStatusModel( policyKey ):
   iePolicyConfig = gv.ieCliConfig.policyConfig
   ieExitGroupStatus = gv.ieExitGroupStatus.exitGroupStatus
   policyModel = SfeIePolicyStatusModel()
   updatePolicyModel( policyModel, policyKey )
   vniAvtId, polName = policyKey.vniAvtId, policyKey.policyName

   for egCfg in iePolicyConfig[ polName ].exitGroup.values():
      egName = egCfg.exitGroupName
      egKey = GroupVniAvtKey( vniAvtId, egName )
      exitGroupInPolicy = SfeIeExitGroupInPolicyModel()
      if egKey in ieExitGroupStatus:
         policyStatus = gv.iePolicyStatus.policyStatus[ policyKey ]
         exitGroupInPolicy.isWinning = ( egName ==
                                         policyStatus.winningExitGroup.groupName )
      else:
         exitGroupInPolicy.isWinning = False
      exitGroupInPolicy.exitGroup = getDetailedIeExitGroupStatusModel( egKey )
      policyModel.exitGroups.append( exitGroupInPolicy )
   return policyModel

def appendPolicyStatusInShowModel( showModel, inputPolKey, detail ):
   iePolicyStatus = gv.iePolicyStatus.policyStatus
   policyStatusModel = showModel.policyStatus
   updMethod = getDetailedIePolicyStatusModel if detail else getIePolicyStatusModel
   if inputPolKey.vniAvtId:
      if inputPolKey in iePolicyStatus:
         policyStatusModel.append( updMethod( inputPolKey ) )
   else:
      for polKey in iePolicyStatus:
         if polKey.policyName == inputPolKey.policyName:
            policyStatusModel.append( updMethod( polKey ) )

# ---------------------------------------------------------------------------------
# The 'show internet-exit policy [ POLICY [ vrf VRF avt AVT ] [ detail ]' command
# ---------------------------------------------------------------------------------
def getSfeIePolicyShowModel( mode, args ):
   showModel = SfeIePolicyShowModel()
   iePolicyConfig = gv.ieCliConfig.policyConfig

   # get a list of policyNames for which status needs to be displayed
   argPolicy = args.get( 'POLICY', '' )
   if argPolicy:
      if argPolicy not in iePolicyConfig:
         return showModel
      policyNames = [ argPolicy ]
   else:
      policyNames = iePolicyConfig

   detail = 'detail' in args
   # get the vniAndAvtId if filter is provided
   vrf = args.get( 'VRF' )
   if vrf:
      vniId, avtId = getVniAvtId( mode, vrf, args[ 'AVT' ], verifyVni=False )
      if ( ( vniId is None and vrf != 'default' ) or ( avtId is None ) ):
         return showModel
      if vrf == 'default':
         for policyVni in [ 0, vniId ]:
            # if 'default' vrf is not configured under vxlan, the vniId will be None
            if policyVni is None:
               continue
            vniAvtId = VniAndAvtId.makeVniAndAvtId( policyVni, avtId )
            policyKey = PolicyVniAvtKey( vniAvtId, policyNames[ 0 ] )
            appendPolicyStatusInShowModel( showModel, policyKey, detail )
         return showModel
   else:
      # filter is not specified
      vniId, avtId = ( 0, 0 )

   vniAvtId = VniAndAvtId.makeVniAndAvtId( vniId, avtId )
   # append the policyStatus into the show model
   for policyName in policyNames:
      policyKey = PolicyVniAvtKey( vniAvtId, policyName )
      appendPolicyStatusInShowModel( showModel, policyKey, detail )
   return showModel

class SfeIePolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show internet-exit policy [ POLICY [ VRF AVT ] ] [ detail ]'
   data = {
      'internet-exit' : ieMatcher,
      'policy' : 'Show Internet exit policy information',
      'POLICY' : policyNameMatcher,
      'VRF' : AvtVrfExprFactory( vrfFilterHelpStr ),
      'AVT' : AvtNameExprFactory( avtFilterHelpStr ),
      'detail' : 'Detailed policy information',
   }
   cliModel = SfeIePolicyShowModel
   handler = getSfeIePolicyShowModel

BasicCli.addShowCommandClass( SfeIePolicyCmd )

# ---------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# ---------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.ieCliConfig = LazyMount.mount( entityManager,
                                 'ie/cli/config',
                                 'SfeInternetExit::InternetExitCliConfig', 'r' )
   gv.iePolicyStatus = LazyMount.mount( entityManager,
                                 'ie/status/policy',
                                 'SfeInternetExit::IePolicyDir', 'r' )
   gv.ieExitGroupStatus = LazyMount.mount( entityManager,
                                 'ie/status/exitGroup',
                                 'SfeInternetExit::IeExitGroupDir', 'r' )
   gv.ieExitOptionStatus = LazyMount.mount( entityManager,
                                 'ie/status/exitOption',
                                 'SfeInternetExit::IeExitOptionDir', 'r' )
   gv.siConnectionStatus = LazyMount.mount( entityManager,
                                 'si/status/connection',
                                 'SfeServiceInsertion::SiConnectionStatusDir', 'r' )
