# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Str, List, Bool, Int

def printFormatted( line, leadingSpaces, sep='' ):
   spaces = ' ' * leadingSpaces
   print( spaces, line, sep=sep )

def displayExitOption( exitOptionStatus, leadingSpaces=0 ):
   exitName = exitOptionStatus.optionKey.name
   exitStr = "Exit: "
   # if there is no name for the exit, don't print extra spaces
   if exitName:
      exitStr += exitName + " "
   exitStr += "(" + exitOptionStatus.optionKey.type + ")"
   printFormatted( exitStr, leadingSpaces )
   leadingSpaces += 2
   isActive = "yes" if exitOptionStatus.isActive else "no"
   printFormatted( f"Active: {isActive}", leadingSpaces )
   exitOptionType = exitOptionStatus.optionKey.type
   if exitOptionType == "local":
      if exitOptionStatus.intf:
         printFormatted( f"Interface: {exitOptionStatus.intf}", leadingSpaces )
         printFormatted( f"VRF: {exitOptionStatus.vrf}", leadingSpaces )

def displayExitGroup( exitGroupStatus, isWinning, leadingSpaces=0 ):
   printFormatted( f"Group: {exitGroupStatus.groupName}", leadingSpaces )
   isWinningGroup = "yes" if isWinning else "no"
   printFormatted( f"Winning: {isWinningGroup}", leadingSpaces + 2 )
   for exitOption in exitGroupStatus.exitOption:
      displayExitOption( exitOption, leadingSpaces + 2 )

def displayPolicy( policyStatus, leadingSpaces=0 ):
   printFormatted( f"Policy: {policyStatus.policyName}", leadingSpaces )
   printFormatted( f"VRF: {policyStatus.vrfName}, VNI: {policyStatus.vniId}",
                   leadingSpaces + 2 )
   printFormatted( f"AVT: {policyStatus.avtName}, AVT ID: {policyStatus.avtId}",
                   leadingSpaces + 2 )
   for egInPolicy in policyStatus.exitGroups:
      displayExitGroup( egInPolicy.exitGroup, egInPolicy.isWinning,
                        leadingSpaces + 2 )

class SfeIeExitKeyModel( Model ):
   type = Str( help="Type of the exit" )
   name = Str( help="Name of the exit" )

class SfeIeExitOptionStatusModel( Model ):
   optionKey = Submodel( valueType=SfeIeExitKeyModel,
                         help="Composite key of the exit option" )
   isActive = Bool( help="Exit option is active" )
   intf = Str( help="Result interface of the connection", optional=True )
   vrf = Str( help="VRF of the result interface", optional=True )

class SfeIeExitGroupStatusModel( Model ):
   groupName = Str( help="Name of the exit group" )
   exitOption = List( valueType=SfeIeExitOptionStatusModel,
                      help="Collection of exit options under this exit group" )

class SfeIeExitGroupInPolicyModel( Model ):
   isWinning = Bool( help="This is the winning exit group", default=False )
   exitGroup = Submodel( valueType=SfeIeExitGroupStatusModel,
                         help="Exit group model" )

class SfeIePolicyStatusModel( Model ):
   policyName = Str( help="Name of the policy" )
   vrfName = Str( help="VRF name" )
   vniId = Int( help="VNI identifier" )
   avtName = Str( help="AVT name" )
   avtId = Int( help="AVT identifier" )
   # This list stores only the winning exit group and we use it because in future we
   # will support showing all exitGroups under the policy.
   exitGroups = List( valueType=SfeIeExitGroupInPolicyModel,
                      help="Exit groups configured for this policy" )

# model for 'show internet-exit policy POLICY' command
class SfeIePolicyShowModel( Model ):
   policyStatus = List( valueType=SfeIePolicyStatusModel,
                        help="SFE Internet exit policy status" )

   def render( self ):
      for policy in self.policyStatus:
         displayPolicy( policy )
