#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Int, Model, Dict, Str, Enum, Float, List
from CliPlugin import SfeIpsecShowCliLib
from TableOutput import createTable, Format
import ipaddress
import Tac
from collections import defaultdict
from socket import ntohl
from Ark import timestampToStr, switchTimeToUtc
from ArnetModel import IpGenericAddress
from Arnet import IpGenAddr

IpsecHmacAlgorithmEnum = Tac.Type( "Ipsec::IpsecHmacAlgorithm" )
IpsecEspAlgorithmEnum = Tac.Type( "Ipsec::IpsecEspAlgorithm" )

# Maps used to specify strings that are output in the show commands
IPSEC_AUTH_SHOW_MAP = { IpsecHmacAlgorithmEnum.md5 : 'md5',
                        IpsecHmacAlgorithmEnum.sha1 : 'sha1',
                        IpsecHmacAlgorithmEnum.sha256 : 'sha256',
                        IpsecHmacAlgorithmEnum.sha384 : 'sha384',
                        IpsecHmacAlgorithmEnum.sha512 : 'sha512',
                        IpsecHmacAlgorithmEnum.nullhash : 'null',
                        'unknownAuthentication' : 'unknown', }

IPSEC_ENCR_SHOW_MAP = { IpsecEspAlgorithmEnum.des : 'des',
                        IpsecEspAlgorithmEnum.aes128 : 'aes128',
                        IpsecEspAlgorithmEnum.aes256 : 'aes256',
                        IpsecEspAlgorithmEnum.aes128gcm64 : 'aes128gcm64',
                        IpsecEspAlgorithmEnum.aes128gcm128 : 'aes128gcm128',
                        IpsecEspAlgorithmEnum.aes256gcm128 : 'aes256gcm128',
                        IpsecEspAlgorithmEnum.nullesp : 'null',
                        'unknownEncryption' : 'unknown', }

IPSEC_REKEY_SHOW_MAP = { 'inactive' : 'n/a',
                         'inProgress' : 'in progress', }

IPSEC_PROGRAMMED_LOCATION_MAP = { 'softwareOnly' : 'software',
                                  'softwareAndHardware' : 'software, hardware', }

# Enums taken from rte_crypto_sym.h
# Some of the fields in the SA information we get is set using those (original) enums
class RteCryptoEncryptionType:
   RTE_CRYPTO_CIPHER_NULL = 1
   RTE_CRYPTO_AEAD_AES_GCM = 2
   RTE_CRYPTO_CIPHER_AES_CBC = 5

class RteCryptoAuthAlgorithm:
   RTE_CRYPTO_AUTH_NULL = 1
   RTE_CRYPTO_AUTH_MD5_HMAC = 8
   RTE_CRYPTO_AUTH_SHA1_HMAC = 10
   RTE_CRYPTO_AUTH_SHA256_HMAC = 14
   RTE_CRYPTO_AUTH_SHA384_HMAC = 16
   RTE_CRYPTO_AUTH_SHA512_HMAC = 18

def getEncAlgorithmEnum( cipherAlgo, aeadAlgo, keyLen, icvLen ):
   if cipherAlgo == RteCryptoEncryptionType.RTE_CRYPTO_CIPHER_NULL:
      return IpsecEspAlgorithmEnum.nullesp

   if cipherAlgo == RteCryptoEncryptionType.RTE_CRYPTO_CIPHER_AES_CBC:
      if keyLen == 16:
         return IpsecEspAlgorithmEnum.aes128

      if keyLen == 32:
         return IpsecEspAlgorithmEnum.aes256

   # RTE_CRYPTO_CIPHER_3DES_CBC has an original value of 2 (the same as
   # RTE_CRYPTO_AEAD_AES_GCM). So, we use the icvLen to distinguish between
   # the two ciphers.
   if aeadAlgo == RteCryptoEncryptionType.RTE_CRYPTO_AEAD_AES_GCM:
      if icvLen == 0:
         return IpsecEspAlgorithmEnum.des

      if keyLen == 16:
         if icvLen == 8:
            return IpsecEspAlgorithmEnum.aes128gcm64
         if icvLen == 16:
            return IpsecEspAlgorithmEnum.aes128gcm128

      if keyLen == 32:
         if icvLen == 16:
            return IpsecEspAlgorithmEnum.aes256gcm128
         if icvLen == 8:
            return IpsecEspAlgorithmEnum.aes256gcm64

   return 'unknownEncryption'

def getAuthAlgorithmEnum( authAlgo, aeadAlgo ):
   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_NULL:
      return IpsecHmacAlgorithmEnum.nullhash

   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_SHA1_HMAC:
      return IpsecHmacAlgorithmEnum.sha1

   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_SHA256_HMAC:
      return IpsecHmacAlgorithmEnum.sha256

   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_SHA384_HMAC:
      return IpsecHmacAlgorithmEnum.sha384

   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_SHA512_HMAC:
      return IpsecHmacAlgorithmEnum.sha512

   if authAlgo == RteCryptoAuthAlgorithm.RTE_CRYPTO_AUTH_MD5_HMAC:
      return IpsecHmacAlgorithmEnum.md5

   if aeadAlgo != 0:
      return None

   return 'unknownAuthentication'

def getSrcIpAddrGenFromSaStats( statsEntry ):
   addrTemp = ipaddress.ip_address( statsEntry[ 'srcAddr' ] )
   return IpGenAddr( str( addrTemp ) )

def getDstIpAddrGenFromSaStats( statsEntry ):
   addrTemp = ipaddress.ip_address( statsEntry[ 'dstAddr' ] )
   return IpGenAddr( str( addrTemp ) )

def getSrcIpAddrGenFromSaState( stateEntry ):
   addrTemp = ipaddress.ip_address( ntohl( stateEntry[ 'srcAddr' ] ) )
   return IpGenAddr( str( addrTemp ) )

def getDstIpAddrGenFromSaState( stateEntry ):
   addrTemp = ipaddress.ip_address( ntohl( stateEntry[ 'dstAddr' ] ) )
   return IpGenAddr( str( addrTemp ) )

def getMatchingStateIdx( stats, cryptoEntry, spiToIdx ):
   # Use SPI, source and destination address to match stats entry with
   # correct state entry
   if cryptoEntry[ 'spi' ] not in spiToIdx:
      print( f"Couldn't find stats info for SA with SPI {cryptoEntry[ 'spi' ]}" )
      return None

   cryptoEntrySrcAddr = getSrcIpAddrGenFromSaState( cryptoEntry )
   cryptoEntryDstAddr = getDstIpAddrGenFromSaState( cryptoEntry )

   for i in spiToIdx[ cryptoEntry[ 'spi' ] ]:
      # Use SPI, source and destination address to match stats entry with
      # correct state entry
      if stats[ i ].saDirection != cryptoEntry[ 'saDirection' ]:
         continue
      if stats[ i ].srcAddr != cryptoEntrySrcAddr:
         continue
      if stats[ i ].dstAddr != cryptoEntryDstAddr:
         continue
      # Found the matching SA
      return i

   print( "Couldn't find stats info for SA with source address: "
          f"{cryptoEntrySrcAddr} and destination address: {cryptoEntryDstAddr}" )
   return None

def sharedCountersTableRender( counters ):
   if not counters:
      return

   table = createTable( ( "Source", "Destination", "Direction", "SPI", "Encap ID",
                          "Packets", "Bytes" ), tableWidth=120 )
   formatLeft = Format( justify="left" )
   formatLeft.noPadLeftIs( True )
   formatRight = Format( justify="right" )
   formatRight.noTrailingSpaceIs( True )
   table.formatColumns( formatLeft, formatLeft, formatLeft, formatRight,
                        formatRight, formatRight, formatRight )
   for row in counters:
      data = counters[ row ]
      table.newRow( data.srcAddr, data.dstAddr, data.saDirection, hex( data.spi ),
                    data.encapId, data.numPackets, data.numBytes )
   output = table.output()
   print( output )

# Model functions for "show platform sfe ipsec counters"
class SfeIpsecCounters( Model ):
   __public__ = False
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   encapId = Int( help="Encapsulation ID" )
   spi = Int( help="SPI value" )
   saDirection = Enum( values=( "egress", "ingress" ), help="SA direction" )
   numPackets = Int( help="Number of packets" )
   numBytes = Int( help="Number of bytes" )

class SfeIpsecShowCountersModel( Model ):
   __public__ = False
   """ Represents ipsecShowCounters """
   counters = Dict( keyType=int, valueType=SfeIpsecCounters,
                 help="A mapping of counter names to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      for row in data:
         cntrs = SfeIpsecCounters()
         ctrEntry = data[ row ]
         cntrs.srcAddr = getSrcIpAddrGenFromSaStats( ctrEntry )
         cntrs.dstAddr = getDstIpAddrGenFromSaStats( ctrEntry )
         cntrs.encapId = ctrEntry[ 'encapId' ]
         cntrs.spi = ctrEntry[ 'spi' ]
         cntrs.saDirection = ctrEntry[ 'saDirection' ]
         cntrs.numPackets = ctrEntry[ 'numPackets' ] + ctrEntry[ 'hwCryptoPkts' ]
         cntrs.numBytes = ctrEntry[ 'numBytes' ] + ctrEntry[ 'hwCryptoBytes' ]
         self.counters[ index ] = cntrs
         index = index + 1

   def render( self ):
      sharedCountersTableRender( self.counters )

# Model functions for "show platform sfe ipsec counters software"
class SfeIpsecShowCountersSwModel( Model ):
   __public__ = False
   """ Represents ipsecShowCounters Sw """
   counters = Dict( keyType=int, valueType=SfeIpsecCounters,
                    help="A mapping of software counter names to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      for row in data:
         cntrs = SfeIpsecCounters()
         ctrEntry = data[ row ]
         cntrs.srcAddr = getSrcIpAddrGenFromSaStats( ctrEntry )
         cntrs.dstAddr = getDstIpAddrGenFromSaStats( ctrEntry )
         cntrs.encapId = ctrEntry[ 'encapId' ]
         cntrs.spi = ctrEntry[ 'spi' ]
         cntrs.saDirection = ctrEntry[ 'saDirection' ]
         cntrs.numPackets = ctrEntry[ 'numPackets' ]
         cntrs.numBytes = ctrEntry[ 'numBytes' ]
         self.counters[ index ] = cntrs
         index = index + 1

   def render( self ):
      sharedCountersTableRender( self.counters )

# Model functions for "show platform sfe ipsec counters hardware"
class SfeIpsecShowCountersHwModel( Model ):
   __public__ = False
   """ Represents ipsecShowCounters Hw """
   counters = Dict( keyType=int, valueType=SfeIpsecCounters,
                    help="A mapping of hardware counter names to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      for row in data:
         cntrs = SfeIpsecCounters()
         ctrEntry = data[ row ]
         cntrs.srcAddr = getSrcIpAddrGenFromSaStats( ctrEntry )
         cntrs.dstAddr = getDstIpAddrGenFromSaStats( ctrEntry )
         cntrs.encapId = ctrEntry[ 'encapId' ]
         cntrs.spi = ctrEntry[ 'spi' ]
         cntrs.saDirection = ctrEntry[ 'saDirection' ]
         cntrs.numPackets = ctrEntry[ 'hwCryptoPkts' ]
         cntrs.numBytes = ctrEntry[ 'hwCryptoBytes' ]
         self.counters[ index ] = cntrs
         index = index + 1

   def render( self ):
      sharedCountersTableRender( self.counters )

# Model functions for "show platform sfe ipsec counters detail"
class SfeIpsecCountersDetail( Model ):
   __public__ = False
   spi = Int( help='SPI value' )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   saDirection = Enum( values=( "egress", "ingress" ), help="SA direction" )
   encapId = Int( help="Encapsulation ID" )
   numPackets = Int( help="Number of packets" )
   numBytes = Int( help="Number of bytes" )
   replayError = Int( help="Number of replay errors" )
   integrityError = Int( help="Number of integrity errors" )
   udpEncapError = Int( help="Number of UDP encap errors" )
   rolloverError = Int( help="Number of rollover errors" )
   numHwDecryptedPkts = Int( help="Number of hardware decrypted packets" )
   numHwDecryptedBytes = Int( help="Number of hardware decrypted bytes" )
   numHwDecryptionDone = \
      Int( help="Number of successful hardware decryption operations" )
   numHwDecryptionFails = \
      Int( help="Number of hardware decryption operation failures" )

class SfeIpsecShowCountersDetModel( Model ):
   __public__ = False
   """ Represents ipsecShowCounters detail """
   counters = Dict( keyType=int, valueType=SfeIpsecCountersDetail,
                    help="A mapping of detail counter names to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      for row in data:
         cntrs = SfeIpsecCountersDetail()
         cntrEntry = data[ row ]
         cntrs.spi = cntrEntry[ 'spi' ]
         cntrs.srcAddr = getSrcIpAddrGenFromSaStats( cntrEntry )
         cntrs.dstAddr = getDstIpAddrGenFromSaStats( cntrEntry )
         cntrs.encapId = cntrEntry[ 'encapId' ]
         cntrs.saDirection = cntrEntry[ 'saDirection' ]
         cntrs.numPackets = cntrEntry[ 'numPackets' ]
         cntrs.numBytes = cntrEntry[ 'numBytes' ]
         cntrs.replayError = cntrEntry[ 'replayError' ]
         cntrs.integrityError = cntrEntry[ 'integrityError' ]
         cntrs.udpEncapError = cntrEntry[ 'udpEncapError' ]
         cntrs.rolloverError = cntrEntry[ 'rolloverError' ]
         cntrs.numHwDecryptedPkts = cntrEntry[ 'hwCryptoPkts' ]
         cntrs.numHwDecryptedBytes = cntrEntry[ 'hwCryptoBytes' ]
         cntrs.numHwDecryptionDone = cntrEntry[ 'hwCryptoDone' ]
         cntrs.numHwDecryptionFails = cntrEntry[ 'hwCryptoErrorPkts' ]
         self.counters[ index ] = cntrs
         index = index + 1

   def render( self ):
      if not self.counters:
         return
      for row in self.counters:
         cntrs = self.counters[ row ]
         print( "SPI:", hex( cntrs.spi ) )
         print( "Source:", cntrs.srcAddr )
         print( "Destination:", cntrs.dstAddr )
         print( "Direction:", cntrs.saDirection )
         print( "Encap ID:", cntrs.encapId )
         print( "Software processed packets:", cntrs.numPackets )
         print( "Software processed bytes:", cntrs.numBytes )
         print( "Hardware decrypted packets:", cntrs.numHwDecryptedPkts )
         print( "Hardware decrypted bytes:", cntrs.numHwDecryptedBytes )
         print( "Hardware decryptions succeeded:", cntrs.numHwDecryptionDone )
         print( "Hardware decryptions failed:", cntrs.numHwDecryptionFails )
         print( "Replay errors:", cntrs.replayError )
         print( "Integrity errors:", cntrs.integrityError )
         print( "UDP encap errors:", cntrs.udpEncapError )
         print( "Rollover errors:", cntrs.rolloverError )
         print( "" )

# Model functions for "show platform sfe ipsec counters error"
class SfeIpsecCountersErrorOutputGlobal( Model ):
   __public__ = False
   spiInvalid = Int( help='Invalid SPI error' )
   packetsDropped = Int( help='Number of dropped packets' )
   inCryptoQueueErrors = Int( help='Errors in packet decryption' )
   outCryptoQueueErrors = Int( help='Errors in packet encryption' )
   replayDropped = Int( help='Anti replay drops' )
   natFail = Int( help='NATT failures' )
   encapInvalid = Int( help='Invalid encapsulation' )
   saLookupError = Int( help='SA lookup error' )
   flowCacheError = Int( help='Flow cache error' )
   badProtocol = Int( help='Bad protocol' )
   inEspInvalid = Int( help='Ingress processing ESP invalid error' )
   outEspInvalid = Int( help='Egress processing ESP invalid error' )
   hwCryptoError = Int( help='Hardware crypto error' )

class SfeIpsecCountersErrorOutputSA( Model ):
   __public__ = False
   spi = Int( help='SPI value' )
   replayDropped = Int( help='Anti replay drops' )
   udpError = Int( help='UDP drops' )
   hwCryptoErrorPkts = Int( help='Hardware crypto error drops' )
   rolloverError = Int( help="Rollover errors" )

class SfeIpsecShowCountersErrModel( Model ):
   __public__ = False
   """ Represents ipsecShowCounters error """
   counters = Dict( keyType=int, valueType=SfeIpsecCountersErrorOutputGlobal,
                    help="A mapping of global drop counter names to values" )
   spiCounters = Dict( keyType=int, valueType=SfeIpsecCountersErrorOutputSA,
                    help="A mapping of per SPI drop counter names to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      cntrs = SfeIpsecCountersErrorOutputGlobal()
      ctrEntry = data[ index ]
      cntrs.spiInvalid = ctrEntry[ 'spiInvalid' ]
      cntrs.packetsDropped = ctrEntry[ 'packetsDropped' ]
      cntrs.replayDropped = ctrEntry[ 'replayDropped' ]
      cntrs.natFail = ctrEntry[ 'natFail' ]
      cntrs.encapInvalid = ctrEntry[ 'encapInvalid' ]
      cntrs.saLookupError = ctrEntry[ 'saLookupError' ]
      cntrs.flowCacheError = ctrEntry[ 'flowCacheError' ]
      cntrs.inCryptoQueueErrors = ctrEntry[ 'inCryptoQueueErrors' ]
      cntrs.outCryptoQueueErrors = ctrEntry[ 'outCryptoQueueErrors' ]
      cntrs.badProtocol = ctrEntry[ 'badProtocol' ]
      cntrs.inEspInvalid = ctrEntry[ 'inEspInvalid' ]
      cntrs.outEspInvalid = ctrEntry[ 'outEspInvalid' ]
      cntrs.hwCryptoError = ctrEntry[ 'hwCryptoError' ]
      self.counters[ index ] = cntrs
      index += 1

      spiCntrEntry = data[ index ]
      entry = 0
      for row in spiCntrEntry:
         cntrsT = SfeIpsecCountersErrorOutputSA()
         spiData = spiCntrEntry[ row ]
         cntrsT.spi = spiData[ 'spi' ]
         cntrsT.replayDropped = spiData[ 'replayError' ]
         cntrsT.udpError = spiData[ 'udpEncapError' ]
         cntrsT.rolloverError = spiData[ 'rolloverError' ]
         cntrsT.hwCryptoErrorPkts = spiData[ 'hwCryptoErrorPkts' ]
         self.spiCounters[ entry ] = cntrsT
         entry += 1

   def render( self ):
      if not self.counters:
         return
      cntrs = self.counters[ 0 ]
      print( "Global stats:" )
      print( "Ingress invalid SPI errors:", cntrs.spiInvalid )
      print( "Ingress processing errors:", cntrs.inEspInvalid )
      print( "Egress processing errors:", cntrs.outEspInvalid )
      print( "Ingress crypto queue errors:", cntrs.inCryptoQueueErrors )
      print( "Egress crypto queue errors:", cntrs.outCryptoQueueErrors )
      print( "Invalid encapsulation ID errors:", cntrs.encapInvalid )
      print( "Egress SA lookup errors:", cntrs.saLookupError )
      print( "NAT encapsulation errors:", cntrs.natFail )
      print( "Replay drop errors:", cntrs.replayDropped )
      print( "Ingress SA processing errors:", cntrs.packetsDropped )
      print( "Flow cache errors:", cntrs.flowCacheError )
      print( "Inner protocol errors:", cntrs.badProtocol )
      print( "Hardware decryption errors:", cntrs.hwCryptoError )
      print( "" )

      if not self.spiCounters:
         return
      table = createTable( ( "SPI", "Replay", "Rollover", "UDP Encap",
                             "Hardware Crypto" ), tableWidth=120 )
      formatRight = Format( justify="right" )
      formatRight.noTrailingSpaceIs( True )
      table.formatColumns( formatRight, formatRight, formatRight, formatRight,
                           formatRight )
      for row in self.spiCounters:
         data = self.spiCounters[ row ]
         table.newRow( hex( data.spi ), data.replayDropped, data.rolloverError,
                       data.udpError, data.hwCryptoErrorPkts )
      output = table.output()
      print( output )

# Model functions for "show platform sfe ipsec state"
class SfeIpsecState( Model ):
   __public__ = False
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   encapId = Int( help="Encapsulation ID" )
   spi = Int( help="SPI value" )
   saDirection = Enum( values=( "egress", "ingress" ), help="SA direction" )
   encryption = Enum( values=IPSEC_ENCR_SHOW_MAP, help="Encryption algorithm" )
   authentication = Enum( values=IPSEC_AUTH_SHOW_MAP, help="Authentication type",
                          optional=True )

# Model functions for "show platform sfe ipsec state detail"
class SfeIpsecStateDetail( SfeIpsecState ):
   __public__ = False
   replayWindow = Int( help="Replay window size" )
   addTime = Float( help="Time at which SA was added" )
   useTime = Float( help="Time when SA was last used" )
   digestLen = Enum( values=( '0', '8', '12', '16' ), help="Digest length in bytes" )
   ivLen = Enum( values=( '0', '8', '16' ), help="IV length in bytes" )
   blockSize = Enum( values=( '4', '16' ), help="Block size in bytes" )
   isRekey = Enum( values=IPSEC_REKEY_SHOW_MAP, help="Status of rekey operation" )
   programmedLocation = Enum( values=IPSEC_PROGRAMMED_LOCATION_MAP,
                              help="Location where SA is programmed" )
   hwPorts = List( valueType=str, help="List of ports onto which the SA is ' \
                   'programmed", optional=True )

# Model functions for "show platform sfe ipsec encapsulation"
class SfeIpsecStateEncapsulation( Model ):
   __public__ = False
   encapId = Int( help="Encapsulation id" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ingressSpiList = Str( help="Ingress spi list" )
   egressSpiList = Str( help="Egress spi list" )

class SfeIpsecShowStateModel( Model ):
   __public__ = False
   """ Represents ipsecShowState """
   states = Dict( keyType=int, valueType=SfeIpsecState,
                  help="A mapping of states to values" )

   def setAttrsFromDict( self, data ):
      index = 0
      statsData = data[ 0 ]
      cryptoData = data[ 1 ]
      spiToIdx = defaultdict( list )
      for row in statsData:
         state = SfeIpsecState()
         statsEntry = statsData[ row ]
         state.spi = statsEntry[ 'spi' ]
         spiToIdx[ state.spi ].append( index )
         state.srcAddr = getSrcIpAddrGenFromSaStats( statsEntry )
         state.dstAddr = getDstIpAddrGenFromSaStats( statsEntry )
         state.encapId = statsEntry[ 'encapId' ]
         state.saDirection = statsEntry[ 'saDirection' ]
         state.encryption = 'unknownEncryption'
         state.authentication = None
         self.states[ index ] = state
         index = index + 1

      for row in cryptoData:
         cryptoEntry = cryptoData[ row ]
         i = getMatchingStateIdx( self.states, cryptoEntry, spiToIdx )
         if i is None:
            continue
         cryptoAlgo = getEncAlgorithmEnum( cryptoEntry[ 'cipherAlgorithmDec' ],
                                           cryptoEntry[ 'aeadAlgorithmDec' ],
                                           cryptoEntry[ 'cipherKeyLen' ],
                                           cryptoEntry[ 'saDigestLen' ] )
         self.states[ i ].encryption = cryptoAlgo
         authAlgo = getAuthAlgorithmEnum( cryptoEntry[ 'authAlgorithmDec' ],
                                          cryptoEntry[ 'aeadAlgorithmDec' ] )
         if authAlgo:
            self.states[ i ].authentication = authAlgo

   def render( self ):
      if not self.states:
         return
      table = createTable( ( "Source", "Destination", "SPI", "Encap ID", "Direction",
                             "Encryption", "Authentication" ), tableWidth=120 )
      formatLeft = Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = Format( justify="right" )
      formatRight.noTrailingSpaceIs( True )
      table.formatColumns( formatLeft, formatLeft, formatRight,
                                    formatRight, formatLeft, formatLeft,
                                    formatLeft )
      for row in self.states:
         data = self.states[ row ]
         table.newRow( data.srcAddr, data.dstAddr, hex( data.spi ), data.encapId,
                       data.saDirection, IPSEC_ENCR_SHOW_MAP[ data.encryption ],
                       IPSEC_AUTH_SHOW_MAP.get( data.authentication, 'n/a' ) )
      output = table.output()
      print( output )
      return

class SfeIpsecShowStateDetailModel( Model ):
   __public__ = False
   """ Represents ipsecShowStateDetail"""
   states = Dict( keyType=int, valueType=SfeIpsecStateDetail,
                  help="A mapping of states to values" )

   def setHwPortsFromBitmask( self, portBitmask, state ):
      bitNum = 0
      bitVal = 1
      portIntfMap = SfeIpsecShowCliLib.getNaciesPortIntfMap()
      while portBitmask >= bitVal:
         if portBitmask & bitVal:
            state.programmedLocation = 'softwareAndHardware'
            if portIntfMap:
               portIntf = portIntfMap.get( bitNum )
               if portIntf:
                  state.hwPorts.append( portIntf.intfId )
         bitNum += 1
         bitVal = 1 << bitNum

   def setAttrsFromDict( self, data ):
      index = 0
      statsData = data[ 0 ]
      cryptoData = data[ 1 ]
      spiToIdx = defaultdict( list )
      for row in statsData:
         state = SfeIpsecStateDetail()
         statsEntry = statsData[ row ]
         state.spi = statsEntry[ 'spi' ]
         spiToIdx[ state.spi ].append( index )
         state.srcAddr = getSrcIpAddrGenFromSaStats( statsEntry )
         state.dstAddr = getDstIpAddrGenFromSaStats( statsEntry )
         state.encapId = statsEntry[ 'encapId' ]
         state.saDirection = statsEntry[ 'saDirection' ]
         state.encryption = 'unknownEncryption'
         state.authentication = None
         state.addTime = float( statsEntry[ 'addTime' ] )
         state.useTime = 0.0
         if statsEntry[ 'useTime' ]:
            state.useTime = switchTimeToUtc( float( statsEntry[ 'useTime' ] ) )
         self.states[ index ] = state
         index = index + 1

      for row in cryptoData:
         cryptoEntry = cryptoData[ row ]
         i = getMatchingStateIdx( self.states, cryptoEntry, spiToIdx )
         if i is None:
            continue
         cryptoAlgo = getEncAlgorithmEnum( cryptoEntry[ 'cipherAlgorithmDec' ],
                                           cryptoEntry[ 'aeadAlgorithmDec' ],
                                           cryptoEntry[ 'cipherKeyLen' ],
                                           cryptoEntry[ 'saDigestLen' ] )
         self.states[ i ].encryption = cryptoAlgo
         authAlgo = getAuthAlgorithmEnum( cryptoEntry[ 'authAlgorithmDec' ],
                                          cryptoEntry[ 'aeadAlgorithmDec' ] )
         if authAlgo:
            self.states[ i ].authentication = authAlgo
         self.states[ i ].digestLen = str( cryptoEntry[ 'saDigestLen' ] )
         self.states[ i ].ivLen = str( cryptoEntry[ 'saIvLen' ] )
         self.states[ i ].isRekey = cryptoEntry[ 'isRekey' ]
         self.states[ i ].replayWindow = cryptoEntry[ 'replayWindow' ]
         self.states[ i ].blockSize = str( cryptoEntry[ 'blockSize' ] )
         self.states[ i ].programmedLocation = 'softwareOnly'
         portBitmask = cryptoEntry[ 'portsProgrammed' ]
         self.setHwPortsFromBitmask( portBitmask, self.states[ i ] )

   def render( self ):
      for row in self.states:
         state = self.states[ row ]
         print( f'SPI: {hex( state.spi )}' )
         print( f'Source: {state.srcAddr}' )
         print( f'Destination: {state.dstAddr}' )
         print( f'Encap ID: {state.encapId}' )
         print( f'Direction: {state.saDirection}' )
         print( f'Encryption: {IPSEC_ENCR_SHOW_MAP[ state.encryption ]}' )
         print( 'Authentication: '
                f"{IPSEC_AUTH_SHOW_MAP.get( state.authentication, 'n/a' )}" )
         print( f'Replay window: {state.replayWindow}' )
         if state.addTime:
            aTime = timestampToStr( state.addTime, relative=False, now=Tac.utcNow() )
            print( f'Add time: {aTime}' )
         else:
            print( 'Add time: n/a' )

         if state.useTime:
            uTime = timestampToStr( state.useTime, relative=False, now=Tac.utcNow() )
            print( f'Use time: {uTime}' )
         else:
            print( 'Use time: n/a' )

         print( f'Digest length: {state.digestLen}' )
         print( f'IV length: {state.ivLen}' )
         print( f'Block size: {state.blockSize}' )
         print( f'AutoVPN rekey: {IPSEC_REKEY_SHOW_MAP.get( state.isRekey )}' )
         print( 'Programmed location: '
                f'{IPSEC_PROGRAMMED_LOCATION_MAP.get( state.programmedLocation )}' )
         if state.hwPorts:
            print( f"Hardware ports programmed: {', '.join( state.hwPorts )}" )
         print( '' )

class SfeIpsecShowStateEncapsulationModel( Model ):
   __public__ = False
   """ Represents ipsecShowStateEncapsulation"""
   encapsulationIds = Dict( keyType=int, valueType=SfeIpsecStateEncapsulation,
                            help="Mapping of encapsulation IDs to security \
                            associations" )

   def addToEncapEntryList( self, saInfo, encapGroupToIdx ):
      entryIdx = None
      key = str( saInfo.encapId ) + ":" + str( saInfo.dstAddr )
      if key not in encapGroupToIdx:
         encapEntry = SfeIpsecStateEncapsulation()
         encapEntry.encapId = saInfo.encapId
         encapEntry.dstAddr = saInfo.dstAddr
         encapEntry.egressSpiList = ''
         encapEntry.ingressSpiList = ''
         entryIdx = len( self.encapsulationIds )
         self.encapsulationIds[ entryIdx ] = encapEntry
         encapGroupToIdx[ key ] = entryIdx
      else:
         entryIdx = encapGroupToIdx[ key ]

      encapEntry = self.encapsulationIds[ entryIdx ]
      if saInfo.saDirection == "egress":
         if encapEntry.egressSpiList:
            encapEntry.egressSpiList = ", ".join( [ encapEntry.egressSpiList,
                                                   hex( saInfo.spi ) ] )
         else:
            encapEntry.egressSpiList = hex( saInfo.spi )
      else:
         if encapEntry.ingressSpiList:
            encapEntry.ingressSpiList = ", ".join( [ encapEntry.ingressSpiList,
                                                   hex( saInfo.spi ) ] )
         else:
            encapEntry.ingressSpiList = hex( saInfo.spi )

   def setAttrsFromDict( self, data ):
      statsData = data[ 0 ]
      encapGroupToIdx = {}
      for row in statsData:
         saInfo = SfeIpsecState()
         statsEntry = statsData[ row ]
         saInfo.encapId = statsEntry[ 'encapId' ]
         saInfo.spi = statsEntry[ 'spi' ]
         saInfo.dstAddr = getDstIpAddrGenFromSaStats( statsEntry )
         saInfo.saDirection = statsEntry[ 'saDirection' ]
         self.addToEncapEntryList( saInfo, encapGroupToIdx )

   def render( self ):
      for row in self.encapsulationIds:
         encapEntry = self.encapsulationIds[ row ]
         print( f'Encap ID: {encapEntry.encapId}' )
         print( f'Destination: {encapEntry.dstAddr}' )
         if encapEntry.ingressSpiList:
            print( f'Ingress SPI list: {encapEntry.ingressSpiList}' )
         if encapEntry.egressSpiList:
            print( f'Egress SPI list: {encapEntry.egressSpiList}' )
         print( '' )
