# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Dict, Model, List, Int
import TableOutput
from ArnetModel import IpGenericPrefix

class SfeRouteCacheHitbitSummary( Model ):
   vrfs = Dict( keyType=str, valueType=int,
                help='Mapping of VRF to hitbit count' )

   def render( self ):
      headings = ( 'VRF', 'No. of prefixes' )

      table = TableOutput.createTable( headings )
      fmtVrfName = TableOutput.Format( justify='left', minWidth=16,
                                       maxWidth=64, wrap=True )
      fmtVrfName.noPadLeftIs( True )
      fmtVrfName.padLimitIs( True )

      fmtCount = TableOutput.Format( justify='left', minWidth=16,
                                     maxWidth=16, wrap=True )
      fmtCount.noPadLeftIs( True )
      fmtCount.padLimitIs( True )
      table.formatColumns( fmtVrfName, fmtCount )

      for vrfName, count in sorted( self.vrfs.items() ):
         table.newRow( vrfName, count )

      print( table.output() )

class HitbitEntry( Model ):
   prefix = IpGenericPrefix( help='IP prefix' )
   hitbit = Int( help='Hitbit value' )

class SfeRouteCacheHitbitVrf( Model ):
   hitbitEntries = List( valueType=HitbitEntry, help='List of hitbit entries' )

   def render( self ):
      headings = ( 'Prefix', 'Hitbit' )

      table = TableOutput.createTable( headings )
      fmtPrefix = TableOutput.Format( justify='left', minWidth=20,
                                      maxWidth=64, wrap=True )
      fmtPrefix.noPadLeftIs( True )
      fmtPrefix.padLimitIs( True )

      fmtHitbit = TableOutput.Format( justify='left', minWidth=16,
                                      maxWidth=16, wrap=True )
      fmtHitbit.noPadLeftIs( True )
      fmtHitbit.padLimitIs( True )
      table.formatColumns( fmtPrefix, fmtHitbit )

      for entry in self.hitbitEntries:
         table.newRow( entry.prefix, entry.hitbit )

      print( table.output() )

class SfeRouteCacheSummary( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr[ 'kickFailCount' ] = 0
      return dictRepr

   ipv4RoutePrefixHwSpaceLeft = Int( help='Space left in HW for IPv4 prefixes' )
   ipv4NotificationFailCounter = Int( help='IPv4 prefix notification fail count' )
   ipv4NotificationSuccessCounter = \
         Int( help='IPv4 prefix notification success count' )
   readKickNotificationCount = Int( help='Read kick notification count' )
   routeCachePrefixReqCount = Int( help='Route cache prefix request count' )
   kickAttemptCount = Int( help='Kick attempt count in BESS' )
   kickFailCount = Int( optional=True,
         help='Kick attempt fail count in BESS' )
   prefixEnqFailCount = Int( help='Prefix enqueue fail count in BESS' )
   hitbitCount = Int( help='Number of IPv4 prefixes in hitbit table' )
   ipv4PrefixesMovedToHwCount = Int( help='Number of IPv4 prefixes moved to HW' )
   ipv4PrefixesAgedOutCount = Int( help='Number of IPv4 prefixes aged out' )
