# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliGlobal
import ConfigMount
from TypeFuture import TacLazyType
from CliToken.Router import routerMatcherForConfig
from CliPlugin import (
        EthIntfCli,
        TunnelIntfCli,
        SubIntfCli,
        SwitchIntfCli,
        ConnectivityMonitorCli,
        IpAddrMatcher,
)
from CliPlugin.SfeServiceInsertionCliLib import (
        SiConnectionConfigContext,
        SiServiceGroupConfigContext,
        SiServiceGroupInstanceConfigContext,
)
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliMode.SfeServiceInsertion import (
        RouterServiceInsertionConfigMode,
        RouterSiConnectionConfigMode,
        RouterSiServiceGroupConfigMode,
        RouterSiServiceGroupInstanceConfigMode,
)
from Arnet import IpGenAddr
from Toggles.WanTECommonToggleLib import (
      toggleAvtRemoteInternetExitEnabled,
)

IntfId = TacLazyType( 'Arnet::IntfId' )
ServiceGroup = TacLazyType( 'SfeServiceInsertion::ServiceGroupConfig' )
ServiceGroupType = TacLazyType( 'Avt::ServiceGroupType' )

gv = CliGlobal.CliGlobal( dict( siCliConfig=None ) )

def siGetConnectionNames( mode ):
   return list( gv.siCliConfig.connectionConfig )

def siGetServiceGroupNames( mode ):
   return list( gv.siCliConfig.serviceGroupConfig )

def siGetServiceGroupInstanceNames( mode ):
   return list( mode.context.getGroupConfig().instanceConfig )

def getInstanceNameByInstanceId( group, instanceId ):
   for instanceName, instanceConfig in group.instanceConfig.items():
      if instanceId == instanceConfig.instanceId:
         return instanceName
   return None

def getGroupNameByGroupId( groupId ):
   for groupName, groupConfig in gv.siCliConfig.serviceGroupConfig.items():
      if groupId == groupConfig.groupId:
         return groupName
   return None

def getServiceGroupType( serviceType ):
   if serviceType == 'internet-exit':
      return ServiceGroupType.internetExit
   return ServiceGroupType.invalid

def removeServiceGroupComments( groupName ):
   BasicCliModes.removeCommentWithKey( f'service-group-{groupName}' )
   group = gv.siCliConfig.serviceGroupConfig[ groupName ]
   for instanceName in group.instanceConfig:
      BasicCliModes.removeCommentWithKey( f'sg-{groupName}-instance-{instanceName}' )

# ===================================================================================
# (config)# router service-insertion
# ===================================================================================

siMatcher = CliMatcher.KeywordMatcher( 'service-insertion',
               helpdesc='Configure network services inserted to data forwarding' )

class RouterServiceInsertionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''router service-insertion'''
   noOrDefaultSyntax = syntax
   data = {
           'router' : routerMatcherForConfig,
           'service-insertion' : siMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      gv.siCliConfig.enabled = True
      childMode = mode.childMode( RouterServiceInsertionConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      BasicCliModes.removeCommentWithKey( 'service-insertion' )
      for connName in gv.siCliConfig.connectionConfig:
         BasicCliModes.removeCommentWithKey( f'connection-{connName}' )
      gv.siCliConfig.connectionConfig.clear()
      for groupName in gv.siCliConfig.serviceGroupConfig:
         removeServiceGroupComments( groupName )
      gv.siCliConfig.serviceGroupConfig.clear()
      gv.siCliConfig.enabled = False

# ===================================================================================
# (config-service-insertion)# connection <connection-name>
# (config-connection-name)# interface (<eth<num>> next-hop <nexthop-addr>) |
#                                     (tunnel<num> <primary | secondary>)
# (config-connection-name)# monitor connectivity host <name>
# ===================================================================================

connectionConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      siGetConnectionNames,
      helpdesc='Name of the connection',
      helpname='WORD' )
connectionIntfMatcher = IntfMatcher()
connectionIntfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
connectionIntfMatcher |= SwitchIntfCli.SwitchIntf.matcher
connectionIntfMatcher |= SubIntfCli.subMatcher
tunnelInterfaceRoleMatcher = CliMatcher.EnumMatcher( {
    'primary' : 'Primary interface',
    'secondary' : 'Secondary interface',
    } )

class SiConnectionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''connection CONNECTION_NAME'''
   noOrDefaultSyntax = syntax
   data = {
           'connection' : 'Configure a network service connection',
           'CONNECTION_NAME' : connectionConfigNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      connName = args[ 'CONNECTION_NAME' ]
      context = SiConnectionConfigContext( gv.siCliConfig, mode, connName )
      context.addConnectionConfig( connName )
      childMode = mode.childMode( RouterSiConnectionConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      connName = args[ 'CONNECTION_NAME' ]
      BasicCliModes.removeCommentWithKey( f'connection-{connName}' )
      context = SiConnectionConfigContext( gv.siCliConfig, mode, connName )
      context.delConnectionConfig( connName )

class SiConnectionModeInterfaceCli( CliCommand.CliCommandClass ):
   syntax = '''interface ( INTF_NAME next-hop IP_ADDR ) |
                         ( TUNNEL role )'''
   noOrDefaultSyntax = '''interface ( INTF_NAME ) | ( TUNNEL ) ...'''
   data = {
           'interface' : 'Interface to be used to reach the network service',
           'INTF_NAME' : connectionIntfMatcher,
           'next-hop' : 'Nexthop to be used',
           'IP_ADDR' : IpAddrMatcher.IpAddrMatcher(
                           helpdesc='The nexthop IPv4 address' ),
           'TUNNEL' : TunnelIntfCli.TunnelIntf.matcher,
           'role' : tunnelInterfaceRoleMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      roleMapper = {
                    'primary' : True,
                    'secondary' : False,
                   }
      if args.get( 'INTF_NAME' ):
         intfId = IntfId( args[ 'INTF_NAME' ].name )
         nextHopAddr = IpGenAddr( args[ 'IP_ADDR' ] )
         mode.context.addConnectionEthInterface( intfId, nextHopAddr )
      elif args.get( 'TUNNEL' ):
         intfId = IntfId( args.get( 'TUNNEL' ).name )
         role = args.get( 'role' )
         mode.context.addConnectionTunnelInterface(
                         intfId, roleMapper.get( role ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if args.get( 'INTF_NAME' ):
         mode.context.delConnectionInterface(
               IntfId( args.get( 'INTF_NAME' ).name ) )
      elif args.get( 'TUNNEL' ):
         mode.context.delConnectionInterface(
               IntfId( args.get( 'TUNNEL' ).name ) )

class SiConnectionModeMonitorCli( CliCommand.CliCommandClass ):
   syntax = '''monitor connectivity host HOST_NAME'''
   noOrDefaultSyntax = '''monitor connectivity ...'''
   data = {
           'monitor' : 'Monitor health',
           'connectivity' : 'Monitor connectivity health',
           'host' : 'To host',
           'HOST_NAME' : ConnectivityMonitorCli.matcherHostName,
          }

   @staticmethod
   def handler( mode, args ):
      mode.context.setConnectivityMonitor( args.get( 'HOST_NAME', '' ) )

   noOrDefaultHandler = handler

serviceGroupConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      siGetServiceGroupNames,
      helpdesc='Name of the network service group',
      helpname='WORD' )
serviceGroupIdMatcher = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Service group id' )
serviceGroupInstanceConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      siGetServiceGroupInstanceNames,
      helpdesc='Name of the network service instance',
      helpname='WORD' )
serviceGroupTypeMatcher = CliMatcher.EnumMatcher( {
      'internet-exit' : 'Configure internet-exit service group type',
      } )
serviceGroupInstanceIdMatcher = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Instance id' )

class SiServiceGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''service group GROUP_NAME id GROUP_ID'''
   noOrDefaultSyntax = '''service group GROUP_NAME ...'''
   data = {
           'service' : 'Configure a network service group',
           'group' : 'Configure a network service group',
           'GROUP_NAME' : serviceGroupConfigNameMatcher,
           'id' : 'Configure a network service group id',
           'GROUP_ID' : serviceGroupIdMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      groupName = args[ 'GROUP_NAME' ]
      groupId = args[ 'GROUP_ID' ]

      context = SiServiceGroupConfigContext( gv.siCliConfig, mode, groupName )

      # GroupId exists in a service group
      existingGroupName = getGroupNameByGroupId( groupId )
      if existingGroupName and existingGroupName != groupName:
         removeServiceGroupComments( existingGroupName )
         context.delGroupConfig( existingGroupName )

      if groupName in gv.siCliConfig.serviceGroupConfig:
         group = gv.siCliConfig.serviceGroupConfig[ groupName ]
         # If we change the groupId of an existing group, then we are clearing the
         # old service group configs.
         if group.groupId != groupId:
            removeServiceGroupComments( groupName )
            context.delGroupConfig( groupName )
            context.addGroupConfig( groupName, groupId )
      else:
         context.addGroupConfig( groupName, groupId )

      childMode = mode.childMode( RouterSiServiceGroupConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupName = args[ 'GROUP_NAME' ]
      if groupName in gv.siCliConfig.serviceGroupConfig:
         removeServiceGroupComments( groupName )
      context = SiServiceGroupConfigContext( gv.siCliConfig, mode, groupName )
      context.delGroupConfig( groupName )

class SiServiceGroupTypeConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''type TYPE'''
   noOrDefaultSyntax = '''type ...'''
   data = {
           'type' : 'Type of the network service group',
           'TYPE' : serviceGroupTypeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      serviceType = args[ 'TYPE' ]
      serviceGroupType = getServiceGroupType( serviceType )
      mode.context.setServiceGroupType( serviceGroupType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceGroupType = ServiceGroupType.invalid
      mode.context.setServiceGroupType( serviceGroupType )

class SiServiceGroupInstanceConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''instance INSTANCE_NAME id INSTANCE_ID'''
   noOrDefaultSyntax = '''instance INSTANCE_NAME ...'''
   data = {
           'instance' : 'Configure an instance of the network service group',
           'INSTANCE_NAME' : serviceGroupInstanceConfigNameMatcher,
           'id' : 'Configure a network service instance id',
           'INSTANCE_ID' : serviceGroupInstanceIdMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      instanceName = args[ 'INSTANCE_NAME' ]
      instanceId = args[ 'INSTANCE_ID' ]
      groupName = mode.context.groupName()
      group = gv.siCliConfig.serviceGroupConfig[ groupName ]
      assert group

      context = SiServiceGroupInstanceConfigContext( group, mode, instanceName )

      # InstanceId exists in a service group
      existingInstanceName = getInstanceNameByInstanceId( group, instanceId )
      if existingInstanceName and existingInstanceName != instanceName:
         BasicCliModes.removeCommentWithKey(
            f'sg-{groupName}-instance-{existingInstanceName}' )
         context.delInstanceConfig( existingInstanceName )

      if instanceName in group.instanceConfig:
         instance = group.instanceConfig[ instanceName ]
         # If we change the instanceId of an existing instance, then we are clearing
         # the old service instance configs
         if instance.instanceId != instanceId:
            BasicCliModes.removeCommentWithKey(
               f'sg-{groupName}-instance-{instanceName}' )
            context.delInstanceConfig( instanceName )
            context.addInstanceConfig( instanceName, instanceId )
      else:
         if len( group.instanceConfig.keys() ) == ServiceGroup.maxInstances:
            mode.addError( 'Can\'t configure more than ' +
                           f'{ServiceGroup.maxInstances} service instances ' +
                            'in a network service group' )
            return
         context.addInstanceConfig( instanceName, instanceId )

      childMode = mode.childMode( RouterSiServiceGroupInstanceConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceName = args[ 'INSTANCE_NAME' ]
      groupName = mode.context.groupName()
      group = gv.siCliConfig.serviceGroupConfig[ groupName ]
      assert group
      BasicCliModes.removeCommentWithKey(
            f'sg-{groupName}-instance-{instanceName}' )
      context = SiServiceGroupInstanceConfigContext( group, mode, instanceName )
      context.delInstanceConfig( instanceName )

class SiServiceGroupInstanceConnConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''connection CONNECTION_NAME'''
   noOrDefaultSyntax = '''connection ...'''
   data = {
           'connection' : 'Connection of the service instance',
           'CONNECTION_NAME' : connectionConfigNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      connectionName = args.get( 'CONNECTION_NAME', '' )
      mode.context.setConnectionName( connectionName )

   noOrDefaultHandler = handler

# Commands that define router service-insertion general attributes (if any)
BasicCli.GlobalConfigMode.addCommandClass( RouterServiceInsertionConfigCmd )

# Commands that define router service-insertion connection mode clis
RouterServiceInsertionConfigMode.addCommandClass( SiConnectionConfigCmd )
RouterSiConnectionConfigMode.addCommandClass( SiConnectionModeInterfaceCli )
RouterSiConnectionConfigMode.addCommandClass( SiConnectionModeMonitorCli )

if toggleAvtRemoteInternetExitEnabled():
   # Commands that define router network service-group mode clis
   RouterServiceInsertionConfigMode.addCommandClass( SiServiceGroupConfigCmd )
   RouterSiServiceGroupConfigMode.addCommandClass( SiServiceGroupTypeConfigCmd )

   # Commands that define router service-group-instance mode clis
   RouterSiServiceGroupConfigMode.addCommandClass( SiServiceGroupInstanceConfigCmd )
   RouterSiServiceGroupInstanceConfigMode.addCommandClass(
         SiServiceGroupInstanceConnConfigCmd )

# ===================================================================================
# Register the plugin
# ===================================================================================

def Plugin( entityManager ):
   gv.siCliConfig = ConfigMount.mount( entityManager,
                       'si/cli/config',
                       'SfeServiceInsertion::ServiceInsertionConfigDir', 'w' )
