# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
import BasicCli
import AgentCommandRequest
import SfeAgent
from io import StringIO
import LazyMount
import CliMatcher
import ShowCommand
import Tac
import ast
from CliPlugin.SfeServiceInsertionShowCliModel import (
      SfeSiAllConnStatusModel,
      SfeSiConnStatusModel,
      SfeSiConnIntfStatusModel,
      SfeSiGroupModel,
)
from Toggles.WanTECommonToggleLib import (
      toggleAvtRemoteInternetExitEnabled,
)

gv = CliGlobal.CliGlobal( dict( siCliConfigDir=None,
         siConnStatusDir=None,
         siConnResolverStatusDir=None,
         siConnIntfStatusDir=None, ) )

siMatcher = CliMatcher.KeywordMatcher(
   keyword='service-insertion',
   helpdesc='Network services inserted to data forwarding',
)

connNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda _: gv.siCliConfigDir.connectionConfig,
   helpdesc='Name of the connection' )

groupNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda _: gv.siCliConfigDir.serviceGroupConfig,
   helpdesc='Name of the service group' )

def getSfeSiConnIntfStatusModel( intfId, monHostKey ):
   model = SfeSiConnIntfStatusModel()
   model.intfId = intfId
   intfStatus = None
   if intfId in gv.siConnIntfStatusDir.connectionInterface:
      intfStatus = gv.siConnIntfStatusDir.connectionInterface[ intfId ]
      model.intfEnabled = intfStatus.intfState
      model.vrf = intfStatus.vrf
   if not monHostKey:
      return model
   if intfStatus and monHostKey in intfStatus.hostState:
      model.monHostReachable = intfStatus.hostState[ monHostKey ]
   else:
      model.monHostReachable = False
   return model

def getSfeSiConnStatusModel( connName ):
   model = SfeSiConnStatusModel()
   model.connName = connName
   siConnConfig = gv.siCliConfigDir.connectionConfig
   siConnStatus = gv.siConnStatusDir.connection
   # If connStatus is there pull values from here
   # otherwise display information form the connConfig
   if connName in siConnStatus:
      connStatus = siConnStatus[ connName ]
      resultConnIntf = connStatus.resultConnIntf
      model.isActive = bool( resultConnIntf.intfId )
      primaryIntf = connStatus.primaryIntf
      secondaryIntf = connStatus.secondaryIntf
      monHost = connStatus.monHostName
   else:
      connConfig = siConnConfig[ connName ]
      model.isActive = False
      primaryIntf = connConfig.primaryIntf
      secondaryIntf = connConfig.secondaryIntf
      monHost = connConfig.monConnectivityHostName
   # populate monHost, primaryIntf, secondaryIntf information
   monHostKey = None
   if monHost:
      model.connectivityMonitor = monHost
      monHostKey = Tac.ValueConst( 'SfeServiceInsertion::ConnectionHost',
                                    connName, monHost )
   if primaryIntf and primaryIntf.intfId:
      model.primaryIntf = getSfeSiConnIntfStatusModel( primaryIntf.intfId,
                                                       monHostKey )
   if secondaryIntf and secondaryIntf.intfId:
      model.secondaryIntf = getSfeSiConnIntfStatusModel( secondaryIntf.intfId,
                                                         monHostKey )
   # populate nextHop information
   if primaryIntf.nextHop.isAddrZero:
      return model
   model.nextHop = primaryIntf.nextHop.stringValue
   if primaryIntf in gv.siConnResolverStatusDir.connectionResolver:
      model.nhResolved = gv.siConnResolverStatusDir.\
            connectionResolver[ primaryIntf ].isValid()
   else:
      model.nhResolved = False
   return model

# ---------------------------------------------------------------------------------
# The 'show service-insertion connection [ NAME ]' command
# ---------------------------------------------------------------------------------
def getSfeSiAllConnStatusModel( mode, args ):
   model = SfeSiAllConnStatusModel()
   siConnConfig = gv.siCliConfigDir.connectionConfig

   connections = []
   if 'NAME' in args:
      connection = args[ 'NAME' ]
      if connection not in siConnConfig:
         return model
      connections.append( connection )
   else:
      connections = list( siConnConfig )

   for connection in connections:
      model.connectionStatus.append( getSfeSiConnStatusModel( connection ) )
   return model

class SfeSiConnectionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service-insertion connection [ NAME ]'
   data = {
      'service-insertion' : siMatcher,
      'connection' : 'Network service connection',
      'NAME' : connNameMatcher,
   }
   cliModel = SfeSiAllConnStatusModel
   handler = getSfeSiAllConnStatusModel

BasicCli.addShowCommandClass( SfeSiConnectionCmd )

# ---------------------------------------------------------------------------------
# The 'show service-insertion group [ NAME ]' command
# ---------------------------------------------------------------------------------
def getSfeSiGroupModel( mode, args ):
   buff = StringIO()
   cmd = 'SiGroup'
   if 'NAME' in args:
      cmd += args[ 'NAME' ]
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         'sfe', cmd, stringBuff=buff )
   output = buff.getvalue()
   try:
      serviceGroups = ast.literal_eval( output )
   except SyntaxError:
      mode.addError( output )
      return SfeSiGroupModel()
   model = SfeSiGroupModel()
   model.assignData( serviceGroups )
   return model

class SfeSiGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service-insertion group [ NAME ]'
   data = {
      'service-insertion' : siMatcher,
      'group' : 'Service insertion group',
      'NAME' : groupNameMatcher,
   }
   cliModel = SfeSiGroupModel
   handler = getSfeSiGroupModel

if toggleAvtRemoteInternetExitEnabled():
   BasicCli.addShowCommandClass( SfeSiGroupCmd )

# ---------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from Sysdb
# ---------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.siCliConfigDir = LazyMount.mount( entityManager, 'si/cli/config',
                           'SfeServiceInsertion::ServiceInsertionConfigDir', 'r' )
   gv.siConnStatusDir = LazyMount.mount( entityManager, 'si/status/connection',
                           'SfeServiceInsertion::SiConnectionStatusDir', 'r' )
   gv.siConnResolverStatusDir = LazyMount.mount( entityManager,
                           'si/status/connectionResolver',
                           'SfeServiceInsertion::SiConnectionResolverDir', 'r' )
   gv.siConnIntfStatusDir = LazyMount.mount( entityManager,
                           'si/status/connectionInterface',
                           'SfeServiceInsertion::SiConnectionInterfaceDir', 'r' )
