# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Str, List, Bool, Int

def printFormatted( line, leadingSpaces, sep='' ):
   spaces = ' ' * leadingSpaces
   print( spaces, line, sep=sep )

def displaySiConnIntfStatus( interfaceStatus, showMonitorState=True,
                      leadingSpaces=0, isPrimary=True ):
   if interfaceStatus.intfId.startswith( "Tunnel" ):
      intfPriority = "primary" if isPrimary else "secondary"
      printFormatted( f"Interface: {interfaceStatus.intfId} ({intfPriority})",
                        leadingSpaces )
   else:
      printFormatted( f"Interface: {interfaceStatus.intfId}", leadingSpaces )
   leadingSpaces += 2

   intfEnabled = "up" if interfaceStatus.intfEnabled else "down"
   printFormatted( f"State: {intfEnabled}", leadingSpaces )
   printFormatted( f"VRF: {interfaceStatus.vrf}", leadingSpaces )
   if showMonitorState:
      monHostReachable = "up" if interfaceStatus.monHostReachable else "down"
      printFormatted( f"Monitor state: {monHostReachable}", leadingSpaces )

def displaySfeSiConnStatus( connection, leadingSpaces=0 ):
   printFormatted( f"Connection: {connection.connName}", leadingSpaces )
   leadingSpaces += 2

   isActive = "yes" if connection.isActive else "no"
   printFormatted( f"Active: {isActive}", leadingSpaces )

   if connection.connectivityMonitor is not None:
      printFormatted( f"Connectivity monitor: {connection.connectivityMonitor}",
                       leadingSpaces )
   if connection.primaryIntf:
      displaySiConnIntfStatus( connection.primaryIntf,
                        connection.connectivityMonitor is not None,
                        leadingSpaces )
   if connection.secondaryIntf:
      displaySiConnIntfStatus( connection.secondaryIntf,
                        connection.connectivityMonitor is not None,
                        leadingSpaces, isPrimary=False )
   if connection.nextHop:
      isNhResolved = "resolved" if connection.nhResolved else "unresolved"
      printFormatted( f"Nexthop: {connection.nextHop} ({isNhResolved})",
                        leadingSpaces )

class SfeSiConnIntfStatusModel( Model ):
   intfId = Str( help="Configured interface" )
   intfEnabled = Bool( help="Interface is enabled", default=False )
   vrf = Str( help="Configured VRF on this interface", default="default" )
   monHostReachable = Bool( help="Monitor host is reachable", optional=True )

class SfeSiConnStatusModel( Model ):
   connName = Str( help="Connection name" )
   isActive = Bool( help="Connection is active or not" )
   connectivityMonitor = Str( help="Configured connectivity monitor hostname",
         optional=True )
   primaryIntf = Submodel( valueType=SfeSiConnIntfStatusModel,
         help="Configured primary interface", optional=True )
   secondaryIntf = Submodel( valueType=SfeSiConnIntfStatusModel,
         help="Configured secondary interface", optional=True )
   nextHop = Str( help="Nexthop of this exit", optional=True )
   nhResolved = Bool( help="Nexthop is resolved", optional=True )

# Model for 'show router service-insertion connectivity <con_name>
class SfeSiAllConnStatusModel( Model ):
   connectionStatus = List( valueType=SfeSiConnStatusModel,
                            help="All Connection Status" )

   def render( self ):
      for connection in self.connectionStatus:
         displaySfeSiConnStatus( connection )

class SfeSiGroupModel( Model ):

   class ServiceGroup( Model ):
      groupId = Int( help="Service group id" )
      groupName = Str( help="Service group name" )
      groupType = Str( help="Service group type" )

      class ServiceInstance( Model ):
         instanceId = Int( help="Service instance id" )
         instanceName = Str( help="Service instance name" )
         connName = Str( optional=True, help="Name of the connection used by" +
                        " instance" )
         connStatus = Str( optional=True, help="If the connection used by " +
                     "instance is contained in SI connection status, it is active" +
                     "otherwise inactive" )

      instances = List( valueType=ServiceInstance, help="List of Service Instances" +
                        " under a service group" )
   serviceGroups = List( valueType=ServiceGroup, help="List of the service groups" )

   def sortData( self ):
      for group in self.serviceGroups:
         group.instances = sorted( group.instances,
                                       key=lambda ele: ele[ 'instanceName' ] )
      self.serviceGroups = sorted( self.serviceGroups,
                                      key=lambda ele: ele[ 'groupName' ] )

   def assignData( self, data ):
      for group in data[ "serviceGroups" ]:
         sg = self.ServiceGroup()
         for key, value in group.items():
            if key == "instances":
               for instance in value:
                  inst = sg.ServiceInstance()
                  inst.setAttrsFromDict( instance )
                  sg.instances.append( inst )
            elif key == "groupType":
               sg.groupType = 'internet' if value == '1' else 'invalid'
            else:
               setattr( sg, key, value )
         self.serviceGroups.append( sg )
      self.sortData()

   def render( self ):
      for group in self.serviceGroups:
         leadingSpaces = 0
         printFormatted( f"Group: {group.groupName} ({group.groupId})",
                         leadingSpaces )
         leadingSpaces += 2
         printFormatted( f"Type: {group.groupType}", leadingSpaces )
         for instance in group.instances:
            printFormatted( f"Instance: {instance.instanceName} "
                            f"({instance.instanceId})", leadingSpaces )
            if instance.connName is not None:
               leadingSpaces += 2
               printFormatted( f"Connection: {instance.connName} "
                               f"({instance.connStatus})", leadingSpaces )
               leadingSpaces -= 2
